/*
 * ip.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * Internet Protocol.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/mbuf.h>
#include <lib/lib.h>
#include <kern/Thread.h>
#include <kern/lock.h>
#include <kern/vm.h>
#include <kern/errno.h>
#include <kern/time.h>
#include <kern/timer.h>
#include <kern/Wait.h>
#include <net/net.h>
#include <net/netlib.h>
#include <net/ethernet.h>
#include <net/arp.h>
#include <net/icmp.h>
#include <net/udp.h>
#include <net/tcp.h>
#include <net/ip.h>


//#define DEBUG_IP 1
#ifdef DEBUG_IP
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


/**************************************************************************
 * IPإå
 **************************************************************************/

//================================== PRIVATE ============================================

/*
 * IPǡΥ
 */
STATIC INLINE int getDataSize(IP_HEADER *ip)
{
	return swapWord(ip->len) - sizeof(IP_HEADER);
}

/*
 * IPǡΥե饰ȥեåȤ
 */
STATIC INLINE uint16_t getFragOffset(IP_HEADER *ip)
{
	return swapWord(ip->fragOffset) & FRAG_OFFSET_MASK;
}

/**************************************************************************
 *
 * 
 *
 **************************************************************************/

//================================== PRIVATE ============================================

enum{
	IP_HEAD_VERSION=4<<4,

	IP_HEAD_FRAG_ON= 0x2000,	/* ե饰ȥǡե饰 */
	IP_HEAD_FRAG_NOT=0x4000,	/* ե饰Բĥե饰 */
};

typedef struct{
	int    lock;
	ushort id;
}HEAD_ID;

#define GET_ID(a)	enter_spinlock(&headId.lock);\
			 		a = swapWord(headId.id++);\
					exit_spinlock(&headId.lock)

static HEAD_ID headId;

/************************************************************************************************
static void viewHead(IP_HEADER *head)
{
	printk("TRANS IP version=%x,head length=%x,TOS=%x,total length=%x,id=%x,fragment=%x\n"
	       "TTL=%x,protocol=%x,sourceIP=%x,destIP=%x\n",
	       head->verhead>>4,head->verhead&0xf,head->tos,swapWord(head->len),head->id,head->frag,
	       head->ttl,head->prot,head->srcip,head->dstip);
}
*************************************************************************************************/

//================================== PUBLIC =============================================

/*
 *ահ"sock""NULL"
 * Transfer IP datagram.
 * return : 0 or error number
 */
int transIp(SOCKET *sock, TRANS_BUF_INFO *tbi, uint dstip, uint16_t frag)
{
	char dstmac[6];
	uchar tos;
	IP_HEADER head;
	int max, total, trans;
	int rest;
	int num = getEtherNum(dstip);
	uint32_t transip = getTransIp(dstip, num);

	/*  MACμ */
	if ((rest = getMac(num, transip, dstmac)) != NOERR){
		return rest;
	}

	// IPץ
	tos = 0;
	if (sock != NULL){
		if (sock->ipOpt.name == IP_TOS){
			tos = sock->ipOpt.value;
		}
	}

	/* Set IP head. */
	head.verhead = IP_HEAD_VERSION | (sizeof(IP_HEADER) / 4);
	head.tos = tos;
	GET_ID(head.id);
	head.ttl = 255;
	head.prot = tbi->ipType;
	head.srcip = getEtherIp(num);
	head.dstip = dstip;
	head.chksum = 0;

	tbi->type = ETHERTYPE_IP;
	tbi->data[0] = (char*)&head;
	tbi->size[0] = sizeof(IP_HEADER);			/* IP header size. */
	max = getEtherMtu(num) - sizeof(IP_HEADER);	/* ǡ */
	total = tbi->size[1] + tbi->size[2];		/* ǡȡ륵 */

	if (total <= max){
		head.len = swapWord(tbi->size[0] + total);
		head.fragOffset = swapWord(frag);
		head.chksum = calcSum((uint*)&head, tbi->size[0]);
/**********************************************
viewHead(&head);
**********************************************/
		transEther(getEtherIfnet(num), tbi, dstmac);
	}
	else{
		head.len = swapWord(tbi->size[0] + max);
		head.fragOffset = swapWord(IP_HEAD_FRAG_ON);
		tbi->size[2] = max-tbi->size[1];
		head.chksum = calcSum((uint*)&head, tbi->size[0]);
/**********************************************
viewHead(&head);
**********************************************/
		transEther(getEtherIfnet(num), tbi, dstmac);

		trans = max;						/* ѥ */
		tbi->data[2] += max-tbi->size[1];
		tbi->size[1] = 0;

		for(;;){
			head.chksum = 0;

			if(total - trans <= max){
				head.len = swapWord(tbi->size[0] + total - trans);
				head.fragOffset = swapWord(trans);
				GET_ID(head.id);
				head.chksum = calcSum((uint*)&head, tbi->size[0]);
				tbi->size[2] = total - trans;
/**********************************************
viewHead(&head);
**********************************************/
				transEther(getEtherIfnet(num), tbi, dstmac);
				break;
			}
			else{
				head.len = swapWord(tbi->size[0] + max);
				head.fragOffset = swapWord(trans | IP_HEAD_FRAG_ON);
				GET_ID(head.id);
				head.chksum = calcSum((uint*)&head, tbi->size[0]);
				tbi->size[2] = max;
/**********************************************
viewHead(&head);
**********************************************/
				transEther(getEtherIfnet(num), tbi, dstmac);
			}

			trans += max;
			tbi->data[2] += max;
		}
	}

	return 0;
}

/**************************************************************************
 *
 * ե饰ơǡη
 *
 **************************************************************************/

//================================== PRIVATE ============================================

/* ե饰ȴإå */
enum{
	DATA_LEN_MAX		= 0x10000,	// IPǡ祵
	FRAG_HEAD_USE		= 1,		// ե饰ȥإå
	FRAG_HEAD_NOT_USE	= 0,		// ե饰ȥإå̤
	REASSENBLED			= 1,		// ƹ۴λ
	FRAG_LIVE_TIME		= 5,		// ե饰ȥǡ¸֡á
	FRAG_HEAD_NUM		= 1,		// ե饰ȥإå
};

typedef struct FRAG_HEAD{
	int 	use;		// ѺѤߥե饰
	int		curLen;		// ߤǡĹIPإåȴ
	uint 	creatTime;	// 
	char	*data;		// ǡǼХåե
}FRAG_HEAD;

static char fragData[DATA_LEN_MAX];

static FRAG_HEAD fragHead[FRAG_HEAD_NUM] = {
	{FRAG_HEAD_NOT_USE,
	0,
	0,
	fragData}
};

/*
 * IPɥ쥹
 */
STATIC INLINE uint32_t getDstIp(FRAG_HEAD *fragHead)
{
	return ((IP_HEADER*)fragHead->data)->dstip;
}

/*
 * IPֹ
 */
STATIC INLINE uint16_t getId(FRAG_HEAD *fragHead)
{
	return ((IP_HEADER*)fragHead->data)->id;
}

/*
 * ե饰ȥإåγ
 */
STATIC void releaseFragHead(FRAG_HEAD *fragHead)
{
	fragHead->use = FRAG_HEAD_NOT_USE;
	fragHead->curLen = 0;
	fragHead->creatTime = 0;
}

/*
 * αΥե饰ȥǡθ
 * ե饰ȥإåɥ쥹 or NULL
 */
STATIC FRAG_HEAD *searchFragHead(IP_HEADER *ip)
{
	int i;

	for (i = 0; i < FRAG_HEAD_NUM; ++i){
		/* IPȼֹ椬 */
		if ((ip->dstip == getDstIp(&fragHead[i])) && (ip->id == getId(&fragHead[i]))){
			return &fragHead[i];
		}
	}

	return NULL;
}

/*
 * ե饰ȥإåν
 * return : NOERR or ERR
 */
STATIC int initFragHead(IP_HEADER *ip)
{
	uint curTime = sys_time(NULL);
	int freeNum;
	IP_HEADER *fragIp;
	
	// ̤ѤΥإåõ
	for (freeNum = 0; freeNum < FRAG_HEAD_NUM; ++freeNum){
		if (FRAG_HEAD_NUM <= freeNum){
			return ERR;
		}
		if (fragHead[freeNum].use == FRAG_HEAD_NOT_USE){
			break;
		}
		else if (FRAG_LIVE_TIME < (curTime - fragHead[freeNum].creatTime)){
			/* ¸֤᤮إå */
			break;
		}
	}

	/* IPإå򥳥ԡ */
	fragIp = (IP_HEADER*)&fragHead[freeNum].data;
	memcpy(fragIp, ip, sizeof(IP_HEADER));
	if (swapWord(ip->fragOffset) & FRAG_MORE){
		fragIp->len = 0;
	}
	else{
		fragIp->len = swapWord(getFragOffset(ip) + getDataSize(ip) + sizeof(IP_HEADER));
	}
	
	// IPǡ򥳥ԡ
	memcpy(fragIp->data + getFragOffset(ip), ip->data, getDataSize(ip));

	fragHead->curLen = getDataSize(ip);		// ߤǡĹ
	fragHead->creatTime = curTime;			// 
	fragHead[freeNum].use = FRAG_HEAD_USE;
	
	return NOERR;
}

/*
 * ե饰Ȥκƹ
 * return : NOERR or REASSENBLED or error number
 */
STATIC int reassemble(IP_HEADER *ip, FRAG_HEAD *fragHead)
{
	IP_HEADER *fragIp = (IP_HEADER*)fragHead->data;

	/* IPǡ򥳥ԡ */
	memcpy(fragIp->data + getFragOffset(ip), ip->data, getDataSize(ip));
	fragHead->curLen += getDataSize(ip);

	/* IPȡĹ */
	if ((swapWord(ip->fragOffset) & FRAG_MORE) == 0){
		fragIp->len = swapWord(getFragOffset(ip) + getDataSize(ip) + sizeof(IP_HEADER));
	}

	/* ٤Ʒ礷 */
	if (getDataSize(fragIp) == fragHead->curLen){
		return REASSENBLED;
	}
	
	return NOERR;
}

/*
 * ե饰ȥǡå
 * return : NOERR or ERR
 */
STATIC int checkFragSize(IP_HEADER *ip)
{
	/* 祵Υå */
	if (DATA_LEN_MAX < getFragOffset(ip) + getDataSize(ip) + sizeof(IP_HEADER)){
		return  ERR;
	}
	return NOERR;
}

/*
 * ե饰ȥǡκƹ
 * return : NOERR or REASSENBLED or error number
 */
STATIC int reassembleFragment(IP_HEADER *ip,void **o_fragHead)
{
	FRAG_HEAD *fragHead;
	
	/* αΥե饰ȥǡθ */
	fragHead = searchFragHead(ip);
	if (fragHead == NULL){
		/* ե饰ȥإåο */
		return initFragHead(ip);
	}
	else{
		/* ե饰ȥǡη */
		if (reassemble(ip, fragHead) == REASSENBLED){
			*o_fragHead = fragHead;
			return REASSENBLED;
		}
	}
	
	return NOERR;
}

/*
 * ե饰ȥǡγ
 */
STATIC void freeFragment(FRAG_HEAD *fragHead)
{
	releaseFragHead(fragHead);
}

/*
 *աether_headerʬ
 * IPǡmbuf쥤ȤǼ
 * return : struct mbuf* or NULL
 */
STATIC struct mbuf *getIpFromFragment(FRAG_HEAD *fragHead)
{
	return m_devget(
		fragHead->data, 
		fragHead->curLen + sizeof(IP_HEADER) + sizeof(struct ether_header), 
		sizeof(struct ether_header),
		NULL, 
		NULL);
}

/**************************************************************************
 *
 * 
 *
 **************************************************************************/

//================================== PUBLIC =============================================

/*
 * Receive IP datagram.
 * parameters : IP header
 * return : 0 or task switch=1
 */
int receiveIp(struct mbuf *mbuf)
{
	IP_HEADER *ip = getMbufDataPointer(mbuf);
	int (*receive)(struct mbuf*);

/*****************************************************************************************
TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

printk("IP version=%x,ip length=%x,TOS=%x,total length=%x,id=%x,fragment=%x "
       "TTL=%x,protocol=%x,sourceIP=%x,destIP=%x\n",
       ip->verhead >> 4,ip->verhead & 0xf,ip->tos,swapWord(ip->len),ip->id,
       swapWord(ip->fragOffset),ip->ttl,ip->prot,ip->srcip,ip->dstip);
*******************************************************************************************/
	// åγǧ
	if (calcSum((uint*)ip, (ip->verhead & 0xf) * 4)){
		return 0;
	}

	// ̥ץȥؿ򥻥å
	switch(ip->prot){
		case IPPROTO_TCP:
			receive = receiveTcp;
			break;
		case IPPROTO_UDP:
			receive = receiveUdp;
			break;
		case IPPROTO_ICMP:
			receive = receiveIcmp;
			break;
		default:
			return 0;
	}

	// ե饰IPǡν
	if ((swapWord(ip->fragOffset) & FRAG_MORE) || (0 < getFragOffset(ip))){
		void *fragment;
		int rest;

		// Υå
		if (checkFragSize(ip) == ERR){
			m_freem(mbuf);
			return 0;
		}
		
		// ե饰Ȥη
		if (reassembleFragment(ip, &fragment) != REASSENBLED){
			m_freem(mbuf);
			return 0;
		}

		rest = receive(getIpFromFragment(fragment));
		freeFragment(fragment);
		m_freem(mbuf);
		return rest;
	}

	return receive(mbuf);
}

/**************************************************************************
 *
 * IPå
 *
 **************************************************************************/

//================================== PRIVATE ============================================

static void *ipTask;
static AggregateList recvBuf;
static int recvLock;

/*
 * IPå
 */
STATIC void IpTask()
{
	struct mbuf *mbuf;
	IP_HEADER *ip;

	mbuf = getRecvBuf(&recvBuf, &recvLock);
	if (mbuf == NULL) {
		waitTaskSignal();
	}
	else {
		ip = getMbufDataPointer(mbuf);
		switch (ip->prot){
		case IPPROTO_TCP:
			doTcpService(mbuf->mh_sock, ip);
			break;
		case IPPROTO_UDP:
			doUdpService(mbuf->mh_sock, ip);
			break;
		case IPPROTO_ICMP:
			// ̤б
			break;
		default:
			break;
		}
		m_freem(mbuf);
	}
}

/*
 * IPåɤεư
 */
STATIC void startIpTask()
{
	switch(sys_fork()){
	case -1:
		printk("Faild startIpTask()\n");
		break;
	case 0:
		ipTask = getCurrentTask();
		initRecvBuf(&recvBuf);
		for (;;) {
			IpTask();
		}
		break;
	}
}

//================================== PUBLIC =============================================

/*
 * IPåɤε
 *առ߻Τߤ˻Ѥ뤳
 */
void setIpTask(SOCKET *sock, struct mbuf *mbuf)
{
	ASSERT(mbuf != NULL);
	mbuf->mh_sock = sock;
	putRecvBuf(mbuf, &recvBuf, &recvLock);
	activeTaskSignal(getWaitTask(ipTask));
}

/**************************************************************************
 *
 * 
 *
 **************************************************************************/

//================================== PUBLIC =============================================

int initIp()
{
	startIpTask();
	
	return 0;
}
