/*
 * tcp.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * Transmission control Protocl.
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dev/console/console.h>
#include <lib/lib.h>
#include <machine/interrupt.h>
#include <kern/errno.h>
#include <kern/kmalloc.h>
#include <kern/Thread.h>
#include <kern/time.h>
#include <kern/lock.h>
#include <kern/ProcSignal.h>
#include <kern/fs.h>
#include <kern/timer.h>
#include <net/net.h>
#include <net/ethernet.h>
#include <net/ip.h>
#include <net/tcp.h>
#include <net/netlib.h>

#include <kern/debug.h>


//#define DEBUG_TCP 1
#ifdef DEBUG_TCP
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


//==================================  ===========================================

enum{
	/* TCP flag. */
	TCP_FLG_FIN = 1<<0,
	TCP_FLG_SYN = 1<<1,
	TCP_FLG_RST = 1<<2,
	TCP_FLG_PSH = 1<<3,
	TCP_FLG_ACK = 1<<4,
	TCP_FLG_URG = 1<<5,

	TCP_WIN_SIZE=0xffff,	/* Window size. */
};

/*
 * MSSǡ
 */
typedef struct{
	uint8_t  kind;
	uint8_t  len;
	uint16_t mss;
}MSS;

//================================== PRIVATE ============================================

/*
 * TCPХȿ֤
 */
STATIC INLINE int getTcpSize(IP_HEADER *ip)
{
	return swapWord(ip->len) - (ip->verhead & 0xf) * 4;
}

/*
 * TCPǡХȿ֤
 */
STATIC int getTcpDataSize(IP_HEADER *ip)
{
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

	return getTcpSize(ip) - (tcp->headlen >> 4) * 4;
}

/**************************************************************************
 *
 * 
 *
 **************************************************************************/

//================================== PRIVATE ============================================

enum{
	/* Header option. */
	TCP_KIND_END=  0,
	TCP_KIND_NOP=  1,
	TCP_KIND_MSS=  2,
	TCP_KIND_WSC=  3,
	TCP_KIND_TSTMP=8,

	TCP_OPT_MAXSIZE=32,		/* TCP ץκ祵 */
};

static SOCKET waitCnctSc;	/* Ԥ󥯡 */
static SOCKET waitLstnSc;	/* å󥯡 */

/*
 * إåꡣ
 */
STATIC INLINE void setPseudoHead(
	in_addr_t dstip,
	in_addr_t srcip,uint16_t size,
	PSEUDO_HEADER *pshead)
{
	pshead->dstip = dstip;
	pshead->srcip = srcip;
	pshead->tmp = 0;
	pshead->prot = IPPROTO_TCP;
	pshead->len = swapWord(size);
}

//-----------------------------  -------------------------------------------

/*
 * TCPȤ
 *աեץǡɬintǳڤ륵ˤ뤳ȡ
 */
STATIC void transSegment(
	SOCKET *sock,
	uint8_t flag,
	void *opt,
	int opt_size,
	void *data,
	int data_size,
	uint16_t wndsize)
{
	char head[sizeof(PSEUDO_HEADER) + sizeof(TCP_HEADER) + TCP_OPT_MAXSIZE];
	TRANS_BUF_INFO tbi;
	TCP_HEADER *send_tcp = (TCP_HEADER*)(head + sizeof(PSEUDO_HEADER));

	// إå
	setPseudoHead(sock->dstip, getEtherIp(getEtherNum(sock->dstip)), sizeof(TCP_HEADER) + opt_size+data_size, (PSEUDO_HEADER*)head);

	// Хåեơ֥
	tbi.data[2] = data;
	tbi.size[2] = data_size;
	tbi.data[1] = (char*)send_tcp;
	tbi.size[1] = sizeof(TCP_HEADER) + opt_size;
	tbi.ipType = IPPROTO_TCP;

	// Make TCP header.
	send_tcp->srcport = sock->srcport;
	send_tcp->dstport = sock->dstport;
	send_tcp->seqnum =  swapInt32(sock->seqnum);
	send_tcp->acknum =  swapInt32(sock->acknum);
	send_tcp->headlen = ((sizeof(TCP_HEADER) + opt_size) / sizeof(uint32_t)) << 4;
	send_tcp->flag = flag;
	send_tcp->wndsize = swapWord(wndsize);
	send_tcp->chksum = 0;
	send_tcp->urgpoint = 0;
	memcpy(send_tcp->option, opt, opt_size);
	send_tcp->chksum = calcSumM((uint*)head, data, sizeof(PSEUDO_HEADER) + sizeof(TCP_HEADER) + opt_size, data_size);
/************************************************************************************************************************
{
	union{
		uint ui;
		uchar uc[4];
	}src_ip;

	src_ip.ui = sock->dstip;
	printk("send srcport=%x,dstport=%x,seqnum=%x,acknum=%x,flag=%x,ip=%u.%u.%u.%u\n",
		swapWord(send_tcp->srcport),swapWord(send_tcp->dstport),swapInt32(send_tcp->seqnum),swapInt32(send_tcp->acknum),
		send_tcp->flag,src_ip.uc[0],src_ip.uc[1],src_ip.uc[2],src_ip.uc[3]);
}
************************************************************************************************************************/
	// 
	transIp(sock, &tbi, sock->dstip, 0);
}

/*
 * FIN
 */
STATIC void sendFinAck(SOCKET *sock)
{
	sock->stat |= SOCK_CNCT_RECV_FIN;
	sock->acknum += 1;
	transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
}

/*
 * ꥻåȥȤ
 */
STATIC void transReset(SOCKET *sock, TCP_HEADER *tcp)
{
	sock->seqnum = 0;
	sock->acknum = swapInt32(tcp->seqnum) + 1;
	transSegment(sock, TCP_FLG_RST, NULL, 0, NULL, 0, TCP_WIN_SIZE);
}

/*
 * ͥ󤬤ʤΥꥻåȥȤ
 */
STATIC void sendReset(IP_HEADER *ip)
{
	SOCKET sock;
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

	sock.srcport = tcp->dstport;
	sock.dstport = tcp->srcport;
	sock.dstip = ip->srcip;
	transReset(&sock, tcp);
}

//----------------------------- ߼ -------------------------------------------

/*
 *Գ
 * ԤåȤõ
 * return : 0 = ̾ｪλ 1 = 򵯤 -1 = 顼
 */
STATIC int searchSocket(struct mbuf *mbuf)
{
	SOCKET *sock;
	IP_HEADER *ip = getMbufDataPointer(mbuf);
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;
/***************************************************************************
enum{COUNT = 7,};
static int cnt = 0;
***************************************************************************/

	/* ͥ󥯤õ */
	for (sock = waitCnctSc.next; sock != &waitCnctSc; sock = sock->next){
		if ((sock->srcport == tcp->dstport) && (sock->dstport == tcp->srcport) && (sock->dstip == ip->srcip)){
/****************************************************************************************
{
	union{
		uint ui;
		uchar uc[4];
	}srcIp;

	if ((ip->srcip == 0x40019ac) && (swapWord(tcp->srcport) == 20) && (++cnt <= COUNT)){
		srcIp.ui = ip->srcip;
		printk(
			"searchSocket() "
//			"%u.%u.%u.%u "
//			"srcport=%u "
//			"dstport=%u "
			"seq=%x "
			"sockAck=%x\n",
//			srcIp.uc[0],srcIp.uc[1],srcIp.uc[2],srcIp.uc[3],
//			swapWord(tcp->srcport),
//			swapWord(tcp->dstport),
			swapInt32(tcp->seqnum),
			sock->acknum);
	}
}
****************************************************************************************/
			/* åȤϴǤƤ롣 */
			if (sock->stat & SOCK_DISCON_RECV){
				setIpTask(sock, mbuf);
				return 1;
			}

			// ǡ򥽥åȤϤ
			putRecvBufTcp(mbuf, &sock->recvBuf, &sock->lockGate);
			if (sock->waitTask != NULL) {
				activeTaskSoonSignal(getWaitTask(sock->waitTask));
			}
			return 1;
		}
	}

	/* å󥯤õ */
	for (sock = waitLstnSc.next; sock != &waitLstnSc; sock = sock->next){
		if ((sock->srcport == tcp->dstport) && ((sock->srcip == INADDR_ANY) || (sock->srcip == ip->dstip))){
			putRecvBufTcp(mbuf, &sock->recvBuf, &sock->lockGate);
			activeTaskSoonSignal(getWaitTask(sock->listenTask));
			return 1;
		}
	}
	
	// Ĥ餺
	setIpTask(NULL, mbuf);

	return 0;
}

//================================== PUBLIC =============================================

/*
 *Գ
 * return : 0 or task switch=1
 */
int receiveTcp(struct mbuf *mbuf)
{
	IP_HEADER *ip = getMbufDataPointer(mbuf);
	int tcpSize;
	PSEUDO_HEADER pseudoHead;

	/* åγǧ */
	tcpSize = getTcpSize(ip);
	setPseudoHead(ip->dstip, ip->srcip, tcpSize, &pseudoHead);
	if (calcSumM((uint*)&pseudoHead, (uint*)ip->data, sizeof(PSEUDO_HEADER), tcpSize)){
		return 0;
	}
	
	/* γǧ */
	if (TCP_MSS_SIZE < getTcpDataSize(ip)){
		return 0;
	}

	return searchSocket(mbuf);
}

//-----------------------------  -------------------------------------------

//================================== PRIVATE ============================================

/*
 * Ԥ
 * return : error number
 */
STATIC int waitRemoteData(
	SOCKET *sock, 
	const uint i_timeout,
	const struct mbuf *i_mbuf)	// Хåե󥯤ƬӤmbuf
{
	if (i_timeout == 0) {
		return NOERR;
	}

	if (refRecvBuf(&sock->recvBuf) == i_mbuf) {
		uint timeout = i_timeout;

		for (;;) {
			uint remainTime;

			// Ԥ
			sock->waitTask = getCurrentTask();
			remainTime = waitTimeSignal(timeout);

			if (refRecvBuf(&sock->recvBuf) != i_mbuf) {
				break;
			}
			if (remainTime == 0) {
				return -ETIMEDOUT;
			}
			timeout = remainTime;
		}
	}

	return NOERR;
}

/**************************************************************************
 *
 * ԣãХץȥ
 *
 **************************************************************************/

//================================== PRIVATE ============================================

enum{
	SEND_BUF_SIZE = 0x1000,		// Хåե
};

/*
 * Get sequence number.
 * return : sequence number
 */
STATIC INLINE uint32_t getSeqNumber()
{
	/* 4msǣ䤹RFC793ˡ */
	return calcMiliSecondFromClock(rdtsc()) * 4;
}

/*
 * åȤ롣
 */
STATIC void ackFromCloseSock(SOCKET *sock, IP_HEADER *ip)
{
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

	sock->acknum += getTcpDataSize(ip);

	/* FINФ */
	if (swapInt32(tcp->acknum) == sock->seqnum){
		sock->stat |= SOCK_CNCT_RECV_FINACK;
	}

	/* FINα */
	if (tcp->flag & TCP_FLG_FIN){
		if (sock->stat & SOCK_CNCT_RECV_FINACK){
			sendFinAck(sock);
		}
		else{
			if (swapInt32(tcp->acknum) == sock->seqnum){
				sock->stat |= SOCK_CNCT_RECV_FINACK;
				sendFinAck(sock);
			}
		}
	}

	sock->waitTime = sys_time(NULL);
}

//----------------------------- إåץ -------------------------------------------

typedef struct{
	uint32_t timeStamp;
	uint32_t timeEcho;
	uint16_t mss;
	uint8_t  winscal;
}OPTIONS;

/*
 * Get TCP options.
 */
STATIC void getTcpOptions(uchar *c, int size, OPTIONS *optst)
{
	uchar *last;

	last = c + size;
	while (c < last){
		switch (*c){
		case TCP_KIND_MSS:
			optst->mss = *(uint16_t*)(c + 2);
			c += 4;
			break;
		case TCP_KIND_WSC:
			optst->winscal = *(uint8_t*)(c + 2);
			c += 3;
			break;
		case TCP_KIND_TSTMP:
			optst->timeStamp = *(uint32_t*)(c + 2);
			optst->timeEcho = *(uint32_t*)(c + 6);
			c += 10;
			break;
		case TCP_KIND_NOP:
			c += 1;
			break;
		case TCP_KIND_END:
			break;
		default:
			c += *(c + 1);
		}
	}
}

//------------------ ǡ¸ ---------------------

/*
 * ǡ¸
 */
STATIC INLINE void storeBuf(
	SOCKET *sock,
	struct mbuf *mbuf)
{
	AggregateList *aggregate = &sock->recvStore;

	ASSERT(mbuf != NULL);

	aggregate->insertEnd(aggregate, &mbuf->list);
}

/*
 * ǡ¸᤹
 */
STATIC INLINE void storeBackBuf(
	SOCKET *sock,
	struct mbuf *mbuf)
{
	AggregateList *aggregate = &sock->recvStore;

	ASSERT(mbuf != NULL);

	aggregate->insertHead(aggregate, &mbuf->list);
}

/*
 * ¸ǡФ
 * return : ǡ or NULL
 */
STATIC INLINE struct mbuf *getStoreBuf(
	SOCKET *sock)
{
	AggregateList *aggregate = &sock->recvStore;

	return aggregate->getHead(aggregate);
}

/*
 * ¸ǡ¸ߤ뤫
 * return : YES or NO
 */
STATIC INLINE int isExistStoreBuf(
	SOCKET *sock)
{
	AggregateList *aggregate = &sock->recvStore;
	
	return (aggregate->refHead(aggregate) != NULL) ? YES : NO;
}

/*
 * ¸ǡ򤹤٤Ƴ
 */
void releaseStoreBuf(
	SOCKET *sock)
{
	AggregateList *aggregate = &sock->recvStore;
	struct mbuf *mbuf;

	while ((mbuf = aggregate->getHead(aggregate)) != NULL) {
		m_freem(mbuf);
	}
}

//-----------------------------  -------------------------------------------

static int waitCnctLock;	/* Ԥѥåȡ */

/*
 * Add to connection link.
 */
STATIC void addWaitCnctLink(SOCKET *sc)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next = waitCnctSc.next;
		sc->prev = &waitCnctSc;
		waitCnctSc.next->prev = sc;
		waitCnctSc.next = sc;
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
}

/*
 * Add to connection link.
 */
STATIC void addWaitLstnLink(SOCKET *sc)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next=waitLstnSc.next;
		sc->prev=&waitLstnSc;
		waitLstnSc.next->prev=sc;
		waitLstnSc.next=sc;
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
}

/*
 * Remove from connection link.
 */
STATIC void removeWaitLink(SOCKET *sc)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next->prev = sc->prev;
		sc->prev->next = sc->next;
		sc->next = sc->prev = sc;
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
}

/*
 * å󥽥åȤݡȤõ
 * return : socket address or NULL
 */
STATIC SOCKET *searchListenSocket(in_port_t srcport)
{
	SOCKET *sc,*search_sc = NULL;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		for (sc = waitLstnSc.next; sc != &waitLstnSc; sc = sc->next)
			if (sc->srcport == srcport)
			{
				search_sc = sc;
				break;
			}
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
	
	return search_sc;
}

enum{
	ACK_EQ_SEQ_EQ = 1,	// ֹ = ֹ桢ֹ = ֹ
	ACK_LW_SEQ_EQ,		// ֹ < ֹ桢ֹ = ֹ
	ACK_EQ_SEQ_LW,		// ֹ = ֹ桢ֹ < ֹ
	ACK_EQ_SEQ_GR,		// ֹ = ֹ桢ֹ > ֹ
};

/*
 * Ȥγǧ
 */
STATIC int isAck(
	TCP_HEADER *tcp, 		// TCP HEADER
	uint32_t sent,			// ֹ
	uint32_t received)		// ֹ
{
	static uchar returnValue[3][3] = {
		{ACK_EQ_SEQ_EQ,	ACK_EQ_SEQ_LW,	ACK_EQ_SEQ_GR},
		{ACK_LW_SEQ_EQ,	ERR,			ERR},
		{ERR,			ERR,			ERR},
	};
	int isAck, isSeq;
	uint32_t tcpAck = swapInt32(tcp->acknum);
	uint32_t tcpSeq = swapInt32(tcp->seqnum);


	/* ե饰ӡ */
	if ((tcp->flag & TCP_FLG_ACK) == 0)
		return ERR;

	/* ACKͤSEQͤ */
	if (tcpAck == sent){
		isAck = 0;
	}
	else if (tcpAck < sent){
		isAck = 1;
	}
	else{
		isAck = 2;
	}
	if (tcpSeq == received){
		isSeq = 0;
	}
	else if (tcpSeq < received){
		isSeq = 1;
	}
	else{
		isSeq = 2;
	}

	return returnValue[isAck][isSeq];
}

STATIC void initReceive()
{
	waitCnctSc.next = waitCnctSc.prev = &waitCnctSc;
	waitLstnSc.next = waitLstnSc.prev = &waitLstnSc;
}

/**************************************************************************
 *
 * å󥹥å
 *
 **************************************************************************/

enum{
	LISTEN_WAIT_MAX = 5,	/* listenԤॢȻ֡á */
};

//================================== PRIVATE ============================================

/*
 * åȤ
 * return :  error number
 */
STATIC int makeNewSocket(SOCKET *sock, struct mbuf *mbuf, SOCKET **o_newSock)
{
	SOCKET *newSock;
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	OPTIONS optst;
	int error;

	error = makeSocket(sock->sockType, IPPROTO_TCP, &newSock);
	if (error < 0){
		return error;
	}

	ip = getMbufDataPointer(mbuf);
	tcp = (TCP_HEADER*)ip->data;

	/* ץ(MSSΤ߼) */
	optst.mss = TCP_MSS_SIZE;
	getTcpOptions(tcp->option, getTcpDataSize(ip), &optst);

	/*
	 * åȤꡣ
	 */
	memcpy(&newSock->fs, &sock->fs, sizeof(SOCKET) - OFFSETOF(SOCKET, fs));
	newSock->sc				= newSock;
	newSock->sockType		= sock->sockType;
	newSock->stat			= SOCK_LINK_ON;
	newSock->sinf			= sock->sinf;
	newSock->srcport		= tcp->dstport;
	newSock->dstport		= tcp->srcport;
	newSock->dstip			= ip->srcip;
	newSock->srcip			= sock->srcip;
	initRecvBuf(&newSock->recvBuf);
	AggregateListConstructor(&newSock->recvStore);
	newSock->waitTask		= sock->waitTask;
	newSock->seqnum			= getSeqNumber();
	newSock->acknum			= swapInt32(tcp->seqnum) + 1;
	newSock->next			= NULL;
	newSock->prev			= NULL;
	newSock->waitNext		= NULL;
	newSock->waitTime		= 0;
	newSock->mss			= optst.mss;
	newSock->window			= swapWord(tcp->wndsize);
	newSock->listenTask	= NULL;
	newSock->newsc_next		= NULL;
//	newSock->sockOpt		= NULL;
//	newSock->ipOpt			= NULL;
	newSock->lockGate		= 0;

	*o_newSock = newSock;
	
	return NOERR;
}

/*
 * åȤå󥽥åȤ³
 * ֿॢȤΥåȤ
 * parameters : å󥽥å,å
 */
STATIC void addNewSocket(SOCKET *sock, SOCKET *newSock)
{
	enum{DEL_NUM = 3};
	SOCKET **p;
	SOCKET *delSock[DEL_NUM];
	int delNum = 0;
	uint currentTime = sys_time(NULL);
	int eflag;
	int i;
	
	// åȤǸ³
	newSock->waitTime = currentTime;
	newSock->newsc_next = NULL;
	eflag = enterCli();
	enter_spinlock(&sock->lockGate);
	{
		for (p = &sock->newsc_next;; p = &(*p)->newsc_next) {
			if (*p == NULL){
				*p = newSock;
				break;
			}

			// ֿॢȤΥåȤ
			if ((LISTEN_WAIT_MAX < currentTime - (*p)->waitTime) && (refRecvBuf(&sock->recvBuf) != NULL)) {
				if (delNum < DEL_NUM) {
					delSock[delNum] = *p;
					*p = (*p)->newsc_next;
					++delNum;
					if (*p == NULL) {
						*p = newSock;
						break;
					}
				}
			}
		}
	}
	exit_spinlock(&sock->lockGate);
	exitCli(eflag);
	
	// ֿॢȥåȤ򥳥ͥ󥯤
	for (i = 0; i < delNum; ++i){
		removeWaitLink(delSock[i]);
		kfree(delSock[i]);
	}
}

/*
 * listenԤ³
 */
STATIC void listenConnect(SOCKET *sock)
{
	struct mbuf *mbuf = getRecvBuf(&sock->recvBuf, &sock->lockGate);
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	MSS mss;
	SOCKET *newSock;
	
	ASSERT(mbuf != NULL);

	ip = getMbufDataPointer(mbuf);
	tcp = (TCP_HEADER*)ip->data;

	/* ³׵᥻Ȥγǧ */
	if ((tcp->flag & TCP_FLG_SYN) && (tcp->acknum == 0)){
		;
	}
	else{
		sendReset(ip);
		m_freem(mbuf);
		return;
	}

	/* åȤκ */
	if (makeNewSocket(sock, mbuf, &newSock) != NOERR){
		sendReset(ip);
		m_freem(mbuf);
		return;
	}
	m_freem(mbuf);
	
	/* ͥ󥯤³ */
	addWaitCnctLink(newSock);

	/* إåץꡣ */
	mss.kind = TCP_KIND_MSS;
	mss.len = 4;
	mss.mss = swapWord(TCP_MSS_SIZE);

	/* SYNȤ */
	transSegment(newSock, TCP_FLG_SYN|TCP_FLG_ACK, &mss, sizeof(mss), NULL, 0, TCP_WIN_SIZE);

	/* å󥽥åȤ³ */
	addNewSocket(sock, newSock);
}

/*
 * åɽλ롼
 */
STATIC void listenRoutin(SOCKET *sock)
{
	sock->listenTask = getCurrentTask();
	
	/* ͥγ */
	addWaitLstnLink(sock);

	for (;;) {
		if (refRecvBuf(&sock->recvBuf) == NULL) {
			waitTaskSignal();

			/* å֤ǤʤХåɽλ */
			if ((sock->stat & SOCK_LISTEN) == 0) {
				break;
			}
		}
		else {
			listenConnect(sock);
		}
	}

	/*  */
	kfree(sock);
	sys_exit(0);
}

/*
 * ³ѿåȤƥåȤå󥽥åȤ
 * ǽԤϻ֤
 * return : ³ѿå or NULL
 */
STATIC SOCKET *getFromListenSocket(SOCKET *sc)
{
	SOCKET **p,*rest = NULL;
	int eflag;
	
	eflag = enterCli();
	enter_spinlock(&sc->lockGate);
	{
		for (p = &sc->newsc_next; *p != NULL; p = &(*p)->newsc_next){
			if (refRecvBuf(&(*p)->recvBuf) != NULL){
				rest = *p;
				*p = rest->newsc_next;
				rest->waitTime = 0;
				break;
			}
		}
	}
	exit_spinlock(&sc->lockGate);
	exitCli(eflag);
	
	return rest;
}

/*
 * å󥽥åȥ󥯤餹٤ƤΥåȤ
 */
STATIC void releaseListenSocket(SOCKET *sc)
{
	SOCKET *p,*q;
	
	for (p = sc->newsc_next; p != NULL; p = q){
		q = p->newsc_next;
		releaseRecvBuf(&p->recvBuf);
		kfree(p);
	}
}

/*
 * å󥹥åɤκ
 * parameters : socket,file discripter
 * return     : 0 or error number
 */
STATIC int makeListenThread(SOCKET *sock, int fd)
{
	/* listenѥץư */
	switch(forkKernelThread()) {
	case -1:
		return -ENOBUFS;
	case 0:
		// СåȤǤ褦
		// եǥץȥȤ򸺤餹
		sys_close(fd);

		listenRoutin(sock);
		// Not reached.
	}
	
	/* listenץưޤԤ */
	while (sock->next == sock){
		wait_task();
	}

	return 0;
}

/*
 * å󥹥åɤνλ
 * parameters : socket
 */
STATIC void endListenThread(SOCKET *sock)
{
	sock->stat &= ~SOCK_LISTEN;
	activeTaskSignal(getWaitTask(sock->listenTask));
}

/***************************************************************************************************
 * 
 * TODO : FINΥֹꡢХåե
 * sock->seqnum					ϥֹ桡ѤμΥֹ
 * sock->remoteAck				ѥֹ
 * sock->window					ǽơ˹
 * sock->acknum					ѥǡֹ桡ʼѤμΥֹ
****************************************************************************************************/

/*
 * Хåե
 * return : error number
 */
STATIC void sendFromBuf(
	SOCKET *sock, 
	const size_t i_start,		// Хåեϥֹ
	const size_t sendLen,		// ǡĹ
	const uint8_t transFlag)	// ե饰
{
//	uint32_t maxSeqnum = sock->seqnum;

	sock->seqnum = i_start;
	transSegment(sock, transFlag, NULL, 0, getSendBufPointer(i_start, &sock->sendBuf), sendLen, SEND_BUF_SIZE);

	/*
	 * Υǡ׵ᤷƤ뤳Ȥ⤢Τ
	 * ֹϵƤ
	 */
//	sock->seqnum = (maxSeqnum < i_start + sendLen) ? i_start + sendLen : maxSeqnum;
	sock->seqnum += sendLen;

	if (transFlag & TCP_FLG_FIN) {
		sock->seqnum += 1;
	}
}

/*
 * mssޤwindowޤǤ
 */
STATIC void sendSoonFromBuf(
	SOCKET *sock,
	const size_t i_start)		// ϥֹ
{
	size_t sendMaxSize = (sock->window < sock->mss) ? sock->window : sock->mss;		// 祵
	size_t saveSendSize = getSendBufSendSize(i_start, &sock->sendBuf);
	size_t sendSize = (sendMaxSize <= saveSendSize) ? sendMaxSize : saveSendSize;

	if (0 < sendSize) {
		sendFromBuf(sock, i_start, sendSize, TCP_FLG_PSH | TCP_FLG_ACK);
		sock->window -= sendSize;
	}
}

/*
 * Ʊ
 */
STATIC INLINE void reply(
	SOCKET *sock,
	const int recvFlag)		// ե饰
{
	if (recvFlag & TCP_FLG_FIN) {
		sendFinAck(sock);
	}
	else {
		transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
	}
}

/*
 * 羮
 * return :  = 0,  = 1,  = 2
 */
STATIC INLINE int condition(
	int64_t src,	// 
	int64_t dst)	// Ӹ
{
	if (dst < src) {
		return 0;
	}
	else if (dst == src) {
		return 1;
	}
	else {
		return 2;
	}
}

/*
 * ǡ롣
 * Ԥ֤ꤵƤʤмǡ̵Ƚλ
 * return : ǡХåե or NULL
 */
STATIC struct mbuf *getFromRemote(
	SOCKET *sock,
	const uint timeout,		// Ԥ֡á
	int *o_error)			// error number
{
	enum {
		WATI_COUNT_MAX = 20		// Ԥ
	};
	struct mbuf *mbuf;
	struct mbuf *cmpMbuf = NULL;
//	int waitCnt;

//	for (waitCnt = 0; waitCnt < WATI_COUNT_MAX; ++waitCnt) {
	for (;;) {
		IP_HEADER *ip;
		TCP_HEADER *tcp;
		uint32_t tcpAck;
		uint32_t tcpSeq;

		// mbufա󥯥ХåեƬȰפԤΤ̾NULL
		*o_error = waitRemoteData(sock, timeout, cmpMbuf);
		if (*o_error != NOERR) {
			return NULL;
		}
		cmpMbuf = NULL;

		mbuf = getRecvBuf(&sock->recvBuf, &sock->lockGate);
		if (mbuf == NULL) {
			// Ԥ֤ꤵƤкԤ
			if (0 < timeout) {
				continue;
			}
			else {
				*o_error = NOERR;
				return NULL;
			}
		}
		ip = getMbufDataPointer(mbuf);
		tcp = (TCP_HEADER*) ip->data;
		tcpAck = swapInt32(tcp->acknum);
		tcpSeq = swapInt32(tcp->seqnum);

		// ꥻåȥե饰ʤ饳ͥ
		if (tcp->flag & TCP_FLG_RST) {
			m_freem(mbuf);
			*o_error = -ECONNRESET;
			return NULL;
		}

		/*
		 * ֹγǧ
		 * 		
		 * 				
		 * 																								
		 * 										ľ줿ǽ				¸Ʊ
		 * 											ǡ¸λ							
		 *											ǡ̵кƼԤ
		 * 											ʬαϤƤʤǽ					Ʊ
		 *																								ASSERT
		 *											ľ줿ǽ				¸Ʊ
		 * 											ǡ¸λ							
		 *											ǡ̵кƼԤ
		 * 											ʬαϤƤʤǽ					Ʊ
		 * 											ʬϤƤʤǽ롢				
		 *																								
		 *
		 * ǥǡ̵ס֤סֺƱסֺפ³ޤϥͥåȥ۾ȽǤ
		 * ͥǡ
		 */
		{
			enum {
				DO_OK,			// 
				DO_REPLY,		// Ʊ
				DO_RESEND,		// 
				DO_BACK,		// ¸
				DO_RESET,		// Ѻ
				DO_ERROR,		// 
				DO_ASSERT,		// ᥽åɥ顼
				
				DO_RESEND_COUNT = 2, // ׵ᥫ
			};
			static const int nextDo[3][3][3] = {
				{			//  > 
					{			//  > 
						DO_ERROR, DO_RESET, DO_ERROR},
					{			//  = 
						DO_BACK,		//  > 
						DO_OK,			//  = 
						DO_REPLY},		//  < 
					{			//  < 
						DO_BACK,		//  < 
						DO_OK,			//  = 
						DO_REPLY}		//  > 
				},
				{			//  = 
					{			//  > 
						DO_ASSERT, DO_ASSERT, DO_ASSERT},
					{			//  = 
						DO_BACK,		//  > 
						DO_OK,			//  = 
						DO_REPLY},		//  < 
					{			//  < 
						DO_RESEND, DO_RESEND, DO_RESEND}
				},
				{			//  < 
					{	DO_ERROR, DO_ERROR, DO_ERROR},
					{	DO_ERROR, DO_ERROR, DO_ERROR},
					{	DO_ERROR, DO_ERROR, DO_ERROR}
				}
			};

			switch (nextDo
				[condition(tcpAck, sock->remoteAck)]
				[condition(tcpAck, sock->seqnum)]
				[condition(tcpSeq, sock->acknum)]) {
			case DO_OK:
				sock->remoteAck = tcpAck;
				sock->window = swapWord(tcp->wndsize);
				sock->acknum += getTcpDataSize(ip);
				sock->resend = 0;

				if (0 < getTcpDataSize(ip)) {
					// ΥǡƤʤб
					if (refRecvBuf(&sock->recvBuf) == NULL) {
						transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
					}
				}
				if (tcp->flag & TCP_FLG_FIN) {
					sendFinAck(sock);
				}

				*o_error = NOERR;
				return mbuf;
			case DO_REPLY:
				m_freem(mbuf);
				
				// Ʊ
				reply(sock, tcp->flag);
				
				sock->remoteAck = tcpAck;
				break;
			case DO_RESEND:
				m_freem(mbuf);
				++sock->resend;

				// 
				if (DO_RESEND_COUNT <= sock->resend) {
					sock->window = swapWord(tcp->wndsize);
					sendSoonFromBuf(sock, tcpAck);
				}
				break;
			case DO_BACK:
				// Хåե󥯤᤹
				putRecvBufTcp(mbuf, &sock->recvBuf, &sock->lockGate);
				cmpMbuf = mbuf;

				// Ʊ
				reply(sock, tcp->flag);

				sock->remoteAck = tcpAck;
				break;
			case DO_RESET:
				m_freem(mbuf);
				sock->remoteAck = tcpAck;
				sock->seqnum = tcpAck;
				sock->resend = 0;
				break;
			case DO_ERROR:
				m_freem(mbuf);
				break;
			case DO_ASSERT:
				// 
			default:
				ASSERT(0);
			}
		}
	}

	*o_error = -ECONNRESET;
	return NULL;
}

/*
 * ǡ
 * return :  or error number
 */
STATIC int sendData(
	SOCKET *sock,			// å
	const char *msg,		// ǡ
	const size_t i_size)	// 
{
	enum {
		WAIT_TIME = 5000	// Ԥ ms
	};
	const char *sendBuf = msg;
	size_t restSize;			// ̤¸

	for (restSize = i_size; 0 < restSize;) {
		// 
		{
			struct mbuf *mbuf;
			IP_HEADER *ip;
			int error;

			mbuf = getFromRemote(sock, 0, &error);
			if (error != NOERR) {
				return error;
			}
			if (mbuf != NULL) {
				ip = getMbufDataPointer(mbuf);
				if (0 < getTcpDataSize(ip)) {
					storeBuf(sock, mbuf);
				}
				else {
					m_freem(mbuf);
				}
			}
			while ((getSendBufSpaceSize(sock->remoteAck, &sock->sendBuf) == 0) || (sock->window == 0)) {
				mbuf = getFromRemote(sock, WAIT_TIME, &error);
				if (error != NOERR) {
					return error;
				}
				if (mbuf != NULL) {
					ip = getMbufDataPointer(mbuf);
					if (0 < getTcpDataSize(ip)) {
						storeBuf(sock, mbuf);
					}
					else {
						m_freem(mbuf);
					}
				}
			}
		}

		// ǡ¸
		{
			size_t spaceSize = getSendBufSpaceSize(sock->remoteAck, &sock->sendBuf);
			size_t saveSize = (spaceSize < restSize) ? spaceSize : restSize;

			writeSendBuf(saveSize, sendBuf, &sock->sendBuf);
			restSize -= saveSize;
			sendBuf += saveSize;
		}

		// ̤ǡޤϣˤʤä
		{
			size_t saveSendSize = getSendBufSendSize(sock->seqnum, &sock->sendBuf);

			if (saveSendSize == getSendBufSaveSize(sock->seqnum, &sock->sendBuf)) {
				size_t sendMaxSize = (sock->window < sock->mss) ? sock->window : sock->mss;
				if (sendMaxSize <= saveSendSize) {
					sendFromBuf(sock, sock->seqnum, sendMaxSize, TCP_FLG_PSH | TCP_FLG_ACK);
					sock->window -= sendMaxSize;
				}
			}
			else {
				sendSoonFromBuf(sock, sock->seqnum);
			}
		}
	}

	return i_size;
}

/*
 * ǡ
 * return : ǡ or NULL
 */
STATIC struct mbuf *getRecvData(
	SOCKET *sock,
	int *o_error)
{
	enum {
		WAIT_TIME = 5000,	// Ԥ ms
		WAIT_MAX = 3,		// Ԥ
	};
	struct mbuf *mbuf;
	int waitCnt;

	// Хåե˻ĤäƤǡ
	while (0 < getSendBufSaveSize(sock->seqnum, &sock->sendBuf)) {
		sendSoonFromBuf(sock, sock->seqnum);
	}
	
	// ¸ǡ
	mbuf = getStoreBuf(sock);
	if (mbuf != NULL) {
		*o_error = NOERR;
		return mbuf;
	}

	// ǡޤԤ
	for (waitCnt = 0; waitCnt < WAIT_MAX; ++waitCnt) {
		IP_HEADER *ip;
		TCP_HEADER *tcp;

		mbuf = getFromRemote(sock, WAIT_TIME, o_error);
		if (*o_error != NOERR) {
			return NULL;
		}

		ip = getMbufDataPointer(mbuf);
		tcp = (TCP_HEADER*)ip->data;
		if (0 < getTcpDataSize(ip)) {
			*o_error = NOERR;
			return mbuf;
		}
		else if (tcp->flag & TCP_FLG_FIN) {
			*o_error = NOERR;
			return mbuf;
		}
		else {
			m_freem(mbuf);
		}
	}
	
	*o_error = -ETIMEDOUT;

	return NULL;
}

/***************************************************************************************************
 *
 * ͥ
 *
 ***************************************************************************************************/

//================================== PRIVATE ============================================

static SOCKET *finSocket;		/* 楽åȥ󥯡 */

/*
 * ǳϥåȤ󥯤³롣
 * parameters : socket
 */
STATIC void addCloseSocket(SOCKET *sock)
{
	sock->waitNext = finSocket;
	finSocket = sock;
	sock->waitTime = sys_time(NULL);
}

/*
 * ǳϥåȤǡॢȤõ
 */
STATIC void delCloseTimeout()
{
	enum{
		CNCT_CLOSE_WAIT_TIME = 5,	/* ǽॢȻs */
	};
	uint time;

	time = sys_time(NULL);
	while (finSocket != NULL) {
		if (CNCT_CLOSE_WAIT_TIME <= time - finSocket->waitTime) {
			SOCKET *delSock = finSocket;

			finSocket = finSocket->waitNext;

			// åȳ
			removeWaitLink(delSock);
			freeSendBuf(&delSock->sendBuf);
			releaseRecvBuf(&delSock->recvBuf);
			releaseStoreBuf(delSock);
			kfree(delSock);
		}
		else {
			break;
		}
	}
}

/*
 * ͥ
 * parameters :
 */
STATIC void disconnect(SOCKET *sock)
{
	// Хåե˻ĤäƤǡ
	while (0 < getSendBufSaveSize(sock->seqnum, &sock->sendBuf)) {
		sendSoonFromBuf(sock, sock->seqnum);
	}

	/* FINȤ */
	transSegment(sock, TCP_FLG_FIN | TCP_FLG_ACK, NULL, 0, NULL, 0, SEND_BUF_SIZE);
	sock->seqnum += 1;
	sock->stat |= SOCK_CNCT_TRANS_FIN;

	delCloseTimeout();		/* ॢȥåȤ롣 */
	addCloseSocket(sock);	/* åȥ󥯤³롣 */
}

/*
 * ͥΩ
 * parameters : socket
 * return : error number
 */
STATIC int connectTo(SOCKET *sock)
{
	enum {
		WAIT_CONNECT = 7,			// ³ॢ
	};
	struct mbuf *mbuf;
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	OPTIONS optst;
	MSS mss;
	int error;

	/* ͥ󥯤³ */
	addWaitCnctLink(sock);
	
	/* إåץꡣ */
	mss.kind=TCP_KIND_MSS;
	mss.len= 4;
	mss.mss= swapWord(TCP_MSS_SIZE);

	/* SYNȤ */
	transSegment(sock, TCP_FLG_SYN, &mss, sizeof(mss), NULL, 0, TCP_WIN_SIZE);
	sock->seqnum += 1;

	/* Ȥμ */
	error = waitRemoteData(sock, WAIT_CONNECT, NULL);
	if (error != NOERR) {
		goto ERR;
	}
	mbuf = getRecvBuf(&sock->recvBuf, &sock->lockGate);

	ASSERT(mbuf != NULL);

	ip = getMbufDataPointer(mbuf);
	tcp = (TCP_HEADER*)ip->data;
	if (((tcp->flag & TCP_FLG_SYN) == 0) || (isAck(tcp, sock->seqnum, 0) != ACK_EQ_SEQ_GR)) {
		if (tcp->flag & TCP_FLG_RST){
			return -ECONNREFUSED;
		}
		transReset(sock, tcp);
		error = -EAGAIN;
		goto ERR;
	}

	/* ץγǧ(MSSΤ߼) */
	optst.mss = TCP_MSS_SIZE;
	getTcpOptions(tcp->option, getTcpDataSize(ip), &optst);
	sock->mss = optst.mss;

	/* ǧ */
	sock->window = swapWord(tcp->wndsize);
	sock->acknum = swapInt32(tcp->seqnum) + 1;
	sock->remoteAck = swapInt32(tcp->acknum);
	sock->stat |= SOCK_CNCT_ON;
	transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);

	error = allocSendBuf(sock->seqnum, &sock->sendBuf);
	if (error != NOERR) {
		goto ERR;
	}

	m_freem(mbuf);

	return NOERR;
ERR:
	/* ͥ󥯤 */
	removeWaitLink(sock);
	releaseRecvBuf(&sock->recvBuf);
	releaseStoreBuf(sock);

	return error;
}

/**************************************************************************
 *
 * TCPӥؿ
 *
 **************************************************************************/

//================================== PUBLIC =============================================

/*
 * IPåɤƤӽФ
 */
void doTcpService(SOCKET *sock, IP_HEADER *ip)
{
	if (sock == NULL){
		sendReset(ip);
	}
	else{
		ackFromCloseSock(sock, ip);
	}
}

/*
 * åȤ򥳥ͥ󤫤鳫롣
 * return ; 0 or 1(åȤƤϤʤ)
 */
int releaseSockFromConect(SOCKET *sock)
{
	enum {
		WAIT_TIME = 5000		// Ԥ ms
	};

	if (sock->stat & SOCK_CNCT_ON) {
		int isFin = 0;

		// Хåե˻ĤäƤǡ
		while (0 < getSendBufSaveSize(sock->seqnum, &sock->sendBuf)) {
			struct mbuf *mbuf;
			int error;

			sendSoonFromBuf(sock, sock->seqnum);
			mbuf = getFromRemote(sock, WAIT_TIME, &error);
			if (mbuf != NULL) {
				m_freem(mbuf);
			}
		}

		sock->stat |= SOCK_DISCON_RECV | SOCK_DISCON_SEND;

		// ѥåȤ˱
		while (refRecvBuf(&sock->recvBuf) != NULL){
			struct mbuf *mbuf = getRecvBuf(&sock->recvBuf, &sock->lockGate);
			IP_HEADER *ip;
			TCP_HEADER *tcp;

			ASSERT(mbuf != NULL);
			ip = getMbufDataPointer(mbuf);
			tcp = (TCP_HEADER*)ip->data;
			sock->acknum += getTcpDataSize(ip);
			isFin = tcp->flag & TCP_FLG_FIN;
			m_freem(mbuf);
		}
		if (isFin && ((sock->stat & SOCK_CNCT_RECV_FIN) == 0)) {
			sendFinAck(sock);
		}
		
		if ((sock->stat & SOCK_CNCT_TRANS_FIN) == 0) {
			disconnect(sock);
		}

		return 1;
	}

	if (sock->stat & SOCK_LISTEN){
		removeWaitLink(sock);
		releaseRecvBuf(&sock->recvBuf);
		releaseListenSocket(sock);
		releaseStoreBuf(sock);
		endListenThread(sock);

		return 1;
	}

	if (sock->stat & SOCK_LINK_ON){
		removeWaitLink(sock);
		releaseRecvBuf(&sock->recvBuf);
		releaseStoreBuf(sock);
	}

	return 0;
}

/**************************************************************************
 *
 * åȥӥؿ
 *
 **************************************************************************/

STATIC int send(SOCKET *sock, void *msg, const size_t i_size, int flags, struct sockaddr *to)
{
	if ((sock->stat & SOCK_CNCT_ON) == 0){
		return -ENOTCONN;
	}
	
	return sendData(sock, msg, i_size);
}

STATIC int recv(SOCKET *sock, struct msghdr *msg)
{
	enum{
		RECEIVE_TIMEOUT = 5,	// ॢ
	};
	char *data;
	int allSize;
	int tcpDataSize;
	int dataSize;
	int cpSize;
	int iovIndex;
	int iovOffset;
	int error;

	// ͥγǧ
	if ((sock->stat & SOCK_CNCT_ON) == 0){
		return -ENOTCONN;
	}

	// ǤFINȤƤ뤫
	if (sock->stat & SOCK_CNCT_RECV_FIN) {
		if (isExistStoreBuf(sock) == NO) {
			return 0;
		}
	}

	allSize = iovIndex = iovOffset = 0;
	for (;;){
		struct mbuf *mbuf;
		IP_HEADER *ip;
		TCP_HEADER *tcp;
		
		// ǡμ
		mbuf = getRecvData(sock, &error);
		if (error != NOERR) {
			return error;
		}
		ip = getMbufDataPointer(mbuf);
		tcp = (TCP_HEADER*)ip->data;

		// 桼Хåե˥ԡ
		tcpDataSize = getTcpDataSize(ip);
		dataSize = tcpDataSize - mbuf->mh_offset;
		data = (char*) tcp + getTcpSize(ip) - dataSize;
		cpSize = copyIovec(data, dataSize, msg->msg_iov, msg->msg_iovlen, &iovIndex, &iovOffset);
		allSize += cpSize;
		mbuf->mh_offset += cpSize;

		// 桼ХåեǡϤƤʤȤϤꤨʤ
		ASSERT((iovIndex == msg->msg_iovlen) || (mbuf->mh_offset == tcpDataSize));

		// 뤫
		if (tcp->flag & TCP_FLG_FIN) {		// FINȤ
			if (allSize == 0) {				// ǡ̵
				m_freem(mbuf);
			}
			else {
				// 桼EOFǡ֤˰ö¸ƼθƤӽФԤġ
				storeBackBuf(sock, mbuf);
			}
		}
		else if (mbuf->mh_offset == tcpDataSize) {		// ǡ桼Ϥ
			m_freem(mbuf);
		}
		else {
			// IPХåե¸
			storeBackBuf(sock, mbuf);
		}

		// λ뤫
		if (iovIndex == msg->msg_iovlen) {				// 桼ХåեäѤ
			break;
		}
		else if (refRecvBuf(&sock->recvBuf) == NULL) {	// Υǡʤ
			break;
		}
	}

	return allSize;
}

STATIC int shutdown(SOCKET *sock, int flag)
{
	if ((sock->stat & SOCK_CNCT_ON) == 0) {
		return -ENOTCONN;
	}

	switch(flag) {
	case SHUT_RD:
		sock->stat |= SOCK_DISCON_RECV;
		break;
	case SHUT_WR:
		sock->stat |= SOCK_DISCON_SEND;
		break;
	case SHUT_RDWR:
		sock->stat |= SOCK_DISCON_RECV | SOCK_DISCON_SEND;
		break;
	}

	if ((sock->stat & (SOCK_DISCON_RECV | SOCK_DISCON_SEND)) && (sock->stat & SOCK_CNCT_TRANS_FIN) == 0) {
		disconnect(sock);
	}

	return 0;
}

STATIC int connect(SOCKET *sock)
{
	if (sock->stat & SOCK_CNCT_ON)
		return -EISCONN;

	/* Set connect structure. */
	sock->next = sock->prev = sock;
	sock->stat = SOCK_LINK_ON;
	sock->seqnum = getSeqNumber();
	sock->acknum = 0;

	/* ͥγΩ */
	return connectTo(sock);
}

STATIC int listen(SOCKET *sock,int fd)
{
	if (sock->stat & SOCK_LISTEN)
		return -EADDRINUSE;

	if (searchListenSocket(sock->srcport) != NULL)
		return -EADDRINUSE;

	sock->next = sock->prev = sock;
	sock->stat = SOCK_LINK_ON | SOCK_LISTEN;
	sock->waitTask = getCurrentTask();
	
	/* å󥹥åɤγ */
	makeListenThread(sock, fd);
	
	return 0;
}

STATIC int accept(SOCKET *sock,SOCKET **o_newSock,uint32_t *o_srcIp)
{
	SOCKET *newSock;
	struct mbuf *mbuf;
	int error;
	
	for (;;){
		for (;;){
			if (isSigint(TaskGetTaskSignal(getCurrentTask())) == YES){
				return -EINTR;
			}

			newSock = getFromListenSocket(sock);
			if (newSock != NULL){
				break;
			}

			/* Ԥ */
			waitTaskSignal();
			newSock = getFromListenSocket(sock);
			if (newSock != NULL){
				break;
			}
		}
		mbuf = refRecvBuf(&newSock->recvBuf);

		for (;;){
			IP_HEADER *ip = getMbufDataPointer(mbuf);
			TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

			/* γǧ */
			switch (isAck(tcp, newSock->seqnum, newSock->acknum)){
				case -1:
					/*  */
				case ACK_LW_SEQ_EQ:
					detachRecvBuf(mbuf, &newSock->recvBuf, &newSock->lockGate);
					m_freem(mbuf);
					mbuf = refRecvBuf(&newSock->recvBuf);
					if (mbuf == NULL){
						kfree(newSock);
						goto NEXT;
					}
					break;
				case ACK_EQ_SEQ_GR:
					// ȤΥֹ椬礭ȤϡΥȤ夷ǽ
					goto NEXT;
					break;
				default:
					/* ׵ */
					if (tcp->flag & TCP_FLG_RST){
						releaseRecvBuf(&newSock->recvBuf);
						releaseStoreBuf(newSock);
						kfree(newSock);
						goto NEXT;
					}
					else{
						newSock->remoteAck = swapInt32(tcp->acknum);
						goto OUT;
					}
			}
		}
NEXT:	;
	}
OUT:
	detachRecvBuf(mbuf, &newSock->recvBuf, &newSock->lockGate);
	m_freem(mbuf);

	newSock->stat |= SOCK_CNCT_ON;
	newSock->seqnum += 1;

	error = allocSendBuf(newSock->seqnum, &newSock->sendBuf);
	if (error < 0) {
		return error;
	}

	*o_srcIp = newSock->dstip;
	*o_newSock = newSock;

	return 0;
}

STATIC int poll(SOCKET *sock, int events)
{
	// ͥγǧ 
	if (((sock->stat & SOCK_CNCT_ON) == 0) && ((sock->stat & SOCK_LISTEN) == 0)) {
		return -ENOTCONN;
	}

	// Хåե˻ĤäƤǡ
	while (0 < getSendBufSaveSize(sock->seqnum, &sock->sendBuf)) {
		sendSoonFromBuf(sock, sock->seqnum);
	}

	if (events & (POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI)) {
		if (sock->stat & SOCK_LISTEN) {
			if (sock->newsc_next != NULL){
				return POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI;
			}
		}
		else {
			if (refRecvBuf(&sock->recvBuf) != NULL) {
				return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
			}
		}
		sock->waitTask = getCurrentTask();

		return 0;
	}
	if (events & (POLLOUT | POLLWRNORM | POLLWRBAND)) {
			return POLLOUT | POLLWRNORM | POLLWRBAND;
	}

	return 0;
}

/**************************************************************************
 *
 * 
 *
 **************************************************************************/

static SOCKET_INFO sockInfo = {send, recv, connect, shutdown, listen, accept, poll};

//================================== PUBLIC =============================================

int initTcp()
{
	registSocket(&sockInfo,IPPROTO_TCP);
	initReceive();

	return 0;
}

/***************************************************************************/
void test_tcp()
{
	printk("waitNext=%x\n",*(uint*)0x16be64);
}
/****************************************************************************/
