/*
 * Copyright (c) 1997, Stefan Esser <se@freebsd.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice unmodified, this list of conditions, and the following
 *    disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/pci/pcivar.h,v 1.41.2.2 2002/01/10 12:08:22 mdodd Exp $
 *
 */


#ifndef PCI_PCIVAR_H
#define PCI_PCIVAR_H


#include <sys/types.h>
#include <sys/queue.h>
#include <sys/bus.h>
#include <sys/bus_private.h>
#include <kern/dev_method.h>


typedef u_int32_t pci_addr_t;

#define PCI_MAXMAPS_0	6	/* max. no. of memory/port maps */
#define PCI_MAXMAPS_1	2	/* max. no. of maps for PCI to PCI bridge */
#define PCI_MAXMAPS_2	1	/* max. no. of maps for CardBus bridge */

#define PCI_PPBIOBASE(h,l)   ((((h)<<16) + ((l)<<8)) & ~0xfff)
#define PCI_PPBIOLIMIT(h,l)  ((((h)<<16) + ((l)<<8)) | 0xfff)
#define PCI_PPBMEMBASE(h,l)  (((l)<<16) & ~0xfffff)
#define PCI_PPBMEMLIMIT(h,l) (((l)<<16) | 0xfffff)

/* map register information */
#define PCI_MAPMEM	0x01	/* memory map */
#define PCI_MAPMEMP	0x02	/* prefetchable memory map */
#define PCI_MAPPORT	0x04	/* port map */

enum pci_device_ivars {
	PCI_IVAR_SUBVENDOR,
	PCI_IVAR_SUBDEVICE,
	PCI_IVAR_VENDOR,
	PCI_IVAR_DEVICE,
	PCI_IVAR_DEVID,
	PCI_IVAR_CLASS,
	PCI_IVAR_SUBCLASS,
	PCI_IVAR_PROGIF,
	PCI_IVAR_REVID,
	PCI_IVAR_INTPIN,
	PCI_IVAR_IRQ,
	PCI_IVAR_BUS,
	PCI_IVAR_SLOT,
	PCI_IVAR_FUNCTION,
	PCI_IVAR_SECONDARYBUS,
	PCI_IVAR_SUBORDINATEBUS,
	PCI_IVAR_HOSE,
};


/* config header information common to all header types */
typedef struct pcicfg {
	struct device *dev;			/* device which owns this */
	void		*hdrspec;		/* pointer to header type specific data */
	u_int16_t	subvendor;		/* card vendor ID */
	u_int16_t	subdevice;		/* card device ID, assigned by card vendor */
	u_int16_t	vendor;			/* chip vendor ID */
	u_int16_t	device;			/* chip device ID, assigned by chip vendor */
	u_int16_t	cmdreg;			/* disable/enable chip and PCI options */
	u_int16_t	statreg;		/* supported PCI features and error state */
	u_int8_t	baseclass;		/* chip PCI class */
	u_int8_t	subclass;		/* chip PCI subclass */
	u_int8_t	progif;			/* chip PCI programming interface */
	u_int8_t	revid;			/* chip revision ID */
	u_int8_t	hdrtype;		/* chip config header type */
	u_int8_t	cachelnsz;		/* cache line size in 4byte units */
	u_int8_t	intpin;			/* PCI interrupt pin */
	u_int8_t	intline;		/* interrupt line (IRQ for PC arch) */
	u_int8_t	mingnt;			/* min. useful bus grant time in 250ns units */
	u_int8_t	maxlat;			/* max. tolerated bus grant latency in 250ns */
	u_int8_t	lattimer;		/* latency timer in units of 30ns bus cycles */
	u_int8_t	mfdev;			/* multi-function device (from hdrtype reg) */
	u_int8_t	nummaps;		/* actual number of PCI maps used */
	u_int8_t	hose;			/* hose which bus is attached to */
	u_int8_t	bus;			/* config space bus address */
	u_int8_t	slot;			/* config space slot address */
	u_int8_t	func;			/* config space function number */
	u_int8_t	secondarybus;	/* bus on secondary side of bridge, if any */
	u_int8_t	subordinatebus;	/* topmost bus number behind bridge, if any */
	u_int16_t	pp_cap;			/* PCI power management capabilities */
	u_int8_t	pp_status;		/* config space address of PCI power status reg */
	u_int8_t	pp_pmcsr;		/* config space address of PMCSR reg */
	u_int8_t	pp_data;		/* config space address of PCI power data reg */
} pcicfgregs;


typedef struct {
	pci_addr_t	pmembase;	/* base address of prefetchable memory */
	pci_addr_t	pmemlimit;	/* topmost address of prefetchable memory */
	u_int32_t	membase;	/* base address of memory window */
	u_int32_t	memlimit;	/* topmost address of memory window */
	u_int32_t	iobase;		/* base address of port window */
	u_int32_t	iolimit;	/* topmost address of port window */
	u_int16_t	secstat;	/* secondary bus status register */
	u_int16_t	bridgectl;	/* bridge control register */
	u_int8_t	seclat;		/* CardBus latency timer */
} pcih1cfgregs;


/* additional type 2 device config header information (CardBus bridge) */
typedef struct {
	u_int32_t	membase0;	/* base address of memory window */
	u_int32_t	memlimit0;	/* topmost address of memory window */
	u_int32_t	membase1;	/* base address of memory window */
	u_int32_t	memlimit1;	/* topmost address of memory window */
	u_int32_t	iobase0;	/* base address of port window */
	u_int32_t	iolimit0;	/* topmost address of port window */
	u_int32_t	iobase1;	/* base address of port window */
	u_int32_t	iolimit1;	/* topmost address of port window */
	u_int32_t	pccardif;	/* PC Card 16bit IF legacy more base addr. */
	u_int16_t	secstat;	/* secondary bus status register */
	u_int16_t	bridgectl;	/* bridge control register */
	u_int8_t	seclat;		/* CardBus latency timer */
} pcih2cfgregs;

/*
 * Simplified accessors for pci devices
 */
#define PCI_ACCESSOR(A, B, T)										\
static inline T pci_get_ ## A(device_t dev)							\
{																	\
	uintptr_t v;													\
	BUS_READ_IVAR(device_get_parent(dev), dev, PCI_IVAR_ ## B, &v);	\
	return (T) v;													\
}																	\
static inline void pci_set_ ## A(device_t dev, T t)					\
{																	\
	u_long v = (u_long) t;											\
	BUS_WRITE_IVAR(device_get_parent(dev), dev, PCI_IVAR_ ## B, v);	\
}
PCI_ACCESSOR(subvendor,			SUBVENDOR,		u_int16_t)
PCI_ACCESSOR(subdevice,			SUBDEVICE,		u_int16_t)
PCI_ACCESSOR(vendor,			VENDOR,			u_int16_t)
PCI_ACCESSOR(device,			DEVICE,			u_int16_t)
PCI_ACCESSOR(devid,				DEVID,			u_int32_t)
PCI_ACCESSOR(class,				CLASS,			u_int8_t)
PCI_ACCESSOR(subclass,			SUBCLASS,		u_int8_t)
PCI_ACCESSOR(progif,			PROGIF,			u_int8_t)
PCI_ACCESSOR(revid,				REVID,			u_int8_t)
PCI_ACCESSOR(intpin,			INTPIN,			u_int8_t)
PCI_ACCESSOR(irq,				IRQ,			u_int8_t)
PCI_ACCESSOR(bus,				BUS,			u_int8_t)
PCI_ACCESSOR(slot,				SLOT,			u_int8_t)
PCI_ACCESSOR(function,			FUNCTION,		u_int8_t)
PCI_ACCESSOR(secondarybus,		SECONDARYBUS,	u_int8_t)
PCI_ACCESSOR(subordinatebus,	SUBORDINATEBUS,	u_int8_t)
PCI_ACCESSOR(hose,				HOSE,			u_int32_t)

static inline u_int32_t pci_read_config(device_t dev, int reg, int width)
{
	return PCI_READ_CONFIG(device_get_parent(dev), dev, reg, width);
}

static inline void pci_write_config(device_t dev, int reg, u_int32_t val, int width)
{
	PCI_WRITE_CONFIG(device_get_parent(dev), dev, reg, val, width);
}

static inline void pci_enable_busmaster(device_t dev)
{
	PCI_ENABLE_BUSMASTER(device_get_parent(dev), dev);
}

static inline void pci_disable_busmaster(device_t dev)
{
	PCI_DISABLE_BUSMASTER(device_get_parent(dev), dev);
}

static inline void pci_enable_io(device_t dev, int space)
{
	PCI_ENABLE_IO(device_get_parent(dev), dev, space);
}

static inline void pci_disable_io(device_t dev, int space)
{
	PCI_DISABLE_IO(device_get_parent(dev), dev, space);
}


extern const char *pci_ata_match(struct device *dev);
extern const char *pci_usb_match(struct device *dev);
extern const char *pci_vga_match(struct device *dev);
extern int initPciNew();


#endif /* _PCIVAR_H_ */
