/*-
 * Copyright (c) 1996-1999
 * Kazutaka YOKOTA (yokota@zodiac.mech.utsunomiya-u.ac.jp)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/dev/kbd/atkbdc.c,v 1.5.2.2 2002/03/31 11:02:02 murray Exp $
 * from kbdio.c,v 1.13 1998/09/25 11:55:46 yokota Exp
 *
 * 2008: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/rman.h>
#include <stdlib.h>
#include <isa/isareg.h>
#include <machine/clock.h>
#include <i386/bus_at386.h>
#include <kbd/atkbdcreg.h>

//=====================================  ===================================================

#define MAXKBDC		1
#define kbdcp(p)			((atkbdc_softc_t *)(p))
#define nextq(i)			(((i) + 1) % KBDQ_BUFSIZE)
#define availq(q)			((q)->head != (q)->tail)
#define emptyq(q)			((q)->tail = (q)->head = 0)
#define read_status(k)		(bus_space_read_1((k)->iot, (k)->ioh1, 0))
#define read_data(k)		(bus_space_read_1((k)->iot, (k)->ioh0, 0))
#define write_command(k, d)	(bus_space_write_1((k)->iot, (k)->ioh1, 0, (d)))
#define write_data(k, d)	(bus_space_write_1((k)->iot, (k)->ioh0, 0, (d)))

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static atkbdc_softc_t default_kbdc;
static atkbdc_softc_t *atkbdc_softc[MAXKBDC] = { &default_kbdc };

static int atkbdc_setup(
	atkbdc_softc_t *sc, 
	bus_space_tag_t tag, 
	bus_space_handle_t h0,
	bus_space_handle_t h1)
{
	if (sc->ioh0 == 0) {	/* XXX */
		sc->command_byte = -1;
		sc->command_mask = 0;
		sc->lock = FALSE;
		sc->kbd.head = sc->kbd.tail = 0;
		sc->aux.head = sc->aux.tail = 0;
	}
	sc->iot = tag;
	sc->ioh0 = h0;
	sc->ioh1 = h1;

	return 0;
}

/*
 * return : TRUE or FALSE
 */
static int addq(
	kqueue *q, 
	int c)
{
	if (nextq(q->tail) != q->head) {
		q->q[q->tail] = c;
		q->tail = nextq(q->tail);
		return TRUE;
	}
	return FALSE;
}

/*
 * return : 
 */
static int removeq(
	kqueue *q)
{
	int c;

	if (q->tail != q->head) {
		c = q->q[q->head];
		q->head = nextq(q->head);
		return c;
	}
	return -1;
}

/* 
 * device I/O routines
 */
static int wait_while_controller_busy(
	struct atkbdc_softc *kbdc)
{
	/* CPU will stay inside the loop for 100msec at most */
	int retry = 5000;
	int f;

	while ((f = read_status(kbdc)) & KBDS_INPUT_BUFFER_FULL) {
		if ((f & KBDS_BUFFER_FULL) == KBDS_KBD_BUFFER_FULL) {
			DELAY(KBDD_DELAYTIME);
			addq(&kbdc->kbd, read_data(kbdc));
		} 
		else if ((f & KBDS_BUFFER_FULL) == KBDS_AUX_BUFFER_FULL) {
			DELAY(KBDD_DELAYTIME);
			addq(&kbdc->aux, read_data(kbdc));
		}
		DELAY(KBDC_DELAYTIME);
		if (--retry < 0)
			return FALSE;
	}
	return TRUE;
}

/* wait for data from the keyboard */
static int wait_for_kbd_data(
	struct atkbdc_softc *kbdc)
{
	/* CPU will stay inside the loop for 200msec at most */
	int retry = 10000;
	int f;

	while ((f = read_status(kbdc) & KBDS_BUFFER_FULL) != KBDS_KBD_BUFFER_FULL) {
		if (f == KBDS_AUX_BUFFER_FULL) {
			DELAY(KBDD_DELAYTIME);
			addq(&kbdc->aux, read_data(kbdc));
		}
		DELAY(KBDC_DELAYTIME);
		if (--retry < 0) {
			return 0;
		}
	}
	DELAY(KBDD_DELAYTIME);
	return f;
}

/*
 * wait for any data; whether it's from the controller, 
 * the keyboard, or the aux device.
 */
static int wait_for_data(
	struct atkbdc_softc *kbdc)
{
	/* CPU will stay inside the loop for 200msec at most */
	int retry = 10000;
	int f;

	while ((f = read_status(kbdc) & KBDS_ANY_BUFFER_FULL) == 0) {
		DELAY(KBDC_DELAYTIME);
		if (--retry < 0) {
			return 0;
		}
	}
	DELAY(KBDD_DELAYTIME);
	return f;
}

/* 
 * wait for an ACK(FAh), RESEND(FEh), or RESET_FAIL(FCh) from the keyboard.
 * queue anything else.
 */
static int wait_for_kbd_ack(
	struct atkbdc_softc *kbdc)
{
	/* CPU will stay inside the loop for 200msec at most */
	int retry = 10000;
	int f;
	int b;

	while (retry-- > 0) {
		if ((f = read_status(kbdc)) & KBDS_ANY_BUFFER_FULL) {
			DELAY(KBDD_DELAYTIME);
			b = read_data(kbdc);
			if ((f & KBDS_BUFFER_FULL) == KBDS_KBD_BUFFER_FULL) {
				if ((b == KBD_ACK) || (b == KBD_RESEND) || (b == KBD_RESET_FAIL)) {
					return b;
				}
				addq(&kbdc->kbd, b);
			} 
			else if ((f & KBDS_BUFFER_FULL) == KBDS_AUX_BUFFER_FULL) {
				addq(&kbdc->aux, b);
			}
		}
		DELAY(KBDC_DELAYTIME);
	}
	return -1;
}

//===================================== PUBLIC =====================================================

int atkbdc_probe_unit(
	int unit, 
	struct resource *port0, 
	struct resource *port1)
{
	if (rman_get_start(port0) <= 0) {
		return ENXIO;
	}
	if (rman_get_start(port1) <= 0) {
		return ENXIO;
	}
	return 0;
}

int atkbdc_attach_unit(
	int unit, 
	atkbdc_softc_t *sc, 
	struct resource *port0,
	struct resource *port1)
{
	return atkbdc_setup(sc, rman_get_bustag(port0), rman_get_bushandle(port0), rman_get_bushandle(port1));
}

/* the backdoor to the keyboard controller! XXX */
int atkbdc_configure(void)
{
	bus_space_tag_t tag;
	bus_space_handle_t h0;
	bus_space_handle_t h1;
	int port0;
	int port1;

	port0 = IO_KBD;
	resource_int_value("atkbdc", 0, "port", &port0);
	port1 = IO_KBD + KBD_STATUS_PORT;

	/* XXX: tag should be passed from the caller */
	tag = I386_BUS_SPACE_IO;

#if notyet
	bus_space_map(tag, port0, IO_KBDSIZE, 0, &h0);
	bus_space_map(tag, port1, IO_KBDSIZE, 0, &h1);
#else
	h0 = (bus_space_handle_t)port0;
	h1 = (bus_space_handle_t)port1;
#endif
	return atkbdc_setup(atkbdc_softc[0], tag, h0, h1);
}

/* open a keyboard controller */
KBDC atkbdc_open(
	int unit)
{
	if (unit <= 0) {
		unit = 0;
	}
	if (MAXKBDC <= unit) {
		return NULL;
	}
	if ((atkbdc_softc[unit]->port0 != NULL) || (atkbdc_softc[unit]->ioh0 != 0)) {		/* XXX */
		return (KBDC)atkbdc_softc[unit];
	}
	return NULL;
}

/* set/reset polling lock */
int kbdc_lock(
	KBDC p, 
	int lock)
{
	int prevlock;

	prevlock = kbdcp(p)->lock;
	kbdcp(p)->lock = lock;

	return (prevlock != lock);
}

/* 
 * read one byte from any source; whether from the controller,
 * the keyboard, or the aux device
 */
int read_controller_data(
	KBDC p)
{
	if (availq(&kbdcp(p)->kbd)) {
		return removeq(&kbdcp(p)->kbd);
	}
	if (availq(&kbdcp(p)->aux)) {
		return removeq(&kbdcp(p)->aux);
	}
	if (!wait_for_data(kbdcp(p))) {
		return -1;		/* timeout */
	}
	return read_data(kbdcp(p));
}

/* read one byte from the keyboard */
int read_kbd_data(KBDC p)
{
	if (availq(&kbdcp(p)->kbd)) {
		return removeq(&kbdcp(p)->kbd);
	}
	if (!wait_for_kbd_data(kbdcp(p))) {
		return -1;		/* timeout */
	}
	return read_data(kbdcp(p));
}

/* read one byte from the keyboard, but return immediately if 
 * no data is waiting
 */
int read_kbd_data_no_wait(
	KBDC p)
{
	int f;

	if (availq(&kbdcp(p)->kbd)) {
		return removeq(&kbdcp(p)->kbd);
	}
	f = read_status(kbdcp(p)) & KBDS_BUFFER_FULL;
	if (f == KBDS_AUX_BUFFER_FULL) {
		DELAY(KBDD_DELAYTIME);
		addq(&kbdcp(p)->aux, read_data(kbdcp(p)));
		f = read_status(kbdcp(p)) & KBDS_BUFFER_FULL;
	}
	if (f == KBDS_KBD_BUFFER_FULL) {
		DELAY(KBDD_DELAYTIME);
		return read_data(kbdcp(p));
	}
	return -1;		/* no data */
}

/* write a one byte command to the controller */
int write_controller_command(
	KBDC p, 
	int c)
{
	if (!wait_while_controller_busy(kbdcp(p))) {
		return FALSE;
	}
	write_command(kbdcp(p), c);
	return TRUE;
}

/* write a one byte data to the controller */
int write_controller_data(
	KBDC p, 
	int c)
{
	if (!wait_while_controller_busy(kbdcp(p))) {
		return FALSE;
	}
	write_data(kbdcp(p), c);
	return TRUE;
}

/* write a one byte keyboard command */
int write_kbd_command(KBDC p, int c)
{
	if (!wait_while_controller_busy(kbdcp(p))) {
		return FALSE;
	}
	write_data(kbdcp(p), c);
	return TRUE;
}

/* send a command to the keyboard and wait for ACK */
int send_kbd_command(
	KBDC p, 
	int c)
{
	int retry = KBD_MAXRETRY;
	int res = -1;

	while (retry-- > 0) {
	if (!write_kbd_command(p, c))
		continue;
		res = wait_for_kbd_ack(kbdcp(p));
		if (res == KBD_ACK)
			break;
	}
	return res;
}

/* send a command and a data to the keyboard, wait for ACKs */
int send_kbd_command_and_data(
	KBDC p, 
	int c, 
	int d)
{
	int retry;
	int res = -1;

	for (retry = KBD_MAXRETRY; retry > 0; --retry) {
		if (!write_kbd_command(p, c)) {
			continue;
		}
		res = wait_for_kbd_ack(kbdcp(p));
		if (res == KBD_ACK) {
			break;
		}
		else if (res != KBD_RESEND) {
			return res;
		}
	}
	if (retry <= 0) {
		return res;
	}

	for (retry = KBD_MAXRETRY, res = -1; retry > 0; --retry) {
		if (!write_kbd_command(p, d)) {
			continue;
		}
		res = wait_for_kbd_ack(kbdcp(p));
		if (res != KBD_RESEND) {
			break;
		}
	}
	return res;
}

/* discard any data from the keyboard or the aux device */
void empty_both_buffers(
	KBDC p, 
	int wait)
{
	int t;
	int f;
	int delta = 2;

	for (t = wait; t > 0; ) { 
		if ((f = read_status(kbdcp(p))) & KBDS_ANY_BUFFER_FULL) {
			DELAY(KBDD_DELAYTIME);
			(void)read_data(kbdcp(p));
			t = wait;
		} 
		else {
			t -= delta;
		}
		DELAY(delta*1000);
	}

	emptyq(&kbdcp(p)->kbd);
	emptyq(&kbdcp(p)->aux);
}

int test_controller(
	KBDC p)
{
	int retry = KBD_MAXRETRY;
	int again = KBD_MAXWAIT;
	int c = KBD_DIAG_FAIL;

	while (retry-- > 0) {
		empty_both_buffers(p, 10);
		if (write_controller_command(p, KBDC_DIAGNOSE)) {
			break;
		}
	}
	if (retry < 0) {
		return FALSE;
	}

	emptyq(&kbdcp(p)->kbd);
	while (0 < again--) {
		/* wait awhile */
		DELAY(KBD_RESETDELAY*1000);
		c = read_controller_data(p);	/* DIAG_DONE/DIAG_FAIL */
		if (c != -1) { 	/* wait again if the controller is not ready */
			break;
		}
	}
	return (c == KBD_DIAG_DONE);
}

int test_kbd_port(
	KBDC p)
{
	int retry = KBD_MAXRETRY;
	int again = KBD_MAXWAIT;
	int c = -1;

	while (retry-- > 0) {
		empty_both_buffers(p, 10);
		if (write_controller_command(p, KBDC_TEST_KBD_PORT)) {
			break;
		}
	}
	if (retry < 0) {
		return FALSE;
	}

	emptyq(&kbdcp(p)->kbd);
	while (again-- > 0) {
		c = read_controller_data(p);
		if (c != -1) { 	/* try again if the controller is not ready */
			break;
		}
	}
	return c;
}

//--------------------------------------------------------------------------------------------------
// Getter
//--------------------------------------------------------------------------------------------------

atkbdc_softc_t *atkbdc_get_softc(
	int unit)
{
	atkbdc_softc_t *sc;

	if ((int) (sizeof(atkbdc_softc) / sizeof(atkbdc_softc[0])) <= unit) {
		return NULL;
	}
	sc = atkbdc_softc[unit];
	if (sc == NULL) {
		sc = atkbdc_softc[unit] = malloc(sizeof(*sc));
		if (sc == NULL) {
			return NULL;
		}
		bzero(sc, sizeof(*sc));
	}
	return sc;
}

int kbdc_get_device_mask(
	KBDC p)
{
	return kbdcp(p)->command_mask;
}

int get_controller_command_byte(
	KBDC p)
{
	if (kbdcp(p)->command_byte != -1) {
		return kbdcp(p)->command_byte;
	}
	if (!write_controller_command(p, KBDC_GET_COMMAND_BYTE)) {
		return -1;
	}
	emptyq(&kbdcp(p)->kbd);
	kbdcp(p)->command_byte = read_controller_data(p);
	return kbdcp(p)->command_byte;
}

//--------------------------------------------------------------------------------------------------
// Setter
//--------------------------------------------------------------------------------------------------

void kbdc_set_device_mask(
	KBDC p, 
	int mask)
{
	kbdcp(p)->command_mask = mask & (KBD_KBD_CONTROL_BITS | KBD_AUX_CONTROL_BITS);
}

int set_controller_command_byte(
	KBDC p, 
	int mask, 
	int command)
{
	if (get_controller_command_byte(p) == -1) {
		return FALSE;
	}

	command = (kbdcp(p)->command_byte & ~mask) | (command & mask);
	if (command & KBD_DISABLE_KBD_PORT) {
		if (!write_controller_command(p, KBDC_DISABLE_KBD_PORT)) {
			return FALSE;
		}
	}
	if (!write_controller_command(p, KBDC_SET_COMMAND_BYTE)) {
		return FALSE;
	}
	if (!write_controller_data(p, command)) {
		return FALSE;
	}
	kbdcp(p)->command_byte = command;

	return TRUE;
}

/* NOTE: enable the keyboard port but disable the keyboard 
 * interrupt before calling "reset_kbd()".
 */
int reset_kbd(
	KBDC p)
{
	int retry = KBD_MAXRETRY;
	int again = KBD_MAXWAIT;
	int c = KBD_RESEND;		/* keep the compiler happy */

	while (0 < retry--) {
		empty_both_buffers(p, 10);
		if (!write_kbd_command(p, KBDC_RESET_KBD)) {
			continue;
		}
		emptyq(&kbdcp(p)->kbd);
		c = read_controller_data(p);
		if (c == KBD_ACK) {	/* keyboard has agreed to reset itself... */
			break;
		}
	}
	if (retry < 0) {
		return FALSE;
	}

	while (again-- > 0) {
		/* wait awhile, well, in fact we must wait quite loooooooooooong */
		DELAY(KBD_RESETDELAY * 1000);
		c = read_controller_data(p);	/* RESET_DONE/RESET_FAIL */
		if (c != -1) { 	/* wait again if the controller is not ready */
			break;
		}
	}
	if (c != KBD_RESET_DONE) {
		return FALSE;
	}
	return TRUE;
}

//--------------------------------------------------------------------------------------------------
// Yes or No
//--------------------------------------------------------------------------------------------------

/* check if any data is waiting to be processed */
int kbdc_data_ready(KBDC p)
{
    return (availq(&kbdcp(p)->kbd) || availq(&kbdcp(p)->aux) || (read_status(kbdcp(p)) & KBDS_ANY_BUFFER_FULL));
}

//--------------------------------------------------------------------------------------------------
// Search
//--------------------------------------------------------------------------------------------------

