/*-
 * Copyright (c) 1999 Kazutaka YOKOTA <yokota@zodiac.mech.utsunomiya-u.ac.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/dev/kbd/kbd.c,v 1.17.2.2 2001/07/30 16:46:43 yokota Exp $
 *
 * 2008: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/errno.h>
#include <sys/queue.h>
#include <sys/systm.h>
#include <sys/libkern.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <kbd/kbdreg.h>
#include <kbd/kbdreg.h>

//=====================================  ===================================================

#define ARRAY_DELTA	4

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static	SLIST_HEAD(, keyboard_driver) keyboard_drivers = SLIST_HEAD_INITIALIZER(keyboard_drivers);

/*
 * We need at least one entry each in order to initialize a keyboard
 * for the kernel console.  The arrays will be increased dynamically
 * when necessary.
 */
static int				keyboards = 1;
static keyboard_t		*kbd_ini;
static keyboard_t		**keyboard = &kbd_ini;
static keyboard_switch_t *kbdsw_ini;
static keyboard_switch_t **kbdsw = &kbdsw_ini;

#define set_lockkey_state(k, s, l)									\
	if (!((s) & l ## DOWN)) {										\
		int i;														\
		(s) |= l ## DOWN;											\
		(s) ^= l ## ED;												\
		i = (s) & LOCK_MASK;										\
		(*kbdsw[(k)->kb_index]->ioctl)((k), KDSETLED, (caddr_t)&i); \
	}

static u_int save_accent_key(
	keyboard_t *kbd, 
	u_int key, 
	int *accents)
{
	int i;

	/* make an index into the accent map */
	i = key - F_ACC + 1;
	if ((i > kbd->kb_accentmap->n_accs) || (kbd->kb_accentmap->acc[i - 1].accchar == 0)) {
		/* the index is out of range or pointing to an empty entry */
		*accents = 0;
		return ERRKEY;
	}

	/* 
	 * If the same accent key has been hit twice, produce the accent char
	 * itself.
	 */
	if (i == *accents) {
		key = kbd->kb_accentmap->acc[i - 1].accchar;
		*accents = 0;
		return key;
	}

	/* remember the index and wait for the next key  */
	*accents = i; 
	return NOKEY;
}

static u_int make_accent_char(
	keyboard_t *kbd, 
	u_int ch, 
	int *accents)
{
	struct acc_t *acc;
	int i;

	acc = &kbd->kb_accentmap->acc[*accents - 1];
	*accents = 0;

	/* 
	 * If the accent key is followed by the space key,
	 * produce the accent char itself.
	 */
	if (ch == ' ') {
		return acc->accchar;
	}

	/* scan the accent map */
	for (i = 0; i < NUM_ACCENTCHARS; ++i) {
		if (acc->map[i][0] == 0) {	/* end of table */
			break;
		}
		if (acc->map[i][0] == ch) {
			return acc->map[i][1];
		}
	}
	/* this char cannot be accented... */
	return ERRKEY;
}

static int kbd_realloc_array(void)
{
	keyboard_t **new_kbd;
	keyboard_switch_t **new_kbdsw;
	int newsize;
	int s;

	s = spltty();
	newsize = ((keyboards + ARRAY_DELTA) / ARRAY_DELTA) * ARRAY_DELTA;
	new_kbd = malloc(sizeof(*new_kbd)*newsize);
	if (new_kbd == NULL) {
		splx(s);
		return ENOMEM;
	}
	new_kbdsw = malloc(sizeof(*new_kbdsw)*newsize);
	if (new_kbdsw == NULL) {
		free(new_kbd);
		splx(s);
		return ENOMEM;
	}
	bzero(new_kbd, sizeof(*new_kbd)*newsize);
	bzero(new_kbdsw, sizeof(*new_kbdsw)*newsize);
	bcopy(keyboard, new_kbd, sizeof(*keyboard)*keyboards);
	bcopy(kbdsw, new_kbdsw, sizeof(*kbdsw)*keyboards);
	if (1 < keyboards) {
		free(keyboard);
		free(kbdsw);
	}
	keyboard = new_kbd;
	kbdsw = new_kbdsw;
	keyboards = newsize;
	splx(s);

	return 0;
}

/* diagnostic dump */
static char *get_kbd_type_name(
	int type)
{
	static struct {
		int type;
		char *name;
	} name_table[] = {
		{ KB_84,	"AT 84" },
		{ KB_101,	"AT 101/102" },
		{ KB_OTHER,	"generic" },
	};
	int i;

	for (i = 0; i < (int) (sizeof(name_table) / sizeof(name_table[0])); ++i) {
		if (type == name_table[i].type) {
			return name_table[i].name;
		}
	}
	return "unknown";
}

//===================================== PUBLIC =====================================================

/*
 * The back door for the console driver; configure keyboards
 * This function is for the kernel console to initialize keyboards
 * at very early stage.
 */
int kbd_configure(
	int flags)
{
	const keyboard_driver_t *p;

	SLIST_FOREACH(p, &keyboard_drivers, link) {
		if (p->configure != NULL) {
			(*p->configure)(flags);
		}
	}

	return 0;
}

/* declare a new keyboard driver */
int kbd_add_driver(
	keyboard_driver_t *driver)
{
	if (SLIST_NEXT(driver, link)) {
		return EINVAL;
	}
	SLIST_INSERT_HEAD(&keyboard_drivers, driver, link);
	return 0;
}

int kbd_delete_driver(
	keyboard_driver_t *driver)
{
	SLIST_REMOVE(&keyboard_drivers, driver, keyboard_driver, link);
	SLIST_NEXT(driver, link) = NULL;
	return 0;
}

/* register a keyboard and associate it with a function table */
int kbd_register(
	keyboard_t *kbd)
{
	const keyboard_driver_t *p;
	int index;

	for (index = 0; index < keyboards; ++index) {
		if (keyboard[index] == NULL) {
			break;
		}
	}
	if (keyboards <= index) {
		if (kbd_realloc_array()) {
			return -1;
		}
	}

	kbd->kb_index = index;
	KBD_UNBUSY(kbd);
	KBD_VALID(kbd);
	kbd->kb_active = 0;	/* disabled until someone calls kbd_enable() */
	kbd->kb_token = NULL;
	kbd->kb_callback.kc_func = NULL;
	kbd->kb_callback.kc_arg = NULL;

	SLIST_FOREACH(p, &keyboard_drivers, link) {
		if (strcmp(p->name, kbd->kb_name) == 0) {
			keyboard[index] = kbd;
			kbdsw[index] = p->kbdsw;
			return index;
		}
	}

	return -1;
}

int kbd_unregister(
	keyboard_t *kbd)
{
	int error;
	int s;

	if ((kbd->kb_index < 0) || (kbd->kb_index >= keyboards)) {
		return ENOENT;
	}
	if (keyboard[kbd->kb_index] != kbd) {
		return ENOENT;
	}

	s = spltty();
	if (KBD_IS_BUSY(kbd)) {
		error = (*kbd->kb_callback.kc_func)(kbd, KBDIO_UNLOADING, kbd->kb_callback.kc_arg);
		if (error) {
			splx(s);
			return error;
		}
		if (KBD_IS_BUSY(kbd)) {
			splx(s);
			return EBUSY;
		}
	}
	KBD_INVALID(kbd);
	keyboard[kbd->kb_index] = NULL;
	kbdsw[kbd->kb_index] = NULL;

	splx(s);
	return 0;
}

/* initialize the keyboard_t structure */
void kbd_init_struct(
	keyboard_t *kbd, 
	char *name, 
	int type, 
	int unit, 
	int config,
	int port, 
	int port_size)
{
	kbd->kb_flags = KB_NO_DEVICE;	/* device has not been found */
	kbd->kb_name = name;
	kbd->kb_type = type;
	kbd->kb_unit = unit;
	kbd->kb_config = config & ~KB_CONF_PROBE_ONLY;
	kbd->kb_led = 0;		/* unknown */
	kbd->kb_io_base = port;
	kbd->kb_io_size = port_size;
	kbd->kb_data = NULL;
	kbd->kb_keymap = NULL;
	kbd->kb_accentmap = NULL;
	kbd->kb_fkeytab = NULL;
	kbd->kb_fkeytab_size = 0;
	kbd->kb_delay1 = KB_DELAY1;	/* these values are advisory only */
	kbd->kb_delay2 = KB_DELAY2;
	kbd->kb_count = 0L;
	bzero(kbd->kb_lastact, sizeof(kbd->kb_lastact));
}

int genkbd_commonioctl(
	keyboard_t *kbd, 
	u_long cmd, 
	caddr_t arg)
{
	keyarg_t *keyp;
	fkeyarg_t *fkeyp;
	int s;
	int i;

	s = spltty();
	switch (cmd) {

	case KDGKBINFO:		/* get keyboard information */
		((keyboard_info_t *)arg)->kb_index = kbd->kb_index;
		i = imin(strlen(kbd->kb_name) + 1, sizeof(((keyboard_info_t *)arg)->kb_name));
		bcopy(kbd->kb_name, ((keyboard_info_t *)arg)->kb_name, i);
		((keyboard_info_t *)arg)->kb_unit = kbd->kb_unit;
		((keyboard_info_t *)arg)->kb_type = kbd->kb_type;
		((keyboard_info_t *)arg)->kb_config = kbd->kb_config;
		((keyboard_info_t *)arg)->kb_flags = kbd->kb_flags;
		break;

	case KDGKBTYPE:		/* get keyboard type */
		*(int *)arg = kbd->kb_type;
		break;

	case KDGETREPEAT:	/* get keyboard repeat rate */
		((int *)arg)[0] = kbd->kb_delay1;
		((int *)arg)[1] = kbd->kb_delay2; 
		break;

	case GIO_KEYMAP:	/* get keyboard translation table */
		bcopy(kbd->kb_keymap, arg, sizeof(*kbd->kb_keymap));
		break;
	case PIO_KEYMAP:	/* set keyboard translation table */
#ifndef KBD_DISABLE_KEYMAP_LOAD
		bzero(kbd->kb_accentmap, sizeof(*kbd->kb_accentmap));
		bcopy(arg, kbd->kb_keymap, sizeof(*kbd->kb_keymap));
		break;
#else
		splx(s);
		return ENODEV;
#endif

	case GIO_KEYMAPENT:	/* get keyboard translation table entry */
		keyp = (keyarg_t *)arg;
		if (keyp->keynum >= sizeof(kbd->kb_keymap->key) / sizeof(kbd->kb_keymap->key[0])) {
			splx(s);
			return EINVAL;
		}
		bcopy(&kbd->kb_keymap->key[keyp->keynum], &keyp->key, sizeof(keyp->key));
		break;
	case PIO_KEYMAPENT:	/* set keyboard translation table entry */
#ifndef KBD_DISABLE_KEYMAP_LOAD
		keyp = (keyarg_t *)arg;
		if (keyp->keynum >= sizeof(kbd->kb_keymap->key) / sizeof(kbd->kb_keymap->key[0])) {
			splx(s);
			return EINVAL;
		}
		bcopy(&keyp->key, &kbd->kb_keymap->key[keyp->keynum], sizeof(keyp->key));
		break;
#else
		splx(s);
		return ENODEV;
#endif

	case GIO_DEADKEYMAP:	/* get accent key translation table */
		bcopy(kbd->kb_accentmap, arg, sizeof(*kbd->kb_accentmap));
		break;
	case PIO_DEADKEYMAP:	/* set accent key translation table */
#ifndef KBD_DISABLE_KEYMAP_LOAD
		bcopy(arg, kbd->kb_accentmap, sizeof(*kbd->kb_accentmap));
		break;
#else
		splx(s);
		return ENODEV;
#endif

	case GETFKEY:		/* get functionkey string */
		fkeyp = (fkeyarg_t *)arg;
		if (fkeyp->keynum >= kbd->kb_fkeytab_size) {
			splx(s);
			return EINVAL;
		}
		bcopy(kbd->kb_fkeytab[fkeyp->keynum].str, fkeyp->keydef, kbd->kb_fkeytab[fkeyp->keynum].len);
		fkeyp->flen = kbd->kb_fkeytab[fkeyp->keynum].len;
		break;
	case SETFKEY:		/* set functionkey string */
#ifndef KBD_DISABLE_KEYMAP_LOAD
		fkeyp = (fkeyarg_t *)arg;
		if (fkeyp->keynum >= kbd->kb_fkeytab_size) {
			splx(s);
			return EINVAL;
		}
		kbd->kb_fkeytab[fkeyp->keynum].len = imin(fkeyp->flen, MAXFK);
		bcopy(fkeyp->keydef, kbd->kb_fkeytab[fkeyp->keynum].str, kbd->kb_fkeytab[fkeyp->keynum].len);
		break;
#else
		splx(s);
		return ENODEV;
#endif

	default:
		splx(s);
		return ENOIOCTL;
	}

	splx(s);
	return 0;
}

void genkbd_diag(
	keyboard_t *kbd, 
	int level)
{
	if (level > 0) {
		printf("kbd%d: %s%d, %s (%d), config:0x%x, flags:0x%x", 
			kbd->kb_index, kbd->kb_name, kbd->kb_unit, get_kbd_type_name(kbd->kb_type), kbd->kb_type, kbd->kb_config, kbd->kb_flags);
		if (kbd->kb_io_base > 0) {
			printf(", port:0x%x-0x%x", kbd->kb_io_base, kbd->kb_io_base + kbd->kb_io_size - 1);
		}
		printf("\n");
	}
}

int genkbd_keyaction(
	keyboard_t *kbd, 
	int keycode, 
	int up, 
	int *shiftstate,
	int *accents)
{
	struct keyent_t *key;
	int state = *shiftstate;
	int action;
	int f;
	int i;

	i = keycode;
	f = state & (AGRS | ALKED);
	if ((f == AGRS1) || (f == AGRS2) || (f == ALKED)) {
		i += ALTGR_OFFSET;
	}
	key = &kbd->kb_keymap->key[i];
	i = ((state & SHIFTS) ? 1 : 0) | ((state & CTLS) ? 2 : 0) | ((state & ALTS) ? 4 : 0);
	if (((key->flgs & FLAG_LOCK_C) && (state & CLKED)) || ((key->flgs & FLAG_LOCK_N) && (state & NLKED)) ) {
		i ^= 1;
	}

	if (up) {	/* break: key released */
		action = kbd->kb_lastact[keycode];
		kbd->kb_lastact[keycode] = NOP;
		switch (action) {
		case LSHA:
			if (state & SHIFTAON) {
				set_lockkey_state(kbd, state, ALK);
				state &= ~ALKDOWN;
			}
			action = LSH;
			/* FALL THROUGH */
		case LSH:
			state &= ~SHIFTS1;
			break;
		case RSHA:
			if (state & SHIFTAON) {
				set_lockkey_state(kbd, state, ALK);
				state &= ~ALKDOWN;
			}
			action = RSH;
			/* FALL THROUGH */
		case RSH:
			state &= ~SHIFTS2;
			break;
		case LCTRA:
			if (state & SHIFTAON) {
				set_lockkey_state(kbd, state, ALK);
				state &= ~ALKDOWN;
			}
			action = LCTR;
			/* FALL THROUGH */
		case LCTR:
			state &= ~CTLS1;
			break;
		case RCTRA:
			if (state & SHIFTAON) {
				set_lockkey_state(kbd, state, ALK);
				state &= ~ALKDOWN;
			}
			action = RCTR;
			/* FALL THROUGH */
		case RCTR:
			state &= ~CTLS2;
			break;
		case LALTA:
			if (state & SHIFTAON) {
				set_lockkey_state(kbd, state, ALK);
				state &= ~ALKDOWN;
			}
			action = LALT;
			/* FALL THROUGH */
		case LALT:
			state &= ~ALTS1;
			break;
		case RALTA:
			if (state & SHIFTAON) {
				set_lockkey_state(kbd, state, ALK);
				state &= ~ALKDOWN;
			}
			action = RALT;
			/* FALL THROUGH */
		case RALT:
			state &= ~ALTS2;
			break;
		case ASH:
			state &= ~AGRS1;
			break;
		case META:
			state &= ~METAS1;
			break;
		case NLK:
			state &= ~NLKDOWN;
			break;
		case CLK:
			state &= ~CLKDOWN;
			break;
		case SLK:
			state &= ~SLKDOWN;
			break;
		case ALK:
			state &= ~ALKDOWN;
			break;
		case NOP:
			/* release events of regular keys are not reported */
			*shiftstate &= ~SHIFTAON;
			return NOKEY;
		}
		*shiftstate = state & ~SHIFTAON;
		return (SPCLKEY | RELKEY | action);
	} 
	else {	/* make: key pressed */
		action = key->map[i];
		state &= ~SHIFTAON;
		if (key->spcl & (0x80 >> i)) {
			/* special keys */
			if (kbd->kb_lastact[keycode] == NOP) {
				kbd->kb_lastact[keycode] = action;
			}
			if (kbd->kb_lastact[keycode] != action) {
				action = NOP;
			}
			switch (action) {
			/* LOCKING KEYS */
			case NLK:
				set_lockkey_state(kbd, state, NLK);
				break;
			case CLK:
				set_lockkey_state(kbd, state, CLK);
				break;
			case SLK:
				set_lockkey_state(kbd, state, SLK);
				break;
			case ALK:
				set_lockkey_state(kbd, state, ALK);
				break;
			/* NON-LOCKING KEYS */
			case SPSC: case RBT:  case SUSP: case STBY:
			case DBG:  case NEXT: case PREV: case PNC:
			case HALT: case PDWN:
				*accents = 0;
				break;
			case BTAB:
				*accents = 0;
				action |= BKEY;
				break;
			case LSHA:
				state |= SHIFTAON;
				action = LSH;
				/* FALL THROUGH */
			case LSH:
				state |= SHIFTS1;
				break;
			case RSHA:
				state |= SHIFTAON;
				action = RSH;
				/* FALL THROUGH */
			case RSH:
				state |= SHIFTS2;
				break;
			case LCTRA:
				state |= SHIFTAON;
				action = LCTR;
				/* FALL THROUGH */
			case LCTR:
				state |= CTLS1;
				break;
			case RCTRA:
				state |= SHIFTAON;
				action = RCTR;
				/* FALL THROUGH */
			case RCTR:
				state |= CTLS2;
				break;
			case LALTA:
				state |= SHIFTAON;
				action = LALT;
				/* FALL THROUGH */
			case LALT:
				state |= ALTS1;
				break;
			case RALTA:
				state |= SHIFTAON;
				action = RALT;
				/* FALL THROUGH */
			case RALT:
				state |= ALTS2;
				break;
			case ASH:
				state |= AGRS1;
				break;
			case META:
				state |= METAS1;
				break;
			case NOP:
				*shiftstate = state;
				return NOKEY;
			default:
				/* is this an accent (dead) key? */
				*shiftstate = state;
				if (F_ACC <= action && action <= L_ACC) {
					action = save_accent_key(kbd, action, accents);
					switch (action) {
					case NOKEY:
					case ERRKEY:
						return action;
					default:
						if (state & METAS) {
							return (action | MKEY);
						}
						else {
							return action;
						}
					}
					/* NOT REACHED */
				}
				/* other special keys */
				if (*accents > 0) {
					*accents = 0;
					return ERRKEY;
				}
				if (action >= F_FN && action <= L_FN) {
					action |= FKEY;
				}
				/* XXX: return fkey string for the FKEY? */
				return (SPCLKEY | action);
			}
			*shiftstate = state;
			return (SPCLKEY | action);
		} 
		else {
			/* regular keys */
			kbd->kb_lastact[keycode] = NOP;
			*shiftstate = state;
			if (*accents > 0) {
				/* make an accented char */
				action = make_accent_char(kbd, action, accents);
				if (action == ERRKEY) {
					return action;
				}
			}
			if (state & METAS) {
				action |= MKEY;
			}
			return action;
		}
	}
	/* NOT REACHED */
}

//--------------------------------------------------------------------------------------------------
// Getter
//--------------------------------------------------------------------------------------------------

/* get a keyboard structure */
keyboard_t *kbd_get_keyboard(
	int index)
{
	if ((index < 0) || (keyboards <= index)) {
		return NULL;
	}
	if (keyboard[index] == NULL) {
		return NULL;
	}
	if (!KBD_IS_VALID(keyboard[index])) {
		return NULL;
	}
	return keyboard[index];
}

/* find a funciton table by the driver name */
keyboard_switch_t *kbd_get_switch(
	char *driver)
{
	const keyboard_driver_t *p;

	SLIST_FOREACH(p, &keyboard_drivers, link) {
		if (strcmp(p->name, driver) == 0) {
			return p->kbdsw;
		}
	}

	return NULL;
}

/* get a pointer to the string associated with the given function key */
u_char *genkbd_get_fkeystr(
	keyboard_t *kbd, 
	int fkey, 
	size_t *len)
{
	if (kbd == NULL) {
		return NULL;
	}
	fkey -= F_FN;
	if (fkey > kbd->kb_fkeytab_size) {
		return NULL;
	}
	*len = kbd->kb_fkeytab[fkey].len;
	return kbd->kb_fkeytab[fkey].str;
}

keyboard_switch_t *kbdGetKbdsw(
	keyboard_t *this)
{
	return kbdsw[this->kb_index];
}
	
//--------------------------------------------------------------------------------------------------
// Setter
//--------------------------------------------------------------------------------------------------

void kbd_set_maps(
	keyboard_t *kbd, 
	keymap_t *keymap, 
	accentmap_t *accmap,
	fkeytab_t *fkeymap, 
	int fkeymap_size)
{
	kbd->kb_keymap = keymap;
	kbd->kb_accentmap = accmap;
	kbd->kb_fkeytab = fkeymap;
	kbd->kb_fkeytab_size = fkeymap_size;
}

//--------------------------------------------------------------------------------------------------
// Yes or No
//--------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------------------
// Search
//--------------------------------------------------------------------------------------------------

/* find the keyboard specified by a driver name and a unit number */
int kbd_find_keyboard(
	char *driver, 
	int unit)
{
	int i;

	for (i = 0; i < keyboards; ++i) {
		if (keyboard[i] == NULL) {
			continue;
		}
		if (!KBD_IS_VALID(keyboard[i])) {
			continue;
		}
		if (strcmp("*", driver) && strcmp(keyboard[i]->kb_name, driver)) {
			continue;
		}
		if ((unit != -1) && (keyboard[i]->kb_unit != unit)) {
			continue;
		}
		return i;
	}
	return -1;
}
