/*
 * ext2.c
 *
 * Copyright 2003, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * EXT2ե륷ƥࡣ
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/libkern.h>
#include <sys/conf.h>
#include <machine/limits.h>
#include <machine/lock.h>
#include <lib/lib.h>
#include <lib/lib_path.h>
#include <kern/vm.h>
#include <kern/kmalloc.h>
#include <kern/fs.h>
#include <kern/device.h>
#include <kern/devfs.h>
#include <kern/proc.h>
#include <kern/time.h>
#include <kern/BlockCache.h>
#include <fs/ext2.h>

#include <kern/debug.h>

#define DEBUG_EXT2_FS 1
#ifdef DEBUG_EXT2_FS
	#define STATIC
//	#define INLINE
	#define INLINE	inline
#else
	#define STATIC static
	#define INLINE	inline
#endif

/************************************************************************************
 *
 * 
 *
 *************************************************************************************/

enum{
	INT_LOG					= 2,		/* intΥХȿΣ򺬤Ȥп */
	UINT_LOG				= INT_LOG,

	EXT2_BLOCK_BASE			=1024,		/* ֥å׻뤿Υ١Хȿ */
	EXT2_BLOCK_BASE_LOG		= 10,		/* ١֥åΣ򺬤Ȥп */

	EXT2_SUPER_BLK_POSITION	= 1024,		/* ѡ֥åΰ֡ƬΥХȿˡ */

	EXT2_GROUP_DESC_BLK		= 1,		/* group descriptor block number. */

	/* File system states at ss_state". */
	EXT2_VALID_FS			= 0x0001,	/* Unmounted cleanly */
	EXT2_ERROR_FS			= 0x0002,	/* Errors detected */

	/* Maximal mount counts. */
	EXT2_DFL_MAX_MNT_COUNT	= 20,
	EXT2_DFL_CHECKINTERVAL	= 0,

	/* Magic number at "s_magic". */
	EXT2_SUPER_MAGIC		= 0xEF53,

	/* Behaviour when detecting errors. */
	EXT2_ERRORS_CONTINUE	= 1,		/* continue as if nothing happened. */
	EXT2_ERRORS_RO			= 2,		/* remount read-only. */
	EXT2_ERRORS_PANIC		= 3,		/* cause a kernel panic. */
	EXT2_ERRORS_DEFAULT		= EXT2_ERRORS_CONTINUE,

	/* revision level value. */
	EXT2_GOOD_OLD_REV		= 0,		/* original format. */
	EXT2_DYNAMIC_REV		= 1,		/* V2 format with dynamic inode sizes. */

	/* EXT2_*_INO values at "s_first_ino". */
	EXT2_BAD_INO			= 0x01, 	/* bad blocks inode */
	EXT2_ROOT_INO			= 0x02, 	/* root directory inode */
	EXT2_ACL_IDX_INO		= 0x03, 	/* ACL index inode (deprecated?) */
	EXT2_ACL_DATA_INO		= 0x04, 	/* ACL data inode (deprecated?) */
	EXT2_BOOT_LOADER_INO	= 0x05,		/* boot loader inode */
	EXT2_UNDEL_DIR_INO		= 0x06,		/* undelete directory inode */
};

/* super block. */
typedef struct{
	uint s_inodes_count;			/* 00,total number of inodes. */
	uint s_blocks_count;			/* 04,total number of blocks. */
	uint s_r_blocks_count;			/* 08,total number of blocks reserved for the usage of the super user. */
	uint s_free_blocks_count;		/* 0c,total number of free blocks. */
	uint s_free_inodes_count;		/* 10,total number of free inodes. */
	uint s_first_data_block;		/* 14,the block containing the superblock structure.block size larger than 1KB=0,block size of 1KB=1. */
	uint s_log_block_size;			/* 18,number of bits to shift left the value 1024 for computing block size. */
	int s_log_frag_size;			/* 1c,number of bits to shift the value 1024 for computing fragment size. */
	uint s_blocks_per_group;		/* 20,total number of blocks per group. */
	uint s_frags_per_group;			/* 24,total number of fragments per group. */
	uint s_inodes_per_group;		/* 28,total number of inodes per group. */
	uint s_mtime;					/* 2c,Unix time,last time the file system was mounted. */
	uint s_wtime;					/* 30,Unix time,last write access to the file system. */
	ushort s_mnt_count;				/* 34,how many time the file system was mounted. */
	short s_max_mnt_count;			/* 36,the maximum number of mount times. */
	ushort s_magic;					/* 38,magic number 0xEF53. */
	ushort s_state;					/* 3a,file system state. */
	ushort s_errors;				/* 3c,behaviour when an error is detected. */
	ushort s_minor_rev_level;		/* 3e,minor revision level. */
	uint s_lastcheck;				/* 40,Unix time,last file system check. */
	uint s_checkinterval;			/* 44,between file system checks. */
	uint s_creator_os;				/* 48,os that created the file system. */
	uint s_rev_level;				/* 4c,revision level value. */
	ushort s_def_resuid;			/* 4e,default user id for reserved blocks. */
	ushort s_def_resgid;			/* 50,default group id for reserved blocks. */

	/* EXT2_DYNAMIC_REV Specific */
	uint s_first_ino;				/* 54,index to the first inode useable for standard files. */
	ushort s_inode_size;			/* 58,size of the inode structure. */
	ushort s_block_group_nr;		/* 5a,block group number hosting this superblock. */
	uint s_feature_compat;			/* 5c,bitmask of compatible features. */
	uint s_feature_incompat;		/* 60,bitmask of incompatible features. */
	uint s_feature_ro_compat;		/* 64,bitmask of read-only features. */
	uchar s_uuid[16];				/* 68,volume id. */
	char s_volume_name[16];			/* 78,volume name,mostly unusued. */
	char s_last_mounted[64];		/* 88,last mounted directory path,not normally used. */
	uint s_algo_bitmap;				/* compression algorithms. */

	/* Performance Hints */
	uchar s_prealloc_blocks;		/* nmber of blocks to try to preallocate. */
	uchar s_prealloc_dir_blocks;	/* number to preallocate for dirs. */
	ushort s_padding1;

	/* Journaling Support */
	uchar s_journal_uuid[16];		/*  */
	uint s_journal_inum;			/*  */
	uint s_journal_dev;				/*  */
	uint s_last_orphan;				/*  */

	/* ꡼¸ȼ */
	int blockSize;					/* ֥å */
	int sectorSize;					/*  */
	int sectors;					/* 1֥åΥ */
}SUPER_BLOCK;

/* groupe descriptor. */
typedef struct{
	uint bg_block_bitmap;			/* 0,the first block of the "block bitmap". */
	uint bg_inode_bitmap;			/* 4,the first block of the "inode bitmap". */
	uint bg_inode_table;			/* 8,the first block of the "inode table". */
	ushort bg_free_blocks_count;	/* c,total number of free blocks. */
	ushort bg_free_inodes_count;	/* e,total number of free inodes. */
	ushort bg_used_dirs_count;		/* 10,number of inodes allocated to directories. */
	ushort bg_pad;
	uint bg_reserved[3];
}GROUP_DESC;

/*================================== ѥ֥å ========================================*/

/* ӥåȥޥåפǥХ֥ͤåΰ֤롣 */
static const uchar emptyBit[]={
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,7,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,0,
};

// ޥȻ
static SUPER_BLOCK *superBlock[MAX_DEVICE_OPEN];
static GROUP_DESC *grpDesc[MAX_DEVICE_OPEN];
static int grpDescNum[MAX_DEVICE_OPEN];			// 롼ץǥץ
static int grpDescBlocks[MAX_DEVICE_OPEN];		// 롼ץǥץ֥å

STATIC INLINE int EXT2_BLOCK_SIZE(
	void *devDsc)
{
	return superBlock[getDevInodeNum(devDsc)]->blockSize;
}

STATIC INLINE int EXT2_SECTOR_NUM(
	void *devDsc)
{
	return superBlock[getDevInodeNum(devDsc)]->sectors;
}

STATIC INLINE int EXT2_SECTOR_SIZE(
	void *devDsc)
{
	return superBlock[getDevInodeNum(devDsc)]->sectorSize;
}

/*
 * ֥å饻򻻽Ф롣
 * parameters : device inode,block
 * return : secter number
 */
STATIC INLINE uint blkToSect(
	void *devDsc,
	uint blk)
{
	return EXT2_SECTOR_NUM(devDsc) * blk;
}

/*
 * ѡ֥åȥ롼ץǥץǥ˽᤹
 *ԳԲġ
 * return : 0 or error=-1
 */
STATIC int writeBackSbGd(
	void *devDsc)
{
	const int din = getDevInodeNum(devDsc);
	SUPER_BLOCK *sb = superBlock[din];
	int sectors = sb->sectors;
	void *buf;

	/* 롼ץǥץǥ˽᤹ */
	BlockCacheWrite(
		getDevInfo(devDsc), grpDesc[din], grpDescBlocks[din] * sectors, blkToSect(devDsc, EXT2_GROUP_DESC_BLK + sb->s_first_data_block));
	BlockCacheWrite(getDevInfo(devDsc), grpDesc[din], grpDescBlocks[din] * sectors,
		blkToSect(devDsc, EXT2_GROUP_DESC_BLK + sb->s_first_data_block + sb->s_blocks_per_group));

	/* ѡ֥åǥ˽᤹ */
	buf = kmalloc(sb->blockSize);
	if (buf==NULL){
		return -1;
	}
	/*BlockCacheRead(getDevInfo(devDsc), buf, sectors, blkToSect(devDsc, sb->s_first_data_block));*/
	memset(buf, 0, sb->blockSize);
	if (sb->s_first_data_block == 1){
		memcpy(buf, sb, sizeof(SUPER_BLOCK));
	}
	else{
		memcpy((char*)buf + EXT2_SUPER_BLK_POSITION, sb, sizeof(SUPER_BLOCK));
	}
	BlockCacheWrite(getDevInfo(devDsc), buf, sectors, blkToSect(devDsc, sb->s_first_data_block));
	if (sb->s_first_data_block == 0){
		memset((char*)buf + EXT2_SUPER_BLK_POSITION, 0,sizeof(SUPER_BLOCK));
		memcpy(buf, sb, sizeof(SUPER_BLOCK));
	}
	BlockCacheWrite(getDevInfo(devDsc), buf, sectors, blkToSect(devDsc, sb->s_first_data_block + sb->s_blocks_per_group));
	kfree(buf);

	return 0;
}

/***********************************************************************
 *
 * Block
 *
 ***********************************************************************/

//================================== PRIVATE ============================================

/* ӥåȥޥå׹¤Ρ */
typedef struct{
	uchar *bitmap;	/* EXT2롼ץ֥åӥåȥޥåץХåե */
	int group;		/* ߤΥ롼 */
	int offset;		/* ߤΥӥåȥޥåפΥХȥեåȡ */
}BLOCK_BITMAP;

static BLOCK_BITMAP blockBitmap[MAX_DEVICE_OPEN];
static int blkLock[MAX_DEVICE_OPEN];			// ֥åԥå

/*
 * KХȥ֥å
 */
STATIC INLINE int is4Kbyte(
	void *devDsc)
{
	return EXT2_BLOCK_SIZE(devDsc) / 4096;
}

/*
 * ֥å롣
 *ԳԲġ
 * parameters : device inode
 * return : block number or 0
 */
STATIC u_int32_t getBlock(
	void *devDsc,
	int *o_error)
{
	const int din = getDevInodeNum(devDsc);
	GROUP_DESC *grpDsc = grpDesc[din];
	SUPER_BLOCK *superBlk = superBlock[din];
	int block = 0;

	enter_spinlock(&blkLock[din]);
	{
		int group;
		int offset;
		int last;

		// ߤΥ롼פ˶֥åʤ
		if (grpDsc[blockBitmap[din].group].bg_free_blocks_count == 0){
			int sectors = superBlk->sectors;

			// ߤΥӥåȥޥåפ᤹
			*o_error = BlockCacheWrite(
				getDevInfo(devDsc), blockBitmap[din].bitmap, sectors, blkToSect(devDsc, grpDsc[blockBitmap[din].group].bg_block_bitmap));
			if (*o_error != NOERR){
				goto EXIT;
			}

			// ֥åΤ륰롼פõ
			for (group = blockBitmap[din].group + 1;; ++group){
				if (group == grpDescNum[din]){
					group = 0;
				}
				if (group == blockBitmap[din].group){
					*o_error = -ENOSPC;
					goto EXIT;
				}
				if (0 < grpDsc[group].bg_free_blocks_count){
					break;
				}
			}
			*o_error = BlockCacheRead(getDevInfo(devDsc), blockBitmap[din].bitmap, sectors, blkToSect(devDsc, grpDsc[group].bg_block_bitmap));
			if (*o_error != NOERR){
				goto EXIT;
			}

			// ֥åɤʤХ֥å֤
			blockBitmap[din].group = group;
			blockBitmap[din].offset = 0;
		}

		/* ӥåȥޥåפθ */
		last = ROUNDUP_DIV(superBlk->s_blocks_per_group , CHAR_BIT);
		for (offset = blockBitmap[din].offset; blockBitmap[din].bitmap[offset] == 0xff;) {
			if (last == ++offset){
				offset = 0;
			}

			// ӥåȥޥåפϤ
			ASSERT(blockBitmap[din].offset != offset)
		}
		blockBitmap[din].offset = offset;
		block = offset * CHAR_BIT + emptyBit[blockBitmap[din].bitmap[offset]] ;
		blockBitmap[din].bitmap[offset] |= 1 << emptyBit[blockBitmap[din].bitmap[offset]];
		
		/* ֥åȤ򸺤餹 */
		--grpDsc[blockBitmap[din].group].bg_free_blocks_count;
		--superBlk->s_free_blocks_count;
		
		block += blockBitmap[din].group * superBlk->s_blocks_per_group + superBlk->s_first_data_block;
	}
EXIT:
	exit_spinlock(&blkLock[din]);

	*o_error = NOERR;
	return block;
}

/*
 * ֥å롣
 *ԳԲġ
 * parameters : device inode,block number
 */
STATIC void releaseBlock(
	void *devDsc,
	uint block)
{
	const int din = getDevInodeNum(devDsc);
	SUPER_BLOCK *sb = superBlock[din];
	int sectors = sb->sectors;
	int group;
	int a;

	// ֥åӥåȥޥåפθ
	group = (block - sb->s_first_data_block) / sb->s_blocks_per_group;
	enter_spinlock(&blkLock[din]);
	{
		if (group != blockBitmap[din].group) {
			if (BlockCacheWrite(getDevInfo(devDsc), blockBitmap[din].bitmap, sectors,
				blkToSect(devDsc, grpDesc[din][blockBitmap[din].group].bg_block_bitmap)) != NOERR){
				return;
			}
			if (BlockCacheRead(
				getDevInfo(devDsc), blockBitmap[din].bitmap, sectors, blkToSect(devDsc, grpDesc[din][group].bg_block_bitmap)) != NOERR){
				return;
			}
			blockBitmap[din].group = group;
		}

		/* ӥåȥޥåפ˽񤭹ࡣ */
		a = block - group * sb->s_blocks_per_group - sb->s_first_data_block;
		blockBitmap[din].bitmap[a / CHAR_BIT] &= ~(1 << a % CHAR_BIT);

		/* ֥åȤ䤹 */
		++grpDesc[din][group].bg_free_blocks_count;
		++sb->s_free_blocks_count;
	}
	exit_spinlock(&blkLock[din]);
}

/*
 * ޥȻν
 * return : groupe descriptor number or error number
 */
STATIC int initBlock(
	void *devDsc)
{
	const int din = getDevInodeNum(devDsc);
	SUPER_BLOCK *sb = superBlock[din];
	GROUP_DESC *gd = grpDesc[din];
	size_t ioSize;
	int group;
	int error;

	blockBitmap[din].bitmap = kmalloc(sb->blockSize);
	if (blockBitmap[din].bitmap == NULL){
		return -ENOMEM;
	}

	// ֥å˶Τ륰롼פΥӥåȥޥåפɤ߽Ф
	for (group = 0; gd[group].bg_free_blocks_count == 0; ++group){
		if(group == grpDescNum[din]){
			return -ENOSPC;
		}
	}
	error = read_direct(getDevInfo(devDsc), blockBitmap[din].bitmap, sb->sectors, blkToSect(devDsc, gd[group].bg_block_bitmap), &ioSize);
	if (error != NOERR){
		return error;
	}
	blockBitmap[din].group = group;
	blockBitmap[din].offset = 0;

	return blockBitmap[din].group;
}


/*
 * ޥȻν
 *ԳԲġ
 * parameters : device inode
 */
STATIC void umountBlock(
	void *devDsc)
{
	const int din = getDevInodeNum(devDsc);

	BlockCacheWrite(
		getDevInfo(devDsc), blockBitmap[din].bitmap, EXT2_SECTOR_NUM(devDsc), 
		blkToSect(devDsc, grpDesc[din][blockBitmap[din].group].bg_block_bitmap));
	kfree(blockBitmap[din].bitmap);
}

/***********************************************************************
 *
 * Inode
 *
 ***********************************************************************/

enum{
	/* at i_mode */
	EXT2_S_IFMT		= 0xF000,	/* format mask */
	EXT2_S_IFSOCK	= 0xC000,	/* socket */
	EXT2_S_IFLNK	= 0xA000,	/* symbolic link */
	EXT2_S_IFREG	= 0x8000,	/* regular file */
	EXT2_S_IFBLK	= 0x6000,	/* block device */
	EXT2_S_IFDIR	= 0x4000,	/* directory */
	EXT2_S_IFCHR	= 0x2000,	/* character device */
	EXT2_S_IFIFO	= 0x1000,	/* fifo */
	EXT2_S_ISUID	= 0x0800,	/* SUID */
	EXT2_S_ISGID	= 0x0400,	/* SGID */
	EXT2_S_ISVTX	= 0x0200,	/* sticky bit */
	EXT2_S_IRWXU	= 0x01C0,	/* user access rights mask */
	EXT2_S_IRUSR	= 0x0100,	/* read */
	EXT2_S_IWUSR	= 0x0080,	/* write */
	EXT2_S_IXUSR	= 0x0040,	/* execute */
	EXT2_S_IRWXG	= 0x0038,	/* group access rights mask */
	EXT2_S_IRGRP	= 0x0020,	/* read */
	EXT2_S_IWGRP	= 0x0010,	/* write */
	EXT2_S_IXGRP	= 0x0008,	/* execute */
	EXT2_S_IRWXO	= 0x0007,	/* others access rights mask */
	EXT2_S_IROTH	= 0x0004,	/* read */
	EXT2_S_IWOTH	= 0x0002,	/* write */
	EXT2_S_IXOTH	= 0x0001,	/* execute */

	/* behaviour control flags at "i_flags". */
	EXT2_SECRM_FL		= 0x00000001,	/* secure deletion */
	EXT2_UNRM_FL		= 0x00000002,	/* record for undelete */
	EXT2_COMPR_FL		= 0x00000004,	/* compressed file */
	EXT2_SYNC_FL		= 0x00000008,	/* synchronous updates */
	EXT2_IMMUTABLE_FL	= 0x00000010,	/* immutable file */
	EXT2_APPEND_FL		= 0x00000020,	/* append only */
	EXT2_NODUMP_FL		= 0x00000040,	/* do not dump/delete file */
	EXT2_NOATIME_FL		= 0x00000080,	/* do not update .i_atime */
	EXT2_DIRTY_FL		= 0x00000100,	/* dirty (file is in use?) */
	EXT2_COMPRBLK_FL	= 0x00000200,	/* compressed blocks */
	EXT2_NOCOMPR_FL		= 0x00000400,	/* access raw compressed data */
	EXT2_ECOMPR_FL		= 0x00000800,	/* compression error */
	EXT2_BTREE_FL		= 0x00010000,	/* b-tree format directory */
	EXT2_INDEX_FL		= 0x00010000,	/* Hash indexed directory */

	EXT2_ROOT_INODE_INDEX = 2,			/* 롼inodeΥǥå */

	/* indirect data block index. */
	INODE_DATA_INDIRECT1	= 12,		/* ܻȥ֥å͡ */
	INODE_DATA_INDIRECT2	= 13,		/* Ŵܻȥ֥å͡ */
	INODE_DATA_INDIRECT3	= 14,		/* Ŵܻȥ֥å͡ */

	INODE_DATA_BLOCK_NUM	= 15,		/* ǡ֥å */
	INODE_DIRECT_BLOCK_NUM	= INODE_DATA_INDIRECT1,	/* ľܥ֥å */

	INODE_BLOCK_SIZE	= 512,			/* inodei_blocksΥ֥åñ̥Хȿ */
};

typedef struct{
	ushort i_mode;			/* 00,format of the described file and the access rights. */
	ushort i_uid;			/* 02,user id. */
	uint i_size;			/* 04,size of the file in bytes. */
	uint i_atime;			/* 08,Unix time,last time this file was accessed. */
	uint i_ctime;			/* 0c,Unix time,when file was created. */
	uint i_mtime;			/* 10,Unix time,last time this file was modified. */
	uint i_dtime;			/* 14,Unix time,when file was deleted. */
	ushort i_gid;			/* 18,group id. */
	ushort i_links_count;	/* 1a,how many times this inode is linked. */
	uint i_blocks;			/* 1c,both currently in used and currently reserved blocks.number of 512 bytes block. */
	uint i_flags;			/* 20,behave when accessing the data for this inode. */
	uint i_number;			/* 24,OS dependent 1,inode number(open) */
	/*
	 * data block number.
	 * 112 direct block number.
	 * 13 indirect block number.
	 * 14 bi-indirect block number.
	 * 15 tri-indirect block number.
	 * Each indirect block array contains as many entries of 32bit block numbers as possible
	 * (to fill one entire block).
	 */
	uint i_block[INODE_DATA_BLOCK_NUM];		/* 28 */
	uint i_generation;		/* 64,file version (used by NFS). */
	uint i_file_acl;		/* 68,block number containing the extended attributes, */
	uint i_dir_acl;			/* 6c,"high size" of the file. */
	uint i_faddr;			/* 70,location of the last file fragment. */
	uint i_osd2[3];			/* 74,OS dependent 2 */
}INODE;

typedef struct{
	const char *path;
	INODE *inodeData;
	INODE *inodeBuf;
}INODE_PARAM;

static BLOCK_BITMAP inodeBitmap[MAX_DEVICE_OPEN];

/*
 * inodeֹ椫inodeΥ֥åХåեɤ߹ࡣ
 *ԳԲġ
 * return : error number
 */
STATIC int readInode(
	void *devDsc, 
	uint inumber, 
	INODE *sectorBuf,
	INODE **o_inode)
{
	const int din = getDevInodeNum(devDsc);
	int gdesc;
	int index;
	uint sector;
	int error;

	gdesc = (inumber - 1) / superBlock[din]->s_inodes_per_group;
	index = (inumber - 1) % superBlock[din]->s_inodes_per_group;
	sector = blkToSect(devDsc, grpDesc[din][gdesc].bg_inode_table) + index * sizeof(INODE) / EXT2_SECTOR_SIZE(devDsc);
	error = BlockCacheRead(getDevInfo(devDsc), sectorBuf, 1, sector);
	if (error != NOERR){
		return error;
	}

	*o_inode = sectorBuf + (index % (EXT2_SECTOR_SIZE(devDsc) / sizeof(INODE)));
	return NOERR;
}

/*
 * inodeǥ˽񤭹ࡣ
 *ԳԲġ
 * return : error number
 */
STATIC int writeInode(
	void *devDsc,
	uint inumber,
	INODE *sectorBuf)
{
	const int din = getDevInodeNum(devDsc);
	int gdesc;
	int index;
	uint sector;
	int rest;

	gdesc = (inumber - 1) / superBlock[din]->s_inodes_per_group;
	index = (inumber - 1) % superBlock[din]->s_inodes_per_group;
	sector = blkToSect(devDsc,grpDesc[din][gdesc].bg_inode_table) + index * sizeof(INODE) / EXT2_SECTOR_SIZE(devDsc);
	rest = BlockCacheWrite(getDevInfo(devDsc), sectorBuf, 1, sector);
	if (rest != NOERR) {
		return rest;
	}

	return NOERR;
}

/*
 * inodeֹ롣
 *ԳԲġ
 * return : inode number or error = 0
 */
STATIC uint getNewInode(
	void *devDsc,
	int type)
{
	const int din = getDevInodeNum(devDsc);
	GROUP_DESC *gd = grpDesc[din];
	SUPER_BLOCK *sb = superBlock[din];
	uint inode;
	int i, last;

	/* ߤΥ롼פ˶inodeʤ */
	if (gd[inodeBitmap[din].group].bg_free_inodes_count == 0){
		int sectors=sb->sectors;

		/* ߤΥӥåȥޥåפ᤹ */
		if (BlockCacheWrite(
			getDevInfo(devDsc), inodeBitmap[din].bitmap, sectors, blkToSect(devDsc, gd[inodeBitmap[din].group].bg_inode_bitmap)) != NOERR){
			return 0;
		}

		for (i = inodeBitmap[din].group + 1;; ++i)
		{
			if (i == grpDescNum[din]){
				i = 0;
			}
			if (i == inodeBitmap[din].group){
				return 0;
			}
			if (gd[i].bg_free_inodes_count != 0){
				break;
			}
		}
		if (BlockCacheRead(getDevInfo(devDsc), inodeBitmap[din].bitmap, sectors, blkToSect(devDsc, gd[i].bg_inode_bitmap)) != NOERR){
			return 0;
		}
		inodeBitmap[din].group = i;
	}

	/* ӥåȥޥåפθ */
	i = inodeBitmap[din].offset;
	last = sb->s_inodes_per_group / CHAR_BIT;
	while (inodeBitmap[din].bitmap[i] == 0xff) {
		if (++i >= last) {
			i=0;
		}
	}
	inodeBitmap[din].offset=i;
	inode=i*CHAR_BIT+emptyBit[inodeBitmap[din].bitmap[i]];
	inodeBitmap[din].bitmap[i]|=1<<emptyBit[inodeBitmap[din].bitmap[i]];

	/* 롼ץǥץȥѡ֥å򹹿롣 */
	--gd[inodeBitmap[din].group].bg_free_inodes_count;
	--sb->s_free_inodes_count;
	
	/* 롼ץǥץΥǥ쥯ȥꥫȤ䤹 */
	if ((type & EXT2_S_IFMT) == EXT2_S_IFDIR)
		grpDesc[din][inodeBitmap[din].group].bg_used_dirs_count += 1;

	return inode+inodeBitmap[din].group*sb->s_inodes_per_group+1;
}

/*
 * inodeӥåȥޥåפγ롼ץǥץѹ
 *ԳԲġ
 * return : error number
 */
STATIC int releaseInodeBitAndGrp(
	void *devDsc, 
	int inumber, 
	int type)
{
	const int din = getDevInodeNum(devDsc);
	uchar *bitmap;
	int grp;
	SUPER_BLOCK *sb = superBlock[din];
	int sectors = sb->sectors;
	int a, rest;

	/* ֥åӥåȥޥåפθ */
	grp = (inumber - 1) / (sb->s_inodes_per_group);
	if (grp != inodeBitmap[din].group){
		if ((bitmap = kmalloc(sb->blockSize)) == NULL){
			return -ENOMEM;
		}
		rest = BlockCacheRead(getDevInfo(devDsc), bitmap, sectors, blkToSect(devDsc, grpDesc[din][grp].bg_inode_bitmap));
		if (rest != NOERR){
			return rest;
		}
	}
	else{
		bitmap = inodeBitmap[din].bitmap;
	}

	/* ӥåȥޥåפ˽񤭹ࡣ */
	a = inumber - grp * sb->s_inodes_per_group - 1;
	bitmap[a / CHAR_BIT] &= ~(1 << a % CHAR_BIT);
	if (grp != inodeBitmap[din].group)
	{
		/* ߤΥ֥å롼פȰפ뤫 */
		if (grp == blockBitmap[din].group)
		{
			uchar *buf;

			buf = inodeBitmap[din].bitmap;
			inodeBitmap[din].bitmap = bitmap;
			bitmap = buf;
			inodeBitmap[din].group = grp;
			inodeBitmap[din].offset = a / CHAR_BIT;
		}
		rest = BlockCacheWrite(getDevInfo(devDsc), bitmap, sectors, blkToSect(devDsc, grpDesc[din][grp].bg_inode_bitmap));
		if (rest != NOERR){
			return rest;
		}
		kfree(bitmap);
	}

	/* ֥åȤ䤹 */
	++grpDesc[din][grp].bg_free_inodes_count;
	++sb->s_free_inodes_count;

	/* ǥ쥯ȥʤǥ쥯ȥꥫȤ򸺤餹 */
	if ((type & EXT2_S_IFMT) == EXT2_S_IFDIR)
		--grpDesc[din][grp].bg_used_dirs_count;

	return NOERR;
}

/*
 * ޥȻν
 *ԳԲġ
 * return : error nomber
 */
STATIC int initInode(
	void *devDsc, 
	int groupNum)	// 롼ץǥץֹ
{
	const int din = getDevInodeNum(devDsc);
	GROUP_DESC *gd = grpDesc[din];
	SUPER_BLOCK *sb = superBlock[din];
	INODE *inode;
	INODE *sectBuf;
	size_t ioSize;
	int i;
	int error;

	// inodeӥåȥޥåפɤ߹
	inodeBitmap[din].bitmap = kmalloc(sb->blockSize);
	if (inodeBitmap[din].bitmap == NULL) {
		return -ENOMEM;
	}
	inodeBitmap[din].group = groupNum;
	inodeBitmap[din].offset = 0;

	i = groupNum;
	do {
		if (0 < gd[i].bg_free_inodes_count) {
			error = read_direct(getDevInfo(devDsc), inodeBitmap[din].bitmap, sb->sectors, blkToSect(devDsc, gd[i].bg_inode_bitmap), &ioSize);
			if (error != NOERR) {
				return error;
			}
			inodeBitmap[din].group = i;
			for (i = 0; inodeBitmap[din].bitmap[i] == 0xff; ++i);
			inodeBitmap[din].offset = i;
			break;
		}
		if (++i == grpDescNum[din]) {
			i = 0;
		}
	} while (i != groupNum);

	/* 롼inodeν */
	sectBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (sectBuf == NULL) {
		return -ENOMEM;
	}
	error = readInode(devDsc, EXT2_ROOT_INODE_INDEX, sectBuf, &inode);
	if (error != NOERR) {
		goto END;
	}
	inode->i_number = EXT2_ROOT_INODE_INDEX;
	error = writeInode(devDsc, EXT2_ROOT_INODE_INDEX, sectBuf);
	if (error != NOERR) {
		goto END;
	}
END:
	kfree(sectBuf);
	return error;
}

/*
 * ޥȻν
 *ԳԲġ
 */
STATIC int umountInode(
	void *devDsc)
{
	const int din = getDevInodeNum(devDsc);

	/* inodeӥåȥޥåפǥ˽᤹ */
	BlockCacheWrite(
		getDevInfo(devDsc), inodeBitmap[din].bitmap, EXT2_SECTOR_NUM(devDsc), 
		blkToSect(devDsc, grpDesc[din][inodeBitmap[din].group].bg_inode_bitmap));
	kfree(inodeBitmap[din].bitmap);

	return 0;
}

STATIC INLINE u_int32_t getSize(const INODE *inode)
{
	return inode->i_size;
}

//====================================== IO ========================================================

enum {
	INDIRECT_MAX = 4		// ܥ֥åΡ
};

typedef struct {
	u_int32_t blk[INDIRECT_MAX];	// ֥åֹ
	void *blkBuf[INDIRECT_MAX];		// ֥åХåե
	int upload[INDIRECT_MAX];		// ե饰1 =  or 0
} BLK_CACHE;

/*
 * ܥ֥å
 */
STATIC INLINE uint getIndirectNode(
	void *devDsc,
	const int node)		// ܥΡɿ
{
	static uint blocks[2][4] =	{
		{INODE_DIRECT_BLOCK_NUM, 256,  256  * 256,  256  * 256  * 256},
		{INODE_DIRECT_BLOCK_NUM, 1024, 1024 * 1024, 1024 * 1024 * 1024}
	};

	return blocks[is4Kbyte(devDsc)][node];
}

/*
 * ǡ֥åå
 */
STATIC void initBlkCache(BLK_CACHE *blkCache)
{
	memset(blkCache, 0, sizeof(*blkCache));
}

/*
 * ǡ֥åå
 * ܥΡɤȤ˹򤹤
 *ԳԲġ
 * return : ֥åХåե or NULL
 */
STATIC void *getBlkCache(
	void *devDsc,
	const u_int32_t i_blk,
	const int node,
	BLK_CACHE *blkCache,
	int *o_error)
{
	ASSERT(node < INDIRECT_MAX);

	if (blkCache->blk[node] == i_blk) {
		*o_error = NOERR;
		return blkCache->blkBuf[node];
	}
	else {
		if (blkCache->blk[node] != 0) {
			if (blkCache->upload[node] == 1) {
				// ߤΥ֥ååϥǥ˽᤹
				*o_error = BlockCacheWrite(
					getDevInfo(devDsc), blkCache->blkBuf[node], EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, blkCache->blk[node]));
				if (*o_error != NOERR) {
					return NULL;
				}
			}
		}
		else {
			// Хåե
			blkCache->blkBuf[node] = kmalloc(EXT2_BLOCK_SIZE(devDsc));
			if (blkCache->blkBuf[node] == NULL) {
				*o_error = -ENOMEM;
				return NULL;
			}
		}

		// ǥɤ߹
		*o_error = BlockCacheRead(getDevInfo(devDsc), blkCache->blkBuf[node], EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, i_blk));
		if (*o_error != NOERR) {
			return NULL;
		}
		blkCache->blk[node] = i_blk;
		blkCache->upload[node] = 0;
		
		*o_error = NOERR;
		return blkCache->blkBuf[node];
	}
}

/*
 * ǡ֥åå򿷤Ƥ
 * ֥åե饰ΩƤ
 *ԳԲġ
 * return : ֥åХåե or NULL
 */
STATIC void *getNewBlkCache(
	void *devDsc,
	const u_int32_t i_blk,
	const int node,
	BLK_CACHE *blkCache,
	int *o_error)
{
	ASSERT(node < INDIRECT_MAX);

	if (blkCache->blk[node] != 0) {
		if (blkCache->upload[node] == 1) {
			// ߤΥ֥ååϥǥ˽᤹
			*o_error = BlockCacheWrite(
				getDevInfo(devDsc), blkCache->blkBuf[node], EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, blkCache->blk[node]));
			if (*o_error != NOERR) {
				return NULL;
			}
		}
	}
	else {
		// ǽΥХåե
		blkCache->blkBuf[node] = kmalloc(EXT2_BLOCK_SIZE(devDsc));
		if (blkCache->blkBuf[node] == NULL) {
			*o_error = -ENOMEM;
			return NULL;
		}
	}

	memset(blkCache->blkBuf[node], 0, EXT2_BLOCK_SIZE(devDsc));
	blkCache->blk[node] = i_blk;
	blkCache->upload[node] = 0;

	*o_error = NOERR;
	return blkCache->blkBuf[node];
}

/*
 * ǡ֥åå峫
 *ԳԲġ
 * return : error number
 */
STATIC int freeBlkCache(
	void *devDsc,
	BLK_CACHE *blkCache)
{
	int node;
	int error;
	
	for (node = 0; node < INDIRECT_MAX; ++node) {
		if (blkCache->blk[node] != 0) {
			if (blkCache->upload[node] == 1) {
				error = BlockCacheWrite(
					getDevInfo(devDsc), blkCache->blkBuf[node], EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, blkCache->blk[node]));
				if (error != NOERR) {
					for (; node < INDIRECT_MAX; ++node) {
						kfree(blkCache->blkBuf[node]);
					}
					return error;
				}
			}

			kfree(blkCache->blkBuf[node]);
		}
	}
	
	return NOERR;
}

/*
 * ǡ֥åιե饰ΩƤ
 *աwriteɬƤӽФ
 * return : error number
 */
STATIC void updateBlk(
	const int node,
	BLK_CACHE *blkCache)
{
	blkCache->upload[node] = 1;
}

/*
 * եåȤΥ֥åǡ
 *ա̤֥åƤΤǡREADǥե륵ĶƸƤӽФʤȡ
 *ԳԲġ
 * return : ֥åǡ or NULL
 */
STATIC char *getBlkData(
	void *devDsc,				// device inode
	const int64_t i_offset,		// ǡХȥեå
	INODE *inode,				// inode
	BLK_CACHE *blkCache,		// ǡ֥åå幽¤
	int *o_error)
{
	uint blkNum = i_offset / EXT2_BLOCK_SIZE(devDsc);	// ֥åֹ
	int node;											// ܥΡɿ

	// ܥΡɿȳΡɤǤХ֥å򻻽
	node = 0;
	for (; getIndirectNode(devDsc, node) <= blkNum; ++node) {
		if (node == INDIRECT_MAX) {
			*o_error = -EFBIG;
			return NULL;
		}
		blkNum -= getIndirectNode(devDsc, node);
	}

	// ܥ֥å򤿤ɤäƥǡ֥åХåե
	{
		u_int32_t *blkBuf;
		int array;

		blkBuf = inode->i_block;
		array = (node == 0) ? blkNum : INODE_DATA_INDIRECT1 + node - 1;
		for (;;) {
			if (blkBuf[array] == 0) {
				// ֥å򿷤˳Ƥ
				blkBuf[array] = getBlock(devDsc, o_error);
				if (blkBuf[array] == 0) {
					return NULL;
				}
				
				blkBuf = getNewBlkCache(devDsc, blkBuf[array], node, blkCache, o_error);
				if (*o_error != NOERR) {
					return NULL;
				}
			}
			else {
				blkBuf = getBlkCache(devDsc, blkBuf[array], node, blkCache, o_error);
				if (*o_error != NOERR) {
					return NULL;
				}
			}
			
			if (node == 0) {
				break;
			}

			if (--node == 0) {
				array = blkNum;
			}
			else {
				array = blkNum / getIndirectNode(devDsc, node);
				blkNum %= getIndirectNode(devDsc, node);
			}

			if (blkBuf[array] == 0) {
				updateBlk(node + 1, blkCache);
			}
		}

		*o_error = NOERR;
		return (char*)blkBuf;
	}
}

/*
 * ǡ֥åɤ
 *ԳԲġ
 * return : error number
 */
STATIC int readData(
	void *devDsc,
	const u_int32_t i_offset,	// ϥХȥեå
	const int64_t i_bytes,		// ɤ߹ߥХȿ
	char *m_readBuf,			// ɤ߹ߥХåե
	INODE *inode)				// եinode
{
	BLK_CACHE blkCache;
	char *readBuf = m_readBuf;
	char *blkData;
	int64_t offset;
	int64_t last;
	size_t readBytes;
	int error;

	initBlkCache(&blkCache);
	
	// ǽΥ֥å̤ɤ
	blkData = getBlkData(devDsc, i_offset, inode, &blkCache, &error);
	if (error != NOERR) {
		goto END;
	}
	readBytes = ROUNDUP(i_offset, EXT2_BLOCK_SIZE(devDsc)) - i_offset;
	if (i_bytes < readBytes) {
		readBytes = i_bytes;
	}
	memcpy(readBuf, blkData + (i_offset % EXT2_BLOCK_SIZE(devDsc)), readBytes);
	readBuf += readBytes;

	// ǸΥ֥åޤɤ
	last = ROUNDDOWN(i_offset + i_bytes, EXT2_BLOCK_SIZE(devDsc));
	offset = ROUNDUP(i_offset, EXT2_BLOCK_SIZE(devDsc));
	for (; offset < last; offset += EXT2_BLOCK_SIZE(devDsc)) {
		blkData = getBlkData(devDsc, offset, inode, &blkCache, &error);
		if (error != NOERR) {
			goto END;
		}
		memcpy(readBuf, blkData, EXT2_BLOCK_SIZE(devDsc));
		readBuf += EXT2_BLOCK_SIZE(devDsc);
	}
		
	if (offset < i_offset + i_bytes) {
		// ǸΥ֥å̤񤭹
		blkData = getBlkData(devDsc, offset, inode, &blkCache, &error);
		if (error != NOERR) {
			goto END;
		}
		readBytes = i_offset + i_bytes - offset;
		memcpy(readBuf, blkData, readBytes);
	}
	
	error = NOERR;
END:
	freeBlkCache(devDsc, &blkCache);
	
	return error;
}

/*
 * ǡ֥å˽񤭹ࡣ
 *ԳԲġ
 * return : error number
 */
STATIC int writeData(
	void *devDsc,				// device
	const u_int32_t i_offset,	// ϥХȥեå
	const uint i_bytes,			// 񤭹ߥХȿ
	const void *i_writeBuf,		// 񤭹ߥХåե
	INODE *m_inode)				// inode
{
	BLK_CACHE blkCache;
	const char *writeBuf = i_writeBuf;
	char *blkData;
	int64_t offset;
	int64_t last;
	size_t writeBytes;
	int error;

	initBlkCache(&blkCache);
	
	// 񤭹ǽޤǤ̤ƥ֥åƤ
	if (ROUNDUP(getSize(m_inode), EXT2_BLOCK_SIZE(devDsc)) < ROUNDDOWN(i_offset, EXT2_BLOCK_SIZE(devDsc))) {
		last = ROUNDDOWN(i_offset, EXT2_BLOCK_SIZE(devDsc));
		for (offset = ROUNDUP(getSize(m_inode), EXT2_BLOCK_SIZE(devDsc)); offset < last; offset += EXT2_BLOCK_SIZE(devDsc)) {
			getBlkData(devDsc, offset, m_inode, &blkCache, &error);
			if (error != NOERR) {
				goto END;
			}
		}
	}
	
	// ǽΥ֥å̤񤭹
	blkData = getBlkData(devDsc, i_offset, m_inode, &blkCache, &error);
	if (error != NOERR) {
		goto END;
	}
	writeBytes = ROUNDUP(i_offset, EXT2_BLOCK_SIZE(devDsc)) - i_offset;
	if (i_bytes < writeBytes) {
		writeBytes = i_bytes;
	}
	memcpy(blkData + (i_offset % EXT2_BLOCK_SIZE(devDsc)), writeBuf, writeBytes);
	updateBlk(0, &blkCache);
	writeBuf += writeBytes;

	// ǸΥ֥åޤǽ񤭹
	last = ROUNDDOWN(i_offset + i_bytes, EXT2_BLOCK_SIZE(devDsc));
	offset = ROUNDUP(i_offset, EXT2_BLOCK_SIZE(devDsc));
	for (; offset < last; offset += EXT2_BLOCK_SIZE(devDsc)) {
		blkData = getBlkData(devDsc, offset, m_inode, &blkCache, &error);
		if (error != NOERR) {
			goto END;
		}
		memcpy(blkData, writeBuf, EXT2_BLOCK_SIZE(devDsc));
		updateBlk(0,&blkCache);
		writeBuf += EXT2_BLOCK_SIZE(devDsc);
	}
		
	if (offset < i_offset + i_bytes) {
		// ǸΥ֥å̤񤭹
		blkData = getBlkData(devDsc, offset, m_inode, &blkCache, &error);
		if (error != NOERR) {
			goto END;
		}
		writeBytes = i_offset + i_bytes - offset;
		memcpy(blkData, writeBuf, writeBytes);
		updateBlk(0, &blkCache);
	}
	
	error = NOERR;
END:
	freeBlkCache(devDsc, &blkCache);
	
	return error;
}

//====================================== release ========================================================

/*
 * ֥åγ
 *ԳԲġ
 * return : error number
 */
STATIC int releaseBlockTable(
	void *i_devDsc, 
	const int i_node,		// ܥ٥0 = ľܥ֥åơ֥
	uint *m_table[])
{
	uint *table = m_table[i_node];
	int last = EXT2_BLOCK_SIZE(i_devDsc) / sizeof(uint);
	int i;

	if (i_node == 0) {
		/*
		 * ľܥ֥å
		 */
		for (i = 0; i < last; ++i) {
			if (table[i] != 0) {
				releaseBlock(i_devDsc, table[i]);
			}
		}
	}
	else {
		/*
		 * ܥ֥å
		 */
		for (i = 0; i < last; ++i) {
			int error;

			if (table[i] != 0) {
				error = BlockCacheRead(getDevInfo(i_devDsc), m_table[i_node - 1], EXT2_SECTOR_NUM(i_devDsc), blkToSect(i_devDsc, table[i]));
				if (error != NOERR) {
					return error;
				}
				error = releaseBlockTable(i_devDsc, i_node - 1, m_table);
				if (error != NOERR) {
					return error;
				}
				releaseBlock(i_devDsc, table[i]);
			}
		}
	}

	return NOERR;
}

/*
 * ǡ֥å롣
 *ԳԲġ
 * return : error number
 */
STATIC int releaseDataBlock(
	void *i_devDsc,
	INODE *m_inode)
{
	uint *table[3];
	int tbl;
	int i;
	int error = NOERR;
	
	// ľܥ֥å
	for (i = 0; i < INODE_DATA_INDIRECT1; ++i) {
		if (m_inode->i_block[i] != 0) {
			releaseBlock(i_devDsc, m_inode->i_block[i]);
			m_inode->i_block[i] = 0;
		}
	}

	// ܥ֥å
	tbl = 0;
	for (i = INODE_DATA_INDIRECT1; i < INODE_DATA_BLOCK_NUM; ++i) {
		table[tbl] = kmalloc(EXT2_BLOCK_SIZE(i_devDsc));
		if (table[tbl] == NULL) {
			error =  -ENOMEM;
			break;
		}

		if (m_inode->i_block[i] != 0) {
			error = BlockCacheRead(
				getDevInfo(i_devDsc), table[tbl], EXT2_SECTOR_NUM(i_devDsc), blkToSect(i_devDsc, m_inode->i_block[i]));
			if (error != NOERR) {
				++tbl;
				break;
			}
			error = releaseBlockTable(i_devDsc, tbl, table);
			if (error != NOERR) {
				++tbl;
				break;
			}
			releaseBlock(i_devDsc, m_inode->i_block[i]);
			m_inode->i_block[i] = 0;
		}

		++tbl;
	}

	// ܥơ֥Хåեγ
	for (i = 0; i < tbl; ++i) {
		kfree(table[i]);
	}

	if (error == NOERR) {
		m_inode->i_size = 0;
	}

	return error;
}

/*
 * ̾եʤinode󥯥Ȥ򸺤餷ƥ󥯣ʤ
 * ǥ쥯ȥʤ󥯥ȣΤȤ
 *ԳԲġ
 * return : errnor number
 */
STATIC int unlinkInode(
	void *devDsc,
	INODE *m_inode)
{
	INODE *sectorBuf;
	INODE *inode;
	int error;
	
	if ((m_inode->i_mode & EXT2_S_IFMT) == EXT2_S_IFDIR) {	// ǥ쥯ȥ
		ASSERT(2 <= m_inode->i_links_count);

		if (2 == m_inode->i_links_count){
			m_inode->i_links_count = 0;
		}
		else {
			return -ENOTEMPTY;
		}
	}
	else {													// ¾Υե륿
		m_inode->i_links_count -= 1;

		ASSERT(0 <= (short)m_inode->i_links_count);
	}

	/* ǡ֥å롣 */
	error = releaseDataBlock(devDsc, m_inode);
	if (error != NOERR) {
		return error;
	}

	/* inodeΰ */
	error = releaseInodeBitAndGrp(devDsc, m_inode->i_number, m_inode->i_mode);
	if (error != NOERR) {
		return error;
	}

	/* delete timeꤹ */
	m_inode->i_dtime = (uint) sys_time(NULL);

	/* inode᤹ */
	sectorBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (sectorBuf == NULL){
		return -ENOMEM;
	}
	error = readInode(devDsc, m_inode->i_number, sectorBuf, &inode);
	if (error != NOERR){
		goto ERR;
	}
	memcpy(inode, m_inode, sizeof(INODE));
	error = writeInode(devDsc, m_inode->i_number, sectorBuf);
	if (error != NOERR) {
		goto ERR;
	}
	
	error = NOERR;
ERR:
	kfree(sectorBuf);

	return error;
}

/********************************************************************************************************
 *
 * Directory
 *
 ********************************************************************************************************/

enum{
	EXT2_NAME_LEN = 255,

	/* directory file types. */
	EXT2_DIR_UNKNOWN = 0,
	EXT2_DIR_REG_FILE,
	EXT2_DIR_DIR,
	EXT2_DIR_CHRDEV,
	EXT2_DIR_BLKDEV,
	EXT2_DIR_FIFO,
	EXT2_DIR_SOCK,
	EXT2_DIR_SYMLINK,
	EXT2_DIR_MAX,
};

typedef struct{
	uint inode;				/* inode number. */
	ushort rec_len;			/* directory entry length. */
	uchar name_len;			/* name length. */
	uchar file_type;		/* file type. */
	char name[0];			/* File name. */
}DIR_ENTRY;

typedef struct{
	const char *path;
	DIR_ENTRY *dirBuf;
	uint blkNumber;
}DIR_PARAM;

typedef int (*READ_DIR)(void*, uint*, int, DIR_PARAM*, int, int (**READ_DIR)());

#define ROUNDUP4(a) (((a+3)>>2)<<2)		/* 4Ƿ夲 */

//================================== PRIVATE ============================================

/*
 * search entry.
 * return : directory entry address or failed=NULL
 */
STATIC DIR_ENTRY *searchEntry(
	void *devDsc,
	const char *path,
	DIR_ENTRY *entry)		// entry buffer
{
	DIR_ENTRY *last;

	last = (DIR_ENTRY*)((char*)entry + EXT2_BLOCK_SIZE(devDsc));
	if (entry->inode == 0) {
		entry = (DIR_ENTRY*)((char*)entry + entry->rec_len);
	}

	for (;entry < last; entry = (DIR_ENTRY*)((char*)entry + entry->rec_len)) {
		if (pathCmpName(entry->name, path, entry->name_len) == 0) {
			return entry;
		}
	}

	return NULL;
}

/*
 * ȥ꡼õ
 * return : directory entry address or failed = NULL
 */
STATIC DIR_ENTRY *searchEntrySpace(
	void *devDsc,
	DIR_PARAM *dparam)
{
	int addSize = strlen(dparam->path) + sizeof(DIR_ENTRY);
	int entSize;
	DIR_ENTRY *entry;
	DIR_ENTRY *new;
	char *next;
	char *last;

	entry = dparam->dirBuf;
	if (entry->inode == 0) {
		entry->name_len = addSize - sizeof(DIR_ENTRY);
		return entry;
	}

	last = (char*) dparam->dirBuf + EXT2_BLOCK_SIZE(devDsc);
	for (next = (char*) dparam->dirBuf; next < last; next += entry->rec_len) {
		entry = (DIR_ENTRY*) next;
		entSize = ROUNDUP4(entry->name_len + sizeof(DIR_ENTRY));
		if (addSize <= (entry->rec_len - entSize)) {
			new = (DIR_ENTRY*) ((char*) entry + entSize);
			new->rec_len = entry->rec_len - entSize;		/* ȥ꡼ */
			new->name_len = addSize - sizeof(DIR_ENTRY);	/* ͡ॵ */
			entry->rec_len = entSize;
			return new;
		}
	}

	return NULL;
}

/*
 * ȥ꡼뤫å롣
 * return : 0 or ȥ꡼=-1
 */
STATIC int checkEntry(
	void *devDsc,
	DIR_PARAM *dparam)
{
	DIR_ENTRY *entry;

	entry=dparam->dirBuf;

	if (entry->inode == 0) {
		if(entry->rec_len == EXT2_BLOCK_SIZE(devDsc)) {
			return 0;
		}
	}
	else {
		if (pathCmpName(entry->name, ".", entry->name_len) == 0) {
			entry = (DIR_ENTRY*) ((char*) entry + entry->rec_len);
			if (entry->rec_len == EXT2_BLOCK_SIZE(devDsc) - (sizeof(DIR_ENTRY) + sizeof(int))) {
				return 0;
			}
		}
	}

	return -1;
}


/*
* ȥ꡼롣
 * parameters : delete entry,directory buffer
 */
STATIC INLINE void delEntry(DIR_ENTRY *entry,DIR_ENTRY *dirBuf)
{
	DIR_ENTRY *p;

	if (entry == dirBuf) {
		entry->inode = 0;
	}
	else {
		for (p = dirBuf; (char*) p + p->rec_len != (char*) entry; p = (DIR_ENTRY*) ((char*) p + p->rec_len));
		p->rec_len += entry->rec_len;
	}
}

/*
 * ľܥ֥åɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 *ԳԲġ
 * return : directory entry address or error number
 */
STATIC int readDirectDir(
	void *devDsc,
	uint *table,				// block table
	int blk_num,				// number of block table indexs
	DIR_PARAM *dparam,
	int node,					// node=0
	READ_DIR *readDirBlock)
{
	DIR_ENTRY *entry;
	int i;
	int error;

	for (i = 0; i < blk_num; ++i){
		error = BlockCacheRead(getDevInfo(devDsc), dparam->dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, table[i]));
		if (error != NOERR){
			return -EIO;
		}
		entry = searchEntry(devDsc, dparam->path, dparam->dirBuf);
		if (entry != NULL) {
			dparam->blkNumber = table[i];
			return (int)entry;
		}
	}

	return -ENOENT;
}

/*
 * ľܥ֥åɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 *ԳԲġ
 * return : directory entry address or error number
 */
STATIC int searchDirectDir(
	void *devDsc,
	uint *table,			// block table
	int blk_num,			// number of block table indexs
	DIR_PARAM *dparam,
	int node,				// node=0
	READ_DIR *readDirBlock)
{
	DIR_ENTRY *entry;
	int i;
	int error;

	for (i = 0; i < blk_num; ++i){
		error = BlockCacheRead(getDevInfo(devDsc), dparam->dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, table[i]));
		if (error != NOERR){
			return error;
		}
		entry = searchEntrySpace(devDsc, dparam);
		if (entry != NULL) {
			dparam->blkNumber = table[i];
			return (int)entry;
		}
	}

	return -ENOENT;
}

/*
 * ľܥ֥åɹߡ
 *ԳԲġ
 * return : -ENOENT = ʤ or error number
 */
STATIC int checkDirectDir(
	void *devDsc,
	uint *table,			// block table
	int blk_num,			// number of block table indexs
	DIR_PARAM *dparam,
	int node,				// node=0
	READ_DIR *readDirBlock)
{
	int i;
	int error;

	for (i = 0; i < blk_num; ++i) {
		error = BlockCacheRead(getDevInfo(devDsc), dparam->dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, table[i]));
		if (error != NOERR) {
			return error;
		}
		if (checkEntry(devDsc, dparam) == -1) {
			return -EEXIST;
		}
	}

	return -ENOENT;
}

/*
 * Υǥ쥯ȥ֥åõ
 * DIR_PARAMblkNumber0ʤǽΥ֥å0ʾʤblkNumberμΥ֥å֤
 * return : next block or error number
 */
STATIC int searchNextBlock(
	void *dummy,
	uint *table,			// block table
	int blk_num,			// number of block table indexs
	DIR_PARAM *dparam,
	int node,				// node=0
	READ_DIR *readDirBlock)
{
	int i;

	if (dparam->blkNumber == 0) {
		return *table;
	}
	else {
		for (i = 0; i < blk_num; ++i) {
			if (table[i] == dparam->blkNumber) {
				if (++i < blk_num) {
					return table[i];
				}
				dparam->blkNumber = 0;
				break;
			}
		}
	}

	return -ENOENT;
}

/*
 * ɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 *ԳԲġ
 * return : directory entry address of error number
 */
STATIC int readIndirectDir(
	void *devDsc,
	uint *table,			// block table
	int blk_num,			// number of block table indexs
	DIR_PARAM *dparam,
	int node,
	READ_DIR *readDirBlock)
{
	const int din = getDevInodeNum(devDsc);
	uint *buf;
	int nblk = 1 << ((EXT2_BLOCK_BASE_LOG + superBlock[din]->s_log_block_size - INT_LOG) * node);
	int rest;
	int i;

	buf = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (buf == NULL){
		return -ENOMEM;
	}

	for (i = 0; nblk < blk_num; blk_num -= nblk, ++i){
		rest = BlockCacheRead(getDevInfo(devDsc), buf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, table[i]));
		if (rest != NOERR){
			goto END;
		}
		rest = readDirBlock[node - 1](devDsc, buf, nblk, dparam, node - 1, readDirBlock);
		if (rest != -ENOENT){
			goto END;
		}
	}
	rest = BlockCacheRead(getDevInfo(devDsc), buf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, table[i]));
	if (rest != NOERR){
		goto END;
	}
	rest = readDirBlock[node - 1](devDsc, buf, blk_num, dparam, node - 1, readDirBlock);

END:
	kfree(buf);

	return rest;
}


/*
 * read directory.
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 *ԳԲġ
 * return : directory address or error number
 */
STATIC int readDir(
	void *devDsc, 
	INODE *inode, 
	DIR_PARAM *dparam, 
	READ_DIR *readDirBlock)
{
	uint *buf;
	int blk_num = ROUNDUP_DIV(inode->i_size, EXT2_BLOCK_SIZE(devDsc));
	int nblk;
	int rest;
	int blk, i;

	/* ľܥ֥å */
	blk = (blk_num >= INODE_DATA_INDIRECT1)? INODE_DATA_INDIRECT1 : blk_num;
	rest = readDirBlock[0](devDsc, inode->i_block, blk, dparam, 0, readDirBlock);
	if (rest != -ENOENT){
		return rest;
	}
	if ((blk_num -= blk) == 0){
		return -ENOENT;
	}

	/* ܥ֥å */
	buf = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (buf == NULL){
		return -ENOMEM;
	}
	nblk = EXT2_BLOCK_SIZE(devDsc) / sizeof(uint);
	blk = nblk;
	for (i = INODE_DATA_INDIRECT1; ; ++i){
		rest = BlockCacheRead(getDevInfo(devDsc), buf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, inode->i_block[i]));
		if (rest != NOERR){
			goto END;
		}
		rest = readDirBlock[i - INODE_DATA_INDIRECT1](devDsc, buf, (blk_num > blk)? blk : blk_num, dparam, i - INODE_DATA_INDIRECT1, readDirBlock);
		if (rest != -ENOENT){
			goto END;
		}
		if ((blk_num -= blk) <= 0){
			break;
		}
		blk *= nblk;
	}
END:
	kfree(buf);

	return rest;
}


static READ_DIR readDirBlock[] = {readDirectDir, readIndirectDir, readIndirectDir, readIndirectDir};
static READ_DIR addDirBlock[] = {searchDirectDir, readIndirectDir, readIndirectDir, readIndirectDir};
static READ_DIR checkDirBlock[] = {checkDirectDir, readIndirectDir, readIndirectDir, readIndirectDir};
static READ_DIR searchDirBlock[] = {searchNextBlock, readIndirectDir, readIndirectDir, readIndirectDir};

/*
 * ǥ쥯ȥ˥ȥ꡼ĤäƤ뤫ǧ롣
 *ԳԲġ
 * return : ȥ꡼ʤ=-ENOENT or error number
 */
STATIC int checkDir(
	void *devDsc,
	INODE *inode)	// directory inode
{
	DIR_PARAM param;
	int error;

	param.dirBuf = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (param.dirBuf == NULL) {
		return -ENOMEM;
	}
	error = readDir(devDsc, inode, &param, checkDirBlock);
	kfree(param.dirBuf);

	return error;
}

// inodeե륿פǥ쥯ȥե륿פѴ
STATIC uchar getDirFileTypeFromFileType(const ushort type)
{
	switch (type & EXT2_S_IFMT) {
	case EXT2_S_IFSOCK:
		return EXT2_DIR_SOCK;
	case EXT2_S_IFLNK:
		return EXT2_DIR_SYMLINK;
	case EXT2_S_IFREG:
		return EXT2_DIR_REG_FILE;
	case EXT2_S_IFBLK:
		return EXT2_DIR_BLKDEV;
	case EXT2_S_IFDIR:
		return EXT2_DIR_DIR;
	case EXT2_S_IFCHR:
		return EXT2_DIR_CHRDEV;
	case EXT2_S_IFIFO:
		return EXT2_DIR_FIFO;
	default:
		return EXT2_DIR_UNKNOWN;
	}
}

/*
 * ѥindoe
 *ԳԲġ
 * return : inodeinodeѥХåեݥ󥿡 or NULL
 */
STATIC INODE *getInodeFromPath(
	void *devDsc,			// ǥХǥץ
	const char *i_path,		// ѥ
	INODE *i_inode,			// ǥ쥯ȥinode
	INODE *m_inodeBuf,		// inodeѥХåե
	DIR_ENTRY *m_dirBuf,	// ǥ쥯ȥѥХåե
	const char **o_path,	// Υѥݥ
	int *o_error)			// error number
{
	INODE *retInode;
	DIR_PARAM dparam;
	DIR_ENTRY *entry;
	const char *path;
	int rest;

	/* ǥ쥯ȥꥨȥ꡼õ */
	dparam.path = i_path;
	dparam.dirBuf = m_dirBuf;
	rest = readDir(devDsc, i_inode, &dparam, readDirBlock);
	if (rest < 0) {
		*o_error = rest;
		return NULL;
	}
	entry = (DIR_ENTRY*) rest;

	/* inode */
	*o_error = readInode(devDsc, entry->inode, m_inodeBuf, &retInode);
	if (*o_error != NOERR) {
		return NULL;
	}
	retInode->i_number = entry->inode;

	/* '/'μ˥ѥʤ롣 */
	path = i_path + entry->name_len;
	for (; *path == '/'; ++path);
	*o_path = path;

	*o_error = NOERR;
	return retInode;
}

/*
 * ƥǥ쥯ȥޤǸơparam˼Υѥȥǥ쥯ȥinode롣
 *ԳԲġ
 * return : error number
 */
STATIC int searchParentDir(
	void *devDsc, 
	INODE_PARAM *param, 
	DIR_ENTRY *dirBuf)
{
	DIR_ENTRY *entry;
	DIR_PARAM dparam;
	int rest;

	dparam.dirBuf = dirBuf;

	for (dparam.path = param->path; isLastPath(dparam.path) == NO;){
		/* ǥ쥯ȥꥨȥ꡼õ */
		if ((rest = readDir(devDsc, param->inodeData, &dparam, readDirBlock)) < 0){
			return rest;
		}
		entry = (DIR_ENTRY*)rest;
		if (entry->file_type != EXT2_DIR_DIR){
			return -ENOTDIR;
		}

		// inode
		rest = readInode(devDsc, entry->inode, param->inodeBuf, &param->inodeData);
		if (rest != NOERR){
			return rest;
		}
		param->inodeData->i_number = entry->inode;
		
		/* '/'μ˥ѥʤ롣 */
		for (dparam.path += entry->name_len + 1; *dparam.path == '/'; ++dparam.path){
		}
	}
	param->path = dparam.path;

	return NOERR;
}

/*
 *ԳԲġ
 * return : error number
 */
STATIC int getInodeData(void *devDsc, const uint inodeNum, INODE_PARAM *param)
{
	// inodeǡ롣
	int error = readInode(devDsc, inodeNum, param->inodeBuf, &param->inodeData);
	if (error != NOERR){
		return error;
	}
	param->inodeData->i_number = inodeNum;

	return NOERR;
}

/*
 * ƥǥ쥯ȥ˥ȥ꡼ä롣
 *ԳԲġ
 * return : error number
 */
STATIC int addDirEnt(
	void *devDsc,
	uchar type,			// file type
	uint inumber,		// inode number
	INODE_PARAM *param, 
	DIR_ENTRY *dirBuf)
{
	const int din = getDevInodeNum(devDsc);
	DIR_ENTRY *new_ent;
	DIR_PARAM dparam;
	int rest;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ѥ¸ߤ뤫ǧ */
	if (0 < readDir(devDsc, param->inodeData, &dparam, readDirBlock)) {
		return -EEXIST;
	}

	/* ǥ쥯ȥ꤫ȥ꡼õ */
	rest = readDir(devDsc, param->inodeData, &dparam, addDirBlock);
	if (rest == -ENOENT) {
		/* ˥ǥ쥯ȥ˥֥åƤ롣 */
		if (0 < superBlock[din]->s_free_blocks_count) {
			new_ent = dparam.dirBuf;
			new_ent->rec_len = EXT2_BLOCK_SIZE(devDsc);
			new_ent->name_len = strlen(dparam.path);
			memcpy(new_ent->name, param->path, new_ent->name_len);
			new_ent->inode = inumber;
			new_ent->file_type = type;

			rest = writeData(devDsc, param->inodeData->i_size, EXT2_BLOCK_SIZE(devDsc), new_ent, param->inodeData);
			if (rest != NOERR) {
				return rest;
			}
			param->inodeData->i_size += EXT2_BLOCK_SIZE(devDsc);
			param->inodeData->i_blocks = ROUNDUP_DIV(param->inodeData->i_size, INODE_BLOCK_SIZE);
		}
		else{
			return -ENOSPC;
		}
	}
	else if (0 < rest){
		/* ȥ꡼롣 */
		new_ent = (DIR_ENTRY*)rest;
		memcpy(new_ent->name,param->path,new_ent->name_len);
		new_ent->inode = inumber;
		new_ent->file_type = type;
		rest = BlockCacheWrite(getDevInfo(devDsc), dparam.dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, dparam.blkNumber));
		if (rest != NOERR){
			return rest;
		}
	}
	else{
		return rest;
	}

	/* ƥǥ쥯ȥinode򹹿 */
	param->inodeData->i_atime = param->inodeData->i_mtime = (uint)sys_time(NULL);
	if (type == EXT2_DIR_DIR) {
		param->inodeData->i_links_count += 1;
	}
	rest = writeInode(devDsc, param->inodeData->i_number, param->inodeBuf);
	if (rest != NOERR) {
		return rest;
	}

	return NOERR;
}

/*
 * ƥǥ쥯ȥ꤫ե륨ȥ꡼롣
 * ȥ꡼inodeϰparam롣
 *ԳԲġ
 * return : error number
 */
STATIC int delFileEntry(
	void *devDsc,
	const int type,
	INODE_PARAM *param,
	DIR_ENTRY *dirBuf)
{
	DIR_ENTRY *entry;
	DIR_PARAM dparam;
	int rest;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if ((rest = readDir(devDsc,param->inodeData,&dparam,readDirBlock)) < 0) {
		return rest;
	}
	entry = (DIR_ENTRY*)rest;

	if (entry->file_type == getDirFileTypeFromFileType(type)) {
		rest = readInode(devDsc,entry->inode,param->inodeBuf, &param->inodeData);
		if (rest != NOERR) {
			return rest;
		}
		param->inodeData->i_number = entry->inode;
		delEntry(entry,dparam.dirBuf);
		rest = BlockCacheWrite(getDevInfo(devDsc), dparam.dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, dparam.blkNumber));
		if (rest != NOERR) {
			return rest;
		}
	}
	else {
		return -ENOENT;
	}

	return NOERR;
}


/*
 * ƥǥ쥯ȥ꤫ǥ쥯ȥꥨȥ꡼롣
 * ȥ꡼inodeϰparam롣
 *ԳԲġ
 * return error number
 */
STATIC int delDirEntry(
	void *devDsc,
	INODE_PARAM *param,
	DIR_ENTRY *dirBuf)
{
	DIR_ENTRY *entry;
	DIR_PARAM dparam;
	INODE *parent_inode;
	INODE *parent_ibuf;
	int rest;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	rest = readDir(devDsc, param->inodeData, &dparam, readDirBlock);
	if (rest < 0){
		return rest;
	}
	entry = (DIR_ENTRY*)rest;

	if (entry->file_type == EXT2_DIR_DIR){
		parent_inode = param->inodeData;
		parent_ibuf = param->inodeBuf;

		param->inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
		if (param->inodeBuf == NULL){
			return -ENOMEM;
		}

		rest = readInode(devDsc, entry->inode, param->inodeBuf, &param->inodeData);
		if (rest != NOERR){
			goto ERR;
		}
		param->inodeData->i_number = entry->inode;
		rest = checkDir(devDsc, param->inodeData);
		if (rest != -ENOENT){
			goto ERR;
		}
		delEntry(entry, dparam.dirBuf);
		rest = BlockCacheWrite(getDevInfo(devDsc), dparam.dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, dparam.blkNumber));
		if (rest != NOERR){
			goto ERR;
		}
		--parent_inode->i_links_count;
		rest = writeInode(devDsc, parent_inode->i_number, parent_ibuf);
		if (rest != NOERR) {
			goto ERR;
		}

		kfree(parent_ibuf);
	}
	else{
		return -ENOENT;
	}

	return NOERR;
ERR:
	kfree(parent_ibuf);

	return rest;
}

/*
 * ȥ꡼inodeoldnew촹롣
 *ԳԲġ
 * return : error number
 */
STATIC int changeInode(
	void *devDsc,
	INODE_PARAM *old_param,
	INODE_PARAM *new_param,
	DIR_ENTRY *old_dbuf,
	DIR_ENTRY *new_dbuf,
	const char *old_path,
	const char *new_path)
{
	DIR_ENTRY *old_ent;
	DIR_ENTRY *new_ent;
	DIR_PARAM old_dparam;
	DIR_PARAM new_dparam;
	int rest;

	/* oldΥȥ꡼õ */
	old_dparam.dirBuf = old_dbuf;
	old_dparam.path = old_param->path;
	rest = readDir(devDsc, old_param->inodeData, &old_dparam, readDirBlock);
	if (rest < 0) {
		return rest;
	}
	old_ent = (DIR_ENTRY*) rest;

	/* newΥȥ꡼õ */
	new_dparam.dirBuf = new_dbuf;
	new_dparam.path = new_param->path;
	rest = readDir(devDsc, new_param->inodeData, &new_dparam, readDirBlock);

	if (0 < rest) {				/* newΥȥ꡼¸ߤ롣 */
		new_ent = (DIR_ENTRY*) rest;
		if (old_ent->file_type != new_ent->file_type) {
			return (old_ent->file_type == EXT2_DIR_REG_FILE) ? -EISDIR : -ENOTDIR;
		}

		/* pathͭγǧ */
		if (old_ent->file_type == EXT2_DIR_DIR) {
			if (old_path[cmpStrNum(old_path, new_path)] == '\0')
				return -EINVAL;
		}

		// newΥǡinode
		rest = readInode(devDsc, new_ent->inode, new_param->inodeBuf, &new_param->inodeData);
		if (rest != NOERR) {
			return rest;
		}
		if (new_ent->file_type == EXT2_DIR_DIR) {
			rest = checkDir(devDsc, new_param->inodeData);
			if (rest != -ENOENT) {
				return rest;
			}
		}
		rest = unlinkInode(devDsc, new_param->inodeData);
		if (rest != NOERR) {
			return rest;
		}

		/* ȥ꡼inodeʥС촹롣 */
		new_ent->inode = old_ent->inode;
		rest = BlockCacheWrite(getDevInfo(devDsc), new_dparam.dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, new_dparam.blkNumber));
		if (rest != NOERR){
			return rest;
		}
	}
	else if (rest == -ENOENT) {	// newΥȥ꡼¸ߤʤ
		/* ե̾γǧ */
		if (EXT2_NAME_LEN < strlen(new_param->path)){
			return -ENAMETOOLONG;
		}
		/* newȥ꡼ */
		if ((rest = addDirEnt(devDsc, old_ent->file_type, old_ent->inode, new_param, new_dbuf)) < 0){
			return rest;
		}
	}
	else{
		return rest;
	}

	/* oldȥ꡼κ */
	rest = readDir(devDsc, old_param->inodeData, &old_dparam, readDirBlock);
	if (rest < 0){
		return rest;	/* ɤ߹ߡ */
	}
	delEntry(old_ent, old_dparam.dirBuf);
	rest = BlockCacheWrite(getDevInfo(devDsc), old_dparam.dirBuf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, old_dparam.blkNumber));
	if (rest != NOERR){
		return rest;
	}

	return NOERR;
}

/*
 * ̾μΥȥ꡼̾ơΥȥ꡼ΥեåȤ֤
 * Υȥ꡼̵0֤
 *ԳԲġ
 * return : next index or 0 or error number
 */
STATIC int getNextEntry(
	void *devDsc, 
	uint block,		// directory block
	int index,		// entry index
	char *o_name)
{
	DIR_ENTRY *buf, *entry;
	int rest;

	if (index == EXT2_BLOCK_SIZE(devDsc)){
		return 0;
	}

	buf = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (buf == NULL){
		return -ENOMEM;
	}
	rest = BlockCacheRead(getDevInfo(devDsc), buf, EXT2_SECTOR_NUM(devDsc), blkToSect(devDsc, block));
	if (rest != NOERR){
		goto END;
	}

	entry = (DIR_ENTRY*)((char*)buf + index);
	if (entry->inode == 0){
		entry=(DIR_ENTRY*)((char*)entry+entry->rec_len);
	}

	if ((char*)entry == (char*)buf + EXT2_BLOCK_SIZE(devDsc)){
		rest = 0;
	}
	else{
		memcpy(o_name, entry->name, entry->name_len);
		o_name[entry->name_len] = '\0';
		rest = (uint)entry - (uint)buf + entry->rec_len;
	}
END:
	kfree(buf);

	return rest;
}

/*
 * Υǥ쥯ȥ֥å롣Υ֥åʤ-ENOENT֤
 *ԳԲġ
 * retur : next block or error number
 */
STATIC int INLINE getNextBlock(
	void *devDsc,
	INODE *inode,
	uint block)
{
	DIR_PARAM dparam;

	dparam.blkNumber = block;
	return readDir(devDsc, inode, &dparam, searchDirBlock);
}


/*
 * ǥ쥯ȥΥǥ쥯ȥ֥åν
 *ԳԲġ
 * return : error number
 */
STATIC int initDirBlock(
	void *devDsc,
	uint block,
	uint inumber,			// inode number
	uint parent_inumber)	// parent inode number
{
	DIR_ENTRY *dent,*dent2;
	int error=0;
	int rest;


	if ((dent = kmalloc(EXT2_SECTOR_SIZE(devDsc))) == NULL) {
		return -ENOMEM;
	}
	memset(dent, 0, EXT2_SECTOR_SIZE(devDsc));			/* ꥢʤȥǥ쥯ȥǷٹθˤʤ롣 */

	/* "."ǥ쥯ȥɲá */
	dent->inode=inumber;
	dent->rec_len=sizeof(DIR_ENTRY)+sizeof(int);
	dent->name_len=1;
	dent->file_type = EXT2_DIR_DIR;
	dent->name[0]='.';

	dent2=(DIR_ENTRY*)((char*)dent+dent->rec_len);

	/* ".."ǥ쥯ȥɲá */
	dent2->inode=parent_inumber;
	dent2->rec_len=EXT2_BLOCK_SIZE(devDsc) - dent->rec_len;
	dent2->name_len=2;
	dent2->file_type = EXT2_DIR_DIR;
	dent2->name[0]='.';
	dent2->name[1]='.';

	rest = BlockCacheWrite(getDevInfo(devDsc), dent, 1, blkToSect(devDsc, block));
	if (rest != NOERR){
		error = rest;
	}

	kfree(dent);

	return error;
}

//--------------------------------------------------------------------------------------------------
// System call
//--------------------------------------------------------------------------------------------------

/*
 * ե
 *ԳԲġ
 * return : error number
 */
STATIC int creatFile(
	const char *path, 
	void *devDsc,
	uint inumber,		// ǥ쥯ȥinodeֹ
	ushort mode, 
	uint *o_number)
{
	uint parentInumber;
	uint block = 0;
	DIR_ENTRY *dirBuf;
	INODE *inode;
	INODE_PARAM param;
	void *proc;
	int rest,err_rest;
	int error;

	/* ѥθ */
	if ((param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc))) == NULL) {
		return -ENOMEM;
	}
	if ((dirBuf = kmalloc(EXT2_BLOCK_SIZE(devDsc))) == NULL) {
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path = path;
	rest = readInode(devDsc, inumber, param.inodeBuf, &param.inodeData);
	if (rest != NOERR) {
		goto ERR1;
	}
	rest = searchParentDir(devDsc, &param, dirBuf);
	if (rest != NOERR){
		goto ERR1;
	}
	parentInumber = param.inodeData->i_number;

	/* inode롣 */
	inumber = getNewInode(devDsc, mode);
	if (inumber == 0) {
		rest = -ENOSPC;
		goto ERR1;
	}

	/* ǥ쥯ȥꥨȥ꡼ƥǥ쥯ȥɲä롣 */
	if ((rest = addDirEnt(devDsc, getDirFileTypeFromFileType(mode), inumber, &param, dirBuf)) < 0){
		goto ERR2;
	}

	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR){
		/* ǥ쥯ȥ롣 */
		block = getBlock(devDsc, &error);
		if (block == 0){
			rest = error;
			goto ERR3;
		}
		rest = initDirBlock(devDsc, block, inumber, param. inodeData->i_number);
		if (rest < 0){
			goto ERR4;
		}
	}

	/* inode롣 */
	rest = readInode(devDsc, inumber, param.inodeBuf, &inode);
	if (rest != NOERR) {
		goto ERR4;
	}
	memset(inode, 0, sizeof(INODE));
	proc = getCurrentProc();
	inode->i_mode = mode;
	inode->i_uid = getUid(proc);
	inode->i_gid = getGid(proc);
	inode->i_atime = inode->i_ctime = inode->i_mtime = (uint)sys_time(NULL);
	inode->i_number = inumber;
	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR) {
		inode->i_links_count = 2;
		inode->i_size = EXT2_BLOCK_SIZE(devDsc);
		inode->i_blocks = inode->i_size / INODE_BLOCK_SIZE;
		inode->i_block[0] = block;
	}
	else {
		inode->i_links_count = 1;
	}
	rest = writeInode(devDsc, inumber, param. inodeBuf);
	if (rest != NOERR) {
		goto ERR4;
	}

	kfree(dirBuf);
	kfree(param.inodeBuf);
	
	*o_number = inumber;

	return NOERR;

ERR4:
	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR) {
		releaseBlock(devDsc,block);
	}
ERR3:
	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR) {
		INODE *inode;

		err_rest = readInode(devDsc, parentInumber, param.inodeBuf, &inode);
		if (err_rest == NOERR) {
			inode->i_links_count -= 1;
			writeInode(devDsc, parentInumber, param.inodeBuf);
		}
	}
ERR2:
	releaseInodeBitAndGrp(devDsc, inumber, EXT2_S_IFLNK);
ERR1:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}

/*
 *ԳԲġ
 */
STATIC int mount(void *devDsc, void **o_rootInodeObj)
{
	const int din = getDevInodeNum(devDsc);
	char *buf;
	INODE *inodeBuf;
	INODE *inode;
	size_t ioSize;
	int groupNum;
	int error;

	// Ǥ˥ޥȤƤ뤫ǧ
	if (superBlock[din] != NULL){
		return -EBUSY;
	}

	// ѡ֥åɤ߹ࡣ
	buf = kmalloc(PAGE_SIZE);
	if (buf == NULL){
		return -ENOMEM;
	}
	error = read_direct(getDevInfo(devDsc), buf, PAGE_SIZE / DEV_BSIZE, 0, &ioSize);
	if (error != NOERR){
		goto ERR;
	}
	superBlock[din] = kmalloc(sizeof(SUPER_BLOCK));
	if (superBlock[din] == NULL){
		error = -ENOMEM;
		goto ERR;
	}
	memcpy(superBlock[din], buf + EXT2_SUPER_BLK_POSITION, sizeof(SUPER_BLOCK));

	// ޥåʥСγǧ
	if (superBlock[din]->s_magic != EXT2_SUPER_MAGIC) {
		error = -ENOTBLK;
		goto ERR;
	}

	// ॹѡ֥åǤ
	superBlock[din]->blockSize = EXT2_BLOCK_BASE << superBlock[din]->s_log_block_size;
	superBlock[din]->sectorSize = DEV_BSIZE;
	superBlock[din]->sectors = EXT2_BLOCK_SIZE(devDsc) / EXT2_SECTOR_SIZE(devDsc);

	// 롼ץǥץɤ߹
	grpDescNum[din] = ROUNDUP_DIV(superBlock[din]->s_blocks_count, superBlock[din]->s_blocks_per_group);
	grpDescBlocks[din] = ROUNDUP_DIV(grpDescNum[din] * sizeof(GROUP_DESC), EXT2_BLOCK_SIZE(devDsc));
	grpDesc[din] = kmalloc(EXT2_BLOCK_SIZE(devDsc) * grpDescBlocks[din]);
	if (grpDesc[din] == NULL){
		error = -ENOMEM;
		goto ERR2;
	}
	error = read_direct(getDevInfo(devDsc), grpDesc[din], EXT2_SECTOR_NUM(devDsc) * grpDescBlocks[din],
		blkToSect(devDsc, EXT2_GROUP_DESC_BLK + superBlock[din]->s_first_data_block), &ioSize);
	if (error != NOERR){
		goto ERR3;
	}

	// ֥åطν
	groupNum = initBlock(devDsc);
	if (groupNum < 0){
		goto ERR3;
	}

	// inodeطν
	error = initInode(devDsc, groupNum);
	if (error < 0){
		goto ERR3;
	}

	// root inodeɤ߽Ф
	error = readInode(devDsc, EXT2_ROOT_INODE_INDEX, (INODE*) buf, &inode);
	if (error != NOERR){
		goto ERR3;
	}
	inodeBuf = kmalloc(sizeof(INODE));
	if (inodeBuf == NULL){
		error = -ENOMEM;
		goto ERR3;
	}
	memcpy(inodeBuf, inode, sizeof(INODE));
	*o_rootInodeObj = inodeBuf;

	kfree(buf);

	return 0;

ERR3:
	kfree(grpDesc[din]);
	grpDesc[din] = NULL;
ERR2:
	kfree(superBlock[din]);
	superBlock[din] = NULL;
ERR:
	kfree(buf);

	return error;
}

/*
 *ԳԲġ
 */
STATIC int umount(
	void *devDsc,
	void *rootInodeObj)
{
	const int din = getDevInodeNum(devDsc);

	umountInode(devDsc);
	umountBlock(devDsc);
	writeBackSbGd(devDsc);

	kfree(rootInodeObj);

	kfree(grpDesc[din]);
	grpDesc[din] = NULL;

	kfree(superBlock[din]);
	superBlock[din] = NULL;

	return NOERR;
}

STATIC int lookup(const char *i_path, void *devDsc, void *inodeObj, uint *o_inodeNum, const char **o_path)
{
	const char *path;		// ѥ
	const char *nextPath;
	INODE *inodeBuf;
	INODE *inode;
	DIR_ENTRY *dirBuf;
	int error;

	ASSERT(i_path != NULL);

	inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (inodeBuf == NULL) {
		return -ENOMEM;
	}

	// ǥ쥯ȥ긡ѥХåեgetInodeFromPath()ǻѤ뤬ʣƤӽФΤ
	// ΨΤ˸ƤӽФǳƤ롣
	dirBuf = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (dirBuf == NULL) {
		kfree(inodeBuf);
		return -ENOMEM;
	}

	inode = inodeObj;
	for (path = nextPath = i_path; (nextPath[0] != '\0') && ((inode->i_mode & EXT2_S_IFMT) != EXT2_S_IFLNK);) {
		path = nextPath;
		inode = getInodeFromPath(devDsc, path, inode, inodeBuf, dirBuf, &nextPath, &error);
		if (error != NOERR){
			goto END;
		}
	}
	*o_path = path;
	*o_inodeNum = inode->i_number;
	if ((inode->i_mode & EXT2_S_IFMT) == EXT2_S_IFLNK) {
		// ܥåȯ
		error = SYMLINK;
	}
	else {
		error = NOERR;
	}
END:
	kfree(dirBuf);
	kfree(inodeBuf);

	return error;
}

/*
 *ԳԲġ
 */
STATIC int creat(const char *path, void *devDsc, void *inodeObj, const int mode, uint *o_inodeNum)
{
	INODE *dirInode = inodeObj;

	ASSERT(path != NULL);

	// ե
	return creatFile(path, devDsc, dirInode->i_number, mode, o_inodeNum);
}

/*
 *ԳԲġ
 */
STATIC int open(void *devDsc, const uint inodeNum, int oflag, void **o_inodeObj, int *o_type)
{
	INODE *inodeData;
	INODE_PARAM inodeParam;
	int error;

	/* ƥХåեΥ */
	inodeParam.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (inodeParam.inodeBuf == NULL) {
		return -ENOMEM;
	}
	error = getInodeData(devDsc, inodeNum, &inodeParam);
	if (error < 0) {
		goto END;
	}

	inodeData = kmalloc(sizeof(INODE));
	if (inodeData == NULL) {
		error = -ENOMEM;
		goto END;
	}
	*inodeData = *inodeParam.inodeData;

	*o_inodeObj = inodeData;
	*o_type = inodeData->i_mode & EXT2_S_IFMT;

	error = NOERR;
END:
	kfree(inodeParam.inodeBuf);

	return error;
}

/*
 *ԳԲġ
 */
STATIC int close(void *devDsc, void *inodeObj)
{
	INODE *sectBuf;
	INODE *diskInode;
	INODE *inode = inodeObj;
	int rest;

	// inode֥Ȥ񤭤Ƴ
	sectBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (sectBuf == NULL){
		return -ENOMEM;
	}
	rest = readInode(devDsc, inode->i_number, sectBuf, &diskInode);
	if (rest != NOERR){
		goto END;
	}
	if (diskInode->i_dtime == 0){
		memcpy(diskInode, inode, sizeof(INODE));
		rest = writeInode(devDsc, inode->i_number, sectBuf);
		if (rest != NOERR) {
			goto END;
		}
	}

	rest = NOERR;
END:
	kfree(sectBuf);
	kfree(inode);

	return rest;
}

/*
 *ԳԲġ
 */
STATIC int read(
	void *devDsc, 
	void *inodeObj, 
	void *buf, 
	uint i_size, 
	uint begin)
{
	INODE *inode = inodeObj;
	size_t size;
	int error;

	// γǧ
	if (inode->i_size < begin) {
		return -EOVERFLOW;
	}
	if (begin + i_size <= inode->i_size) {
		size = i_size;
	}
	else {
		size = inode->i_size - begin;
	}
	if (size == 0) {
		return 0;
	}

	// եɤ
	error = readData(devDsc, begin, size, buf, inode);
	if (error != NOERR) {
		return error;
	}

	/* դι */
	inode->i_atime = (uint) sys_time(NULL);

	return size;
}

/*
 *ԳԲġ
 */
STATIC int write(
	void *devDsc, 
	void *inodeObj, 
	void *buf, 
	uint size, 
	uint begin)
{
	const int din = getDevInodeNum(devDsc);
	INODE *inode = inodeObj;
	int error;

	// γǧ
	if (inode->i_size < begin + size){
		if(superBlock[din]->s_free_blocks_count <= (begin + size - inode->i_size) / EXT2_BLOCK_SIZE(devDsc)){
			return -ENOSPC;
		}
	}
	if (size == 0) {
		return 0;
	}

	/* եؽ񤭹ߡ */
	error = writeData(devDsc, begin, size, buf, inode);
	if (error != NOERR) {
		return error;
	}

	/* ι */
	if (inode->i_size < begin + size){
		int inodeNum = EXT2_BLOCK_SIZE(devDsc) / sizeof(uint32_t);
		int blkNum = ROUNDUP_DIV(begin + size, EXT2_BLOCK_SIZE(devDsc));

		inode->i_size = begin + size;

		if (INODE_DATA_INDIRECT1 < blkNum){
			if (INODE_DATA_INDIRECT1 + inodeNum < blkNum){
				if (INODE_DATA_INDIRECT1 + (inodeNum + 1) * inodeNum < blkNum){
					blkNum += ROUNDUP_DIV(blkNum - INODE_DATA_INDIRECT1 - inodeNum, inodeNum * inodeNum);
				}
				blkNum += ROUNDUP_DIV(blkNum - INODE_DATA_INDIRECT1, inodeNum);
			}
			blkNum += 1;
		}
		inode->i_blocks = blkNum * (EXT2_BLOCK_SIZE(devDsc) / INODE_BLOCK_SIZE);
	}

	/* դι */
	inode->i_atime = inode->i_mtime = sys_time(NULL);

	return size;
}

STATIC int mknod(const char *path, void *devDsc, void *inodeObj, const int mode, const uint dev)
{
	return -EOPNOTSUPP;
}

/*
 *աNULLѥԲġ
 *ԳԲġ
 */
STATIC int mkdir(const char *path, void *devDsc, void *dirInodeObj, int mode)
{
	INODE *dirInode = dirInodeObj;
	uint inumber;

	return creatFile(path, devDsc, dirInode->i_number, mode | S_IFDIR, &inumber);
}

/*
 *աNULLѥԲġ
 *ԳԲġ
 */
STATIC int delete(const char *path, void *devDsc, void *dirInodeObj, int type)
{
	INODE *dirInode = dirInodeObj;
	DIR_ENTRY *dirBuf;
	INODE_PARAM param;
	int rest;

	/* ƥǥ쥯ȥθ */
	param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (param.inodeBuf == NULL){
		return -ENOMEM;
	}
	dirBuf = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (dirBuf == NULL){
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path = path;
	rest = readInode(devDsc, dirInode->i_number, param.inodeBuf, &param.inodeData);
	if (rest != NOERR) {
		goto END;
	}
	rest = searchParentDir(devDsc, &param, dirBuf);
	if (rest != NOERR){
		goto END;
	}

	rest = -EOPNOTSUPP;
	if ((type & S_IFREG) != 0) {
		rest = delFileEntry(devDsc, S_IFREG, &param, dirBuf);
		if (rest == NOERR) {
			rest = unlinkInode(devDsc, param.inodeData);
			goto END;
		}
	}
	if ((type & S_IFLNK) != 0) {
		rest = delFileEntry(devDsc, S_IFLNK, &param, dirBuf);
		if (rest == NOERR) {
			rest = unlinkInode(devDsc, param.inodeData);
			goto END;
		}
	}
	if ((type & S_IFDIR) != 0) {
		rest = delDirEntry(devDsc, &param, dirBuf);
		if (rest == NOERR) {
			rest = unlinkInode(devDsc, param.inodeData);
			goto END;
		}
	}

END:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}

/*
 *ԳԲġ
 */
STATIC int opendir(void *devDsc, void *dirInodeObj, uint *o_block, int *o_index)
{
	INODE *dirInode = dirInodeObj;
	INODE_PARAM param;
	INODE *inode;
	int rest;

	param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (param.inodeBuf == NULL){
		return -ENOMEM;
	}
	rest = readInode(devDsc, dirInode->i_number, param.inodeBuf, &inode);
	if (rest != NOERR) {
		goto ERR;
	}
	*o_block = inode->i_block[0];
	*o_index = 0;

	rest = NOERR;
ERR:
	kfree(param.inodeBuf);
	return rest;
}

/*
 *ԳԲġ
 */
STATIC int readdir(void *devDsc, void *inodeObj, uint *m_block, int *m_index, char *o_name)
{
	INODE *dirInode = inodeObj;
	INODE *inode;
	INODE_PARAM param;
	int rest;

	for(;;){
		rest = getNextEntry(devDsc, *m_block, *m_index, o_name);
		if(0 < rest){
			*m_index = rest;
			rest = 0;
			break;
		}
		else if(rest < 0){
			break;
		}

		if ((param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc))) == NULL) {
			return -ENOMEM;
		}
		rest = readInode(devDsc, dirInode->i_number, param.inodeBuf, &inode);
		if (rest != NOERR) {
			kfree(param.inodeBuf);
			break;
		}
		rest = getNextBlock(devDsc, inode, *m_block);
		kfree(param.inodeBuf);
		if(rest == -ENOENT){		/* Υ֥å̵ */
			*o_name = '\0';
			rest = 0;
			break;
		}
		else if(rest < 0){
			break;
		}

		*m_block = rest;
		*m_index = 0;
	}

	return rest;
}

STATIC int stat(void *devDsc, void *inodeObj, struct stat *m_stat)
{
	const int din = getDevInodeNum(devDsc);
	INODE *inode = inodeObj;

	m_stat->st_dev		= din;
	m_stat->st_mode		= inode->i_mode;
	m_stat->st_nlink	= inode->i_links_count;
	m_stat->st_uid		= inode->i_uid;
	m_stat->st_gid		= inode->i_gid;
	m_stat->st_size		= inode->i_size;
	m_stat->st_atime	= inode->i_atime;
	m_stat->st_mtime	= inode->i_mtime;
	m_stat->st_ctime	= inode->i_ctime;
	m_stat->st_blksize	= EXT2_BLOCK_BASE << superBlock[din]->s_log_block_size;
	m_stat->st_ino		= inode->i_number;										/* Ȥꤢinode롣 */
	m_stat->st_blocks	= ROUNDUP(m_stat->st_size,m_stat->st_blksize) / 512;	/* ѤƤ֥å(512Хñ) */
	m_stat->st_rdev		= 0;
/************** ̤ *******************************/
	m_stat->st_flags = 0;
/*****************************************************/

	return NOERR;
}

/*
 *ԳԲġ
 */
STATIC int rename(void *devDsc, void *oldDirInodeObj, const char *oldPath, void *newDirInodeObj, const char *newPath)
{
	INODE *oldDirInode = oldDirInodeObj;
	INODE *newDirInode = newDirInodeObj;
	DIR_ENTRY *dirBuf1, *dirBuf2;
	INODE_PARAM old_param, new_param;
	int rest;

	new_param.inodeBuf = NULL;
	dirBuf2 = NULL;

	/* old pathõ */
	dirBuf1 = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (dirBuf1 == NULL){
		return -ENOMEM;
	}
	old_param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (old_param.inodeBuf == NULL){
		rest = -ENOMEM;
		goto END;
	}
	old_param.path = oldPath;
	rest = readInode(devDsc, oldDirInode->i_number, old_param.inodeBuf, &old_param.inodeData);
	if (rest != NOERR) {
		goto END2;
	}
	rest = searchParentDir(devDsc, &old_param, dirBuf1);
	if (rest != NOERR){
		goto END2;
	}

	/* new pathõ */
	new_param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (new_param.inodeBuf == NULL) {
		rest = -ENOMEM;
		goto END2;
	}
	new_param.path = newPath;
	rest = readInode(devDsc, newDirInode->i_number, new_param.inodeBuf, &new_param.inodeData);
	if (rest != NOERR){
		goto END3;
	}
	rest = searchParentDir(devDsc,&new_param, dirBuf1);
	if (rest != NOERR){
		goto END3;
	}

	/* old inodenew inode촹롣 */
	dirBuf2 = kmalloc(EXT2_BLOCK_SIZE(devDsc));
	if (dirBuf2 == NULL){
		rest = -ENOMEM;
		goto END3;
	}
	rest = changeInode(devDsc, &old_param, &new_param, dirBuf1, dirBuf2, oldPath, newPath);
	if (rest < 0){
		goto END4;
	}

	rest = NOERR;
END4:
	kfree(dirBuf2);
END3:
	kfree(new_param.inodeBuf);
END2:
	kfree(old_param.inodeBuf);
END:
	kfree(dirBuf1);

	return rest;
}

STATIC int chattr(void *devDsc, void *inodeObj, uint mode, uint uid, uint gid, uint atime, uint mtime)
{
	INODE *inode = inodeObj;

	inode->i_mode 	= (inode->i_mode & ~0x1ff) | (mode & 0x1ff);
	inode->i_uid 	= uid;
	inode->i_gid 	= gid;
	inode->i_atime 	= atime;
	inode->i_mtime 	= mtime;

	return NOERR;
}

/*
 *ԳԲġ
 */
STATIC int link(void *devDsc, void *dstInodeObj, void *srcInodeObj, const char *srcPath)
{
	INODE *dstInode = dstInodeObj;
	INODE *srcInode = srcInodeObj;
	DIR_ENTRY *dirBuf;
	INODE_PARAM param;
	int rest;

	if ((param.inodeBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc))) == NULL){
		return -ENOMEM;
	}
	if ((dirBuf = kmalloc(EXT2_BLOCK_SIZE(devDsc))) == NULL){
		kfree(param.inodeBuf);
		return -ENOMEM;
	}

	/* Υǥ쥯ȥinodeΰõ */
	param.path = srcPath;
	rest = readInode(devDsc,srcInode->i_number,param.inodeBuf, &param.inodeData);
	if (rest != NOERR) {
		goto END;
	}
	if ((rest = searchParentDir(devDsc,&param,dirBuf)) != NOERR){
		goto END;
	}

	/* Υե̾Υå */
	if (strlen(param.path) > EXT2_NAME_LEN){
		rest = -ENAMETOOLONG;
		goto END;
	}

	/* ǥ쥯ȥꥨȥ꡼ɲä롣 */
	if ((rest = addDirEnt(devDsc, getDirFileTypeFromFileType(EXT2_S_IFREG), dstInode->i_number, &param, dirBuf)) < 0){
		goto END;
	}

	/* 󥯤䤹 */
	dstInode->i_links_count += 1;
	
	rest = NOERR;
END:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}

/*
 *ԳԲġ
 */
STATIC int symlink(const char *filePath, void *devDsc, void *dirInodeObj, char *linkPath)
{
	uint inumber;
	INODE *inode;
	INODE *sectBuf;
	INODE *dirInode = dirInodeObj;
	int rest;

	rest = creatFile(filePath, devDsc, dirInode->i_number, S_IFLNK | S_IRWXU | S_IRWXG | S_IRWXO, &inumber);
	if (rest < 0){
		return rest;
	}

	/* inodeɤ߽Ф */
	sectBuf = kmalloc(EXT2_SECTOR_SIZE(devDsc));
	if (sectBuf == NULL){
		return -ENOMEM;
	}
	rest = readInode(devDsc, inumber, sectBuf, &inode);
	if (rest != NOERR){
		goto END;
	}

	/* 󥯥ѥ񤭹 */
	rest = write(devDsc, inode, linkPath, strlen(linkPath), 0);
	if (rest < 0){
		goto END;
	}
	
	/* inode᤹ */
	rest = writeInode(devDsc, inumber, sectBuf);
	if (rest != NOERR) {
		goto END;
	}

	rest = NOERR;
END:
	kfree(sectBuf);

	return rest;
}

/*
 *ԳԲġ
 */
STATIC int ioctl(void *devDsc, void *inodeObj, int cmd, caddr_t param, int fflag)
{
	int rest;

	switch(cmd){
		case I_TRUNC:
			/* truncate. */
			rest = releaseDataBlock(devDsc, inodeObj);
			if(rest != NOERR) {
				return rest;
			}
	}

	return NOERR;
}

STATIC int poll(void *inodeObj, int events)
{
	return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
}


STATIC int statfs(void *devDsc, struct statfs *m_statfs)
{
	const int din = getDevInodeNum(devDsc);
	SUPER_BLOCK *superBlk = superBlock[din];

	m_statfs->f_bsize		= superBlk->blockSize;				/* File system block size. */
	m_statfs->f_iosize		= superBlk->blockSize;				/* optimal transfer block size */
	m_statfs->f_blocks		= superBlk->s_blocks_count;			/* total data blocks in file system */
	m_statfs->f_bfree		= superBlk->s_free_blocks_count;	/* free blocks in fs */
	m_statfs->f_bavail		= superBlk->s_free_blocks_count;	/* free blocks avail to non-superuser */
	m_statfs->f_files		= superBlk->s_inodes_count;			/* total file nodes in file system */
	m_statfs->f_ffree		= superBlk->s_free_inodes_count;	/* free file nodes in fs */
	m_statfs->f_fsid.val[0] = 0;								/* file system id */
	m_statfs->f_fsid.val[1] = 0;
	m_statfs->f_owner		= 0;								/* user that mounted the filesystem */
	m_statfs->f_type		= MOUNT_EXT2FS;						/* type of filesystem */
	m_statfs->f_flags		= 0;								/* Bit mask of f_flag values */
	memcpy(m_statfs->f_mntfromname, "ext2fs", 7);				/* mounted filesystem */

	return NOERR;
}


/***********************************************************************
 *
 * Init
 *
 ***********************************************************************/

static FS ext2Fs={
	"ext2",
	mount,
	umount,
	lookup,
	creat,
	open,
	close,
	read,
	write,
	ioctl,
	poll,
	opendir,
	readdir,
	rename,
	mknod,
	mkdir,
	stat,
	delete,
	chattr,
	link,
	symlink,
	statfs,
};

/*
 * GLOBAL
 */
int initExt2tFs()
{
	int i;

	for (i = 0; i < MAX_DEVICE_OPEN; ++i) {
		grpDesc[i] = NULL;
		superBlock[i] = NULL;
	}

	return regist_fs(&ext2Fs);
}

/**************************************************************************************/

int test_ext2(int din)
{
	printk("test_ext2(() group=0x%x\n", blockBitmap[din].group);

	return 0;
}

