/*
 * apm.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <lib/lib.h>
#include <i386/lib.h>
#include <i386/segment.h>
#include "apm.h"


enum{
	// Power state flag.
	APM_PWRFLG_ENABL	= 0x0,	// APM Enabled.
	APM_PWRFLG_STANDBY	= 0x1,	// Standby.
	APM_PWRFLG_SUSPEND	= 0x2,	// Suspend.
	APM_PWRFLG_OFF		= 0x3,	// Power off.
	
	// Errpr coad.
	APM_ERR_NOTPRESEN = 0x86,	// APM not present.
};

typedef struct{
	uint eax;
	uint ebx;
	uint ecx;
	uint edx;
	uint esi;
	uint edi;
} APM_FUNC_ARGS;

volatile FARCALL_ADDR apmfarCall;	/* Indirect address for APM far call. */
extern ushort apmVersion;
extern ushort apmFlag;
extern uint apmCodeSegBase;			/* Real mode segment base address. */
extern uint apmCode16SegBase;		/* Real mode segment base address. */
extern uint apmDataSegBase;			/* Real mode segment base address. */
extern uint apmOffset;
extern uint apmCodeSegLen;
extern uint apmCode16SegLen;
extern uint apmDataSegLen;

extern int apmBiosCall(APM_FUNC_ARGS *args);
extern int apmBiosCallReal(APM_FUNC_ARGS *args);

//================================== PRIVATE ============================================

//================================== PUBLIC =============================================

/*
 * 顼ɥå
 * return : ʸ
 */
const char *getApmErrorMessage(
	const int errorCode)		// 顼
{
	switch (errorCode) {
	case 0x01:
		return "Power management functionality disabled.";
	case 0x02:
		return "Real mode interface connection already established.";
	case 0x03:
		return "Interface not connected.";
	case 0x05:
		return "16-bit protected mode interface already established.";
	case 0x06:
		return "16-bit protected mode interface not supported.";
	case 0x07:
		return "32-bit protected mode interface already established.";
	case 0x08:
		return "32-bit protected mode interface not supported.";
	case 0x09:
		return "Unrecognized device ID.";
	case 0x0A:
		return "Parameter value out of range.";
	case 0x0B:
		return "Interface not engaged.";
	case 0x0C:
		return "Function not supported.";
	case 0x0D:
		return "Resume timer disabled (valid for get resume timer value only).";
	case 0x60:
		return "Unable to enter requested state.";
	case 0x80:
		return "No power management events pending.";
	case 0x86:
		return "APM not present.";
	default:
		return "Unknown error.";
	}
}

/*
 * 顼ɥå
 * return : ʸ
 */
const char *getApmPowerStateMessage(
	const int state)
{
	switch (state) {
	case 0:
		return "APM Enabled.";
	case 1:
		return "Standby.";
	case 2:
		return "Suspend.";
	case 3:
		return "Off.";
	default:
		return "Unknown state.";
	}
}

/*
 * Get APM Driver Version.
 * return : driver version(BCD) or -1
 */
int getApmDriverVersion()
{
	APM_FUNC_ARGS args;
	int error;

	if ((apmFlag & APM_FLG_32MODE) == 0){
		return APM_ERR_NOTPRESEN;
	}

	args.eax = APM_FUNC_DRVVRS;
	args.ebx = 0;				/* APM BIOS device number. */
	args.ecx = apmVersion;
	error = apmBiosCall(&args);
	if (error == 0){
		return args.eax;
	}
	else{
		return -1;
	}
}

/*
 * Get Power State.
 * return NOERR or apm error number
 */
int getApmPowerState(
	int *o_state)
{
	APM_FUNC_ARGS args;
	int error;

	if ((apmFlag & APM_FLG_32MODE) == 0){
		return APM_ERR_NOTPRESEN;
	}

	args.eax = APM_FUNC_GETSTA;
	args.ebx = APM_DEV_ALLDEV;
	error = apmBiosCall(&args);
	if (error == NOERR) {
		*o_state = args.ecx & 0xff;
		return NOERR;
	}
	else {
		return error;
	}
}

/*
 * Set power off.
 * return NOERR or apm error number
 */
int setApmPowerOff()
{
	APM_FUNC_ARGS args;

	if ((apmFlag & APM_FLG_32MODE) == 0){
		return APM_ERR_NOTPRESEN;
	}

	args.eax = APM_FUNC_SETPWR;
	args.ebx = APM_DEV_ALLDEV;
	args.ecx = APM_PWRFLG_OFF;
	args.edx = 0;
	return apmBiosCall(&args);
}

/*
 * 
 */
void InitApm()
{
	if (apmFlag == 0){		/* APM not supported. */
		return;
	}

	/* APM version. */
	printk("APM : BIOS version %d.%d\n", apmVersion >> 8, apmVersion & 0xff);

	if ((apmFlag & APM_FLG_32MODE) == 0){	/* 32-bit protected mode interface not supported. */
		return;
	}

	if(apmOffset == 0){
		printk("APM 32bit mode interface connection error! error coad : %x\n", apmCodeSegBase >> 8);
	}

	/* APMͥѥȤꡣ */
	set_gdt(APM_CODE_DES, apmCodeSegBase << 4, apmCodeSegLen, TYPE_KERNEL_CODE);			/* Code segment. */
	set_gdt(APM_CODE16_DES, apmCode16SegBase << 4, apmCode16SegLen, TYPE_KERNEL_CODE16);	/* 16bit code segment. */
	set_gdt(APM_DATA_DES, apmDataSegBase << 4, apmDataSegLen, TYPE_KERNEL_DATA);			/* Data segment. */
	apmfarCall.des = APM_CODE_DES;
	apmfarCall.addr = apmOffset;
}
