/*
 * isa.c
 *
 * Copyright 2006, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ɣӣХϢؿ
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/bus.h>
#include <sys/bus_private.h>
#include <sys/systm.h>
#include <machine/interrupt.h>
#include <isa/isareg.h>
#include <isa/isa_dma.h>
#include <isa/isavar.h>
#include <isa/ic/i8237.h>
#include <kern/kmalloc.h>
#include <kern/device.h>
#include <lib/lib.h>
#include <kern/debug.h>


//#define DEBUG_ISA 1
#ifdef DEBUG_ISA
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


/****************************************************************************************
* DMA
*****************************************************************************************/


//================================== PRIVATE ============================================


enum{
	DMA_CHANNEL_MAX = 8,	// DMAͥ
};

/*
**  Register definitions for DMA controller 1 (channels 0..3):
*/
#define	DMA1_CHN(c)	(IO_DMA1 + 1*(2*(c)))	/* addr reg for channel c */
#define	DMA1_SMSK	(IO_DMA1 + 1*10)		/* single mask register */
#define	DMA1_MODE	(IO_DMA1 + 1*11)		/* mode register */
#define	DMA1_FFC	(IO_DMA1 + 1*12)		/* clear first/last FF */

/*
**  Register definitions for DMA controller 2 (channels 4..7):
*/
#define	DMA2_CHN(c)	(IO_DMA2 + 2*(2*(c)))	/* addr reg for channel c */
#define	DMA2_SMSK	(IO_DMA2 + 2*10)		/* single mask register */
#define	DMA2_MODE	(IO_DMA2 + 2*11)		/* mode register */
#define	DMA2_FFC	(IO_DMA2 + 2*12)		/* clear first/last FF */

static struct dmaBuf{
	caddr_t			buf;	// Хåեɥ쥹
	int				last;	// 饹ȥɥ쥹
	struct dmaBuf	*next;	// ѥ󥯡ɥ쥹˥
}dmaBuf[DMA_CHANNEL_MAX];

static u_int8_t	dma_inuse = 0;		// User for acquire/release
static u_int8_t	dma_auto_mode = 0;
static u_int8_t	dma_busy = 0;		// Used in isa_dmastart()
static u_int8_t	dma_bounced = 0;

/* high byte of address is stored in this port for i-th dma channel */
static int dmapageport[8] = { 0x87, 0x83, 0x81, 0x82, 0x8f, 0x8b, 0x89, 0x8a };


// DMAѥХåեƤ롣
//աեХåեϰϤϤϥͥ뤬ɤ꡼ɥ쥹˰¸롣
// return : error number
STATIC int allocDmaBuf(const int channel, const size_t size)
{
	enum{
		DMA_BUF_BEGIN	= 0,
		DMA_BUF_LAST	= 0x1000,
	};
	static struct dmaBuf *dmaBufTop = NULL;
	struct dmaBuf **nextBuf;
	int first;
	
	// ̤ѥ꡼Ƭ򸡺
	first = DMA_BUF_BEGIN;
	nextBuf = &dmaBufTop;
	for (; *nextBuf != NULL; nextBuf = &(*nextBuf)->next){
		first = (*nextBuf)->last;
	}
	if (DMA_BUF_LAST < first + size){
		return -ENOMEM;
	}
	
	// DMAХåե¤Τ
	dmaBuf[channel].buf = (void*)first;
	dmaBuf[channel].last = first + size;
	*nextBuf = &dmaBuf[channel];
	
	return NOERR;
}


//================================== PUBLIC =============================================


//FreeBSD
// Register a DMA channel's usage.  Usually called from a device driver
// in open() or during its initialization.
// return : error number
int isa_dma_acquire(int chan)
{
	if (dma_inuse & (1 << chan)) {
		printf("isa_dma_acquire: channel %d already in use\n", chan);
		return -EBUSY;
	}
	dma_inuse |= (1 << chan);
	dma_auto_mode &= ~(1 << chan);

	return (0);
}


// isa_dmastart(): program 8237 DMA controller channel, avoid page alignment
// problems by using a bounce buffer.
void isa_dmastart(int flags, caddr_t addr, u_int nbytes, int chan)
{
	int waport;
	caddr_t newaddr;

	dma_busy |= (1 << chan);

	if (dmaBuf[chan].last - (int)dmaBuf[chan].buf < nbytes){
		panic("isa_dmastart: bad buffer"); 
	}
	dma_bounced |= (1 << chan);
	ASSERT(0 < dmaBuf[chan].last);
	newaddr = dmaBuf[chan].buf;

	/* copy bounce buffer on write */
	if (!(flags & ISADMA_READ)){
		bcopy(addr, newaddr, nbytes);
	}
	addr = newaddr;

	if ((flags & ISADMA_RAW) == ISADMA_RAW) {
	    dma_auto_mode |= (1 << chan);
	}
	else { 
	    dma_auto_mode &= ~(1 << chan);
	}

	if ((chan & 4) == 0) {
		/*
		 * Program one of DMA channels 0..3.  These are
		 * byte mode channels.
		 */
		/* set dma channel mode, and reset address ff */

		/* If ISADMA_RAW flag is set, then use autoinitialise mode */
		if ((flags & ISADMA_RAW) == ISADMA_RAW) {
			if ((flags & ISADMA_READ) == ISADMA_READ){
				outb(DMA1_MODE, DMA37MD_AUTO | DMA37MD_WRITE | chan);
			}
			else{
				outb(DMA1_MODE, DMA37MD_AUTO | DMA37MD_READ | chan);
			}
		}
		else if ((flags & ISADMA_READ) == ISADMA_READ){
			outb(DMA1_MODE, DMA37MD_SINGLE | DMA37MD_WRITE | chan);
		}
		else{
			outb(DMA1_MODE, DMA37MD_SINGLE | DMA37MD_READ | chan);
		}
		outb(DMA1_FFC, 0);

		/* send start address */
		waport = DMA1_CHN(chan);
		outb(waport, (uint)addr);
		outb(waport, (uint)addr>>8);
		outb(dmapageport[chan], (uint)addr>>16);

		/* send count */
		outb(waport + 1, --nbytes);
		outb(waport + 1, nbytes>>8);

		/* unmask channel */
		outb(DMA1_SMSK, chan);
	}
	else {
		/*
		 * Program one of DMA channels 4..7.  These are
		 * word mode channels.
		 */
		/* set dma channel mode, and reset address ff */

		/* If ISADMA_RAW flag is set, then use autoinitialise mode */
		if ((flags & ISADMA_RAW) == ISADMA_RAW) {
			if ((flags & ISADMA_READ) == ISADMA_READ){
				outb(DMA2_MODE, DMA37MD_AUTO|DMA37MD_WRITE|(chan&3));
			}
			else{
				outb(DMA2_MODE, DMA37MD_AUTO|DMA37MD_READ|(chan&3));
			}
		}
		else if ((flags & ISADMA_READ) == ISADMA_READ){
			outb(DMA2_MODE, DMA37MD_SINGLE|DMA37MD_WRITE|(chan&3));
		}
		else{
			outb(DMA2_MODE, DMA37MD_SINGLE|DMA37MD_READ|(chan&3));
		}
		outb(DMA2_FFC, 0);

		/* send start address */
		waport = DMA2_CHN(chan - 4);
		outb(waport, (uint)addr>>1);
		outb(waport, (uint)addr>>9);
		outb(dmapageport[chan], (uint)addr>>16);

		/* send count */
		nbytes >>= 1;
		outb(waport + 2, --nbytes);
		outb(waport + 2, nbytes>>8);

		/* unmask channel */
		outb(DMA2_SMSK, chan & 3);
	}
}


void isa_dmadone(int flags, caddr_t addr, int nbytes, int chan)
{  
	if (((dma_busy & (1 << chan)) == 0) && (dma_auto_mode & (1 << chan)) == 0 ){
		printf("isa_dmadone: channel %d not busy\n", chan);
	}

	if ((dma_auto_mode & (1 << chan)) == 0){
		outb(chan & 4 ? DMA2_SMSK : DMA1_SMSK, (chan & 3) | 4);
	}

	if (dma_bounced & (1 << chan)) {
		/* copy bounce buffer on read */
		if ((flags & ISADMA_READ) == ISADMA_READ){
			bcopy(dmaBuf[chan].buf, addr, nbytes);
		}
		dma_bounced &= ~(1 << chan);
	}
	dma_busy &= ~(1 << chan);
}


// Setup a DMA channel's bounce buffer.
void isa_dmainit(int chan, u_int bouncebufsize)
{
	if (allocDmaBuf(chan, bouncebufsize) != NOERR){
		printf("isa_dmainit(%d, %d) failed\n", chan, bouncebufsize);
	}
}


/****************************************************************************************
* 
*****************************************************************************************/


//================================== PRIVATE ============================================


struct device isaDevice;


static int isa_probe(device_t dev)
{
	printk("isa_probe() : unsuported function.\n");
	return 0;
}

static int isa_attach(device_t dev)
{
	printk("isa_attach() : unsuported function.\n");
	return 0;
}

static device_t isa_add_child(device_t dev, int order, const char *name, int unit)
{
	printk("isa_add_child() : unsuported function.\n");
	return 0;
}

static int isa_print_child(device_t bus, device_t dev)
{
	printk("isa_print_child() : unsuported function.\n");
	return 0;
}

static void isa_probe_nomatch(device_t dev, device_t child)
{
	bus_print_child_header(dev, child);
	printf(" failed to probe");
	bus_print_child_footer(dev, child);
}

static int isa_read_ivar(device_t bus, device_t dev, int index, uintptr_t * result)
{
	printk("isa_read_ivar() : unsuported function.\n");
	return 0;
}

static int isa_write_ivar(device_t bus, device_t dev, int index, uintptr_t value)
{
	printk("isa_write_ivar() : unsuported function.\n");
	return 0;
}

static void isa_child_detached(device_t dev, device_t child)
{
	printk("isa_child_detached() : unsuported function.\n");
}

static void isa_driver_added(
	device_t dev,		// ƥǥХ
	driver_t *driver)	// ҥɥ饤С¤
{
    device_t child;
    char *nameUnit;
    int nameBufLen;

	nameBufLen = strlen(driver->name) + 5;
	nameUnit = kmalloc(nameBufLen);
	if (nameUnit == NULL){
		return;
	}
    for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)){
		if (device_get_state(child) != DS_NOTPRESENT){
			continue;
		}
		if (!device_is_enabled(child)){
			continue;
		}

		snprintf(nameUnit, nameBufLen, "%s%d", driver->name, child->unit);
		if (strcmp(child->nameunit, nameUnit) == 0){
			device_probe_and_attach(child);
		}
	}
	kfree(nameUnit);
}

// return : struct resource or 0 = error
struct resource *isa_alloc_resource(
	device_t bus, 
	device_t dev, 
	int type, 
	int *rid_ioport, 
	u_long start, 
	u_long end, 
	u_long count, 
	u_int flags)
{
	struct resource *resource;
	int rest;

	resource = kmalloc(sizeof(*resource));
	if (resource == NULL){
		return 0;
	}
	memset(resource, 0, sizeof(*resource));
	switch (type){
		case SYS_RES_IRQ:
			if(resource_int_value(dev->devclass->name, dev->unit, "irq", &rest) != 0){
				goto ERR;
			}
			resource->r_start = rest;
			break;
		case SYS_RES_DRQ:
			if(resource_int_value(dev->devclass->name, dev->unit, "drq", &rest) != 0){
				goto ERR;
			}
			resource->r_start = rest;
			break;
		case SYS_RES_MEMORY:
			ASSERT(0);
			break;
		case SYS_RES_IOPORT:
			if(resource_int_value(dev->devclass->name, dev->unit, "ioport", &rest) != 0){
				goto ERR;
			}
			resource->r_bustag = I386_BUS_SPACE_IO;
			resource->r_bushandle = rest;
			break;
		default:
			ASSERT(0);
	}

	return resource;
ERR:
	kfree(resource);
	return 0;
}


static int isa_setup_intr(
	device_t parent,
	device_t dev,
	struct resource *irq,
	int intrFlag,
	void (*handler)(void *),
	void *softc,
	void *opt)
{
	setupIntr(irq, handler, softc);
	release_irq_mask(irq->r_start);

	return NOERR;
}


static device_method_t isa_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe,			isa_probe),
	DEVMETHOD(device_attach,		isa_attach),
	DEVMETHOD(device_detach,		bus_generic_detach),
	DEVMETHOD(device_shutdown,		bus_generic_shutdown),
	DEVMETHOD(device_suspend,		bus_generic_suspend),
	DEVMETHOD(device_resume,		bus_generic_resume),

	/* Bus interface */
	DEVMETHOD(bus_add_child,		isa_add_child),
	DEVMETHOD(bus_print_child,		isa_print_child),
	DEVMETHOD(bus_probe_nomatch,	isa_probe_nomatch),	
	DEVMETHOD(bus_read_ivar,		isa_read_ivar),
	DEVMETHOD(bus_write_ivar,		isa_write_ivar),
	DEVMETHOD(bus_child_detached,	isa_child_detached),
	DEVMETHOD(bus_driver_added,		isa_driver_added),
	DEVMETHOD(bus_alloc_resource,	isa_alloc_resource),
	DEVMETHOD(bus_setup_intr,		isa_setup_intr),

	{ 0, 0 }
};

static driver_t isa_driver = {
	"isa",
	isa_methods,
	1,			/* no softc */
};


//================================== PUBLIC =============================================


// return : error number
int initIsa()
{
	int i;
	int error;

	memset(&isaDevice, 0, sizeof(struct device));
	isaDevice.state = DS_ATTACHED;
	isaDevice.driver = &isa_driver;
	TAILQ_INIT(&isaDevice.children);
	
	// ǥХ饹κ
	error = device_set_devclass(&isaDevice, "isa");
	if (error != NOERR){
		return error;
	}

	// ҥǥХϿ
	for (i = resource_query_string(-1, "at", "isa0"); i != -1; i = resource_query_string(i, "at", "isa0")){
		if (device_add_child(&isaDevice, resource_query_name(i), resource_query_unit(i)) == 0){
			printf("Failed add device : %s\n", resource_query_name(i));
		}
	}
	return NOERR;
}
