/*
 * BlockCache.c
 *
 * Copyright 2009, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 *Գסե֥åå
 *ե֥åBLOCK_SIZEܿǡPAGE_SIZEʲǡģ֥åPAGE_SIZEޤʤ
 */

#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/limits.h>
#include <sys/conf.h>
#include <sys/uio.h>
#include <lib/lib.h>
#include <lib/AggregateList.h>
#include <lib/IteratorList.h>
#include <lib/bitmap.h>
#include <machine/interrupt.h>
#include <machine/lock.h>
#include <kern/kmalloc.h>
#include <kern/TaskSignal.h>
#include <kern/TaskWait.h>
#include <kern/proc.h>
#include <kern/PhysMem.h>
#include <module/Module.h>
#include <kern/BlockCache.h>

#include <kern/debug.h>

//#define DEBUG_BLOCK_CACHE 1
#ifdef DEBUG_BLOCK_CACHE
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif

//=====================================  ===================================================

enum {
	RESERVE_CACHE		= 0x80000,						// ͽ󥭥å奵
	HASH_MAX			= 2000,							// ϥåꥹȿ
	IOVEC_MAX 			= MAXPHYS / PAGE_SIZE,			// iovec
	CACHE_BLOCK_SIZE	= DEV_BSIZE,					// å֥å
	CACHE_BLOCKS		= PAGE_SIZE / CACHE_BLOCK_SIZE,	// å֥å
};

typedef struct {
	PhysMemObj	*physMemObj;	// ʪ꡼֥
	dev_t		dev;			// ǥХ
	uint		block;			// å֥åֹ
	List		hashList;		// ϥåꥹ
	List		writeList;		// ǥХƱ񤭹Ԥꥹ
	u_int8_t 	bitmap;			// ֥åӥåȥޥå
} BlockCache;

//===================================== Х륤ݡ =======================================

extern BlockCacheObj *PhysMemGetCache(PhysMemObj*);
extern BlockCacheObj *PhysMemGetNewCache();
extern int PhysMemAllocBlockCacheReserve(const uint);
extern void PhysMemPutCache(PhysMemObj*);
extern void PhysMemPutBackCache(PhysMemObj*);
extern void PhysMemSetDirty(PhysMemObj*);
extern void PhysMemClearDirty(PhysMemObj*);
extern char *PhysMemGetPhysAddr(PhysMemObj*);
extern int callUserMethod(ThreadObj*, const void*, const void*, const uint, const uint, const uint, const uint, const uint, const uint);

//===================================== PRIVATE ====================================================

//--------------------------------------------------------------------------------------------------
// ֥åǥХɤ߹
//--------------------------------------------------------------------------------------------------

static char *readBuf;

/*
 * ǥХ饭åɤ߹ࡣ
 * return : error number
 */
STATIC int readDevice(
	BlockCache *this,
	char *buf)
{
	int rest;
	
	rest = moduleReadDev(this->dev, buf, PAGE_SIZE, this->block, NULL);
	if (rest < 0) {
		return rest;
	}
	else if (rest != PAGE_SIZE) {
		return -EIO;
	}
	else {
		return NOERR;
	}
}

/*
 * ̤å֥å򥭥å夹
 * return : error number
 */
STATIC int readToCache(
	BlockCache *this)
{
	int error;

	if (this->bitmap == 0) {
		error = readDevice(this, PhysMemGetPhysAddr(this->physMemObj));
	}
	else {
		static int spinLock = 0;
		char *cache = PhysMemGetPhysAddr(this->physMemObj);
		int i;

		// readBuf¾
		enter_spinlock(&spinLock);
		{
			error = readDevice(this, readBuf);
			if (error == NOERR) {
				for (i = 0; i < CACHE_BLOCKS; ++i) {
					if (bitmapIsSet(&this->bitmap, 1 << i) == NO) {
						memcpy(cache + CACHE_BLOCK_SIZE * i, readBuf + CACHE_BLOCK_SIZE * i, CACHE_BLOCK_SIZE);
					}
				}
			}
		}
		exit_spinlock(&spinLock);
	}
	
	return error;
}

//--------------------------------------------------------------------------------------------------
// ֥åǥХƱ񤭹
// ֥åPAGE_SIZEñ
//--------------------------------------------------------------------------------------------------

static ThreadObj *writeThread;						// Ʊ񤭹ߥͥ륹å
static AggregateList aggrWriteList;					// 񤭹ԤꥹȽ
static AggregateListMethod aggrWriteMethod;			// 񤭹Ԥꥹȥ᥽å
static int writeSpinLock;							// 񤭹Ԥꥹȥԥå
static AggregateList aggrWriteDevice;				// ǥХ񤭹ߥꥹȽ
static AggregateListMethod aggrWriteDeviceMethod;	// ǥХ񤭹ߥꥹȥ᥽å

/*
 * ǥХؽ񤭹
 * return : error number
 */
STATIC int writeDevice()
{
	struct iovec *iovec;
	BlockCache *head;
	BlockCache *next;
	int iov_cnt;
	int size;
	BitmapInfo info;
	IteratorList iterator;

	head = aggrWriteDeviceMethod.refHead(&aggrWriteDevice);
	iov_cnt = 0;
	size = 0;
	iovec = entryAllocParamMem(getUserStackPoint(), sizeof(struct iovec) * IOVEC_MAX);

	ASSERT(head != NULL);

	IteratorListConstruct(&iterator, &aggrWriteDevice, &aggrWriteDeviceMethod);
	for (next = iterator.next(&iterator);;) {
		// iovec
		getBitmapRange(next->bitmap, &info);
		iovec[iov_cnt].iov_base = PhysMemGetPhysAddr(next->physMemObj) + (DEV_BSIZE * info.low);
		iovec[iov_cnt].iov_len = DEV_BSIZE * info.count;
		size += DEV_BSIZE * info.count;
		++iov_cnt;
		
		if (iterator.hasNext(&iterator) == BOOL_FALSE) {
			break;
		}

		if (info.low + info.count == CHAR_BIT) {
#ifdef DEBUG
			int block = next->block;
#endif
			next = iterator.next(&iterator);

			ASSERT(block + CHAR_BIT == next->block);
		}
		else {
			int error;

			getBitmapRange(head->bitmap, &info);
			error = moduleWriteUio(head->dev, head->block + info.low, size, iovec, iov_cnt);
			if (error != 0) {
				return error;
			}
			getBitmapRange(next->bitmap, &info);
			bitmapReset(info.low, info.count, &next->bitmap);

			head = next;
			iov_cnt = 0;
			size = 0;
			iovec = entryAllocParamMem(getUserStackPoint(), sizeof(struct iovec) * IOVEC_MAX);
		}
	}

	getBitmapRange(head->bitmap, &info);
	return moduleWriteUio(head->dev, head->block + info.low, size, iovec, iov_cnt);
}

/*
 * åƱǥХǥ֥å˼Ф
 *aggrWriteListϥǥХ硢֥å¤Ǥ
 * return : å奻åȿ
 */
STATIC int setDeviceWriteList()
{
	BlockCache *this;
	BlockCache *next;
	int cnt;
	
	this = aggrWriteMethod.getHead(&aggrWriteList);
	if (this == NULL) {
		return 0;
	}
	aggrWriteDeviceMethod.insertEnd(&aggrWriteDevice, &this->writeList);
	cnt = 1;

	for (next = aggrWriteMethod.refHead(&aggrWriteList); next != NULL; next = aggrWriteMethod.refHead(&aggrWriteList)) {
		ASSERT(0 < this->bitmap);

		if (this->dev == next->dev) {
			BitmapInfo info;
			
			getBitmapRange(this->bitmap, &info);
			if (info.high != CHAR_BIT) {
				break;
			}

			getBitmapRange(next->bitmap, &info);
			if (0 < info.low) {
				break;
			}

			if (this->block + CHAR_BIT != next->block) {
				break;
			}

			this = aggrWriteMethod.getHead(&aggrWriteList);
			aggrWriteDeviceMethod.insertEnd(&aggrWriteDevice, &this->writeList);
			if (IOVEC_MAX <= ++cnt) {
				break;
			}
		}
		else {
			break;
		}
	}

	return cnt;
}

/*
 * ǥХ硢֥å˥ꥹȤ
 *writeSpinLockǥå줿֤ǸƤӽФ뤳
 */
STATIC void insertWriteList(
	BlockCache *this)
{
	IteratorList iterator;

	IteratorListConstruct(&iterator, &aggrWriteList, &aggrWriteMethod);
	for (;;) {
		if (iterator.hasNext(&iterator) == BOOL_FALSE) {
			aggrWriteMethod.insertEnd(&aggrWriteList, &this->writeList);
			break;
		}

		BlockCache *next = iterator.next(&iterator);
		if (this->dev < next->dev) {
			aggrWriteMethod.insertPrev(&aggrWriteList, &next->writeList, &this->writeList);
			break;
		}
		if (this->dev == next->dev) {
			if (this->block < next->block) {
				aggrWriteMethod.insertPrev(&aggrWriteList, &next->writeList, &this->writeList);
				break;
			}
			
			ASSERT(this->block != next->block);
//			else if (this->block == next->block) {
//				aggrWriteMethod.insertPrev(&aggrWriteList, &next->writeList, &this->writeList);
//				aggrWriteMethod.removeEntry(&aggrWriteList, &next->writeList);
//				PhysMemPutBackCache(next->physMemObj);
//				break;
//			}
		}
	}
}

/*
 * Ʊ񤭹߽
 */
STATIC void asyncWrite()
{
	for (;;) {
		int cnt;
		int eflag = enterCli();
		enter_spinlock(&writeSpinLock);
		{
			// 񤭹ԤФ
			cnt = setDeviceWriteList();
		}
		exit_spinlock(&writeSpinLock);
		exitCli(eflag);

		if (0 < cnt) {
			BlockCache *this;
			int error = writeDevice();
			if (error != NOERR) {
				printk("writeDevice() error! : %d\n", error);
			}

			// ͳꥹȤ᤹
			while ((this = aggrWriteDeviceMethod.getHead(&aggrWriteDevice)) != NULL) {
				PhysMemClearDirty(this->physMemObj);
				PhysMemPutBackCache(this->physMemObj);
			}
		}
		else{
			TaskWait();
		}
	}
}

/*
 * Ʊ񤭹ߥͥ륹åɤεư
 * return : NOERR or ERR
 */
STATIC int startAsyncWriteThread()
{
	switch (forkKernelThread()) {
		case -1:
			printk("Faild startAsyncWriteThread\n");
			return ERR;
		case 0:
			writeThread = getCurrentTask();

			// λ륷ʥ򤹤٤ƥޥ
			setSigAllMask(TaskGetTaskSignal(writeThread));

			asyncWrite();

			// äƤʤ
	}

	return NOERR;
}

//--------------------------------------------------------------------------------------------------
// ϥåꥹ
//--------------------------------------------------------------------------------------------------
/*
 * ϥåꥹȤˤʤ֥åʣΥƱ˥줿硢ʣƱ֥å
 * å夵ƤޤΤǡƱϥåϥȥߥå˥褦ˤɬפ롣
 */

static AggregateList aggrHashList[HASH_MAX];	// ϥåꥹȽ
static AggregateListMethod aggrHashMethod;		// ϥåꥹȥ᥽å
static int hashSpinLock;						// ϥåꥹȥԥå

/*
 * ֥åֹ椫ϥåͤ򻻽Ф
 * return : ϥå
 */
STATIC INLINE int calcHash(
	const uint block)
{
	return (block / 8) % HASH_MAX;
}

/*
 * ϥåꥹȤ饭å򸡺
 *hashSpinLockǥåƤ뤳
 * return : BlockCache or NULL
 */
STATIC BlockCache *getHashList(
	dev_t dev,
	const uint block)
{
	IteratorList iterator;
	int hash = calcHash(block);
	BlockCache *this = NULL;

	IteratorListConstruct(&iterator, &aggrHashList[hash], &aggrHashMethod);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		BlockCache *next = iterator.next(&iterator);
		if ((next->dev == dev) && (next->block == block)) {
			this = next;
			aggrHashMethod.removeEntry(&aggrHashList[hash], &this->hashList);
			break;
		}
	}

	return this;
}

/*
 * ϥåꥹȤ³
 *hashSpinLockǥåƤ뤳
 */
STATIC void insertHashList(
	BlockCache *this,
	const uint block)
{
	int hash = calcHash(block);
	aggrHashMethod.insertHead(&aggrHashList[hash], &this->hashList);

}

/*
 * ϥåꥹȤ
 *hashSpinLockǥåƤ뤳
 */
STATIC void removeHashList(
	BlockCache *this,
	const uint block)
{
	int hash = calcHash(block);
	aggrHashMethod.removeEntry(&aggrHashList[hash], &this->hashList);
}

//--------------------------------------------------------------------------------------------------
// ֥åå嶦̥ե󥯥
//--------------------------------------------------------------------------------------------------

/*
 * åꤹ
 */
STATIC INLINE void resetCache(
	BlockCache *this,
	dev_t dev,
	const uint block)
{
	removeHashList(this, this->block);
	this->dev = dev;
	this->block = block;
	this->bitmap = 0;
}

STATIC BlockCache *getblk(
	dev_t dev,
	const uint block)
{
	BlockCache *this;

	enter_spinlock(&hashSpinLock);
	{
		this = getHashList(dev, block);
		if (this != NULL) {
			this = (BlockCache*) PhysMemGetCache(this->physMemObj);
			if (this == NULL) {
				// å
				this = (BlockCache*) PhysMemGetNewCache();
				resetCache(this, dev, block);
			}
		}
		else {
			// å
			this = (BlockCache*) PhysMemGetNewCache();
			resetCache(this, dev, block);
		}

		insertHashList(this, block);
	}
	exit_spinlock(&hashSpinLock);

	return this;
}

/*
 * return : error number
 */
static inline int cacheRead(
	BlockCache *this,
	void *buf,				// writeХåե
	const uint block,		// åⳫϥ֥å
	const uint count)		// å֥å
{
	if (bitmapIsSet(&this->bitmap, bitmapGet(block, count)) == NO) {
		int error = readToCache(this);
		if (error != NOERR) {
			PhysMemPutBackCache(this->physMemObj);
			return error;
		}
		bitmapSet(0, CACHE_BLOCKS, &this->bitmap);
	}
	memcpy(buf, PhysMemGetPhysAddr(this->physMemObj) + block * CACHE_BLOCK_SIZE, count * CACHE_BLOCK_SIZE);
	
	return NOERR;
}

static inline void cacheWrite(
	BlockCache *this,
	void *buf,				// writeХåե
	const uint block,		// åⳫϥ֥å
	const uint count)		// å֥å
{
	memcpy(PhysMemGetPhysAddr(this->physMemObj) + block * CACHE_BLOCK_SIZE, buf, count * CACHE_BLOCK_SIZE);
	bitmapSet(block, count, &this->bitmap);
}

//===================================== PUBLIC =====================================================

/*
 *ʪ꡼˼¹
 * return : error number
 */
int BlockCacheInit()
{
	int i;
	int error;

#ifdef DEBUG
	if (sizeof(BlockCache) != sizeof(BlockCacheObj)) {
		printk("BlockCacheInit() object size error! BlockCache=%d BlockCacheObj=%d\n", sizeof(BlockCache), sizeof(BlockCacheObj));
		idle();
	}
#endif
	// ֥ååѤ˥꡼ͽ
	error = PhysMemAllocBlockCacheReserve(RESERVE_CACHE);
	if (error != NOERR) {
		return error;
	}

	// ꥹȽν
	AggregateListConstructor(&aggrWriteList, &aggrWriteMethod);
	AggregateListConstructor(&aggrWriteDevice, &aggrWriteDeviceMethod);
	for (i = 0; i < HASH_MAX; ++i) {
		AggregateListConstructor(&aggrHashList[i], &aggrHashMethod);
	}

	// Ʊ񤭹ߥåɤεư
	startAsyncWriteThread();

	// readХåեγ
	readBuf = kmalloc(PAGE_SIZE);
	if (readBuf == NULL) {
		return -ENOMEM;
	}

	return NOERR;
}

void BlockCacheConstructor(
	BlockCacheObj *blockCacheObj,
	PhysMemObj *physMemObj)
{
	BlockCache *this = (BlockCache*) blockCacheObj;

	listConstructor(&this->writeList, this);
	listConstructor(&this->hashList, this);
	this->physMemObj = physMemObj;
	this->dev = NULL;
	this->block = 0;
	this->bitmap = 0;
}

//--------------------------------------------------------------------------------------------------
// ֥åǥХƱ񤭹
//--------------------------------------------------------------------------------------------------

/*
 * Ʊ񤭹ߥꥹȤ˲ä
 */
void BlockCacheAsyncWrite(
	BlockCacheObj *blockCacheObj)
{
	BlockCache *this = (BlockCache*) blockCacheObj;
	int eflag;
	
	eflag = enterCli();
	enter_spinlock(&writeSpinLock);
	{
		insertWriteList(this);
	}
	exit_spinlock(&writeSpinLock);
	exitCli(eflag);

	// ٱ񤭹ߥ򵯤
	TaskAwake(getTaskWait(writeThread));
}

/*
 * Ʊ񤭹Ԥꡩ
 * return : YES of NO
 */
int BlockCacheIsAsyncWrite()
{
	int condition;
	int eflag = enterCli();
	enter_spinlock(&writeSpinLock);
	{
		ASSERT(0 <= aggrWriteMethod.getCount(&aggrWriteList));
		ASSERT(0 <= aggrWriteDeviceMethod.getCount(&aggrWriteDevice));

		if ((0 < aggrWriteMethod.getCount(&aggrWriteList)) ||
			(0 < aggrWriteDeviceMethod.getCount(&aggrWriteDevice))) {
			condition = YES;
		}
		else {
			condition = NO;
		}
	}
	exit_spinlock(&writeSpinLock);
	exitCli(eflag);
	
	return condition;
}

//--------------------------------------------------------------------------------------------------
// ֥åå嶦̥ե󥯥
//--------------------------------------------------------------------------------------------------

/*
 *աճߥϥɥ餫θƤӽФԲ
 *ԳԲġ
 * return : error number
 */
int BlockCacheRead(
	dev_t dev,				// ǥХ
	void *i_buf,			// readХåե
	const uint i_blocks,	// ֥å
	const uint i_block)		// ϥ֥å
{
	BlockCache *this;
	uint resid = i_blocks;
	uchar *buf = i_buf;
	uint boundBlock = ROUNDDOWN(i_block, CACHE_BLOCKS);		// å֥å
	uint relateBlock = i_block - boundBlock;				// åХ֥å
	uint count = ((resid + relateBlock) < CACHE_BLOCKS) ? resid : CACHE_BLOCKS - relateBlock;	// å֥å

	while (0 < resid) {
		int error;

		this = getblk(dev, boundBlock);

		error = cacheRead(this, buf, relateBlock, count);
		if (error != NOERR) {
			return error;
		}
		buf += count * CACHE_BLOCK_SIZE;

		// åͳꥹȤ᤹
		PhysMemPutCache(this->physMemObj);
	
		boundBlock += CACHE_BLOCKS;
		relateBlock = 0;
		resid -= count;
		count = (resid < CACHE_BLOCKS) ? resid : CACHE_BLOCKS;
	}

	return NOERR;
}

/*
 *աճߥϥɥ餫θƤӽФԲ
 *ԳԲġ
 * return : error number
 */
int BlockCacheWrite(
	dev_t dev,				// ǥХ
	void *i_buf,			// writeХåե
	const uint i_blocks,	// ֥å
	const uint i_block)		// ϥ֥å
{
	BlockCache *this;
	uint resid = i_blocks;
	uchar *buf = i_buf;
	uint boundBlock = ROUNDDOWN(i_block, CACHE_BLOCKS);		// å֥å
	uint relateBlock = i_block - boundBlock;				// åХ֥å
	uint count = ((resid + relateBlock) < CACHE_BLOCKS) ? resid : CACHE_BLOCKS - relateBlock;	// å֥å

	while (0 < resid) {
		// å
		this = getblk(dev, boundBlock);

		cacheWrite(this, buf, relateBlock, count);
		buf += count * CACHE_BLOCK_SIZE;

		// 򥻥å
		PhysMemSetDirty(this->physMemObj);
		
		// åͳꥹȤ᤹
		PhysMemPutCache(this->physMemObj);
		
		boundBlock += CACHE_BLOCKS;
		relateBlock = 0;
		resid -= count;
		count = (resid < CACHE_BLOCKS) ? resid : CACHE_BLOCKS;
	}

	return NOERR;
}

//**************************************************************************************************

#ifdef DEBUG
void testBlockCache()
{
}
#endif
