/*
 * PhysMem.c
 *
 * Copyright 2009, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 *Գסʪ꡼
 */

#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <lib/lib.h>
#include <lib/AggregateList.h>
#include <machine/Entry.h>
#include <machine/interrupt.h>
#include <machine/lock.h>
#include <sys/Thread.h>
#include <kern/TaskWait.h>
#include <kern/BlockCache.h>
#include <kern/PhysMem.h>
#include <kern/debug.h>

//#define DEBUG_PHYSMEM
#ifdef DEBUG_PHYSMEM
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif

//=====================================  ===================================================

/*
 * ڡñʪ꡼ǥץ
 */
typedef struct PhysMem {
	List			list;
	short			refCount;		// ȥ
	ushort			heap_num;		// Heap number
	int				lock;			// ԥå

	// ֥åå
	int				blockCacheFlag;	// ֥ååե饰
	BlockCacheObj	blockCacheObj;	// ֥åå奪֥
} PhysMem;

enum{
	PHYSICAL_MEM_DESC_AREA = KERNEL_DATA_BEG,	// ʪ꡼ǥץꥢϥɥ쥹 */
};

// ֥ååѥե饰
enum {
	BLK_RESERVE	= 1 << 0,		// ͽ꡼
	BLK_DIRTY	= 1 << 1,		// åѹ
	BLK_CACHE	= 1 << 2,		// å夵Ƥʤ
};

//===================================== Х륤ݡ =======================================

extern void BlockCacheConstructor(BlockCacheObj*, PhysMemObj*);
extern void BlockCacheAsyncWrite(BlockCacheObj*);
extern int BlockCacheIsAsyncWrite();

//===================================== PRIVATE ====================================================

static AggregateList aggrFreeKernMem;
static AggregateList aggrFreeUserMem;
static AggregateListMethod aggrMethod;
static int freeListLock;				// ͳꥹѥԥå
static uint allMemSize;					// ʪ꡼

static void PhysMemConstructor(
	PhysMem *this)
{
	listConstructor(&this->list, this);
	this->refCount = 0;
	this->heap_num = 0;
	this->lock = 0;

	// ֥ååν
	this->blockCacheFlag = 0;
	BlockCacheConstructor(&this->blockCacheObj, (PhysMemObj*) this);
}

/*
 * ǥץɥ쥹ʪ꡼ɥ쥹
 * return : ʪ꡼ɥ쥹
 */
STATIC INLINE void *getPhysAddr(
	PhysMem *this)
{
	return (void*) (((uint) this - PHYSICAL_MEM_DESC_AREA) / sizeof(PhysMem) * PAGE_SIZE);
}

/*
 * ʪ꡼ɥ쥹ǥץɥ쥹
 * return : PhysMem address
 */
STATIC INLINE PhysMem *getDscFromPhysAddr(
	const void *physAddr)			// ʪ꡼ɥ쥹
{
	return (PhysMem*) ((uint) physAddr / PAGE_SIZE * sizeof(PhysMem) + PHYSICAL_MEM_DESC_AREA);
}

/*
 * ʪ꡼¬ꤹ
 * return : ʪ꡼
 */
STATIC uint findAllMemsize()
{
	enum {EXIST_MEM_SIZE = 0x800000};		/* ꡼¬ñ */
	uint volatile *point;

	/* ꡼ñ̤Ȥ¬ꤹ*/
	for (point = (uint*) EXIST_MEM_SIZE - 1; ; point += EXIST_MEM_SIZE / sizeof(uint)) {
		*point = 0x5a5a5a5a;
		if (*point != 0x5a5a5a5a) {
			break;
		}
		else {
			*point = 0xf0f0f0f0;
			if(*point != 0xf0f0f0f0) {
				break;
			}
		}
	}

	return (uint) point - EXIST_MEM_SIZE + sizeof(uint);
}

/*
 * ʪ꡼ǥץ
 */
STATIC void initPhysMemAll()
{
	PhysMem *this = (PhysMem*) PHYSICAL_MEM_DESC_AREA;
	int c_dsc;
	int last;

	// ʪ꡼1MbyteϻѺѤ
	for (c_dsc = 0; c_dsc < 0x100000 / PAGE_SIZE; ++c_dsc) {
		this[c_dsc].refCount = 1;
	}

	// Ĥե꡼ꤹ
	last = allMemSize / PAGE_SIZE;
	for (; c_dsc < last; ++c_dsc) {
		PhysMemConstructor(&this[c_dsc]);

		if (c_dsc * PAGE_SIZE < KERNEL_END) {
			aggrMethod.insertEnd(&aggrFreeKernMem, &this[c_dsc].list);
		}
		else {
			aggrMethod.insertEnd(&aggrFreeUserMem, &this[c_dsc].list);
		}
#ifdef DEBUG
		addFreeMemSize();
#endif
	}

	// ǥץʬʪ꡼Ƥ
	last = ROUNDUP(PHYSICAL_MEM_DESC_AREA + allMemSize / PAGE_SIZE * sizeof(PhysMem), PAGE_SIZE) / PAGE_SIZE;
	for (c_dsc = PHYSICAL_MEM_DESC_AREA / PAGE_SIZE; c_dsc < last; ++c_dsc) {
		aggrMethod.removeEntry(&aggrFreeKernMem, &this[c_dsc].list);
		this[c_dsc].refCount = 1;
#ifdef DEBUG
		subtractFreeMemSize();
#endif
	}
}

/*
 * ǥץե꡼󥯤Ф
 * return : ǥץɥ쥹 or NULL
 */
STATIC PhysMem *getFreeMem(
	AggregateList *aggrFree,
	int *spinLock)
{
	PhysMem *this;
	int eflag = enterCli();
	enter_spinlock(spinLock);
	{
		for (;;) {
			this = aggrMethod.getHead(aggrFree);
			if (this == NULL) {
				break;
			}
			if (BIT_ISSET(this->blockCacheFlag, BLK_DIRTY)) {
				exit_spinlock(spinLock);
				exitCli(eflag);

				BlockCacheAsyncWrite(&this->blockCacheObj);

				eflag = enterCli();
				enter_spinlock(spinLock);
			}
			else {
				BIT_CLR(this->blockCacheFlag, BLK_CACHE);
				this->refCount = 1;
				break;
			}
		}
	}
	exit_spinlock(spinLock);
	exitCli(eflag);

	return this;
}

/*
 * ǥץե꡼󥯤ˤĤʤ
 */
STATIC void setFreeMem(
	PhysMem *this)
{
	if (this->refCount <= 0) {
		int eflag = enterCli();
		enter_spinlock(&freeListLock);
		{
			if ((uint) getPhysAddr(this) < KERNEL_DATA_END) {
				aggrMethod.insertHead(&aggrFreeKernMem, &this->list);
			}
			else {
				aggrMethod.insertHead(&aggrFreeUserMem, &this->list);
			}
			this->heap_num = 0;
			this->refCount = 0;
		}
		exit_spinlock(&freeListLock);
		exitCli(eflag);
#ifdef DEBUG
		addFreeMemSize();
#endif
	}
}

/*
 * ֤ͥϢ³꡼ڡƤ
 *ճ߶ػߤfreeListLockäƤ뤳
 * return : ƥɥ쥹 or NULL
 */
STATIC void *getMultiFreeMem(
	PhysMem *this,		// Ƭ
	const int num)		// ƥڡ
{
	int i;

	for (i = 0; i < num; ++i) {
		if (aggrMethod.isLink(&aggrFreeKernMem, &(this + i)->list) == NO) {
			// ֥åå񤭽Ф椫֥åå񤭽Ф̤Υ
			return NULL;
		}
	}
	for (i = 0; i < num; ++i) {
		aggrMethod.removeEntry(&aggrFreeKernMem, &(this + i)->list);
		(this + i)->refCount = 1;
		BIT_CLR((this + i)->blockCacheFlag, BLK_CACHE);
#ifdef DEBUG
		subtractFreeMemSize();
#endif
	}

	return getPhysAddr(this);
}

//--------------------------------------------------------------------------------------------------
// ֥åå
//--------------------------------------------------------------------------------------------------

static AggregateList aggrReserveList;		// ͽ꡼ꥹȽ
static int reserveListLock;					// ͽꥹѥԥå

/*
 * åԤ
 */
STATIC void waitCache(
	AggregateList *aggrFree)
{
	while (aggrMethod.getCount == 0);
}

//===================================== PUBLIC =====================================================

void PhysMemInit()
{
	// ʪ꡼¬ꤹ
	allMemSize = findAllMemsize();
	printk("Memory size = %d MB\n", allMemSize / 0x100000);

	AggregateListConstructor(&aggrFreeKernMem, &aggrMethod);
	AggregateListConstructor(&aggrFreeUserMem, &aggrMethod);

	// ʪ꡼ǥץ
	initPhysMemAll();
}

/*
 * ʪ꡼
 * return : ʪ꡼
 */
uint getPhysMemSize()
{
	return allMemSize;
}

/*
 * ֥ͥڡ꡼Ƥ
 * return : ʪ꡼ or NULL
 */
void *allocKernelPage()
{
	PhysMem *this;

	this = getFreeMem(&aggrFreeKernMem, &freeListLock);
	if (this == NULL) {
		return NULL;
	}
#ifdef DEBUG
	subtractFreeMemSize();
#endif
	return getPhysAddr(this);
}

/*
 * 桼֥ڡ꡼Ƥ
 * Ƥڡϲ
 * return : ʪ꡼ or NULL
 */
void *allocUserPage()
{
	PhysMem *this;

	this = getFreeMem(&aggrFreeUserMem, &freeListLock);
	if (this == NULL) {
		return allocKernelPage();
	}
#ifdef DEBUG
	subtractFreeMemSize();
#endif
	return getPhysAddr(this);
}

/*
 * ֤ͥϢ³꡼ڡƤ
 * return : ƥɥ쥹 or NULL
 */
void *allocMultiKernelPage(
	const int num)
{
	uint maxmem = (allMemSize < KERNEL_DATA_END) ? allMemSize : KERNEL_DATA_END;
	PhysMem *this;
	void *rest = NULL;
	int again;

	do {
		int eflag = enterCli();
		enter_spinlock(&freeListLock);
		{
			int cnt = 0;
			again = 0;

			for(this = getDscFromPhysAddr((const void*) maxmem) - 1 ;(PhysMem*) PHYSICAL_MEM_DESC_AREA <= this; --this) {
				if (this->refCount == 0) {
					if ((BIT_ISSET(this->blockCacheFlag, BLK_DIRTY)) && (aggrMethod.isLink(&aggrFreeKernMem, &this->list) == YES)) {
						aggrMethod.removeEntry(&aggrFreeKernMem, &this->list);
						exit_spinlock(&freeListLock);
						exitCli(eflag);

						BlockCacheAsyncWrite(&this->blockCacheObj);

						eflag = enterCli();
						enter_spinlock(&freeListLock);
					}
					if (++cnt == num) {
						rest = getMultiFreeMem(this, num);
						if (rest != NULL) {
							break;
						}
						again = 1;
						cnt=0;
					}
				}
				else {
					cnt=0;
				}
			}
		}
		exit_spinlock(&freeListLock);
		exitCli(eflag);
	} while ((rest == NULL) && (again == 1));

	return rest;
}

/*
 * ʪ꡼ڡλȥȤ䤹
 */
void linkPhysTable(
	const void *i_addr)		// ʪɥ쥹
{
	PhysMem *this = getDscFromPhysAddr(i_addr);
	++this->refCount;
}

/*
 * ʪ꡼ڡλȥȤ򸺤餹
 */
void unlinkPhysTable(
	const void *addr)		// ʪɥ쥹
{
	PhysMem *this = getDscFromPhysAddr(addr);

	--this->refCount;
#ifdef DEBUG
	if (this->refCount < 0) {
		uint *esp = (uint*) &addr - 1;
		printk("Warning! This physTable has unlinked : addr=0x%x function=0x%x stack=0x%x 0x%x 0x%x\n", 
			addr, esp[0], esp[8], esp[9], esp[10]);
	}
#endif
	setFreeMem(this);
}

/*
 *꡼ȥإåֹꤹ
 */
void setAllocNum(
	const void *addr,
	const int num)
{
	getDscFromPhysAddr(addr)->heap_num = num;
}

/*
 * ʪ꡼ڡԥå
 */
void enterPhysMemLock(
	const void *i_addr)
{
	enter_spinlock(&getDscFromPhysAddr(i_addr)->lock);
}

/*
 * ʪ꡼ڡԥå
 */
void exitPhysMemLock(
	const void *i_addr)
{
	exit_spinlock(&getDscFromPhysAddr(i_addr)->lock);
}

//--------------------------------------------------------------------------------------------------
// Getter
//--------------------------------------------------------------------------------------------------

/*
 * ʪ꡼ڡλȥȤ
 * return : ȥ
 */
int getPhysPageLinkCount(
	const void *i_addr)		// ʪɥ쥹
{
	return getDscFromPhysAddr(i_addr)->refCount;
}

/*
 *꡼ȥإåֹ
 * return : ȥإåֹ
 */
int getAllocNum(
	const void *addr)
{
	return getDscFromPhysAddr(addr)->heap_num;
}

//--------------------------------------------------------------------------------------------------
// ֥åå
//--------------------------------------------------------------------------------------------------

/*
 * ֥ååͽ꡼Ƥ
 * return : error number
 */
int PhysMemAllocBlockCacheReserve(
	const uint size)	// ͽ󥵥ڡڤ
{
	int i;

	AggregateListConstructor(&aggrReserveList, &aggrMethod);

	for (i = 0; i < size / PAGE_SIZE; ++i) {
		PhysMem *this = getFreeMem(&aggrFreeKernMem, &freeListLock);
		if (this == NULL) {
			return -ENOMEM;
		}

		BIT_SET(this->blockCacheFlag, BLK_RESERVE);
		aggrMethod.insertEnd(&aggrReserveList, &this->list);
#ifdef DEBUG
		subtractFreeMemSize();
#endif
	}

	return NOERR;
}

/*
 * ͳꥹȤ鳺å
 *BLK_NOCACHEե饰ϥȥߥå˥åȡȤʤФʤ餤ΤǡPhysMemGetNewCache()Ʊå˸ƤФ뤳
 * return : BlockCacheObj or NULL
 */
BlockCacheObj *PhysMemGetCache(
	PhysMemObj *physMemObj)
{
	PhysMem *this = (PhysMem*) physMemObj;
	AggregateList *aggrList;
	int *spinLock;
	BlockCacheObj *blockCacheObj = NULL;
	int eflag;
	
	if (BIT_ISSET(this->blockCacheFlag, BLK_RESERVE)) {
		aggrList = &aggrReserveList;
		spinLock = &reserveListLock;
	}
	else {
		aggrList = &aggrFreeKernMem;
		spinLock = &freeListLock;
	}

	eflag = enterCli();
	enter_spinlock(spinLock);
	{
		for (;;) {
			if (BIT_ISSET(this->blockCacheFlag, BLK_CACHE)) {
				if (aggrMethod.isLink(aggrList, &this->list) == YES) {
					// åФ
					aggrMethod.removeEntry(aggrList, &this->list);
					blockCacheObj = &this->blockCacheObj;
					break;
				}
				else {
					// åȤƼФ椫Ʊ񤭹Ԥ
					exit_spinlock(spinLock);
					exitCli(eflag);

					waitCache(aggrList);

					eflag = enterCli();
					enter_spinlock(spinLock);
				}
			}
			else {
				// ˥ȤƤ
				break;
			}
		}
	}
	exit_spinlock(spinLock);
	exitCli(eflag);

	return blockCacheObj;
}

/*
 * ͳꥹƬΥå
 *BLK_NOCACHEե饰ϥȥߥå˥åȡȤʤФʤ餤ΤǡPhysMemGetCache()Ʊå˸ƤФ뤳
 * return : BlockCacheObj
 */
BlockCacheObj *PhysMemGetNewCache()
{
	PhysMem *this;

	for (;;) {
		this = getFreeMem(&aggrFreeKernMem, &freeListLock);
		if (this != NULL) {
			break;
		}

		this = getFreeMem(&aggrReserveList, &reserveListLock);
		if (this != NULL) {
			break;
		}

		waitCache(&aggrReserveList);
	}

	this->refCount = 0;
	BIT_SET(this->blockCacheFlag, BLK_CACHE);

	return &this->blockCacheObj;
}

/*
 * åͳꥹȤκǸ᤹
 */
void PhysMemPutCache(
	PhysMemObj *physMemObj)
{
	PhysMem *this = (PhysMem*) physMemObj;
	AggregateList *aggrList;
	int *spinLock;
	int eflag;

	if (BIT_ISSET(this->blockCacheFlag, BLK_RESERVE)) {
		aggrList = &aggrReserveList;
		spinLock = &reserveListLock;
	}
	else {
		aggrList = &aggrFreeKernMem;
		spinLock = &freeListLock;
	}

	eflag = enterCli();
	enter_spinlock(spinLock);
	{
		aggrMethod.insertEnd(aggrList, &this->list);
	}
	exit_spinlock(spinLock);
	exitCli(eflag);
}

/*
 * åͳꥹȤƬ᤹
 */
void PhysMemPutBackCache(
	PhysMemObj *physMemObj)
{
	PhysMem *this = (PhysMem*) physMemObj;
	AggregateList *aggrList;
	int *spinLock;
	int eflag;

	if (BIT_ISSET(this->blockCacheFlag, BLK_RESERVE)) {
		aggrList = &aggrReserveList;
		spinLock = &reserveListLock;
	}
	else {
		aggrList = &aggrFreeKernMem;
		spinLock = &freeListLock;
	}

	eflag = enterCli();
	enter_spinlock(spinLock);
	{
		aggrMethod.insertHead(aggrList, &this->list);
	}
	exit_spinlock(spinLock);
	exitCli(eflag);
}

/*
 * ߤꥭå򤹤٤Ʊ񤭹ߤ
 */
void PhysMemSync()
{
	PhysMem *this;
	int eflag;

	for (this = aggrMethod.refHead(&aggrFreeKernMem); this != NULL;) {
		PhysMem *next = aggrMethod.refNext(&aggrFreeKernMem, &this->list);

		if (BIT_ISSET(this->blockCacheFlag, BLK_DIRTY)) {
			eflag = enterCli();
			enter_spinlock(&freeListLock);
			{
				aggrMethod.removeEntry(&aggrFreeKernMem, &this->list);
			}
			exit_spinlock(&freeListLock);
			exitCli(eflag);

			// ǥХƱ񤭹
			BlockCacheAsyncWrite(&this->blockCacheObj);
		}
		this = next;
	}

	// ֥ååͽʬ
	for (this = aggrMethod.refHead(&aggrReserveList); this != NULL;) {
		PhysMem *next = aggrMethod.refNext(&aggrReserveList, &this->list);

		if (BIT_ISSET(this->blockCacheFlag, BLK_DIRTY)) {
			eflag = enterCli();
			enter_spinlock(&freeListLock);
			{
				aggrMethod.removeEntry(&aggrReserveList, &this->list);
			}
			exit_spinlock(&freeListLock);
			exitCli(eflag);

			// ǥХƱ񤭹
			BlockCacheAsyncWrite(&this->blockCacheObj);
		}
		this = next;
	}

	// Ʊ񤭹߽λԤ
	while (BlockCacheIsAsyncWrite() == YES) {
		mili_timer(TASK_TIME);
	}
}

/*
 * å˹ե饰ΩƤ
 */
void PhysMemSetDirty(
	PhysMemObj *physMemObj)
{
	PhysMem *this = (PhysMem*) physMemObj;
	BIT_SET(this->blockCacheFlag ,BLK_DIRTY);
}

/*
 * å˹ե饰򥯥ꥢ
 */
void PhysMemClearDirty(
	PhysMemObj *physMemObj)
{
	PhysMem *this = (PhysMem*) physMemObj;
	BIT_CLR(this->blockCacheFlag ,BLK_DIRTY);
}

/*
 * ʪ꡼ɥ쥹
 * return : ʪ꡼ɥ쥹
 */
char *PhysMemGetPhysAddr(
	PhysMemObj *physMemObj)
{
	PhysMem *this = (PhysMem*) physMemObj;
	return getPhysAddr(this);
}

//--------------------------------------------------------------------------------------------------
// ƥॳ
//--------------------------------------------------------------------------------------------------

/*
 * å夫ǥХ˥ǡ᤹
 */
int sys_sync()
{
	PhysMemSync();

	return 0;
}

//**************************************************************************************************

#ifdef DEBUG
#include <lib/IteratorList.h>

void test_physmem(
	void *addr)
{
	IteratorList iterator;
	
	IteratorListConstruct(&iterator, &aggrReserveList, &aggrMethod);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		PhysMem *next = iterator.next(&iterator);
		if ((void*) getPhysAddr(next) == addr) {
			printk("test_physmem %x\n", addr);
			idle();
		}
	}
}
#endif
