/*
 * TaskWait.c
 *
 * Copyright 2008, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 *Գסե꡼ס
 *
 */

#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/buf.h>
#include <lib/lib.h>
#include <lib/IteratorList.h>
#include <lib/AggregateList.h>
#include <machine/interrupt.h>
#include <machine/lock.h>
#include <kern/ProcSignal.h>
#include <kern/timer.h>
#include <sys/Thread.h>
#include <kern/TaskWait.h>

#include <kern/debug.h>

#define DEBUG_TASK_WAIT 1
#ifdef DEBUG_TASK_WAIT
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif

//=====================================  ===================================================

/*
 * ȹ¤
 *ԱƶեСѹVmTaskObjΥѹ뤳
 */
typedef struct {
	ThreadObj	*task;
	List		list;			// ꥹ
	int			atomic;			// ȥߥåʵԤե饰
	int			awake;			// Ԥե饰
	uint		ident;			// 
	int			spinLock;		// ԥå
} TaskWaitAttr;

/*
 * Ƚ繽¤
 */
typedef struct {
	AggregateList		aggregate;		// ȥꥹȥإå
	volatile uint		semaphore;		// ޥե
	volatile int 		refCnt;			// 楫
	int 				nestCnt;		// 楫
	int					spinLock;		// ԥå
} WaitAggr;

/*
 * ̥ȹ¤
 *ԱƶեСѹVmTaskObjΥѹ뤳
 */
typedef struct {
	AggregateList	aggregate;			// ԤꥹȽ
	uint			ident;				// 
	int				spinLock;			// ԥå
} WaitIdent;

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static AggregateListMethod	aggrMethod;		// ꥹȥ᥽å

/*
 * ȥߥåʵԤ
 *ոƤӽФ¾򤫤뤳
 * return : YES = 塼뤫 or NO
 */
STATIC int waitAtomic(
	TaskWaitAttr *this,
	const int taskState)
{
	if (this->atomic == 0) {
		this->atomic = 1;
		delFromSchedule(taskState);
		return YES;
	}
	else {
		this->atomic = 0;
		return NO;
	}
}

/*
 * ȥߥåʵԤε
 *ոƤӽФ¾򤫤뤳
 *Գ
 * return : YES = NO = Ƥʤ
 */
STATIC int awakeAtomic(
	TaskWaitAttr *this,
	const int taskState,
	int (*addScheduleMethod)(ThreadObj*, const int))
{
	if (addScheduleMethod(this->task, taskState) == YES) {
		addScheduleMethod(this->task, taskState);
//		this->atomic = 0;
		return YES;
	}
	else {
		this->atomic = 1;
		return NO;
	}
}

/*
 * Ԥ
 *ոƤӽФ¾򤫤뤳
 * return : YES = 塼뤫 or NO
 */
STATIC int wait(
	TaskWaitAttr *this,
	const int taskState)
{
	if (this->awake == 0) {
		delFromSchedule(taskState);
		return YES;
	}
	else {
		this->awake = 0;
		return NO;
	}
}

/*
 * Ԥε
 *ոƤӽФ¾򤫤뤳
 *Գ
 * return : YES = NO = Ƥʤ
 */
STATIC int awake(
	TaskWaitAttr *this,
	const int taskState,
	int (*addScheduleMethod)(ThreadObj*, const int))
{
	if (this->atomic == 0) {
		if (addScheduleMethod(this->task, taskState) == YES) {
			this->awake = 0;
			return YES;
		}
	}
	this->awake = 1;
	return NO;
}

//--------------------------------------------------------------------------------------------------
// ̥
//--------------------------------------------------------------------------------------------------

/*
 * ̥Ԥޡ
 * ޡ֤ʤ饿ޡϥåȤʤ
 * return : error number
 */
STATIC int waitTimerIdentify(
	TaskWaitAttr *this,
	WaitIdent *waitIdent,
	const uint ident,
	const uint miliSeconds,
	uint *o_remainTime)			// Ĥ ms
{
	void *timer;
	int isDelSchedule;
	int eflag;

	if (0 < miliSeconds){
		int error = setTaskTimer(miliSeconds, 0, (void*) TaskAwake, this, &timer);
		if (error != NOERR) {
			return error;
		}
	}

	eflag = enterCli();
	enter_spinlock(&waitIdent->spinLock);
	{
		if (waitIdent->ident == ident) {
			waitIdent->ident = 0;
			goto EXIT;
		}

		enter_spinlock(&this->spinLock);
		{
			ASSERT(this->atomic == 0);

			isDelSchedule = wait(this, TASK_WAIT);
		}
		exit_spinlock(&this->spinLock);
		if (isDelSchedule == YES) {
			this->ident = ident;
			aggrMethod.insertEnd(&waitIdent->aggregate, &this->list);
			exit_spinlock(&waitIdent->spinLock);

			wait_task();

			enter_spinlock(&waitIdent->spinLock);
			aggrMethod.removeEntry(&waitIdent->aggregate, &this->list);
		}
	}
EXIT:
	exit_spinlock(&waitIdent->spinLock);
	exitCli(eflag);

	if (0 < miliSeconds){
		*o_remainTime = getRemainTime(timer);
		resetTimer(timer);
	}
	else {
		if (o_remainTime != NULL) {
			*o_remainTime = 0;
		}
	}

	return NOERR;
}

STATIC void wakeIdenify(
	WaitIdent *waitIdent,
	const uint ident)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitIdent->spinLock);
	{
		IteratorList iterator;

		waitIdent->ident = ident;
		IteratorListConstruct(&iterator, &waitIdent->aggregate, &aggrMethod);
		while (iterator.hasNext(&iterator) == BOOL_TRUE) {
			TaskWaitAttr *next = iterator.next(&iterator);
			if (next->ident == ident) {
				awake(next, TASK_WAIT, addScheduleSoon);
				waitIdent->ident = 0;
			}
		}
		
	}
	exit_spinlock(&waitIdent->spinLock);
	exitCli(eflag);
}

//--------------------------------------------------------------------------------------------------
// Ԥ
//--------------------------------------------------------------------------------------------------

/*
 * Ԥޡ
 * ޡ֤ʤ饿ޡϥåȤʤ
 * return : error number
 */
STATIC int waitTimer(
	TaskWaitAttr *this,
	const uint miliSeconds,
	int (*waitMethod)(TaskWaitAttr*, const int),
	int (*awakeMethod)(TaskWaitObj*),
	const int taskState,
	uint *o_remainTime)			// Ĥ ms
{
	void *timer;
	int isDelSchedule;
	int eflag;

	if (0 < miliSeconds){
		int error = setTaskTimer(miliSeconds, 0, (void*) awakeMethod, this, &timer);
		if (error != NOERR) {
			return error;
		}
	}

	eflag = enterCli();
	enter_spinlock(&this->spinLock);
	{
		isDelSchedule = waitMethod(this, taskState);
	}
	exit_spinlock(&this->spinLock);
	if (isDelSchedule == YES) {
		wait_task();
		if (waitMethod == waitAtomic) {
			this->atomic = 0;
		}
		else {
			this->awake = 0;
		}
	}
	exitCli(eflag);

	if (o_remainTime != NULL) {
		if (0 < miliSeconds){
			*o_remainTime = getRemainTime(timer);
		}
		else {
			*o_remainTime = 0;
		}
	}

	if (0 < miliSeconds){
		resetTimer(timer);
	}

	return NOERR;
}

/*
 * 
 * return : YES = NO = Ƥʤ
 */
STATIC int awakeTask(
	TaskWaitAttr *this,
	int (*awakeMethod)(TaskWaitAttr*, const int, int (*)(ThreadObj*, const int)),
	const int taskState,
	int (*addScheduleMethod)(ThreadObj*, const int))
{
	int isWake;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&this->spinLock);
	{
		isWake = awakeMethod(this, taskState, addScheduleMethod);
	}
	exit_spinlock(&this->spinLock);
	exitCli(eflag);
	
	return isWake;
}

//===================================== PUBLIC =====================================================

//--------------------------------------------------------------------------------------------------
// 󥹥ȥ饯
//--------------------------------------------------------------------------------------------------

/*
 * ȥ󥹥ȥ饯
 */
void taskWaitConstruct(
	TaskWaitObj *i_this,
	ThreadObj *i_task)
{
	TaskWaitAttr *this = (TaskWaitAttr*) i_this;

#ifdef DEBUG
	if (sizeof(TaskWaitAttr) != sizeof(TaskWaitObj)) {
		printk("taskWaitConstruct() object size error! TaskWait=%d TaskWaitObj=%d\n", sizeof(TaskWaitAttr), sizeof(TaskWaitObj));
		idle();
	}
#endif
	memset(this, 0, sizeof(*this));
	listConstructor(&this->list, this);
	this->task = i_task;
}

/*
 * ޥեȥ󥹥ȥ饯
 */
void WaitAggrConstructor(
	WaitAggrObj *waitAggrObj)
{
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;

#ifdef DEBUG
	if (sizeof(WaitAggr) != sizeof(WaitAggrObj)) {
		printk("waitConstruct() object size error! WaitAggr=%d WaitAggrObj=%d\n", sizeof(WaitAggr), sizeof(WaitAggrObj));
		idle();
	}
#endif
	memset(waitAggr, 0, sizeof(*waitAggr));
	AggregateListConstructor(&waitAggr->aggregate, &aggrMethod);
}

/*
 * ̥ȥ󥹥ȥ饯
 */
void WaitIdentConstruct(
	WaitIdentObj *waitIdentObj)
{
	WaitIdent *waitIdent = (WaitIdent*) waitIdentObj;

#ifdef DEBUG
	if (sizeof(WaitIdent) != sizeof(WaitIdentObj)) {
		printk("WaitIdentConstruct() object size error! waitIdent=%d waitIdentObj=%d\n", 
			sizeof(waitIdent), sizeof(waitIdentObj));
		idle();
	}
#endif
	memset(waitIdent, 0, sizeof(*waitIdent));
	AggregateListConstructor(&waitIdent->aggregate, &aggrMethod);
}

/*
 * ̥ȥǥȥ饯
 */
void WaitIdentDestructor(
	WaitIdentObj *waitIdentObj)
{
	WaitIdent *waitIdent = (WaitIdent*) waitIdentObj;
	int eflag;

	// ³ꥹȤγ
	eflag = enterCli();
	enter_spinlock(&waitIdent->spinLock);
	{
		TaskWaitAttr *next;
		while ((next = aggrMethod.getHead(&waitIdent->aggregate)) != NULL) {
			addToSchedule(next->task, TASK_WAIT);
		}
	}
	exit_spinlock(&waitIdent->spinLock);
	exitCli(eflag);
}

/*
 * TaskWaitObjꥻåȤ
 */
void TaskWaitReset(
	TaskWaitObj *i_this)
{
	TaskWaitAttr *this = (TaskWaitAttr*) i_this;
	this->atomic = 0;
}

//--------------------------------------------------------------------------------------------------
// Ԥ
//--------------------------------------------------------------------------------------------------

/*
 * ̤꥿Ԥ
 */
void TaskWaitAtomic()
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	waitTimer(this, 0, waitAtomic, NULL, TASK_WAIT, NULL);
}

/*
 * ̤Ԥ
 *Գ
 * return : YES = NO = Ƥʤ
 */
int TaskAwakeAtomicSoon(
	TaskWaitObj *i_this)
{
	TaskWaitAttr *this = (TaskWaitAttr*) i_this;
	return awakeTask(this, awakeAtomic, TASK_WAIT, addScheduleSoon);
}

/*
 * Ԥ
 */
void TaskWait()
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	waitTimer(this, 0, wait, NULL, TASK_WAIT, NULL);
}

/*
 * Ԥޡ
 * return : Ĥ or error number
 */
int TaskWaitTimer(
	const int miliSeconds)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	uint remainTime;
	int error;

	if (miliSeconds == 0){
		return 0;
	}
	error = waitTimer(this, miliSeconds, wait, TaskAwake, TASK_WAIT, &remainTime);
	if (error != NOERR) {
		return error;
	}
	return remainTime;
}

/*
 * Ԥ
 *Գ
 * return : YES = NO = Ƥʤ
 */
int TaskAwake(
	TaskWaitObj *i_this)
{
	TaskWaitAttr *this = (TaskWaitAttr*) i_this;
	return awakeTask(this, awake, TASK_WAIT, addToSchedule);
}

/*
 * Ԥ
 *Գ
 * return : YES = NO = Ƥʤ
 */
int TaskAwakeSoon(
	TaskWaitObj *i_this)
{
	TaskWaitAttr *this = (TaskWaitAttr*) i_this;
	return awakeTask(this, awake, TASK_WAIT, addScheduleSoon);
}

/*
 * ǡ󥿥Ԥ
 */
void TaskWaitDaemon()
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	waitTimer(this, 0, wait, NULL, TASK_INTR_WAIT, NULL);
}

/*
 * ǡԤ
 *Գ
 * return : YES = NO = Ƥʤ
 */
int TaskWaitDaemonActiveSoon(
	TaskWaitObj *i_this)
{
	TaskWaitAttr *this = (TaskWaitAttr*) i_this;
	return awakeTask(this, awake, TASK_INTR_WAIT, addScheduleSoon);
}

/*
 * λ
 */
int exitSchedule()
{
	cli();
	delFromSchedule(TASK_EXIT);

	return YES;
}

/*
 * ߤ
 *奿åޤǳ߶ػߤ뤳
 * return : task switch YES or NO
 */
int stopSchedule()
{
	cli();
	delFromSchedule(TASK_SIGNAL_STOP);

	return YES;
}

/*
 *FreeBSD
 * General sleep call.  Suspends the current process until a wakeup is
 * performed on the specified identifier.  The process will then be made
 * runnable with the specified priority.  Sleeps at most timo/hz seconds
 * (0 means no timeout).  If pri includes PCATCH flag, signals are checked
 * before and after sleeping, else signals are not checked.  Returns 0 if
 * awakened, EWOULDBLOCK if the timeout expires.  If PCATCH is set and a
 * signal needs to be delivered, ERESTART is returned if the current system
 * call should be restarted if possible, and EINTR is returned if the system
 * call should be interrupted by the signal (return EINTR).
 */
int tsleep(
	void *chan, 
	int pri, 
	const char *wmesg, 
	int timeout)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	struct buf *bp = chan;
	uint remainTime;
	int error;

	bp->task = getCurrentTask();
	error = waitTimer(this, timeout, wait, TaskAwake, TASK_WAIT, &remainTime);
	if (error != NOERR) {
		return error;
	}
	if (pri & PCATCH) {
		if (isSigint(TaskGetTaskSignal(getCurrentTask())) == YES) {
			return -EINTR;
		}
	}
	if (remainTime == 0) {
		return -EWOULDBLOCK;
	}

	return NOERR;
}

//--------------------------------------------------------------------------------------------------
// Ԥ塼
//--------------------------------------------------------------------------------------------------

/*
 * ꡼Ԥ
 */
void WaitSleep(
	WaitAggrObj *waitAggrObj)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;
	int isDelSchedule;
	int eflag;
	
	eflag = enterCli();
	enter_spinlock(&waitAggr->spinLock);
	{
		aggrMethod.insertEnd(&waitAggr->aggregate, &this->list);
		enter_spinlock(&this->spinLock);
		{
			isDelSchedule = waitAtomic(this, TASK_WAIT);
		}
		exit_spinlock(&this->spinLock);
	}
	exit_spinlock(&waitAggr->spinLock);
	if (isDelSchedule == YES) {
		wait_task();
		this->atomic = 0;
	}
	exitCli(eflag);
}

/*
 * ꡼ץ򵯤
 */
void WaitWake(
	WaitAggrObj *waitAggrObj)
{
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitAggr->spinLock);
	{
		for (;;) {
			TaskWaitAttr *next = aggrMethod.getHead(&waitAggr->aggregate);
			if (next == NULL) {
				break;
			}
			awakeTask(next, awakeAtomic, TASK_WAIT, addScheduleSoon);
		}
	}
	exit_spinlock(&waitAggr->spinLock);
	exitCli(eflag);
}

//--------------------------------------------------------------------------------------------------
// ޥեѲǽԤ
//--------------------------------------------------------------------------------------------------

/*
 * ޥեԲĤʤ饿Ԥ
 */
void WaitGetSemaphore(
	WaitAggrObj *waitAggrObj)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;

	for (;;) {
		int eflag = enterCli();
		enter_spinlock(&waitAggr->spinLock);

		// ޥեѲ
		if ((waitAggr->semaphore == 0) && (waitAggr->refCnt == 0)) {
			waitAggr->semaphore = (uint) this;
			exit_spinlock(&waitAggr->spinLock);
			exitCli(eflag);
			break;
		}
		// ޥեԲ
		else {
			int isDelSchedule;

			aggrMethod.insertEnd(&waitAggr->aggregate, &this->list);
			enter_spinlock(&this->spinLock);
			{
				isDelSchedule = waitAtomic(this, TASK_WAIT);
			}
			exit_spinlock(&this->spinLock);
			exit_spinlock(&waitAggr->spinLock);
			if (isDelSchedule == YES) {
				wait_task();
				this->atomic = 0;
			}
			exitCli(eflag);
		}
	}
}

/*
 * ޥեѲĤˤ
 */
void WaitReleaseSemaphore(
	WaitAggrObj *waitAggrObj)
{
	TaskWaitAttr *this;
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitAggr->spinLock);
	{
		waitAggr->semaphore = 0;
		this = aggrMethod.getHead(&waitAggr->aggregate);
	}
	exit_spinlock(&waitAggr->spinLock);
	exitCli(eflag);

	if (this != NULL) {
		awakeTask(this, awakeAtomic, TASK_WAIT, addScheduleSoon);
	}
}

/*
 * ȤǥޥեԲĤʤ饿Ԥ
 */
void WaitReference(
	WaitAggrObj *waitAggrObj)
{
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;

	for (;;) {
		int eflag = enterCli();
		enter_spinlock(&waitAggr->spinLock);

		// ޥեѲ
		if (waitAggr->semaphore == 0) {
			++waitAggr->refCnt;
			exit_spinlock(&waitAggr->spinLock);
			exitCli(eflag);
			break;
		}
		// ޥեԲ
		else {
			TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
			int isDelSchedule;

			aggrMethod.insertEnd(&waitAggr->aggregate, &this->list);
			enter_spinlock(&this->spinLock);
			{
				isDelSchedule = waitAtomic(this, TASK_WAIT);
			}
			exit_spinlock(&this->spinLock);
			exit_spinlock(&waitAggr->spinLock);
			if (isDelSchedule == YES) {
				wait_task();
				this->atomic = 0;
			}
			exitCli(eflag);
		}
	}
}

/*
 * 
 */
void WaitEndReference(
	WaitAggrObj *waitAggrObj)
{
	TaskWaitAttr *this;
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;
	int eflag;

	ASSERT(0 < waitAggr->refCnt);

	eflag = enterCli();
	enter_spinlock(&waitAggr->spinLock);
	{
		waitAggr->refCnt -= 1;
		this = aggrMethod.getHead(&waitAggr->aggregate);
	}
	exit_spinlock(&waitAggr->spinLock);
	exitCli(eflag);
	if (this != NULL) {
		awakeTask(this, awakeAtomic, TASK_WAIT, addScheduleSoon);
	}
}

//--------------------------------------------------------------------------------------------------
// ƥॳ
//--------------------------------------------------------------------------------------------------

uint sys_sleep(
	uint seconds)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	uint miliSeconds;

	if (seconds == 0){
		return 0;
	}

	if (waitTimer(this, seconds * 1000, wait, TaskAwake, TASK_WAIT, &miliSeconds) != NOERR) {
		return seconds;
	}

	return miliSeconds * 1000;
}

/*
 * ̥Ԥ
 * General sleep call.  Suspends the current process until a wakeup is
 * performed on the specified identifier.  The process will then be made
 * runnable with the specified priority.  Sleeps at most timo/hz seconds
 * (0 means no timeout).  If pri includes PCATCH flag, signals are checked
 * before and after sleeping, else signals are not checked.  Returns 0 if
 * awakened, EWOULDBLOCK if the timeout expires.  If PCATCH is set and a
 * signal needs to be delivered, ERESTART is returned if the current system
 * call should be restarted if possible, and EINTR is returned if the system
 * call should be interrupted by the signal (return EINTR).
 *եɥ饤С⥸塼뤫ƤФ
 * return : error number
 */
int sysTsleep(
	WaitIdent *waitIdent,
	const uint ident,
	const int timeout,		// ॢms
	const int priority)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	uint remainTime;
	int error;
	
	error = waitTimerIdentify(this, waitIdent, ident, timeout, &remainTime);
	if (error != NOERR) {
		return error;
	}

	if (priority & PCATCH) {
		if (isSigint(TaskGetTaskSignal(getCurrentTask())) == YES) {
			return -EINTR;
		}
	}
	if (0 < timeout) {
		if (remainTime == 0) {
			return -EWOULDBLOCK;
		}
	}

	return 0;
}

/*
 * ̥
 *եɥ饤С⥸塼뤫ƤФ
 * return : 0
 */
int sysWakeup(
	WaitIdent *waitIdent,
	const uint ident)
{
	wakeIdenify(waitIdent, ident);
	return 0;
}

/*
 *եɥ饤С⥸塼뤫ƤФ
 * return : 0
 */
int sysAwake(
	TaskWaitAttr *this)
{
	awakeTask(this, awake, TASK_WAIT, addScheduleSoon);
	return 0;
}

/*
 * Ԥ
 *Գ
 * return : 0
 */
int sysWaitIntr(
	const uint miliSeconds)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	waitTimer(this, miliSeconds, waitAtomic, TaskAwakeAtomicSoon, TASK_WAIT, NULL);

	return 0;
}

/*
 * Ԥ
 *Գ
 * return : 0
 */
int sysAwakeIntr(
	ThreadObj *threadObj)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(threadObj);
	awakeTask(this, awakeAtomic, TASK_WAIT, addScheduleSoon);
	
	return 0;
}

/*
 * ޥեμǤʤеԤ
 * return : 0
 */
int sysGetSemaphore(
	WaitAggrObj *waitAggrObj)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;

	// ޥեƤ饹롼
	if (waitAggr->semaphore != (uint) this) {
		WaitGetSemaphore(waitAggrObj);

		ASSERT(waitAggr->nestCnt == 0);
	}
	++waitAggr->nestCnt;

	return 0;
}

/*
 * ޥեγ
 * return : 0
 */
int sysReleaseSemaphore(
	WaitAggrObj *waitAggrObj)
{
	TaskWaitAttr *this = (TaskWaitAttr*) getTaskWait(getCurrentTask());
	WaitAggr *waitAggr = (WaitAggr*) waitAggrObj;

	ASSERT(waitAggr->semaphore == (uint) this);
	ASSERT(0 < waitAggr->nestCnt);

	if (--waitAggr->nestCnt == 0) {
		WaitReleaseSemaphore(waitAggrObj);
	}
	return 0;
}

//**************************************************************************************************

#ifdef DEBUG
void testTaskWait(
	uint from)
{
	printDebug(80 * 8 + 70, "%x", from);
}
#endif
