/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/uio.h>
#include <lib/IteratorList.h>
#include <lib/AggregateList.h>
#include <lib/lib.h>
#include <lib/lib_path.h>
#include <machine/interrupt.h>
#include <machine/lock.h>
#include <module/Module.h>
#include <kern/kmalloc.h>
#include <kern/timer.h>
#include <kern/ProcSignal.h>
#include <kern/BlockCache.h>
#include <kern/devfs.h>
#include <kern/TaskWait.h>
#include <kern/device.h>

#include <kern/debug.h>

//=====================================  ===================================================

/*
 * ǥХ¤Υꥹȥإå
 */
typedef struct {
	AggregateList		aggregate;		// ǥХꥹȥإå
	AggregateListMethod	aggrMethod;		// ꥹȥ᥽å
	int					lock;			// ԥå
} DEV_SPEC_HEAD;

//===================================== Х륤ݡ =======================================

//================================== PRIVATE ============================================

static DEV_SPEC_HEAD devSpecHead;

/*
 * ǥХ̾õ
 * return : struct specinfo or NULL
 */
static struct specinfo *searchDevByName(
	const char *i_name)
{
	IteratorList iterator;
	struct specinfo *dev;

	enter_spinlock(&devSpecHead.lock);
	{
		IteratorListConstruct(&iterator, &devSpecHead.aggregate, &devSpecHead.aggrMethod);
		for (; ; ) {
			if (iterator.hasNext(&iterator) == BOOL_FALSE) {
				dev = NULL;
				break;
			}

			dev = iterator.next(&iterator);
			if (strcmp(dev->si_name, i_name) == 0) {
				break;
			}
		}
	}
	exit_spinlock(&devSpecHead.lock);
	
	return dev;
}

/*
 * ǥХ̾õ
 * return : struct specinfo or NULL
 */
static struct specinfo *searchDevByDevnum(
	const int i_devNum)
{
	struct specinfo *dev;
	IteratorList iterator;

	enter_spinlock(&devSpecHead.lock);
	{
		IteratorListConstruct(&iterator, &devSpecHead.aggregate, &devSpecHead.aggrMethod);
		for (; ; ) {
			if (iterator.hasNext(&iterator) == BOOL_FALSE) {
				dev = NULL;
				break;
			}

			dev = iterator.next(&iterator);
			if (dev->si_udev == i_devNum) {
				break;
			}
		}
	}
	exit_spinlock(&devSpecHead.lock);
	
	return dev;
}

/*
 * ǥХϿ
 * return : struct specinfo or NULL
 */
static struct specinfo *registDev(
	struct cdevsw *i_cdevsw,
	const size_t i_blockBeg,
	const size_t i_blocks,
	const int i_partType,
	const int i_major,
	const int i_minor,
	const int i_perms,
	const char *i_name)
{
	struct specinfo *dev;

	dev = kmalloc(sizeof(*dev));
	if (dev == NULL) {
		return NULL;
	}
	memset(dev, 0, sizeof(*dev));

	// ƱǥХ̾ʤ
	if (searchDevByName(dev->si_name) != NULL) {
		kfree(dev);
		return NULL;
	}

	// ǥХꤹ
	dev->si_udev		= makeudev(i_major, i_minor);
	dev->si_devsw		= i_cdevsw;
	dev->beginSector	= i_blockBeg;
	dev->sectors		= i_blocks;
	dev->partType		= i_partType;
	memcpy(dev->si_name, i_name, strlen(i_name));
	if (i_cdevsw->d_flags & (D_TAPE | D_TTY | D_MEM)) {		// 饯ǥХ
		dev->si_flags = S_IFCHR | i_perms;
	}
	else if ((i_cdevsw->d_flags & D_DISK) == D_DISK) {		// ֥åǥХ
		dev->si_flags = S_IFBLK | i_perms;
	}
	else{
		// Ȥꤢ饯ǥХˤƤ
		dev->si_flags = S_IFCHR | i_perms;
	}

	// ꥹȤ³
	listConstructor(&dev->list, dev);
	enter_spinlock(&devSpecHead.lock);
	{
		devSpecHead.aggrMethod.insertHead(&devSpecHead.aggregate, &dev->list);
	}
	exit_spinlock(&devSpecHead.lock);

	return dev;
}

/*
 * ǥХϿ
 */
static void unregistDev(
	struct specinfo *m_dev)
{
	enter_spinlock(&devSpecHead.lock);
	{
		devSpecHead.aggrMethod.removeEntry(&devSpecHead.aggregate, &m_dev->list);
	}
	exit_spinlock(&devSpecHead.lock);

	kfree(m_dev);
}

/*
 * cdevsw¤ΤNULLؿѴ롣
 */
static void compile_devsw(
	struct cdevsw *devsw)
{
	if (devsw->d_open == NULL) {
		devsw->d_open = noopen;
	}
	if (devsw->d_close == NULL) {
		devsw->d_close = noclose;
	}
	if (devsw->d_read == NULL) {
		devsw->d_read = noread;
	}
	if (devsw->d_write == NULL) {
		devsw->d_write = nowrite;
	}
	if (devsw->d_ioctl == NULL) {
		devsw->d_ioctl = noioctl;
	}
	if (devsw->d_poll == NULL) {
		devsw->d_poll = nopoll;
	}
	if (devsw->d_mmap == NULL) {
		devsw->d_mmap = nommap;
	}
	if (devsw->d_strategy == NULL) {
		devsw->d_strategy = nostrategy;
	}
	if (devsw->d_dump == NULL) {
		devsw->d_dump = nodump;
	}
	if (devsw->d_psize == NULL) {
		devsw->d_psize = nopsize;
	}
	if (devsw->d_kqfilter == NULL) {
		devsw->d_kqfilter = nokqfilter;
	}
}

//----------------------------------------------------------
// ǥХ
//----------------------------------------------------------

static void (*intrHandler[IRQ_ENTRY])(void *);
static void *intrArg[IRQ_ENTRY];

static int deviceIntr(int irq)
{
	intrHandler[irq](intrArg[irq]);

	return 1;
}

//----------------------------------------------------------
// ǥХڥ졼
//----------------------------------------------------------

/*
 * ǥХڥ졼˥СȤ롣
 * return : error number
 */
static int convReadDev(
	struct specinfo *i_dev,
	void *buf,
	const size_t size,	// READ֥å
	const size_t begin,	// Ƭ֥å
	const int ioflag,
	size_t *o_size)		// READѥ֥å
{
	struct iovec *iovec;
	struct uio uio;
	int error;

	ASSERT(0 <= i_dev->sectSize);

	iovec = kmalloc(sizeof(*iovec));
	iovec->iov_base = buf;
	iovec->iov_len = size * i_dev->sectSize;

	setUio(
		1, 
		UIO_READ, 
		begin * i_dev->sectSize,
		iovec->iov_len,
		iovec,
		&uio);
	error = i_dev->si_devsw->d_read(i_dev, &uio, ioflag);
	*o_size = size - (uio.uio_resid / i_dev->sectSize);
	
	kfree(iovec);
	
	return error;
}

/*
 * ǥХڥ졼˥СȤ롣
 * return : error number
 */
static int convWriteDev(
	struct specinfo *dev,
	void *buf,
	const size_t size,	// WRITE֥å
	const size_t begin,	// Ƭ֥å
	const int ioflag,
	size_t *o_size)		// WRITEѥ֥å
{
	struct iovec *iovec;
	struct uio uio;
	int error;

	ASSERT(0 <= dev->sectSize);

	iovec = kmalloc(sizeof(*iovec));
	iovec->iov_base = buf;
	iovec->iov_len = size * dev->sectSize;

	setUio(
		1, 
		UIO_WRITE, 
		begin * dev->sectSize,
		iovec->iov_len,
		iovec,
		&uio);
	error = dev->si_devsw->d_write(dev, &uio, ioflag);
	*o_size = size - (uio.uio_resid / dev->sectSize);
	
	kfree(iovec);
	
	return error;
}

/*
 * ľܥ֥åǥХIO
 * return : error number
 */
static int ioDirect(
	void *i_dev,		// ǥХ
	void *buf,
	size_t i_blocks,
	size_t begin,
	int (*moduleIo)(void*, void*, const size_t, const off_t, void*),
	size_t *o_blocks)	// READѥ֥å
{
	size_t reblocks = i_blocks;

	while (0 < reblocks) {
		size_t blocks = (reblocks <= MAXPHYS / DEV_BSIZE) ? reblocks : MAXPHYS / DEV_BSIZE;
		int rest = moduleIo(
			i_dev,
			buf,
			blocks * DEV_BSIZE,
			begin,
			NULL);
		if (rest < 0) {
			return rest;
		}
		else if (0 < rest) {
			*o_blocks = reblocks - ROUNDUP_DIV(rest, DEV_BSIZE);
			return NOERR;
		}

		reblocks -= blocks;
	}
	
	*o_blocks = i_blocks;
	return NOERR;
}

/*
 * ˥᥸㡼ֹ롣
 * return : ᥸㡼ֹ
 */
static int getNewMajor()
{
	static int majorNum = 0;

	return majorNum++;
}

/*
 * ֥åǥХread
 * return : IO Хȿ or error number
 */
static int readBlk(
	struct specinfo *dev,
	void *m_buf,				// Хåե
	const uint i_bytes,			// Хȿ
	const uint i_offset)		// ϥХȥեå
{
	const uint blkSize = dev->sectSize;
	char *destBuf = m_buf;
	uint resid = i_bytes;
	uint offset = i_offset;
	char *blkBuf = NULL;
	uint copyBytes;
	struct iovec iovec;
	struct uio uio;
	int error;

	// ǽΥ֥å̤ɤ
	if (offset % blkSize) {
		uint exclBytes = offset % blkSize;

		blkBuf = kmalloc(blkSize);
		if (blkBuf == NULL) {
			return -ENOMEM;
		}
		iovec.iov_base = blkBuf;
		iovec.iov_len = blkSize;
		setUio(1, UIO_READ, ROUNDDOWN(offset, blkSize), blkSize, &iovec, &uio);
		error = dev->si_devsw->d_read(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}
		
		copyBytes = blkSize - exclBytes;
		if (resid < copyBytes) {
			copyBytes = resid;
		}
		memcpy(destBuf, blkBuf + exclBytes, copyBytes);
		destBuf += copyBytes;
		resid -= copyBytes;
		offset += copyBytes;
	}

	// ǸΥ֥åޤɤ
	while (blkSize <= resid) {
		copyBytes = (MAXPHYS <= resid) ? MAXPHYS : ROUNDDOWN(resid, blkSize);
		iovec.iov_base = destBuf;
		iovec.iov_len = copyBytes;
		setUio(1, UIO_READ, offset, copyBytes, &iovec, &uio);
		error = dev->si_devsw->d_read(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}
		resid -= copyBytes;
		if (0 < uio.uio_resid) {
			resid += uio.uio_resid;
			goto END;
		}
		destBuf += copyBytes;
		offset += copyBytes;
	}
		
	// ǸΥ֥å̤񤭹
	if (0 < resid) {
		if (blkBuf == NULL) {
			blkBuf = kmalloc(blkSize);
			if (blkBuf == NULL) {
				return -ENOMEM;
			}
		}
		iovec.iov_base = blkBuf;
		iovec.iov_len = blkSize;
		setUio(1, UIO_READ, offset, blkSize, &iovec, &uio);
		error = dev->si_devsw->d_read(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}
		memcpy(destBuf, blkBuf, resid);
		resid = 0;
	}
END:
	if (blkBuf != NULL) {
		kfree(blkBuf);
	}
	return i_bytes - resid;
ERR:
	if (blkBuf != NULL) {
		kfree(blkBuf);
	}
	return error;
}

/*
 * ֥åǥХwrite
 * return : IO Хȿ or error number
 */
static int writeBlk(
	struct specinfo *dev,
	void *m_buf,				// Хåե
	const uint i_bytes,			// Хȿ
	const uint i_offset)		// ϥХȥեå
{
	const uint blkSize = dev->sectSize;
	char *destBuf = m_buf;
	uint resid = i_bytes;
	uint offset = i_offset;
	char *blkBuf = NULL;
	uint copyBytes;
	struct iovec iovec;
	struct uio uio;
	int error;

	// ǽΥ֥å̤ɤ
	if (offset % blkSize) {
		uint exclBytes = offset % blkSize;

		blkBuf = kmalloc(blkSize);
		if (blkBuf == NULL) {
			return -ENOMEM;
		}
		iovec.iov_base = blkBuf;
		iovec.iov_len = blkSize;
		setUio(1, UIO_READ, ROUNDDOWN(offset, blkSize), blkSize, &iovec, &uio);
		error = dev->si_devsw->d_read(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}

		copyBytes = blkSize - exclBytes;
		if (resid < copyBytes) {
			copyBytes = resid;
		}
		memcpy(blkBuf + exclBytes, destBuf, copyBytes);
		iovec.iov_base = blkBuf;
		iovec.iov_len = blkSize;
		setUio(1, UIO_WRITE, ROUNDDOWN(offset, blkSize), blkSize, &iovec, &uio);
		error = dev->si_devsw->d_write(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}

		destBuf += copyBytes;
		resid -= copyBytes;
		offset += copyBytes;
	}

	// ǸΥ֥åޤɤ
	while (blkSize <= resid) {
		copyBytes = (MAXPHYS <= resid) ? MAXPHYS : ROUNDDOWN(resid, blkSize);
		iovec.iov_base = destBuf;
		iovec.iov_len = copyBytes;
		setUio(1, UIO_WRITE, offset, copyBytes, &iovec, &uio);
		error = dev->si_devsw->d_write(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}
		resid -= copyBytes;
		destBuf += copyBytes;
		offset += copyBytes;
	}
		
	// ǸΥ֥å̤񤭹
	if (0 < resid) {
		if (blkBuf == NULL) {
			blkBuf = kmalloc(blkSize);
			if (blkBuf == NULL) {
				return -ENOMEM;
			}
		}
		iovec.iov_base = blkBuf;
		iovec.iov_len = blkSize;
		setUio(1, UIO_READ, offset, blkSize, &iovec, &uio);
		error = dev->si_devsw->d_read(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}

		memcpy(blkBuf, destBuf, resid);
		iovec.iov_base = blkBuf;
		iovec.iov_len = blkSize;
		setUio(1, UIO_WRITE, offset, blkSize, &iovec, &uio);
		error = dev->si_devsw->d_write(dev, &uio, 0);
		if (error != NOERR) {
			goto ERR;
		}
	}

	if (blkBuf != NULL) {
		kfree(blkBuf);
	}
	return i_bytes;
ERR:
	if (blkBuf != NULL) {
		kfree(blkBuf);
	}
	return error;
}

//================================== PUBLIC =============================================

/*
 * ǥХƥν
 * return : error number
 */
int initDevice()
{
	AggregateListConstructor(&devSpecHead.aggregate, &devSpecHead.aggrMethod);

	return NOERR;
}

/*
 * ǥХϿ
 * return : struct specinfo or NODEV
 */
struct specinfo *make_dev(
	struct cdevsw *devsw, 
	int minor,				// ޥʡֹ
	uid_t uid,
	gid_t gid,
	int perms,				// ե⡼
	const char *fmt, ...)
{
	char name[MAX_DEVICE_NAME + 1];

	// cdevsw¤ΤNULLؿ
	compile_devsw(devsw);
	
	// ǥХ̾
	snprintf(name, MAX_DEVICE_NAME + 1, "%s%d", devsw->d_name, minor);
	name[MAX_DEVICE_NAME] = '\0';

	return registDev(
		devsw,
		0,
		0,
		0,
		getNewMajor(),
		minor,
		perms,
		name);
}

/*
 * ֥åǥХϿ
 * return : struct specinfo or NULL
 */
struct specinfo *makeBlkDev(
	struct cdevsw *m_cdevsw,
	const size_t i_blockBeg,
	const size_t i_blocks,
	const int i_perms)
{
	// cdevsw¤ΤNULLؿ
	compile_devsw(m_cdevsw);

	return registDev(
		m_cdevsw,
		i_blockBeg,
		i_blocks,
		0,
		getNewMajor(),
		0,
		i_perms,
		m_cdevsw->d_name);
}

/*
 * Delete from regist table
 * return : error number
 */
void destroy_dev(
	struct specinfo *dev)
{
	unregistDev(dev);
}

/*
 * ǥХֹõ
 * return : error number
 */
struct specinfo *getDevByDevnum(
	const int i_devnum)
{
	return searchDevByDevnum(i_devnum);
}

//----------------------------------------------------------
// ǥХڥ졼
//----------------------------------------------------------

/*
 * UIOꤹ
 * return : error number
 */
void setUio(
	const int iovecCnt,		// struct iovec
	enum uio_rw uioRw,		// UIO_READ or UIO_WRITE
	const off_t offset,		// Ƭեå
	const int resid,		// 
	struct iovec *iovec,
	struct uio *m_uio)		// ꤹstruct uio
{
	m_uio->uio_iov = iovec;
	m_uio->uio_iovcnt = iovecCnt;
	m_uio->uio_offset = offset;
	m_uio->uio_resid = resid;
//	m_uio->uio_segflg = UIO_SYSSPACE;
	m_uio->uio_segflg = UIO_USERSPACE;
	m_uio->uio_rw = uioRw;
//	m_uio->uio_procp = NULL;
}

int noopen(struct specinfo *dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noclose(struct specinfo *dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noread(struct specinfo *dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int nowrite(struct specinfo *dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int noioctl(struct specinfo *dev, u_long cmd, caddr_t data, int flags, struct proc *p)
{
	return (-ENODEV);
}

int nokqfilter(struct specinfo *dev, struct knote *kn)
{
	return (-ENODEV);
}

int nommap(struct specinfo *dev, vm_offset_t offset, int nprot)
{
	/* Don't return ENODEV.  That would allow mapping address ENODEV! */
	return (-1);
}

int nodump(struct specinfo *dev)
{
	return (-ENODEV);
}

/*
 * nopoll
 */
int seltrue(struct specinfo *dev, int events, struct proc *p)
{
	return (events & (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM));
}

/*
 *FreeBSD
 * return : cdevsw
 */
struct cdevsw *devsw(
	struct specinfo *dev)
{
	if (dev->si_devsw != NULL) {
		return dev->si_devsw;
	}
	else {
		ASSERT(0);
		return NULL;
	}
}

/*
 *FreeBSD
 * cdevsw¤ΤåϿ롣
 * return : error number
 */
int cdevsw_add(struct cdevsw *newentry)
{
	compile_devsw(newentry);
	if (newentry->d_maj < 0 || newentry->d_maj >= NUMCDEVSW) {
		printf("%s: ERROR: driver has bogus cdevsw->d_maj = %d\n", newentry->d_name, newentry->d_maj);
		return -EINVAL;
	}
	if (newentry->d_bmaj >= NUMCDEVSW) {
		printf("%s: ERROR: driver has bogus cdevsw->d_bmaj = %d\n",newentry->d_name, newentry->d_bmaj);
		return -EINVAL;
	}
	if (newentry->d_bmaj >= 0 && (newentry->d_flags & D_DISK) == 0) {
		printf("ERROR: \"%s\" bmaj but is not a disk\n", newentry->d_name);
		return -EINVAL;
	}
	if (newentry->d_bmaj < 0){
		return (0);
	}

	return (0);
}

//----------------------------------------------------------
// ǥХ
//----------------------------------------------------------

/*
 * ǥХߤ
 * return : error number
 */
void setupIntr(
	struct resource *irq,
	void (*handler)(void *),
	void *softc)
{
	irq_entry[irq->r_start] = deviceIntr;
	intrHandler[irq->r_start] = handler;
	intrArg[irq->r_start] = softc;
}

//----------------------------------------------------------
// ľžؿ
//----------------------------------------------------------

/*
 * return : error number
 */
int read_direct(
	void *i_dev,		// ǥХ
	void *buf,
	size_t i_blocks,
	size_t begin,
	size_t *o_blocks)	// readѥ֥å
{
	return ioDirect(i_dev, buf, i_blocks, begin, moduleReadDev, o_blocks);
}

/*
 * return : error number
 */
int write_direct(
	void *i_dev,		// ǥХ
	void *buf,
	size_t i_blocks,
	size_t begin,
	size_t *o_blocks)	// writeѥ֥å
{
	return ioDirect(i_dev, buf, i_blocks, begin, moduleWriteDev, o_blocks);
}

//--------------------------------------------------------------------------------------------------
// եե󥯥
//--------------------------------------------------------------------------------------------------

/*
 * ǥХ򥪡ץ󤹤
 * return : error number
 */
int devOpenDev(
	struct specinfo *m_dev,
	const int i_oflag,
	void **o_privateData)
{
	if ((m_dev->si_flags & S_IFMT) == S_IFCHR) {
		m_dev->sectSize = 1;
	}

	return m_dev->si_devsw->d_open(m_dev, i_oflag, 0, NULL);
}

/*
 * ǥХ򥯥
 * return : error number
 */
int devCloseDev(
	struct specinfo *m_dev,
	void *i_privateData)
{
	return m_dev->si_devsw->d_close(m_dev, 0, 0, NULL);
}

/*
 * ǥХɤ߹
 *ԳԲġ
 * return : read bytes or error number
 */
int devReadDev(
	struct specinfo *i_dev,
	void *m_buf,				// ߥХåե
	const size_t i_bytes,		// ɤ߹ߥХȿ
	const size_t i_offset,		// ɤ߹߳ϥХȥեå
	void *i_privateData)
{
	char *readBuf = m_buf;
	size_t ioSize;				// žѥ
	int error;

	switch (i_dev->si_flags & S_IFMT) {
	// 饯ǥХ
	case S_IFCHR:
		error = convReadDev(i_dev, readBuf, i_bytes, 0, 0, &ioSize);
		if (error != NOERR){
			return error;
		}
		else{
			return ioSize;
		}
	// ֥åǥХ
	case S_IFBLK:
/****************************************************************************************************************/
printk("readBlk m_buf=%x bytes=%d offset=%x PageDir=%x\n", 
	m_buf, i_bytes, i_offset, getPageDir(getVmTask(getCurrentTask())));
/****************************************************************************************************************/
		return readBlk(i_dev, m_buf, i_bytes, i_offset);
	// root directory
	case S_IFDIR:
		return 0;
	default:
		ASSERT(0);
		return -ENOENT;
	}
}

/*
 * ǥХ˽񤭹
 *ԳԲġ
 * return : write bytes or error number
 */
int devWriteDev(
	struct specinfo *i_dev,
	void *i_buf,				// ɤ߹ߥХåե
	const size_t i_bytes,		// ߥХȿ
	const size_t i_offset,		// ߳ϥХȥեå
	void *i_privateData)
{
	char *writeBuf = i_buf;
	size_t ioSize;				// žѥ
	int error;

	switch (i_dev->si_flags & S_IFMT){
	// 饯ǥХ
	case S_IFCHR:
		error = convWriteDev(i_dev, writeBuf, i_bytes, 0, 0, &ioSize);
		if (error != NOERR){
			return error;
		}
		else{
			return ioSize;
		}
	// ֥åǥХ
	case S_IFBLK:
		return writeBlk(i_dev, i_buf, i_bytes, i_offset);
	// root directory
	case S_IFDIR:
		return 0;
	default:
		ASSERT(0);
		return -ENOENT;
	}
}

/*
 * ǥХIOCTL
 * return : error number
 */
int devIoctlDev(
	struct specinfo *i_dev,
	const int cmd,
	caddr_t param,
	const int fflag,
	void *i_privateData)
{
	return i_dev->si_devsw->d_ioctl(i_dev, cmd, param, fflag, NULL);
}

/*
 * ǥХPOLL
 * return : error number
 */
int devPollDev(
	struct specinfo *i_dev,
	const int events,
	void *i_privateData)
{
	return i_dev->si_devsw->d_poll(i_dev, events, NULL);
}

/*
 * ǥХơȤμ
 * return : error number
 */
void devStatDev(
	struct specinfo *i_dev,
	DEV_STAT *m_devStat)
{
	ASSERT(i_dev != NULL);

	m_devStat->major		= major(i_dev);
	m_devStat->minor		= minor(i_dev);
	m_devStat->prt_type		= i_dev->partType;
	m_devStat->all_sect		= i_dev->sectors;
	m_devStat->sect_size	= i_dev->sectSize;
	m_devStat->flags		= i_dev->si_flags;
	m_devStat->name			= i_dev->si_name;
}

void statDev(
	void *i_dev,
	DEV_STAT *m_devStat)
{
	if (((struct specinfo*) i_dev)->moduleFlag == 0) {
		devStatDev(i_dev, m_devStat);
	}
	else {
		moduleStatDev(i_dev, m_devStat);
	}
}

int major(
	const dev_t x)
{
	if (x == NODEV) {
		return NOUDEV;
	}

	return getMajor(x->si_udev);
}

int minor(
	const dev_t x)
{
	if (x == NODEV) {
		return NOUDEV;
	}

	return getMinor(x->si_udev);
}

udev_t makeudev(
	int x, 
	int y)
{
	return ((x << 8) | y);
}

//*************************************************************************************************
void test_device()
{
}
