/*
 * time.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/limits.h>
#include <dev/8254.h>
#include <machine/syscall.h>
#include <machine/sp.h>
#include <machine/mp.h>
#include <machine/clock.h>
#include <machine/interrupt.h>
#include <lib/lib.h>
#include <sys/Thread.h>
#include <kern/vm.h>
#include <kern/ProcSignal.h>
#include <kern/time.h>

#include <kern/debug.h>


//#define DEBUG_TIME 1
#ifdef DEBUG_TIME
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


/***************************************************************************************************
 *
 * ִ
 *
 ***************************************************************************************************/

//================================== PRIVATE ============================================

enum {
	UPDATE_TIME = 500,		// ֳֹms
};

/*
 * ƥ
 */
static struct systime {
	uint64	miliSecond;		// Number of mili seconds since 00:00:00.1.1.1970.
	uint64	clock;			// CPUå
} systime;

static uint64 clock_1micro;		// cpu clock/s
static uint64 clock_1mili;		// cpu clock/ms

//FreeBSD
STATIC INLINE void timevalfix(struct timeval *t1)
{
	if (t1->tv_usec < 0) {
		t1->tv_sec--;
		t1->tv_usec += 1000000;
	}
	if (t1->tv_usec >= 1000000) {
		t1->tv_sec++;
		t1->tv_usec -= 1000000;
	}
}

/*
 * ߤΥߥ UNIX TIME 
 * return : ms
 */
STATIC uint64 getMilisecond()
{
	return systime.miliSecond;
}
//================================== PUBLIC =============================================

void initTime(void)
{
	uint clock = calcCpuClock();

	/*
	 * ޥäȣߥäCPUå򻻽Ф
	 */
	clock_1micro = clock / 1000000 + 1;		/* 1micros */
	clock_1mili = clock / 1000 + 1;			/* 1ms */
}

/*
 * ֤ν
 *յư˸ƤӽФ
 */
void setTime(
	uint seconds)
{
	systime.miliSecond = (uint64) seconds * 1000;
}

/*
 * ֤򹹿롣
 *Գ
 *ճԲĤǸƤӽФ롣
 *        0.5äȤ˸ƤӽФ롣
 */
void updateTime()
{
	uint64 currentClock = getCpuClock();
	uint64 diffClock = currentClock - systime.clock;
	
	systime.miliSecond += diffClock / clock_1mili;
	systime.clock = currentClock;
}

void gettimeofday(struct timeval *m_tp)
{
	uint64 sum;

	sum = systime.miliSecond * 1000 + (getCpuClock() - systime.clock) / clock_1micro;

	m_tp->tv_sec =  sum / 1000000;
	m_tp->tv_usec = sum % 1000000;
}

void getmicrouptime(struct timeval *m_tp)
{
	uint64 current = getCpuClock();

	m_tp->tv_sec = current / (clock_1micro * 1000000);
	m_tp->tv_usec = (current % (clock_1micro * 1000000)) / clock_1micro;
}

void microtime(struct timeval *tv)
{
	gettimeofday(tv);
}

/*
 *FreeBSD
 * Add and subtract routines for timevals.
 * N.B.: subtract routine doesn't deal with
 * results which are before the beginning,
 * it just gets very confused in this case.
 * Caveat emptor.
 */
void timevaladd(struct timeval *t1, struct timeval *t2)
{
	t1->tv_sec += t2->tv_sec;
	t1->tv_usec += t2->tv_usec;
	timevalfix(t1);
}

/*
 *FreeBSD
 */
void timevalsub(struct timeval *t1, struct timeval *t2)
{
	t1->tv_sec -= t2->tv_sec;
  	t1->tv_usec -= t2->tv_usec;
	timevalfix(t1);
}

/*
 * ưηвä
 */
time_t time_second()
{
	return getCpuClock() / clock_1mili / 1000;
}

/*
 * cpuå¬
 * ֤ : å / second
 */
uint calcCpuClock(void)
{
	uint clock;
	uint64 beg;

 	setStartClock();
 	beg = getCpuClock();
	timeIntervalTwentiethSecond();
	clock = (getCpuClock() - beg) * 20;

	return clock;
}

/*
 * Υåͤвᤷ֤
 * return : в(ms)
 */
uint getMiliSecondsFromClock(
	uint64 begin)		// clock
{
	return (getCpuClock() - begin) / clock_1mili;
}

/*
 * CPUåͤms֤׻
 * return : ms
 */
uint calcMiliSecondFromClock(
	uint64 clock)		// CPUå
{
	return clock / clock_1mili;
}

/*
 * CPUåͤs֤׻
 * return : s
 */
uint64 calcMicroSecondFromClock(
	uint64 clock)		// CPUå
{
	return clock / clock_1micro;
}

/*
 * micro secondsCPUåͤ򻻽Ф
 * return : CPUå
 */
uint64 calcClockFromMicroSecond(
	uint64 micro)		// s
{
	return micro * clock_1micro;
}

/*
 * mili secondsCPUåͤ򻻽Ф
 * return : CPUå
 */
uint64 calcClockFromMiliSecond(
	uint64 mili)		// s
{
	return mili * clock_1mili;
}

/***************************************************************************************************
 *
 * ƥॳ
 *
 ***************************************************************************************************/

int sys_time(int *tp)
{
	int currentSecond = (int) (getMilisecond() / 1000);

	if (tp != NULL) {
		if (vmIsReadArea(getVmProc(getCurrentProc()), tp, sizeof(int)) == -1) {
			return -EFAULT;
		}
		*tp = currentSecond;
	}

	return currentSecond;
}

int sys_settime(
	uint seconds)
{
	if (getUid(getCurrentProc()) != 0) {
		return -EPERM;
	}

	// ֤
	systime.miliSecond = seconds * 1000;

	return 0;
}

int sys_gettimeofday(struct timeval *tp, void *tzp)
{
	if (vmIsReadArea(getVmProc(getCurrentProc()), tp, sizeof(struct timeval)) == -1) {
		return 0;
	}
	gettimeofday(tp);

	return 0;	
}

/*
 * micro seconds¬
 */
int sysCalcMicroSeconds(
	int flag)		// 1:¬ 2:¬λ
{
	static uint64 begin;

	switch (flag) {
	case 1:
		begin = getCpuClock();
		break;
	case 2:
		printk("Past time = %u\n", calcMicroSecondFromClock(getCpuClock() - begin));
		break;
	default:
		break;
	}
	
	return NOERR;
}

/*************************************************************************************/
void test_time(int cnt)
{
	static int go = NO;
	
	if (go == YES){
		printk("test_time() %d\n", cnt);
	}
	if (cnt == 0){
		go = YES;
	}
}
