/*
 * timer.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/callout.h>
#include <sys/systm.h>
#include <sys/limits.h>
#include <lib/IteratorList.h>
#include <lib/AggregateList.h>
#include <machine/lib.h>
#include <machine/clock.h>
#include <machine/interrupt.h>
#include <machine/lock.h>
#include <dev/8254.h>
#include <dev/MC146818.h>
#include <kern/proc.h>
#include <kern/ProcSignal.h>
#include <kern/vm.h>
#include <kern/kmalloc.h>
#include <kern/time.h>
#include <kern/timer.h>

#include <kern/debug.h>


//#define DEBUG_TIMER 1
#ifdef DEBUG_TIMER
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif

//==================================  ===========================================

enum{
	TIMER_MAX = 10,			// Υޡͭ
	
	// ޡե饰
	FLG_NODELETE	= 1,		// ॢȸʤ
};

/*
 * ޡإå
 */
typedef struct {
	AggregateList		aggrList;					// ޡ󥯽
	AggregateListMethod	aggrMethod;					// 󥯥᥽å
	uint				(*getPastTime)();			// ޡߤޤǤθߤηвms
	void				(*setTimer)(const uint);	// ޡms
	int					lock;						// ԥå
} TIMER_HEAD;

/*
 * ޡ¤
 */
typedef struct {
	uint		init;				// ³ޡ
	void		(*handle)(void*);	// ϥɥ
	void		*arg;				// ϥɥ
	List		ownedList;			// ͭ󥯥ꥹ
	int			flag;				// ե饰

	// ޡ
	uint		count;				// Ĥ
	List		timerList;			// ޡꥹ
	TIMER_HEAD	*timerHead;			// ³Ƥ륿ޡإå³ƤʤNULL
} Timer;

//================================== PRIVATE =============================================

/*
 * 顼ϥɥ
 *Գ
 */
STATIC void handleAlarm(
	void *proc)
{
	procSendSignal(proc, SIGALRM);
}

/*
 * ޡ󥯤饿ޡ֤ꤹ
 */
STATIC void setTimeTimerLink(
	TIMER_HEAD *i_timerHead)	// ³西ޡإå
{
	AggregateList *aggrList = &i_timerHead->aggrList;
	AggregateListMethod *aggrMethod = &i_timerHead->aggrMethod;
	Timer *timer = aggrMethod->refHead(aggrList);
	
	if (timer == NULL) {
		return;
	}

	// ƬΥޡ֤ꤹ
	if (timer->count == 0) {
		i_timerHead->setTimer(1);
	}
	else {
		i_timerHead->setTimer(timer->count);
	}
}

//--------------------------------------------------------------------------------------------------
// ޡ
//--------------------------------------------------------------------------------------------------

/*
 * 󥿡Х륿ޡإå
 */
static TIMER_HEAD intervalTimer;

/*
 * ꥢ륿ޡإå
 */
static TIMER_HEAD realTimer;

/*
 * ޡإåν
 */
STATIC void initTimerHead()
{
	// 󥿡Х륿ޡإåν
	AggregateListConstructor(&intervalTimer.aggrList, &intervalTimer.aggrMethod);
	intervalTimer.getPastTime = getIntervalTimerPastTime;
	intervalTimer.setTimer = setIntervalTimer;
	intervalTimer.lock = 0;
	
	// ꥢ륿ޡإåν
	AggregateListConstructor(&realTimer.aggrList, &realTimer.aggrMethod);
	realTimer.getPastTime = getRealTimerPastTime;
	realTimer.setTimer = setRealTimer;
	realTimer.lock = 0;
}

/*
 * ޡ¤Τ򥿥ޡإå˻ֽ³롣
 *ճ߶ػߡޡإåȤ˥å򤫤ƸƤӽФ뤳
 */
STATIC void linkTimer(
	Timer *this,				// ³륿ޡ¤
	TIMER_HEAD *m_timerHead,	// ³西ޡإå
	const int i_time)			// ޡms
{
	uint time = i_time;
	AggregateList *aggrList = &m_timerHead->aggrList;
	AggregateListMethod *aggrMethod = &m_timerHead->aggrMethod;
	IteratorList iterator;

	IteratorListConstruct(&iterator, aggrList, aggrMethod);
	for (;;) {
		Timer *timer;
		
		if (iterator.hasNext(&iterator) == BOOL_FALSE) {
			aggrMethod->insertEnd(aggrList, &this->timerList);
			break;
		}

		timer = iterator.next(&iterator);
		if (time <= timer->count) {
			timer->count -= time;
			aggrMethod->insertPrev(aggrList, &timer->timerList, &this->timerList);
			break;
		}
		else {
			time -= timer->count;
		}
	}
	this->count = time;
}

/*
 * ޡ¤Τ򥿥ޡإåڤΥ
 *ճ߶ػߡޡإåȤ˥å򤫤ƸƤӽФ뤳
 */
STATIC void unlinkTimer(
	Timer *this,				// ڤΥޡ¤
	TIMER_HEAD *i_timerHead)	// ³西ޡإå
{
	AggregateList *aggrList = &i_timerHead->aggrList;
	AggregateListMethod *aggrMethod = &i_timerHead->aggrMethod;

	if (i_timerHead->aggrMethod.isLink(&i_timerHead->aggrList, &this->timerList) == YES) {
		Timer *next = aggrMethod->refNext(aggrList, &this->timerList);
		if (next != NULL) {
			next->count += this->count;
		}
		aggrMethod->removeEntry(aggrList, &this->timerList);
	}
}

/*
 * ֤вᤷޡ򥿥ޡإåڤΥƼ롣
 *ճ߶ػߡޡإåȤ˥å򤫤ƸƤӽФ뤳
 */
STATIC void getTimeoutTimer(
	const uint i_time,						// вms
	TIMER_HEAD *m_timerHead,				// ³西ޡإå
	AggregateList *aggrUnlinkList,			// ޡ¤Τ³ñꥹȥإå
	AggregateListMethod *aggrUnlinkMehotd)	// ޡ¤Τ³ñꥹȥ᥽å
{
	uint time = i_time;
	AggregateList *aggrList = &m_timerHead->aggrList;
	AggregateListMethod *aggrMethod = &m_timerHead->aggrMethod;
	Timer *timer;

	for (timer = aggrMethod->refHead(aggrList); timer != NULL;) {
		Timer *next = aggrMethod->refNext(aggrList, &timer->timerList);

		if (timer->count <= time) {
			aggrMethod->removeEntry(aggrList, &timer->timerList);
			time -= timer->count;
			timer->count = 0;
			aggrUnlinkMehotd->insertHead(aggrUnlinkList, &timer->timerList);
		}
		else {
			timer->count -= time;
			break;
		}
		
		timer = next;
	}
}

/*
 * ޡ󥯤֤򸺤餹
 *ճ߶ػߡޡإåȤ˥å򤫤ƸƤӽФ뤳
 */
STATIC void subtractTimeTimerLink(
	TIMER_HEAD *i_timerHead,	// ³西ޡإå
	const uint time)			//  ms
{
	AggregateList *aggrList = &i_timerHead->aggrList;
	AggregateListMethod *aggrMethod = &i_timerHead->aggrMethod;
	Timer *timer = aggrMethod->refHead(aggrList);
	
	if (timer == NULL) {
		return;
	}

	ASSERT(time <= timer->count);
	
	timer->count -= time;
}

//-------------------------------------- Ͽ ----------------------------------------------

/*
 * ޡ󥯤³Ƥ뤫
 *ճ߶ػߤǸƤӽФ뤳
 * return : YES or NO
 */
STATIC int isRegistTimer(
	const Timer *this)
{
	TIMER_HEAD *timerHead = this->timerHead;

	ASSERT(this->timerHead != NULL);
	
	if (timerHead->aggrMethod.isLink(&timerHead->aggrList, &this->timerList) == YES) {
		return YES;
	}
	return (this->flag != 0) ? YES : NO;
}

/*
 * ޡ¤Τ򥿥ޡإåϿ
 *ճ߶ػߤǸƤӽФ뤳
 * return : error number
 */
STATIC void registTimer(
	Timer  *this,
	const uint i_time,	// ޡms
	const int flag)
{
	TIMER_HEAD *timerHead;
	int eflags;

	// ޡإå
	if (i_time < REAL_TIMER_RATE) {
		timerHead = &intervalTimer;
	}
	else {
		timerHead = &realTimer;
	}

	eflags = enterCli();
	enter_spinlock(&timerHead->lock);
	{
		uint pastTime;

		// ޡηв֤
		pastTime = timerHead->getPastTime();

		// ޡ󥯤³
		subtractTimeTimerLink(timerHead, pastTime);
		linkTimer(this, timerHead, i_time);
		this->timerHead = timerHead;

		// ǥХޡ֤
		setTimeTimerLink(timerHead);
	}
	exit_spinlock(&timerHead->lock);
	exitCli(eflags);
	
	this->flag = flag;
}

/*
 * ޡ¤Τ򥿥ޡإå
 */
STATIC void unregistTimer(
	Timer  *this)
{
	TIMER_HEAD *timerHead = this->timerHead;
	int eflags;

	eflags = enterCli();
	enter_spinlock(&timerHead->lock);
	{
		unlinkTimer(this, timerHead);
		this->flag = 0;
	}
	exit_spinlock(&timerHead->lock);
	exitCli(eflags);
}

//-------------------------------------- ޡ ----------------------------------------------

/*
 * ֤вᤷޡϥɥ¹Ԥ롣
 *ԳաԹ
 */
STATIC void doTimer(
	const uint i_time,			// в
	TIMER_HEAD *m_timerHead)	// ³إå
{
	AggregateList aggrListUnlilnkTimer;				// ñꥹ
	AggregateListMethod aggrMethodUnlilnkTimer;
	IteratorList iterator;
	int eflags;

	AggregateListConstructor(&aggrListUnlilnkTimer, &aggrMethodUnlilnkTimer);

	eflags = enterCli();
	enter_spinlock(&m_timerHead->lock);
	{
		// ॢȥޡμ
		getTimeoutTimer(i_time, m_timerHead, &aggrListUnlilnkTimer, &aggrMethodUnlilnkTimer);
	}
	exit_spinlock(&m_timerHead->lock);
	exitCli(eflags);

	// ޡϥɥμ¹
	IteratorListConstruct(&iterator, &aggrListUnlilnkTimer, &aggrMethodUnlilnkTimer);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		Timer *timer = iterator.next(&iterator);

		timer->handle(timer->arg);
	}

	eflags = enterCli();
	enter_spinlock(&m_timerHead->lock);
	{
		Timer *timer;

		while ((timer = aggrMethodUnlilnkTimer.getHead(&aggrListUnlilnkTimer)) != NULL) {
			// ³ޡͤк³
			if (0 < timer->init) {
				linkTimer(timer, m_timerHead, timer->init);
				timer->timerHead = m_timerHead;
			}
		}

		// ǥХޡ֤
		setTimeTimerLink(m_timerHead);
	}
	exit_spinlock(&m_timerHead->lock);
	exitCli(eflags);
}

//--------------------------------------------------------------------------------------------------
// ޡͭ
//--------------------------------------------------------------------------------------------------

/*
 * ƥॿޡͭإå
 */
static TIMER_OWNER systemTimer;

/*
 * ޡͭإåν
 */
STATIC INLINE void initOwnerHead(
	TIMER_OWNER *m_owner)
{
	AggregateListConstructor(&m_owner->aggrList, &m_owner->aggrMethod);
	m_owner->lock = 0;
}

/*
 * ͭإåѥޡõ
 *ճ߶ػߤǸƤФ뤳
 * return : NOERR or -ENOENT
 */
STATIC int findDelTimer(
	TIMER_OWNER *i_owner,		// ͭإå
	Timer **o_timer)
{
	IteratorList iterator;

	IteratorListConstruct(&iterator, &i_owner->aggrList, &i_owner->aggrMethod);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		Timer *timer = iterator.next(&iterator);
		
		if ((isRegistTimer(timer) == NO) && ((timer->flag & FLG_NODELETE) == 0)) {
			*o_timer = timer;
			return NOERR;
		}
	}

	return -ENOENT;
}

/*
 * ͭ󥯤˥ޡƤƤʤп˥ޡơޡ롣
 *ճ߶ػߤǸƤФ뤳
 * return : error number
 */
STATIC int allocTimer(
	const uint initTime,	// ַв˺Ƥꤹ륿ޡ
	void (*handle)(void*),	// ַв˼¹Ԥ
	void *arg,				// ַв˼¹Ԥΰ
	TIMER_OWNER *m_owner,	// ޡͭ
	Timer **o_timer)
{
	Timer *timer;

	// ޡõƤʤгƤ
	if (findDelTimer(m_owner, &timer) == -ENOENT) {
		AggregateList *aggrList = &m_owner->aggrList;
		AggregateListMethod *aggrMethod = &m_owner->aggrMethod;

		if (TIMER_MAX <= aggrMethod->getCount(aggrList)) {
			WARNING(0);
			return -ENOMEM;
		}

		timer = kmalloc(sizeof(*timer));
		if (timer == NULL){
			return -ENOMEM;
		}
		listConstructor(&timer->ownedList, timer);
		aggrMethod->insertHead(aggrList, &timer->ownedList);
	}

	listConstructor(&timer->timerList, timer);
	timer->init = initTime;
	timer->handle = handle;
	timer->arg = arg;

	*o_timer = timer;

	return NOERR;
}

/*
 * ͭ󥯤
 *ճ߶ػߤǸƤФ뤳
 *unregistTimer()ϿƤ뤳
 * return : error number
 */
/*
STATIC void freeTimer(
	Timer *this,
	TIMER_OWNER *i_owner)
{
	AggregateList *aggrList = &i_owner->aggrList;
	AggregateListMethod *aggrMethod = &i_owner->aggrMethod;

	enter_spinlock(&systemTimer.lock);
	{
		aggrMethod->removeEntry(aggrList, &this->ownedList);
	}
	exit_spinlock(&systemTimer.lock);
	kfree(this);
}
*/

/*
 * ͭإåϥɥõ
 *ճ߶ػߤǸƤФ뤳
 * return : Timer or NULL
 */
STATIC Timer *findTimerByHandle(
	TIMER_OWNER *i_owner,		// ͭإå
	void (*handle)(void*))
{
	IteratorList iterator;

	IteratorListConstruct(&iterator, &i_owner->aggrList, &i_owner->aggrMethod);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		Timer *timer = iterator.next(&iterator);
		
		if (timer->handle == handle) {
			return timer;
		}
	}

	return NULL;
}

//================================== PUBLIC ==============================================

/*
 * ޡν
 *ե֡Ȼ˸ƤӽФ
 */
void initTimer(void)
{
#ifdef DEBUG
	if (sizeof(Timer) != sizeof(TimerObj)) {
		printk("initTimer() object size error! Timer=%d TimerObj=%d\n", sizeof(Timer), sizeof(TimerObj));
		idle();
	}
#endif
	// ƥॿޡإåν
	initTimerHead();

	// ƥॿޡͭإåν
	initOwnerHead(&systemTimer);
}

/*
 * ץޡν
 *եץ˸ƤӽФ
 */
void initProcessTimer(
	void *m_timer)
{
	initOwnerHead(m_timer);
}

//--------------------------------------------------------------------------------------------------
// 
//--------------------------------------------------------------------------------------------------

/*
 *Գ
 * 󥿡Х륿ޡμ¹
 */
void doIntervalTimer(
	const uint time)		// вᤷ(ms)
{
	doTimer(time, &intervalTimer);
}

/*
 *Գ
 * ꥢ륯åޡμ¹
 */
void doRealTimer(
	const uint time)		// вᤷ(ms)
{
	doTimer(time, &realTimer);
}

//--------------------------------------------------------------------------------------------------
// ޡϿ
//--------------------------------------------------------------------------------------------------

/*
 * ƥॿޡƤ
 * return : error number
 */
int setSystemTimer(
	const uint i_time,		// ޡ
	const uint initTime,	// ַв˺ꤹ륿ޡ
	void (*handle)(void*),	// ַв˼¹Ԥ
	void *arg,				// ַв˼¹Ԥΰ
	void **o_timer)
{
	int error;

	enter_spinlock(&systemTimer.lock);
	{
		error = allocTimer(initTime, handle, arg, &systemTimer, (Timer**) o_timer);
	}
	exit_spinlock(&systemTimer.lock);
	
	if (error == NOERR) {
		registTimer(*o_timer, i_time, 0);
		return NOERR;
	}
	else {
		return error;
	}
}

/*
 * ץޡƤ
 * return : error number
 */
int setTaskTimer(
	const uint i_time,		// ޡ
	const uint initTime,	// ַв˺Ƥꤹ륿ޡ
	void (*handle)(void*),	// ַв˼¹Ԥ
	void *arg,				// ַв˼¹Ԥΰ
	void **o_timer)
{
	int eflags;
	int error = NOERR;

	eflags = enterCli();
	{
		error = allocTimer(initTime, handle, arg, getTaskTimer(getCurrentTask()), (Timer**)o_timer);
		if (error == NOERR) {
			registTimer(*o_timer, i_time, FLG_NODELETE);
		}
	}
	exitCli(eflags);

	return error;
}

/*
 * 顼ॿޡƤ
 * return : error number
 */
int setAlarmTimer(
	const uint i_time,		// ޡms
	const uint initTime,	// ַв˺Ƥꤹ륿ޡms
	TimerObj *m_Timer)		// Υ顼ॿޡ
{
	int error;
	int eflags;

	eflags = enterCli();
	{
		Timer *timer;

		// 顼ॿޡˤв
		timer = findTimerByHandle(getTaskTimer(getCurrentTask()), handleAlarm);
		if (timer != NULL) {
			*m_Timer = *(TimerObj*) timer;
			unregistTimer(timer);
		}
		else {
			Timer *nullTimer = (Timer*) m_Timer;
			nullTimer->init = 0;
			nullTimer->timerHead = NULL;
		}

		// ޡ
		if (0 < i_time){
			error = allocTimer(initTime, handleAlarm, getCurrentProc(), getTaskTimer(getCurrentTask()), (Timer**) &timer);
			if (error == NOERR) {
				registTimer(timer, i_time, 0);
			}
		}
		else {
			error = NOERR;
		}
	}
	exitCli(eflags);

	return error;
}

/*
 * ޡ
 * return : error number
 */
void resetTimer(
	void *m_timer)
{
	Timer *timer = (Timer*) m_timer;

	// Ͽ
	unregistTimer(timer);
}

/*
 *Թ
 * ץͭޡ
 */
void releaseProcessTimer(
	TIMER_OWNER *m_owner)
{
	Timer *timer;

	while ((timer = m_owner->aggrMethod.getHead(&m_owner->aggrList)) != NULL) {
		// Ͽ
		unregistTimer(timer);
		kfree(timer);
	}
}

//--------------------------------------------------------------------------------------------------
// GETTER
//--------------------------------------------------------------------------------------------------

/*
 * ֤
 * return : ms
 */
uint getInitTime(
	TimerObj *i_timer)
{
	Timer *timer = (Timer*) i_timer;
	return timer->init;
}

/*
 * Ĥ֤
 *Ի
 * return : Ĥms
 */
uint getRemainTime(
	TimerObj *i_timer)
{
	Timer *this = (Timer*) i_timer;
	TIMER_HEAD *timerHead = this->timerHead;
	uint time;
	int eflags;

	if (timerHead == NULL) {
		return 0;
	}

	time = 0;
	eflags = enterCli();
	enter_spinlock(&timerHead->lock);
	{
		if (timerHead->aggrMethod.isLink(&timerHead->aggrList, &this->timerList) == YES) {
			IteratorList iterator;

			IteratorListConstruct(&iterator, &timerHead->aggrList, &timerHead->aggrMethod);
			while (iterator.hasNext(&iterator) == BOOL_TRUE) {
				Timer *next = iterator.next(&iterator);

				// Ĥ֤û
				time += next->count;
				if (next == this) {
					break;
				}
			}
		}
	}
	exit_spinlock(&timerHead->lock);
	exitCli(eflags);

	return time;
}

/*
 * 顼ॿޡ
 */
void getAlarmTimer(
	TimerObj *m_timer)
{
	Timer *timer;
	int eflags;

	eflags = enterCli();
	{
		timer = findTimerByHandle(getTaskTimer(getCurrentTask()), handleAlarm);
	}
	exitCli(eflags);
	if (timer != NULL) {
		*m_timer = *(TimerObj*) timer;
	}
	else {
		timer = (Timer*) m_timer;
		timer->init = 0;
		timer->timerHead = NULL;
	}
}

//--------------------------------------------------------------------------------------------------
// ¾ޡ
//--------------------------------------------------------------------------------------------------

/*
 * timeout --
 *	Execute a function after a specified length of time.
 *
 * untimeout --
 *	Cancel previous timeout function call.
 *
 * callout_handle_init --
 *	Initialize a handle so that using it with untimeout is benign.
 *
 *	See AT&T BCI Driver Reference Manual for specification.  This
 *	implementation differs from that one in that although an 
 *	identification value is returned from timeout, the original
 *	arguments to timeout as well as the identifier are used to
 *	identify entries for untimeout.
 *
 *Գ
 *ա¾ץƤФ礢
 */
void callout_handle_init(struct callout_handle *callout)
{
	callout->callout = NULL;
}

struct callout_handle timeout(
	timeout_t *ftn,
	void *arg, 
	int to_ticks)		// ms
{
	void *timer;
	struct callout_handle callout;

	if (to_ticks == 0) {
		callout.callout = NULL;
		return callout;
	}

	if (setSystemTimer(to_ticks, 0, ftn, arg, &timer) == NOERR) {
		callout.callout = timer;
	}
	else {
		printk("Failed timeout()!\n");
		callout.callout = NULL;
	}

	return callout;
}

/*
 *Գ
 */
void untimeout(
	timeout_t *ftn, 
	void *arg, 
	struct callout_handle callout)
{
	Timer *this = callout.callout;

	if (this == NULL) {
		return;
	}
	resetTimer(callout.callout);
}

void callout_init(void *c)
{
	// Ȥꤢ⤷ʤ
}

void callout_reset(void *c, int to_ticks, void (*ftn)(void *), void *arg)
{
	// Ȥꤢ⤷ʤ
}

/*
 * return : 0 : ȥåcallout̵ or 1 : ȥåcalloutͭ
 */
int callout_stop(void *c)
{
	// Ȥꤢ⤷ʤ
	return 0;
}

/*
 * micro seccond ޡ
 */
void micro_timer(
	uint count)		// ¬(s)
{
	uint64 end;

	end = getCpuClock();
	end += calcClockFromMicroSecond(count);
	while (getCpuClock() < end);
}

/*
 * mili seccond timer
 */
void mili_timer(
	uint count)		// ¬(ms)
{
	uint64 end;

	end = getCpuClock();
	end += calcClockFromMiliSecond(count);
	while (getCpuClock() < end);
}

//--------------------------------------------------------------------------------------------------
// ƥॳ
//--------------------------------------------------------------------------------------------------

int sys_alarm(
	uint seconds)
{
	TimerObj dummy;
	int error = setAlarmTimer(seconds * 1000, 0, &dummy);

	return error;
}

void sysMicroDelay(
	const uint micro)
{
	micro_timer(micro);
}

//*************************************************************************************************

#ifdef DEBUG
void test_timer(
	int num)
{
	printDebug(80 * 10 + 70, "test-%d  ", num);
}
#endif
