/*
 * vfs.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * To do
 *	opendir()λͤlookup()бˤ롣
 */

/******************************************************************************************************
 *
 * ۥե륷ƥ
 *
 *  ޥȻȥǥ쥯ȥѹե륪ץե¹Ի˥ȥ꡼ɲäinodeObjǤ롣
 * 
 * ¥եθ           ¥ե
 * lookup()                           deleteObj()
 * creat()
 *                                   
 * ¥ե򥪡ץ             ¥ե򥯥
 * open()                             close()
 *                                   
 * ۥȥ꡼κ             ۥȥ꡼
 * VentConstructor()                  VentDestructor()
 *                                   
 * ۥե륷ƥ³       ۥե륷ƥफ
 * addVent()
 *                                   
 *                            
 * linkVent()                         unlinkVent()
 *******************************************************************************************************/

#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <sys/conf.h>
#include <lib/lib_path.h>
#include <lib/Hierarchy.h>
#include <kern/kmalloc.h>
#include <kern/devfs.h>
#include <kern/fs.h>
#include <sys/Thread.h>
#include <kern/time.h>
#include <kern/device.h>
#include <kern/TaskWait.h>
#include <kern/vfs.h>

#include <kern/debug.h>

//#define TEST_VFS

#define DEBUG_VFS 1
#ifdef DEBUG_VFS
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif

//=====================================  ===================================================

enum {
	SYMLINK_LOOP_MAX = 10,		// ܥå󥯺ݻ
	EXIST_VENT = 1,				// ۥȥ꡼
	
	FUNC_NORMAL = 0,
	FUNC_END,
};

/*
 * ۥȥ꡼
 * Note!
 *  fsinodeΰ֤ϥåȹ¤Τȥ󥯤ƤΤա
 */
typedef struct VENTRY{
	FS				*fs;		// ե륷ƥ
	void			*inodeObj;	// inode object ݥ
	Hierarchy		hierarchy;
	struct VENTRY	*mount;		// ޥȥȥ꡼
	struct VENTRY	*mounted;	// ޥȸȥ꡼
	void			*din;		// ǥХǥץ
	int				type;		// ե륿
	ushort			linkCount;	// ץ󡢥ޥȡȥǥ쥯ȥȥå
	uchar			del;		// ׵᥵YES or NO
	uchar			len;		// Name size.
	char			name[0];	// ѥ͡
} VENTRY;

/*
 * ܥå󥯥롼ץåѥơ֥
 */
typedef struct {
	int count;							// ܥå󥯥
	void *din[SYMLINK_LOOP_MAX];		// ǥХinode
	uint inodeNum[SYMLINK_LOOP_MAX];	// indoeֹ
} SYMLINK_LOOP;

/*
 * lookupPathѽϥѥ᡼
 */
typedef struct {
	VENTRY *vent;				// ۥȥ꡼
	void *din;					// ǥХǥץ
	FS *fs;						// ե륷ƥ๽¤
	uint inodeNum;				// inodeֹ
	const char *parentPath;		// ƥǥ쥯ȥѥ
	VENTRY *parentVent;			// ƥǥ쥯ȥinode֥
	char path[PATH_MAX + 1];	// ܥå󥯥ѥХåե
} LOOKUP_OUT;

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static char rootEntArea[sizeof(VENTRY) + 1];
static VENTRY *rootEnt = (VENTRY*)&rootEntArea;	// 롼ȥȥ꡼

//--------------------------------------------------------------------------------------------------
// ȥ꡼
//--------------------------------------------------------------------------------------------------

/*
 * ޥȥȥ꡼ޥȤƤʤä鼫ȥ꡼֤
 *Ի
 * return : VENTRY
 */
STATIC VENTRY *getMountUp(
	VENTRY *i_vent)
{
	VENTRY *vent;
	if (i_vent == NULL) {
		return NULL;
	}
	for (vent = i_vent; vent->mounted != NULL; vent = vent->mounted);
	return vent;
}

/*
 * ޥȥȥ꡼ޥȤƤʤä鼫ȥ꡼֤
 *Ի
 * return : VENTRY
 */
STATIC VENTRY *getMountDown(
	VENTRY *i_vent)
{
	VENTRY *vent;
	if (i_vent == NULL) {
		return NULL;
	}
	for (vent = i_vent; vent->mount != NULL; vent = vent->mount);
	return vent;
}

/*
 * ƥȥ꡼롢ޥȤƤޥοƥȥ꡼
 *Ի
 * return : VENTRY
 */
STATIC VENTRY *getParent(
	VENTRY *i_vent)
{
	VENTRY *vent = getMountUp(i_vent);

	ASSERT(i_vent != NULL);

	if (vent == rootEnt) {
		return rootEnt;
	}
	else {
		return HierarchyGetParent(&vent->hierarchy);
	}
}

/*
 * 	ҥȥ꡼롢ޥȤƤޥȥȥ꡼λҥȥ꡼
 *Ի
 * return : VENTRY
 */
STATIC VENTRY *getChild(
	VENTRY *i_vent)
{
	VENTRY *vent = getMountDown(i_vent);
	
	ASSERT(i_vent != NULL);
	
	return HierarchyGetChild(&vent->hierarchy);
}

/*
 * pathȥåץǥ쥯ȥ롣
 *Ի
 * return : VENTRY
 */
STATIC VENTRY *getTopVdir(
	const char *path, 
	const char **o_path)
{
	VENTRY *vent;
	char *p = (char*)path;
	if (*p == '/') {
		vent = rootEnt;
		while(*++p == '/');
		*o_path = p;
	}
	else{
		vent = getCurrentDir();
		*o_path = p;
	}

	return getMountDown(vent);
}

/*
 * ȥ꡼ؤ˸Ƥ
 *ԺƵաԻ
 */
STATIC void searchUp(
	VENTRY *vent,
	int (*func)(VENTRY*, void*),		// return : FUNC_NORMAL or FUNC_END  λ
	void *arg,
	void (*funcAfter)(VENTRY*, void*),
	void *argAfter)
{
	VENTRY *parent;

	if (vent == NULL) {
		return;
	}

	parent = getParent(vent);
	if (func(vent, arg) == FUNC_END) {
		return;
	}
	searchUp(parent, func, arg, funcAfter, argAfter);
	funcAfter(vent, argAfter);
}

/*
 * ȥ꡼ؤ򲼤˸Ƥ
 *ԺƵաԻ
 */
STATIC void searchDown(
	VENTRY *vent,
	int (*func)(VENTRY*, void*),		// return : FUNC_NORMAL or FUNC_END  λ
	void *arg,
	void (*funcAfter)(VENTRY*, void*),
	void *argAfter)
{
	VENTRY *next;

	for (next = getMountDown(vent); next != NULL; next = getMountDown(HierarchyGetNext(&next->hierarchy))) {
		if (func(next, arg) == FUNC_END) {
			break;
		}
		searchDown(getChild(next), func, arg, funcAfter, argAfter);
		funcAfter(next, argAfter);
	}
}

/*
 * ȥ꡼ѥǸ
 *Ի
 * return : VENTRY or NULL
 */
STATIC VENTRY *searchPath(
	VENTRY *vent,
	const char *i_path,
	void (*func)(VENTRY*, const char*, void*),
	void *arg,
	void (*funcAfter)(VENTRY*, const char*, void*),
	void *argAfter)
{
	VENTRY *next = getMountDown(vent);
	const char *path = i_path;

	while (*path != '\0') {
		VENTRY *before = next;
		const char *beforePath = path; 

		func(next, path, arg);

		if (pathCmpName(".", path, 1) == 0){
			// ⤷ʤ
		}
		else if (pathCmpName("..", path, 2) == 0){
			next = getParent(next);
		}
		else {
			next = getChild(next);
			while (next != NULL) {
				if (pathCmpName(next->name, path, next->len) == 0) {
					break;
				}
				next = HierarchyGetNext(&next->hierarchy);
			}
			if (next == NULL) {
				break;
			}
		}

		funcAfter(before, beforePath, argAfter);

		path = pathGetNext(path);
	}
	
	return next;
}

/*
 * Ѵؿ
 * ⤷ʤ
 */
STATIC int funcIntNull()
{
	// ⤷ʤ
	return FUNC_NORMAL;
}

STATIC void funcVoidNull()
{
	// ⤷ʤ
}

//--------------------------------------------------------------------------------------------------
// ѥ̤äƥԡ
//--------------------------------------------------------------------------------------------------

/*
 * copyPathBack()Ѵؿ¤
 */
typedef struct {
	char *path;			// ѥХåե
	size_t size;		// ѥХåե
	size_t len;			// ԡѥ
	int (*condition)(VENTRY *);
} COPY_PATH;

/*
 * Ѵؿѥԡ
 * return : FUNC_NORMAL or FUNC_END
 */
STATIC int funcCopyPath(
	VENTRY *i_vent,
	void *arg)
{
	COPY_PATH *copyPath = arg;

	if (copyPath->condition(i_vent) == YES) {
		size_t len;
		VENTRY *vent = getMountUp(i_vent);
		len = (copyPath->size < vent->len) ? copyPath->size : vent->len;
		memcpy(copyPath->path, vent->name, len);
		copyPath->len = len;

		return FUNC_END;
	}
	else {
		return FUNC_NORMAL;
	}
}

/*
 * Ѵؿѥԡ
 */
STATIC void funcAfterCopyPath(
	VENTRY *vent,
	void *arg)
{
	COPY_PATH *copyPath = arg;
	size_t len;

	if (copyPath->size <= copyPath->len) {
		return;
	}
	
	copyPath->path[copyPath->len++] = '/';
	len = ((copyPath->size - copyPath->len) < vent->len) ? (copyPath->size - copyPath->len) : vent->len;
	memcpy(copyPath->path + copyPath->len, vent->name, len);
	copyPath->len += len;
}

/*
 * copyPathBack()Ϥؿ
 */
STATIC int isRoot(
	VENTRY *vent)
{
	return (vent == rootEnt) ? YES : NO;
}

STATIC int isMounted(
	VENTRY *vent)
{
	return (vent->mounted != NULL) ? YES : NO;
}

STATIC int isParent(
	VENTRY *vent)
{
	return (vent->inodeObj != NULL) ? YES : NO;
}

/*
 * ѥäƥХåե˥ԡ롣
 *Ի
 * return : ԡ
 */
STATIC int copyPathBack(
	VENTRY *vent,
	char *pathBuf,					// ԡХåե
	const size_t size,				// ԡХåե
	int (*condition)(VENTRY *))		// ̤¾
{
	COPY_PATH copyPath;
	
	copyPath.path = pathBuf;
	copyPath.size = size;
	copyPath.len = 0;
	copyPath.condition = condition;
	searchUp(vent, funcCopyPath, &copyPath, funcAfterCopyPath, &copyPath);
	return copyPath.len;
}

//--------------------------------------------------------------------------------------------------
// ѥθ
//--------------------------------------------------------------------------------------------------

/*
 * searchPath()Ѵؿ¤
 */
typedef struct {
	VENTRY *vent;
	const char *path;			// ѥ
} SEARCH_PATH;

/*
 * Ѵؿ
 * ̥ȥ꡼
 *Ի
 */
STATIC void funcSearchPath(
	VENTRY *i_vent,
	const char *path,
	void *arg)
{
	VENTRY *vent = getMountDown(i_vent);
	SEARCH_PATH *searchPath = arg;
	if ((vent->inodeObj != NULL) || 
		(vent->mounted != NULL ) || 
		(vent == rootEnt)) {
		searchPath->vent = vent;
		searchPath->path = path;
	}
}

/*
 * ѥ饨ȥ꡼򸡺롣
 *Ի
 * return : error number
 */
STATIC int searchVent(
	VENTRY *i_vent,				// inode֥
	const char *i_path,			// ѥ
	const char **o_parentPath,	// inode֥Ȥƥǥ쥯ȥѥ
	VENTRY **o_parentVent,		// inode֥Ȥƥǥ쥯ȥ겾ۥȥ꡼
	VENTRY **o_vent)			// Ǹβۥȥ꡼
{
	VENTRY *vent;
	SEARCH_PATH searchPathArg;
	
	vent = searchPath(i_vent, i_path, funcSearchPath, &searchPathArg, funcVoidNull, NULL);
	if (vent != NULL) {
		*o_vent = vent;
		return NOERR;
	}
	else {
		*o_parentVent = searchPathArg.vent;
		*o_parentPath = searchPathArg.path;
		return -ENOENT;
	}
}

//--------------------------------------------------------------------------------------------------
// ¥եκ
//--------------------------------------------------------------------------------------------------

/*
 * deleteObjFromVent()Ѵؿ¤
 */
typedef struct {
	VENTRY *vent;
} DELETE_OBJ;

/*
 * Ѵؿ
 * ̥ȥ꡼
 * return : FUNC_NORMAL or FUNC_END
 */
STATIC int funcHighVent(
	VENTRY *vent,
	void *arg)
{
	DELETE_OBJ *deleteObj = arg;
	if (vent->inodeObj != NULL) {
		deleteObj->vent = vent;
	}
	return FUNC_NORMAL;
}

/*
 * ¥ե뤿ᡢ̳ؤinodeObjΤ륨ȥ꡼򸡺Υѥ롣
 *ԹաԳԲġ
 * return : error number
 */
STATIC int deleteObjFromVent(
	VENTRY *vent,		// ȥ꡼
	const int type)		// ȥ꡼
{
	char path[PATH_MAX + 1];
	DELETE_OBJ deleteObj;
	int len;

	searchUp(vent, funcHighVent, &deleteObj, funcVoidNull, NULL);
	len = copyPathBack(vent, path, PATH_MAX + 1, isParent);
	if (PATH_MAX < len) {
		return -ENAMETOOLONG;
	}
	path[len] = '\0';

	return deleteObj.vent->fs->delete(path, deleteObj.vent->din, deleteObj.vent->inodeObj, type);
}

//--------------------------------------------------------------------------------------------------
// Constructor Destructor
//--------------------------------------------------------------------------------------------------

/*
 * ȥ꡼Υ
 * return : VENTRY or NULL
 */
STATIC VENTRY *allocVent(
	const int nameSize)
{
	return kmalloc(sizeof(VENTRY) + nameSize);
}

/*
 * ȥ꡼γ
 */
STATIC void freeVent(
	VENTRY *vent)
{
	kfree(vent);
}

/*
 *Թ
 */
STATIC void VentConstructor(
	VENTRY *vent,
	const char *path,
	VENTRY *i_parent)
{
	memset(vent, 0, sizeof(*vent));
	HierarchyConstructor(&vent->hierarchy, vent);

	// 󥯤ɲä
	if (i_parent != NULL) {
		VENTRY *parent = getMountDown(i_parent);
		HierarchyAdd(&vent->hierarchy, &parent->hierarchy);
	}

	vent->len = pathLen(path);
	memcpy(vent->name, path, vent->len);
}

/*
 *Թ
 * return : error number
 */
STATIC int VentDestructor(
	VENTRY *vent)
{
	HierarchyRemove(&vent->hierarchy);
	freeVent(vent);

	return NOERR;
}

/*
 * ȥ꡼˼¥ե򥻥åȤ
 *Ի
 */
STATIC void setVent(
	VENTRY *vent,
	void *devDsc,
	FS *fs,
	void *inodeObj,
	const int type)
{
	vent->din = devDsc;
	vent->fs = fs;
	vent->inodeObj = inodeObj;
	vent->type = type;
	vent->del = NO;
}

/*
 * ȥ꡼
 *ԹաԳԲġ
 * return : error number
 */
STATIC int deleteVent(
	VENTRY *vent)
{
	int error;

	if (0 < vent->linkCount) {
		return -EBUSY;
	}
	if (0 < HierarchyGetChildCount(&vent->hierarchy)) {
		return -EBUSY;
	}
	if (vent->mount != NULL) {
		return -EBUSY;
	}
	if (vent->mounted != NULL) {
		VENTRY *mounted = vent->mounted;
		vent->fs->umount(vent->din, vent->inodeObj);
		mounted->mount = NULL;
	}

	//ե򥯥
	if (vent->inodeObj != NULL) {
		error = vent->fs->close(vent->din, vent->inodeObj);
		if (error != NOERR) {
			return error;
		}
		vent->inodeObj = NULL;
	}
	
	//ե
	if (vent->del == YES) {
		error = deleteObjFromVent(vent, vent->type);
		if (error != NOERR) {
			return error;
		}
	}

	VentDestructor(vent);

	return NOERR;
}

//--------------------------------------------------------------------------------------------------
// ѥǲۥȥ꡼
//--------------------------------------------------------------------------------------------------

/*
 * Ѵؿ
 * ̥ȥ꡼
 */
STATIC void funcAddVent(
	VENTRY *vent,
	const char *path,
	void *arg)
{
	SEARCH_PATH *searchPath = arg;
	searchPath->vent = vent;
	searchPath->path = path;
}

/*
 * Ѵؿ
 * ̥ȥ꡼
 */
STATIC void funcAfterAddVent(
	VENTRY *vent,
	const char *path,
	void *arg)
{
	if (pathCmpName("..", path, 2) == 0) {
		if ((vent->inodeObj == NULL) && (HierarchyGetChildCount(&vent->hierarchy) == 0)) {
			deleteVent(vent);
		}
	}
}

/*
 * ȥ꡼ѥɲä롣
 *հοƥȥ꡼Ʊե륷ƥǤʤФʤʤ
 *Թ
 * return : error number
 */
STATIC int addVent(
	VENTRY *parentVent,		//ƥȥ꡼
	const char *i_path,		//ƤΥѥ
	VENTRY **o_vent)
{
	SEARCH_PATH searchPathArg;
	VENTRY *new;

	searchPathArg.path = i_path;
	searchPathArg.vent = parentVent;
	while ((new = searchPath(searchPathArg.vent, searchPathArg.path, funcAddVent, &searchPathArg, funcAfterAddVent, NULL)) == NULL) {
		VENTRY *vent = allocVent(pathLen(searchPathArg.path));
		if (vent == NULL) {
			return -ENOMEM;
		}
		VentConstructor(vent, searchPathArg.path, searchPathArg.vent);
		setVent(vent, searchPathArg.vent->din, searchPathArg.vent->fs, NULL, S_IFDIR);
	}

	*o_vent = new;

	return NOERR;
}

//--------------------------------------------------------------------------------------------------
// ѥ̤äƲۥȥ꡼
//--------------------------------------------------------------------------------------------------

/*
 * Ѵؿ
 * ѥԡ
 * return : FUNC_NORMAL or FUNC_END
 */
STATIC int funcUnlinkVent(
	VENTRY *vent,
	void *arg)
{
	if (deleteVent(vent) == NOERR) {
		return FUNC_NORMAL;
	}
	else {
		return FUNC_END;
	}
}

/*
 * ۥȥ꡼򥯥
 *դθƤӽФɬΤߤˤʤ뤳ȡƱ֤äƸƤӽФʤȡ
 *ԹաԳԲġ
 */
STATIC void unlinkVent(
	VENTRY *vent)
{
	ASSERT(0 < vent->linkCount);

	--vent->linkCount;

	// ȥ꡼̤äƺ
	searchUp(vent, funcUnlinkVent, NULL, funcVoidNull, NULL);
}

//--------------------------------------------------------------------------------------------------
// ե륷ƥΥޥ
//--------------------------------------------------------------------------------------------------

/*
 * Ѵؿ
 * ޥ
 */
STATIC void funcUmountFs(
	VENTRY *i_vent,
	void *arg)
{
	VENTRY *vent;

	for (vent = i_vent; vent != NULL; vent = vent->mounted) {
		if (vent->mounted != NULL) {
			vent->fs->umount(vent->din, vent->inodeObj);
		}
	};
}

/*
 *ԳԲġ
 */
STATIC void umountFs(
	VENTRY *vent)
{
	searchDown(vent, funcIntNull, NULL, funcUmountFs, NULL);
}

//--------------------------------------------------------------------------------------------------
// ¥եθ
//--------------------------------------------------------------------------------------------------

/*
 * ܥå󥯥롼ץơ֥ν
 */
STATIC void initSymlinkLoop(SYMLINK_LOOP *m_symlinkLoop)
{
	m_symlinkLoop->count = 0;
}

/*
 * ܥå󥯥롼ץå
 * return : error number
 */
STATIC int symlinkLoop(
	void *devDsc,
	const uint inodeNum,
	SYMLINK_LOOP *m_symlinkLoop)
{
	int i;

	ASSERT(m_symlinkLoop->count <= SYMLINK_LOOP_MAX);
	
	if (m_symlinkLoop->count == SYMLINK_LOOP_MAX) {
		return -ELOOP;
	}
	for (i = 0; i < m_symlinkLoop->count; ++i) {
		if ((m_symlinkLoop->din[i] == devDsc) && (m_symlinkLoop->inodeNum[i] == inodeNum)) {
			return -ELOOP;
		}
	}
	
	m_symlinkLoop->din[m_symlinkLoop->count] = devDsc;
	m_symlinkLoop->inodeNum[m_symlinkLoop->count] = inodeNum;
	++m_symlinkLoop->count;
	
	return NOERR;
}

/*
 * ܥå󥯤Υѥ򥻥åȤ롣
 *ԻաԳԲġ
 * return : error number
 */
STATIC int setSymlinkPath(
	void *devDsc,
	const FS *fs,
	const uint inodeNum,
	const char *path,
	const char *symlinkPath,
	char *m_path,					// ѥĹ PATH_MAX + 1 Ǥ뤳
	SYMLINK_LOOP *m_symlinkLoop)
{
	char *pathBuf;
	const char *nextPath;
	void *inodeObj;
	int lenPrev;
	int len;
	int dummy;
	int error;

	error = symlinkLoop(devDsc, inodeNum, m_symlinkLoop);
	if (error != NOERR) {
		return error;
	}

	pathBuf = m_path;

	// ܥå󥯥ѥ
	error = fs->open(devDsc, inodeNum, O_RDONLY, &inodeObj, &dummy);
	if (error != NOERR) {
		return error;
	}
	len = fs->read(devDsc, inodeObj, pathBuf, PATH_MAX, 0);
	if (fs->close(devDsc, inodeObj) != NOERR) {
		ASSERT(0);
	}
	if (len < 0) {
		return len;
	}
	
	if (*pathBuf == '/') {
		// ХѥʤΤ˥ѥղäʤ
		lenPrev = 0;
	}
	else {
		// ܥåޤǤΥѥղ
		lenPrev = symlinkPath - path;
		if (PATH_MAX < lenPrev + len) {
			return -ENAMETOOLONG;
		}
		memmove(pathBuf + lenPrev, pathBuf, len);
		memcpy(pathBuf, path, lenPrev);
	}
	pathBuf += lenPrev + len;

	// ܥå󥯸Υѥ
	nextPath = pathGetNext(symlinkPath);
	if (*nextPath != '\0') {
		int lenNext;

		nextPath -= 1;			// '/'ޤ᤹
		lenNext = strnlen(nextPath, PATH_MAX + 1);
		if (PATH_MAX < lenPrev + len + lenNext) {
			return -ENAMETOOLONG;
		}
		memcpy(pathBuf, nextPath, lenNext);
		pathBuf += lenNext;
	}
	*pathBuf = '\0';

	return NOERR;
}

/*
 * ¥ե򸡺
 *ԻաԳԲġ
 * return : error number or SYMLINK
 */
STATIC int lookupObj(
	const char *i_path, 	// ѥ
	void *devDsc,
	const FS *fs,
	void *i_inodeObj,
	char *m_path,			// ܥåȯˤĤѥѥХåե
	SYMLINK_LOOP *m_symlinkLoop,
	uint *o_inodeNum)		// inodeֹ
{
	uint inodeNum;
	const char *symlinkPath;
	int error;

	if(*i_path=='\0'){
		return -ENOENT;
	}

	error = fs->lookup(i_path, devDsc, i_inodeObj, &inodeNum, &symlinkPath);
	switch (error) {
	case NOERR:
		break;
	case SYMLINK:		// ܥå
		// ѥ
		error = setSymlinkPath(devDsc, fs, inodeNum, i_path, symlinkPath, m_path, m_symlinkLoop);
		if (error != NOERR) {
			return error;
		}
		return SYMLINK;
	default:
		return error;
	}

	*o_inodeNum = inodeNum;

	return error;
}

/*
 * ¥ե򸡺
 * ܥå󥯤⸡
 *ԻաԳԲġ
 * return : error number or SYMLINK
 */
STATIC int lookupObjSymlink(
	const char *i_path, 	// ѥ
	void *devDsc,
	const FS *fs,
	void *i_inodeObj,
	char *m_path,			// ܥåȯˤĤѥѥХåե
	SYMLINK_LOOP *m_symlinkLoop,
	uint *o_inodeNum)		// inodeֹ
{
	uint inodeNum;
	const char *symlinkPath;
	int error;

	if(*i_path=='\0'){
		return -ENOENT;
	}

	error = fs->lookup(i_path, devDsc, i_inodeObj, &inodeNum, &symlinkPath);
	switch (error) {
	case NOERR:
		break;
	case SYMLINK:		// ܥå
		// ѥκǸΥȥ꡼ʤｪλ
		if (*pathGetNext(symlinkPath) == '\0') {
			error = NOERR;
		}
		else {
			error = setSymlinkPath(devDsc, fs, inodeNum, i_path, symlinkPath, m_path, m_symlinkLoop);
			if (error != NOERR) {
				return error;
			}
			return SYMLINK;
		}
		break;
	default:
		return error;
	}

	*o_inodeNum = inodeNum;

	return error;
}

/*
 * ۥȥ꡼ȼ¥եõܥåб
 * Ǥ˲ۥȥ꡼Хȥѥ᡼˲ۥȥ꡼򥻥åȤEXIST_VENT֤
 *ԻաԳԲġ
 */
STATIC int lookupPath(
	const char *i_path,
	int lookupFunc(const char *, void*, const FS *, void *, char *, SYMLINK_LOOP *, uint *),
	LOOKUP_OUT *lookupOut)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	SYMLINK_LOOP symlinkLoop;
	uint inodeNum;
	int error;

	initSymlinkLoop(&symlinkLoop);

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	for (;;) {
		error = searchVent(vent, path, &parentPath, &parentVent, &vent);
		if (error == NOERR) {				// ۥեˤ
			lookupOut->vent = vent;
			return EXIST_VENT;
		}
		else if (error == -ENOENT) {		// ۥե̵
			error = lookupFunc(parentPath, parentVent->din, parentVent->fs, parentVent->inodeObj, lookupOut->path, &symlinkLoop, &inodeNum);
			if (error == SYMLINK) {	// ܥå
				path = lookupOut->path;
				if (*path == '/') {
					// Хѥξ롼Ȳۥȥ꡼
					vent = getTopVdir(path, &path);
				}
			}
			else {
				break;
			}
		}
		else {
			return error;
		}
	}

	lookupOut->din = parentVent->din;
	lookupOut->fs = parentVent->fs;
	lookupOut->inodeNum = inodeNum;
	lookupOut->parentPath = parentPath;
	lookupOut->parentVent = parentVent;

	return error;
}

//--------------------------------------------------------------------------------------------------
// 
//--------------------------------------------------------------------------------------------------

/*
 * struct statfs˳Ǽ롣
 *Ի
 * return : error number
 */
STATIC int getFsstat(
	VENTRY *i_vent, 
	struct statfs *statfs)
{
	VENTRY *vent = getMountDown(i_vent);
	int error;
	DEV_STAT devstat;

	if ((error = vent->fs->statfs(vent->din, statfs)) < 0){
		return error;
	}

	statDev(getDevInfo(i_vent->din), &devstat);
	if (devstat.minor > 0){
		sprintk(statfs->f_mntfromname, "/dev/%s%d", devstat.name, devstat.minor);
	}
	else{
		sprintk(statfs->f_mntfromname, "/dev/%s", devstat.name);
	}

	copyPathBack(vent, statfs->f_mntonname, MNAMELEN, &isMounted);

	return NOERR;
}

/*
 * return : error number
 */
STATIC int chmod(
	const VENTRY *vent,
	const mode_t mode)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	// øΥå
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}
	if ((getUid(proc) != 0) && (getUid(proc) != stat.st_uid)) {
		return -EPERM;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, mode, stat.st_uid, stat.st_gid, stat.st_atime, stat.st_mtime);
}

/*
 * return : error number
 */
STATIC int chuid(
	const VENTRY *vent,
	const uid_t owner)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	ASSERT(owner != -1);

	// øΥå
	if (getUid(proc) != 0) {
		return -EPERM;
	}

	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, stat.st_mode, owner, stat.st_gid, stat.st_atime, stat.st_mtime);
}

/*
 * return : error number
 */
STATIC int chgrp(
	const VENTRY *vent,
	const gid_t group)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	ASSERT(group != -1);

	// øΥå
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}
	if (((getUid(proc) != 0) && (getUid(proc) != stat.st_uid)) || (group != getGid(proc))) {
		return -EPERM;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, stat.st_mode, stat.st_uid, group, stat.st_atime, stat.st_mtime);
}

/*
 * return : error number
 */
STATIC inline int chutime(
	const VENTRY *vent,
	const uint atime,
	const uint mtime)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	// øΥå
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}
	if ((getUid(proc) != 0) && (getUid(proc) != stat.st_uid)) {
		return -EPERM;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, stat.st_mode, stat.st_uid, stat.st_gid, atime, mtime);
}

/*
 * ۥեˤвۥȥ꡼̵м¥ե򥪡ץ󤷤Ʋۥȥ꡼ʲۥեɲäʤˤ롣
 *ԹաԳԲġ
 * return : error number
 */
STATIC int openVent(
	const char *i_path,
	const int flag,			// openե饰
	int lookupFunc(const char *, void*, const FS *, void *, char *, SYMLINK_LOOP *, uint *),
	VENTRY **o_vent)		// inode
{
	LOOKUP_OUT lookupOut;
	int error;

	error = lookupPath(i_path, lookupFunc, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		*o_vent = lookupOut.vent;
		break;
	case NOERR: {		// ۥե̵
		void *inodeObj;
		int type;
		VENTRY *vent;

		error = lookupOut.fs->open(lookupOut.din, lookupOut.inodeNum, flag, &inodeObj ,&type);
		if (error != NOERR){
			return error;
		}
		vent = allocVent(0);
		if (vent == NULL) {
			lookupOut.fs->close(lookupOut.din, inodeObj);
			return -ENOENT;
		}
		memset(vent, 0, sizeof(*vent));
		setVent(vent, lookupOut.din, lookupOut.fs, inodeObj, type);
		*o_vent = vent;
		break;
	}
	default:
		return error;
	}

	linkVent(*o_vent);

	return NOERR;
}

/*
 *ԹաԳԲġ
 * openVent()Ǽۥȥ꡼롣
 */
STATIC void closeVent(
	void *i_vent)
{
	VENTRY *vent = i_vent;

	ASSERT(0 < vent->linkCount);

	--vent->linkCount;
	if (vent->linkCount == 0) {
		vent->fs->close(vent->din, vent->inodeObj);
		freeVent(vent);
	}
}

//--------------------------------------------------------------------------------------------------
// ȥå
//--------------------------------------------------------------------------------------------------

static WaitAggrObj waitObj;		// ۥե륷ƥࡦե륷ƥॳå

/*
 * ۥե륷ƥλȳ
 */
STATIC INLINE void refVfs()
{
	WaitReference(&waitObj);
}

/*
 * ۥե륷ƥλȽλ
 */
STATIC INLINE void refEndVfs()
{
	WaitEndReference(&waitObj);
}

/*
 * ۥե륷ƥι
 */
STATIC INLINE void modifyVfs()
{
	WaitGetSemaphore(&waitObj);
}

/*
 * ۥե륷ƥιλ
 */
STATIC INLINE void modifyEndVfs()
{
	WaitReleaseSemaphore(&waitObj);
}

#ifdef TEST_VFS
static int dummyClose()
{
	return NOERR;
}

enum {
	FIND_OK = 0,
	FIND_NO,
};

typedef struct {
	VENTRY *vent;
	int state;
} FindVent;

static int findVend(
	VENTRY *vent,
	void *arg)
{
	FindVent *find = arg;
	if (vent == find->vent) {
		find->state = FIND_OK;
		return FUNC_END;
	}
	else {
		find->state = FIND_NO;
		return FUNC_NORMAL;
	}
}

static void testVfs()
{
	VENTRY *vent1;
	const char *path1 = "vent1";
	VENTRY *vent1_1;
	const char *path1_1 = "vent1/dir/vent1_1";
	VENTRY *vent1_2;
	const char *path1_2 = "dir/vent1_2";
	VENTRY *vent1_3;
	const char *path1_3 = "vent1/dir/vent1_2/dir/dir/../../../../../vent1/dir/vent1_3";
	int error;
	FS fs = {
		.umount = (void*) dummyClose,
		.close = (void*) dummyClose,
	};

	error = mountToVfs(&fs, (void*) 1, (void*) 0, rootEnt);
	if (error != NOERR) {
		panic("mountToVfs() error. %d %s line=%d\n", error, __FILE__, __LINE__);
	}

	error = addVent(rootEnt, path1, &vent1);
	if (error != NOERR) {
		panic("addVent() error. %d %s line=%d\n", error, __FILE__, __LINE__);
	}
	vent1->inodeObj = (void*) 1;

	error = addVent(rootEnt, path1_1, &vent1_1);
	if (error != NOERR) {
		panic("addVent() error. error. %d %s line=%d\n", error, __FILE__, __LINE__);
	}
	vent1_1->inodeObj = (void*) 11;

	error = addVent(vent1, path1_2, &vent1_2);
	if (error != NOERR) {
		panic("addVent() error. error. %d %s line=%d\n", error, __FILE__, __LINE__);
	}
	vent1_2->inodeObj = (void*) 12;

	error = addVent(rootEnt, path1_3, &vent1_3);
	if (error != NOERR) {
		panic("addVent() error. error. %d %s line=%d\n", error, __FILE__, __LINE__);
	}
	vent1_3->inodeObj = (void*) 13;

	{
		VENTRY *vent;
		const char *parentPath;
		VENTRY *parent;
		int error = searchVent(rootEnt, "vent1/dir/vent1_2/../../dir/vent1_3", &parentPath, &parent, &vent);
		switch (error) {
		case NOERR:
			if (vent != vent1_3) {
				panic("searchVent error. %s line=%d\n", __FILE__, __LINE__);
			}
			break;
		case -ENOENT:
			panic("searchVent error. %d %s line=%d\n", error, __FILE__, __LINE__);
			break;
		default:
			panic("searchVent error. %d %s line=%d\n", error, __FILE__, __LINE__);
		}

		error = searchVent(vent1, "dir/vent1_3", &parentPath, &parent, &vent);
		switch (error) {
		case NOERR:
			if (vent != vent1_3) {
				panic("searchVent error. %s line=%d\n", __FILE__, __LINE__);
			}
			break;
		case -ENOENT:
			panic("searchVent error.  %d %s line=%d\n", error, __FILE__, __LINE__);
			break;
		default:
			panic("searchVent error. %d %s line=%d\n", error, __FILE__, __LINE__);
		}

		error = searchVent(rootEnt, "vent1/dir/vent1_2/dir/dir/../../../vent1_4", &parentPath, &parent, &vent);
		switch (error) {
		case NOERR:
			if (vent == vent1_3) {
				panic("searchVent error. %s line=%d\n", __FILE__, __LINE__);
			}
			break;
		case -ENOENT:
			if ((strcmp(parentPath, "dir/dir/../../../vent1_4") != 0) ||
				(parent != vent1_2)) {
				panic("searchVent error.  %d %s line=%d\n", error, __FILE__, __LINE__);
			}
			break;
		default:
			panic("searchVent error.  %d %s line=%d\n", error, __FILE__, __LINE__);
		}
	}

	{
		FindVent find = {
			.vent = vent1_3,
		};
		searchDown(rootEnt, findVend, &find, funcVoidNull, NULL);
		if (find.state != FIND_OK) {
			panic("searchDown error.  %d %s line=%d\n", error, __FILE__, __LINE__);
		}
	}
	vent1->linkCount = 1;
	vent1_1->linkCount = 1;
	vent1_2->linkCount = 1;
	vent1_3->linkCount = 1;
	unlinkVent(vent1_3);
	unlinkVent(vent1_2);
	unlinkVent(vent1_1);
	unlinkVent(vent1);
	error = umountPath("/");
	if (error != NOERR) {
		panic("umountPath error.  %d %s line=%d\n", error, __FILE__, __LINE__);
	}
	
	printk("Test OK! ");
}
#endif

//===================================== PUBLIC =====================================================

void VfsInit()
{
	/* rootȥ꡼ν */
	VentConstructor(rootEnt, "/", NULL);
	setVent(rootEnt, 0, NULL, NULL, S_IFDIR);
	rootEnt->linkCount	= 2;		// ǥեȥȥǥ쥯ȥʤΤǣꤹ롣

	// ȥåν
	WaitAggrConstructor(&waitObj);

	printk("Init vfs ");
#ifdef TEST_VFS
	testVfs();
#endif
	printk("\n");
}

/*
 *դǤ˥󥯤Ƥ뤳
 *Թ
 */
void linkVent(
	void *i_vent)
{
	VENTRY *vent = i_vent;

	++vent->linkCount;
}

//--------------------------------------------------------------------------------------------------
// ۡ¥ե
//--------------------------------------------------------------------------------------------------

/*
 * Υե륷ƥΥ롼inodeޥȤ롣
 * return : error number
 */
int mountToVfs(
	FS *fs,
	void *devDsc,
	void *rootInodeObj,
	void *m_dstVent)
{
	VENTRY *dstVent = m_dstVent;
	VENTRY *mountVent;
	char name[PATH_MAX + 1];
	int error;

	modifyVfs();

	// Ǥ˥ޥȤƤ뤫
	if (dstVent->mount != NULL){
		error = -EBUSY;
		goto ERR;
	}

	// ޥȥȥ꡼κ
	mountVent = allocVent(dstVent->len);
	if (mountVent == NULL){
		error = -ENOMEM;
		goto ERR;
	}

	// ޥȥȥ꡼ν
	memcpy(name,  dstVent->name, dstVent->len);
	name[dstVent->len] = '\0';
	VentConstructor(mountVent, name, NULL);
	mountVent->mounted	= dstVent;
	setVent(mountVent, devDsc, fs, rootInodeObj, S_IFDIR);
	linkVent(mountVent);

	// ޥȤ
	dstVent->mount = mountVent;

	error = NOERR;
ERR:
	modifyEndVfs();
	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int umountPath(
	const char *i_path)
{
	LOOKUP_OUT lookupOut;
	int error;

	modifyVfs();

	error = lookupPath(i_path, lookupObj, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		if (lookupOut.vent->type == S_IFDIR){
			unlinkVent(lookupOut.vent);
			error = NOERR;
		}
		else {
			error = -ENOENT;
		}
		break;
	case NOERR: {		// ۥե̵
		error = -ENOENT;
		break;
	}
	default:
		break;
	}

	modifyEndVfs();

	return error;
}

/*
 * ۥե륷ƥ򸡺ƥޥȤƤե륷ƥ򥢥ޥȤ롣
 *ԳԲġ
 */
void umountAllFs()
{
	umountFs(rootEnt);
}

/*
 * ¥եκץ
 * return : error number
 */
int creatObj(
	const char *i_path,
	const mode_t mode)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	uint inodeNum;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	error = searchVent(vent, path, &parentPath, &parentVent, &vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		break;
	case -ENOENT:	// ۥե̵
		error = parentVent->fs->creat(parentPath, parentVent->din, parentVent->inodeObj, mode, &inodeNum);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return NOERR;
}

/*
 *ԳԲġ
 * return : error number
 */
int mkdirPath(
	const char *i_path,
	const mode_t mode)
{
	LOOKUP_OUT lookupOut;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = lookupPath(i_path, lookupObj, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		//
	case NOERR:			// ¥եˤ
		error = -EEXIST;
		break;
	case -ENOENT:			// ¥ե̵
		error = lookupOut.fs->mkdir(lookupOut.parentPath, lookupOut.din, lookupOut.parentVent->inodeObj, mode);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ܥå󥯤κ
 * return : error number
 */
int creatSymlink(
	const char *i_path,		// ܥå󥯥ѥ
	const mode_t mode,		// ܥå󥯥ե⡼
	char *linkPath)			// 󥯥ѥ
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	uint inodeNum;
	void *inodeObj;
	int len;
	VENTRY *dummy1;
	int dummy2;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	error = searchVent(vent, path, &parentPath, &parentVent, &dummy1);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		goto ERR;
	case -ENOENT:	// ۥե̵
		break;
	default:
		goto ERR;
	}

	error = parentVent->fs->creat(parentPath, parentVent->din, parentVent->inodeObj, mode, &inodeNum);
	if (error != NOERR) {
		goto ERR;
	}
	error = parentVent->fs->open(parentVent->din, inodeNum, O_WRONLY, &inodeObj ,&dummy2);
	if (error != NOERR){
		goto ERR;
	}
	len = strnlen(linkPath, PATH_MAX + 1);
	error = parentVent->fs->write(parentVent->din, inodeObj, linkPath, len, 0);
	if (0 < error) {
		error = NOERR;
	}
	parentVent->fs->close(parentVent->din, inodeObj);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ۥȥ꡼򥪡ץ󤹤
 * ۥȥ꡼ˤʤкƥץ󤹤롣
 *ԳԲġ
 * return : error number
 */
int openPath(
	const char *i_path,
	const int flag,
	void **o_vent)
{
	LOOKUP_OUT lookupOut;
	VENTRY *vent;
	int error;

	modifyVfs();

	error = lookupPath(i_path, lookupObj, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		vent = lookupOut.vent;
		error = NOERR;
		break;
	case NOERR: {		// ۥե̵
		void *inodeObj;
		int type;

		error = lookupOut.fs->open(lookupOut.din, lookupOut.inodeNum, flag, &inodeObj ,&type);
		if (error != NOERR){
			goto ERR;
		}

		// ۥǥ쥯ȥκ
		error = addVent(lookupOut.parentVent, lookupOut.parentPath, &vent);
		if (error != NOERR){
			lookupOut.fs->close(lookupOut.din, inodeObj);
			goto ERR;
		}

		setVent(vent, lookupOut.din, lookupOut.fs, inodeObj, type);
		break;
	}
	default:
		goto ERR;
	}

	linkVent(vent);
	*o_vent = vent;
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ۥȥ꡼򥯥롣
 *ԳԲġ
 * return : error number
 */
void VfsClose(void *i_vent)
{
	VENTRY *vent = i_vent;

	modifyVfs();
	unlinkVent(vent);
	modifyEndVfs();
}

/*
 * ¥ե򥪡ץ󤷤Ʋۥե륷ƥ³ʤǲۥȥ꡼֤
 *ԳԲġ
 * return : error number
 */
int getVent(
	const char *i_path,
	void **o_vent)			// inode
{
	int error;
	
	modifyVfs();
	error = openVent(i_path, O_RDWR, lookupObj, (VENTRY**) o_vent);
	modifyEndVfs();

	return error;
}

/*
 * getVent()Ǽȥ꡼
 *ԳԲġ
 * return : error number
 */
void releaseVent(
	void *vent)
{
	modifyVfs();
	closeVent(vent);
	modifyEndVfs();
}

/*
 * ¥ե롣
 *ԳԲġ
 * return : error number
 */
int deleteObj(
	const char *i_path,
	const int type)
{
	LOOKUP_OUT lookupOut;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = lookupPath(i_path, lookupObjSymlink, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		lookupOut.vent->del = YES;
		error = -EBUSY;
		break;
	case NOERR:			// ۥե̵
		error = lookupOut.fs->delete(lookupOut.parentPath, lookupOut.din, lookupOut.parentVent->inodeObj, type);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int renamePath(
	const char *oldPath,
	const char *newPath)
{
	LOOKUP_OUT lookupOutOld;
	LOOKUP_OUT lookupOutNew;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	// renameե򸡺
	error = lookupPath(oldPath, lookupObjSymlink, &lookupOutOld);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		error = -EBUSY;
		goto ERR;
	case NOERR:			// ¥եˤ
		break;
	default:
		goto ERR;
	}

	// renameե򸡺
	error = lookupPath(newPath, lookupObjSymlink, &lookupOutNew);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		//
	case NOERR:			// ¥եˤ
		//
	case -ENOENT:		// ¥ե̵
		break;
	default:
		goto ERR;
	}

	if (lookupOutOld.din == lookupOutNew.din) {
		error = lookupOutOld.fs->rename(lookupOutOld.din, 
			lookupOutOld.parentVent->inodeObj, lookupOutOld.parentPath, 
			lookupOutNew.parentVent->inodeObj, lookupOutNew.parentPath);
	}
	else {
		error = -EXDEV;
	}
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ѥե륹ơȤ
 *ԳԲġ
 * return : error number
 */
int statPath(
	const char *i_path,
	struct stat *m_stat)
{
	VENTRY *vent;
	int error;

	modifyVfs();

	error = openVent(i_path, O_RDONLY, lookupObj, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = vent->fs->stat(vent->din, vent->inodeObj, m_stat);
	closeVent(vent);
ERR:
	modifyEndVfs();

	return error;
}

/*
 * ѥե륹ơȤ
 *ԳԲġ
 * return : error number
 */
int statPathSymlink(
	const char *i_path,
	struct stat *m_stat)
{
	VENTRY *vent;
	int error;

	modifyVfs();

	error = openVent(i_path, O_RDONLY, lookupObjSymlink, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = vent->fs->stat(vent->din, vent->inodeObj, m_stat);
	closeVent(vent);
ERR:
	modifyEndVfs();
	
	return error;
}

/*
 * return : error number
 */
int statfsPath(
	const char *i_path,
	struct statfs *m_statfs)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	int error;

	// ۥե륷ƥλȳ
	refVfs();

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	/* ǥ쥯ȥõ */
	error = searchVent(vent, path, &parentPath, &parentVent, &vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (vent->mounted == NULL) {
			VENTRY *p = vent;

			// ̣ʤʤΤα
//			for (p = vent; p != rootEnt; p = p->parent) {
//				if (p->din == vent->din){
//					break;
//				}
//			}
			memcpy(m_statfs->f_mntonname, p->name, p->len);			/* directory on which mounted */
			m_statfs->f_mntonname[p->len] = '\0';
		}
		else{
			memcpy(m_statfs->f_mntonname, vent->name, vent->len);	/* directory on which mounted */
			m_statfs->f_mntonname[vent->len] = '\0';
		}
		break;
	case -ENOENT:	// ۥե̵
		goto ERR;
	default:
		goto ERR;
	}
	
	error = getFsstat(vent, m_statfs);

ERR:
	// ۥե륷ƥλȽλ
	refEndVfs();
	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int readlinkPath(
	const char *i_path,
	const size_t bufSize,
	char *readBuf)
{
	VENTRY *vent;
	struct stat stat;
	size_t size;
	int error;

	modifyVfs();

	error = openVent(i_path, O_RDONLY, lookupObjSymlink, &vent);
	if (error != NOERR) {
		goto  ERR;
	}

	/* ե륿פǧ */
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != 0){
		goto ERR2;
	}
	if ((stat.st_mode & S_IFMT) != S_IFLNK){
		error = -EINVAL;
		goto ERR2;
	}
	
	/* 󥯥ѥɤ */
	size = (stat.st_size < bufSize)? stat.st_size : bufSize;
	error = vent->fs->read(vent->din, vent->inodeObj, readBuf, size, 0);
ERR2:	
	closeVent(vent);
ERR:
	modifyEndVfs();

	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int chownPath(
	const char *i_path,
	const uid_t owner,
	const gid_t group)
{
	VENTRY *vent;
	int error;

	modifyVfs();

	error = openVent(i_path, O_RDWR, lookupObjSymlink, &vent);
	if (error != NOERR) {
		goto ERR;
	}

	if (owner != -1) {
		error = chuid(vent, owner);
		goto ERR2;
	}
	if (group != -1) {
		error = chgrp(vent, group);
	}
ERR2:
	closeVent(vent);
ERR:
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int chownVent(
	void *i_vent,
	const uid_t owner,
	const gid_t group)
{
	VENTRY *vent = i_vent;
	int error = NOERR;

	// ۥե륷ƥι
	modifyVfs();

	if (owner != -1) {
		error = chuid(vent, owner);
		goto ERR;
	}
	if (group != -1) {
		error = chgrp(vent, group);
	}
ERR:
	modifyEndVfs();

	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int utimesPath(
	const char *i_path,
	const struct timeval *times)
{
	VENTRY *vent;
	int error;

	modifyVfs();

	error = openVent(i_path, O_RDWR, lookupObj, &vent);
	if (error != NOERR) {
		goto ERR;
	}

	if (times != NULL) {
		error = chutime(vent, times[0].tv_sec, times[1].tv_sec);
	}
	else {
		error = chutime(vent, sys_time(NULL), sys_time(NULL));
	}

	closeVent(vent);
ERR:
	modifyEndVfs();

	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int linkPath(
	const char *fromPath,
	const char *toPath)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *fromVent;
	VENTRY *vent;
	int error;

	modifyVfs();

	error = openVent(fromPath, O_RDONLY, lookupObj, &fromVent);
	if (error != NOERR) {
		goto ERR;
	}
	
	//ȥǥ쥯ȥ롣
	vent = getTopVdir(toPath, &path);

	/* toPathΥե򸡺롣 */
	error = searchVent(vent, path, &parentPath, &parentVent, &vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		break;
	case -ENOENT:	// ۥե̵
		if (parentVent->din != fromVent->din) {
			error = -EXDEV;
		}
		else {
			error = parentVent->fs->link(parentVent->din, fromVent->inodeObj, parentVent->inodeObj, parentPath);
		}
		break;
	default:
		break;
	}

	closeVent(fromVent);
ERR:
	modifyEndVfs();

	return error;
}

/*
 *Գء
 * return : ۥȥ꡼
 */
void *getChdirVentPath(
	const char *i_path,
	int *o_error)
{
	VENTRY *vent;
	VENTRY *mountedVent;

	*o_error = openPath(i_path, O_RDWR, (void**) &vent);
	if (*o_error != NOERR) {
		return NULL;
	}

	modifyVfs();

	mountedVent = getMountUp(vent);
	if (mountedVent != vent) {
		unlinkVent(vent);
		linkVent(mountedVent);
	}
	if (mountedVent->type != S_IFDIR){
		unlinkVent(mountedVent);
		*o_error = -ENOTDIR;
		mountedVent = NULL;
		goto ERR;
	}
ERR:
	modifyEndVfs();
	
	return mountedVent;
}

/*
 * return : ۥȥ꡼
 */
void *getChdirVent(
	void *i_vent,
	int *o_error)
{
	VENTRY *vent;

	modifyVfs();
	
	vent = getMountUp(i_vent);
	if (vent->type != S_IFDIR) {
		*o_error = -ENOTDIR;
		vent = NULL;
		goto ERR;
	}
	linkVent(vent);

	*o_error = NOERR;
ERR:
	modifyEndVfs();

	return vent;
}

/*
 * retern : poll condition
 */
int pollVent(
	void *i_vent,
	int events)			// events
{
	VENTRY *vent = i_vent;

	return vent->fs->poll(vent->inodeObj, events);
}

/*
 * ѥ롼ȤޤäƥХåե˥ԡ롣
 * return : ԡ
 */
int copyRootPath(
	void *vent,				// 饹ȥȥ꡼
	const size_t size,		// ԡ祵
	char *m_path)			// ԡХåե
{
	int len;

	refVfs();
	len = copyPathBack(vent, m_path, size, isRoot);
	refEndVfs();

	return len;
}

/*
 * searchMount()Ѵؿ¤
 */
typedef struct {
	struct statfs *statfs;
	uint count;
} MOUNT_STAT;

/*
 * Ѵؿ
 * ե륹ơȤμ
 * return : FUNC_NORMAL or FUNC_END
 */
STATIC int funcMountStat(
	VENTRY *vent,
	void *arg)
{
	MOUNT_STAT *mountStat = arg;

	if (mountStat->count <= 0){
		return FUNC_END;
	}

	if (vent->mounted != NULL) {
		if (mountStat->statfs != NULL) {
			if (getFsstat(vent, mountStat->statfs) == NOERR){
				++mountStat->statfs;
			}
		}
		--mountStat->count;
	}

	return FUNC_NORMAL;
}

/*
 * ޥȥȥ꡼򸡺롣
 * return : struct statfsسǼ or ޥȿstruct statfs=NULL
 */
int searchMount(
	struct statfs *statfs, 
	const int i_count)
{
	MOUNT_STAT mountStat;

	mountStat.statfs = statfs;
	mountStat.count = i_count;

	refVfs();
	searchDown(rootEnt, funcMountStat, &mountStat, funcVoidNull, NULL);
	refEndVfs();

	return i_count - mountStat.count;
}

/*
 * ۥȥ꡼ե륿פ
 * return : ե륿
 */
int getTypeVent(void *i_vent)
{
	VENTRY *vent = i_vent;

	return vent->type;
}

/*
 * 롼Ȳۥȥ꡼롣
 * return : 롼Ȳۥȥ꡼
 */
void *getRootEnt()
{
	return rootEnt;
}

/*
 * ü̾롣
 * return : error number
 */
int getTtynameVent(
	void *i_vent,
	char *m_name)
{
	VENTRY *vent = i_vent;

	if (getDevInodeNum(vent->din) == DEVICE_ROOT_INODE) {
		return getTtyname(vent->inodeObj, m_name);
	}
	else {
		return -ENODEV;
	}
}

/*
 *ԳԲġ
 * return : error number
 */
int chmodPath(
	const char *i_path,
	const mode_t mode)
{
	VENTRY *vent;
	int error;

	modifyVfs();

	error = openVent(i_path, O_RDWR, lookupObj, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = chmod(vent, mode);
	closeVent(vent);
ERR:
	modifyEndVfs();

	return error;
}

/*
 * ۥȥ꡼Υǡɤ
 *ԳԲġ
 * return : read bytes or error number
 */
int readVent(
	void *i_vent,
	const size_t size,		// read
	const size_t offset,	// readϥեå
	void *m_buf)			// readХåե
{
	VENTRY *vent = i_vent;
	return vent->fs->read(vent->din, vent->inodeObj, m_buf, size, offset);
}

/*
 * ۥȥ꡼Υǡ˽񤭹
 *ԳԲġ
 * return : read bytes or error number
 */
int writeVent(
	void *i_vent,
	const size_t size,		// write
	const size_t offset,	// writeϥեå
	void *m_buf)			// writeХåե
{
	VENTRY *vent = i_vent;

	return vent->fs->write(vent->din, vent->inodeObj, m_buf, size, offset);
}

/*
 *ԳԲġ
 * return : error number
 */
int ioctlVent(
	void *i_vent,
	const int cmd,
	caddr_t parm,
	int fflag)
{
	VENTRY *vent = i_vent;

	return vent->fs->ioctl(vent->din, vent->inodeObj, cmd, parm, fflag);
}

/*
 *ԳԲġ
 * return : error number
 */
int opendirVent(
	void *i_vent,
	uint *o_block,
	int *o_index)
{
	VENTRY *vent = i_vent;
	int error;

	refVfs();
	error = vent->fs->opendir(vent->din, vent->inodeObj, o_block, o_index);
	refEndVfs();
	
	return error;
}

/*
 *ԳԲġ
 * return : error number
 */
int readdirVent(
	void *i_vent,
	uint *m_block,
	int *m_index,
	char *m_name)
{
	VENTRY *vent = i_vent;

	return vent->fs->readdir(vent->din, vent->inodeObj, m_block, m_index, m_name);
}

/*
 * return : error number
 */
int statVent(
	void *i_vent,
	struct stat *m_stat)
{
	VENTRY *vent = i_vent;

	return vent->fs->stat(vent->din, vent->inodeObj, m_stat);
}

/*
 * return : error number
 */
int chmodVent(
	void *i_vent,
	const mode_t mode)
{
	int error;

	modifyVfs();
	error = chmod(i_vent, mode);
	modifyEndVfs();

	return error;
}
