/*
 * arp.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ARP protocol.
 */

#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <lib/lib.h>
#include <lib/IteratorList.h>
#include <lib/AggregateList.h>
#include <machine/interrupt.h>
#include <machine/lock.h>
#include <kern/kmalloc.h>
#include <sys/Thread.h>
#include <kern/TaskWait.h>
#include <kern/timer.h>
#include <kern/ProcSignal.h>
#include <net/net.h>
#include <net/ethernet.h>
#include <net/ip.h>
#include <net/arp.h>

//#define DEBUG_ARP 1
#ifdef DEBUG_ARP
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif

//=====================================  ===================================================

enum{
	/* Hard type. */
	HARD_TYPE_ETHER= 1,			/* Ethernet. */

	/* Protocol type. */
	PROT_TYPE_IP=0x0800,		/* IP. */

	/* Operation code. */
	OPE_CODE_ARP_REQ= 1,		/* ARP request. */
	OPE_CODE_ARP_REP= 2,		/* ARP repry. */
	OPE_CODE_RARP_REQ=3,		/* RARP request. */
	OPE_CODE_RARP_REP=4,		/* RARP repry. */

	ARP_CACHE_NUM = 64,					/* ARPå */
	ARP_CACHE_HASH = ARP_CACHE_NUM / 2,	/* ARPåϥå */

	RETRY_COUNT		= 3,		// ȥ饤
	REQUEST_TIMEOUT	= 2000,		// ꥯȥॢms
};

typedef struct{
	uint ip;
	char mac[6];
	char next;
}ARP_CACHE;

typedef struct REPLY_WAIT{
	uint32_t	ip;			/* Request IP address. */
	void		*task;
	int			isReply;	/* Reply flag. */
	char		mac[6];		/* Reply mac buffer. */
	List		list;		// 󥯥ꥹ
} REPLY_WAIT;

//===================================== PRIVATE ====================================================

//--------------------------------------------------------------------------------------------------
// ARPå
//--------------------------------------------------------------------------------------------------

static ARP_CACHE arpCache[ARP_CACHE_NUM];
static char cacheEmpty=0;
static char hashTbl[ARP_CACHE_HASH];
static int cacheLock=0;

/*
 * Search previous index address.
 * parameters : IP address
 * return : prev pointer or NULL
 */
static inline char *searchPrevIndex(uint ip)
{
	uchar tbl;
	int i;

	tbl=(ip>>24)%ARP_CACHE_HASH;
	if(hashTbl[tbl]==-1)return &hashTbl[tbl];
	else
		for(i=hashTbl[tbl];;i=arpCache[i].next)
		{
			if(arpCache[i].ip==ip)return NULL;
			if(arpCache[i].next==-1)return &arpCache[i].next;
		}
}

/*
 * åβꡣ
 */
static inline int snatchCache()
{
	int num;
	int i;

	for(i=0;hashTbl[i]==-1;++i);
	num=hashTbl[i];
	hashTbl[i]=arpCache[num].next;

	return num;
}

/*
 * Search cache.
 * parameters : IP address
 * return : Mac address or failed=NULL
 */
static inline char *searchCache(uint ip)
{
	uchar tbl;
	int i;

	tbl = (ip >> 24) % ARP_CACHE_HASH;
	for (i = hashTbl[tbl]; i != -1; i = arpCache[i].next){
		if (arpCache[i].ip == ip){
			return arpCache[i].mac;
		}
	}

	return NULL;
}

/*
 * Add cache.
 */
static void addCache(uint ip,char *mac)
{
	char *prev;
	char num;

	enter_spinlock(&cacheLock);
	{
		if((prev=searchPrevIndex(ip))==NULL)
			memcpy(searchCache(ip),mac,6);
		else
		{
			if(cacheEmpty==-1)num=snatchCache();
			else
			{
				num=cacheEmpty;
				cacheEmpty=arpCache[(int)num].next;
			}

			arpCache[(int)num].ip=ip;
			memcpy(arpCache[(int)num].mac,mac,6);
			arpCache[(int)num].next=-1;
			*prev=num;
		}
	}
	exit_spinlock(&cacheLock);
}

/*
 * Init arp cache.
 */
static void initCache()
{
	int i;

	/* å󥯤Ϣ롣 */
	for(i=0;i<ARP_CACHE_NUM;++i)arpCache[i].next=i+1;
	arpCache[ARP_CACHE_NUM-1].next=-1;

	/* ϥåơ֥ν */
	memset(hashTbl,0xff,ARP_CACHE_HASH);
}

//--------------------------------------------------------------------------------------------------
// 
//--------------------------------------------------------------------------------------------------

static char broadcastMac[]={0xff,0xff,0xff,0xff,0xff,0xff};

/*
 * Transfer ARP frame.
 */
static void transArp(int num, uint dstip, char *dstmac, ushort opecode)
{
	ARP_HEADER head;
	TRANS_BUF_INFO tbi;

	/* Make ARP head. */
	head.hardType = swapWord(HARD_TYPE_ETHER);
	head.protType = swapWord(PROT_TYPE_IP);
	head.hardAddrLen = 6;
	head.protAddrLen = 4;
	head.opeCode = swapWord(opecode);
	getEtherMac(num, head.srcMac);
	head.srcIp = getEtherIp(num);
	if (dstmac == broadcastMac){
		memset(head.dstMac,0,6);
	}
	else{
		memcpy(head.dstMac, dstmac, 6);
	}
	head.dstIp = dstip;

	/* Set transmission buffer structure. */
	tbi.data[0] = (char*)&head;
	tbi.size[0] = sizeof(ARP_HEADER);
	tbi.size[1] = 0;
	tbi.size[2] = 0;
	tbi.type = ETHERTYPE_ARP;

	transEther(getEtherIfnet(num), &tbi, dstmac);
}

//--------------------------------------------------------------------------------------------------
// ꥯ
//--------------------------------------------------------------------------------------------------

static AggregateList aggrReply;
static AggregateListMethod aggrMethod;
static int waitGate;				// Ԥѥå

/*
 * ARPν
 */
STATIC void readyReply(
	const uint32_t dstIp,
	REPLY_WAIT *m_replyWait)
{
	int eflags;

	// 
	m_replyWait->ip = dstIp;
	m_replyWait->task = getCurrentTask();
	m_replyWait->isReply = NO;
	listConstructor(&m_replyWait->list, m_replyWait);

	// 󥯤³
	eflags = enterCli();
	enter_spinlock(&waitGate);
	{
		aggrMethod.insertHead(&aggrReply, &m_replyWait->list);
	}
	exit_spinlock(&waitGate);
	exitCli(eflags);
}

/*
 * return : YES or NO
 */
STATIC INLINE int isReply(REPLY_WAIT *wait)
{
	return wait->isReply;
}

STATIC INLINE void releaseReplyWait(REPLY_WAIT *wait)
{
	int eflags;

	// 󥯤
	eflags = enterCli();
	enter_spinlock(&waitGate);
	{
		aggrMethod.removeEntry(&aggrReply, &wait->list);
	}
	exit_spinlock(&waitGate);
	exitCli(eflags);
}

/*
 * Receive reply.
 *Գ
 * return : 0 or 1
 */
STATIC int receiveReply(const uint32_t srcIp, char *mac)
{
	IteratorList iterator;
	int taskSwitch = 0;
	int eflags;

	eflags = enterCli();
	enter_spinlock(&waitGate);
	{
		IteratorListConstruct(&iterator, &aggrReply, &aggrMethod);
		while (iterator.hasNext(&iterator) == BOOL_TRUE) {
			REPLY_WAIT *next = iterator.next(&iterator);

			if (next->ip == srcIp){
				memcpy(next->mac, mac, 6);
				next->isReply = YES;
				TaskAwakeSoon(getTaskWait(next->task));
				taskSwitch = 1;
				break;
			}
		}
	}
	exit_spinlock(&waitGate);
	exitCli(eflags);

	return taskSwitch;
}

//===================================== PUBLIC =====================================================

// Get destination MAC.
// return : error number
int getMac(const int num, const uint32_t dstIp, char *m_mac)
{
	char *cacheMac;
	REPLY_WAIT *wait;
	int error;
	int i;

	// å򸡺
	cacheMac = searchCache(dstIp);
	if (cacheMac != NULL){
		memcpy(m_mac, cacheMac, 6);
		return NOERR;
	}

	// 
	wait = kmalloc(sizeof(*wait));
	if (wait == NULL) {
		return -ENOMEM;
	}
	readyReply(dstIp, wait);

	// ꥯARP
	error = -ENETUNREACH;
	for (i = 0; i < RETRY_COUNT; ++i){
		transArp(num, dstIp, broadcastMac, OPE_CODE_ARP_REQ);

		TaskWaitTimer(REQUEST_TIMEOUT);

		if (isReply(wait) == YES){
			addCache(dstIp, wait->mac);
			memcpy(m_mac, wait->mac, 6);
			error = NOERR;
			break;
		}
	}

	// 
	releaseReplyWait(wait);
	kfree(wait);

	return error;
}

/*
 * Receive ARP frame.
 *Գ
 * parameters : device number,ARP_HEADER address
 * return : task switch on=1,off=0
 */
int receiveArp(int num,ARP_HEADER *head)
{
	int rest = 0;
/***************************************************************************************************************
uchar *srcip,*dstip;

srcip=(uchar*)&head->srcIp;
dstip=(uchar*)&head->dstIp;
printk("ARP_HEADER htype=%x,ptype=%x,hsize=%x,psize=%x,opecode=%x\n"
		"source mac=%d-%d-%d-%d-%d-%d,IP=%d-%d-%d-%d\n"
		"dest   mac=%d-%d-%d-%d-%d-%d,IP=%d-%d-%d-%d\n",
       swapWord(head->hardType),swapWord(head->protType),head->hardAddrLen,head->protAddrLen,
       swapWord(head->opeCode),
       head->srcMac[0],head->srcMac[1],head->srcMac[2],head->srcMac[3],head->srcMac[4],head->srcMac[5],
       srcip[0],srcip[1],srcip[2],srcip[3],
       head->dstMac[0],head->dstMac[1],head->dstMac[2],head->dstMac[3],head->dstMac[4],head->dstMac[5],
       dstip[0],dstip[1],dstip[2],dstip[3]);
***************************************************************************************************************/
	/* IPγǧ */
	if (head->dstIp != getEtherIp(num)){
		return 0;
	}

	/* ڥ졼󥳡ɤγǧ */
	switch (swapWord(head->opeCode)){
		case OPE_CODE_ARP_REQ:
			transArp(num, head->srcIp, head->srcMac, OPE_CODE_ARP_REP);
			addCache(head->srcIp, head->srcMac);
			break;
		case OPE_CODE_ARP_REP:
			rest = receiveReply(head->srcIp, head->srcMac);
			break;
		case OPE_CODE_RARP_REQ:
		case OPE_CODE_RARP_REP:
			break;
	}

	return rest;
}

/*
 * Init arp protocol.
 */
void initArp()
{
	initCache();
	AggregateListConstructor(&aggrReply, &aggrMethod);
}

/***************************************************************************/

#ifdef DEBUG
int test_arp()
{
	char mac[6];

	printk("test_arp\n");
	if(getMac(0,IP_ADDR(172,25,0,2),mac) != NOERR)
		printk("Failed getMac!\n");
	else
		printk("MAC=%d-%d-%d-%d-%d-%d\n",mac[0],mac[1],mac[2],mac[3],mac[4],mac[5]);

	return 0;
}
#endif
