/*
 * malloc.c
 */

#include <sys/types.h>
#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

//=====================================  ===================================================

enum{
	MIN_SIZE	= 8,		/* ǾХȿ */
	HEADER_NUM	= 33,		/* إåơ֥ */
	MAGIC_NUMBER= 0xfafa,	/* إåޥåʥС */
};

typedef union MALLOC_HEAD {
	union MALLOC_HEAD *next;
	struct{
		ushort index;
		ushort magic;
	} head;
} MALLOC_HEAD;

//===================================== Х륤ݡ =======================================

extern void enter_spinlock(int*);
extern void exit_spinlock(int*);

//===================================== PRIVATE ====================================================

static const char indexNumber[]={
	0,1,2,2,3,3,3,3,4,4,4,4,4,4,4,4,
	5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
};

static MALLOC_HEAD *nextHead[HEADER_NUM];
static int spinLock;

//===================================== PUBLIC =====================================================

void *malloc(
	size_t size)
{
	int index;
	size_t rsize = size + sizeof(MALLOC_HEAD);
	MALLOC_HEAD *head;

	if (size == 0) {
		return  NULL;
	}
	if (UINT_MAX - sizeof(MALLOC_HEAD) < size) {
		return  NULL;
	}

	if (rsize < MIN_SIZE) {
		rsize = MIN_SIZE;
	}

	/* ǥå */
	if (rsize < 0x100) {
		index = indexNumber[rsize];
	}
	else if (rsize < 0x10000) {
		index = indexNumber[rsize >> 8] + 8;
	}
	else if (rsize < 0x1000000) {
		index = indexNumber[rsize >> 16] + 16;
	}
	else {
		index = indexNumber[rsize >> 24] + 24;
		if (index == 32) {
			return NULL;
		}
	}

	enter_spinlock(&spinLock);
	{
		// ꡼
		if (nextHead[index] != NULL) {
			head = nextHead[index];
			nextHead[index] = head->next;
		}
		else {
			head = sbrk(1 << index);
			if (head == NULL) {
				exit_spinlock(&spinLock);
				return NULL;
			}
		}
		head->head.magic = MAGIC_NUMBER;
		head->head.index = index;
	}
	exit_spinlock(&spinLock);

	return (void*) (head + 1);
}

void free(
	void *ptr)
{
	MALLOC_HEAD *head;
	int index;

	if (ptr == NULL) {
		return;
	}

	head = (MALLOC_HEAD*) ptr - 1;
	if (head->head.magic != MAGIC_NUMBER)	/* ҡפ˲Ƥ롣 */
	{
		fprintf(stderr, "free() : Heap is uncomplete! ptr=0x%x Call from 0x%x\n", ptr, *((uint*)&ptr - 1));
		return;
	}

	enter_spinlock(&spinLock);
	{
		index = head->head.index;
		head->next = nextHead[index];
		nextHead[index] = head;
	}
	exit_spinlock(&spinLock);
}

void *realloc(
	void *ptr,
	size_t size)
{
	MALLOC_HEAD *head;
	size_t rsize = size + sizeof(MALLOC_HEAD);

	if (ptr == NULL) {
		return malloc(size);
	}
	if (size == 0) {
		free(ptr);
		return NULL;
	}

	head = (MALLOC_HEAD*)ptr-1;

	// ҡפ˲Ƥ
	if (head->head.magic != MAGIC_NUMBER) {
		return NULL;
	}

	if ((1 << head->head.index) < rsize) {
		void *mem = malloc(size);
		if (mem == NULL) {
			return NULL;
		}
		memcpy(mem, ptr, (1 << head->head.index) - sizeof(MALLOC_HEAD));
		free(ptr);
		return mem;
	}
	else {
		return ptr;
	}
}
