/*	$NetBSD: ata_wdc.c,v 1.85 2006/09/30 15:56:18 itohy Exp $	*/

/*
 * Copyright (c) 1998, 2001, 2003 Manuel Bouyer.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Manuel Bouyer.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1998, 2004 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Charles M. Hannum, by Onno van der Linden and by Manuel Bouyer.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * 2009: modified by minoru murashima.
 */

#include <sys/cdefs.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <machine/cpu.h>
#include <machine/bus.h>
#include <ic/wdcreg.h>
#include <ic/wdcvar.h>
#include <ata/atavar.h>
#include <ata/atareg.h>

//=====================================  ===================================================

#define ATA_DELAY 10000 /* 10s for a drive I/O */
#define WDC_ATA_NOERR 0x00 /* Drive doesn't report an error */
#define WDC_ATA_RECOV 0x01 /* There was a recovered error */
#define WDC_ATA_ERR   0x02 /* Drive reports an error */

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static int wdc_ata_bio_intr(struct ata_channel*, struct ata_xfer*, int);

static int wdc_ata_err(
	struct ata_drive_datas *drvp, 
	struct ata_bio *ata_bio)
{
	struct ata_channel *chp = drvp->chnl_softc;
	ata_bio->error = 0;
	if (chp->ch_status & WDCS_BSY) {
		ata_bio->error = TIMEOUT;
		return WDC_ATA_ERR;
	}

	if (chp->ch_status & WDCS_DWF) {
		ata_bio->error = ERR_DF;
		return WDC_ATA_ERR;
	}

	if (chp->ch_status & WDCS_ERR) {
		ata_bio->error = ERROR;
		ata_bio->r_error = chp->ch_error;
		if (ata_bio->r_error & (WDCE_BBK | WDCE_UNC | WDCE_IDNF | WDCE_ABRT | WDCE_TK0NF | WDCE_AMNF)) {
			return WDC_ATA_ERR;
		}
		return WDC_ATA_NOERR;
	}

	if (chp->ch_status & WDCS_CORR) {
		ata_bio->flags |= ATA_CORR;
	}
	return WDC_ATA_NOERR;
}

static void wdc_ata_bio_done(
	struct ata_channel *chp, 
	struct ata_xfer *xfer)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct ata_bio *ata_bio = xfer->c_cmd;
	int drive = xfer->c_drive;

	callout_stop(&chp->ch_callout);

	/* feed back residual bcount to our caller */
	ata_bio->bcount = xfer->c_bcount;

	/* mark controller inactive and free xfer */
	chp->ch_queue->active_xfer = NULL;
	ata_free_xfer(chp, xfer);

	if (chp->ch_drive[drive].drive_flags & DRIVE_WAITDRAIN) {
		ata_bio->error = ERR_NODEV;
		chp->ch_drive[drive].drive_flags &= ~DRIVE_WAITDRAIN;
		wakeup(&chp->ch_queue->active_xfer);
	}
	ata_bio->flags |= ATA_ITSDONE;
	(*chp->ch_drive[drive].drv_done)(chp->ch_drive[drive].drv_softc);
	atastart(chp);
}

static void _wdc_ata_bio_start(
	struct ata_channel *chp, 
	struct ata_xfer *xfer)
{
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	struct ata_bio *ata_bio = xfer->c_cmd;
	struct ata_drive_datas *drvp = &chp->ch_drive[xfer->c_drive];
	int wait_flags = (xfer->c_flags & C_POLL) ? AT_POLL : 0;
	u_int16_t cyl;
	u_int8_t head, sect, cmd = 0;
	int nblks;

#if NATA_DMA
	if (xfer->c_flags & (C_DMA | C_PIOBM)) {
		if (drvp->n_xfers <= NXFER) {
			drvp->n_xfers++;
		}
	}
#endif
again:
	/*
	 * When starting a multi-sector transfer, or doing single-sector transfers...
	 */
	if (xfer->c_skip == 0 || (ata_bio->flags & ATA_SINGLE) != 0) {
		if (ata_bio->flags & ATA_SINGLE) {
			nblks = 1;
		}
		else {
			nblks = xfer->c_bcount / ata_bio->lp->d_secsize;
		}
		if (ata_bio->flags & ATA_LBA48) {
			sect = 0;
			cyl =  0;
			head = 0;
		} 
		else if (ata_bio->flags & ATA_LBA) {
			sect = (ata_bio->blkno >> 0) & 0xff;
			cyl = (ata_bio->blkno >> 8) & 0xffff;
			head = (ata_bio->blkno >> 24) & 0x0f;
			head |= WDSD_LBA;
		} 
		else {
			int blkno = ata_bio->blkno;
			sect = blkno % ata_bio->lp->d_nsectors;
			sect++;    /* Sectors begin with 1, not 0. */
			blkno /= ata_bio->lp->d_nsectors;
			head = blkno % ata_bio->lp->d_ntracks;
			blkno /= ata_bio->lp->d_ntracks;
			cyl = blkno;
			head |= WDSD_CHS;
		}
#if NATA_DMA
		if (xfer->c_flags & C_DMA) {
			int dma_flags = 0;
			u_int8_t cmd;

			dma_flags = (ata_bio->flags & ATA_READ) ?  WDC_DMA_READ : 0;
			if (ata_bio->flags & ATA_LBA48) {
				dma_flags |= WDC_DMA_LBA48;
			}
			ata_bio->nblks = nblks;
			ata_bio->nbytes = xfer->c_bcount;
			cmd = (ata_bio->flags & ATA_READ) ? WDCC_READDMA : WDCC_WRITEDMA;

	    	/* Init the DMA channel. */
			(*wdc->dma_init)(wdc->dma_arg, chp->ch_channel, xfer->c_drive, xfer->uio, dma_flags);

			/* Initiate command */
			if (wdc->select) {
				wdc->select(chp, xfer->c_drive);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (xfer->c_drive << 4));
			switch(wdc_wait_for_ready(chp, ATA_DELAY, wait_flags)) {
			case WDCWAIT_OK:
				break;
			case WDCWAIT_TOUT:
				goto timeout;
			case WDCWAIT_THR:
				return;
			}
			if (ata_bio->flags & ATA_LBA48) {
			    wdccommandext(chp, xfer->c_drive, atacmd_to48(cmd), (u_int64_t)ata_bio->blkno, nblks);
			} 
			else {
			    wdccommand(chp, xfer->c_drive, cmd, cyl, head, sect, nblks, 0);
			}

			/* Wait for IRQ (either real) */
			if ((ata_bio->flags & ATA_POLL) == 0) {
				chp->ch_flags |= ATACH_IRQ_WAIT;
				chp->waitTask = ata_bio->task;
			} 

			/* start the DMA channel */
			chp->ch_flags |= ATACH_DMA_WAIT;
			(*wdc->dma_start)(wdc->dma_arg, chp->ch_channel, xfer->c_drive);

			/* start timeout machinery */
			if ((xfer->c_flags & C_POLL) == 0) {
				callout_reset(&chp->ch_callout, ATA_DELAY / 1000 * hz, wdctimeout, chp);
			}

			/* wait for irq */
			goto intr;
		}
// do_pio:
#endif	/* NATA_DMA */
		ata_bio->nblks = min(nblks, ata_bio->multi);
		ata_bio->nbytes = ata_bio->nblks * ata_bio->lp->d_secsize;
		KASSERT(nblks == 1 || (ata_bio->flags & ATA_SINGLE) == 0);
		if (1 < ata_bio->nblks) {
			cmd = (ata_bio->flags & ATA_READ) ? WDCC_READMULTI : WDCC_WRITEMULTI;
		} 
		else {
			cmd = (ata_bio->flags & ATA_READ) ? WDCC_READ : WDCC_WRITE;
		}
		/* Initiate command! */
		if (wdc->select) {
			wdc->select(chp, xfer->c_drive);
		}
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (xfer->c_drive << 4));
		switch(wdc_wait_for_ready(chp, ATA_DELAY, wait_flags)) {
		case WDCWAIT_OK:
			break;
		case WDCWAIT_TOUT:
			goto timeout;
		case WDCWAIT_THR:
			return;
		}

		if ((ata_bio->flags & ATA_POLL) == 0) {
			chp->ch_flags |= ATACH_IRQ_WAIT;
			chp->waitTask = ata_bio->task;
		} 

		if (ata_bio->flags & ATA_LBA48) {
		    wdccommandext(chp, xfer->c_drive, atacmd_to48(cmd), (u_int64_t) ata_bio->blkno, nblks);
		} 
		else {
			wdccommand(chp, xfer->c_drive, cmd, cyl, head, sect, nblks,
				(ata_bio->lp->d_type == DTYPE_ST506) ? ata_bio->lp->d_precompcyl / 4 : 0);
		}
		/* start timeout machinery */
		if ((xfer->c_flags & C_POLL) == 0) {
			callout_reset(&chp->ch_callout, ATA_DELAY / 1000 * hz, wdctimeout, chp);
		}
	} 
	else if (1 < ata_bio->nblks) {
		panic("ata pio mode is not support");
		/* The number of blocks in the last stretch may be smaller. */
		nblks = xfer->c_bcount / ata_bio->lp->d_secsize;
		if (ata_bio->nblks > nblks) {
			ata_bio->nblks = nblks;
			ata_bio->nbytes = xfer->c_bcount;
		}
	}

	/* If this was a write and not using DMA, push the data. */
	if ((ata_bio->flags & ATA_READ) == 0) {
		int i;

		/*
		 * we have to busy-wait here, we can't rely on running in
		 * thread context.
		 */
		if (wdc_wait_for_drq(chp, ATA_DELAY, AT_POLL) != 0) {
			printf("%s:%d:%d: timeout waiting for DRQ, st=0x%02x, err=0x%02x\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, 
				chp->ch_status, chp->ch_error);
			if (wdc_ata_err(drvp, ata_bio) != WDC_ATA_ERR) {
				ata_bio->error = TIMEOUT;
			}
			wdc_ata_bio_done(chp, xfer);
			return;
		}
		if (wdc_ata_err(drvp, ata_bio) == WDC_ATA_ERR) {
			wdc_ata_bio_done(chp, xfer);
			return;
		}

		for (i = 0; i < xfer->uio->uio_iovcnt; ++i) {
			wdc->dataout_pio(chp, drvp->drive_flags, xfer->uio->uio_iov[i].iov_base, xfer->uio->uio_iov[i].iov_len);
		}
	}

#if NATA_DMA
intr:
#endif
	/* Wait for IRQ (either real or polled) */
	if ((ata_bio->flags & ATA_POLL) == 0) {
		/*
		 *ԸƤեॢȤ
		 *Ի͡	if ((xfer->c_flags & C_POLL) == 0) {
		 *				callout_reset(&chp->ch_callout, ATA_DELAY / 1000 * hz, wdctimeout, chp);
		 *			}
		 */
		waitIntr(ATA_DELAY);
		chp->waitTask = NULL;
		if (wdc_ata_bio_intr(chp, xfer, 1) == 0) {
			printf("There is a wrong thing at wdcintr!\n");
		}
	} 
	else {
		/* Wait for at last 400ns for status bit to be valid */
		delay(1);
#if NATA_DMA
		if (chp->ch_flags & ATACH_DMA_WAIT) {
			wdc_dmawait(chp, xfer, ATA_DELAY);
			chp->ch_flags &= ~ATACH_DMA_WAIT;
		}
#endif
		wdc_ata_bio_intr(chp, xfer, 0);
		if ((ata_bio->flags & ATA_ITSDONE) == 0) {
			goto again;
		}
	}
	return;

timeout:
	printf("%s:%d:%d: not ready, st=0x%02x, err=0x%02x\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, chp->ch_status, 
		chp->ch_error);
	if (wdc_ata_err(drvp, ata_bio) != WDC_ATA_ERR) {
		ata_bio->error = TIMEOUT;
	}
	wdc_ata_bio_done(chp, xfer);
	return;
}

static int wdc_ata_bio_intr(
	struct ata_channel *chp, 
	struct ata_xfer *xfer, 
	int irq)
{
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct ata_bio *ata_bio = xfer->c_cmd;
	struct ata_drive_datas *drvp = &chp->ch_drive[xfer->c_drive];
	int drv_err;

	/* Is it not a transfer, but a control operation? */
	if (drvp->state < READY) {
		printf("%s:%d:%d: bad state %d in wdc_ata_bio_intr\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, drvp->state);
		panic("wdc_ata_bio_intr: bad state");
	}

	/*
	 * if we missed an interrupt in a PIO transfer, reset and restart.
	 * Don't try to continue transfer, we may have missed cycles.
	 */
	if ((xfer->c_flags & (C_TIMEOU | C_DMA)) == C_TIMEOU) {
		ata_bio->error = TIMEOUT;
		wdc_ata_bio_done(chp, xfer);
		return 1;
	}

	/* Ack interrupt done by wdc_wait_for_unbusy */
	if (wdc_wait_for_unbusy(chp, (irq == 0) ? ATA_DELAY : 0, AT_POLL) < 0) {
		if (irq && (xfer->c_flags & C_TIMEOU) == 0) {
			return 0; /* IRQ was not for us */
		}
		printf("%s:%d:%d: device timeout, c_bcount=%d, c_skip%d\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, 
			xfer->c_bcount, xfer->c_skip);
		ata_bio->error = TIMEOUT;
		wdc_ata_bio_done(chp, xfer);
		return 1;
	}
	if (wdc->irqack) {
		wdc->irqack(chp);
	}

	drv_err = wdc_ata_err(drvp, ata_bio);

#if NATA_DMA
	/* If we were using DMA, Turn off the DMA channel and check for error */
	if (xfer->c_flags & C_DMA) {
		if (ata_bio->flags & ATA_POLL) {
			/*
			 * IDE drives deassert WDCS_BSY before transfer is
			 * complete when using DMA. Polling for DRQ to deassert
			 * is not enough DRQ is not required to be
			 * asserted for DMA transfers, so poll for DRDY.
			 */
			if (wdcwait(chp, WDCS_DRDY | WDCS_DRQ, WDCS_DRDY, ATA_DELAY, ATA_POLL) == WDCWAIT_TOUT) {
				printf("%s:%d:%d: polled transfer timed out (st=0x%x)\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, 
					chp->ch_status);
				ata_bio->error = TIMEOUT;
				drv_err = WDC_ATA_ERR;
			}
		}
		if (wdc->dma_status != 0) {
			if (drv_err != WDC_ATA_ERR) {
				ata_bio->error = ERR_DMA;
				drv_err = WDC_ATA_ERR;
			}
		}
		if (chp->ch_status & WDCS_DRQ) {
			if (drv_err != WDC_ATA_ERR) {
				printf("%s:%d:%d: intr with DRQ (st=0x%x)\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, chp->ch_status);
				ata_bio->error = TIMEOUT;
				drv_err = WDC_ATA_ERR;
			}
		}
		if (drv_err != WDC_ATA_ERR) {
			goto end;
		}
		if (ata_bio->r_error & WDCE_CRC || ata_bio->error == ERR_DMA) {
			ata_dmaerr(drvp, (xfer->c_flags & C_POLL) ? AT_POLL : 0);
		}
	}
#endif	/* NATA_DMA */

	/* if we had an error, end */
	if (drv_err == WDC_ATA_ERR) {
		wdc_ata_bio_done(chp, xfer);
		return 1;
	}

	/* If this was a read and not using DMA, fetch the data. */
	if ((ata_bio->flags & ATA_READ) != 0) {
		int i;

		if ((chp->ch_status & WDCS_DRQ) != WDCS_DRQ) {
			printf("%s:%d:%d: read intr before drq\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive);
			ata_bio->error = TIMEOUT;
			wdc_ata_bio_done(chp, xfer);
			return 1;
		}
		
		for (i = 0; i < xfer->uio->uio_iovcnt; ++i) {
			wdc->datain_pio(chp, drvp->drive_flags, xfer->uio->uio_iov[i].iov_base, xfer->uio->uio_iov[i].iov_len);
		}
	}

#if NATA_DMA
end:
#endif
	ata_bio->blkno += ata_bio->nblks;
	ata_bio->blkdone += ata_bio->nblks;
	xfer->c_skip += ata_bio->nbytes;
	xfer->c_bcount -= ata_bio->nbytes;
	/* See if this transfer is complete. */
	if (0 < xfer->c_bcount) {
		if ((ata_bio->flags & ATA_POLL) == 0) {
			/* Start the next operation */
			_wdc_ata_bio_start(chp, xfer);
		} 
		else {
			/* Let _wdc_ata_bio_start do the loop */
			return 1;
		}
	} 
	else { /* Done with this transfer */
		ata_bio->error = NOERROR;
		wdc_ata_bio_done(chp, xfer);
	}
	return 1;
}

static void wdc_ata_bio_kill_xfer(
	struct ata_channel *chp, 
	struct ata_xfer *xfer,
    int reason)
{
	struct ata_bio *ata_bio = xfer->c_cmd;
	int drive = xfer->c_drive;

	ata_free_xfer(chp, xfer);

	ata_bio->flags |= ATA_ITSDONE;
	switch (reason) {
	case KILL_GONE:
		ata_bio->error = ERR_NODEV;
		break;
	case KILL_RESET:
		ata_bio->error = ERR_RESET;
		break;
	default:
		printf("wdc_ata_bio_kill_xfer: unknown reason %d\n", reason);
		panic("wdc_ata_bio_kill_xfer");
	}
	ata_bio->r_error = WDCE_ABRT;
	(*chp->ch_drive[drive].drv_done)(chp->ch_drive[drive].drv_softc);
}

static void wdc_ata_bio_start(
	struct ata_channel *chp, 
	struct ata_xfer *xfer)
{
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	struct ata_bio *ata_bio = xfer->c_cmd;
	struct ata_drive_datas *drvp = &chp->ch_drive[xfer->c_drive];
	int wait_flags = (xfer->c_flags & C_POLL) ? AT_POLL : 0;
	const char *errstring;

	/* Do control operations specially. */
	if (__predict_false(drvp->state < READY)) {
		/*
		 * Actually, we want to be careful not to mess with the control
		 * state if the device is currently busy, but we can assume
		 * that we never get to this point if that's the case.
		 */
		/* If it's not a polled command, we need the kernel thread */
		if ((xfer->c_flags & C_POLL) == 0 && (chp->ch_flags & ATACH_TH_RUN) == 0) {
			chp->ch_queue->queue_freeze++;
			wakeup(&chp->ch_thread);
			return;
		}
		/*
		 * disable interrupts, all commands here should be quick
		 * enouth to be able to poll, and we don't go here that often
		 */
		bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT | WDCTL_IDS);
		if (wdc->select) {
			wdc->select(chp, xfer->c_drive);
		}
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (xfer->c_drive << 4));
		DELAY(10);
		errstring = "wait";
		if (wdcwait(chp, WDCS_DRDY, WDCS_DRDY, ATA_DELAY, wait_flags)) {
			goto ctrltimeout;
		}
		wdccommandshort(chp, xfer->c_drive, WDCC_RECAL);
		/* Wait for at last 400ns for status bit to be valid */
		DELAY(1);
		errstring = "recal";
		if (wdcwait(chp, WDCS_DRDY, WDCS_DRDY, ATA_DELAY, wait_flags)) {
		goto ctrltimeout;
		}
		if (chp->ch_status & (WDCS_ERR | WDCS_DWF)) {
			goto ctrlerror;
		}
		/* Don't try to set modes if controller can't be adjusted */
		if (atac->atac_set_modes == NULL) {
			goto geometry;
		}
		/* Also don't try if the drive didn't report its mode */
		if ((drvp->drive_flags & DRIVE_MODE) == 0) {
			goto geometry;
		}
		wdccommand(chp, drvp->drive, SET_FEATURES, 0, 0, 0, 0x08 | drvp->PIO_mode, WDSF_SET_MODE);
		errstring = "piomode";
		if (wdcwait(chp, WDCS_DRDY, WDCS_DRDY, ATA_DELAY, wait_flags)) {
			goto ctrltimeout;
		}
		if (chp->ch_status & (WDCS_ERR | WDCS_DWF)) {
			goto ctrlerror;
		}
#if NATA_DMA
#if NATA_UDMA
		if (drvp->drive_flags & DRIVE_UDMA) {
			wdccommand(chp, drvp->drive, SET_FEATURES, 0, 0, 0, 0x40 | drvp->UDMA_mode, WDSF_SET_MODE);
		} 
		else
#endif
		if (drvp->drive_flags & DRIVE_DMA) {
			wdccommand(chp, drvp->drive, SET_FEATURES, 0, 0, 0, 0x20 | drvp->DMA_mode, WDSF_SET_MODE);
		} 
		else {
			goto geometry;
		}
		errstring = "dmamode";
		if (wdcwait(chp, WDCS_DRDY, WDCS_DRDY, ATA_DELAY, wait_flags)) {
		goto ctrltimeout;
		}
		if (chp->ch_status & (WDCS_ERR | WDCS_DWF)) {
			goto ctrlerror;
		}
#endif	/* NATA_DMA */
geometry:
		if (ata_bio->flags & ATA_LBA) {
			goto multimode;
		}
		wdccommand(chp, xfer->c_drive, WDCC_IDP, ata_bio->lp->d_ncylinders, ata_bio->lp->d_ntracks - 1, 0, ata_bio->lp->d_nsectors,
			(ata_bio->lp->d_type == DTYPE_ST506) ? ata_bio->lp->d_precompcyl / 4 : 0);
		errstring = "geometry";
		if (wdcwait(chp, WDCS_DRDY, WDCS_DRDY, ATA_DELAY, wait_flags)) {
			goto ctrltimeout;
		}
		if (chp->ch_status & (WDCS_ERR | WDCS_DWF)) {
			goto ctrlerror;
		}
multimode:
		if (ata_bio->multi == 1) {
		goto ready;
		}
		wdccommand(chp, xfer->c_drive, WDCC_SETMULTI, 0, 0, 0, ata_bio->multi, 0);
		errstring = "setmulti";
		if (wdcwait(chp, WDCS_DRDY, WDCS_DRDY, ATA_DELAY, wait_flags)) {
			goto ctrltimeout;
		}
		if (chp->ch_status & (WDCS_ERR | WDCS_DWF)) {
			goto ctrlerror;
		}
ready:
		drvp->state = READY;
		/*
		 * The drive is usable now
		 */
		bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT);
		delay(10); /* some drives need a little delay here */
	}

	_wdc_ata_bio_start(chp, xfer);
	return;

ctrltimeout:
	printf("%s:%d:%d: %s timed out\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, errstring);
	ata_bio->error = TIMEOUT;
	goto ctrldone;
ctrlerror:
	printf("%s:%d:%d: %s ", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, errstring);
	if (chp->ch_status & WDCS_DWF) {
		printf("drive fault\n");
		ata_bio->error = ERR_DF;
	} 
	else {
		printf("error (%x)\n", chp->ch_error);
		ata_bio->r_error = chp->ch_error;
		ata_bio->error = ERROR;
	}
ctrldone:
	drvp->state = 0;
	wdc_ata_bio_done(chp, xfer);
	bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT);
	return;
}

/*
 * Handle block I/O operation. Return ATACMD_COMPLETE, ATACMD_QUEUED, or
 * ATACMD_TRY_AGAIN. Must be called at splbio().
 */
static int wdc_ata_bio(
	struct ata_drive_datas *drvp, 
	struct ata_bio *ata_bio)
{
	struct ata_xfer *xfer;
	struct ata_channel *chp = drvp->chnl_softc;
	struct atac_softc *atac = chp->ch_atac;

	xfer = ata_get_xfer(ATAXF_NOSLEEP);
	if (xfer == NULL) {
		return ATACMD_TRY_AGAIN;
	}
	if (atac->atac_cap & ATAC_CAP_NOIRQ) {
		ata_bio->flags |= ATA_POLL;
	}
	if (ata_bio->flags & ATA_POLL) {
		xfer->c_flags |= C_POLL;
	}
#if NATA_DMA
	if ((drvp->drive_flags & (DRIVE_DMA | DRIVE_UDMA)) && (ata_bio->flags & ATA_SINGLE) == 0) {
		xfer->c_flags |= C_DMA;
	}
#endif
	xfer->c_drive = drvp->drive;
	xfer->c_cmd = ata_bio;
	xfer->c_databuf = ata_bio->databuf;
	xfer->c_bcount = ata_bio->bcount;
	xfer->c_start = wdc_ata_bio_start;
	xfer->c_intr = wdc_ata_bio_intr;
	xfer->c_kill_xfer = wdc_ata_bio_kill_xfer;
	xfer->uio = ata_bio->uio;
//************************************************************
	chp->ch_flags |= ATACH_TH_RUN;
//************************************************************
	ata_exec_xfer(chp, xfer);
	return (ata_bio->flags & ATA_ITSDONE) ? ATACMD_COMPLETE : ATACMD_QUEUED;
}

static int wdc_ata_addref(
	struct ata_drive_datas *drvp)
{
	struct ata_channel *chp = drvp->chnl_softc;

	return (ata_addref(chp));
}

static void wdc_ata_delref(
	struct ata_drive_datas *drvp)
{
	struct ata_channel *chp = drvp->chnl_softc;

	ata_delref(chp);
}

const struct ata_bustype wdc_ata_bustype = {
	SCSIPI_BUSTYPE_ATA,
	wdc_ata_bio,
	wdc_reset_drive,
	wdc_reset_channel,
	wdc_exec_command,
	ata_get_params,
	wdc_ata_addref,
	wdc_ata_delref,
	ata_kill_pending,
};

//===================================== PUBLIC =====================================================

