/*	$NetBSD: wdc.c,v 1.243.2.3 2007/01/29 22:25:16 bouyer Exp $ */

/*
 * Copyright (c) 1998, 2001, 2003 Manuel Bouyer.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by Manuel Bouyer.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1998, 2003, 2004 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Charles M. Hannum, by Onno van der Linden and by Manuel Bouyer.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * 2009: modified by minoru murashima.
 */

/*
 * CODE UNTESTED IN THE CURRENT REVISION:
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/queue.h>
#include <sys/buf.h>
#include <sys/lock.h>
#include <sys/callout.h>
#include <machine/cpu.h>
#include <machine/bus.h>
#include <stddef.h>
#include <stdlib.h>
#include <ata/atavar.h>
#include <ata/atareg.h>
#include <ic/wdcreg.h>
#include <ic/wdcvar.h>

//=====================================  ===================================================

/*
 * timeout when waiting for BSY to deassert when probing.
 * set to 5s. From the standards this could be up to 31, but we can't
 * wait that much at boot time, and 5s seems to be enouth.
 */
#define WDC_PROBE_WAIT 5

#define WDCDELAY  100 /* 100 microseconds */
#define WDCNDELAY_RST (WDC_RESET_WAIT * 1000 / WDCDELAY)

/* When polling wait that much and then tsleep for 1/hz seconds */
#define WDCDELAY_POLL 1 /* ms */

/* Flags to wdcreset(). */
#define	RESET_POLL	1
#define	RESET_SLEEP	0	/* wdcreset() will use tsleep() */

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

//#if NWD > 0
extern const struct ata_bustype wdc_ata_bustype; /* in ata_wdc.c */
//#else
/* A fake one, the autoconfig will print "wd at foo ... not configured */
//const struct ata_bustype wdc_ata_bustype = {
//	SCSIPI_BUSTYPE_ATA,
//	NULL,				/* wdc_ata_bio */
//	NULL,				/* wdc_reset_drive */
//	wdc_reset_channel,
//	wdc_exec_command,
//	NULL,				/* ata_get_params */
//	NULL,				/* wdc_ata_addref */
//	NULL,				/* wdc_ata_delref */
//	NULL				/* ata_kill_pending */
//};
//#endif

static void __wdcerror(
	struct ata_channel *chp, 
	const char *msg)
{
	struct atac_softc *atac = chp->ch_atac;
	struct ata_xfer *xfer = chp->ch_queue->active_xfer;

	if (xfer == NULL) {
		printf("%s:%d: %s\n", atac->atac_dev.dv_xname, chp->ch_channel, msg);
	}
	else {
		printf("%s:%d:%d: %s\n", atac->atac_dev.dv_xname, chp->ch_channel, xfer->c_drive, msg);
	}
}

static void __wdccommand_done_end(
	struct ata_channel *chp, 
	struct ata_xfer *xfer)
{
	struct ata_command *ata_c = xfer->c_cmd;

	ata_c->flags |= AT_DONE;
	ata_free_xfer(chp, xfer);
	if (ata_c->flags & AT_WAIT) {
		wakeup(ata_c);
	}
	else if (ata_c->callback) {
		ata_c->callback(ata_c->callback_arg);
	}
	atastart(chp);
	return;
}

static void __wdccommand_kill_xfer(
	struct ata_channel *chp, 
	struct ata_xfer *xfer,
    int reason)
{
	struct ata_command *ata_c = xfer->c_cmd;

	switch (reason) {
	case KILL_GONE:
		ata_c->flags |= AT_GONE;
		break;
	case KILL_RESET:
		ata_c->flags |= AT_RESET;
		break;
	default:
		printf("__wdccommand_kill_xfer: unknown reason %d\n", reason);
		panic("__wdccommand_kill_xfer");
	}
	__wdccommand_done_end(chp, xfer);
}

static void __wdccommand_done(
	struct ata_channel *chp, 
	struct ata_xfer *xfer)
{
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	struct ata_command *ata_c = xfer->c_cmd;

	if (chp->ch_status & WDCS_DWF) {
		ata_c->flags |= AT_DF;
	}
	if (chp->ch_status & WDCS_ERR) {
		ata_c->flags |= AT_ERROR;
		ata_c->r_error = chp->ch_error;
	}
	if ((ata_c->flags & AT_READREG) != 0 && device_is_active(&atac->atac_dev) && (ata_c->flags & (AT_ERROR | AT_DF)) == 0) {
		ata_c->r_head = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0);
		ata_c->r_count = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_seccnt], 0);
		ata_c->r_sector = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0);
		ata_c->r_cyl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_hi], 0) << 8;
		ata_c->r_cyl |= bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
		ata_c->r_error = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_error], 0);
		ata_c->r_features = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_features], 0);
	}
	callout_stop(&chp->ch_callout);
	chp->ch_queue->active_xfer = NULL;
	if (ata_c->flags & AT_POLL) {
		/* enable interrupts */
		bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT);
		delay(10); /* some drives need a little delay here */
	}
	if (chp->ch_drive[xfer->c_drive].drive_flags & DRIVE_WAITDRAIN) {
		__wdccommand_kill_xfer(chp, xfer, KILL_GONE);
		chp->ch_drive[xfer->c_drive].drive_flags &= ~DRIVE_WAITDRAIN;
		wakeup(&chp->ch_queue->active_xfer);
	} 
	else {
		__wdccommand_done_end(chp, xfer);
	}
}

static int __wdccommand_intr(
	struct ata_channel *chp, 
	struct ata_xfer *xfer, 
	int irq)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	struct ata_command *ata_c = xfer->c_cmd;
	int bcount = ata_c->bcount;
	char *data = ata_c->data;
	int wflags;
	int drive_flags;

	if (ata_c->r_command == WDCC_IDENTIFY || ata_c->r_command == ATAPI_IDENTIFY_DEVICE) {
		/*
		 * The IDENTIFY data has been designed as an array of
		 * u_int16_t, so we can byteswap it on the fly.
		 * Historically it's what we have always done so keeping it
		 * here ensure binary backward compatibility.
		 */
		 drive_flags = DRIVE_NOSTREAM |  chp->ch_drive[xfer->c_drive].drive_flags;
	} 
	else {
		/*
		 * Other data structure are opaque and should be transfered
		 * as is.
		 */
		drive_flags = chp->ch_drive[xfer->c_drive].drive_flags;
	}

	if ((ata_c->flags & (AT_WAIT | AT_POLL)) == (AT_WAIT | AT_POLL)) {
		/* both wait and poll, we can tsleep here */
		wflags = AT_WAIT | AT_POLL;
	} 
	else {
		wflags = AT_POLL;
	}

 again:
	/*
	 * after a ATAPI_SOFT_RESET, the device will have released the bus.
	 * Reselect again, it doesn't hurt for others commands, and the time
	 * penalty for the extra regiter write is acceptable,
	 * wdc_exec_command() isn't called often (mosly for autoconfig)
	 */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (xfer->c_drive << 4));
	if ((ata_c->flags & AT_XFDONE) != 0) {
		/*
		 * We have completed a data xfer. The drive should now be
		 * in its initial state
		 */
		if (wdcwait(chp, ata_c->r_st_bmask | WDCS_DRQ, ata_c->r_st_bmask, (irq == 0)  ? ata_c->timeout : 0, wflags) ==  WDCWAIT_TOUT) {
			if (irq && (xfer->c_flags & C_TIMEOU) == 0) {
				return 0; /* IRQ was not for us */
			}
			ata_c->flags |= AT_TIMEOU;
		}
		goto out;
	}
	if (wdcwait(chp, ata_c->r_st_pmask, ata_c->r_st_pmask, (irq == 0)  ? ata_c->timeout : 0, wflags) == WDCWAIT_TOUT) {
		if (irq && (xfer->c_flags & C_TIMEOU) == 0) {
			return 0; /* IRQ was not for us */
		}
		ata_c->flags |= AT_TIMEOU;
		goto out;
	}
	if (wdc->irqack) {
		wdc->irqack(chp);
	}
	if (ata_c->flags & AT_READ) {
		if ((chp->ch_status & WDCS_DRQ) == 0) {
			ata_c->flags |= AT_TIMEOU;
			goto out;
		}
		wdc->datain_pio(chp, drive_flags, data, bcount);

		/* at this point the drive should be in its initial state */
		ata_c->flags |= AT_XFDONE;
		/*
		 * XXX checking the status register again here cause some
		 * hardware to timeout.
		 */
	} 
	else if (ata_c->flags & AT_WRITE) {
		if ((chp->ch_status & WDCS_DRQ) == 0) {
			ata_c->flags |= AT_TIMEOU;
			goto out;
		}
		wdc->dataout_pio(chp, drive_flags, data, bcount);
		ata_c->flags |= AT_XFDONE;
		if ((ata_c->flags & AT_POLL) == 0) {
			chp->ch_flags |= ATACH_IRQ_WAIT; /* wait for interrupt */
			callout_reset(&chp->ch_callout, mstohz(ata_c->timeout), wdctimeout, chp);
			return 1;
		} 
		else {
			goto again;
		}
	}
 out:
	__wdccommand_done(chp, xfer);
	return 1;
}

static void __wdccommand_start(
	struct ata_channel *chp, 
	struct ata_xfer *xfer)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	int drive = xfer->c_drive;
	int wait_flags = (xfer->c_flags & C_POLL) ? AT_POLL : 0;
	struct ata_command *ata_c = xfer->c_cmd;

	if (wdc->select) {
		wdc->select(chp, drive);
	}
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (drive << 4));
	switch(wdcwait(chp, ata_c->r_st_bmask | WDCS_DRQ, ata_c->r_st_bmask, ata_c->timeout, wait_flags)) {
	case WDCWAIT_OK:
		break;
	case WDCWAIT_TOUT:
		ata_c->flags |= AT_TIMEOU;
		__wdccommand_done(chp, xfer);
		return;
	case WDCWAIT_THR:
		return;
	}
	if (ata_c->flags & AT_POLL) {
		/* polled command, disable interrupts */
		bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT | WDCTL_IDS);
	}
	wdccommand(chp, drive, ata_c->r_command, ata_c->r_cyl, ata_c->r_head, ata_c->r_sector, ata_c->r_count, ata_c->r_features);

	if ((ata_c->flags & AT_POLL) == 0) {
		chp->ch_flags |= ATACH_IRQ_WAIT; /* wait for interrupt */
		callout_reset(&chp->ch_callout, ata_c->timeout / 1000 * hz, wdctimeout, chp);
		return;
	}
	/*
	 * Polled command. Wait for drive ready or drq. Done in intr().
	 * Wait for at last 400ns for status bit to be valid.
	 */
	delay(10);	/* 400ns delay */
	__wdccommand_intr(chp, xfer, 0);
}

/*
 * Wait for a drive to be !BSY, and have mask in its status register.
 * return -1 for a timeout after "timeout" ms.
 */
static int __wdcwait(
	struct ata_channel *chp, 
	int mask, 
	int bits, 
	int timeout)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	u_char status;
	int xtime = 0;

	chp->ch_error = 0;

	timeout = timeout * 1000 / WDCDELAY; /* delay uses microseconds */

	for (;;) {
		chp->ch_status = status = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
		if ((status & (WDCS_BSY | mask)) == bits) {
			break;
		}
		if (timeout < ++xtime) {
			return(WDCWAIT_TOUT);
		}
		delay(WDCDELAY);
	}
	if (status & WDCS_ERR) {
		chp->ch_error = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_error], 0);
	}
	return(WDCWAIT_OK);
}

static int __wdcwait_reset(
	struct ata_channel *chp, 
	int drv_mask, 
	int poll)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	int timeout, nloop;
	u_int8_t st0 = 0, st1 = 0;

	if (poll) {
		nloop = WDCNDELAY_RST;
	}
	else {
		nloop = WDC_RESET_WAIT * hz / 1000;
	}
	/* wait for BSY to deassert */
	for (timeout = 0; timeout < nloop; timeout++) {
		if ((drv_mask & 0x01) != 0) {
			if (wdc->select) {
				wdc->select(chp,0);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM); /* master */
			delay(10);
			st0 = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
		}
		if ((drv_mask & 0x02) != 0) {
			if (wdc->select) {
				wdc->select(chp,1);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | 0x10); /* slave */
			delay(10);
			st1 = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
		}

		if ((drv_mask & 0x01) == 0) {
			/* no master */
			if ((drv_mask & 0x02) != 0 && (st1 & WDCS_BSY) == 0) {
				/* No master, slave is ready, it's done */
				goto end;
			}
			if ((drv_mask & 0x02) == 0) {
				/* No master, no slave: it's done */
				goto end;
			}
		} 
		else if ((drv_mask & 0x02) == 0) {
			/* no slave */
			if ((drv_mask & 0x01) != 0 && (st0 & WDCS_BSY) == 0) {
				/* No slave, master is ready, it's done */
				goto end;
			}
		} 
		else {
			/* Wait for both master and slave to be ready */
			if ((st0 & WDCS_BSY) == 0 && (st1 & WDCS_BSY) == 0) {
				goto end;
			}
		}
		if (poll) {
			delay(WDCDELAY);
		}
		else {
			tsleep(&nloop, PRIBIO, "atarst", 1);
		}
	}
	/* Reset timed out. Maybe it's because drv_mask was not right */
	if (st0 & WDCS_BSY) {
		drv_mask &= ~0x01;
	}
	if (st1 & WDCS_BSY) {
		drv_mask &= ~0x02;
	}
end:

	return drv_mask;
}

static int wdcreset(
	struct ata_channel *chp, 
	int poll)
{
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	int drv_mask1, drv_mask2;

	wdc->reset(chp, poll);

	drv_mask1 = (chp->ch_drive[0].drive_flags & DRIVE) ? 0x01:0x00;
	drv_mask1 |= (chp->ch_drive[1].drive_flags & DRIVE) ? 0x02:0x00;
	drv_mask2 = __wdcwait_reset(chp, drv_mask1, (poll == RESET_SLEEP) ? 0 : 1);
	if (drv_mask2 != drv_mask1) {
		printf("%s channel %d: reset failed for", atac->atac_dev.dv_xname, chp->ch_channel);
		if ((drv_mask1 & 0x01) != 0 && (drv_mask2 & 0x01) == 0) {
			printf(" drive 0");
		}
		if ((drv_mask1 & 0x02) != 0 && (drv_mask2 & 0x02) == 0) {
			printf(" drive 1");
		}
		printf("\n");
	}
	bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT);
	return  (drv_mask1 != drv_mask2) ? 1 : 0;
}

/* Put all disk in RESET state */
void wdc_reset_drive(
	struct ata_drive_datas *drvp, 
	int flags)
{
	struct ata_channel *chp = drvp->chnl_softc;
//	struct atac_softc *atac = chp->ch_atac;

	ata_reset_channel(chp, flags);
}

static int wdcprobe1(
	struct ata_channel *chp, 
	int poll)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	u_int8_t st0 = 0, st1 = 0, sc, sn, cl, ch;
	u_int8_t ret_value = 0x03;
	u_int8_t drive;
	int s;
	int wdc_probe_count = poll ? (WDC_PROBE_WAIT / WDCDELAY) : (WDC_PROBE_WAIT * hz);

	/*
	 * Sanity check to see if the wdc channel responds at all.
	 */

//	s = splbio();
	if ((wdc->cap & WDC_CAPABILITY_NO_EXTRA_RESETS) == 0) {
		while (wdc_probe_count-- > 0) {
			if (wdc->select) {
				wdc->select(chp,0);
			}

			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM);
			delay(10);	/* 400ns delay */
			st0 = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);

			if (wdc->select) {
				wdc->select(chp,1);
			}

			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | 0x10);
			delay(10);	/* 400ns delay */
			st1 = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
			if ((st0 & WDCS_BSY) == 0) {
				break;
			}
		}

		if (st0 == 0xff || st0 == WDSD_IBM) {
			ret_value &= ~0x01;
		}
		if (st1 == 0xff || st1 == (WDSD_IBM | 0x10)) {
			ret_value &= ~0x02;
		}
		/* Register writability test, drive 0. */
		if (ret_value & 0x01) {
			if (wdc->select) {
				wdc->select(chp,0);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM);
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0, 0x02);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
			if (cl != 0x02) {
				ret_value &= ~0x01;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0, 0x01);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
			if (cl != 0x01) {
				ret_value &= ~0x01;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0, 0x01);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0);
			if (cl != 0x01) {
				ret_value &= ~0x01;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0, 0x02);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0);
			if (cl != 0x02) {
				ret_value &= ~0x01;
			}
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
			if (cl != 0x01) {
				ret_value &= ~0x01;
			}
		}
		/* Register writability test, drive 1. */
		if (ret_value & 0x02) {
			if (wdc->select) {
			     wdc->select(chp,1);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | 0x10);
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0, 0x02);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
			if (cl != 0x02) {
				ret_value &= ~0x02;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0, 0x01);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
			if (cl != 0x01) {
				ret_value &= ~0x02;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0, 0x01);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0);
			if (cl != 0x01) {
				ret_value &= ~0x02;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0, 0x02);
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0);
			if (cl != 0x02) {
				ret_value &= ~0x02;
			}
			cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
			if (cl != 0x01) {
				ret_value &= ~0x02;
			}
		}

		if (ret_value == 0) {
//			splx(s);
			return 0;
		}
	}

	wdc->reset(chp, RESET_POLL);
	DELAY(2000);
	(void) bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_error], 0);
	bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT);
//	splx(s);

	ret_value = __wdcwait_reset(chp, ret_value, poll);

	/* if reset failed, there's nothing here */
	if (ret_value == 0) {
		return 0;
	}

	/*
	 * Test presence of drives. First test register signatures looking
	 * for ATAPI devices. If it's not an ATAPI and reset said there may
	 * be something here assume it's ATA or OLD.  Ghost will be killed
	 * later in attach routine.
	 */
	for (drive = 0; drive < chp->ch_ndrive; drive++) {
		if ((ret_value & (0x01 << drive)) == 0) {
			continue;
		}
		if (wdc->select) {
			wdc->select(chp,drive);
		}
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (drive << 4));
		delay(10);	/* 400ns delay */
		/* Save registers contents */
		sc = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_seccnt], 0);
		sn = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0);
		cl = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0);
		ch = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_hi], 0);

		/*
		 * sc & sn are supposted to be 0x1 for ATAPI but in some cases
		 * we get wrong values here, so ignore it.
		 */
		s = splbio();
		if (cl == 0x14 && ch == 0xeb) {
			chp->ch_drive[drive].drive_flags |= DRIVE_ATAPI;
		} 
		else {
			chp->ch_drive[drive].drive_flags |= DRIVE_ATA;
			if ((wdc->cap & WDC_CAPABILITY_PREATA) != 0) {
				chp->ch_drive[drive].drive_flags |= DRIVE_OLD;
			}
		}
		splx(s);
	}
	return (ret_value);
}

static void wdc_datain_pio(
	struct ata_channel *chp, 
	int flags, 
	void *bf, 
	size_t len)
{
	struct wdc_regs *wdr = CHAN_TO_WDC_REGS(chp);

#ifndef __NO_STRICT_ALIGNMENT
	if ((uintptr_t)bf & 1) {
		goto unaligned;
	}
	if ((flags & DRIVE_CAP32) && ((uintptr_t)bf & 3)) {
		goto unaligned;
	}
#endif

	if (flags & DRIVE_NOSTREAM) {
		if (flags & DRIVE_CAP32) {
			bus_space_read_multi_4(wdr->data32iot, wdr->data32ioh, 0, bf, len >> 2);
			bf = (char *)bf + (len & ~3);
			len &= 3;
		}
		if (len) {
			bus_space_read_multi_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0, bf, len >> 1);
		}
	} 
	else {
		if (flags & DRIVE_CAP32) {
			bus_space_read_multi_stream_4(wdr->data32iot, wdr->data32ioh, 0, bf, len >> 2);
			bf = (char *)bf + (len & ~3);
			len &= 3;
		}
		if (len) {
			bus_space_read_multi_stream_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0, bf, len >> 1);
		}
	}
	return;

#ifndef __NO_STRICT_ALIGNMENT
unaligned:
	if (flags & DRIVE_NOSTREAM) {
		if (flags & DRIVE_CAP32) {
			while (3 < len) {
				uint32_t val;

				val = bus_space_read_4(wdr->data32iot, wdr->data32ioh, 0);
				memcpy(bf, &val, 4);
				bf = (char *)bf + 4;
				len -= 4;
			}
		}
		while (1 < len) {
			uint16_t val;

			val = bus_space_read_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0);
			memcpy(bf, &val, 2);
			bf = (char *)bf + 2;
			len -= 2;
		}
	} 
	else {
		if (flags & DRIVE_CAP32) {
			while (3 < len) {
				uint32_t val;

				val = bus_space_read_stream_4(wdr->data32iot, wdr->data32ioh, 0);
				memcpy(bf, &val, 4);
				bf = (char *)bf + 4;
				len -= 4;
			}
		}
		while (1 < len) {
			uint16_t val;

			val = bus_space_read_stream_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0);
			memcpy(bf, &val, 2);
			bf = (char *)bf + 2;
			len -= 2;
		}
	}
#endif
}

static void wdc_dataout_pio(
	struct ata_channel *chp, 
	int flags, 
	void *bf, 
	size_t len)
{
	struct wdc_regs *wdr = CHAN_TO_WDC_REGS(chp);

#ifndef __NO_STRICT_ALIGNMENT
	if ((uintptr_t)bf & 1) {
		goto unaligned;
	}
	if ((flags & DRIVE_CAP32) && ((uintptr_t)bf & 3)) {
		goto unaligned;
	}
#endif

	if (flags & DRIVE_NOSTREAM) {
		if (flags & DRIVE_CAP32) {
			bus_space_write_multi_4(wdr->data32iot, wdr->data32ioh, 0, bf, len >> 2);
			bf = (char *)bf + (len & ~3);
			len &= 3;
		}
		if (len) {
			bus_space_write_multi_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0, bf, len >> 1);
		}
	} 
	else {
		if (flags & DRIVE_CAP32) {
			bus_space_write_multi_stream_4(wdr->data32iot, wdr->data32ioh, 0, bf, len >> 2);
			bf = (char *)bf + (len & ~3);
			len &= 3;
		}
		if (len) {
			bus_space_write_multi_stream_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0, bf, len >> 1);
		}
	}
	return;

#ifndef __NO_STRICT_ALIGNMENT
unaligned:
	if (flags & DRIVE_NOSTREAM) {
		if (flags & DRIVE_CAP32) {
			while (3 < len) {
				uint32_t val;

				memcpy(&val, bf, 4);
				bus_space_write_4(wdr->data32iot, wdr->data32ioh, 0, val);
				bf = (char *)bf + 4;
				len -= 4;
			}
		}
		while (1 < len) {
			uint16_t val;

			memcpy(&val, bf, 2);
			bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0, val);
			bf = (char *)bf + 2;
			len -= 2;
		}
	} 
	else {
		if (flags & DRIVE_CAP32) {
			while (3 < len) {
				uint32_t val;

				memcpy(&val, bf, 4);
				bus_space_write_stream_4(wdr->data32iot, wdr->data32ioh, 0, val);
				bf = (char *)bf + 4;
				len -= 4;
			}
		}
		while (1 < len) {
			uint16_t val;

			memcpy(&val, bf, 2);
			bus_space_write_stream_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0, val);
			bf = (char *)bf + 2;
			len -= 2;
		}
	}
#endif
}

//===================================== PUBLIC =====================================================

void wdctimeout(
	void *arg)
{
	struct ata_channel *chp = (struct ata_channel *)arg;
#if NATA_DMA || NATA_PIOBM
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
#endif
	struct ata_xfer *xfer = chp->ch_queue->active_xfer;
	int s;

	s = splbio();
	if ((chp->ch_flags & ATACH_IRQ_WAIT) != 0) {
		__wdcerror(chp, "lost interrupt");
		printf("\ttype: %s tc_bcount: %d tc_skip: %d\n", (xfer->c_flags & C_ATAPI) ?  "atapi" : "ata", xfer->c_bcount, xfer->c_skip);
#if NATA_DMA || NATA_PIOBM
		if (chp->ch_flags & ATACH_DMA_WAIT) {
			wdc->dma_status = (*wdc->dma_finish)(wdc->dma_arg, chp->ch_channel, xfer->c_drive, WDC_DMAEND_ABRT);
			chp->ch_flags &= ~ATACH_DMA_WAIT;
		}
#endif
		/*
		 * Call the interrupt routine. If we just missed an interrupt,
		 * it will do what's needed. Else, it will take the needed
		 * action (reset the device).
		 * Before that we need to reinstall the timeout callback,
		 * in case it will miss another irq while in this transfer
		 * We arbitray chose it to be 1s
		 */
		callout_reset(&chp->ch_callout, hz, wdctimeout, chp);
		panic("callout_reset not support");
		xfer->c_flags |= C_TIMEOU;
		chp->ch_flags &= ~ATACH_IRQ_WAIT;
		xfer->c_intr(chp, xfer, 1);
	} 
	else {
		__wdcerror(chp, "missing untimeout");
	}
	splx(s);
}

/*
 * Send a command. The drive should be ready.
 * Assumes interrupts are blocked.
 */
void wdccommand(
	struct ata_channel *chp, 
	u_int8_t drive, 
	u_int8_t command,
    u_int16_t cylin, 
    u_int8_t head, 
    u_int8_t sector, 
    u_int8_t count,
    u_int8_t features)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];

	if (wdc->select) {
		wdc->select(chp,drive);
	}

	/* Select drive, head, and addressing mode. */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (drive << 4) | head);
	/* Load parameters into the wd_features register. */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_features], 0, features);
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_seccnt], 0, count);
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sector], 0, sector);
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0, cylin);
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_hi], 0, cylin >> 8);

	/* Send command. */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_command], 0, command);
	return;
}

int wdc_exec_command(
	struct ata_drive_datas *drvp, 
	struct ata_command *ata_c)
{
	struct ata_channel *chp = drvp->chnl_softc;
	struct atac_softc *atac = chp->ch_atac;
	struct ata_xfer *xfer;
	int s, ret;

	/* set up an xfer and queue. Wait for completion */
	xfer = ata_get_xfer(ata_c->flags & AT_WAIT ? ATAXF_CANSLEEP : ATAXF_NOSLEEP);
	if (xfer == NULL) {
		return ATACMD_TRY_AGAIN;
	 }

	if (atac->atac_cap & ATAC_CAP_NOIRQ) {
		ata_c->flags |= AT_POLL;
	}
	if (ata_c->flags & AT_POLL) {
		xfer->c_flags |= C_POLL;
	}
	if (ata_c->flags & AT_WAIT) {
		xfer->c_flags |= C_WAIT;
	}
	xfer->c_drive = drvp->drive;
	xfer->c_databuf = ata_c->data;
	xfer->c_bcount = ata_c->bcount;
	xfer->c_cmd = ata_c;
	xfer->c_start = __wdccommand_start;
	xfer->c_intr = __wdccommand_intr;
	xfer->c_kill_xfer = __wdccommand_kill_xfer;

	s = splbio();
	ata_exec_xfer(chp, xfer);
	if (ata_c->flags & AT_DONE) {
		ret = ATACMD_COMPLETE;
	} 
	else {
		if (ata_c->flags & AT_WAIT) {
			while ((ata_c->flags & AT_DONE) == 0) {
				tsleep(ata_c, PRIBIO, "wdccmd", 0);
			}
			ret = ATACMD_COMPLETE;
		} 
		else {
			ret = ATACMD_QUEUED;
		}
	}
	splx(s);
	return ret;
}

/*
 * Send a 48-bit addressing command. The drive should be ready.
 * Assumes interrupts are blocked.
 */
void wdccommandext(
	struct ata_channel *chp, 
	u_int8_t drive, 
	u_int8_t command,
    u_int64_t blkno, 
    u_int16_t count)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];

	if (wdc->select) {
		wdc->select(chp,drive);
	}

	/* Select drive, head, and addressing mode. */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, (drive << 4) | WDSD_LBA);

	if (wdc->cap & WDC_CAPABILITY_WIDEREGS) {
		bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_features], 0, 0);
		bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_seccnt], 0, count);
		bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_lo], 0, (((blkno >> 16) & 0xff00) | (blkno & 0x00ff)));
		bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_mi], 0, (((blkno >> 24) & 0xff00) | ((blkno >> 8) & 0x00ff)));
		bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_hi], 0, (((blkno >> 32) & 0xff00) | ((blkno >> 16) & 0x00ff)));
	} 
	else {
		/* previous */
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_features], 0, 0);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_seccnt], 0, count >> 8);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_lo], 0, blkno >> 24);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_mi], 0, blkno >> 32);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_hi], 0, blkno >> 40);

		/* current */
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_features], 0, 0);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_seccnt], 0, count);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_lo], 0, blkno);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_mi], 0, blkno >> 8);
		bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_lba_hi], 0, blkno >> 16);
	}

	/* Send command. */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_command], 0, command);
	return;
}

/*
 * Simplified version of wdccommand().  Unbusy/ready/drq must be
 * tested by the caller.
 */
void wdccommandshort(
	struct ata_channel *chp, 
	int drive, 
	int command)
{
//	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];

	if (wdc->select) {
		wdc->select(chp,drive);
	}

	/* Select drive. */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (drive << 4));

	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_command], 0, command);
}

void wdc_do_reset(
	struct ata_channel *chp, 
	int poll)
{
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	int s = 0;

	if (poll != RESET_SLEEP) {
		s = splbio();
	}
	if (wdc->select) {
		wdc->select(chp,0);
	}
	/* master */
	bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM);
//	delay(10);	/* 400ns delay */
	delay(100);	/* 400ns delay */
	/* assert SRST, wait for reset to complete */
	bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_RST | WDCTL_IDS | WDCTL_4BIT);
	delay(2000);
	(void) bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_error], 0);
	bus_space_write_1(wdr->ctl_iot, wdr->ctl_ioh, wd_aux_ctlr, WDCTL_4BIT | WDCTL_IDS);
	delay(10);	/* 400ns delay */
	if (poll != RESET_SLEEP) {
		/* ACK interrupt in case there is one pending left */
		if (wdc->irqack) {
			wdc->irqack(chp);
		}
		splx(s);
	}
}

void wdc_reset_channel(
	struct ata_channel *chp, 
	int flags)
{
	TAILQ_HEAD(, ata_xfer) reset_xfer;
	struct ata_xfer *xfer, *next_xfer;
#if NATA_DMA || NATA_PIOBM
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
#endif

	TAILQ_INIT(&reset_xfer);

	chp->ch_flags &= ~ATACH_IRQ_WAIT;

	/*
	 * if the current command if on an ATAPI device, issue a
	 * ATAPI_SOFT_RESET
	 */
	xfer = chp->ch_queue->active_xfer;
	if (xfer && xfer->c_chp == chp && (xfer->c_flags & C_ATAPI)) {
		wdccommandshort(chp, xfer->c_drive, ATAPI_SOFT_RESET);
		if (flags & AT_WAIT) {
			tsleep(&flags, PRIBIO, "atardl", mstohz(1) + 1);
		}
		else {
			delay(1000);
		}
	}

	/* reset the channel */
	if (flags & AT_WAIT) {
		(void) wdcreset(chp, RESET_SLEEP);
	}
	else {
		(void) wdcreset(chp, RESET_POLL);
	}

	/*
	 * wait a bit after reset; in case the DMA engines needs some time
	 * to recover.
	 */
	if (flags & AT_WAIT) {
		tsleep(&flags, PRIBIO, "atardl", mstohz(1) + 1);
	}
	else {
		delay(1000);
	}
	/*
	 * look for pending xfers. If we have a shared queue, we'll also reset
	 * the other channel if the current xfer is running on it.
	 * Then we'll dequeue only the xfers for this channel.
	 */
	if ((flags & AT_RST_NOCMD) == 0) {
		/*
		 * move all xfers queued for this channel to the reset queue,
		 * and then process the current xfer and then the reset queue.
		 * We have to use a temporary queue because c_kill_xfer()
		 * may requeue commands.
		 */
		for (xfer = TAILQ_FIRST(&chp->ch_queue->queue_xfer);
		    xfer != NULL; xfer = next_xfer) {
			next_xfer = TAILQ_NEXT(xfer, c_xferchain);
			if (xfer->c_chp != chp) {
				continue;
			}
			TAILQ_REMOVE(&chp->ch_queue->queue_xfer, xfer, c_xferchain);
			TAILQ_INSERT_TAIL(&reset_xfer, xfer, c_xferchain);
		}
		xfer = chp->ch_queue->active_xfer;
		if (xfer) {
			if (xfer->c_chp != chp) {
				ata_reset_channel(xfer->c_chp, flags);
			}
			else {
				callout_stop(&chp->ch_callout);
#if NATA_DMA || NATA_PIOBM
				/*
				 * If we're waiting for DMA, stop the
				 * DMA engine
				 */
				if (chp->ch_flags & ATACH_DMA_WAIT) {
					(*wdc->dma_finish)(wdc->dma_arg, chp->ch_channel, xfer->c_drive, WDC_DMAEND_ABRT_QUIET);
					chp->ch_flags &= ~ATACH_DMA_WAIT;
				}
#endif
				chp->ch_queue->active_xfer = NULL;
				if ((flags & AT_RST_EMERG) == 0) {
					xfer->c_kill_xfer( chp, xfer, KILL_RESET);
				}
			}
		}

		for (xfer = TAILQ_FIRST(&reset_xfer);
		    xfer != NULL; xfer = next_xfer) {
			next_xfer = TAILQ_NEXT(xfer, c_xferchain);
			TAILQ_REMOVE(&reset_xfer, xfer, c_xferchain);
			if ((flags & AT_RST_EMERG) == 0) {
				xfer->c_kill_xfer(chp, xfer, KILL_RESET);
			}
		}
	}
}

/*
 * Call __wdcwait(), polling using tsleep() or waking up the kernel
 * thread if possible
 */
int wdcwait(
	struct ata_channel *chp, 
	int mask, 
	int bits, 
	int timeout, 
	int flags)
{
	int error, i, timeout_hz = mstohz(timeout);

	if (timeout_hz == 0 || (flags & (AT_WAIT | AT_POLL)) == AT_POLL) {
		error = __wdcwait(chp, mask, bits, timeout);
	}
	else {
		error = __wdcwait(chp, mask, bits, WDCDELAY_POLL);
		if (error != 0) {
			if ((chp->ch_flags & ATACH_TH_RUN) ||
			    (flags & AT_WAIT)) {
				/*
				 * we're running in the channel thread
				 * or some userland thread context
				 */
				for (i = 0; i < timeout_hz; i++) {
					if (__wdcwait(chp, mask, bits, WDCDELAY_POLL) == 0) {
						error = 0;
						break;
					}
					tsleep(&chp, PRIBIO, "atapoll", 1);
				}
			} 
			else {
				/*
				 * we're probably in interrupt context,
				 * ask the thread to come back here
				 */
				chp->ch_queue->queue_freeze++;
				wakeup(&chp->ch_thread);
				return(WDCWAIT_THR);
			}
		}
	}
	return (error);
}

#if NATA_DMA
/*
 * Busy-wait for DMA to complete
 */
int wdc_dmawait(
	struct ata_channel *chp, 
	struct ata_xfer *xfer, 
	int timeout)
{
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	int xtime;

	for (xtime = 0;  xtime < timeout * 1000 / WDCDELAY; xtime++) {
		wdc->dma_status = (*wdc->dma_finish)(wdc->dma_arg, chp->ch_channel, xfer->c_drive, WDC_DMAEND_END);
		if ((wdc->dma_status & WDC_DMAST_NOIRQ) == 0) {
			return 0;
		}
		delay(WDCDELAY);
	}
	/* timeout, force a DMA halt */
	wdc->dma_status = (*wdc->dma_finish)(wdc->dma_arg, chp->ch_channel, xfer->c_drive, WDC_DMAEND_ABRT);
	return 1;
}
#endif

/*
 * Initialize the "shadow register" handles for a standard wdc controller.
 */
void wdc_init_shadow_regs(
	struct ata_channel *chp)
{
	struct wdc_regs *wdr = CHAN_TO_WDC_REGS(chp);

	wdr->cmd_iohs[wd_status] = wdr->cmd_iohs[wd_command];
	wdr->cmd_iohs[wd_features] = wdr->cmd_iohs[wd_error];
}

/*
 * Allocate a wdc_regs array, based on the number of channels.
 */
void wdc_allocate_regs(
	struct wdc_softc *wdc)
{
	wdc->regs = malloc(wdc->sc_atac.atac_nchannels * sizeof(struct wdc_regs));
}

/*
 * Interrupt routine for the controller.  Acknowledge the interrupt, check for
 * errors on the current operation, mark it done if necessary, and start the
 * next request.  Also check for a partially done transfer, and continue with
 * the next chunk if so.
 */
int wdcintr(
	void *arg)
{
	struct ata_channel *chp = arg;
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	struct ata_xfer *xfer;
	int ret;

	if (!device_is_active(&atac->atac_dev)) {
		return (0);
	}
	if ((chp->ch_flags & ATACH_IRQ_WAIT) == 0) {
		/* try to clear the pending interrupt anyway */
		(void)bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
		return (0);
	}
	xfer = chp->ch_queue->active_xfer;

#if NATA_DMA || NATA_PIOBM
	if (chp->ch_flags & ATACH_DMA_WAIT) {
		wdc->dma_status = (*wdc->dma_finish)(wdc->dma_arg, chp->ch_channel, xfer->c_drive, WDC_DMAEND_END);
		if (wdc->dma_status & WDC_DMAST_NOIRQ) {
			/* IRQ not for us, not detected by DMA engine */
			return 0;
		}
		chp->ch_flags &= ~ATACH_DMA_WAIT;
	}
#endif
	chp->ch_flags &= ~ATACH_IRQ_WAIT;
//****************************************************************************************
	if (chp->waitTask != NULL) {
		awakeIntr(chp->waitTask);
		return 1;
	}
//****************************************************************************************
	ret = xfer->c_intr(chp, xfer, 1);
	if (ret == 0) { /* irq was not for us, still waiting for irq */
		chp->ch_flags |= ATACH_IRQ_WAIT;
	}
	return (ret);
}

int wdcprobe(
	struct ata_channel *chp)
{
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	/* default reset method */
	if (wdc->reset == NULL) {
		wdc->reset = wdc_do_reset;
	}

	return (wdcprobe1(chp, 1));
}

/* Test to see controller with at last one attached drive is there.
 * Returns a bit for each possible drive found (0x01 for drive 0,
 * 0x02 for drive 1).
 * Logic:
 * - If a status register is at 0xff, assume there is no drive here
 *   (ISA has pull-up resistors).  Similarly if the status register has
 *   the value we last wrote to the bus (for IDE interfaces without pullups).
 *   If no drive at all -> return.
 * - reset the controller, wait for it to complete (may take up to 31s !).
 *   If timeout -> return.
 * - test ATA/ATAPI signatures. If at last one drive found -> return.
 * - try an ATA command on the master.
 */
void wdc_drvprobe(struct ata_channel *chp)
{
	struct ataparams params;
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);
	struct wdc_regs *wdr = &wdc->regs[chp->ch_channel];
	u_int8_t st0 = 0, st1 = 0;
	int i, j, error, s;

	if (wdcprobe1(chp, 0) == 0) {
		/* No drives, abort the attach here. */
		return;
	}

	/* for ATA/OLD drives, wait for DRDY, 3s timeout */
	for (i = 0; i < mstohz(3000); i++) {
		if (chp->ch_drive[0].drive_flags & (DRIVE_ATA|DRIVE_OLD)) {
			if (wdc->select) {
				wdc->select(chp,0);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM);
			delay(10);	/* 400ns delay */
			st0 = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
		}

		if (chp->ch_drive[1].drive_flags & (DRIVE_ATA|DRIVE_OLD)) {
			if (wdc->select) {
				wdc->select(chp,1);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | 0x10);
			delay(10);	/* 400ns delay */
			st1 = bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_status], 0);
		}

		if (((chp->ch_drive[0].drive_flags & (DRIVE_ATA|DRIVE_OLD)) == 0 ||
		    (st0 & WDCS_DRDY)) && ((chp->ch_drive[1].drive_flags & (DRIVE_ATA|DRIVE_OLD)) == 0 ||
		    (st1 & WDCS_DRDY))) {
			break;
		}
		tsleep(&params, PRIBIO, "atadrdy", 1);
	}
	s = splbio();
	if ((st0 & WDCS_DRDY) == 0) {
		chp->ch_drive[0].drive_flags &= ~(DRIVE_ATA|DRIVE_OLD);
	}
	if ((st1 & WDCS_DRDY) == 0) {
		chp->ch_drive[1].drive_flags &= ~(DRIVE_ATA|DRIVE_OLD);
	}
	splx(s);

	/* Wait a bit, some devices are weird just after a reset. */
	delay(5000);

	for (i = 0; i < chp->ch_ndrive; i++) {
		/* XXX This should be done by other code. */
		chp->ch_drive[i].chnl_softc = chp;
		chp->ch_drive[i].drive = i;

#if NATA_DMA
		/*
		 * Init error counter so that an error withing the first xfers
		 * will trigger a downgrade
		 */
		chp->ch_drive[i].n_dmaerrs = NERRS_MAX-1;
#endif

		/* If controller can't do 16bit flag the drives as 32bit */
		if ((atac->atac_cap & (ATAC_CAP_DATA16 | ATAC_CAP_DATA32)) == ATAC_CAP_DATA32) {
			s = splbio();
			chp->ch_drive[i].drive_flags |= DRIVE_CAP32;
			splx(s);
		}
		if ((chp->ch_drive[i].drive_flags & DRIVE) == 0) {
			continue;
		}

		/* Shortcut in case we've been shutdown */
		if (chp->ch_flags & ATACH_SHUTDOWN) {
			return;
		}

		/*
		 * Issue an identify, to try to detect ghosts.
		 * Note that we can't use interrupts here, because if there
		 * is no devices, we will get a command aborted without
		 * interrupts.
		 */
		error = ata_get_params(&chp->ch_drive[i], AT_WAIT | AT_POLL, &params);
		if (error != CMD_OK) {
			tsleep(&params, PRIBIO, "atacnf", mstohz(1000));

			/* Shortcut in case we've been shutdown */
			if (chp->ch_flags & ATACH_SHUTDOWN) {
				return;
			}

			error = ata_get_params(&chp->ch_drive[i], AT_WAIT | AT_POLL, &params);
		}
		if (error == CMD_OK) {
			/* If IDENTIFY succeeded, this is not an OLD ctrl */
			s = splbio();
			for (j = 0; j < chp->ch_ndrive; j++) {
				chp->ch_drive[j].drive_flags &= ~DRIVE_OLD;
			}
			splx(s);
		} 
		else {
			s = splbio();
			chp->ch_drive[i].drive_flags &= ~(DRIVE_ATA | DRIVE_ATAPI);
			splx(s);
			if ((chp->ch_drive[i].drive_flags & DRIVE_OLD) == 0) {
				continue;
			}

			/*
			 * Pre-ATA drive ?
			 * Test registers writability (Error register not
			 * writable, but cyllo is), then try an ATA command.
			 */
			if (wdc->select) {
				wdc->select(chp,i);
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_sdh], 0, WDSD_IBM | (i << 4));
			delay(10);	/* 400ns delay */
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_error], 0, 0x58);
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0, 0xa5);
			if (bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_error], 0) == 0x58 ||
			    bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_cyl_lo], 0) != 0xa5) {
				    s = splbio();
				    chp->ch_drive[i].drive_flags &= ~DRIVE_OLD;
				    splx(s);
				    continue;
			}
			if (wdc_wait_for_ready(chp, 10000, 0) == WDCWAIT_TOUT) {
				s = splbio();
				chp->ch_drive[i].drive_flags &= ~DRIVE_OLD;
				splx(s);
				continue;
			}
			bus_space_write_1(wdr->cmd_iot, wdr->cmd_iohs[wd_command], 0, WDCC_RECAL);
			delay(10);	/* 400ns delay */
			if (wdc_wait_for_ready(chp, 10000, 0) == WDCWAIT_TOUT) {
				s = splbio();
				chp->ch_drive[i].drive_flags &= ~DRIVE_OLD;
				splx(s);
			} 
			else {
				s = splbio();
				for (j = 0; j < chp->ch_ndrive; j++) {
					chp->ch_drive[j].drive_flags &= ~(DRIVE_ATA | DRIVE_ATAPI);
				}
				splx(s);
			}
		}
	}
}

void wdcattach(
	struct ata_channel *chp)
{
	struct atac_softc *atac = chp->ch_atac;
	struct wdc_softc *wdc = CHAN_TO_WDC(chp);

	KASSERT(chp->ch_ndrive > 0 && chp->ch_ndrive < 3);

	/* default data transfer methods */
	if (wdc->datain_pio == NULL) {
		wdc->datain_pio = wdc_datain_pio;
	}
	if (wdc->dataout_pio == NULL) {
		wdc->dataout_pio = wdc_dataout_pio;
	}
	/* default reset method */
	if (wdc->reset == NULL) {
		wdc->reset = wdc_do_reset;
	}

	/* initialise global data */
	if (atac->atac_bustype_ata == NULL) {
		atac->atac_bustype_ata = &wdc_ata_bustype;
	}
	if (atac->atac_probe == NULL) {
		atac->atac_probe = wdc_drvprobe;
	}
#if NATAPIBUS > 0
	if (atac->atac_atapibus_attach == NULL) {
		atac->atac_atapibus_attach = wdc_atapibus_attach;
	}
#endif

	ata_channel_attach(chp);
}

/*
 * the bit bucket
 */
void wdcbit_bucket(
	struct ata_channel *chp, 
	int size)
{
	struct wdc_regs *wdr = CHAN_TO_WDC_REGS(chp);

	for (; 2 <= size; size -= 2) {
		(void)bus_space_read_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0);
	}
	if (size) {
		(void)bus_space_read_1(wdr->cmd_iot, wdr->cmd_iohs[wd_data], 0);
	}
}

