/* $NetBSD: kern_auth.c,v 1.32.2.4 2007/01/07 10:51:15 bouyer Exp $ */

/*-
 * Copyright (c) 2005, 2006 Elad Efrat <elad@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Elad Efrat.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * 2008: modified by minoru murashima.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/queue.h>
#include <sys/lock.h>
#include <stddef.h>
#include <sys/kauth.h>

//=====================================  ===================================================

/* 
 * Credentials.
 */
struct kauth_cred {
	struct simplelock cr_lock;	/* lock on cr_refcnt */
	u_int cr_refcnt;		/* reference count */
	uid_t cr_uid;			/* user id */
	uid_t cr_euid;			/* effective user id */
	uid_t cr_svuid;			/* saved effective user id */
	gid_t cr_gid;			/* group id */
	gid_t cr_egid;			/* effective group id */
	gid_t cr_svgid;			/* saved effective group id */
	u_int cr_ngroups;		/* number of groups */
	gid_t cr_groups[NGROUPS];	/* group memberships */
};

/*
 * Listener.
 */
struct kauth_listener {
	kauth_scope_callback_t			func;			/* callback */
	kauth_scope_t					scope;			/* scope backpointer */
	u_int							refcnt;			/* reference count */
	SIMPLEQ_ENTRY(kauth_listener)	listener_next;	/* listener list */
};

/*
 * Scope.
 */
struct kauth_scope {
	const char						*id;			/* scope name */
	void							*cookie;		/* user cookie */
	u_int							nlisteners;		/* # of listeners */
	SIMPLEQ_HEAD(, kauth_listener)	listenq;		/* listener list */
	SIMPLEQ_ENTRY(kauth_scope)		next_scope;		/* scope list */
};

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static boolean_t listeners_have_been_loaded = FALSE;
static kauth_scope_t kauth_builtin_scope_device;

//===================================== PUBLIC =====================================================

/*
 * Authorize a request.
 *
 * scope - the scope of the request as defined by KAUTH_SCOPE_* or as
 *	   returned from kauth_register_scope().
 * credential - credentials of the user ("actor") making the request.
 * action - request identifier.
 * arg[0-3] - passed unmodified to listener(s).
 */
int kauth_authorize_action(
	kauth_scope_t scope, 
	kauth_cred_t cred,
	kauth_action_t action, 
	void *arg0, 
	void *arg1,
	void *arg2, 
	void *arg3)
{
	kauth_listener_t listener;
	int error, allow, fail;

	KASSERT(cred != NULL);
	KASSERT(action != 0);

	/* Short-circuit requests coming from the kernel. */
	if (cred == NOCRED || cred == FSCRED) {
		return (0);
	}

	KASSERT(scope != NULL);

	if (!listeners_have_been_loaded) {
		KASSERT(SIMPLEQ_EMPTY(&scope->listenq));
		return (0);
	}

	fail = 0;
	allow = 0;
	SIMPLEQ_FOREACH(listener, &scope->listenq, listener_next) {
		error = listener->func(cred, action, scope->cookie, arg0, arg1, arg2, arg3);

		if (error == KAUTH_RESULT_ALLOW) {
			allow = 1;
		}
		else if (error == KAUTH_RESULT_DENY) {
			fail = 1;
		}
	}

	return ((allow && !fail) ? 0 : EPERM);
};

int kauth_authorize_device_tty(
	kauth_cred_t cred, 
	kauth_action_t action,
    struct tty *tty)
{
	return (kauth_authorize_action(kauth_builtin_scope_device, cred, action, tty, NULL, NULL, NULL));
}

uid_t kauth_cred_geteuid(
	kauth_cred_t cred)
{
	KASSERT(cred != NULL);

	return (cred->cr_euid);
}

