/*	$NetBSD: kern_lock.c,v 1.102.2.2 2007/08/01 14:45:46 liamjfoy Exp $	*/

/*-
 * Copyright (c) 1999, 2000 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe of the Numerical Aerospace Simulation Facility,
 * NASA Ames Research Center.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Ross Harvey.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the NetBSD
 *	Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1995
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code contains ideas from software contributed to Berkeley by
 * Avadis Tevanian, Jr., Michael Wayne Young, and the Mach Operating
 * System project at Carnegie-Mellon University.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)kern_lock.c	8.18 (Berkeley) 5/21/95
 *
 * 2009: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/lock.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <dev/lockstat.h>

//=====================================  ===================================================

#define	INTERLOCK_ACQUIRE(lkp, flags, s)				\
do {													\
	simple_lock(&(lkp)->lk_interlock);					\
} while (/*CONSTCOND*/ 0)

#define	WEHOLDIT(lkp, pid, lid, cpu_id)					\
	(((lkp)->lk_flags & LK_SPIN) != 0 ?					\
	 ((lkp)->lk_cpu == (cpu_id)) :						\
	 ((lkp)->lk_lockholder == (pid) && (lkp)->lk_locklwp == (lid)))

#define	WAKEUP_WAITER(lkp)										\
do {															\
	if (((lkp)->lk_flags & (LK_SPIN | LK_WAIT_NONZERO)) == LK_WAIT_NONZERO) {	\
		wakeup((lkp));											\
	}															\
} while (/*CONSTCOND*/0)

#define	INTERLOCK_RELEASE(lkp, flags, s)						\
do {															\
	simple_unlock(&(lkp)->lk_interlock);						\
} while (/*CONSTCOND*/ 0)

#define	SETHOLDER(lkp, pid, lid, cpu_id)				\
do {													\
	if ((lkp)->lk_flags & LK_SPIN)						\
		(lkp)->lk_cpu = cpu_id;							\
	else {												\
		(lkp)->lk_lockholder = pid;						\
		(lkp)->lk_locklwp = lid;						\
	}													\
} while (/*CONSTCOND*/0)

#define COUNT(lkp, p, cpu_id, x)
#define COUNT_CPU(cpu_id, x)
#define	DONTHAVEIT(lkp)					/* nothing */
#define	HAVEIT(lkp)						/* nothing */
#define	SPINLOCK_SPINCHECK_DECL			/* nothing */
#define	SPINLOCK_SPINCHECK				/* nothing */

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

/*
 * Acquire a resource.
 */
static int acquire(
	volatile struct lock **lkpp, 
	int *s, 
	int extflags,
    int drain, 
    int wanted, 
    uintptr_t ra)
{
	int error;
	volatile struct lock *lkp = *lkpp;
	LOCKSTAT_TIMER(slptime);

	KASSERT(drain || (wanted & LK_WAIT_NONZERO) == 0);

	if (extflags & LK_SPIN) {
		int interlocked;

		SPINLOCK_SPINCHECK_DECL;

		if (!drain) {
			lkp->lk_waitcount++;
			lkp->lk_flags |= LK_WAIT_NONZERO;
		}
		for (interlocked = 1;;) {
			SPINLOCK_SPINCHECK;
			if ((lkp->lk_flags & wanted) != 0) {
				if (interlocked) {
					INTERLOCK_RELEASE(lkp, LK_SPIN, *s);
					interlocked = 0;
				}
				SPINLOCK_SPIN_HOOK;
			} 
			else if (interlocked) {
				break;
			} 
			else {
				INTERLOCK_ACQUIRE(lkp, LK_SPIN, *s);
				interlocked = 1;
			}
		}
		if (!drain) {
			lkp->lk_waitcount--;
			if (lkp->lk_waitcount == 0)
				lkp->lk_flags &= ~LK_WAIT_NONZERO;
		}
		KASSERT((lkp->lk_flags & wanted) == 0);
		error = 0;	/* sanity */
	} 
	else {
		for (error = 0; (lkp->lk_flags & wanted) != 0; ) {
			if (drain)
				lkp->lk_flags |= LK_WAITDRAIN;
			else {
				lkp->lk_waitcount++;
				lkp->lk_flags |= LK_WAIT_NONZERO;
			}
			/* XXX Cast away volatile. */
			LOCKSTAT_START_TIMER(slptime);
			error = ltsleep(drain ?
			    (volatile const void *)&lkp->lk_flags :
			    (volatile const void *)lkp, lkp->lk_prio,
			    lkp->lk_wmesg, lkp->lk_timo, &lkp->lk_interlock);
			LOCKSTAT_STOP_TIMER(slptime);
			LOCKSTAT_EVENT_RA((void *)(uintptr_t)lkp, LB_LOCKMGR | LB_SLEEP, 1, slptime, ra);
			if (!drain) {
				lkp->lk_waitcount--;
				if (lkp->lk_waitcount == 0) {
					lkp->lk_flags &= ~LK_WAIT_NONZERO;
				}
			}
			if (error) {
				break;
			}
			if (extflags & LK_SLEEPFAIL) {
				error = ENOLCK;
				break;
			}
			if (lkp->lk_newlock != NULL) {
				simple_lock(&lkp->lk_newlock->lk_interlock);
				simple_unlock(&lkp->lk_interlock);
				if (lkp->lk_waitcount == 0) {
					wakeup(&lkp->lk_newlock);
				}
				*lkpp = lkp = lkp->lk_newlock;
			}
		}
	}

	return error;
}

//===================================== PUBLIC =====================================================

/*
 * Initialize a lock; required before use.
 */
void lockinit(
	struct lock *lkp, 
	int prio, 
	const char *wmesg, 
	int timo, 
	int flags)
{
	memset(lkp, 0, sizeof(struct lock));
	simple_lock_init(&lkp->lk_interlock);
	lkp->lk_flags = flags & LK_EXTFLG_MASK;
	if (flags & LK_SPIN) {
		lkp->lk_cpu = LK_NOCPU;
	}
	else {
		lkp->lk_lockholder = LK_NOPROC;
		lkp->lk_newlock = NULL;
		lkp->lk_prio = prio;
		lkp->lk_timo = timo;
	}
	lkp->lk_wmesg = wmesg;	/* just a name for spin locks */
}

/*
 * Set, change, or release a lock.
 *
 * Shared requests increment the shared count. Exclusive requests set the
 * LK_WANT_EXCL flag (preventing further shared locks), and wait for already
 * accepted shared locks and shared-to-exclusive upgrades to go away.
 */
int lockmgr(
	volatile struct lock *lkp, 
	u_int flags,
    struct simplelock *interlkp)
{
	int error;
	pid_t pid = 0;
	lwpid_t lid = 0;
	int extflags;
	cpuid_t cpu_num = 0;
//	struct lwp *l = curlwp;
	int lock_shutdown_noblock = 0;
	int s = 0;

	error = 0;

	/* LK_RETRY is for vn_lock, not for lockmgr. */
	KASSERT((flags & LK_RETRY) == 0);

	INTERLOCK_ACQUIRE(lkp, lkp->lk_flags, s);
	if (flags & LK_INTERLOCK) {
		simple_unlock(interlkp);
	}
	extflags = (flags | lkp->lk_flags) & LK_EXTFLG_MASK;

	if (extflags & LK_SPIN) {
		pid = LK_KERNPROC;
		lid = 0;
	} 
	else {
//		if (l == NULL) {
//			if (!doing_shutdown) {
//				panic("lockmgr: no context");
//			} 
//			else {
//				l = &lwp0;
//				if (panicstr && (!(flags & LK_NOWAIT))) {
//					flags |= LK_NOWAIT;
//					lock_shutdown_noblock = 1;
//				}
//			}
//		}
//		lid = l->l_lid;
//		pid = l->l_proc->p_pid;
		lid = pthread_self();
		pid = getpid();
	}
//	cpu_num = cpu_number();

	/*
	 * Once a lock has drained, the LK_DRAINING flag is set and an
	 * exclusive lock is returned. The only valid operation thereafter
	 * is a single release of that exclusive lock. This final release
	 * clears the LK_DRAINING flag and sets the LK_DRAINED flag. Any
	 * further requests of any sort will result in a panic. The bits
	 * selected for these two flags are chosen so that they will be set
	 * in memory that is freed (freed memory is filled with 0xdeadbeef).
	 * The final release is permitted to give a new lease on life to
	 * the lock by specifying LK_REENABLE.
	 */
	if (lkp->lk_flags & (LK_DRAINING | LK_DRAINED)) {
		lkp->lk_flags &= ~LK_DRAINING;
		if ((flags & LK_REENABLE) == 0) {
			lkp->lk_flags |= LK_DRAINED;
		}
	}

	switch (flags & LK_TYPE_MASK) {

	case LK_SHARED:
		if (WEHOLDIT(lkp, pid, lid, cpu_num) == 0) {
			/*
			 * If just polling, check to see if we will block.
			 */
			if ((extflags & LK_NOWAIT) && (lkp->lk_flags &
			    (LK_HAVE_EXCL | LK_WANT_EXCL | LK_WANT_UPGRADE))) {
				error = EBUSY;
				break;
			}
			/*
			 * Wait for exclusive locks and upgrades to clear.
			 */
			error = acquire(&lkp, &s, extflags, 0, LK_HAVE_EXCL | LK_WANT_EXCL | LK_WANT_UPGRADE, RETURN_ADDRESS);
			if (error) {
				break;
			}
			lkp->lk_sharecount++;
			lkp->lk_flags |= LK_SHARE_NONZERO;
			COUNT(lkp, l, cpu_num, 1);
			break;
		}
		/*
		 * We hold an exclusive lock, so downgrade it to shared.
		 * An alternative would be to fail with EDEADLK.
		 */
		lkp->lk_sharecount++;
		lkp->lk_flags |= LK_SHARE_NONZERO;
		COUNT(lkp, l, cpu_num, 1);
		/* fall into downgrade */

	case LK_DOWNGRADE:
		if (WEHOLDIT(lkp, pid, lid, cpu_num) == 0 || lkp->lk_exclusivecount == 0) {
			panic("lockmgr: not holding exclusive lock");
		}
		lkp->lk_sharecount += lkp->lk_exclusivecount;
		lkp->lk_flags |= LK_SHARE_NONZERO;
		lkp->lk_exclusivecount = 0;
		lkp->lk_recurselevel = 0;
		lkp->lk_flags &= ~LK_HAVE_EXCL;
		SETHOLDER(lkp, LK_NOPROC, 0, LK_NOCPU);
		DONTHAVEIT(lkp);
		WAKEUP_WAITER(lkp);
		break;

	case LK_EXCLUPGRADE:
		/*
		 * If another process is ahead of us to get an upgrade,
		 * then we want to fail rather than have an intervening
		 * exclusive access.
		 */
		if (lkp->lk_flags & LK_WANT_UPGRADE) {
			lkp->lk_sharecount--;
			if (lkp->lk_sharecount == 0) {
				lkp->lk_flags &= ~LK_SHARE_NONZERO;
			}
			COUNT(lkp, l, cpu_num, -1);
			error = EBUSY;
			break;
		}
		/* fall into normal upgrade */

	case LK_UPGRADE:
		/*
		 * Upgrade a shared lock to an exclusive one. If another
		 * shared lock has already requested an upgrade to an
		 * exclusive lock, our shared lock is released and an
		 * exclusive lock is requested (which will be granted
		 * after the upgrade). If we return an error, the file
		 * will always be unlocked.
		 */
		if (WEHOLDIT(lkp, pid, lid, cpu_num) || lkp->lk_sharecount <= 0) {
			panic("lockmgr: upgrade exclusive lock");
		}
		lkp->lk_sharecount--;
		if (lkp->lk_sharecount == 0) {
			lkp->lk_flags &= ~LK_SHARE_NONZERO;
		}
		COUNT(lkp, l, cpu_num, -1);
		/*
		 * If we are just polling, check to see if we will block.
		 */
		if ((extflags & LK_NOWAIT) &&
		    ((lkp->lk_flags & LK_WANT_UPGRADE) ||
		     lkp->lk_sharecount > 1)) {
			error = EBUSY;
			break;
		}
		if ((lkp->lk_flags & LK_WANT_UPGRADE) == 0) {
			/*
			 * We are first shared lock to request an upgrade, so
			 * request upgrade and wait for the shared count to
			 * drop to zero, then take exclusive lock.
			 */
			lkp->lk_flags |= LK_WANT_UPGRADE;
			error = acquire(&lkp, &s, extflags, 0, LK_SHARE_NONZERO, RETURN_ADDRESS);
			lkp->lk_flags &= ~LK_WANT_UPGRADE;
			if (error) {
				WAKEUP_WAITER(lkp);
				break;
			}
			lkp->lk_flags |= LK_HAVE_EXCL;
			SETHOLDER(lkp, pid, lid, cpu_num);
			HAVEIT(lkp);
			if (lkp->lk_exclusivecount != 0) {
				panic("lockmgr: non-zero exclusive count");
			}
			lkp->lk_exclusivecount = 1;
			if (extflags & LK_SETRECURSE) {
				lkp->lk_recurselevel = 1;
			}
			COUNT(lkp, l, cpu_num, 1);
			break;
		}
		/*
		 * Someone else has requested upgrade. Release our shared
		 * lock, awaken upgrade requestor if we are the last shared
		 * lock, then request an exclusive lock.
		 */
		if (lkp->lk_sharecount == 0) {
			WAKEUP_WAITER(lkp);
		}
		/* fall into exclusive request */

	case LK_EXCLUSIVE:
		if (WEHOLDIT(lkp, pid, lid, cpu_num)) {
			/*
			 * Recursive lock.
			 */
			if ((extflags & LK_CANRECURSE) == 0 && lkp->lk_recurselevel == 0) {
				if (extflags & LK_RECURSEFAIL) {
					error = EDEADLK;
					break;
				} 
				else {
					panic("lockmgr: locking against myself");
				}
			}
			lkp->lk_exclusivecount++;
			if (extflags & LK_SETRECURSE && lkp->lk_recurselevel == 0) {
				lkp->lk_recurselevel = lkp->lk_exclusivecount;
			}
			COUNT(lkp, l, cpu_num, 1);
			break;
		}
		/*
		 * If we are just polling, check to see if we will sleep.
		 */
		if ((extflags & LK_NOWAIT) && (lkp->lk_flags &
		     (LK_HAVE_EXCL | LK_WANT_EXCL | LK_WANT_UPGRADE |
		     LK_SHARE_NONZERO))) {
			error = EBUSY;
			break;
		}
		/*
		 * Try to acquire the want_exclusive flag.
		 */
		error = acquire(&lkp, &s, extflags, 0, LK_HAVE_EXCL | LK_WANT_EXCL, RETURN_ADDRESS);
		if (error) {
			break;
		}
		lkp->lk_flags |= LK_WANT_EXCL;
		/*
		 * Wait for shared locks and upgrades to finish.
		 */
		error = acquire(&lkp, &s, extflags, 0, LK_HAVE_EXCL | LK_WANT_UPGRADE | LK_SHARE_NONZERO, RETURN_ADDRESS);
		lkp->lk_flags &= ~LK_WANT_EXCL;
		if (error) {
			WAKEUP_WAITER(lkp);
			break;
		}
		lkp->lk_flags |= LK_HAVE_EXCL;
		SETHOLDER(lkp, pid, lid, cpu_num);
		HAVEIT(lkp);
		if (lkp->lk_exclusivecount != 0) {
			panic("lockmgr: non-zero exclusive count");
		}
		lkp->lk_exclusivecount = 1;
		if (extflags & LK_SETRECURSE) {
			lkp->lk_recurselevel = 1;
		}
		COUNT(lkp, l, cpu_num, 1);
		break;

	case LK_RELEASE:
		if (lkp->lk_exclusivecount != 0) {
			if (WEHOLDIT(lkp, pid, lid, cpu_num) == 0) {
				if (lkp->lk_flags & LK_SPIN) {
					panic("lockmgr: processor %lu, not exclusive lock holder %lu unlocking", cpu_num, lkp->lk_cpu);
				} 
				else {
					panic("lockmgr: pid %d, not exclusive lock holder %d unlocking", pid, lkp->lk_lockholder);
				}
			}
			if (lkp->lk_exclusivecount == lkp->lk_recurselevel) {
				lkp->lk_recurselevel = 0;
			}
			lkp->lk_exclusivecount--;
			COUNT(lkp, l, cpu_num, -1);
			if (lkp->lk_exclusivecount == 0) {
				lkp->lk_flags &= ~LK_HAVE_EXCL;
				SETHOLDER(lkp, LK_NOPROC, 0, LK_NOCPU);
				DONTHAVEIT(lkp);
			}
		} 
		else if (lkp->lk_sharecount != 0) {
			lkp->lk_sharecount--;
			if (lkp->lk_sharecount == 0) {
				lkp->lk_flags &= ~LK_SHARE_NONZERO;
			}
			COUNT(lkp, l, cpu_num, -1);
		}
		WAKEUP_WAITER(lkp);
		break;

	case LK_DRAIN:
		/*
		 * Check that we do not already hold the lock, as it can
		 * never drain if we do. Unfortunately, we have no way to
		 * check for holding a shared lock, but at least we can
		 * check for an exclusive one.
		 */
		if (WEHOLDIT(lkp, pid, lid, cpu_num)) {
			panic("lockmgr: draining against myself");
		}
		/*
		 * If we are just polling, check to see if we will sleep.
		 */
		if ((extflags & LK_NOWAIT) && (lkp->lk_flags &
		     (LK_HAVE_EXCL | LK_WANT_EXCL | LK_WANT_UPGRADE |
		     LK_SHARE_NONZERO | LK_WAIT_NONZERO))) {
			error = EBUSY;
			break;
		}
		error = acquire(&lkp, &s, extflags, 1,
		    LK_HAVE_EXCL | LK_WANT_EXCL | LK_WANT_UPGRADE |
		    LK_SHARE_NONZERO | LK_WAIT_NONZERO, RETURN_ADDRESS);
		if (error) {
			break;
		}
		lkp->lk_flags |= LK_HAVE_EXCL;
		if ((extflags & LK_RESURRECT) == 0) {
			lkp->lk_flags |= LK_DRAINING;
		}
		SETHOLDER(lkp, pid, lid, cpu_num);
		HAVEIT(lkp);
		lkp->lk_exclusivecount = 1;
		/* XXX unlikely that we'd want this */
		if (extflags & LK_SETRECURSE) {
			lkp->lk_recurselevel = 1;
		}
		COUNT(lkp, l, cpu_num, 1);
		break;

	default:
		INTERLOCK_RELEASE(lkp, lkp->lk_flags, s);
		panic("lockmgr: unknown locktype request %d", flags & LK_TYPE_MASK);
		/* NOTREACHED */
	}
	if ((lkp->lk_flags & (LK_WAITDRAIN|LK_SPIN)) == LK_WAITDRAIN &&
	    ((lkp->lk_flags & (LK_HAVE_EXCL | LK_WANT_EXCL | LK_WANT_UPGRADE |
	      LK_SHARE_NONZERO | LK_WAIT_NONZERO)) == 0)) {
		lkp->lk_flags &= ~LK_WAITDRAIN;
		wakeup(&lkp->lk_flags);
	}
	/*
	 * Note that this panic will be a recursive panic, since
	 * we only set lock_shutdown_noblock above if panicstr != NULL.
	 */
	if (error && lock_shutdown_noblock) {
		panic("lockmgr: deadlock (see previous panic)");
	}

	INTERLOCK_RELEASE(lkp, lkp->lk_flags, s);
	return (error);
}
