/* $NetBSD: subr_autoconf.c,v 1.115 2006/10/02 02:59:38 chs Exp $ */

/*
 * Copyright (c) 1996, 2000 Christopher G. Demetriou
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *          This product includes software developed for the
 *          NetBSD Project.  See http://www.NetBSD.org/ for
 *          information about NetBSD.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * --(license Id: LICENSE.proto,v 1.1 2000/06/13 21:40:26 cgd Exp )--
 */

/*
 * Copyright (c) 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This software was developed by the Computer Systems Engineering group
 * at Lawrence Berkeley Laboratory under DARPA contract BG 91-66 and
 * contributed to Berkeley.
 *
 * All advertising materials mentioning features or use of this software
 * must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Lawrence Berkeley Laboratories.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * from: Header: subr_autoconf.c,v 1.12 93/02/01 19:31:48 torek Exp  (LBL)
 *
 *	@(#)subr_autoconf.c	8.3 (Berkeley) 5/17/94
 *
 * 2008: modified by minoru murashima.
 */

#include <sys/errno.h>
#include <sys/queue.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

//=====================================  ===================================================

#define	ROOT ((device_t)NULL)

/* Hooks to finalize configuration once all real devices have been found. */
struct finalize_hook {
	TAILQ_ENTRY(finalize_hook) f_list;
	int (*f_func)(device_t);
	device_t f_dev;
};

struct matchinfo {
	cfsubmatch_t	fn;
	struct device	*parent;
	const int		*locs;
	void			*aux;
	struct cfdata	*match;
	int				pri;
};

struct deferred_config {
	TAILQ_ENTRY(deferred_config) dc_queue;
	device_t dc_dev;
	void (*dc_func)(device_t);
};

#define	STREQ(s1, s2)			\
	(*(s1) == *(s2) && strcmp((s1), (s2)) == 0)

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

/*
 * ơ֥롢ꤵ
 */
static struct cfdriver **cfdriver_list_initial;
static struct cfattachinit *cfattachinit;
static struct cfdata *cfdata;
static short *cfroots;

static int config_initialized;		/* config_init() has been called. */

/*
 * List of all cfdriver structures.  We use this to detect duplicates
 * when other cfdrivers are loaded.
 */
struct cfdriverlist allcfdrivers = LIST_HEAD_INITIALIZER(&allcfdrivers);

/*
 * List of cfdata tables.  We always have one such list -- the one
 * built statically when the kernel was configured.
 */
struct cftablelist allcftables;
static struct cftable initcftable;

struct deferred_config_head deferred_config_queue;
struct deferred_config_head interrupt_config_queue;
static TAILQ_HEAD(, finalize_hook) config_finalize_list;
struct devicelist alldevs;		/* list of all devices */
TAILQ_HEAD(deferred_config_head, deferred_config);
static volatile int config_pending;		/* semaphore for mountroot */

static const char * const msgs[3] = { "", " not configured\n", " unsupported\n" };

/*
 * Look up a cfdriver by name.
 */
struct cfdriver *config_cfdriver_lookup(
	const char *name)
{
	struct cfdriver *cd;

	LIST_FOREACH(cd, &allcfdrivers, cd_list) {
		if (STREQ(cd->cd_name, name)) {
			return (cd);
		}
	}

	return (NULL);
}

/*
 * Look up a cfattach by name.
 */
static struct cfattach *config_cfattach_lookup_cd(
	struct cfdriver *cd, 
	const char *atname)
{
	struct cfattach *ca;

	LIST_FOREACH(ca, &cd->cd_attach, ca_list) {
		if (STREQ(ca->ca_name, atname)) {
			return (ca);
		}
	}

	return (NULL);
}

/*
 * Process a deferred configuration queue.
 */
static void config_process_deferred(
	struct deferred_config_head *queue,
	device_t parent)
{
	struct deferred_config *dc, *ndc;

	for (dc = TAILQ_FIRST(queue); dc != NULL; dc = ndc) {
		ndc = TAILQ_NEXT(dc, dc_queue);
		if (parent == NULL || dc->dc_dev->dv_parent == parent) {
			TAILQ_REMOVE(queue, dc, dc_queue);
			(*dc->dc_func)(dc->dc_dev);
			free(dc);
			config_pending_decr();
		}
	}
}

/*
 * Apply the matching function and choose the best.  This is used
 * a few times and we want to keep the code small.
 */
static void mapply(
	struct matchinfo *m, 
	cfdata_t cf)
{
	int pri;

	if (m->fn != NULL) {
		pri = (*m->fn)(m->parent, cf, m->locs, m->aux);
	} 
	else {
		pri = config_match(m->parent, cf, m->aux);
	}
	if (pri > m->pri) {
		m->match = cf;
		m->pri = pri;
	}
}

/* just like sprintf(buf, "%d") except that it works from the end */
static char *number(
	char *ep, 
	int n)
{

	*--ep = 0;
	while (n >= 10) {
		*--ep = (n % 10) + '0';
		n /= 10;
	}
	*--ep = n + '0';
	return (ep);
}

/*
 * Helper function: check whether the driver supports the interface attribute
 * and return its descriptor structure.
 */
static const struct cfiattrdata *cfdriver_get_iattr(
	const struct cfdriver *cd, 
	const char *ia)
{
	const struct cfiattrdata * const *cpp;

	if (cd->cd_attrs == NULL) {
		return (0);
	}

	for (cpp = cd->cd_attrs; *cpp; cpp++) {
		if (STREQ((*cpp)->ci_name, ia)) {
			/* Match. */
			return (*cpp);
		}
	}
	return (0);
}

/*
 * Determine if `parent' is a potential parent for a device spec based
 * on `cfp'.
 */
static int cfparent_match(
	const device_t parent, 
	const struct cfparent *cfp)
{
	struct cfdriver *pcd;

	/* We don't match root nodes here. */
	if (cfp == NULL) {
		return (0);
	}

	pcd = parent->dv_cfdriver;
	KASSERT(pcd != NULL);

	/*
	 * First, ensure this parent has the correct interface
	 * attribute.
	 */
	if (!cfdriver_get_iattr(pcd, cfp->cfp_iattr)) {
		return (0);
	}

	/*
	 * If no specific parent device instance was specified (i.e.
	 * we're attaching to the attribute only), we're done!
	 */
	if (cfp->cfp_parent == NULL) {
		return (1);
	}

	/*
	 * Check the parent device's name.
	 */
	if (STREQ(pcd->cd_name, cfp->cfp_parent) == 0) {
		return (0);	/* not the same parent */
	}

	/*
	 * Make sure the unit number matches.
	 */
	if (cfp->cfp_unit == DVUNIT_ANY ||	/* wildcard */
	    cfp->cfp_unit == parent->dv_unit) {
		return (1);
	}

	/* Unit numbers don't match. */
	return (0);
}

//===================================== PUBLIC =====================================================

/*
 * Initialize the autoconfiguration data structures.  Normally this
 * is done by configure(), but some platforms need to do this very
 * early (to e.g. initialize the console).
 */
void config_init(void)
{
	const struct cfattachinit *cfai;
	int i, j;

	if (config_initialized) {
		return;
	}

	/* allcfdrivers is statically initialized. */
	for (i = 0; cfdriver_list_initial[i] != NULL; i++) {
		if (config_cfdriver_attach(cfdriver_list_initial[i]) != 0) {
			panic("configure: duplicate `%s' drivers", cfdriver_list_initial[i]->cd_name);
		}
	}

	for (cfai = &cfattachinit[0]; cfai->cfai_name != NULL; cfai++) {
		for (j = 0; cfai->cfai_list[j] != NULL; j++) {
			if (config_cfattach_attach(cfai->cfai_name, cfai->cfai_list[j]) != 0) {
				panic("configure: duplicate `%s' attachment of `%s' driver", cfai->cfai_list[j]->ca_name, cfai->cfai_name);
			}
		}
	}

	TAILQ_INIT(&allcftables);
	initcftable.ct_cfdata = cfdata;
	TAILQ_INSERT_TAIL(&allcftables, &initcftable, ct_list);

	TAILQ_INIT(&deferred_config_queue);
	TAILQ_INIT(&interrupt_config_queue);
	TAILQ_INIT(&config_finalize_list);
	TAILQ_INIT(&alldevs);

	config_initialized = 1;
}

/*
 * Invoke the "match" routine for a cfdata entry on behalf of
 * an external caller, usually a "submatch" routine.
 */
int config_match(
	device_t parent, 
	cfdata_t cf, 
	void *aux)
{
	struct cfattach *ca;

	ca = config_cfattach_lookup(cf->cf_name, cf->cf_atname);
	if (ca == NULL) {
		/* No attachment for this entry, oh well. */
		return (0);
	}

	return ((*ca->ca_match)(parent, cf, aux));
}

device_t config_attach(
	device_t parent, 
	cfdata_t cf, 
	void *aux, 
	cfprint_t print)
{
	return (config_attach_loc(parent, cf, NULL, aux, print));
}

/*
 * As above, but for pseudo-devices.  Pseudo-devices attached in this
 * way are silently inserted into the device tree, and their children
 * attached.
 *
 * Note that because pseudo-devices are attached silently, any information
 * the attach routine wishes to print should be prefixed with the device
 * name by the attach routine.
 */
device_t config_attach_pseudo(
	cfdata_t cf)
{
	device_t dev;
	struct cfdriver *cd;
	struct cfattach *ca;
	size_t lname, lunit;
	const char *xunit;
	int myunit;
	char num[10];

	cd = config_cfdriver_lookup(cf->cf_name);
	if (cd == NULL) {
		return (NULL);
	}

	ca = config_cfattach_lookup_cd(cd, cf->cf_atname);
	if (ca == NULL) {
		return (NULL);
	}

	if (ca->ca_devsize < sizeof(struct device)) {
		panic("config_attach_pseudo");
	}

	/*
	 * We just ignore cf_fstate, instead doing everything with
	 * cf_unit.
	 *
	 * XXX Should we change this and use FSTATE_NOTFOUND and
	 * XXX FSTATE_STAR?
	 */

	if (cf->cf_unit == DVUNIT_ANY) {
		for (myunit = 0; myunit < cd->cd_ndevs; myunit++) {
			if (cd->cd_devs[myunit] == NULL) {
				break;
			}
		}
		/*
		 * myunit is now the unit of the first NULL device pointer.
		 */
	} 
	else {
		myunit = cf->cf_unit;
		if (myunit < cd->cd_ndevs && cd->cd_devs[myunit] != NULL) {
			return (NULL);
		}
	}

	/* compute length of name and decimal expansion of unit number */
	lname = strlen(cd->cd_name);
	xunit = number(&num[sizeof(num)], myunit);
	lunit = &num[sizeof(num)] - xunit;
	if (sizeof(dev->dv_xname) < lname + lunit) {
		panic("config_attach_pseudo: device name too long");
	}

	/* get memory for all device vars */
	dev = (device_t)malloc(ca->ca_devsize);
	if (!dev) {
		panic("config_attach_pseudo: memory allocation for device softc failed");
	}
	memset(dev, 0, ca->ca_devsize);
	TAILQ_INSERT_TAIL(&alldevs, dev, dv_list);	/* link up */
	dev->dv_class = cd->cd_class;
	dev->dv_cfdata = cf;
	dev->dv_cfdriver = cd;
	dev->dv_cfattach = ca;
	dev->dv_unit = myunit;
	memcpy(dev->dv_xname, cd->cd_name, lname);
	memcpy(dev->dv_xname + lname, xunit, lunit);
	dev->dv_parent = ROOT;
	dev->dv_flags = DVF_ACTIVE;	/* always initially active */
//	dev->dv_properties = prop_dictionary_create();
	KASSERT(dev->dv_properties != NULL);

	/* put this device in the devices array */
	config_makeroom(dev->dv_unit, cd);
	if (cd->cd_devs[dev->dv_unit]) {
		panic("config_attach_pseudo: duplicate %s", dev->dv_xname);
		
	}
	cd->cd_devs[dev->dv_unit] = dev;

	(*ca->ca_attach)(ROOT, dev, NULL);
	config_process_deferred(&deferred_config_queue, dev);
	return (dev);
}

/*
 * Detach a device.  Optionally forced (e.g. because of hardware
 * removal) and quiet.  Returns zero if successful, non-zero
 * (an error code) otherwise.
 *
 * Note that this code wants to be run from a process context, so
 * that the detach can sleep to allow processes which have a device
 * open to run and unwind their stacks.
 */
int config_detach(
	device_t dev, 
	int flags)
{
	struct cftable *ct;
	cfdata_t cf;
	const struct cfattach *ca;
	struct cfdriver *cd;
	int rv = 0, i;

	cd = dev->dv_cfdriver;
	KASSERT(cd != NULL);

	ca = dev->dv_cfattach;
	KASSERT(ca != NULL);

	/*
	 * Ensure the device is deactivated.  If the device doesn't
	 * have an activation entry point, we allow DVF_ACTIVE to
	 * remain set.  Otherwise, if DVF_ACTIVE is still set, the
	 * device is busy, and the detach fails.
	 */
	if (ca->ca_activate != NULL) {
		rv = config_deactivate(dev);
	}

	/*
	 * Try to detach the device.  If that's not possible, then
	 * we either panic() (for the forced but failed case), or
	 * return an error.
	 */
	if (rv == 0) {
		if (ca->ca_detach != NULL) {
			rv = (*ca->ca_detach)(dev, flags);
		}
		else {
			rv = EOPNOTSUPP;
		}
	}
	if (rv != 0) {
		if ((flags & DETACH_FORCE) == 0) {
			return (rv);
		}
		else {
			panic("config_detach: forced detach of %s failed (%d)", dev->dv_xname, rv);
		}
	}

	/*
	 * The device has now been successfully detached.
	 */

	/* notify the parent that the child is gone */
	if (dev->dv_parent) {
		device_t p = dev->dv_parent;
		if (p->dv_cfattach->ca_childdetached) {
			(*p->dv_cfattach->ca_childdetached)(p, dev);
		}
	}

	/*
	 * Mark cfdata to show that the unit can be reused, if possible.
	 */
	TAILQ_FOREACH(ct, &allcftables, ct_list) {
		for (cf = ct->ct_cfdata; cf->cf_name; cf++) {
			if (STREQ(cf->cf_name, cd->cd_name)) {
				if (cf->cf_fstate == FSTATE_FOUND &&
				    cf->cf_unit == dev->dv_unit) {
					cf->cf_fstate = FSTATE_NOTFOUND;
				}
#ifdef __BROKEN_CONFIG_UNIT_USAGE
				/*
				 * Note that we can only re-use a starred
				 * unit number if the unit being detached
				 * had the last assigned unit number.
				 */
				if (cf->cf_fstate == FSTATE_STAR &&
				    cf->cf_unit == dev->dv_unit + 1) {
					cf->cf_unit--;
				}
#endif /* __BROKEN_CONFIG_UNIT_USAGE */
			}
		}
	}

	/*
	 * Unlink from device list.
	 */
	TAILQ_REMOVE(&alldevs, dev, dv_list);

	/*
	 * Remove from cfdriver's array, tell the world (unless it was
	 * a pseudo-device), and free softc.
	 */
	cd->cd_devs[dev->dv_unit] = NULL;
	if (dev->dv_cfdata != NULL && (flags & DETACH_QUIET) == 0) {
		aprint_normal("%s detached\n", dev->dv_xname);
	}
	if (dev->dv_locators) {
		free(dev->dv_locators);
	}
	KASSERT(dev->dv_properties != NULL);
//	prop_object_release(dev->dv_properties);
	free(dev);

	/*
	 * If the device now has no units in use, deallocate its softc array.
	 */
	for (i = 0; i < cd->cd_ndevs; i++) {
		if (cd->cd_devs[i] != NULL) {
			break;
		}
	}
	if (i == cd->cd_ndevs) {		/* nothing found; deallocate */
		free(cd->cd_devs);
		cd->cd_devs = NULL;
		cd->cd_ndevs = 0;
	}

	/*
	 * Return success.
	 */
	return (0);
}

int config_deactivate(
	device_t dev)
{
	const struct cfattach *ca = dev->dv_cfattach;
	int rv = 0, oflags = dev->dv_flags;

	if (ca->ca_activate == NULL) {
		return (EOPNOTSUPP);
	}

	if (dev->dv_flags & DVF_ACTIVE) {
		dev->dv_flags &= ~DVF_ACTIVE;
		rv = (*ca->ca_activate)(dev, DVACT_DEACTIVATE);
		if (rv) {
			dev->dv_flags = oflags;
		}
	}
	return (rv);
}

int config_stdsubmatch(
	device_t parent, 
	cfdata_t cf, 
	const int *locs, 
	void *aux)
{
	const struct cfiattrdata *ci;
	const struct cflocdesc *cl;
	int nlocs, i;

	ci = cfiattr_lookup(cf->cf_pspec->cfp_iattr, parent->dv_cfdriver);
	KASSERT(ci);
	nlocs = ci->ci_loclen;
	for (i = 0; i < nlocs; i++) {
		cl = &ci->ci_locdesc[i];
		/* !cld_defaultstr means no default value */
		if ((!(cl->cld_defaultstr)
		     || (cf->cf_loc[i] != cl->cld_default))
		    && cf->cf_loc[i] != locs[i]) {
			return (0);
		}
	}

	return (config_match(parent, cf, aux));
}

/*
 * Add a cfdriver to the system.
 */
int config_cfdriver_attach(
	struct cfdriver *cd)
{
	struct cfdriver *lcd;

	/* Make sure this driver isn't already in the system. */
	LIST_FOREACH(lcd, &allcfdrivers, cd_list) {
		if (STREQ(lcd->cd_name, cd->cd_name)) {
			return (EEXIST);
		}
	}

	LIST_INIT(&cd->cd_attach);
	LIST_INSERT_HEAD(&allcfdrivers, cd, cd_list);

		return (0);
}

/*
 * Add a cfattach to the specified driver.
 */
int config_cfattach_attach(
	const char *driver, 
	struct cfattach *ca)
{
	struct cfattach *lca;
	struct cfdriver *cd;

	cd = config_cfdriver_lookup(driver);
	if (cd == NULL) {
		return (ESRCH);
	}

	/* Make sure this attachment isn't already on this driver. */
	LIST_FOREACH(lca, &cd->cd_attach, ca_list) {
		if (STREQ(lca->ca_name, ca->ca_name)) {
			return (EEXIST);
		}
	}

	LIST_INSERT_HEAD(&cd->cd_attach, ca, ca_list);

	return (0);
}

/*
 * Attach a found device.  Allocates memory for device variables.
 */
device_t config_attach_loc(
	device_t parent, 
	cfdata_t cf,
	const int *locs, 
	void *aux, 
	cfprint_t print)
{
	device_t dev;
	struct cftable *ct;
	struct cfdriver *cd;
	struct cfattach *ca;
	size_t lname, lunit;
	const char *xunit;
	int myunit;
	char num[10];
	const struct cfiattrdata *ia;

	cd = config_cfdriver_lookup(cf->cf_name);
	KASSERT(cd != NULL);

	ca = config_cfattach_lookup_cd(cd, cf->cf_atname);
	KASSERT(ca != NULL);

	if (ca->ca_devsize < sizeof(struct device)) {
		panic("config_attach");
	}

	if (cf->cf_fstate == FSTATE_STAR) {
		for (myunit = cf->cf_unit; myunit < cd->cd_ndevs; myunit++) {
			if (cd->cd_devs[myunit] == NULL) {
				break;
			}
		}
	} 
	else {
		/*
		 * myunit is now the unit of the first NULL device pointer,
		 * or max(cd->cd_ndevs,cf->cf_unit).
		 */
		myunit = cf->cf_unit;
		KASSERT(cf->cf_fstate == FSTATE_NOTFOUND);
		cf->cf_fstate = FSTATE_FOUND;
	}

	/* compute length of name and decimal expansion of unit number */
	lname = strlen(cd->cd_name);
	xunit = number(&num[sizeof(num)], myunit);
	lunit = &num[sizeof(num)] - xunit;
	if (sizeof(dev->dv_xname) < lname + lunit) {
		panic("config_attach: device name too long");
	}

	/* get memory for all device vars */
	dev = (device_t)malloc(ca->ca_devsize);
	if (!dev) {
		panic("config_attach: memory allocation for device softc failed");
	}
	memset(dev, 0, ca->ca_devsize);
	TAILQ_INSERT_TAIL(&alldevs, dev, dv_list);	/* link up */
	dev->dv_class = cd->cd_class;
	dev->dv_cfdata = cf;
	dev->dv_cfdriver = cd;
	dev->dv_cfattach = ca;
	dev->dv_unit = myunit;
	memcpy(dev->dv_xname, cd->cd_name, lname);
	memcpy(dev->dv_xname + lname, xunit, lunit);
	dev->dv_parent = parent;
	dev->dv_flags = DVF_ACTIVE;	/* always initially active */
	if (locs) {
		KASSERT(parent); /* no locators at root */
		ia = cfiattr_lookup(cf->cf_pspec->cfp_iattr, parent->dv_cfdriver);
		dev->dv_locators = malloc(ia->ci_loclen * sizeof(int));
		memcpy(dev->dv_locators, locs, ia->ci_loclen * sizeof(int));
	}
//	dev->dv_properties = prop_dictionary_create();
//	KASSERT(dev->dv_properties != NULL);

	aprint_naive("Found ");
	/*
	 * We want the next two printfs for normal, verbose, and quiet,
	 * but not silent (in which case, we're twiddling, instead).
	 */
	if (parent == ROOT) {
		aprint_naive("%s (root)", dev->dv_xname);
		aprint_normal("%s (root)", dev->dv_xname);
	} 
	else {
		aprint_naive("%s at %s", dev->dv_xname, parent->dv_xname);
		aprint_normal("%s at %s", dev->dv_xname, parent->dv_xname);
		if (print) {
			(void) (*print)(aux, NULL);
		}
	}

	/* put this device in the devices array */
	config_makeroom(dev->dv_unit, cd);
	if (cd->cd_devs[dev->dv_unit]) {
		panic("config_attach: duplicate %s", dev->dv_xname);
	}
	cd->cd_devs[dev->dv_unit] = dev;

	/*
	 * Before attaching, clobber any unfound devices that are
	 * otherwise identical.
	 */
	TAILQ_FOREACH(ct, &allcftables, ct_list) {
		for (cf = ct->ct_cfdata; cf->cf_name; cf++) {
			if (STREQ(cf->cf_name, cd->cd_name) && cf->cf_unit == dev->dv_unit) {
				if (cf->cf_fstate == FSTATE_NOTFOUND) {
					cf->cf_fstate = FSTATE_FOUND;
				}
			}
		}
	}
	(*ca->ca_attach)(parent, dev, aux);
	config_process_deferred(&deferred_config_queue, dev);

	return (dev);
}

/*
 * Expand the size of the cd_devs array if necessary.
 */
void config_makeroom(
	int n, 
	struct cfdriver *cd)
{
	int old, new;
	void **nsp;

	if (n < cd->cd_ndevs) {
		return;
	}

	/*
	 * Need to expand the array.
	 */
	old = cd->cd_ndevs;
	if (old == 0) {
		new = 4;
	}
	else {
		new = old * 2;
	}
	while (new <= n) {
		new *= 2;
	}
	cd->cd_ndevs = new;
	nsp = malloc(new * sizeof(void *));
	if (nsp == NULL) {
		panic("config_attach: %sing dev array", old != 0 ? "expand" : "creat");
	}
	memset(nsp + old, 0, (new - old) * sizeof(void *));
	if (old != 0) {
		memcpy(nsp, cd->cd_devs, old * sizeof(void *));
		free(cd->cd_devs);
	}
	cd->cd_devs = nsp;
}

/*
 * Manipulate the config_pending semaphore.
 */
void config_pending_incr(void)
{
	config_pending++;
}

void config_pending_decr(void)
{
	config_pending--;
	if (config_pending == 0) {
//		wakeup(&config_pending);
	}
}

//--------------------------------------------------------------------------------------------------
// Getter
//--------------------------------------------------------------------------------------------------

void *device_private(
	device_t dev)
{
	/*
	 * For now, at least, "struct device" is the first thing in
	 * the driver's private data.  So, we just return ourselves.
	 */
	return (dev);
}

cfdata_t device_cfdata(
	device_t dev)
{
	return (dev->dv_cfdata);
}

int device_unit(device_t dev)
{
	return (dev->dv_unit);
}

device_t device_parent(
	device_t dev)
{
	return (dev->dv_parent);
}

//--------------------------------------------------------------------------------------------------
// Setter
//--------------------------------------------------------------------------------------------------

/*
 *եǥХ˸ƤӽФ뤳
 */
void setInitialTable(
	struct cfdriver **i_cfdriverList,
	struct cfattachinit *i_cfattachinit,
	struct cfdata *i_cfdata,
	short *i_cfroots)
{
	cfdriver_list_initial = i_cfdriverList;
	cfattachinit = i_cfattachinit;
	cfdata = i_cfdata;
	cfroots = i_cfroots;
}

//--------------------------------------------------------------------------------------------------
// Search
//--------------------------------------------------------------------------------------------------

/*
 * As above, but for root devices.
 */
device_t config_rootfound(
	const char *rootname, 
	void *aux)
{
	cfdata_t cf;

	if ((cf = config_rootsearch((cfsubmatch_t)NULL, rootname, aux)) != NULL) {
		return (config_attach(ROOT, cf, aux, (cfprint_t)NULL));
	}
	aprint_error("root device %s not configured\n", rootname);
	return (NULL);
}

/*
 * Find the given root device.
 * This is much like config_search, but there is no parent.
 * Don't bother with multiple cfdata tables; the root node
 * must always be in the initial table.
 */
cfdata_t config_rootsearch(
	cfsubmatch_t fn, 
	const char *rootname, 
	void *aux)
{
	cfdata_t cf;
	const short *p;
	struct matchinfo m;

	m.fn = fn;
	m.parent = ROOT;
	m.aux = aux;
	m.match = NULL;
	m.pri = 0;
	m.locs = 0;
	/*
	 * Look at root entries for matching name.  We do not bother
	 * with found-state here since only one root should ever be
	 * searched (and it must be done first).
	 */
	for (p = cfroots; *p >= 0; p++) {
		cf = &cfdata[*p];
		if (strcmp(cf->cf_name, rootname) == 0) {
			mapply(&m, cf);
		}
	}
	return (m.match);
}

/*
 * Look up a cfattach by driver/attachment name.
 */
struct cfattach *config_cfattach_lookup(
	const char *name, 
	const char *atname)
{
	struct cfdriver *cd;

	cd = config_cfdriver_lookup(name);
	if (cd == NULL) {
		return (NULL);
	}

	return (config_cfattach_lookup_cd(cd, atname));
}

device_t config_found(
	device_t parent, 
	void *aux, 
	cfprint_t print)
{
	return (config_found_sm_loc(parent, NULL, NULL, aux, print, NULL));
}

device_t config_found_ia(
	device_t parent, 
	const char *ifattr, 
	void *aux,
    cfprint_t print)
{
	return (config_found_sm_loc(parent, ifattr, NULL, aux, print, NULL));
}

/*
 * The given `aux' argument describes a device that has been found
 * on the given parent, but not necessarily configured.  Locate the
 * configuration data for that device (using the submatch function
 * provided, or using candidates' cd_match configuration driver
 * functions) and attach it, and return true.  If the device was
 * not configured, call the given `print' function and return 0.
 */
device_t config_found_sm_loc(
	device_t parent,
	const char *ifattr, 
	const int *locs, 
	void *aux,
	cfprint_t print, 
	cfsubmatch_t submatch)
{
	cfdata_t cf;

	if ((cf = config_search_loc(submatch, parent, ifattr, locs, aux))) {
		return(config_attach_loc(parent, cf, locs, aux, print));
	}
	if (print) {
//		if (config_do_twiddle) {
//			twiddle();
//		}
//		aprint_normal("%s", msgs[(*print)(aux, parent->dv_xname)]);
	}

	return (NULL);
}

/*
 * Iterate over all potential children of some device, calling the given
 * function (default being the child's match function) for each one.
 * Nonzero returns are matches; the highest value returned is considered
 * the best match.  Return the `found child' if we got a match, or NULL
 * otherwise.  The `aux' pointer is simply passed on through.
 *
 * Note that this function is designed so that it can be used to apply
 * an arbitrary function to all potential children (its return value
 * can be ignored).
 */
cfdata_t config_search_loc(
	cfsubmatch_t fn, 
	device_t parent,
	const char *ifattr, 
	const int *locs, 
	void *aux)
{
	struct cftable *ct;
	cfdata_t cf;
	struct matchinfo m;

	KASSERT(config_initialized);
	KASSERT(!ifattr || cfdriver_get_iattr(parent->dv_cfdriver, ifattr));

	m.fn = fn;
	m.parent = parent;
	m.locs = locs;
	m.aux = aux;
	m.match = NULL;
	m.pri = 0;

	TAILQ_FOREACH(ct, &allcftables, ct_list) {
		for (cf = ct->ct_cfdata; cf->cf_name; cf++) {

			/* We don't match root nodes here. */
			if (!cf->cf_pspec) {
				continue;
			}

			/*
			 * Skip cf if no longer eligible, otherwise scan
			 * through parents for one matching `parent', and
			 * try match function.
			 */
			if (cf->cf_fstate == FSTATE_FOUND) {
				continue;
			}
			if (cf->cf_fstate == FSTATE_DNOTFOUND || cf->cf_fstate == FSTATE_DSTAR) {
				continue;
			}

			/*
			 * If an interface attribute was specified,
			 * consider only children which attach to
			 * that attribute.
			 */
			if (ifattr && !STREQ(ifattr, cf->cf_pspec->cfp_iattr)) {
				continue;
			}

			if (cfparent_match(parent, cf->cf_pspec)) {
				mapply(&m, cf);
			}
		}
	}
	return (m.match);
}

/*
 * Lookup an interface attribute description by name.
 * If the driver is given, consider only its supported attributes.
 */
const struct cfiattrdata *cfiattr_lookup(
	const char *name, 
	const struct cfdriver *cd)
{
	const struct cfdriver *d;
	const struct cfiattrdata *ia;

	if (cd) {
		return (cfdriver_get_iattr(cd, name));
	}

	LIST_FOREACH(d, &allcfdrivers, cd_list) {
		ia = cfdriver_get_iattr(d, name);
		if (ia) {
			return (ia);
		}
	}
	return (0);
}

/*
 *	Look up a device instance for a given driver.
 */
void *device_lookup(
	cfdriver_t cd, 
	int unit)
{
	if (unit < 0 || unit >= cd->cd_ndevs) {
		return (NULL);
	}
	
	return (cd->cd_devs[unit]);
}

boolean_t device_is_active(
	device_t dev)
{
	return ((dev->dv_flags & DVF_ACTIVE) != 0);
}

int device_locator(device_t dev, u_int locnum)
{
	KASSERT(dev->dv_locators != NULL);
	return (dev->dv_locators[locnum]);
}

cfdata_t config_search_ia(
	cfsubmatch_t fn, 
	device_t parent, 
	const char *ifattr,
    void *aux)
{
	return (config_search_loc(fn, parent, ifattr, NULL, aux));
}

