/*	$NetBSD: subr_disk.c,v 1.83 2006/11/25 11:59:58 scw Exp $	*/

/*-
 * Copyright (c) 1996, 1997, 1999, 2000 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe of the Numerical Aerospace Simulation Facility,
 * NASA Ames Research Center.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the NetBSD
 *	Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1982, 1986, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)ufs_disksubr.c	8.5 (Berkeley) 1/21/94
 *
 * 2009: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/lock.h>
#include <sys/iostat.h>
#include <sys/buf.h>
#include <stdlib.h>
#include <string.h>
#include <machine/int_fmtio.h>
#include <sys/disklabel.h>
#include <sys/disk.h>

//=====================================  ===================================================

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static void disk_init0(
	struct disk *diskp)
{
	/*
	 * Initialize the wedge-related locks and other fields.
	 */
	lockinit(&diskp->dk_rawlock, PRIBIO, "dkrawlk", 0, 0);
	lockinit(&diskp->dk_openlock, PRIBIO, "dkoplk", 0, 0);
	LIST_INIT(&diskp->dk_wedges);
	diskp->dk_nwedges = 0;
	diskp->dk_labelsector = LABELSECTOR;
	disk_blocksize(diskp, DEV_BSIZE);
}

static void disk_attach0(
	struct disk *diskp)
{
	/*
	 * Allocate and initialize the disklabel structures.  Note that
	 * it's not safe to sleep here, since we're probably going to be
	 * called during autoconfiguration.
	 */
	diskp->dk_label = malloc(sizeof(struct disklabel));
//	diskp->dk_cpulabel = malloc(sizeof(struct cpu_disklabel));
//	if ((diskp->dk_label == NULL) || (diskp->dk_cpulabel == NULL)) {
	if (diskp->dk_label == NULL) {
		panic("disk_attach: can't allocate storage for disklabel");
	}

	memset(diskp->dk_label, 0, sizeof(struct disklabel));
//	memset(diskp->dk_cpulabel, 0, sizeof(struct cpu_disklabel));

	/*
	 * Set up the stats collection.
	 */
	diskp->dk_stats = iostat_alloc(IOSTAT_DISK, diskp, diskp->dk_name);
}

static void disk_detach0(
	struct disk *diskp)
{
	/*
	 * Remove from the drivelist.
	 */
	iostat_free(diskp->dk_stats);

	/*
	 * Release the disk-info dictionary.
	 */
//	if (diskp->dk_info) {
//		prop_object_release(diskp->dk_info);
//		diskp->dk_info = NULL;
//	}

	/*
	 * Free the space used by the disklabel structures.
	 */
	free(diskp->dk_label);
//	free(diskp->dk_cpulabel);
}

//===================================== PUBLIC =====================================================

/*
 * Attach a disk.
 */
void disk_attach(struct disk *diskp)
{
	disk_init0(diskp);
	disk_attach0(diskp);
}

/*
 * Detach a disk.
 */
void disk_detach(
	struct disk *diskp)
{
	(void) lockmgr(&diskp->dk_openlock, LK_DRAIN, NULL);
	disk_detach0(diskp);
}

/*
 * Mark the disk as busy for metrics collection.
 */
void disk_busy(
	struct disk *diskp)
{
	iostat_busy(diskp->dk_stats);
}

/*
 * Finished disk operations, gather metrics.
 */
void disk_unbusy(
	struct disk *diskp, 
	long bcount, 
	int read)
{
	iostat_unbusy(diskp->dk_stats, bcount, read);
}

/*
 * Set the physical blocksize of a disk, in bytes.
 * Only necessary if blocksize != DEV_BSIZE.
 */
void disk_blocksize(
	struct disk *diskp, 
	int blocksize)
{
	diskp->dk_blkshift = DK_BSIZE2BLKSHIFT(blocksize);
	diskp->dk_byteshift = DK_BSIZE2BYTESHIFT(blocksize);
}

/*
 * Compute checksum for disk label.
 */
u_int dkcksum(
	struct disklabel *lp)
{
	u_short *start, *end;
	u_short sum = 0;

	start = (u_short *)lp;
	end = (u_short *)&lp->d_partitions[lp->d_npartitions];
	while (start < end) {
		sum ^= *start++;
	}
	return (sum);
}

/*
 * disk_ioctl --
 *	Generic disk ioctl handling.
 */
int disk_ioctl(
	struct disk *diskp, 
	u_long cmd, 
	caddr_t data, 
	int flag,
	struct lwp *l)
{
	int error;

	switch (cmd) {
/*	case DIOCGDISKINFO:
	    {
		struct plistref *pref = (struct plistref *) data;

		if (diskp->dk_info == NULL) {
			error = ENOTSUP;
		}
		else {
			error = prop_dictionary_copyout_ioctl(pref, cmd, diskp->dk_info);
		}
		break;
	    }
*/
	default:
		error = EPASSTHROUGH;
	}

	return (error);
}

/*
 * Disk error is the preface to plaintive error messages
 * about failing disk transfers.  It prints messages of the form

hp0g: hard error reading fsbn 12345 of 12344-12347 (hp0 bn %d cn %d tn %d sn %d)

 * if the offset of the error in the transfer and a disk label
 * are both available.  blkdone should be -1 if the position of the error
 * is unknown; the disklabel pointer may be null from drivers that have not
 * been converted to use them.  The message is printed with printf
 * if pri is LOG_PRINTF, otherwise it uses log at the specified priority.
 * The message should be completed (with at least a newline) with printf
 * or addlog, respectively.  There is no trailing space.
 */
#ifndef PRIdaddr
#define PRIdaddr PRId64
#endif
void diskerr(
	const struct buf *bp, 
	const char *dname, 
	const char *what, 
	int pri,
    int blkdone, 
    const struct disklabel *lp)
{
	int unit = DISKUNIT(bp->b_dev), part = DISKPART(bp->b_dev);
	void (*pr)(const char *, ...);
	char partname = 'a' + part;
	daddr_t sn;

	if (/*CONSTCOND*/0) {
		/* Compiler will error this is the format is wrong... */
		printf("%" PRIdaddr, bp->b_blkno);
	}

//	if (pri != LOG_PRINTF) {
//		static const char fmt[] = "";
//		log(pri, fmt);
//		pr = addlog;
//	} 
//	else {
		pr = printf;
//	}
	(*pr)("%s%d%c: %s %sing fsbn ", dname, unit, partname, what, bp->b_flags & B_READ ? "read" : "writ");
	sn = bp->b_blkno;
	if (bp->b_bcount <= DEV_BSIZE) {
		(*pr)("%" PRIdaddr, sn);
	}
	else {
		if (0 <= blkdone) {
			sn += blkdone;
			(*pr)("%" PRIdaddr " of ", sn);
		}
		(*pr)("%" PRIdaddr "-%" PRIdaddr "", bp->b_blkno, bp->b_blkno + (bp->b_bcount - 1) / DEV_BSIZE);
	}
	if (lp && (0 <= blkdone || (uint32_t) bp->b_bcount <= lp->d_secsize)) {
		sn += lp->d_partitions[part].p_offset;
		(*pr)(" (%s%d bn %" PRIdaddr "; cn %" PRIdaddr "", dname, unit, sn, sn / lp->d_secpercyl);
		sn %= lp->d_secpercyl;
		(*pr)(" tn %" PRIdaddr " sn %" PRIdaddr ")", sn / lp->d_nsectors, sn % lp->d_nsectors);
	}
}

/*
 * Determine the size of the transfer, and make sure it is
 * within the boundaries of the partition. Adjust transfer
 * if needed, and signal errors or early completion.
 */
int bounds_check_with_label(
	struct disk *dk, 
	struct buf *bp, 
	int wlabel)
{
	struct disklabel *lp = dk->dk_label;
	struct partition *p = lp->d_partitions + DISKPART(bp->b_dev);
	uint64_t p_size, p_offset, labelsector;
	int64_t sz;
	struct uio *uio = bp->uio;
	uint64_t uioOffset = howmany(uio->uio_offset, DEV_BSIZE);

	/* Protect against division by zero. XXX: Should never happen?!?! */
	if (lp->d_secpercyl == 0) {
		bp->b_error = EINVAL;
		goto bad;
	}

	p_size = p->p_size << dk->dk_blkshift;
	p_offset = p->p_offset << dk->dk_blkshift;
#if RAW_PART == 3
	labelsector = lp->d_partitions[2].p_offset;
#else
	labelsector = lp->d_partitions[RAW_PART].p_offset;
#endif
	labelsector = (labelsector + dk->dk_labelsector) << dk->dk_blkshift;

//	sz = howmany(bp->b_bcount, DEV_BSIZE);
	sz = howmany(uio->uio_iovcnt * PAGE_SIZE, DEV_BSIZE);
//	if ((bp->b_blkno + sz) > p_size) {
	if (p_size < (uio->uio_offset / DEV_BSIZE + sz)) {
//		sz = p_size - bp->b_blkno;
		sz = p_size - uio->uio_offset / DEV_BSIZE;
		if (sz == 0) {
			/* If exactly at end of disk, return EOF. */
//			bp->b_resid = bp->b_bcount;
			uio->uio_resid = uio->uio_iovcnt * PAGE_SIZE;
			return (0);
		}
		if (sz < 0) {
			/* If past end of disk, return EINVAL. */
			bp->b_error = EINVAL;
			goto bad;
		}
//		/* Otherwise, truncate request. */
//		bp->b_bcount = sz << DEV_BSHIFT;
	}

	/* Overwriting disk label? */
//	if (bp->b_blkno + p_offset <= labelsector &&
//		bp->b_blkno + p_offset + sz > labelsector &&
	if (uioOffset + p_offset <= labelsector &&
		uioOffset + p_offset + sz > labelsector &&
		(bp->b_flags & B_READ) == 0 && !wlabel) {
		bp->b_error = EROFS;
		goto bad;
	}

	/* calculate cylinder for disksort to order transfers with */
//	bp->b_cylinder = (bp->b_blkno + p->p_offset) / (lp->d_secsize / DEV_BSIZE) / lp->d_secpercyl;
	return (1);

bad:
	bp->b_flags |= B_ERROR;
	return (-1);
}

/*
 * Searches the iostatlist for the disk corresponding to the
 * name provided.
 */
struct disk *disk_find(
	const char *name)
{
	struct io_stats *stat;

	stat = iostat_find(name);

	if ((stat != NULL) && (stat->io_type == IOSTAT_DISK)) {
		return stat->io_parent;
	}

	return (NULL);
}

/*
 * Bounds checking against the media size, used for the raw partition.
 * The sector size passed in should currently always be DEV_BSIZE,
 * and the media size the size of the device in DEV_BSIZE sectors.
 */
int bounds_check_with_mediasize(
	struct buf *bp, 
	int secsize, 
	uint64_t mediasize)
{
	int64_t sz;

	sz = howmany(bp->b_bcount, secsize);

	if (mediasize < bp->b_blkno + sz) {
		sz = mediasize - bp->b_blkno;
		if (sz == 0) {
			/* If exactly at end of disk, return EOF. */
			bp->b_resid = bp->b_bcount;
			goto done;
		}
		if (sz < 0) {
			/* If past end of disk, return EINVAL. */
			bp->b_error = EINVAL;
			goto bad;
		}
		/* Otherwise, truncate request. */
		bp->b_bcount = sz << DEV_BSHIFT;
	}

	return 1;

bad:
	bp->b_flags |= B_ERROR;
done:
	return 0;
}

/*
 * struct deviceǥХֹ򻻽Ф
 * return : error number
 */
int diskComputeDev(
	struct device *dv,
	const int part,		// ѡƥֹ
	dev_t *o_devp)
{
	int majdev;

	majdev = devsw_name2blk(dv->dv_xname, NULL, 0);
	if (majdev == -1) {
		return (ENODEV);
	}

	if (part == 0) {
		*o_devp = makedev(majdev, device_unit(dv));
	}
	else {
		*o_devp = MAKEDISKDEV(majdev, device_unit(dv), part);
	}

	return (0);
}
