/*	$NetBSD: subr_iostat.c,v 1.11 2006/08/23 17:19:32 christos Exp $	*/
/*	NetBSD: subr_disk.c,v 1.69 2005/05/29 22:24:15 christos Exp	*/

/*-
 * Copyright (c) 1996, 1997, 1999, 2000 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe of the Numerical Aerospace Simulation Facility,
 * NASA Ames Research Center.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the NetBSD
 *	Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1982, 1986, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)ufs_disksubr.c	8.5 (Berkeley) 1/21/94
 *
 * 2009: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/systm.h>
#include <sys/timevar.h>
#include <sys/lock.h>
#include <stdlib.h>
#include <string.h>
#include <sys/iostat.h>

//=====================================  ===================================================

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static struct iostatlist_head iostatlist = TAILQ_HEAD_INITIALIZER(iostatlist);
static int iostat_count;		/* number of drives in global drivelist */
static struct simplelock iostatlist_slock = SIMPLELOCK_INITIALIZER;

//===================================== PUBLIC =====================================================

/*
 * Allocate and initialise memory for the i/o statistics.
 */
struct io_stats *iostat_alloc(
	int32_t type, 
	void *parent, 
	const char *name)
{
	struct io_stats *stats;

	stats = malloc(sizeof(struct io_stats));
	if (stats == NULL) {
		panic("iostat_alloc: cannot allocate memory for stats buffer");
	}

	stats->io_type = type;
	stats->io_parent = parent;
	(void)strlcpy(stats->io_name, name, sizeof(stats->io_name));

	/*
	 * Set the attached timestamp.
	 */
	getmicrouptime(&stats->io_attachtime);

	/*
	 * Link into the drivelist.
	 */
	simple_lock(&iostatlist_slock);
	TAILQ_INSERT_TAIL(&iostatlist, stats, io_link);
	iostat_count++;
	simple_unlock(&iostatlist_slock);

	return stats;
}

/*
 * Remove i/o from stats collection.
 */
void iostat_free(
	struct io_stats *stats)
{
	/*
	 * Remove from the iostat list.
	 */
	if (iostat_count == 0) {
		panic("iostat_free: iostat_count == 0");
	}
	simple_lock(&iostatlist_slock);
	TAILQ_REMOVE(&iostatlist, stats, io_link);
	iostat_count--;
	simple_unlock(&iostatlist_slock);
	free(stats);
}

/*
 * Increment a iostat busy counter.  If the counter is going from
 * 0 to 1, set the timestamp.
 */
void iostat_busy(
	struct io_stats *stats)
{
	if (stats->io_busy++ == 0) {
		getmicrouptime(&stats->io_timestamp);
	}
}

/*
 * Decrement a iostat busy counter, increment the byte count, total busy
 * time, and reset the timestamp.
 */
void iostat_unbusy(
	struct io_stats *stats, 
	long bcount, 
	int read)
{
	struct timeval dv_time, diff_time;

	if (stats->io_busy-- == 0) {
		printf("%s: busy < 0\n", stats->io_name);
		panic("iostat_unbusy");
	}

	getmicrouptime(&dv_time);

	timersub(&dv_time, &stats->io_timestamp, &diff_time);
	timeradd(&stats->io_time, &diff_time, &stats->io_time);

	stats->io_timestamp = dv_time;
	if (bcount > 0) {
		if (read) {
			stats->io_rbytes += bcount;
			stats->io_rxfer++;
		} 
		else {
			stats->io_wbytes += bcount;
			stats->io_wxfer++;
		}
	}
}

/*
 * Searches the iostatlist for the iostat corresponding to the
 * name provided.
 */
struct io_stats *iostat_find(
	const char *name)
{
	struct io_stats *iostatp;

	KASSERT(name != NULL);

	simple_lock(&iostatlist_slock);
	TAILQ_FOREACH(iostatp, &iostatlist, io_link) {
		if (strcmp(iostatp->io_name, name) == 0) {
			break;
		}
	}
	simple_unlock(&iostatlist_slock);

	return iostatp;
}

