/*	$NetBSD: subr_prf.c,v 1.103 2006/09/03 17:06:36 martin Exp $	*/

/*-
 * Copyright (c) 1986, 1988, 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)subr_prf.c	8.4 (Berkeley) 5/4/95
 *
 * 2010: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <machine/int_mwgwtypes.h>
#include <machine/stdarg.h>
#include <stddef.h>
#include <string.h>
#include <sys/kprintf.h>

//=====================================  ===================================================

#define KPRINTF_BUFSIZE		(sizeof(quad_t) * NBBY / 3 + 2)

/*
 * macros for converting digits to letters and vice versa
 */
#define	to_digit(c)	((c) - '0')
#define is_digit(c)	((unsigned)to_digit(c) <= 9)
#define	to_char(n)	((n) + '0')

/*
 * flags used during conversion.
 */
#define	ALT			0x001		/* alternate form */
#define	HEXPREFIX	0x002		/* add 0x or 0X prefix */
#define	LADJUST		0x004		/* left adjustment */
#define	LONGDBL		0x008		/* long double; unimplemented */
#define	LONGINT		0x010		/* long integer */
#define	QUADINT		0x020		/* quad integer */
#define	SHORTINT	0x040		/* short integer */
#define	MAXINT		0x080		/* intmax_t */
#define	PTRINT		0x100		/* intptr_t */
#define	SIZEINT		0x200		/* size_t */
#define	ZEROPAD		0x400		/* zero (as opposed to blank) pad */
#define FPT			0x800		/* Floating point number */

/*
 * To extend shorts properly, we need both signed and unsigned
 * argument extraction methods.
 */
#define	SARG() \
	(flags&MAXINT ? va_arg(ap, intmax_t) : \
	    flags & PTRINT ? va_arg(ap, intptr_t) : \
	    flags & SIZEINT ? va_arg(ap, ssize_t) : /* XXX */ \
	    flags & QUADINT ? va_arg(ap, quad_t) : \
	    flags & LONGINT ? va_arg(ap, long) : \
	    flags & SHORTINT ? (long)(short)va_arg(ap, int) : \
	    (long)va_arg(ap, int))
#define	UARG() \
	(flags&MAXINT ? va_arg(ap, uintmax_t) : \
	    flags & PTRINT ? va_arg(ap, uintptr_t) : \
	    flags & SIZEINT ? va_arg(ap, size_t) : \
	    flags & QUADINT ? va_arg(ap, u_quad_t) : \
	    flags & LONGINT ? va_arg(ap, u_long) : \
	    flags & SHORTINT ? (u_long)(u_short)va_arg(ap, int) : \
	    (u_long)va_arg(ap, u_int))

#define KPRINTF_PUTCHAR(C) {							\
	if (oflags == TOBUFONLY) {							\
		if ((vp != NULL) && (sbuf == tailp)) {			\
			ret += 1;		/* indicate error */		\
			goto overflow;								\
		}												\
		*sbuf++ = (C);									\
	} 													\
	else {												\
		putchar(C);										\
	}													\
}

//===================================== Х륤ݡ =======================================

extern int sprintf(char *, const char *, ...);
extern int snprintf(char *s, size_t n, const char *format, ...);
extern int snprintf(char *s, size_t n, const char *format, ...);
extern int writeConsole(const char*, size_t);

//===================================== PRIVATE ====================================================

static const char hexdigits[] = "0123456789abcdef";
static const char HEXDIGITS[] = "0123456789ABCDEF";

static void putchar(
	int chr)
{
	writeConsole((char*) &chr, 1);
}

//===================================== PUBLIC =====================================================
/*
 * kprintf: scaled down version of printf(3).
 *
 * this version based on vfprintf() from libc which was derived from software contributed to Berkeley by Chris Torek.
 *
 * NOTE: The kprintf mutex must be held if we're going TOBUF or TOCONS!
 */

/*
 * Guts of kernel printf.  Note, we already expect to be in a mutex!
 */
int kprintf(
	const char *fmt0, 
	int oflags, 
	void *vp, 
	char *sbuf, 
	va_list ap)
{
	const char *fmt;		/* format string */
	int ch;					/* character from fmt */
	int n;					/* handy integer (short term usage) */
	char *cp;				/* handy char pointer (short term usage) */
	int flags;				/* flags as above */
	int ret;				/* return value accumulator */
	int width;				/* width from format (%8d), or 0 */
	int prec;				/* precision from format (%.3d), or -1 */
	char sign;				/* sign prefix (' ', '+', '-', or \0) */

	u_quad_t _uquad;		/* integer arguments %[diouxX] */
	enum { OCT, DEC, HEX } base;/* base for [diouxX] conversion */
	int dprec;				/* a copy of prec if [diouxX], 0 otherwise */
	int realsz;				/* field size expanded by dprec */
	int size;				/* size of converted field or string */
	const char *xdigs;		/* digits for [xX] conversion */
	char bf[KPRINTF_BUFSIZE]; /* space for %c, %[diouxX] */
	char *tailp;			/* tail pointer for snprintf */

	tailp = NULL;			/* XXX: shutup gcc */
	if (oflags == TOBUFONLY && (vp != NULL)) {
		tailp = *(char **)vp;
	}

	cp = NULL;			/* XXX: shutup gcc */
	size = 0;			/* XXX: shutup gcc */
	fmt = fmt0;
	ret = 0;
	xdigs = NULL;		/* XXX: shut up gcc warning */

	/*
	 * Scan the format for conversions (`%' character).
	 */
	for (;;) {
		while (*fmt != '%' && *fmt) {
			ret++;
			KPRINTF_PUTCHAR(*fmt++);
		}
		if (*fmt == 0) {
			goto done;
		}

		fmt++;		/* skip over '%' */

		flags = 0;
		dprec = 0;
		width = 0;
		prec = -1;
		sign = '\0';

rflag:	ch = *fmt++;
reswitch:
	switch (ch) {
		case ' ':
			/*
			 * ``If the space and + flags both appear, the space
			 * flag will be ignored.''
			 *	-- ANSI X3J11
			 */
			if (!sign) {
				sign = ' ';
			}
			goto rflag;
		case '#':
			flags |= ALT;
			goto rflag;
		case '*':
			/*
			 * ``A negative field width argument is taken as a
			 * - flag followed by a positive field width.''
			 *	-- ANSI X3J11
			 * They don't exclude field widths read from args.
			 */
			if (0 <= (width = va_arg(ap, int))) {
				goto rflag;
			}
			width = -width;
			/* FALLTHROUGH */
		case '-':
			flags |= LADJUST;
			goto rflag;
		case '+':
			sign = '+';
			goto rflag;
		case '.':
			if ((ch = *fmt++) == '*') {
				n = va_arg(ap, int);
				prec = n < 0 ? -1 : n;
				goto rflag;
			}
			n = 0;
			while (is_digit(ch)) {
				n = 10 * n + to_digit(ch);
				ch = *fmt++;
			}
			prec = n < 0 ? -1 : n;
			goto reswitch;
		case '0':
			/*
			 * ``Note that 0 is taken as a flag, not as the
			 * beginning of a field width.''
			 *	-- ANSI X3J11
			 */
			flags |= ZEROPAD;
			goto rflag;
		case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			n = 0;
			do {
				n = 10 * n + to_digit(ch);
				ch = *fmt++;
			} while (is_digit(ch));
			width = n;
			goto reswitch;
		case 'h':
			flags |= SHORTINT;
			goto rflag;
		case 'j':
			flags |= MAXINT;
			goto rflag;
		case 'l':
			if (*fmt == 'l') {
				fmt++;
				flags |= QUADINT;
			} 
			else {
				flags |= LONGINT;
			}
			goto rflag;
		case 'q':
			flags |= QUADINT;
			goto rflag;
		case 't':
			flags |= PTRINT;
			goto rflag;
		case 'z':
			flags |= SIZEINT;
			goto rflag;
		case 'c':
			*(cp = bf) = va_arg(ap, int);
			size = 1;
			sign = '\0';
			break;
		case 'D':
			flags |= LONGINT;
			/*FALLTHROUGH*/
		case 'd':
		case 'i':
			_uquad = SARG();
			if ((quad_t)_uquad < 0) {
				_uquad = -_uquad;
				sign = '-';
			}
			base = DEC;
			goto number;
		case 'n':
			if (flags & MAXINT) {
				*va_arg(ap, intmax_t *) = ret;
			}
			else if (flags & PTRINT) {
				*va_arg(ap, intptr_t *) = ret;
			}
			else if (flags & SIZEINT) {
				*va_arg(ap, ssize_t *) = ret;
			}
			else if (flags & QUADINT) {
				*va_arg(ap, quad_t *) = ret;
			}
			else if (flags & LONGINT) {
				*va_arg(ap, long *) = ret;
			}
			else if (flags & SHORTINT) {
				*va_arg(ap, short *) = ret;
			}
			else {
				*va_arg(ap, int *) = ret;
			}
			continue;	/* no output */
		case 'O':
			flags |= LONGINT;
			/*FALLTHROUGH*/
		case 'o':
			_uquad = UARG();
			base = OCT;
			goto nosign;
		case 'p':
			/*
			 * ``The argument shall be a pointer to void.  The
			 * value of the pointer is converted to a sequence
			 * of printable characters, in an implementation-
			 * defined manner.''
			 *	-- ANSI X3J11
			 */
			/* NOSTRICT */
			_uquad = (u_long)va_arg(ap, void *);
			base = HEX;
			xdigs = hexdigits;
			flags |= HEXPREFIX;
			ch = 'x';
			goto nosign;
		case 's':
			if ((cp = va_arg(ap, char *)) == NULL) {
				/*XXXUNCONST*/
				cp = __UNCONST("(null)");
			}
			if (0 <= prec) {
				/*
				 * can't use strlen; can only look for the
				 * NUL in the first `prec' characters, and
				 * strlen() will go further.
				 */
				char *p = memchr(cp, 0, prec);

				if (p != NULL) {
					size = p - cp;
					if (prec < size) {
						size = prec;
					}
				} 
				else {
					size = prec;
				}
			} 
			else {
				size = strlen(cp);
			}
			sign = '\0';
			break;
		case 'U':
			flags |= LONGINT;
			/*FALLTHROUGH*/
		case 'u':
			_uquad = UARG();
			base = DEC;
			goto nosign;
		case 'X':
			xdigs = HEXDIGITS;
			goto hex;
		case 'x':
			xdigs = hexdigits;
hex:		_uquad = UARG();
			base = HEX;
			/* leading 0x/X only if non-zero */
			if (flags & ALT && _uquad != 0) {
				flags |= HEXPREFIX;
			}

			/* unsigned conversions */
nosign:		sign = '\0';
			/*
			 * ``... diouXx conversions ... if a precision is
			 * specified, the 0 flag will be ignored.''
			 *	-- ANSI X3J11
			 */
number:		if (0 <= (dprec = prec)) {
				flags &= ~ZEROPAD;
			}

			/*
			 * ``The result of converting a zero value with an
			 * explicit precision of zero is no characters.''
			 *	-- ANSI X3J11
			 */
			cp = bf + KPRINTF_BUFSIZE;
			if (_uquad != 0 || prec != 0) {
				/*
				 * Unsigned mod is hard, and unsigned mod
				 * by a constant is easier than that by
				 * a variable; hence this switch.
				 */
				switch (base) {
				case OCT:
					do {
						*--cp = to_char(_uquad & 7);
						_uquad >>= 3;
					} while (_uquad);
					/* handle octal leading 0 */
					if (flags & ALT && *cp != '0') {
						*--cp = '0';
					}
					break;

				case DEC:
					/* many numbers are 1 digit */
					while (10 <= _uquad) {
						*--cp = to_char(_uquad % 10);
						_uquad /= 10;
					}
					*--cp = to_char(_uquad);
					break;

				case HEX:
					do {
						*--cp = xdigs[_uquad & 15];
						_uquad >>= 4;
					} while (_uquad);
					break;

				default:
					/*XXXUNCONST*/
					cp = __UNCONST("bug in kprintf: bad base");
					size = strlen(cp);
					goto skipsize;
				}
			}
			size = bf + KPRINTF_BUFSIZE - cp;
		skipsize:
			break;
		default:	/* "%?" prints ?, unless ? is NUL */
			if (ch == '\0') {
				goto done;
			}
			/* pretend it was %c with argument ch */
			cp = bf;
			*cp = ch;
			size = 1;
			sign = '\0';
			break;
		}

		/*
		 * All reasonable formats wind up here.  At this point, `cp'
		 * points to a string which (if not flags&LADJUST) should be
		 * padded out to `width' places.  If flags&ZEROPAD, it should
		 * first be prefixed by any sign or other prefix; otherwise,
		 * it should be blank padded before the prefix is emitted.
		 * After any left-hand padding and prefixing, emit zeroes
		 * required by a decimal [diouxX] precision, then print the
		 * string proper, then emit zeroes required by any leftover
		 * floating precision; finally, if LADJUST, pad with blanks.
		 *
		 * Compute actual size, so we know how much to pad.
		 * size excludes decimal prec; realsz includes it.
		 */
		realsz = dprec > size ? dprec : size;
		if (sign) {
			realsz++;
		}
		else if (flags & HEXPREFIX) {
			realsz+= 2;
		}

		/* adjust ret */
		ret += width > realsz ? width : realsz;

		/* right-adjusting blank padding */
		if ((flags & (LADJUST|ZEROPAD)) == 0) {
			n = width - realsz;
			while (0 < n--) {
				KPRINTF_PUTCHAR(' ');
			}
		}

		/* prefix */
		if (sign) {
			KPRINTF_PUTCHAR(sign);
		} 
		else if (flags & HEXPREFIX) {
			KPRINTF_PUTCHAR('0');
			KPRINTF_PUTCHAR(ch);
		}

		/* right-adjusting zero padding */
		if ((flags & (LADJUST|ZEROPAD)) == ZEROPAD) {
			n = width - realsz;
			while (0 < n--) {
				KPRINTF_PUTCHAR('0');
			}
		}

		/* leading zeroes from decimal precision */
		n = dprec - size;
		while (0 < n--) {
			KPRINTF_PUTCHAR('0');
		}

		/* the string or number proper */
		while (size--) {
			KPRINTF_PUTCHAR(*cp++);
		}
		/* left-adjusting padding (always blank) */
		if (flags & LADJUST) {
			n = width - realsz;
			while (0 < n--) {
				KPRINTF_PUTCHAR(' ');
			}
		}
	}

done:
	if ((oflags == TOBUFONLY) && (vp != NULL)) {
		*(char **)vp = sbuf;
	}
//	(*v_flush)();
overflow:
	return (ret);
	/* NOTREACHED */
}

/*
 * vprintf: print a message to the console and the log [already have va_alist]
 */
void vprintf(
	const char *fmt, 
	va_list ap)
{
//	int s;

	KPRINTF_MUTEX_ENTER(s);
	kprintf(fmt, TOCONS | TOLOG, NULL, NULL, ap);
	KPRINTF_MUTEX_EXIT(s);
}

/*
 * bitmask_snprintf: print an interpreted bitmask to a buffer
 *
 * => returns pointer to the buffer
 */
char *bitmask_snprintf(
	u_quad_t val, 
	const char *p, 
	char *bf, 
	size_t buflen)
{
	char *bp, *q;
	size_t left;
	const char *sbase;
	char snbuf[KPRINTF_BUFSIZE];
	int base, bit, ch, len, sep;
	u_quad_t field;

	bp = bf;
	memset(bf, 0, buflen);

	/*
	 * Always leave room for the trailing NULL.
	 */
	left = buflen - 1;

	/*
	 * Print the value into the buffer.  Abort if there's not
	 * enough room.
	 */
	if (buflen < KPRINTF_BUFSIZE) {
		return (bf);
	}

	ch = *p++;
	base = ch != '\177' ? ch : *p++;
	sbase = base == 8 ? "%qo" : base == 10 ? "%qd" : base == 16 ? "%qx" : 0;
	if (sbase == 0) {
		return (bf);	/* punt if not oct, dec, or hex */
	}

	snprintf(snbuf, sizeof(snbuf), sbase, val);
	for (q = snbuf ; *q ; q++) {
		*bp++ = *q;
		left--;
	}

	/*
	 * If the value we printed was 0 and we're using the old-style format,
	 * or if we don't have room for "<x>", we're done.
	 */
	if (((val == 0) && (ch != '\177')) || left < 3) {
		return (bf);
	}

#define PUTBYTE(b, c, l) do {		\
	*(b)++ = (c);					\
	if (--(l) == 0)					\
		goto out;					\
} while (/*CONSTCOND*/ 0)

#define PUTSTR(b, p, l) do {		\
	int c;							\
	while ((c = *(p)++) != 0) {		\
		*(b)++ = c;					\
		if (--(l) == 0)				\
			goto out;				\
	}								\
} while (/*CONSTCOND*/ 0)

	/*
	 * Chris Torek's new bitmask format is identified by a leading \177
	 */
	sep = '<';
	if (ch != '\177') {
		/* old (standard) format. */
		for (;(bit = *p++) != 0;) {
			if (val & (1 << (bit - 1))) {
				PUTBYTE(bp, sep, left);
				for (; (ch = *p) > ' '; ++p) {
					PUTBYTE(bp, ch, left);
				}
				sep = ',';
			} 
			else {
				for (; *p > ' '; ++p) {
					continue;
				}
			}
		}
	} 
	else {
		/* new quad-capable format; also does fields. */
		field = val;
		while ((ch = *p++) != '\0') {
			bit = *p++;	/* now 0-origin */
			switch (ch) {
			case 'b':
				if (((u_int)(val >> bit) & 1) == 0) {
					goto skip;
				}
				PUTBYTE(bp, sep, left);
				PUTSTR(bp, p, left);
				sep = ',';
				break;
			case 'f':
			case 'F':
				len = *p++;	/* field length */
				field = (val >> bit) & ((1ULL << len) - 1);
				if (ch == 'F') {	/* just extract */
					break;
				}
				PUTBYTE(bp, sep, left);
				sep = ',';
				PUTSTR(bp, p, left);
				PUTBYTE(bp, '=', left);
				sprintf(snbuf, sbase, field);
				q = snbuf; PUTSTR(bp, q, left);
				break;
			case '=':
			case ':':
				/*
				 * Here "bit" is actually a value instead,
				 * to be compared against the last field.
				 * This only works for values in [0..255],
				 * of course.
				 */
				if ((int)field != bit) {
					goto skip;
				}
				if (ch == '=') {
					PUTBYTE(bp, '=', left);
				}
				PUTSTR(bp, p, left);
				break;
			default:
			skip:
				while (*p++ != '\0') {
					continue;
				}
				break;
			}
		}
	}
	if (sep != '<') {
		PUTBYTE(bp, '>', left);
	}

out:
	return (bf);

#undef PUTBYTE
#undef PUTSTR
}

/*
 * printf: print a message to the console and the log
 */
void printf(
	const char *fmt, ...)
{
	va_list ap;
//	int s;

	KPRINTF_MUTEX_ENTER(s);

	va_start(ap, fmt);
	kprintf(fmt, TOCONS | TOLOG, NULL, NULL, ap);
	va_end(ap);

	KPRINTF_MUTEX_EXIT(s);
}

/*
 * aprint_normal: Send to console unless AB_QUIET.  Always goes
 * to the log.
 */
void aprint_normal(
	const char *fmt, ...)
{
	va_list ap;
//	int s;
	int flags = TOLOG;

	KPRINTF_MUTEX_ENTER(s);

	va_start(ap, fmt);
	kprintf(fmt, flags, NULL, NULL, ap);
	va_end(ap);

	KPRINTF_MUTEX_EXIT(s);
}

void aprint_error(
	const char *fmt, ...)
{
	va_list ap;
//	int s;
	int flags = TOLOG;

	KPRINTF_MUTEX_ENTER(s);

	va_start(ap, fmt);
	kprintf(fmt, flags, NULL, NULL, ap);
	va_end(ap);

	KPRINTF_MUTEX_EXIT(s);
}

void aprint_naive(
	const char *str, ...)
{
	// ⤷ʤ
}

/*
 * panic: handle an unresolvable fatal error
 *
 * prints "panic: <message>" and reboots.   if called twice (i.e. recursive
 * call) we avoid trying to sync the disk and just reboot (to avoid
 * recursive panics).
 */

void panic(
	const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	printf("panic: ");
	vprintf(fmt, ap);
	printf("\n");
	va_end(ap);

	exit(0);
}
