/*	$NetBSD: wsmux.c,v 1.46 2006/11/16 01:33:31 christos Exp $	*/

/*
 * Copyright (c) 1998, 2005 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Author: Lennart Augustsson <lennart@augustsson.net>
 *         Carlstedt Research & Technology
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * 2010: modified by minoru murashima.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/fcntl.h>
#include <sys/queue.h>
#include <sys/device.h>
#include <sys/ttycom.h>
#include <sys/filio.h>
#include <unistd.h>
#include <stddef.h>
#include <string.h>
#include <wscons/wsconsio.h>
#include <wscons/wsksymdef.h>
#include <wscons/wsmuxvar.h>

//=====================================  ===================================================

#define WSMUXDEV(n) ((n) & 0x7f)

//===================================== Х륤ݡ =======================================

extern int snprintf(char *s, size_t n,const char *format, ...);

//===================================== PRIVATE ====================================================

/* Keep track of all muxes that have been allocated */
static int nwsmux = 0;
static struct wsmux_softc **wsmuxdevs;

/*
 * Add mux unit as a child to muxsc.
 */
static int wsmux_add_mux(
	int unit, 
	struct wsmux_softc *muxsc)
{
	struct wsmux_softc *sc, *m;

	sc = wsmux_getmux(unit);
	if (sc == NULL) {
		return (ENXIO);
	}

	if (sc->sc_base.me_parent != NULL || sc->sc_base.me_evp != NULL) {
		return (EBUSY);
	}

	/* The mux we are adding must not be an ancestor of itself. */
	for (m = muxsc; m != NULL ; m = m->sc_base.me_parent) {
		if (m == sc) {
			return (EINVAL);
		}
	}

	return (wsmux_attach_sc(muxsc, &sc->sc_base));
}

/* Common part of opening a mux. */
static void wsmux_do_open(
	struct wsmux_softc *sc, 
	struct wseventvar *evar)
{
	struct wsevsrc *me;

	sc->sc_base.me_evp = evar; /* remember event variable, mark as open */

	/* Open all children. */
	CIRCLEQ_FOREACH(me, &sc->sc_cld, me_next) {
		/* ignore errors, failing children will not be marked open */
		(void)wsevsrc_open(me, evar);
	}
}

/* Common part of closing a mux. */
static void wsmux_do_close(
	struct wsmux_softc *sc)
{
	struct wsevsrc *me;

	/* Close all the children. */
	CIRCLEQ_FOREACH(me, &sc->sc_cld, me_next) {
		(void)wsevsrc_close(me);
		me->me_evp = NULL;
	}
}

/*
 * Open of a mux via the parent mux.
 */
static int wsmux_mux_open(
	struct wsevsrc *me, 
	struct wseventvar *evar)
{
	struct wsmux_softc *sc = (struct wsmux_softc *)me;

	wsmux_do_open(sc, evar);

	return (0);
}

/*
 * Close of a mux via the parent mux.
 */
static int wsmux_mux_close(
	struct wsevsrc *me)
{
	me->me_evp = NULL;
	wsmux_do_close((struct wsmux_softc *)me);
	return (0);
}

/*
 * ioctl of a mux via the parent mux, continuation of wsmuxioctl().
 */
static int wsmux_do_ioctl(
	struct device *dv, 
	u_long cmd, 
	caddr_t data, 
	int flag,
	struct lwp *lwp)
{
	struct wsmux_softc *sc = (struct wsmux_softc *)dv;
	struct wsevsrc *me;
	int error, ok;
	int s, n;
	struct wseventvar *evar;
	struct wscons_event event;
	struct wsmux_device_list *l;

	switch (cmd) {
	case WSMUXIO_INJECTEVENT:
		/* Inject an event, e.g., from moused. */
		evar = sc->sc_base.me_evp;
		if (evar == NULL) {
			/* No event sink, so ignore it. */
			return (0);
		}

		s = spltty();
		event.type = ((struct wscons_event *)data)->type;
		event.value = ((struct wscons_event *)data)->value;
		error = wsevent_inject(evar, &event, 1);
		splx(s);

		return error;
	case WSMUXIO_ADD_DEVICE:
#define d ((struct wsmux_device *)data)
		switch (d->type) {
#if NWSMOUSE > 0
		case WSMUX_MOUSE:
			return (wsmouse_add_mux(d->idx, sc));
#endif
#if NWSKBD > 0
		case WSMUX_KBD:
			return (wskbd_add_mux(d->idx, sc));
#endif
		case WSMUX_MUX:
			return (wsmux_add_mux(d->idx, sc));
		default:
			return (EINVAL);
		}
	case WSMUXIO_REMOVE_DEVICE:
		/* Locate the device */
		CIRCLEQ_FOREACH(me, &sc->sc_cld, me_next) {
			if (me->me_ops->type == d->type &&
			    device_unit(&me->me_dv) == d->idx) {
				wsmux_detach_sc(me);
				return (0);
			}
		}
		return (EINVAL);
#undef d

	case WSMUXIO_LIST_DEVICES:
		l = (struct wsmux_device_list *)data;
		n = 0;
		CIRCLEQ_FOREACH(me, &sc->sc_cld, me_next) {
			if (WSMUX_MAXDEV <= n) {
				break;
			}
			l->devices[n].type = me->me_ops->type;
			l->devices[n].idx = device_unit(&me->me_dv);
			n++;
		}
		l->ndevices = n;
		return (0);
#ifdef WSDISPLAY_COMPAT_RAWKBD
	case WSKBDIO_SETMODE:
		sc->sc_rawkbd = *(int *)data;
		break;
#endif
	case FIONBIO:
		return (0);

	case FIOASYNC:
		evar = sc->sc_base.me_evp;
		if (evar == NULL) {
			return (EINVAL);
		}
		evar->async = *(int *)data != 0;
		return (0);
	case FIOSETOWN:
		evar = sc->sc_base.me_evp;
		if (evar == NULL) {
			return (EINVAL);
		}
//		if (-*(int *)data != evar->io->p_pgid && *(int *)data != evar->io->p_pid) {
		if (-*(int *)data != getpgrp() && *(int *)data != getpid()) {
			return (EPERM);
		}
		return (0);
	case TIOCSPGRP:
		evar = sc->sc_base.me_evp;
		if (evar == NULL) {
			return (EINVAL);
		}
//		if (*(int *)data != evar->io->p_pgid) {
		if (*(int *)data != getpgrp()) {
			return (EPERM);
		}
		return (0);
	default:
		break;
	}

	if (sc->sc_base.me_evp == NULL
#if NWSDISPLAY > 0
	    && sc->sc_base.me_dispdv == NULL
#endif
	    ) {
		return (EACCES);
	}

	/* Return 0 if any of the ioctl() succeeds, otherwise the last error */
	error = 0;
	ok = 0;
	CIRCLEQ_FOREACH(me, &sc->sc_cld, me_next) {
		error = wsevsrc_ioctl(me, cmd, data, flag, lwp);
		if (!error) {
			ok = 1;
		}
	}
	if (ok) {
		error = 0;
		if (cmd == WSKBDIO_SETENCODING) {
			sc->sc_kbd_layout = *((kbd_t *)data);
		}

	}

	return (error);
}

/*
 * Display ioctl() of a mux via the parent mux.
 */
static int wsmux_do_displayioctl(
	struct device *dv, 
	u_long cmd, 
	caddr_t data, 
	int flag,
	struct lwp *l)
{
	struct wsmux_softc *sc = (struct wsmux_softc *)dv;
	struct wsevsrc *me;
	int error, ok;

#ifdef WSDISPLAY_COMPAT_RAWKBD
	if (cmd == WSKBDIO_SETMODE) {
		sc->sc_rawkbd = *(int *)data;
	}
#endif

	/*
	 * Return 0 if any of the ioctl() succeeds, otherwise the last error.
	 * Return EPASSTHROUGH if no mux component accepts the ioctl.
	 */
	error = EPASSTHROUGH;
	ok = 0;
	CIRCLEQ_FOREACH(me, &sc->sc_cld, me_next) {
		if (me->me_ops->ddispioctl != NULL) {
			error = wsevsrc_display_ioctl(me, cmd, data, flag, l);
			if (!error) {
				ok = 1;
			}
		}
	}
	if (ok) {
		error = 0;
	}

	return (error);
}

#if NWSDISPLAY > 0
static int wsmux_set_display(
	struct wsmux_softc *sc, 
	struct device *displaydv)
{
	struct device *odisplaydv;
	struct wsevsrc *me;
	struct wsmux_softc *nsc = displaydv ? sc : NULL;
	int error, ok;

	odisplaydv = sc->sc_base.me_dispdv;
	sc->sc_base.me_dispdv = displaydv;

	if (displaydv) {
		printf("%s: connecting to %s\n", sc->sc_base.me_dv.dv_xname, displaydv->dv_xname);
	}
	ok = 0;
	error = 0;
	CIRCLEQ_FOREACH(me, &sc->sc_cld,me_next) {
		if (me->me_ops->dsetdisplay != NULL) {
			error = wsevsrc_set_display(me, &nsc->sc_base);
			if (!error) {
				ok = 1;
#ifdef WSDISPLAY_COMPAT_RAWKBD
				(void)wsevsrc_ioctl(me, WSKBDIO_SETMODE, &sc->sc_rawkbd, 0, 0);
#endif
			}
		}
	}
	if (ok) {
		error = 0;
	}

	if (displaydv == NULL) {
		printf("%s: disconnecting from %s\n", sc->sc_base.me_dv.dv_xname, odisplaydv->dv_xname);
	}

	return (error);
}

/*
 * Set display of a mux via the parent mux.
 */
static int wsmux_evsrc_set_display(
	struct device *dv, 
	struct wsevsrc *ame)
{
	struct wsmux_softc *muxsc = (struct wsmux_softc *)ame;
	struct wsmux_softc *sc = (struct wsmux_softc *)dv;
	struct device *displaydv = muxsc ? muxsc->sc_base.me_dispdv : NULL;

	if (displaydv != NULL) {
		if (sc->sc_base.me_dispdv != NULL) {
			return (EBUSY);
		}
	} 
	else {
		if (sc->sc_base.me_dispdv == NULL) {
			return (ENXIO);
		}
	}

	return wsmux_set_display(sc, displaydv);
}

#else
#define wsmux_evsrc_set_display NULL
#endif /* NWSDISPLAY > 0 */

static struct wssrcops wsmux_srcops = {
	WSMUX_MUX,
	wsmux_mux_open, 
	wsmux_mux_close, 
	wsmux_do_ioctl, 
	wsmux_do_displayioctl,
	wsmux_evsrc_set_display
};

//===================================== PUBLIC =====================================================

/* Attach me as a child to sc. */
int wsmux_attach_sc(
	struct wsmux_softc *sc, 
	struct wsevsrc *me)
{
	int error;

	if (sc == NULL) {
		return (EINVAL);
	}

	me->me_parent = sc;
	CIRCLEQ_INSERT_TAIL(&sc->sc_cld, me, me_next);

	error = 0;
#if NWSDISPLAY > 0
	if (sc->sc_base.me_dispdv != NULL) {
		/* This is a display mux, so attach the new device to it. */
		if (me->me_ops->dsetdisplay != NULL) {
			error = wsevsrc_set_display(me, &sc->sc_base);
			/* Ignore that the console already has a display. */
			if (error == EBUSY) {
				error = 0;
			}
			if (!error) {
#ifdef WSDISPLAY_COMPAT_RAWKBD
				(void)wsevsrc_ioctl(me, WSKBDIO_SETMODE, &sc->sc_rawkbd, 0, 0);
#endif
				if (sc->sc_kbd_layout != KB_NONE) {
					(void)wsevsrc_ioctl(me, WSKBDIO_SETENCODING, &sc->sc_kbd_layout, FWRITE, 0);
				}
			}
		}
	}
#endif
	if (sc->sc_base.me_evp != NULL) {
		/* Mux is open, so open the new subdevice */
		error = wsevsrc_open(me, sc->sc_base.me_evp);
	} 

	if (error) {
		me->me_parent = NULL;
		CIRCLEQ_REMOVE(&sc->sc_cld, me, me_next);
	}

	return (error);
}

/* Remove me from the parent. */
void wsmux_detach_sc(
	struct wsevsrc *me)
{
	struct wsmux_softc *sc = me->me_parent;

#if NWSDISPLAY > 0
	if (sc->sc_base.me_dispdv != NULL) {
		if (me->me_ops->dsetdisplay != NULL)
			/* ignore error, there's nothing we can do */
			(void)wsevsrc_set_display(me, NULL);
	} 
	else
#endif
		if (me->me_evp != NULL) {
		/* mux device is open, so close multiplexee */
		(void)wsevsrc_close(me);
	}

	CIRCLEQ_REMOVE(&sc->sc_cld, me, me_next);
	me->me_parent = NULL;
}

/* Create a new mux softc. */
struct wsmux_softc *wsmux_create(
	const char *name, 
	int unit)
{
	struct wsmux_softc *sc;

	/* XXX This is wrong -- should use autoconfiguraiton framework */

	sc = malloc(sizeof *sc);
	if (sc == NULL) {
		return (NULL);
	}
	memset(sc, 0, sizeof(*sc));
	CIRCLEQ_INIT(&sc->sc_cld);
	snprintf(sc->sc_base.me_dv.dv_xname, sizeof sc->sc_base.me_dv.dv_xname, "%s%d", name, unit);
	sc->sc_base.me_dv.dv_unit = unit;
	sc->sc_base.me_ops = &wsmux_srcops;
	sc->sc_kbd_layout = KB_NONE;
	return (sc);
}

/* Return mux n, create if necessary */
struct wsmux_softc *wsmux_getmux(
	int n)
{
	struct wsmux_softc *sc;
	int i;
	void *new;

	n = WSMUXDEV(n);	/* limit range */

	/* Make sure there is room for mux n in the table */
	if (nwsmux <= n) {
		i = nwsmux;
		nwsmux = n + 1;
		if (i != 0) {
			new = realloc(wsmuxdevs, nwsmux * sizeof (*wsmuxdevs));
		}
		else {
			new = malloc(nwsmux * sizeof (*wsmuxdevs));
		}
		if (new == NULL) {
			printf("wsmux_getmux: no memory for mux %d\n", n);
			return (NULL);
		}
		wsmuxdevs = new;
		for (; i < nwsmux; i++) {
			wsmuxdevs[i] = NULL;
		}
	}

	sc = wsmuxdevs[n];
	if (sc == NULL) {
		sc = wsmux_create("wsmux", n);
		if (sc == NULL) {
			printf("wsmux: attach out of memory\n");
		}
		wsmuxdevs[n] = sc;
	}
	return (sc);
}

