using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using NDac.Datas.QueryGenerators;

namespace NDac.Datas
{
	/// <summary>
	/// データセットホルダーの拡張メソッドのクラスを表します。
	/// </summary>
	public static class DataSetHolderExtension
	{
		/// <summary>
		/// データを更新します。
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <param name="isManualCommand">手動で更新コマンドを作成する場合はtrueを指定します。System.InvalidOperationException対策</param>
		/// <remarks>System.InvalidOperationException[複数のベース テーブルに対する動的 SQL の生成はサポートされません。]が発生する場合の対策として、isManualCommandをtrueに指定して呼び出して下さい。</remarks>
		public static void Update( this DataSetHolder holder, bool isManualCommand )
		{
			if( isManualCommand )
			{
				UpdateByManualCommand( holder );
			}
			else
			{
				holder.Update();
			}
		}

		/// <summary>
		/// 手動コマンドでデータを更新します。System.InvalidOperationException[複数のベース テーブルに対する動的 SQL の生成はサポートされません。]の対策用
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		private static void UpdateByManualCommand( DataSetHolder holder )
		{
			using( SqlConnection connection = new SqlConnection( holder.ConnectionString ) )
			{
				connection.Open();

				Dictionary< string, DataTable > schemaTables = GetSchemaTables( connection, holder );

				using( SqlTransaction transaction = connection.BeginTransaction() )
				{
					foreach( DataTable table in DeadlockAvoidance( holder ) )
					{
						foreach( DataRow row in table.Rows )
						{
							if( QueryGeneratorManager.Instance.HasGenerator( row.RowState ) )
							{
								IQueryGenerator generator = QueryGeneratorManager.Instance[ row.RowState ];

								SqlCommand command = new SqlCommand( generator.Generate( row, schemaTables[ table.TableName ] ), connection );

								command.Transaction = transaction;

								command.ExecuteNonQuery();
							}
						}
					}

					holder.Data.AcceptChanges();

					transaction.Commit();
				}
			}
		}

		/// <summary>
		/// デッドロック回避の為にテーブル名順にソートしたデータテーブルリストを作成します。
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>データテーブルリスト</returns>
		private static List< DataTable > DeadlockAvoidance( DataSetHolder holder )
		{
			List< DataTable > tables = new List< DataTable >();

			foreach( DataTable table in holder.Data.Tables )
			{
				tables.Add( table );
			}

			tables.Sort( ( x, y ) => x.TableName.CompareTo( y.TableName ) );

			return( tables );
		}

		/// <summary>
		/// テーブルスキーマのコレクションを取得します。
		/// </summary>
		/// <param name="connection">接続オブジェクト</param>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルスキーマのコレクション</returns>
		private static Dictionary< string, DataTable > GetSchemaTables( SqlConnection connection, DataSetHolder holder )
		{
			Dictionary< string, DataTable > schemaTables = new Dictionary< string, DataTable >();

			foreach( DataTable table in holder.Data.Tables )
			{
				DataTable		schemaTable	= new DataTable();

				SqlDataAdapter	adapter		= new SqlDataAdapter( string.Format( "SELECT TOP 0 * FROM {0}", table.TableName ), connection );

				adapter.FillSchema( schemaTable, SchemaType.Mapped );

				schemaTables.Add( table.TableName, schemaTable );
			}

			return( schemaTables );
		}
	}
}
