using System;

namespace NDac.Modules.Attributes
{
	/// <summary>
	/// インデックス属性を表します。
	/// </summary>
	[ AttributeUsage( AttributeTargets.Property, Inherited = true, AllowMultiple = true ) ]
	public class IndexAttribute : Attribute
	{
		protected string	_name;
		protected string	_propertyName;
		protected Type		_dataType;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="propertyName">プロパティ名</param>
		/// <param name="dataType">データ型</param>
		public IndexAttribute( string propertyName, Type dataType )
			: this( string.Empty, propertyName, dataType )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="name">キー／インデックス名</param>
		/// <param name="propertyName">プロパティ名</param>
		/// <param name="dataType">データ型</param>
		public IndexAttribute( string name, string propertyName, Type dataType )
		{
			this._name			= name;
			this._propertyName	= propertyName;
			this._dataType		= dataType;
		}

		/// <summary>
		/// キー／インデックス名を表します。
		/// </summary>
		public string Name
		{
			get
			{
				return( this._name );
			}
		}

		/// <summary>
		/// プロパティ名を表します。
		/// </summary>
		public string PropertyName
		{
			get
			{
				return( this._propertyName );
			}
		}

		/// <summary>
		/// データ型を表します。
		/// </summary>
		public Type DataType
		{
			get
			{
				return( this._dataType );
			}
		}

		/// <summary>
		/// プロパティがユニーク属性か否かを表します。
		/// </summary>
		public virtual bool IsUnique
		{
			get
			{
				return( false );
			}
		}

		/// <summary>
		/// プロパティがプライマリ属性か否かを表します。
		/// </summary>
		public virtual bool IsPrimary
		{
			get
			{
				return( false );
			}
		}

		/// <summary>
		/// キーがユニークキーか否かを表します。
		/// </summary>
		public bool IsUniqueKey
		{
			get
			{
				return( this.IsUnique && !this.IsPrimary ? true : false );
			}
		}

		/// <summary>
		/// キーがプライマリキーか否かを表します。
		/// </summary>
		public bool IsPrimaryKey
		{
			get
			{
				return( this.IsUnique && this.IsPrimary ? true : false );
			}
		}
	}
}
