using System;
using System.Data;
using System.Reflection;
using NDac.Datas;
using NDac.Keys;
using NDac.Modules;
using NDac.Modules.Entities;

namespace NDac.Modules.DataTransferObjects
{
	/// <summary>
	/// データ変換オブジェクトの抽象クラスを表します。
	/// </summary>
	[ Serializable ]
	public abstract partial class DataTransferObject : IDataTransferObject< IKey, IEntity< IKey > >
	{
		[ NonSerialized ] protected EntityAdapter< IEntity > _entityAdapter;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public DataTransferObject()
		{
			this._entityAdapter = new EntityAdapter< IEntity >( null, this.FillTo );
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="entity">エンティティ</param>
		public DataTransferObject( IEntity entity )
		{
			this._entityAdapter = new EntityAdapter< IEntity >( entity, this.FillTo );

			this.FillValue( this._entityAdapter.Entity );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		protected abstract void Copy( DataTransferObject source );

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		protected abstract void Copy( IEntity< IKey > source );

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected abstract void FillTo( IEntity entity );

		/// <summary>
		/// エンティティより値を充填します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected abstract void FillValue( IEntity entity );

		/// <summary>
		/// プライマリキーでデータを充填されたテーブルモジュールを取得します
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>テーブルモジュール</returns>
		protected abstract ITableModule LoadModule( IKey primaryKey );

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <typeparam name="T">キーの型</typeparam>
		/// <returns>プライマリキー</returns>
		protected abstract T ToPrimaryKey< T >() where T : class, IKey;

		/// <summary>
		/// ==演算子を表します。
		/// </summary>
		/// <param name="leftOperand">左辺オペランド</param>
		/// <param name="rightOperand">右辺オペランド</param>
		/// <returns>値が等しい場合trueを返します。</returns>
		public static bool operator ==( DataTransferObject leftOperand, DataTransferObject rightOperand )
		{
			return( IsEqualValues( leftOperand, rightOperand ) );
		}

		/// <summary>
		/// !=演算子を表します。
		/// </summary>
		/// <param name="leftOperand">左辺オペランド</param>
		/// <param name="rightOperand">右辺オペランド</param>
		/// <returns>値が等しくない場合trueを返します。</returns>
		public static bool operator !=( DataTransferObject leftOperand, DataTransferObject rightOperand )
		{
			return( !IsEqualValues( leftOperand, rightOperand ) );
		}

		/// <summary>
		/// 新しいエンティティを結合します。
		/// </summary>
		/// <param name="newEntity">新しいエンティティ</param>
		internal void BindNewEntity( IEntity newEntity )
		{
			this._entityAdapter.Entity = newEntity;

			this._entityAdapter.Fill();
		}

		/// <summary>
		/// インスタンスが等しいかどうかを判断します。
		/// </summary>
		/// <param name="obj">比較対象のインスタンス</param>
		/// <returns>インスタンスが等しい場合trueを返します。</returns>
		public override bool Equals( object obj )
		{
			return( IsEqualValues( this, obj ) );
		}

		/// <summary>
		/// ハッシュコードを取得します。
		/// </summary>
		/// <returns>ハッシュコード</returns>
		public override int GetHashCode()
		{
			return( this._entityAdapter.GetHashCode() );
		}

		/// <summary>
		/// データ変換オブジェクト１とデータ変換オブジェクト２の値が等しいかを判定します。
		/// </summary>
		/// <param name="obj1">データ変換オブジェクト１</param>
		/// <param name="obj2">データ変換オブジェクト２</param>
		/// <returns>データ変換オブジェクト１とデータ変換オブジェクト２の値が等しい場合trueを返します。</returns>
		private static bool IsEqualValues( object obj1, object obj2 )
		{
			if( ValueVerifier.HasNullValue( obj1, obj2 ) )
			{
				return( ValueVerifier.IsNullValueAll( obj1, obj2 ) );
			}
			else
			{
				if( obj1.GetType() != obj2.GetType() )
				{
					return( false );
				}
				else
				{
					DataTransferObject dto1 = ( DataTransferObject )obj1;

					DataTransferObject dto2 = ( DataTransferObject )obj2;

					return( dto1._entityAdapter.Entity == dto2._entityAdapter.Entity );
				}
			}
		}

		/// <summary>
		/// プライマリキーでエンティティを充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		protected IEntity LoadEntity( IKey primaryKey )
		{
			ITableModule module = this.LoadModule( primaryKey );

			return( ( IEntity )module[ primaryKey ] );
		}

		#region IDataTransferObject<IKey,IEntity> メンバ

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		void IDataTransferObject< IKey, IEntity< IKey > >.Copy( IDataTransferObject<  IKey, IEntity< IKey > > source )
		{
			this.Copy( ( DataTransferObject )source );
		}

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void IDataTransferObject< IKey, IEntity< IKey > >.Copy( IEntity< IKey > source )
		{
			this.Copy( ( DataTransferObject )source );
		}

		/// <summary>
		/// 内部に保有しているエンティティを取得します。
		/// </summary>
		/// <returns>エンティティ</returns>
		public IEntity< IKey > GetEntity()
		{
			if( this.IsDirty() )
			{
				this._entityAdapter.Fill();
			}

			return( ( IEntity< IKey > )this._entityAdapter.Entity );
		}

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		public IKey GetPrimaryKey()
		{
			return( this.ToPrimaryKey< IKey >() );
		}

		/// <summary>
		/// データベースよりプライマリキーでデータを充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public void Load( IKey primaryKey )
		{
			if( ( this._entityAdapter.Entity == null )
				|| ( ( this._entityAdapter.Entity != null ) && ( this._entityAdapter.Entity.GetPrimaryKey() != primaryKey ) ) )
			{
				this._entityAdapter.Entity = ( IEntity )this.LoadEntity( primaryKey );

				if( this._entityAdapter.Entity != null )
				{
					this.FillValue( this._entityAdapter.Entity );
				}
			}
		}

		#endregion

		#region IDataTransferObject メンバ

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		void IDataTransferObject.Copy( IDataTransferObject source )
		{
			this.Copy( ( DataTransferObject )source );
		}

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void IDataTransferObject.Copy( IEntity source )
		{
			this.Copy( ( DataTransferObject )source );
		}

		/// <summary>
		/// データベースよりデータを削除します。
		/// </summary>
		public void Delete()
		{
			if( this._entityAdapter.Entity == null )
			{
				this._entityAdapter.Entity = this.LoadEntity( this.GetPrimaryKey() );
			}

			if( this._entityAdapter.Entity != null )
			{
				this._entityAdapter.Fill();

				this._entityAdapter.Entity.Row.Delete();

				this._entityAdapter.GetHolder().Update();
			}
		}

		/// <summary>
		/// 内部に保有しているエンティティを取得します。
		/// </summary>
		/// <returns>エンティティ</returns>
		IEntity IDataTransferObject.GetEntity()
		{
			return( ( IEntity )this.GetEntity() );
		}

		/// <summary>
		/// 内部にエンティティを保有しているか判定します。
		/// </summary>
		/// <returns>内部にエンティティを保有している場合trueが返ります。</returns>
		public bool HasEntity()
		{
			return( this._entityAdapter.Entity != null );
		}

		/// <summary>
		/// エンティティの内容が削除されているか否かを判定します。
		/// </summary>
		/// <returns>内容が削除されている場合trueが返ります。</returns>
		public bool IsDeleted()
		{
			return( this._entityAdapter.Entity == null ? false : this._entityAdapter.Entity.IsDeleted() );
		}

		/// <summary>
		/// エンティティのデータが未更新かを判定します。データが更新されている場合はfalseが返ります。それ以外の全ての状態はtrueが返ります。
		/// </summary>
		/// <returns>未更新(新規，変更，削除)の場合trueが返ります。</returns>
		public bool IsDirty()
		{
			if( this._entityAdapter.Entity == null )
			{
				return( true );
			}
			else
			{
				return( this._entityAdapter.IsEquals( this ) ? this._entityAdapter.Entity.IsDirty() : true );
			}
		}

		/// <summary>
		/// エンティティの内容が変更されているか否かを判定します。
		/// </summary>
		/// <returns>内容が変更されている場合trueが返ります。</returns>
		public bool IsModified()
		{
			if( this._entityAdapter.Entity == null )
			{
				return( false );
			}
			else
			{
				return( this._entityAdapter.IsEquals( this ) ? this._entityAdapter.Entity.IsModified() : true );
			}
		}

		/// <summary>
		/// エンティティが新規作成されたデータかを判定します。
		/// </summary>
		/// <returns>新規作成されたデータの場合trueが返ります。</returns>
		public bool IsNew()
		{
			return( this._entityAdapter.Entity == null ? true : this._entityAdapter.Entity.IsNew() );
		}

		/// <summary>
		/// データベースへデータをセーブします。
		/// </summary>
		public void Save()
		{
			if( this._entityAdapter.Entity != null )
			{
				this._entityAdapter.Fill();

				this._entityAdapter.GetHolder().Update();
			}
			else
			{
				IKey key = this.GetPrimaryKey();

				ITableModule module = this.LoadModule( key );

				if( module[ key ] == null )
				{
					module.Insert( this );

					module.Table.DataSet.GetHolder().Update();

					this._entityAdapter.Entity = ( IEntity )module[ key ];
				}
				else
				{
					this._entityAdapter.Fill();

					this._entityAdapter.GetHolder().Update();
				}
			}
		}

		#endregion

		#region IEquatable<IDataTransferObject> メンバ

		/// <summary>
		/// 指定したインスタンスが現在のインスタンスと等しいかどうかを判断します。
		/// </summary>
		/// <param name="other">比較対象のインスタンス</param>
		/// <returns>指定したインスタンスが現在のインスタンスと等しい場合trueを返します。</returns>
		public bool Equals( IDataTransferObject other )
		{
			return( IsEqualValues( this, other ) );
		}

		#endregion
	}

	/// <summary>
	/// データ変換オブジェクトの抽象クラスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	[ Serializable ]
	public abstract partial class DataTransferObject< TKey, TEntity > : DataTransferObject, IDataTransferObject< TKey, TEntity >
		where TKey : class, IKey
		where TEntity : class, IEntity< TKey >
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public DataTransferObject()
			: base()
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="entity">エンティティ</param>
		public DataTransferObject( TEntity entity )
			: base( entity )
		{
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		protected abstract void Copy( DataTransferObject< TKey, TEntity > source );

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		public abstract void Copy( TEntity source );

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected abstract void FillTo( TEntity entity );

		/// <summary>
		/// データ変換オブジェクトにエンティティの内容を反映します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected abstract void FillValue( TEntity entity );

		/// <summary>
		/// プライマリキーでデータを充填されたテーブルモジュールを取得します
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>テーブルモジュール</returns>
		protected abstract ITableModule LoadModule( TKey primaryKey );

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		protected abstract TKey ToPrimaryKey();

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		protected override T ToPrimaryKey< T >()
		{
			IKey primaryKey = this.ToPrimaryKey();

			return( ( T )primaryKey );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		protected override void Copy( DataTransferObject source )
		{
			this.Copy( ( DataTransferObject< TKey, TEntity > )source );
		}

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		protected override void Copy( IEntity< IKey > source )
		{
			this.Copy( ( TEntity )source );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillTo( IEntity entity )
		{
			this.FillTo( ( TEntity )entity );
		}

		#region IDataTransferObject<TKey,TEntity> メンバ

		/// <summary>
		/// 内部に保有しているエンティティを取得します。
		/// </summary>
		/// <returns>エンティティ</returns>
		public new TEntity GetEntity()
		{
			return( ( TEntity )base.GetEntity() );
		}

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		public new TKey GetPrimaryKey()
		{
			return( ( TKey )this.ToPrimaryKey() );
		}

		/// <summary>
		/// データベースよりプライマリキーでデータを充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public void Load( TKey primaryKey )
		{
			if( ( this._entityAdapter.Entity == null )
				|| ( ( this._entityAdapter.Entity != null ) && ( this._entityAdapter.Entity.GetPrimaryKey() != primaryKey ) ) )
			{
				this._entityAdapter.Entity = ( TEntity )this.LoadEntity( primaryKey );

				if( this._entityAdapter.Entity != null )
				{
					this.FillValue( this._entityAdapter.Entity );
				}
			}
		}

		#endregion

		#region IDataTransferObject<TKey,TEntity> メンバ

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		void IDataTransferObject< TKey, TEntity >.Copy( IDataTransferObject< TKey, TEntity > source )
		{
			this.Copy( ( DataTransferObject< TKey, TEntity > )source );
		}

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void IDataTransferObject< TKey, TEntity >.Copy( TEntity source )
		{
			this.Copy( source );
		}

		#endregion
	}
}
