using System;
using System.Collections.Generic;
using System.Data;

namespace NDac.Modules
{
	/// <summary>
	/// エンティティのフィルタ処理条件を表します。
	/// </summary>
	public class FilteringCondition
	{
		private bool _isReadOnly;
		private Dictionary< DataRowState, bool > _conditions;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public FilteringCondition()
		{
			this._isReadOnly = false;

			this._conditions = new Dictionary< DataRowState, bool >();

			this._conditions.Add( DataRowState.Added,		true );

			this._conditions.Add( DataRowState.Deleted,		false );

			this._conditions.Add( DataRowState.Detached,	false );

			this._conditions.Add( DataRowState.Modified,	true );

			this._conditions.Add( DataRowState.Unchanged,	true );
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="hasAdded">DataRowのRowStateがAddedであるものを含めるかを指定します。</param>
		/// <param name="hasDeleted">DataRowのRowStateがDeletedであるものを含めるかを指定します。</param>
		/// <param name="hasDetached">DataRowのRowStateがDetachedであるものを含めるかを指定します。</param>
		/// <param name="hasModified">DataRowのRowStateがModifiedであるものを含めるかを指定します。</param>
		/// <param name="hasUnchanged">DataRowのRowStateがUnchangedであるものを含めるかを指定します。</param>
		public FilteringCondition( bool hasAdded, bool hasDeleted, bool hasDetached, bool hasModified, bool hasUnchanged )
		{
			this._isReadOnly = false;

			this._conditions = new Dictionary< DataRowState, bool >();

			this._conditions.Add( DataRowState.Added,		hasAdded );

			this._conditions.Add( DataRowState.Deleted,		hasDeleted );

			this._conditions.Add( DataRowState.Detached,	hasDetached );

			this._conditions.Add( DataRowState.Modified,	hasModified );

			this._conditions.Add( DataRowState.Unchanged,	hasUnchanged );
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="hasAdded">DataRowのRowStateがAddedであるものを含めるかを指定します。</param>
		/// <param name="hasDeleted">DataRowのRowStateがDeletedであるものを含めるかを指定します。</param>
		/// <param name="hasDetached">DataRowのRowStateがDetachedであるものを含めるかを指定します。</param>
		/// <param name="hasModified">DataRowのRowStateがModifiedであるものを含めるかを指定します。</param>
		/// <param name="hasUnchanged">DataRowのRowStateがUnchangedであるものを含めるかを指定します。</param>
		/// <param name="isReadOnly">プロパティが読み取り専用であるかを指定します。</param>
		private FilteringCondition( bool hasAdded, bool hasDeleted, bool hasDetached, bool hasModified, bool hasUnchanged, bool isReadOnly )
		{
			this._isReadOnly = isReadOnly;

			this._conditions = new Dictionary< DataRowState, bool >();

			this._conditions.Add( DataRowState.Added,		hasAdded );

			this._conditions.Add( DataRowState.Deleted,		hasDeleted );

			this._conditions.Add( DataRowState.Detached,	hasDetached );

			this._conditions.Add( DataRowState.Modified,	hasModified );

			this._conditions.Add( DataRowState.Unchanged,	hasUnchanged );
		}

		private static FilteringCondition _default			= new FilteringCondition( true, false, false, true, true, true );
		private static FilteringCondition _addedOnly		= new FilteringCondition( true, false, false, false, false, true );
		private static FilteringCondition _deletedOnly		= new FilteringCondition( false, true, false, false, false, true );
		private static FilteringCondition _detachedOnly		= new FilteringCondition( false, false, true, false, false, true );
		private static FilteringCondition _modifiedOnly		= new FilteringCondition( false, false, false, true, false, true );
		private static FilteringCondition _unchangedOnly	= new FilteringCondition( false, false, false, false, true, true );
		private static FilteringCondition _all				= new FilteringCondition( true, true, true, true, true, true );

		/// <summary>
		/// 既定のフィルタ処理条件を表します。 DataRowのRowStateがDeletedおよびDetachedのものは抽出されません。
		/// </summary>
		public static FilteringCondition Default
		{
			get
			{
				return( _default );
			}
		}

		/// <summary>
		/// DataRowのRowStateがAddedであるもののみを抽出します。
		/// </summary>
		public static FilteringCondition AddedOnly
		{
			get
			{
				return( _addedOnly );
			}
		}

		/// <summary>
		/// DataRowのRowStateがDeletedであるもののみを抽出します。
		/// </summary>
		public static FilteringCondition DeletedOnly
		{
			get
			{
				return( _deletedOnly );
			}
		}

		/// <summary>
		/// DataRowのRowStateがDetachedであるもののみを抽出します。
		/// </summary>
		public static FilteringCondition DetachedOnly
		{
			get
			{
				return( _detachedOnly );
			}
		}

		/// <summary>
		/// DataRowのRowStateがModifiedであるもののみを抽出します。
		/// </summary>
		public static FilteringCondition ModifiedOnly
		{
			get
			{
				return( _modifiedOnly );
			}
		}

		/// <summary>
		/// DataRowのRowStateがUnchangedであるもののみを抽出します。
		/// </summary>
		public static FilteringCondition UnchangedOnly
		{
			get
			{
				return( _unchangedOnly );
			}
		}

		/// <summary>
		/// 全てのDataRowを抽出します。
		/// </summary>
		public static FilteringCondition All
		{
			get
			{
				return( _all );
			}
		}

		/// <summary>
		/// DataRowのRowStateがAddedであるものを含めるかを表します。
		/// </summary>
		public bool HasAdded
		{
			get
			{
				return( this._conditions[ DataRowState.Added ] );
			}
			set
			{
				if( !this._isReadOnly )
				{
					this._conditions[ DataRowState.Added ] = value;
				}
			}
		}

		/// <summary>
		/// DataRowのRowStateがDeletedであるものを含めるかを表します。
		/// </summary>
		public bool HasDeleted
		{
			get
			{
				return( this._conditions[ DataRowState.Deleted ] );
			}
			set
			{
				if( !this._isReadOnly )
				{
					this._conditions[ DataRowState.Deleted ] = value;
				}
			}
		}

		/// <summary>
		/// DataRowのRowStateがDetachedであるものを含めるかを表します。
		/// </summary>
		public bool HasDetached
		{
			get
			{
				return( this._conditions[ DataRowState.Detached ] );
			}
			set
			{
				if( !this._isReadOnly )
				{
					this._conditions[ DataRowState.Detached ] = value;
				}
			}
		}

		/// <summary>
		/// DataRowのRowStateがModifiedであるものを含めるかを表します。
		/// </summary>
		public bool HasModified
		{
			get
			{
				return( this._conditions[ DataRowState.Modified ] );
			}
			set
			{
				if( !this._isReadOnly )
				{
					this._conditions[ DataRowState.Modified ] = value;
				}
			}
		}

		/// <summary>
		/// DataRowのRowStateがUnchangedであるものを含めるかを表します。
		/// </summary>
		public bool HasUnchanged
		{
			get
			{
				return( this._conditions[ DataRowState.Unchanged ] );
			}
			set
			{
				if( !this._isReadOnly )
				{
					this._conditions[ DataRowState.Unchanged ] = value;
				}
			}
		}

		/// <summary>
		/// データ行がエンティティに変換可能かを判定します。
		/// </summary>
		/// <param name="row">データ行</param>
		/// <returns>データ行がエンティティに変換可能な場合trueが返ります。</returns>
		internal bool CanConvert( DataRow row )
		{
			return( this._conditions[ row.RowState ] );
		}
	}
}
