#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/time.h>
#include <time.h>

#include "main.h"

Options *getOptions(int argc, char *argv[]){
  if(argc < 4){
    printf("usage: %s [-g #GPU] [-o gap_open] [-e gap_extend] [-n #answer] score_matrix_file query.fasta DB.kfasta DB.id \n", argv[0]);
    return NULL;
  }

  argc--; argv++;

  Options *opt = (Options*)malloc(sizeof(Options));

  opt->gap_open = 10;
  opt->gap_extend = 2;
  opt->n_answer = 10;
  opt->n_gpu = -1;

  while(argc > 4){
    if((*argv)[0] != '-') break;
    
    switch((*argv)[1]){
    case 'o':
      argc--; argv++;
      opt->gap_open = atoi(*argv);
      argc--; argv++;
      break;
    case 'e':
      argc--; argv++;
      opt->gap_extend = atoi(*argv);
      argc--; argv++;
      break;
    case 'n':
      argc--; argv++;
      opt->n_answer = atoi(*argv);
      argc--; argv++;
      break;
    case 'g':
      argc--; argv++;
      opt->n_gpu = atoi(*argv);
      argc--; argv++;
      break;
    default:
      argc--; argv++;
    }
  }

  if(argc != 4){
    printf("Invalid option \n");
    return NULL;
  }
  
  opt->matrix_file = argv[0];
  opt->query_fasta_file = argv[1];
  opt->src_fasta_file = argv[2];
  opt->fasta_id_file = argv[3];
  
  return opt;
}

int main(int argc, char *argv[]){
  
  Options *opt = getOptions(argc, argv);
  if(opt == NULL) return 0;

  int i, j;

  //////////////////// Initialize Part Start  ////////////////////

  char **id;
  int n_cell;
  int n_seq;

  readKFastaID(opt->fasta_id_file, &id, &n_seq, &n_cell);

  if(n_seq < opt->n_answer) opt->n_answer = n_seq;

  // get processors information
  Processors *proc = getProcessors(opt->n_gpu);
  
  printf("# of process : %d, # of GPU process : %d, # of CPU process :%d\n",
	 proc->n_proc, proc->n_gpu, proc->n_cpu);
  
  // Read Substitution Matrices for Scoring Amino-Acid Alignments from file
  AminoMatrix *a_mat = readMatrixFromFile(opt->matrix_file, cudaHostAllocPinned);

  TaskPlan *tp = makeTaskPlan(proc,
			      opt->src_fasta_file, 
			      a_mat,
			      opt->gap_open,
			      opt->gap_extend);
  //////////////////// Initialize Part End  ////////////////////
  
  // Read FASTA format query sequences from file
  SeqEntryArray *q_array = fasta_read( opt->query_fasta_file );
  if(q_array == NULL){
    fprintf(stderr, "query fasta file read error\n");
    return 0;
  }
    
  //printf("Initialize part end\n");
  fflush(stdout);
  
  for(i=0; i<(q_array->n); i++){
    // Start stopwatch here
    double start_time = my_clock();

    SeqEntry *se = getSeqEntryArray(q_array, i);
    char *q_seq = getAlignedSeq(se->seq, se->length, WARP_SIZE, PADDING);
    int q_seq_len = strlen(q_seq);
        
    getIndexedSeq(q_seq, q_seq, q_seq_len, tp->a_matrix->column_index, PADDING);

    // If it is using MPI, it have to scatter "q_seq" to all of node here.

    SWResult *result = getSWResultWithMultiThreads(tp, proc, q_seq, q_seq_len,
						   BLOCK_PER_KERNEL_CALL, THREAD_PER_BLOCK, WARP_SIZE,
						   opt->n_answer);
    
    // If it is using MPI, it have to gather "result" from all of node here
    // and sort all of "result" again.

    // Stop stopwatch here
    double end_time = my_clock();

    double elasped_time = end_time - start_time;
    double mcups = n_cell / elasped_time / 1000000 * se->length;

    printf("query:%s\n", se->id);
    printf("Length: %d --- time: %lf (s) MCUPS: %d\n",
	   se->length, elasped_time, (int)mcups);
    
    if(result == NULL) continue;

    printf("----------Display the top %d ----------\n", opt->n_answer);
    for(j=0; j<opt->n_answer; j++){
      if(result[j].score < 0) break;
      printf("score: %d -- %s\n", result[j].score, id[result[j].index]);
    }
  }
  
  return 0;

}

double my_clock(){
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return tv.tv_sec + (double)tv.tv_usec*1e-6;
}

void readKFastaID(char *path, char ***id, int *n_seq, int *n_cell){
  FILE *fp;
  if((fp = fopen(path, "r")) == NULL){
    fprintf(stderr, "File open error! %s\n", path);
    exit(1);
  }

  char *buf;
  buf = (char*)malloc(sizeof(char) * (1024+1));

  fgets(buf, 1024, fp);
  *n_seq = strtol(buf, NULL, 10);
  fgets(buf, 1024, fp);
  *n_cell = strtol(buf, NULL, 10);

  *id = (char**)malloc(sizeof(char*) * (*n_seq));
  
  int i;
  for(i=0; i<(*n_seq); i++){
    char *buf2;
    int idx;
    buf = (char*)malloc(sizeof(char) * (1024+1));
    fgets(buf, 1024, fp);

    buf[strlen(buf)-1] = '\0';
    idx = strtol(buf, &buf2, 10);
    (*id)[idx] = buf2 + 1;
  }
}
