#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "seq_entry_array.h"
#include "fasta_util.h"

#include "seq_block_read.h"

SeqBlock *getSeqBlock(SeqEntryArray *array,
		      int alignment, int seq_per_block, char padding,
		      void *(* mem_allocate_func)(size_t size)){
  int n_block;
  int *block_index;
  int total_block_size;
  char *seq_block;
  int i;

  SeqEntry *se;
  int seq_length;
  int seq_no;

  char *p_block;

  SeqBlock *sb;
  sb = (SeqBlock*) mem_allocate_func (sizeof(SeqBlock));
  if(sb == NULL) return NULL;
   
  n_block = (array->n + seq_per_block - 1)/seq_per_block; 
  
  block_index = (int*) mem_allocate_func (sizeof(int) * (n_block+1));
  if(block_index == NULL) return NULL;

  block_index[0] = 0;
  total_block_size = 0;
  for(i=0; i<n_block; i++){
    seq_no = i*seq_per_block + (seq_per_block-1);
    if(seq_no >= array->n){
      seq_no = array->n-1;
    }
    
    se = getSeqEntryArray(array, seq_no);
    seq_length = se->length;
    seq_length = (int)((seq_length + alignment - 1)/alignment) * alignment;
    total_block_size += seq_length * seq_per_block;
    block_index[i+1] = block_index[i] + seq_length * seq_per_block;
  }

  seq_block = (char*) mem_allocate_func (sizeof(char) * total_block_size);
  if(seq_block == NULL) return NULL;

  memset(seq_block, padding, total_block_size);

  for(i=0; i<n_block; i++){
    p_block = seq_block + block_index[i];
    copyBlock(array, alignment, seq_per_block, i*seq_per_block, p_block);
  }

  sb->n_block = n_block;
  sb->block_idx = block_index;
  sb->seq_block = seq_block;
  sb->size = total_block_size;
  sb->alignment = alignment;
  sb->seq_per_block = seq_per_block;
  sb->padding = padding;
  
  return sb;
}

int copyBlock(SeqEntryArray *array, int alignment ,int seq_per_block, int block_top_idx, char *block){
  int i, j;
  int buf_size;
  
  for(i=0; i<seq_per_block; i++){
    int seq_no = block_top_idx + i;
    SeqEntry *se = getSeqEntryArray(array, seq_no);

    if(se == NULL) break;
    
    for(j=0; j<se->length; j+=alignment){
      buf_size = alignment;
      if(j + buf_size - 1 > se->length){
	buf_size = (se->length)%alignment;
      }
      memmove(block + alignment * i + seq_per_block * j, se->seq + j, buf_size);
    }
  }      

  return 0;    
}

int getSeqBlockSize(SeqBlock *block){
  return block->size;
}

int getSeqBlockCount(SeqBlock *block){
  return block->n_block;
}

char *getSeqBlockSeq(SeqBlock *block, int idx){
  int bid = idx/block->seq_per_block;
  int sid = (idx%block->seq_per_block) * block->alignment;
  return block->seq_block + block->block_idx[bid] + sid;
}

int getSeqBlockSeqLengthAlignmented(SeqBlock *block, int idx){
  int bid = idx/block->seq_per_block;
  return (block->block_idx[bid+1] - block->block_idx[bid]) / block->seq_per_block;
}

void copySeqBlockSeqAlignmented(SeqBlock *block, int idx, char *dst){
  char *src = getSeqBlockSeq(block, idx);
  int stride = block->seq_per_block * block->alignment;
  int l = getSeqBlockSeqLengthAlignmented(block, idx);
  
  do{
    memmove(dst, src, sizeof(char) * block->alignment);
    l -= block->alignment;
    src += stride;
    dst += block->alignment;
  }while(l > 0);
  
}

int cmpSeqBlockAndSeqEntry(SeqBlock *block, SeqEntryArray *array, int idx){
  SeqEntry *se = getSeqEntryArray(array, idx);
  int len = getSeqBlockSeqLengthAlignmented(block, idx);
  
  char *seq_na = se->seq;
  char *seq_a = (char*) malloc (sizeof(char) * (len+1) );
  
  seq_a[len] = '\0';
  copySeqBlockSeqAlignmented(block, idx, seq_a);
  
  int i;
  for(i=0; i<len+1; i++){
    if(seq_na[i] == '\0' && seq_a[i] == '\0') return 0;
    if(seq_na[i] == '\0' && seq_a[i] == block->padding) return 0;
    if(seq_na[i] > seq_a[i]) return 1;
    if(seq_na[i] < seq_a[i]) return -1;
  }

  return 0;
}

char *getAlignedSeq(char *seq, int len, int alignment, char padding){
  char *p_seq = NULL;
  int padded_len = (int)((len + alignment - 1) / alignment) * alignment;

  p_seq = (char*) malloc (sizeof(char) * (padded_len + 1) );
  memmove(p_seq, seq, sizeof(char) * len);
  memset(p_seq + len, padding, padded_len - len);
  p_seq[padded_len] = '\0';

  return p_seq;
}
