// DSpamFilter.cpp: implementation of the CDSpamFilter class.
//
//////////////////////////////////////////////////////////////////////
/*******************************************************************************
 *                                                                             *
 *  This file is part of NotesAntiSpam.                                        *
 *                                                                             *
 *  NotesAntiSpam is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by       *
 *  the Free Software Foundation; either version 2 of the License, or          *
 *  (at your option) any later version.                                        *
 *                                                                             *
 *  NotesAntiSpam is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 *  GNU General Public License for more details.                               *
 *                                                                             *
 *  You should have received a copy of the GNU General Public License          *
 *  along with NotesAntiSpam; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 *                                                                             *
 *  Copyright (c) 2003 Thomas Kriener                                          *
 *                                                                             *
 *******************************************************************************/

/*
 * Parts of this file are from DSPAM and
 * COPYRIGHT (C) 2002-2004 NETWORK DWEEBS CORPORATION
 */

#include "stdafx.h"
#include <Dbghelp.h>
#include "NotesAntiSpam.h"
#include "NotesAntiSpamDlg.h"
#include "DSpamFilter.h"
#include <fstream>
#include "sqlite/sqlite.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

/* MESSAGE PROCESSING */

/* initialize dspam 

   Operating Modes:
      DSM_PROCESS:	Process message
	  DSM_CLASSIFY:	Classify message only (do not write changes)
	  
   Flags:
      DSF_CHAINED	Use Chained Tokens
	  DSF_SIGNATURE	Signature Mode (Use a signature)
	  DSF_NOISE		Use Bayesian Noise Reduction
	  DSF_WHITELIST	Use Automatic Whitelisting

   Training Modes:
      DST_TEFT		Train Everything
	  DST_TOE		Train-on-Error
	  DST_TUM		Train-until-Mature

   Classifications:
      Used to tell libdspam the message has already been classified, and should
	  be processed in such a way that the result will match the classification.
	  
	  DSR_ISSPAM		Message is spam (learn as spam)
	  DSR_ISINNOCENT	Message is innocent (learn as innocent)
	  DSR_NONE		No predetermined classification (classify message)

   Sources:
      Used to tell libdspam the source of the specified classification (if any).
	  
	  DSS_ERROR		Misclassification by libdspam
	  DSS_CORPUS		Corpusfed message
	  DSS_INOCULATION	Message inoculation
	  DSS_NONE		No classification source (use only with DSR_NONE)

   NOTE: When using DSS_ERROR, a DSPAM signature should be provided, OR
         the original message in PRISTINE form (without any DSPAM headers and
         with the original message's headers).
*/ 

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CDSpamFilter::CDSpamFilter()
{
	CString path;

	m_myApp=(CNotesAntiSpamApp*)AfxGetApp();

	m_dspamHome=m_myApp->m_config.m_userpath;
	m_dspamUser="ItsMe";

	path=m_dspamHome;
	if(path[path.GetLength()-1]!='\\')
		path+='\\';
	path+="data\\"+m_dspamUser+'\\';
	// Path has to end with '\'
	MakeSureDirectoryPathExists(path);

	/* Write Pragma-File for SQLite */
	std::ofstream pragmaFile;
	pragmaFile.open(m_dspamHome+"\\sqlite.pragma");
	pragmaFile<<"PRAGMA synchronous = OFF"<<std::endl;
	pragmaFile.close();

	// Initialize dspam database-driver
	dspam_init_driver();
}

CDSpamFilter::~CDSpamFilter()
{
	/* Performs any driver-specific shutdown functions */
	dspam_shutdown_driver();
}

void CDSpamFilter::TrainHamAsSpam(const CString &signature)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	struct _ds_spam_signature SIG;

	/* SPAM REPORTING (AS MISCLASSIFICATION) */
	
	/* We call everything just like before, with these exceptions:
	   - We set the classification to DSR_ISSPAM
	   - We set the source to DSS_ERROR
	   
	   This example will use the original message in pristine form instead of
	   a signature.  See the next example (false positives) for an example using
	   a signature.
	*/
	
	/* Initialize the DSPAM context */
	CTX = dspam_init (m_dspamUser, NULL, m_dspamHome, DSM_PROCESS,
	                  DSF_CHAINED | DSF_SIGNATURE | DSF_NOISE);
	
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed!");
		return;
	}
	
	/* Set up the context for error correction as spam */
	CTX->classification = DSR_ISSPAM;
	CTX->source         = DSS_ERROR;
	// Set Training-Mode
	CTX->training_mode=m_myApp->m_config.m_dSpam_trainMode;
	// Set algorithms
	CTX->algorithms=DSA_GRAHAM | DSA_BURTON | DSP_GRAHAM;

	// Get Signature
    if (_ds_get_signature (CTX, &SIG, signature))
	{
		CString msg;
		msg.Format("ERROR: signature retrieval for '%s' failed", signature);
		AfxMessageBox(msg,MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,msg);
	}
	else
	{
		CTX->signature = &SIG;
	}

	/* Call DSPAM */
	if (dspam_process(CTX, NULL) != 0)
	{
		AfxMessageBox("ERROR: dspam_process failed",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_process failed");
		return;
	}
	
	/* Destroy the context */
	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return;
	}
	
	return;
}

void CDSpamFilter::TrainSpamAsHam(const CString &signature)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	struct _ds_spam_signature SIG;

	/* FALSE POSITIVE REPORTING */
	
	/* Here we submit the message's signature for retraining as a false positive.
	   We make the following changes from our original example: 
	   - We set the classification to DSR_ISINNOCENT
	   - We set the source to DSS_ERROR
	*/
	
	/* Initialize DSPAM context */
	CTX = dspam_init (m_dspamUser, NULL, m_dspamHome, DSM_PROCESS,
	                  DSF_CHAINED | DSF_SIGNATURE | DSF_NOISE);
	
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed");
		return;
	}
	
	/* Set up the context for error correction as innocent */
	CTX->classification = DSR_ISINNOCENT;
	CTX->source         = DSS_ERROR;
	// Set Training-Mode
	CTX->training_mode=m_myApp->m_config.m_dSpam_trainMode;
	// Set algorithms
	CTX->algorithms=DSA_GRAHAM | DSA_BURTON | DSP_GRAHAM;

	// Get Signature
    if (_ds_get_signature (CTX, &SIG, signature))
	{
		CString msg;
		msg.Format("ERROR: signature retrieval for '%s' failed", signature);
		AfxMessageBox(msg,MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,msg);
	}
	else
	{
		CTX->signature = &SIG;
	}
	
	/* Call DSPAM */
	if (dspam_process(CTX, NULL) != 0)
	{
		AfxMessageBox("ERROR: dspam_process failed",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_process failed");
		return;
	}
	
	/* Destroy the context */
	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return;
	}
	
	return;
}

void CDSpamFilter::TrainHam(const CString &mail)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	
	/* Initialize DSPAM context */
	CTX = dspam_init (m_dspamUser, NULL, m_dspamHome, DSM_PROCESS,
	                  DSF_CHAINED | DSF_NOISE);
	
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed");
		return;
	}
	
	/* Set up the context for error correction as innocent */
	CTX->classification = DSR_ISINNOCENT;
	CTX->source         = DSS_CORPUS;
	// Set Training-Mode
	CTX->training_mode=m_myApp->m_config.m_dSpam_trainMode;
	// Set algorithms
	CTX->algorithms=DSA_GRAHAM | DSA_BURTON | DSP_GRAHAM;
	
	/* Call DSPAM */
	if (dspam_process(CTX, mail) != 0)
	{
		AfxMessageBox("ERROR: dspam_process failed",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_process failed");
		return;
	}

	/* Destroy the context */
	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return;
	}
	
	return;
}

void CDSpamFilter::TrainSpam(const CString &mail)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	
	/* Initialize the DSPAM context */
	CTX = dspam_init (m_dspamUser, NULL, m_dspamHome, DSM_PROCESS,
	                  DSF_CHAINED | DSF_NOISE);
	
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed");
		return;
	}
	
	/* Set up the context for error correction as spam */
	CTX->classification = DSR_ISSPAM;
	CTX->source         = DSS_CORPUS;
	// Set Training-Mode
	CTX->training_mode=m_myApp->m_config.m_dSpam_trainMode;
	// Set algorithms
	CTX->algorithms=DSA_GRAHAM | DSA_BURTON | DSP_GRAHAM;
	
	/* Call DSPAM */
	if (dspam_process(CTX, mail) != 0)
	{
		AfxMessageBox("ERROR: dspam_process failed",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_process failed");
		return;
	}
	
	/* Destroy the context */
	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return;
	}

	return;
}



BOOL CDSpamFilter::CheckSpam(const CString &mail, CString& signature)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	bool      retVal=FALSE;
	
	/* STANDARD INBOUND PROCESSING */
	
	/* Initialize the DSPAM context */
	CTX = dspam_init (m_dspamUser, NULL, m_dspamHome, DSM_PROCESS,
	                  DSF_CHAINED | DSF_SIGNATURE | DSF_NOISE);
    
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed");
		// If we don't know exactly: in dubio pro reo (benefit of the doubt)
		return FALSE;
	}

	/* Set up Context for new message */
	//CTX->classification = DSR_NONE;
	//CTX->source         = DSS_CORPUS;
	// Set Training-Mode
	CTX->training_mode=m_myApp->m_config.m_dSpam_trainMode;
	// Set algorithms
	CTX->algorithms=DSA_GRAHAM | DSA_BURTON | DSP_GRAHAM;

	/* Call DSPAM's processor with the message text */
	if (dspam_process (CTX, mail) != 0)
	{
		AfxMessageBox("ERROR: dspam_process failed",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_process failed");
		// If we don't know exactly: in dubio pro reo (benefit of the doubt)
		return FALSE;
	}
	/* Print processing results */
	CString tempStr;
	tempStr.Format("Probability: %2.4f Confidence: %2.4f, Result: %s",
	               CTX->probability, CTX->confidence,
				   (CTX->result == DSR_ISSPAM) ? "Spam" : "Innocent");
#ifdef _DEBUG
	((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"\r\n");
#endif
	m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);

	retVal=(CTX->result == DSR_ISSPAM) ? true : false;
	/* Manage signature */
	if (CTX->signature == NULL)
	{
		AfxMessageBox("ERROR: No signature provided",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: No signature provided");
	}
	else
	{
		signature=SaveSignature(CTX);
	}
   
	/* Destroy the context */
	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return retVal;
	}

	// If we don't know exactly: in dubio pro reo (benefit of the doubt)
	return retVal;
}


bool CDSpamFilter::GetTotals(_ds_spam_totals * totals)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	bool      retVal=FALSE;

	CNotesAntiSpamApp* myApp=(CNotesAntiSpamApp*)AfxGetApp();
	CString dspamHome=myApp->m_config.m_userpath;
	CString dspamUser="ItsMe";

	/* Initialize the DSPAM context */
	CTX = dspam_init (dspamUser, NULL, dspamHome, DSM_TOOLS, DSF_CHAINED);
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed");
		// If we don't know exactly: in dubio pro reo (benefit of the doubt)
		return FALSE;
	}

	totals->innocent_classified=CTX->totals.innocent_classified;
	totals->innocent_corpusfed=CTX->totals.innocent_corpusfed;
	totals->innocent_learned=CTX->totals.innocent_learned;
	totals->innocent_misclassified=CTX->totals.innocent_misclassified;

	totals->spam_classified=CTX->totals.spam_classified;
	totals->spam_corpusfed=CTX->totals.spam_corpusfed;
	totals->spam_learned=CTX->totals.spam_learned;
	totals->spam_misclassified=CTX->totals.spam_misclassified;

	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return false;
	}

	return true;
}

CString CDSpamFilter::SaveSignature(DSPAM_CTX *CTX)
{
	/* Save the DSPAM signature */
	int valid = 0;
	char session[64] = { 0 };
	CString retVal;
    CString tempStr;

	while (valid == 0)
	{
		_ds_create_signature_id (CTX, session, sizeof (session));
		if (_ds_verify_signature (CTX, session))
			valid = 1;
	}
	int x=_ds_set_signature (CTX, CTX->signature, session);
	if(x)
	{
		tempStr.Format("ERROR: _ds_set_signature failed with error %d",x);
		AfxMessageBox(tempStr,MB_OK|MB_ICONERROR);
	}

    tempStr.Format("saved signature as %s", session);
	m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);
	retVal=session;
	return retVal;
}

void CDSpamFilter::CleanupDatabase(bool purgeSigs,
								   int  ageSigs,
								   bool purgeNeutral,
								   int  ageNeutral,
								   bool purgeUnused,
								   int  ageUnused,
								   int  ageHapaxes,
								   int  ageHits1S,
								   int  ageHits1I,
								   bool vaccumDB)
{
	DSPAM_CTX *CTX;    // DSPAM Context
	CString tempStr;
	int delItems;

#ifdef _DEBUG
	AfxMessageBox("dspam_clean starting");
#endif
	m_myApp->debugger.Add(__FILE__,__LINE__,"**** dspam_clean starting ****");

	if (!purgeSigs && !purgeNeutral && !purgeUnused)
	{
		AfxMessageBox("ERROR: Nothing to clean specified");
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Nothing to clean specified");
		return;
	}
	
	/* Initialize the DSPAM context */
	CTX = dspam_init (m_dspamUser, NULL, m_dspamHome, DSM_PROCESS,
	                  DSF_CHAINED | DSF_SIGNATURE | DSF_NOISE);
	
	if (CTX == NULL)
	{
		AfxMessageBox("ERROR: dspam_init failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_init failed");
		return;
	}
	
	if (purgeSigs)
	{
		delItems=ProcessSigs(CTX, ageSigs);
		tempStr.Format("\r\n->%i Signatures deleted",delItems);
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr);
	}
	if (purgeNeutral)
	{
		delItems=ProcessNeutral(CTX, ageNeutral);
		tempStr.Format("\r\n->%i neutral Tokens deleted",delItems);
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr);
	}
	if (purgeUnused)
	{
		delItems=ProcessUnused(CTX, ageUnused, ageHapaxes, ageHits1S, ageHits1I);
		tempStr.Format("\r\n->%i unused Tokens deleted",delItems);
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr);
	}
	if (vaccumDB)
	{
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("\r\n->Vaccum start...");
		VacuumDB(CTX);
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("finished");
	}

	if (dspam_destroy (CTX) != 0)
	{
		AfxMessageBox("ERROR: dspam_destroy failed!",MB_OK|MB_ICONERROR);
		m_myApp->debugger.Add(__FILE__,__LINE__,"ERROR: dspam_destroy failed");
		return;
	}
	return;
}

int CDSpamFilter::ProcessSigs (DSPAM_CTX * CTX, int age)
{
	struct _ds_storage_signature *ss;
	struct nt *del;
	struct nt_node *node;
	int delta;
	CString tempStr;
	int retVal=0;
	
	del = nt_create(NT_CHAR);
	if (del == NULL)
		return -1;

	tempStr.Format("Processing sigs; age: %d", age);
#ifdef _DEBUG
	((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"\r\n");
#endif
	m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);
	
	ss = _ds_get_nextsignature (CTX);
	while (ss != NULL)
	{
		tempStr.Format("Signature: \"%s\"    Created: %s", ss->signature, ctime (&ss->created_on));
		m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);

		delta = (((time (NULL) - ss->created_on) / 60) / 60) / 24;
		if (age == 0 || delta > age)
		{
#ifdef _DEBUG
			((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"->Deleted\r\n");
#endif
			m_myApp->debugger.Add(__FILE__,__LINE__,"--->Deleted");

			nt_add(del, ss->signature);
		}
		ss = _ds_get_nextsignature (CTX);
	}
	
	node = del->first;
	while(node != NULL)
	{
		_ds_delete_signature (CTX, (const char*)(node->ptr));
		retVal++;
		node = node->next; 
	}
	nt_destroy(del);

	return retVal;
}

int CDSpamFilter::ProcessNeutral(DSPAM_CTX *CTX, int age)
{
	struct _ds_storage_record *sr;
	struct _ds_spam_stat s;
	struct lht *del;
	int delta;
	CString tempStr;
	int retVal=0;

	tempStr.Format("Processing probabilities; age: %d", age);
#ifdef _DEBUG
	((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"\r\n");
#endif
	m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);
	
	del = lht_create(1543);
	if (del == NULL)
		return -1;
	sr = _ds_get_nexttoken (CTX);
	while (sr != NULL)
	{
		s.innocent_hits = sr->innocent_hits;
		s.spam_hits = sr->spam_hits;
		s.probability = 0.00000;
		_ds_calc_stat(CTX, sr->token, &s);
		if (s.probability >= 0.3500 && s.probability <= 0.6500)
		{
			delta = (((time (NULL) - sr->last_hit) / 60) / 60) / 24;
			if (age == 0 || delta > age)
			{
				lht_hit(del, sr->token, "");
				retVal++;
			}
		}
		free (sr);
		sr = _ds_get_nexttoken (CTX);
	}
	
	_ds_delall_spamrecords(CTX, del);
	lht_destroy(del);

	return retVal;
}

int CDSpamFilter::ProcessUnused(DSPAM_CTX *CTX, int any, int quota, int nospam, int onehit)
{
	struct _ds_storage_record *sr;
	struct lht *del;
	time_t t = time(NULL);
	int delta;
	CString tempStr;
	int retVal=0;

	tempStr.Format("Processing unused; any: %d quota: %d nospam: %d onehit: %d\n", any, quota, nospam, onehit);
#ifdef _DEBUG
	((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"\r\n");
#endif
	m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);

	del = lht_create(1543);
	if (del == NULL)
		return -1;
	sr = _ds_get_nexttoken (CTX);
	while (sr != NULL)
	{
		delta = (((t - sr->last_hit) / 60) / 60) / 24;
		if (any == 0 || delta > any)
		{ 
			lht_hit(del, sr->token, "");
			retVal++;
		}
		else if ((sr->innocent_hits*2) + sr->spam_hits < 5)
		{ 
			if (quota == 0 || delta > quota)
			{
				lht_hit(del, sr->token, "");
				retVal++;
			}
			else if (sr->innocent_hits == 0 && sr->spam_hits == 1 &&
				(nospam == 0 || delta > nospam))
			{
				lht_hit(del, sr->token, "");
				retVal++;
			}
			else if (sr->innocent_hits == 1 && sr->spam_hits == 0 &&
				(onehit == 0 || delta > onehit))
			{
				lht_hit(del, sr->token, "");
				retVal++;
			}
		}

		free (sr);
		sr = _ds_get_nexttoken (CTX);
	}
    
	_ds_delall_spamrecords(CTX, del);
	lht_destroy(del);
	return retVal;
}

/*
 * Copied from sqlite_drv.h
 */
struct _sqlite_drv_storage
{
  struct sqlite *dbh;				/* database handle */
  struct _ds_spam_totals control_totals;        /* totals at storage init */
  struct _ds_spam_totals merged_totals;         /* totals for merged group */ 

  /* control token data; used to measure deltas from getall to setall 
   * enabling us to code a sql query based on increments/decrements 
   * instead of query-coded data */

  unsigned long long control_token;     /* control token crc */
  long control_sh;              /* control token spam hits at getall */
  long control_ih;              /* control token innocent hits at getall */

  sqlite_vm *iter_token;        /* get_nexttoken iteration result */
  sqlite_vm *iter_sig;          /* get_nextsignature iteration result */

  struct nt *dir_handles;
  int dbh_attached;
};

int CDSpamFilter::VacuumDB(DSPAM_CTX* CTX)
{
	struct _sqlite_drv_storage *s = (struct _sqlite_drv_storage *) CTX->storage;
	char query[128];
	char *err=NULL;
	CString tempStr;
	
	if (s->dbh == NULL)
	{
		tempStr="CDSpamFilter::VacuumDB: invalid database handle (NULL)";
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"\r\n");
		m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);
		return -1;
	}
	strcpy(query,"VACUUM");
	
	if ((sqlite_exec(s->dbh, query, NULL, NULL, &err))!=SQLITE_OK)
	{
		tempStr.Format("ERROR: \"%s\" on Query \"%s\"",err,query);
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tempStr+"\r\n");
		m_myApp->debugger.Add(__FILE__,__LINE__,tempStr);
		return -1;
	}

	return 0;
}
