// NotesMailDB.cpp: implementation of the CNotesMailDB class.
//
//////////////////////////////////////////////////////////////////////

/*******************************************************************************
 *                                                                             *
 *  This file is part of NotesAntiSpam.                                        *
 *                                                                             *
 *  NotesAntiSpam is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by       *
 *  the Free Software Foundation; either version 2 of the License, or          *
 *  (at your option) any later version.                                        *
 *                                                                             *
 *  NotesAntiSpam is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 *  GNU General Public License for more details.                               *
 *                                                                             *
 *  You should have received a copy of the GNU General Public License          *
 *  along with NotesAntiSpam; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 *                                                                             *
 *  Copyright (c) 2003 Thomas Kriener                                          *
 *                                                                             *
 *******************************************************************************/

#include "stdafx.h"
#include <Winsock2.h>
#include "NotesAntiSpam.h"
#include "NotesAntiSpamDlg.h"
#include "NotesMailDB.h"
#include "NotesMail.h"
#include "Debugger.h"
#include "WhitelistEntry.h"
#include "TrainIDCache.h"
#include "dspam/libdspam.h"


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define ERR_BUF_SIZE 2048

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CNotesMailDB::CNotesMailDB()
{
	CString tmp;

	// Get Pointer to Application
	myApp=(CNotesAntiSpamApp*)AfxGetApp();

	// copy originial notes.ini to notesantispam.ini
	CString configFile=CopyIniFile();

	// Make the error handler throw all errors encountered during execution.
	LNSetThrowAllErrors (TRUE);

	try
	{
		// Initialize the API.
		myApp->debugger.Add(__FILE__,__LINE__,"Initializing Notes-Session");
		m_notesSession.Init(configFile);

		// Print Notes-Version in Debug	
		if(myApp->debugger.GetState())
		{
			CString versionStr;
			char msg[1024];

			myApp->debugger.Add(__FILE__,__LINE__,"###################### BEGIN NOTES-VERSION ######################");

			// Get Version-String from Notes
			LNGetErrorMessage( 1, msg, sizeof(msg)-1);
			versionStr.Format("Notes-Version: %s",msg);
			myApp->debugger.Add(__FILE__,__LINE__,versionStr);
			// Get Version of C++-API
			versionStr.Format("API-Program-Version: %s",LNGetAPIProgramVersionString());
			myApp->debugger.Add(__FILE__,__LINE__,versionStr);
			versionStr.Format("API-Library-Version: %s",LNGetAPILibraryVersionString());
			myApp->debugger.Add(__FILE__,__LINE__,versionStr);

			myApp->debugger.Add(__FILE__,__LINE__,"###################### END NOTES-VERSION ######################");
		}
		if(myApp->m_config.m_mailBox=="")
		{
			// get user's default mail database.
			myApp->debugger.Add(__FILE__,__LINE__,"Get Default-Mailbox");
			m_notesSession.GetMailDatabase(&m_mailDb);
		}
		else
		{
			// open specified mail-database
			tmp.Format("Get specific Mailbox %s on server %s",myApp->m_config.m_mailBox,myApp->m_config.m_mailBoxServer);
			myApp->debugger.Add(__FILE__,__LINE__,tmp);
			LNString mailBox=myApp->m_config.m_mailBox;
			LNString mailBoxServer=myApp->m_config.m_mailBoxServer;
			m_notesSession.GetDatabase(mailBox, &m_mailDb, mailBoxServer);
		}
		// Open the database.
		myApp->debugger.Add(__FILE__,__LINE__,"Opening Mailbox");
		m_mailDb.Open();
		// Open Inbox-View
		myApp->debugger.Add(__FILE__,__LINE__,"Opening Inbox");
		m_mailDb.GetViewFolder("($Inbox)",&m_inboxView);
		m_inboxView.Open(LNVFOPENFLAGS_USE_UNREAD_LIST);
	}
	catch (LNSTATUS lnerror)
	{
		char ErrorBuf[ERR_BUF_SIZE];

		LNGetErrorMessage(lnerror, ErrorBuf, ERR_BUF_SIZE);
		tmp.Format("Notes-Error(%i): %s",lnerror,ErrorBuf);
		myApp->debugger.Add(__FILE__,__LINE__,"**** ERROR: "+tmp);
		MessageBox(AfxGetMainWnd()->m_hWnd,tmp,"NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
		return;
	}
	// Make the error handler throw no errors encountered during execution.
	// We know that the Folder "-Spam" might not be there
	LNSetThrowAllErrors (FALSE);
	// Open or create Spam-Folder
	if(m_mailDb.GetViewFolder("-Spam",&m_spamView)!=LNNOERROR)
	{
		if(m_mailDb.CreateFolder("-Spam")!=LNNOERROR)
		{
			myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Unable to create Folder \"-Spam\" !");
			MessageBox(AfxGetMainWnd()->m_hWnd,"Unable to create Folder \"-Spam\" !",
			           "NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return;
		}
		if(m_mailDb.GetViewFolder("-Spam",&m_spamView)!=LNNOERROR)
		{
			myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Unable to open Folder \"-Spam\" !");
			MessageBox(AfxGetMainWnd()->m_hWnd,"Unable to open Folder \"-Spam\" after creation!",
			           "NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return;
		}
	}
	// Open or create Spam-Folder for Spam-Training
	if(m_mailDb.GetViewFolder("-Spam\\TrainSpam",&m_spamTrainView)!=LNNOERROR)
	{
		if(m_mailDb.CreateFolder("-Spam\\TrainSpam")!=LNNOERROR)
		{
			myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Unable to create Folder \"-Spam\\TrainSpam\" !");
			MessageBox(AfxGetMainWnd()->m_hWnd,"Unable to create Folder \"-Spam\\TrainSpam\" !",
			           "NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return;
		}
		if(m_mailDb.GetViewFolder("-Spam\\TrainSpam",&m_spamTrainView)!=LNNOERROR)
		{
			myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Unable to open Folder \"-Spam\\TrainSpam\" !");
			MessageBox(AfxGetMainWnd()->m_hWnd,"Unable to open Folder \"-Spam\\TrainSpam\" after creation!",
			           "NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return;
		}
	}
	// Open or create Spam-Folder for Ham-Training
	if(m_mailDb.GetViewFolder("-Spam\\TrainHam",&m_hamTrainView)!=LNNOERROR)
	{
		if(m_mailDb.CreateFolder("-Spam\\TrainHam")!=LNNOERROR)
		{
			myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Unable to create Folder \"-Spam\\TrainHam\" !");
			MessageBox(AfxGetMainWnd()->m_hWnd,"Unable to create Folder \"-Spam\\TrainHam\" !",
			           "NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return;
		}
		if(m_mailDb.GetViewFolder("-Spam\\TrainHam",&m_hamTrainView)!=LNNOERROR)
		{
			myApp->debugger.Add(__FILE__,__LINE__,"ERROR: Unable to open Folder \"-Spam\\TrainHam\" !");
			MessageBox(AfxGetMainWnd()->m_hWnd,"Unable to open Folder \"-Spam\\TrainHam\" after creation!",
			           "NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return;
		}
	}
	// Make the error handler throw all errors encountered during execution.
	LNSetThrowAllErrors (TRUE);
	try
	{
		myApp->debugger.Add(__FILE__,__LINE__,"Opening SpamView");
		m_spamView.Open();
		myApp->debugger.Add(__FILE__,__LINE__,"Opening TrainSpamView");
		m_spamTrainView.Open();
		myApp->debugger.Add(__FILE__,__LINE__,"Opening TrainHamView");
		m_hamTrainView.Open();
	}
	catch (LNSTATUS lnerror)
	{
		char ErrorBuf[ERR_BUF_SIZE];

		LNGetErrorMessage(lnerror, ErrorBuf, ERR_BUF_SIZE);
		tmp.Format("Notes-Error on Spam-Folder (%i): %s",lnerror,ErrorBuf);
		myApp->debugger.Add(__FILE__,__LINE__,"**** ERROR: "+tmp);
		MessageBox(AfxGetMainWnd()->m_hWnd,tmp,"NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
		return;
	}
}

CNotesMailDB::~CNotesMailDB()
{
	CString tmp;

	// Now for the main processing.
	// Make the error handler throw all errors encountered during execution.
	LNSetThrowAllErrors (TRUE);

	try
	{
		// Close TrainSpam-View
		if(m_spamTrainView.IsOpen())
			m_spamTrainView.Close();
		// Close TrainHam-View
		if(m_hamTrainView.IsOpen())
			m_hamTrainView.Close();
		// Close the Spam-View
		m_spamView.Close();
		// Close the Inbox-View
		m_inboxView.Close();
		// Close the database.
		m_mailDb.Close();
		// Terminate the API.
		m_notesSession.Term();
	}
	catch (LNSTATUS lnerror)
	{
		char ErrorBuf[ERR_BUF_SIZE];

		LNGetErrorMessage(lnerror, ErrorBuf, ERR_BUF_SIZE);
		tmp.Format("Notes-Error(%i): %s",lnerror,ErrorBuf);
		myApp->debugger.Add(__FILE__,__LINE__,"**** ERROR: "+tmp);
		MessageBox(AfxGetMainWnd()->m_hWnd,tmp,"NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
		return;
	}
}

/*
 * searchSpam searches for Spam in the current Mailbox
 *
 * returns TRUE on normal exit
 *         FALSE if interrupted by m_hEventKillSearchThread
 */
BOOL CNotesMailDB::searchSpam(CDebugger *debugger, CSearchThreadInfo* pSearchInfo)
{
	LNINT			index;
	LNINT			numDocs;
	CString         tmp;
	BOOL            noSpam=FALSE;
//	CTrainIDCache	spamCache("idspam.gdbm"); // used if trained on eeverything
//	CTrainIDCache   hamCache("idham.gdbm");   // used if trained on eeverything


	try
	{
		LNItemArray itemArray;

		debugger->Add(__FILE__,__LINE__,"**************************************************************************************");
		debugger->Add(__FILE__,__LINE__,"*  NEW RUN                                                                           *");
		debugger->Add(__FILE__,__LINE__,"**************************************************************************************");
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("\r\n");
#ifdef _DEBUG
		tmp="----New Run----\r\n";
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
#endif

		// Refresh View "Inbox"
		m_inboxView.Refresh();

		numDocs = m_inboxView.GetEntryCount();
		tmp.Format("Document Count:  %i", numDocs);
		debugger->Add(__FILE__,__LINE__,tmp);
		
		for (index = 0; index < numDocs; index++)
		{
			//LNMailMessage curDoc;
			CNotesMail  curDoc;
			LNVFEntry   entry;  // Working entry
			LNText      textItem;
			LNString    tmpLNStr;
			CString     sender,subject;
			CString     dnsbl;
			BOOL        spam=false;
			BOOL        whitelist=false;
			CString     messageID;


			// Check if Thread should be killed
			if (WaitForSingleObject(pSearchInfo->m_hEventKillSearchThread, 0)  == WAIT_OBJECT_0)
			{
				/*
				if (hwndNotifyProgress != NULL)
				{
					// Reset progress indication to 0 (recalculation not begun)
					pWndNotifyProgress->PostMessage(WM_USER_RECALC_IN_PROGRESS);
				}
				*/
				return FALSE; // Terminate this recalculation
			}


			// Get Entry
			entry=m_inboxView[index];
			// we check only unread documents
			if(entry.IsUnread())
			{
				/*
				 * Check document
				 */
				entry.GetDocument(&curDoc);

				// open MailDocument
				curDoc.Open();

				// Check if Document already processed
				messageID=curDoc.GetMessageID();
				if(m_noteIDCache.Check(messageID))
				{
					if(debugger->GetState())
					{
						tmp.Format("**** MessageID %s already scanned ****\r\n",messageID);
						debugger->Add(__FILE__,__LINE__,tmp);
						((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
					}
					else
						((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(".");
					curDoc.Close();
					continue;
				}
				else
				{
					m_noteIDCache.AddEntry(messageID);
					if(debugger->GetState())
					{
						tmp.Format("**** scanning MessageID %s ****\r\n",messageID);
						debugger->Add(__FILE__,__LINE__,tmp);
						((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
					}
					else
						((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("+");
				}


				// Write the mail message to the log file if debug active.
				if(debugger->GetState())
				{
					debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
					debugger->Add(__FILE__,__LINE__,curDoc.dump());
					debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
				}

				// Get Subject for log
				subject=curDoc.GetSubjectString();

				// get Sender
				curDoc.GetSender(&tmpLNStr);
				sender=tmpLNStr.GetPlatformTextPtr();

				// check against whitelist
				if(checkAgainstWhitelist(sender))
				{
					tmp.Format("########### Sender %s listed in Whitelist ###########",sender);
					debugger->Add(__FILE__,__LINE__,tmp);
					whitelist=TRUE;
				}

				// Check with DSpam-Filter
				if(myApp->m_config.m_dSpamFilter && (!whitelist))
				{
					CString signature;
					if(m_dSpamFilter.CheckSpam(curDoc.dump(false),signature))
					{
						if(myApp->m_config.m_dSpam_trainMode==DST_TEFT)
						{
							curDoc.SetDSpamIsLearned(true);
						}
						curDoc.SetDSpamIsSpam(true);
						curDoc.SetDSpamSignature(signature);
						dnsbl="DSpam";
						spam=true;
						tmp.Format("########### DSpam Filter found spam from %s ###########",sender);
						debugger->Add(__FILE__,__LINE__,tmp);
					}
					else
					{
						if(myApp->m_config.m_dSpam_trainMode==DST_TEFT)
						{
							curDoc.SetDSpamIsLearned(true);
						}
						curDoc.SetDSpamIsSpam(false);
						curDoc.SetDSpamSignature(signature);
					}
					curDoc.Save();
				}

				// Check with DNSBL
				if(myApp->m_config.m_dnsblFilter && (!whitelist) && (!spam))
				{
					// Get Received lines for passed Mailgateways
					curDoc.GetReceivedLines(&textItem);
					for (unsigned int i = 0; i < textItem.GetCount(); i++)
					{
						CIPAddress ipAddress;	

						// Stop if Spam already identified
						if(spam) break;

						// get line
						tmpLNStr=textItem[i];
						tmp.Format("Received: %s",tmpLNStr.GetPlatformTextPtr());
						debugger->Add(__FILE__,__LINE__,tmp);

						// extract IP-Address
						ipAddress=extractIPAddress(tmpLNStr);

						tmp.Format("extracted IP-Address: %s",ipAddress.toString());
						debugger->Add(__FILE__,__LINE__,tmp);

						// check against good Cache
						if(m_goodDnsCache.Check(ipAddress))
						{
#ifdef _DEBUG
							tmp.Format("########### IP-Address %s listed in good Cache ###########",ipAddress.toString());
							debugger->Add(__FILE__,__LINE__,tmp);
#endif
							continue;
						}
						// check against bad Cache
						else if(m_dnsblCache.Check(ipAddress))
						{
							spam=true;
							dnsbl=m_dnsblCache.GetDnsbl4IPAddress(ipAddress);
							tmp.Format("########### IP-Address %s listed in Cache by %s ###########",ipAddress.toString(),dnsbl);
							debugger->Add(__FILE__,__LINE__,tmp);
						}
						// check against DNSBL
						else if(checkAgainstDnsbl(ipAddress,dnsbl))
						{
							tmp.Format("########### IP-Address %s listed by %s ###########",ipAddress.toString(),dnsbl);
							debugger->Add(__FILE__,__LINE__,tmp);
							spam=TRUE;
							m_dnsblCache.AddEntry(ipAddress,dnsbl);
						}
						// IP-Address not in DNS Blacklists
						else
						{
							m_goodDnsCache.AddEntry(ipAddress);
						}
					}
				}

				// Close Document
				curDoc.Close();

				// if spam found move document from "Inbox" to "-Spam"
				if(spam&&(!whitelist))
				{
					// create log
					tmp.Format("\r\n%s: spam from \"%s\" with subject \"%s\"\r\n",dnsbl,sender,subject);
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);

					// change subject
					tmp.Format("[SPAM:%s] %s",dnsbl,subject);
					curDoc.Open();
					curDoc.SetSubjectString(tmp);
					curDoc.Save();
					curDoc.Close();

					// move document
					if(m_spamView.FolderAddDocument(curDoc)==LNNOERROR)
						m_inboxView.FolderRemoveDocument(curDoc);
				}
				else
				{
					noSpam=TRUE;
				}
				if(whitelist)
				{
					tmp.Format("\r\nMail from whitelist-sender \"%s\" with subject \"%s\"\r\n",sender,subject);
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
				}
			}
		}

	}
	catch (LNSTATUS lnerror)
	{
		if(lnerror==3221225501)
		{
			((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("IGNORED: Array Subscript out of Rage\r\n");
			return TRUE;
		}
		else if(lnerror==551)
		{
			((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("IGNORED: Document not available\r\n");
			return TRUE;
		}
		else
		{
			char ErrorBuf[ERR_BUF_SIZE];
			
			LNGetErrorMessage(lnerror, ErrorBuf, ERR_BUF_SIZE);
			tmp.Format("Notes-Error(%u): %s",lnerror,ErrorBuf);
			myApp->debugger.Add(__FILE__,__LINE__,"**** ERROR: "+tmp);
			MessageBox(AfxGetMainWnd()->m_hWnd,tmp,"NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
			return TRUE;
		}
	}

	if(noSpam)
	{
		AfxGetMainWnd()->PostMessage(MYWM_INFORMNEWMAIL);
	}

	return TRUE;
}

CIPAddress CNotesMailDB::extractIPAddress(const LNString &lnStr)
{
	CString    str,tmp1,tmp2;

	str=lnStr;
	tmp1=str.Right(str.GetLength()-str.Find('[')-1);
	tmp2=tmp1.Left(tmp1.Find(']'));
/*	if(tmp2=="")
	{
		tmp1=str.Right(str.GetLength()-str.Find('(')-1);
		tmp2=tmp1.Left(tmp1.Find(')'));
		// Check if we found an IP-Adress
        TODO
	}
*/
	return CIPAddress(tmp2);
}

BOOL CNotesMailDB::checkAgainstDnsbl(const CIPAddress &ip, CString &dnsbl)
{
	CString tmp;
	hostent* myHost;

	for(int i=0; i<myApp->m_config.m_dnsbl.GetSize(); i++)
	{
		tmp=ip.reverseIPString()+myApp->m_config.m_dnsbl[i].m_domain;
		myHost=gethostbyname(tmp);
		if(myHost!=NULL)
		{
			dnsbl=myApp->m_config.m_dnsbl[i].m_name;
			return TRUE;
		}
	}
	return FALSE;
}

BOOL CNotesMailDB::checkAgainstWhitelist(const CString &sender)
{
	CWhitelistEntry entry;
	for(int i=0; i<myApp->m_config.m_whitelist.GetSize(); i++)
	{
		entry=myApp->m_config.m_whitelist[i];
		if(sender.Find(entry.m_address)>=0)
			return TRUE;
	}
	return FALSE;
}

CString CNotesMailDB::CopyIniFile()
{
	CString strFrom = "notes.ini";
	CString strTo   = "NotesAntiSpam.ini";
//	CString notesPath;
	char szBuffer[1024];
	CString tmp;

/*	// Read the Lotus-Notes Directory
	if(0>=GetPrivateProfileString("Lotus Applications",
	                              "Notes",
								  "",
								  szBuffer,
								  sizeof(szBuffer)-1,
								  "lotus.ini"))
	{
		MessageBox(NULL,"wrong lotus.ini","NotesAntiSpam",MB_OK|MB_ICONERROR);
		return "";
	}
	notesPath=szBuffer;
	notesPath+="\\";*/

	strFrom=myApp->m_config.m_notesIniFilename;
	strTo=myApp->m_config.m_userpath+strTo;

	// copy original file
	if(!CopyFile(strFrom,strTo,FALSE))
	{
		tmp.Format("***** ERROR: Unable to copy notes.ini from %s to %s",strFrom,strTo);
		myApp->debugger.Add(__FILE__,__LINE__,tmp);
	}

	// Read Addins
	GetPrivateProfileString("Notes",
	                        "EXTMGR_ADDINS",
							"",
							szBuffer,
							sizeof(szBuffer)-1,
							strTo);

	tmp=szBuffer;
	if(tmp.GetLength()>0)
	{
		if(tmp.Find("NASExtPwd")<0)
			tmp+=",NASExtPwd";
	}
	else
	{
		tmp="NASExtPwd";
	}

	WritePrivateProfileString("Notes",
	                          "EXTMGR_ADDINS",
							  tmp,
							  strTo);

	return strTo;
}
/*
 * trainDSpam: trains the dSpam filter based on TrainHam and TrainSpam
 *
 * returns TRUE on normal exit
 *         FALSE if interrupted by m_hEventKillSearchThread
 */
BOOL CNotesMailDB::trainDSpam(CDebugger *debugger, CSearchThreadInfo* pSearchInfo)
{
	LNINT			index;
	LNINT			numDocs;
	CString         tmp;

	try
	{
		LNItemArray itemArray;

		debugger->Add(__FILE__,__LINE__,"**************************************************************************************");
		debugger->Add(__FILE__,__LINE__,"*  NEW TRAINDPSPAM RUN                                                                           *");
		debugger->Add(__FILE__,__LINE__,"**************************************************************************************");
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("\r\n");
#ifdef _DEBUG
		tmp="----New TrainDSpam-Run----\r\n";
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
#endif

		debugger->Add(__FILE__,__LINE__,"TRAIN HAM FROM -Spam\\TrainHam");
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("Train Ham from -Spam\\TrainHam\r\n");

		// Refresh View
		m_hamTrainView.Refresh();

		numDocs = m_hamTrainView.GetEntryCount();
		tmp.Format("Document Count:  %i", numDocs);
		debugger->Add(__FILE__,__LINE__,tmp);
		
		for (index = 0; index < numDocs; index++)
		{
			CNotesMail  curDoc;
			LNVFEntry   entry;  // Working entry
			LNText      textItem;
			LNString    tmpLNStr;
			CString     messageID;

			// Check if Thread should be killed
			if (WaitForSingleObject(pSearchInfo->m_hEventKillSearchThread, 0)  == WAIT_OBJECT_0)
			{
				/*
				if (hwndNotifyProgress != NULL)
				{
					// Reset progress indication to 0 (recalculation not begun)
					pWndNotifyProgress->PostMessage(WM_USER_RECALC_IN_PROGRESS);
				}
				*/
				return FALSE; // Terminate this recalculation
			}


			// Get Entry
			entry=m_hamTrainView[index];
			entry.GetDocument(&curDoc);

			// open MailDocument
			curDoc.Open();
			messageID=curDoc.GetMessageID();

			if(curDoc.GetDSpamIsLearned()&&(!curDoc.GetDSpamIsSpam()))
			{
				if(debugger->GetState())
				{
					tmp.Format("**** MessageID %s already trained as HAM ****",messageID);
					debugger->Add(__FILE__,__LINE__,tmp);
#ifdef _DEBUG
					tmp+="\r\n";
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
#else
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(".");
#endif
				}
				// Close Document
				curDoc.Close();
				continue;
			}
			else
			{
				if(debugger->GetState())
				{
					tmp.Format("**** training NoteID %s as HAM ****\r\n",messageID);
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
				}
				else
				{
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("+");
				}

			}
			/*
			 * Check document
			 */

			// Write the mail message to the log file if debug active.
			if(debugger->GetState())
			{
				debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
				debugger->Add(__FILE__,__LINE__,curDoc.dump());
				debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
			}

			// Remove our own comment from the Subject
			if(curDoc.GetSubjectString().Find("[SPAM:")==0)
			{
				CString tempSubject;
				tempSubject=curDoc.GetSubjectString();
				tempSubject=tempSubject.Right(tempSubject.GetLength()-tempSubject.Find("]")-2);
				curDoc.SetSubjectString(tempSubject);
				curDoc.Save();
			}

			// Check if maybe already trained as spam
			if(curDoc.GetDSpamIsSpam()&&curDoc.GetDSpamIsLearned())
			{
				if(debugger->GetState())
				{
					tmp.Format("**** untraining MessageID %s as SPAM ****\r\n",messageID);
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
				}
				else
				{
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("U");
				}
				// untrain as spam
				CString signature=curDoc.GetDSpamSignature();
				if(signature!="")
					m_dSpamFilter.TrainSpamAsHam(signature);
				else
				{
					m_dSpamFilter.TrainHam(curDoc.dump(false));
				}
			}
			else
			{
				// train as ham
				m_dSpamFilter.TrainHam(curDoc.dump(false));
			}
			curDoc.SetDSpamIsLearned(true);
			curDoc.SetDSpamIsSpam(false);
			curDoc.Save();

			// close MailDocument
			curDoc.Close();

			// move to "Inbox" after trained
			if(m_inboxView.FolderAddDocument(curDoc)==LNNOERROR)
				m_hamTrainView.FolderRemoveDocument(curDoc);
		}

		debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
		debugger->Add(__FILE__,__LINE__,"TRAIN SPAM FROM -Spam\\TrainSpam");
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("Train Spam from -Spam\\TrainSpam\r\n");

		// Refresh View "-Spam\\TrainSpam"
		m_spamTrainView.Refresh();

		numDocs = m_spamTrainView.GetEntryCount();
		tmp.Format("Document Count:  %i", numDocs);
		debugger->Add(__FILE__,__LINE__,tmp);
		
		for (index = 0; index < numDocs; index++)
		{
			CNotesMail  curDoc;
			LNVFEntry   entry;  // Working entry
			LNText      textItem;
			LNString    tmpLNStr;
			CString     messageID;


			// Check if Thread should be killed
			if (WaitForSingleObject(pSearchInfo->m_hEventKillSearchThread, 0)  == WAIT_OBJECT_0)
			{
				/*
				if (hwndNotifyProgress != NULL)
				{
					// Reset progress indication to 0 (recalculation not begun)
					pWndNotifyProgress->PostMessage(WM_USER_RECALC_IN_PROGRESS);
				}
				*/
				return FALSE; // Terminate this recalculation
			}


			// Get Entry
			entry=m_spamTrainView[index];
			entry.GetDocument(&curDoc);

			// open MailDocument
			curDoc.Open();
			messageID=curDoc.GetMessageID();

			if(curDoc.GetDSpamIsLearned()&&curDoc.GetDSpamIsSpam())
			{
				if(debugger->GetState())
				{
					tmp.Format("**** MessageID %s already trained as SPAM ****",messageID);
					debugger->Add(__FILE__,__LINE__,tmp);
#ifdef _DEBUG
					tmp+="\r\n";
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
#else
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(".");
#endif
				}
				// Close Document
				curDoc.Close();
				continue;
			}
			else
			{
				if(debugger->GetState())
				{
					tmp.Format("**** training NoteID %s as SPAM ****\r\n",messageID);
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
				}
				else
				{
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("+");
				}
			}

			/*
			 * Check document
			 */

			// Write the mail message to the log file if debug active.
			if(debugger->GetState())
			{
				debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
				debugger->Add(__FILE__,__LINE__,curDoc.dump());
				debugger->Add(__FILE__,__LINE__,"*******************************************************************************************");
			}

			// Remove our own comment from the Subject
			if(curDoc.GetSubjectString().Find("[SPAM:")==0)
			{
				CString tempSubject;
				tempSubject=curDoc.GetSubjectString();
				tempSubject=tempSubject.Right(tempSubject.GetLength()-tempSubject.Find("]")-2);
				curDoc.SetSubjectString(tempSubject);
				curDoc.Save();
			}

			// Check if maybe already trained as spam
			if(curDoc.GetDSpamIsLearned()&&(!curDoc.GetDSpamIsSpam()))
			{
				if(debugger->GetState())
				{
					tmp.Format("**** untraining MessageID %s as HAM ****\r\n",messageID);
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
				}
				else
				{
					((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("U");
				}
				// untrain as spam
				CString signature=curDoc.GetDSpamSignature();
				if(signature!="")
					m_dSpamFilter.TrainHamAsSpam(signature);
				else
				{
					m_dSpamFilter.TrainSpam(curDoc.dump(false));
				}
			}
			else
			{
				// train as Spam
				m_dSpamFilter.TrainSpam(curDoc.dump(false));
			}
			curDoc.SetDSpamIsLearned(true);
			curDoc.SetDSpamIsSpam(true);
			curDoc.Save();

			// close MailDocument
			curDoc.Close();
			
			// move to "-Spam" after trained
			if(m_spamView.FolderAddDocument(curDoc)==LNNOERROR)
				m_spamTrainView.FolderRemoveDocument(curDoc);
		}
	}
	catch (LNSTATUS lnerror)
	{
		char ErrorBuf[ERR_BUF_SIZE];

		LNGetErrorMessage(lnerror, ErrorBuf, ERR_BUF_SIZE);
		tmp.Format("Notes-Error(%i): %s",lnerror,ErrorBuf);
		myApp->debugger.Add(__FILE__,__LINE__,"**** ERROR: "+tmp);
		MessageBox(AfxGetMainWnd()->m_hWnd,tmp,"NotesAntiSpam",MB_OK|MB_ICONSTOP|MB_SYSTEMMODAL);
		return TRUE;
	}

	return TRUE;
}

/*
 * purgeDSpam: purges unused Tokens in the DSpam-Database
 *
 * returns TRUE on normal exit
 *         FALSE if interrupted by m_hEventKillSearchThread
 */
BOOL CNotesMailDB::purgeDSpam(CDebugger *debugger, CSearchThreadInfo* pSearchInfo)
{
	// Check if Thread should be killed
	if (WaitForSingleObject(pSearchInfo->m_hEventKillSearchThread, 0)  == WAIT_OBJECT_0)
	{
		return FALSE; // Terminate this calculation
	}

	((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("\r\nStarting cleanup of DSpam-Database...\r\n");

	/* TODO : get times from config */
	m_dSpamFilter.CleanupDatabase();

	((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog("\r\n-->finished\r\n");

	// Check if Thread should be killed
	if (WaitForSingleObject(pSearchInfo->m_hEventKillSearchThread, 0)  == WAIT_OBJECT_0)
	{
		return FALSE; // Terminate this calculation
	}

	return TRUE;
}

LNDatetime CNotesMailDB::GetLastModified()
{
	LNDatetime lastMod;
	CString    tmp;

	lastMod.SetDate(1,1,51);
	lastMod.SetTime(0,0,0,0);

	try
	{
		lastMod=m_mailDb.GetLastModified();
	}
	catch (LNSTATUS lnerror)
	{
		char ErrorBuf[ERR_BUF_SIZE];

		LNGetErrorMessage(lnerror, ErrorBuf, ERR_BUF_SIZE);
		tmp.Format("Notes-Error(%i): %s",lnerror,ErrorBuf);
		myApp->debugger.Add(__FILE__,__LINE__,"**** ERROR: "+tmp);
		((CNotesAntiSpamDlg*)AfxGetMainWnd())->AddLog(tmp);
		return lastMod;
	}

	return lastMod;
}
