#!/usr/bin/env ruby
# This software is a part of NOODLYBOX.
# This software is distributed under the terms of the new BSD License.
# Copyright (c) 2008, molelord
# All rights reserved.

# ToDo
# Handling arrayed register.  example : u16 FOOBAR[8]

require 'csv'

if (ARGV.length < 2)
    print <<EOF
Usage : accessor.rb -h foobar.csv
        (for generate a header)
                 or
        accessor.rb -c foobar.csv
        (for generate an implementation)
EOF
    exit 1
end

# The "Register" objects are created from each line.
class Register
    def initialize(addr, type, name, comment)
        @addr     = addr
        @type     = type

        @name     = name
        @elements = 1
        if (name =~ /(.*)\[(\d+)\]$/) then
            @name     = $1
            @elements = $2.to_i
        end

        @comment  = comment
    end
    attr_accessor :addr
    attr_accessor :name
    attr_accessor :elements
    attr_accessor :type
    attr_accessor :comment
end

# ---- make foobar.h  -----------------------------------------------------
def putHeader(commandline, csvname, reg)
printf "// %s\n", commandline
printf "#ifndef %s_H_\n", csvname.upcase
printf "#define %s_H_\n", csvname.upcase

print <<EOF

#if !defined(BITWIDTHTYPES_ARE_DEFINED)
#   define BITWIDTHTYPES_ARE_DEFINED
    typedef unsigned int   u32;
    typedef unsigned short u16;
    typedef unsigned char   u8;
    typedef signed   int   s32;
    typedef signed   short s16;
    typedef signed   char   s8;
#endif

EOF

printf "typedef struct %s {\n", csvname.capitalize
printf "    %s  (*get%s)(void);\n", "int", "Success"

reg.each do |i|
    if (i.elements > 1) then
        printf "    %s  (*get%s[%d])(void);\n", i.type, i.name, i.elements
        printf "    void (*set%s[%d])(%s data);\n", i.name, i.elements, i.type
    else
        printf "    %s  (*get%s)(void);\n", i.type, i.name
        printf "    void (*set%s)(%s data);\n", i.name, i.type
    end
end

printf "} %s;\n", csvname.capitalize

print <<EOF

#if defined(__cplusplus)
extern "C" {
#endif
EOF

printf "%s *%s_instance(void);\n", csvname.capitalize, csvname.capitalize

print <<EOF
#if defined(__cplusplus)
}
#endif

#endif
EOF
end

# ---- make foobar.c  -----------------------------------------------------
def putImplementation(commandline, csvname, reg)
printf "// %s\n", commandline
printf "#include \"%s.h\"\n", csvname
print "\n"
print <<EOF
#if !defined(REALWORLD)
#   include "noodlybox.h"
    static Processor *cpu;
#   define FUNC(t,x) static t  get##x(void) { return cpu->readH(x);} \\
                     static void set##x(t data) { cpu->writeH(x, data);}
#else
#   define FUNC(t,x) static volatile t * const p##x = (u16 *)(x); \\
                     static t  get##x(void) { return *p##x;} \\
                     static void set##x(t data) { *p##x = data;}
#endif

static int getSuccess(void)
{
#if !defined(REALWORLD)
    return cpu->readSuccess();
#else
    return 1;
#endif
}
EOF

print "// Adderss\n"
reg.each do |i|
    printf "#define %-10s %sU\n", i.name, i.addr
end
print "\n"

print "// Private function implementations\n"
reg.each do |i|
    printf "FUNC(%s, %s)\n", i.type, i.name
end
print "\n"

print "// Public function implementation\n"
printf "%s *%s_instance(void)\n", csvname.capitalize, csvname.capitalize
print  "{\n"
printf "    static %s funcinst;\n", csvname.capitalize

print <<EOF
    static int initialized;
    if (!initialized) {
        initialized = 1;
#if !defined(REALWORLD)
        cpu = Processor_instance();
#endif
EOF

printf "        funcinst.get%-10s = get%s;\n", "Success", "Success"
reg.each do |i|
    printf "        funcinst.get%-10s = get%s;\n", i.name, i.name
    printf "        funcinst.set%-10s = set%s;\n", i.name, i.name
end

print <<EOF
    }
    return &funcinst;
}
EOF
end

# ---- main ---------------------------------------------------------------

reg = []

first = true
addr_num    = -1
type_num    = -1
name_num    = -1
comment_num = -1
CSV.open(ARGV[1], 'r') do |row|
    if (first)
        first = false
        i = 0
        while i < row.length
            case row[i]
            when "addr"
                addr_num = i
            when "type"
                type_num = i
            when "name"
                name_num = i
            when "comment"
                comment_num = i
            end
            i += 1
        end
        if (addr_num == -1 || type_num == -1 ||
            name_num == -1 || comment_num == -1)
            print "Error : Title line must include addr, type, name and comment."
            exit 1
        end
    elsif (row.length >= 3)
        reg.push(Register.new(
            row[addr_num],
            row[type_num],
            row[name_num],
            row[comment_num]))
    end
end


commandline = sprintf "accessor.rb %s %s", ARGV[0], ARGV[1]

# The filename of CSV is used for naming.
csvname = ARGV[1].sub(/^.*\//, "").sub(/\.csv$/i, "")

if (ARGV[0] == "-h") then
    putHeader(commandline, csvname, reg)
elsif (ARGV[0] == "-c") then
    putImplementation(commandline, csvname, reg)
end

