# This software is a part of NOODLYBOX.
# This software is distributed under the terms of the new BSD License.
# Copyright (c) 2008, molelord
# All rights reserved.

# Defining echo canceller.
proc silent args {
}

silent [package require cmdline]

# Flying Spaghetti Machine
namespace eval Fsm {
    variable meatball       2
    variable channelMode    ""
    variable channelTarget  ""
    variable waveTclScript  ""
    variable topDesignUnit  tNOODLYBOX
    variable noodlyboxPath  /$topDesignUnit/uBOX
    variable readBodyWidth  1
    variable writeBodyWidth 1

    variable clkPeriod      -1
    variable clkFuture      0
    variable syncRequired   0

    variable nextPhase      "reset"
    variable execAtNextRise ""
    variable execAtNextFall ""
    variable nopCount       0
    variable readBodyCount  0
    variable writeBodyCount 0
    variable result         0
    variable iChannel       0
    variable oChannel       0
    variable clkTick        0
    variable dbg            0

    variable msim           0
    variable isesim         0
    variable forceArray
    variable forceCountArray
    variable forceLastArray
    variable forced

    variable zero           0
    variable one            1

    proc checkEnvironment {} {
        set name [info nameofexecutable]
        # Remove the path and the extension.
        regsub "^.*/" $name "" name
        regsub "\.exe$" $name "" name

        putdbg $name

        if {$name eq "vish"} {
            variable msim   1
        } else {
            variable isesim 1
        }
    }


# In the following procedures, "variable" is not used if it is not modified.

    # Put the debug information.
    proc putdbg {str} {
        if {$Fsm::dbg} { puts stderr $str }
    }

    # The following descriptions absorb the difference
    # of the simulators.
    proc simForce {target value {time "0 ns"}} {
        regsub " ns" $time "" time
        set time [expr $time + $Fsm::clkPeriod * $Fsm::clkFuture]
        append time " ns"

        if {$Fsm::msim} {
            putdbg "force -deposit $target $value $time"
            if {$::hdl eq "vhdl" || $target ne "${Fsm::noodlyboxPath}/i_PHASE"} {
                force -deposit $target $value $time
            }
        } elseif {$Fsm::isesim} {
            variable forceArray
            variable forceCountArray
            variable forceLastArray

            set radix ""
            if {[regsub "^16#" $value "" value] != 0} {
                set radix "-radix hex"
            }

            regsub " ns" $time "" time
            # Convert from ns to femto
            set time [expr $time * 1000 * 1000]

            if {[array get forceArray $target] eq ""} {
                set forceArray($target) "isim force add $target"
                set forceCountArray($target) 0
                if {[array get forceLastArray $target] ne ""} {
                    append forceArray($target) " $forceLastArray($target) -value"
                    incr forceCountArray($target)
                }
            } else {
                append forceArray($target) " -value"
            }
            append forceArray($target) " $value $radix -time $time"
            incr forceCountArray($target)

            # When specified options are too long,
            # ISE Simulator fails in the interpretation.
            if {$forceCountArray($target) == 12} {
                variable syncRequired 1
            }

            # The last value is continued.
            set forceLastArray($target) "$value $radix"
        }
    }

    proc simExamine {a {b ""} {c ""} {d ""}} {
        if {$Fsm::clkFuture != 0} {
            puts stderr "FSM internal error : clkFuture is $Fsm::clkFuture." 
            variable clkTick 1
            ignoreClkRise
            stop
        }

        if {$b eq ""} {
            return [examine $a]
        } elseif {$c eq ""} {
            return [examine $a $b]
        } elseif {$d eq ""} {
            return [examine $a $b $c]
        }
        return     [examine $a $b $c $d]
    }

    proc hookRstNegate {} {
        if {$Fsm::msim} {
            eval "when -label atRstNegate { ${Fsm::noodlyboxPath}/RESET_X == 1 } { Fsm::rstNegate }"
        } elseif {$Fsm::isesim} {
            eval "isim condition add { \$\{Fsm::noodlyboxPath\}/RESET_X == $Fsm::one } {
                Fsm::rstNegate
            } -label atRstNegate"
        }
    }
    proc ignoreRstNegate {} {
        if {$Fsm::msim} {
            nowhen atRstNegate
        } elseif {$Fsm::isesim} {
            isim condition remove -label atRstNegate
        }
    }

    proc hookClkRise {} {
        if {$Fsm::msim} {
            eval "when -label atClkRise { ${Fsm::noodlyboxPath}/CLK == 1 } { Fsm::clkRise }"
        } elseif {$Fsm::isesim} {
            eval "isim condition add { \$\{Fsm::noodlyboxPath\}/CLK == $Fsm::one } {
                Fsm::clkRise
            } -label atClkRise"
            putdbg "trap clkRise"
        }
    }
    proc ignoreClkRise {} {
        if {$Fsm::msim} {
            nowhen atClkRise
        } elseif {$Fsm::isesim} {
            isim condition remove -label atClkRise
        }
    }

    proc hookWakeUp {time unit} {
        if {$Fsm::msim} {
            eval "when -label atWakeUp { \$now == $time $unit } { Fsm::wakeup }"
        } elseif {$Fsm::isesim} {
            putdbg "trap wakeup begin"
            isim force add ${Fsm::noodlyboxPath}/i_WAKEUP $Fsm::zero -time "$time $unit"

            eval "isim condition add ${Fsm::noodlyboxPath}/i_WAKEUP { Fsm::wakeup } -label atWakeUp"
        }
    }
    proc ignoreWakeUp {} {
        if {$Fsm::msim} {
            nowhen atWakeUp
        } elseif {$Fsm::isesim} {
            isim force remove ${Fsm::noodlyboxPath}/i_WAKEUP
            isim condition remove -label atWakeUp
        }
    }

    # Put current phase to i_PHASE signal.
    proc putPhase {str} {
        set phaseStringLen 32
        if {$Fsm::msim} {
            append str [string repeat " " $phaseStringLen]
        } elseif {$Fsm::isesim} {
            append str [string repeat "_" $phaseStringLen]
        }

        set str [string range $str 0 [expr $phaseStringLen - 1]]
        if {$Fsm::msim} {
            simForce ${Fsm::noodlyboxPath}/i_PHASE $str
        }
    }

    proc config {} {
        variable channelMode
        variable channelTarget
        variable iChannel
        variable oChannel

        # Configurable variables.
        variable topDesignUnit
        variable noodlyboxPath
        variable readBodyWidth
        variable writeBodyWidth
        variable dbg

        if {$channelMode eq "file"} {
            set iChannel [open $channelTarget r]
            set oChannel stdout
        } elseif {$channelMode eq "command"} {
            set iChannel [open "|$channelTarget" r+]
            set oChannel $iChannel
            fconfigure $oChannel -buffering line
        }
        #set server localhost
        #set sockchannel [socket $server 9900]

        set longline ""
        while {1} {
            # Get next line.
            set rc [gets $iChannel line]
            if {$rc == -1} {
                puts stderr "FSMError:endOfConfig is not found."
                abort
            }

            regsub "^ +" $line "" line
            putdbg "config: $line"
            if {$line eq "endOfConfig"} {
                return
            }

            # Ignore any comments
            if [regexp "^\s*#" $line] {
                continue
            }
            append longline $line

            # Replace "\".
            set match ""

            if {[regexp "\[\{\}\] *(.)$" $line dummy match]} {
                set to " "
            } elseif {[regexp "(.)$" $line dummy match]} {
                set to ";"
            }

            if {$match eq "\\"} {
                regsub ".$" $longline $to longline
            } else {
                putdbg $longline
                eval $longline
                set longline ""
            }
        }
    }

    proc loadDesign {designUnit} {
        if {$Fsm::msim} {
            if {$::hdl eq "vhdl"} {
                vsim -t ns $designUnit
            } else {
                vsim -t ns -L unisims_ver $designUnit
            }
        }
    }
    
    proc examineClkPeriod {} {
        if {$Fsm::isesim} {
            if {$::hdl eq "vhdl"} {
                set time [simExamine $Fsm::noodlyboxPath/clock_period]
            } else {
                set time [simExamine -dec $Fsm::noodlyboxPath/CLOCK_PERIOD]
            }
        } else {
            set time [simExamine $Fsm::noodlyboxPath/CLOCK_PERIOD]
        }
        regsub "\{" $time "" time
        regsub "\}" $time "" time

        if       {[regsub " ns$" $time "" time]} {
            ;
        } elseif {[regsub " ps$" $time "" time]} {
            set time [expr $time / 1000]
        } elseif {[regsub " fs$" $time "" time]} {
            set time [expr $time / 1000 / 1000]
        }

        putdbg $time
        variable clkPeriod $time
    }

    # Called when -w foo.do is not specified.
    proc defaultWave {designUnit} {
        if {$Fsm::msim} {
            set full $Fsm::clkPeriod
            set half [expr $full / 2]
            append full "ns"
            append half "ns"
            configure wave -gridoffset $half -gridperiod $full -timeline 1
            add wave -hex -r /*
        } elseif {$Fsm::isesim} {
            ntrace select -m /$designUnit -l all
            ntrace start
        }
    }

    proc dataLatch {} {
        variable result [simExamine -hex ${Fsm::noodlyboxPath}/D]
        putdbg "latch: $result"
    }

    proc reset {} {
        putPhase "reset"
        hookRstNegate
    }

    proc rstNegate {} {
        putdbg "rstNegate"
        ignoreRstNegate
        variable nextPhase "fetch"
        hookClkRise
    }

    # Sleep
    proc opSleep {time unit} {
        variable nextPhase "fetch"
        variable clkTick 1

        changeAtOpNop
        putPhase "opSleep $time $unit"

        if {$unit eq "ms"} {
            set time [expr $time * 1000000]
        } elseif {$unit eq "us"} {
            set time [expr $time * 1000]
        }
        set clocks [expr $time / $Fsm::clkPeriod - 1]

        variable clkFuture
        incr clkFuture $clocks
        variable syncRequired 1
    }
    proc wakeup {} {
        ignoreWakeUp
        hookClkRise
    }

    proc changeAtOpNop {} {
        simForce ${Fsm::noodlyboxPath}/i_HIZ  1
        simForce ${Fsm::noodlyboxPath}/i_CS_X 1
        simForce ${Fsm::noodlyboxPath}/i_OE_X 1
        simForce ${Fsm::noodlyboxPath}/i_WE_X 1
    }

    # No operation
    proc opNop {count} {
        variable nextPhase "opNop_tail"
        variable nopCount $count
        variable clkTick 1

        # Update the output directional signals.
        changeAtOpNop

        incr nopCount -1
        if {$nopCount == 0} { set nextPhase "fetch" }
    }
    proc opNop_tail {} {
        variable nopCount
        variable clkTick 1

        incr nopCount -1
        if {$nopCount == 0} { variable nextPhase "fetch" }
    }

    proc changeAtOpRead {addr} {
        simForce ${Fsm::noodlyboxPath}/i_A    $addr
        simForce ${Fsm::noodlyboxPath}/i_HIZ  1
        simForce ${Fsm::noodlyboxPath}/i_CS_X 0
        simForce ${Fsm::noodlyboxPath}/i_OE_X 1
        simForce ${Fsm::noodlyboxPath}/i_WE_X 1
    }
    proc changeAtOpRead_body {} {
        simForce ${Fsm::noodlyboxPath}/i_OE_X 0
        putdbg "changeAtOpRead_body"
        variable syncRequired 1
    }
    proc changeAtOpRead_tail {} {
        simForce ${Fsm::noodlyboxPath}/i_OE_X 1
        dataLatch
    }

    # Read operation
    proc opRead {addr} {
        variable nextPhase "opRead_body"
        variable clkTick 1
        variable readBodyCount $Fsm::readBodyWidth

        # Update the output directional signals.
        changeAtOpRead $addr

        if {$readBodyCount == 0} { set nextPhase "opRead_tail" }
    }
    proc opRead_body {} {
        variable clkTick 1
        variable readBodyCount

        changeAtOpRead_body

        incr readBodyCount -1
        if {$readBodyCount == 0} { variable nextPhase "opRead_tail" }
    }
    proc opRead_tail {} {
        variable nextPhase "fetch"
        variable clkTick 1

        # Update the output directional signals.
        changeAtOpRead_tail
    }

    proc changeAtOpWrite {addr data} {
        simForce ${Fsm::noodlyboxPath}/i_A    $addr
        simForce ${Fsm::noodlyboxPath}/i_D    $data
        simForce ${Fsm::noodlyboxPath}/i_HIZ  0
        simForce ${Fsm::noodlyboxPath}/i_CS_X 0
        simForce ${Fsm::noodlyboxPath}/i_OE_X 1
        simForce ${Fsm::noodlyboxPath}/i_WE_X 1
    }
    proc changeAtOpWrite_body {} {
        simForce ${Fsm::noodlyboxPath}/i_WE_X 0
    }
    proc changeAtOpWrite_tail {} {
        simForce ${Fsm::noodlyboxPath}/i_WE_X 1
    }

    # Write operation
    proc opWrite {addr data} {
        variable nextPhase "opWrite_body"
        variable clkTick 1
        variable writeBodyCount $Fsm::writeBodyWidth

        # Update the output directional signals.
        changeAtOpWrite $addr $data

        if {$writeBodyCount == 0} { set nextPhase "opWrite_tail" }
    }
    proc opWrite_body {} {
        variable clkTick 1
        variable writeBodyCount

        # If it is 1st, ...
        if {$writeBodyCount == $Fsm::writeBodyWidth} {
            # Update the output directional signals.
            changeAtOpWrite_body
        }

        incr writeBodyCount -1
        if {$writeBodyCount == 0} { variable nextPhase "opWrite_tail" }

    }
    proc opWrite_tail {} {
        variable nextPhase "fetch"
        variable clkTick 1

        # Update the output directional signals.
        changeAtOpWrite_tail
    }

    # End of simulation.
    proc endOfSimulation {} {
        variable iChannel
        variable clkTick 1
        variable syncRequired 1

        putdbg "endOfSimulation"
        #close $iChannel
        stop
    }

    # Fetch new line from the channel, and execute it.
    proc fetch {} {
        variable iChannel
        variable oChannel

        set rc [gets $iChannel line]
        if {$rc == -1} {
            # End of file is end of sim.
            endOfSimulation
            putPhase "endOfSimulation"
        } else {
            putdbg "fetch: $line"

            # Ignore any comments
            if [regexp "^\s*#" $line] {
                return
            }

            variable result
            eval $line
            putPhase $line
        }
    }

    # This function is called at least rising edge of the clock.
    proc clkRise {} {
        variable nextPhase
        variable execAtNextRise
        variable execAtNextFall

        ignoreClkRise

        variable clkFuture 0
        while {1} {

            if {$nextPhase ne "fetch"} {
                putdbg $nextPhase
                putPhase $nextPhase
            }

            eval $execAtNextRise
            set execAtNextRise ""

            # $syncRequired == 0 means that examining is not required.
            variable syncRequired 0

            # $clkTick == 0 means that time does not pass.
            variable clkTick 0
            while {$clkTick == 0} {
                # Call procedure that was named $nextPhase.
                eval $nextPhase
            }

            if {$execAtNextFall ne ""} {
                set tmp $clkFuture
                set clkFuture [expr $clkFuture + 0.5]

                eval $execAtNextFall
                set execAtNextFall ""

                set clkFuture $tmp
            }

            incr clkFuture

            if {$syncRequired == 1} {
                if {$Fsm::msim} {
                    putdbg "syncRequired clkFuture:$clkFuture @ $::now"
                } elseif {$Fsm::isesim} {
                    putdbg "syncRequired clkFuture:$clkFuture"

                    variable forceArray
                    variable forced
                    set id [array startsearch forceArray]

                    while {1} {
                        set target [array nextelement forceArray $id]
                        if {$target eq ""} {
                            break
                        }
                        putdbg $target
                        putdbg $forceArray($target)

                        if {[array get forced $target] eq ""} {
                            set forced($target) 1
                        } else {
                            isim force remove $target
                        }

                        eval "$forceArray($target)"
                    }
                    unset forceArray
                }
                if {$clkFuture == 1} {
                    hookClkRise
                } else {
                    hookWakeUp [expr $Fsm::clkPeriod * $clkFuture - 1] "ns"
                }
                break
            }
        }
    }
}

# main ------------------------------------------------------------------------

Fsm::checkEnvironment

# Gather all specified options.
if {$Fsm::msim} {
    set specified ""
    while {$argc > 0} {
        append specified " " $1
        shift
    }
    alias fsm "do fsm.tcl $specified"
} else {
    set specified $env(FSMARGS)
}
Fsm::putdbg "-- $specified --"

while {[::cmdline::getopt specified {f.arg c.arg w.arg l.arg} optvar valvar]} {
    if {$optvar eq "f"} {
        set Fsm::channelMode   "file"
        set Fsm::channelTarget $valvar
    } elseif {$optvar eq "c"} {
        set Fsm::channelMode   "command"
        set Fsm::channelTarget $valvar
    } elseif {$optvar eq "w"} {
        set Fsm::waveTclScript $valvar
    } elseif {$optvar eq "l"} {
        set ::hdl $valvar
    }
}
if {$Fsm::channelMode eq ""} {
    puts stderr "fsm.tcl needs to specify -f file or -c command." 
    abort
}
if {$::hdl eq "vhdl"} {
    set Fsm::zero "'0'"
    set Fsm::one  "'1'"
}

set compileSuccess 1

if {$Fsm::msim} {
    quit -sim
    set compileSuccess [checkResult "project compileoutofdate"]
}

if {$compileSuccess} {

    Fsm::config
    Fsm::loadDesign $Fsm::topDesignUnit
    Fsm::examineClkPeriod

    if {$Fsm::waveTclScript ne ""} {
        source $Fsm::waveTclScript
    } else {
        Fsm::defaultWave $Fsm::topDesignUnit
    }

    Fsm::reset

    set StdArithNoWarnings 1
    run 0 ns
    set StdArithNoWarnings 0
    run -all
}

if {$Fsm::isesim} {
    quit
}
