// $Id: ghkcurrent.h 26 2004-07-03 06:22:39Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_GHKCURRENT_H
#define NSIM_GHKCURRENT_H

#include <nsim/functions.h>
#include <nsim/pow.h>

namespace nsim
{

template <int P, int Q>
class ghkcurrent
{
public:
    ghkcurrent(int zion, double temp, double p_max) :
        p_max_(p_max), temp_(temp), z_ion_(zion) { init_(); }

    void set_z_ion(int val)
    {
        z_ion_ = val;
        init_();
    }

    void set_temp(double val)
    {
        temp_ = val;
        init_();
    }

    void set_p_max(double val)
    {
        p_max_ = val;
    }

    double operator()(double v, double m, double h,
            double ion_in, double ion_out) const
    {
        double x(xi*v), l(lin::apply(x)), e(exp::apply(-x));
        return pow<P>(m)*pow<Q>(h)
            *p_max*l*(ion_in - ion_out*e);
    }

    double deriv_v(double v, double m, double h,
            double ion_in, double ion_out) const
    {
        double x(xi*v), l(lin::apply(x)), e((l - x)/l), dl(lin::deriv(x));
        return pow<P>(m)*pow<Q>(h)
            *p_max*xi*(dl*(ion_in - ion_out*e) - ion_out*l*e);
    }

    double deriv_m(double v, double m, double h,
            double ion_in, double ion_out) const
    {
        double x(xi*v), l(lin::apply(x)), e(exp::apply(-x));
        return P*pow<P-1>(m)*pow<Q>(h)
            *p_max*l*(ion_in - ion_out*e);
    }

    double deriv_h(double v, double m, double h,
            double ion_in, double ion_out) const
    {
        double x(xi*v), l(lin::apply(x)), e(exp::apply(-x));
        return Q*pow<P>(m)*pow<Q-1>(h)
            *p_max*l*(ion_in - ion_out*e);
    }

    double deriv_out(double v, double m, double h,
            double ion_in, double ion_out) const
    {
        double x(xi*v), l(lin::apply(x)), e((l - x)/l);
        return -pow<P>(m)*pow<Q>(h)*p_max*l*e;
    }

    double deriv_in(double v, double m, double h,
            double ion_in, double ion_out) const
    {
        double x(xi*v), l(lin::apply(x));
        return pow<P>(m)*pow<Q>(h)*p_max*l;
    }

private:
    double p_max_;
    double temp_;
    double xi_;
    int z_ion_;

    void init_() { xi_ = z_ion_ * (96.5/8.31)/(273+temp_); }
};

}

#endif
