// $Id: limitcycle.h 26 2004-07-03 06:22:39Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_LIMITCYCLE_H
#define NSIM_LIMITCYCLE_H

#include <deque>
#include <vector>
#include <limits>
#include <nsim/orbit.h>

namespace nsim
{

template <typename V>
class limitcycle
{
public:
    typedef std::size_t size_type;
    typedef V vector_type;
    typedef typename vector_type::value_type value_type;
    typedef orbit<V> orbit_type;
    typedef typename orbit_type::interp_type interp_type;
    typedef typename orbit_type::state_type state_type;
    typedef typename orbit_type::deriv_type deriv_type;
    typedef std::deque<double> rmap_type;
    typedef std::deque<vector_type> xmap_type;
    typedef std::vector<double> diff_type;

    limitcycle(size_type index, value_type val, double eps, double limit) :
        index_(index), value_(val), eps_(eps), limit_(limit) {}

    virtual ~limitcycle() {}

    void set_index(size_type val) { index_ = val; }
    void set_value(double val) { value_ = val; }
    void set_eps(double val) { eps_ = val; }
    void set_limit(double val) { limit_ = val; }

    size_type index() const { return index_; }
    double value() const { return value_; }
    double eps() const { return eps_; }
    double limit() const { return limit_; }

    double cycle() const { return cycle_; }
    orbit_type const& orbit() const { return orbit_; }

    state_type x(double t) const
    {
        t = std::fmod(t, cycle_);
        if (t < 0.0)
            t += cycle_;
        return orbit_.x(t);
    }

    deriv_type d(double t) const
    {
        t = std::fmod(t, cycle_);
        if (t < 0.0)
            t += cycle_;
        return orbit_.d(t);
    }


    virtual void check(diff_type const& diff, double& t)
    {
        std::cerr << "# " << diff.front() <<  '\n';
        if (diff.front() < eps())
            cycle_ = rmap_.back();
        else
            shift(rmap_.size()-1, t);
    }

    template <typename model_type>
        void start(model_type& eq, double& t, vector_type& x, double& h)
        {
            std::cerr << "# start limitcycle\n";
            orbit_.clear();
            cycle_ = 0.0;
            rmap_.clear();
            xmap_.clear();

            h = std::numeric_limits<double>::infinity();
            t = 0.0;
            x.resize(eq.size());
            eq.init(x);
        }

    template <typename model_type, typename Result>
        void accepted(model_type const& eq, Result const& r)
        {
            orbit_.insert(r);
        }

    template <typename model_type>
        bool modify(model_type const& eq, double& t, vector_type& x, double& h)
        {
            interp_type const& interp(orbit_.rbegin()->second);
            if ((interp.x0(index_) < value_)
                    && (value_ <= interp.x1(index_))) {
                rmap_.push_back(interp.find_time(index_, value_));
                xmap_.push_back(interp.x(rmap_.back()));
                if (rmap_.size() == 1) {
                    shift(0, t);
                } else {
                    diff_type diff(rmap_.size()-1);
                    vector_type e(xmap_.back().size());
                    for (size_type i(0); i < diff.size(); ++i) {
                        e.assign(xmap_.back() - xmap_[i]);
                        diff[i] = eq.norm(e, rmap_.back(), xmap_.back());
                    }
                    check(diff, t);
                }
            }
            return false;
        }

    template <typename model_type>
        bool is_active(model_type const& eq,
                double t, vector_type const& x) const
        {
            return (cycle_ == 0.0)
                && (t - (rmap_.empty() ? 0.0 : rmap_.back()) < limit_);
        }

    template <typename model_type>
        void finish(model_type const& eq, double& t, vector_type& x, double& h)
        {
            if (rmap_.empty())
                rmap_.push_back(0.0);
        }

protected:
    rmap_type rmap_;
    xmap_type xmap_;

    void shift(size_type n, double& t)
    {
        double delta(rmap_[n]);
        t -= delta;

        orbit_type tmpo;
        for (typename orbit_type::iterator itr(--orbit_.upper_bound(delta));
                itr != orbit_.end(); ++itr) {
            itr->second.set_t0(itr->second.t0() - delta);
            tmpo.insert_temporary(itr->second);
        }
        orbit_.swap(tmpo);

        for (size_type i(n); i < rmap_.size(); ++i)
            rmap_[i] -= delta;
        for (size_type i(0); i < n; ++i) {
            rmap_.pop_front();
            xmap_.pop_front();
        }
    }

private:
    size_type index_;
    value_type value_;
    double eps_;
    double limit_;

    orbit_type orbit_;
    double cycle_;
};

}

#endif
