#ifndef TEST_HHNEURON_H
#define TEST_HHNEURON_H

#include <nsim/current.h>
#include <nsim/alpha_beta.h>
#include <nsim/functions.h>

namespace hh
{

typedef nsim::scale<nsim::lin> lin;
typedef nsim::scale<nsim::exp> exp;
typedef nsim::scale<nsim::sig> sig;

struct leak_current : public nsim::current<>
{
    leak_current() : nsim::current<>(0.3, -54.4) {}
};

struct na_current : public nsim::current<3, 1>
{
    typedef nsim::alpha_beta<lin, exp> m_base;
    typedef nsim::alpha_beta<exp, sig> h_base;
    struct m_type : public m_base
    {
        m_type() : m_base(lin(-40.0, 10.0, 1.0), exp(-65.0, -18.0, 4.0)) {}
    } m;
    struct h_type : public h_base
    {
        h_type() : h_base(exp(-65.0, -20.0, 0.07), sig(-35.0, 10.0, 1.0)) {}
    } h;
    na_current() : nsim::current<3, 1>(120.0, 50.0) {}
};

struct k_current : public nsim::current<4>
{
    typedef nsim::alpha_beta<lin, exp> n_base;
    struct n_type : public n_base
    {
        n_type() : n_base(lin(-55.0, 10.0, 0.1), exp(-65.0, -80.0, 0.125)) {}
    } n;
    k_current() : nsim::current<4>(36.0, -77.0) {}
};

struct neuron
{
    enum { VM, NA_M, NA_H, K_N, DIM };

    static std::size_t size() { return DIM; }

    double i_app;
    leak_current i_leak;
    na_current i_na;
    k_current i_k;

    mutable double vm, na_m, na_h, k_n;
    mutable double ileak, ina, ik;

    neuron(double iapp = 0.0) : i_app(iapp) {}

    /** get initial value.
     *  @param x [out]
     */
    template <typename T>
        void init(T& x) const
        {
            x[VM] = -60.0;
            x[NA_M] = 0.1;
            x[NA_H] = 0.6;
            x[K_N] = 0.35;
        }

    /** set internal parameters.
     *  @param x [in]
     */
    template <typename T>
        void set(T const& x) const
        {
            vm   = x[VM];
            na_m = x[NA_M];
            na_h = x[NA_H];
            k_n  = x[K_N];

            ileak = i_leak(vm);
            ina   = i_na(vm, na_m, na_h);
            ik    = i_k(vm, k_n);
        }

    /** 
     *  @param t [in]
     *  @param x [in]
     *  @param f [out] f(t, x)
     */
    template <typename T1, typename T2>
        void deriv(double t, T1 const& x, T2& f) const
        {
            set(x);
            f(VM)   = i_app - ileak - ina - ik;
            f(NA_M) = i_na.m.dt(vm, na_m);
            f(NA_H) = i_na.h.dt(vm, na_h);
            f(K_N)  = i_k.n.dt(vm, k_n);
        }

    /** 
     *  @param t [in]
     *  @param x [in]
     *  @param j [out] Df(t, x)
     */
    template <typename V, typename M>
        void jacob(double t, V const& x, M& j) const
        {
            set(x);
            j.clear();

            j(VM, VM) = -i_leak.dv(vm)
                - i_na.dv(vm, na_m, na_h) - i_k.dv(vm, k_n);
            j(VM, NA_M) = -i_na.dm(vm, na_m, na_h);
            j(VM, NA_H) = -i_na.dh(vm, na_m, na_h);
            j(VM, K_N)  = -i_k.dm(vm, k_n);

            j(NA_M, VM)   = i_na.m.dx(vm, na_m);
            j(NA_M, NA_M) = i_na.m.dy(vm, na_m);

            j(NA_H, VM)   = i_na.h.dx(vm, na_h);
            j(NA_H, NA_H) = i_na.h.dy(vm, na_h);

            j(K_N, VM)  = i_k.n.dx(vm, k_n);
            j(K_N, K_N) = i_k.n.dy(vm, k_n);
        }

    /** 
     *  norm for this model.
     *  e could be modified.
     *  @param e [in]
     *  @param t [in] use for calculating scale
     *  @param x [in] use for calculating scale
     *  @return norm of e.
     */
    template <typename V1, typename V2>
        double norm(V1& e, double t, V2 const& x) const
        {
            scale(e);
            return norm_2(e);
        }

    /** 
     *  scale for norm.
     *  @param e [in/out]
     */
    template <typename T>
        void scale(T& x) const
        {
            x[VM] *= 0.01;
        }
};

}

#endif
