#include <iostream>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/io.hpp>

#include <nsim/dopri853.h>
#include <nsim/hiha54.h>
#include <nsim/limitcycle.h>
#include <nsim/phasemodel.h>
#include <nsim/phasecycle.h>

#include "hhneuron.h"

typedef std::size_t size_type;
typedef boost::numeric::ublas::vector<double> vector_type;

typedef nsim::limitcycle<vector_type> limitcycle;
typedef nsim::phasecycle<vector_type> phasecycle;

typedef hh::neuron model_type;
typedef nsim::phasemodel<model_type, limitcycle> phase_type;

template <typename Driver>
struct main_
{
    double iapp;
    double xtol;
    double xeps;
    double ztol;
    double zeps;
    double delta;
    double limit;

    main_(int argc, char** argv, int start = 1) :
        iapp(0.0), xtol(1e-15), xeps(1e-6), ztol(1e-2), zeps(1e-2),
        delta(-1.0), limit(100.0)
    {
        for (int i(start); i < argc; ++i) {
            std::string arg(argv[i]);
            if (arg == "-iapp" && i+1 < argc)
                iapp = boost::lexical_cast<double>(argv[++i]);
            else if (arg == "-xtol" && i+1 < argc)
                xtol = boost::lexical_cast<double>(argv[++i]);
            else if (arg == "-xeps" && i+1 < argc)
                xeps = boost::lexical_cast<double>(argv[++i]);
            else if (arg == "-ztol" && i+1 < argc)
                ztol = boost::lexical_cast<double>(argv[++i]);
            else if (arg == "-zeps" && i+1 < argc)
                zeps = boost::lexical_cast<double>(argv[++i]);
            else if (arg == "-delta" && i+1 < argc)
                delta = boost::lexical_cast<double>(argv[++i]);
            else if (arg == "-limit" && i+1 < argc)
                limit = boost::lexical_cast<double>(argv[++i]);
            else 
                throw std::runtime_error(
std::string("Usage: ") + argv[0] + " [driver_type] [options]\n"
"driver_type:\n"
"    -dopri853\n"
"    -hia54\n"
"options:\n"
"    -iapp val\n"
"    -xtol val\n"
"    -xeps val\n"
"    -ztol val\n"
"    -zeps val\n"
"    -delta val\n"
"    -limit val\n"
"Report bugs to <takekawa@users.sourceforge.jp>.");
        }
    }

    void run()
    {
        Driver driver;
        model_type model(iapp);

        limitcycle limitcycle_(model.VM, 0.0, xeps, limit);
        driver.set_tolerance(xtol);
        driver.integrate(model, limitcycle_);
        if (limitcycle_.cycle() == 0.0)
            throw std::runtime_error("no cycle");

        phase_type phase(model, limitcycle_);
        phasecycle phasecycle_(zeps, delta);
        driver.set_tolerance(ztol);
        driver.integrate(phase, phasecycle_);
        if (phasecycle_.cycle() == 0.0)
            throw std::runtime_error("no phase");

        double dt(limitcycle_.cycle()/10000);
        vector_type v(10000);
        vector_type z(10000);
        for (size_type i(0); i < 10000; ++i) {
            double t(i*dt + delta);
            limitcycle::state_type xx(limitcycle_.x(t));
            phasecycle::state_type zz(phasecycle_.x(t));
            v[i] = xx[0];
            z[i] = zz[0];
        }
        for (size_type i(0); i < 10000; ++i) {
            std::cout << (i*dt + delta)
                << '\t' << v[i]
                << '\t' << z[i]
                << '\n';
        }
    }
};

int main(int argc, char** argv)
try {
    std::string arg1(argc < 2 ? "" : argv[1]);
    if (arg1 == "-dopri853") {
        main_<nsim::dopri853::integrator<vector_type> > app(argc, argv, 2);
        app.run();
    } else if (arg1 == "-hiha54") {
        main_<nsim::hiha54::integrator<vector_type> > app(argc, argv, 2);
        app.run();
    } else {
        main_<nsim::hiha54::integrator<vector_type> > app(argc, argv);
        app.run();
    }
    return 0;
} catch (std::exception& err) {
    std::cerr << err.what() << '\n';
    return 1;
}
