#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <openssl/sha.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>

#include "utils/nt_std_t.h"
#include "utils/text.h"
#include "utils/db.h"
#include "net/nt_socket.h"
#include "net/nt_http.h"


static nt_http_header_tp nt_http_init_header(const char *url);
static void nt_http_free_header(nt_http_header_tp headerp);
static BOOL nt_http_init_header2(
		nt_http_header_tp headerp, const char *out_path);

BOOL nt_http_get(const char *url, const char *out_path,
	const char *referer)
{
	nt_socket_tp socketp;
	int fd = -1, sockfd;
	char data[1024*2];
	int nread, nwrite;
	BOOL result;

	nt_http_header_tp headerp = nt_http_init_header(url);
	if(headerp == NULL)
		return FALSE;
	if(!nt_http_init_header2(headerp, out_path))
		return FALSE;

	if(IS_SET_FLAG(headerp,SSL_FLAG))
		return FALSE;
	
	if(headerp->port == -1)
		headerp->port = 80;

	socketp = nt_socket_init(headerp->port, headerp->host);
	if(socketp == NULL){
		nt_http_free_header(headerp);
		return FALSE;
	}

	strcpy(data, "GET ");
	strcat(data, headerp->param);
	strcat(data," HTTP/1.1\r\nHost: ");
	strcat(data, headerp->host);
	strcat(data, "\r\nUser-Agent: ");
	strcat(data, USER_AGENT);
	if(referer){
		strcat(data, "\r\nReferer: ");
		strcat(data, referer);
	}
	if(headerp->last_modified){
		strcat(data, "\r\nIf-Modified-Since: ");
		strcat(data, headerp->last_modified);
	}
	strcat(data, "\r\nConnection: close\r\n\r\n");
	
	sockfd = nt_socket_connect(socketp, data, strlen(data));
	if(-1 == sockfd){
		nt_socket_free(socketp);
		nt_http_free_header(headerp);
		return FALSE;
	}

	nt_http_response_header_tp responsep = 
			nt_http_alloc_response_header();
	if(responsep == NULL){
		close(sockfd);
		nt_socket_free(socketp);
		nt_http_free_header(headerp);
		return FALSE;
	}

	nread = nt_http_parse_response_header(sockfd, responsep);
	if(nread == -1){
		close(sockfd);
		nt_socket_free(socketp);
		nt_http_free_header(headerp);
		nt_http_free_response_header(responsep);
		return FALSE;
	}

	switch(responsep->status_code){
	case 200:
		nt_http_save_response_header(out_path, responsep);

		fd = open(out_path,
			O_CREAT | O_WRONLY,
			S_IRUSR | S_IWUSR | S_IROTH);
		if(fd == -1){
			result = FALSE;
			fd = -1;
			break;
		}
		do{
			nread = read(sockfd, data, sizeof(data));
			switch(nread){
			case -1:
				break;
			case 0:
				break;
			default:
				nwrite = write(fd, data, nread);
				if(nwrite == -1){
					close(sockfd);
					return	-1;
				}
				break;
			}
		}while(nread > 0);
		result = (nread == 0);
		break;
	case 304: /* Not Modified */
		result = TRUE;
		break;
	default:
		result = FALSE;
	}/* switch */
	
	if(fd > 0)
		close(fd);
	close(sockfd);
	nt_socket_free(socketp);
	nt_http_free_header(headerp);
	nt_http_free_response_header(responsep);

	return result;
}


static BOOL nt_http_init_header2(
		nt_http_header_tp headerp, const char *out_path)
{
    char key[DB_KEY_SIZE_MAX];
    nt_db_log_rec_tp recp;
    datum d_key, d_data;
    const char *log_path;
	DBM *dbm;
	struct stat st;

	assert(headerp);
	assert(out_path);

	if(!nt_db_cpy_key(out_path, key)){
		return FALSE;
	}

	log_path	=	nt_db_get_log_path();
	dbm	=	dbm_open((char*)log_path, O_RDWR | O_CREAT, 0666);
	if(dbm == NULL){
		return FALSE;
	}

	d_key.dsize = strlen(key);
	d_key.dptr = (void*)key;

	d_data = dbm_fetch(dbm, d_key);
	if(d_data.dptr && d_data.dsize == sizeof(nt_db_log_rec_t)){
		recp = (nt_db_log_rec_tp)d_data.dptr;
		headerp->last_modified = nt_trim(recp->last_modified);	
	}else{
		headerp->last_modified = NULL;
	}
	dbm_close(dbm);

	if(0 == stat(out_path, &st)){
		headerp->fsize = st.st_size;
	}else{
		headerp->fsize = -1;
	}

	return TRUE;

}

static nt_http_header_tp nt_http_init_header(const char *url)
{
	nt_http_header_tp headerp;
	char *cptr, *host, *param;
	int port = -1;
	int len;
	BOOL ssl;

	if(0 == strncmp(url, "http://", 7)){
		url += 7;
		ssl = FALSE;
	}else if(0 == strncmp(url, "https://", 8)){
		url += 8;
		ssl = TRUE;
	}else{
		return NULL;
	}

	cptr = strchr(url, '/');
	if(NULL == cptr){
		param = malloc(2);
		if(param == NULL){
			return NULL;
		}
		param[0] = '/';
		param[1] = '\0';
		host = malloc(strlen(url)+1);
		if(host == NULL){;
			return NULL;
		}
		strcpy(host, url);
	}else{
		param = malloc(strlen(cptr)+1);
		if(param == NULL){
			return NULL;
		}
		strcpy(param, cptr);
		len = cptr - url;
		host = malloc(len+1);
		if(host == NULL){
			return NULL;
		}
		memcpy(host, url, len);
		host[len] = '\0';
	}

	cptr = strrchr(host, ':');
	if(cptr != NULL && strlen(cptr) > 0){
		port = atoi(cptr);
		port = (port == 0) ? -1 : port;
		*cptr = '\0';
	}

	headerp = malloc(sizeof(nt_http_header_t));
	if(headerp == NULL){
		free(host);
		free(param);
		return NULL;
	}

	headerp->port = port;
	headerp->host = host;
	headerp->param = param;
	if(ssl){
		SET_FLAG(headerp, SSL_FLAG);
	}else{
		CLR_FLAG(headerp, SSL_FLAG);
	}
	return headerp;
}

static void nt_http_free_header(nt_http_header_tp headerp)
{
	if(headerp->host != NULL)
		free(headerp->host);
	if(headerp->param != NULL)
		free(headerp->param);
	if(headerp->last_modified != NULL)
		free(headerp->last_modified);
	free(headerp);
}
