;;; 1998 Copyright OMURA Shinichi, all rights reserved
;; OBJECT CLASSIFIER -- relation version

;;; see spec.lsp , hisroty.lsp

#|
[state]
+ cyclic category definition

- structure:classifier - category - partition
  this means child* of partition contain different category's partitions

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
|#


(provide 'object-classifier )

(if (and (not (string= (lisp-implementation-type) "Macintosh Common Lisp"))
         (not (string= (lisp-implementation-type) "LispWorks Personal Edition")))
  (require 'my-debug (merge-pathnames "mydebug.lsp" (current-directory)))
  (require 'my-debug  "mydebug.lsp" )
)

#|
(if (not (string= (lisp-implementation-type) "Macintosh Common Lisp"))
  (require 'object-classifier-extension (merge-pathnames "objexten.lsp" (current-directory)))
)
|#
#|
(in-package 'ocp)
|#
;;; expression interface layer ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                       classifier utility               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;

(defmethod slot-exists-boundp (object name)
   (and
        (slot-exists-p object name)
        (slot-boundp object name)
        )
   )


;;; category

(defstruct (categorydef (:type list))
  name
  parent*                       ;; (null parent*) == (t)
  representation                ;; (representation obj partition) is the name of partition                                 ;; (stringp partition) means (lambda(x y) partition)
  constraint                    ;; (constraint obj partition)manes the obj contain this category
                                ;; t means true
  compass                       ;; state changer for own partition
  )

(defun t-p (x) (eq t x))

;;; object layer ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; class definition ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defclass classifier-class (standard-class)
    (
     (definition*     :initform nil :accessor definition*   )
     (category-table  :initform nil :accessor category-table  :type hash-table)
     (partition-class :initform nil :accessor partition-class)
     )
   )

#|
;;; this may be only for FreeLisp...?

(defmethod validate-superclass ((class classifier-class)
				(superclass standard-class))
  t)
|#

(defclass classifier ()
    (
     (t-partition            :initform nil :accessor t-partition)
     )
   (:metaclass classifier-class))


(defclass category ()
  (
   (name            :initarg :name            :initform nil :reader   name)
   (parent*         :initarg :parent*         :initform nil :accessor parent*) ;; category*
   (child*          :initform nil             :accessor child* )  ;; hash for name:category*
   (classifier      :initform nil             :accessor classifier)
   (representation  :initarg :representation  :initform nil :reader representation)
   (constraint      :initarg :constraint      :initform nil :reader   constraint)
   (compass         :initarg :compass         :initform nil :reader   compass)
   )
  )

;;; there always exists t in every category.
;;; it means otherwise.

;;;

(defclass partition ()
  (
   (name      :initarg :name                     :accessor   name)
   (parent*   :initarg :parent* :initform nil    :accessor parent*)  ;; partition*
   (child*    :initform nil  :accessor child*)   ;; partition*
   (category  :initarg :category                 :reader   category)
   (object*   :initform nil  :accessor object*)  ;; any object not satisfy all child*
   (constraint* :initform nil :accessor constraint*)
   )
  )

(defmethod local-constraint ((part partition))
   (list
    (constraint (category part))
    `(lambda (obj par)
       (equal (name ,part)
              (partition-name (category ,part) obj par)))
     )
   )

(defun full-constraint* (part)
   (or  (loop for parent in (parent* part)
              append
              (append (constraint* parent) (local-constraint part))
          )
       (local-constraint part)
       )
   )


(defmethod fullize-constraint ((instance partition))
   (let (constraint*)
         (progn
           (if (setf constraint* (loop for parent in (parent* instance) append
                                   (append (constraint* parent) (local-constraint instance))
                                   ))
;              (push constraint* (constraint* instance))
              (setf (constraint* instance) (local-constraint instance))
              )
           (debug '(constraint*) t "make:: ~S : ~S~%"  (name instance) (constraint* instance))
           )
      )
   )

#|
;;; class property

 (equal (category (child* part))   ;; category is canonial extended.
        (child* (category part)))

 full constraint of category c : i write this as fc(c), then

  fc(t) = t
  fc(c) = fc(b) & (constraint c)  whre (c (b) ...)

 full constraint of partition p : i write this as fp(p), then

  fp(t) = t
  fp(p) = fp(p1) & ... & fp(pn) & (constraint (category p)) &
          (equal (name p) (partition-name (categoyr p) obj)

  this & is functional and

|#


;;; interface ...................................................................

(defun make-user-class-definition (slots)
   (if slots
      (let ((newclassname (gensym)))
         (eval `(defclass ,newclassname (partition) ,slots))
         newclassname
         )
            'partition
      )
   )

(defmacro defclassifier (cname slots &rest categorydef* )
  `(let (cl (category-table (make-hash-table)) category*)
      (setf category* (make-category* ',categorydef* category-table))
      (setf cl (define-classifier
                 ,cname
                 (make-user-class-definition ,slots)
                 category-table
                 category*
                 ))
      (loop for cat in category* do (setf (classifier cat) cl))
      (complete-category-linkage category* category-table)
      )
   )


(defun set-me-in-parent* (parent* me)
  (loop for parent in parent* do
        (pushnew me (child* parent))
        )
  )

(defun complete-category-linkage (category* category-table)
  (loop for category in category* do
        (debug '(cyclic) t "~S~%" (parent* category))
        (setf (parent* category)
              (loop for parent in (parent* category) collect
                    (gethash parent category-table))
              )
        (set-me-in-parent* (parent* category) category)     ;; syncronize parent*
        )
  )

(defmacro define-classifier (cname user-partition-class category-table category*)
   `(let (c )
       (debug '(cyclic) t "user-partition-class = ~S~%" ,user-partition-class)

       (setf c (defclass ,cname (classifier) ()
                  (:metaclass classifier-class)))
       (setf (slot-value c 'partition-class) ,user-partition-class)
       (setf (slot-value c 'definition*) ,category*)
       (setf (slot-value c 'category-table) ,category-table)
       c
       )
   )


(defun put-t-category (category-table)
   (make-category
     t
     nil
     'true
     'true
     nil
     category-table)
   )

(defmethod get-t-category ((cl classifier))
   (gethash t (category-table (class-of cl)) ))



(defun make-category* (categorydef* category-table)
   (cons
     (put-t-category category-table)
     (loop for categorydef in categorydef* collect
       (make-category
         (categorydef-name categorydef)
         (let ((parent* (categorydef-parent* categorydef)))
            ;; I recognize () as (t), and t must be only one
            (if (and (null parent*) (not (eq (categorydef-name categorydef) t)))
              '(t)
              parent* ))
         (let ((pf (categorydef-representation  categorydef))) ;; I manipulate string form
            (if (stringp pf) `(lambda (x y) ,pf) pf))

         (let ((ct (categorydef-constraint categorydef))) ;; I manipulate t for const.
            (if (t-p ct) 'true ct))
        (categorydef-compass categorydef)
         category-table)
       )
     )
   )

(defun make-category (name parent* representation constraint compass category-table)
   (let (me)
      (debug '(lambda) t "representation = ~S~%" representation)
      (setf me (eval `(make-instance      'category
                        :name           ',name
                        :parent*        ',parent*
                        :representation ',representation
                        :constraint     ',constraint
                        :compass        ',compass
                        )))
      (setf (gethash name category-table) me) ;; syncronize category-table
    )
  )

;;; -------------------------------------------------------------------


(defun make-t-partition (cf)
   (make-instance (partition-class (class-of cf))
     :name t
     :category (get-t-category cf)
     )
   )

(defmethod initialize-instance ((cl classifier) &rest arg)
   :after (setf (t-partition cl) (make-t-partition cl))
   )

(defun make-classifier (classifier)
      (make-instance classifier))

;;; type definition -------------------------------------------------

(defun category-list-p (L)
  (loop for e in L always (typep e 'category))
  )

(deftype category-list () '(satisfies category-list-p))

(defun partition-list-p (L)
  (loop for e in L always (typep e 'partition))
  )

(deftype partition-list () '(satisfies partition-list-p))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                           put-object                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; cargo -- avehicle for objects

(defclass cargo ()
    (
     (object   :initarg :object :reader   object)
     (resume   :initform nil    :accessor resume)
     )
   )


(defun put-object* (cf object*)
   (check-type cf classifier)
   (loop for object in object* append
     (put-object cf object))
  )

(defmethod  put-object (cf object &optional history (part nil))
;;; put the object to  t partition
   (put-object
     (t-partition cf)
     (make-instance 'cargo :object object)
     (or history nil)
     )
   )


#|
(defmethod put-object ((container* list) object &optional (part nil))
  ;; I place the object in first eligible category in category*
  (loop for container in container* do
        (put-object container object part)
        )
  )
|#

(defmethod partition-name ((ct category) object part)
    (apply (representation ct) (list object part)))


(defmethod satisfy-p ((ct category) object &optional (part nil))
   (if part
      (apply (constraint ct) (list object part))
      t                                            ;; (null part) means t-partition
      )
   )

(defmethod navigation ((part partition) object)
   "I must return part"
   (if (compass (category part))
      (apply (compass (category part)) (list part object))
      )
   part
   )

(defun add-constraint (child part)
   (check-type child partition)
   (check-type part partition)
   (setf (constraint* child)
         (make-set
           (append (local-constraint child)
             (constraint* part)
             (constraint* child)))
         )
   )

(defun link-sub* (part sub* obj)
   (check-type part partition)
   (check-type sub* partition-list)
   (loop for sub in sub* do
     (pushnew sub (child* part))
     (pushnew part (parent* sub))
     (add-constraint sub part)
     (navigation sub obj)
     (debug '(constraint*) t "link-sub* ~S : ~S~%" (name sub)(constraint* sub))
     (debug '(constraint*) t "link-sub* parent ~S : c-parent ~S~%"
       (parent* sub)(parent* (category sub)))
     )
   sub*
   )

(defun get-category* (par*)
   (check-type par* partition-list)
   (loop for par in par* collect
     (category par))
   )

(defun add-history (par hist)
   (check-type par partition)
   (append hist (list par))
   )

(defun last-parent (hist)
   (car (last hist))
   )


(defun satisfy-constraint*-p (part obj*)
   (loop for obj in obj* always
     (full-satisfy-constraint-p (full-constraint* part) obj part)
     )
   )

(defun search-resume (par history cg)
   (check-type par partition)
   (check-type cg cargo)
   ;; => list of par whose parent* contains (category par)
   (let (subhist)
      (loop for hpar in history
        when (and
                  (member (category par) (parent* (category hpar)))
                  (full-satisfy-p hpar (object cg))
                  (satisfy-constraint*-p par (get-all-object* hpar)))
        append (link-sub* par (list hpar) (object cg))
        )
      )
   )

(defmethod put-object-join ((par partition) (cg cargo))
   (loop for history in (resume cg) append
     (search-resume par history cg)
     )
   )

(defun put-object-sub-partition (par cg history u-sub*)
   (check-type par partition)
   (check-type cg cargo)
  (loop for sub in (set-difference (child* par) u-sub*)
        when (and (satisfy-p (category sub) (object cg) sub)
                  (equal (name sub)
                    (partition-name (category sub) (object cg) sub))
                  (full-satisfy-p sub (object cg))
                  (put-object sub cg (add-history par history)))
        append (list sub))
  )

(defun already-spawned  (par ct cg)
   (check-type par partition)
   (check-type ct category)
   (check-type cg cargo)
   (loop for child in (child* par)
     thereis
     (if (equal ct (category child) ) (full-satisfy-p child (object cg)))
     )
   )

(defun put-object-sub-category (par cg history parent)
   (check-type par partition)
   (check-type cg  cargo)
   (let (sub*)
      (loop for sct in (child* (category par))
          when (and (satisfy-p sct (object cg) par)
                    (not (already-spawned par sct cg))
                    (setf sub*
                          (put-object
                           sct
                           cg
                           (add-history par history)
                           par)))
          append (link-sub* par sub* (object cg))
      )
     )
   )

(defmethod put-object-sub ((par partition) (cg cargo)  &optional history parent)
   (let (u-sub*)
      (append
        (setf u-sub* (put-object-join par cg))
        (put-object-sub-partition par cg history u-sub*)
        (put-object-sub-category par cg history parent)
        )
      )
   )

(defun make-dummy-partition (name category object parent)
   (let ((part (make-instance (partition-class (classifier category))
                 :name name
                 :category category
                 :parent* (and parent (list parent)))
           ))
      (navigation part object)
      part
      )
   )

(defmethod entry-object ((par partition) (cg cargo) history)
   (pushnew (object cg) (object* par))
   (push (add-history par history) (resume cg))
   (debug '(resume) t "resume: ~S   history: ~S~%" (resume cg) history)
   )

(defmethod put-object ((par partition) (cg cargo) &optional history (parent nil))
   (debug '(put-object) t "~A/" (name par))
   (if (satisfy-p (category par) (object cg))
      (let ((sub* (put-object-sub par cg history parent)))
         (if (null sub*)
            (if (full-satisfy-p par (object cg))
               (progn
                 (debug '(put-object) t "{put object ~S in ~A}~%" cg (name par))
                 (entry-object par cg history)
                 (debug '(put-object) t "object* = ~S~%" (object* par))
                 )
               )
            )
         (list par)
         )
      )
   )


(defmethod put-object ((ct category) (cg cargo)  &optional history (parent nil))
   ;; This method is called only when no partition for object exists in ct
   (let ((dummy (make-dummy-partition nil ct (object cg) parent)))
      (if (satisfy-p ct (object cg) dummy)
         (let (name)
            (setf name (partition-name ct (object cg) dummy) )
            (setf (name dummy) name)
            (fullize-constraint dummy)
            (put-object dummy cg history parent)
            )
         )
      )
   )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                           about category                           ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod get-category ((cl classifier))
   (gethash t (category-table (class-of cl)) ))

(defmethod get-sub-category* ((ct category)) (child* ct))

;;;

(defmethod show-category ((cl classifier) &optional indent)
   (terpri t)
   (show-category (get-category cl) (or indent 0))
   )

(defmethod show-category ((ct category) &optional indent)
   (format t "~A/[~A]~%" (nspace indent) (name ct))
   (loop for c in (child* ct) do
     (show-category c (1+ indent))
     )
   )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                           get-object*                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod get-object* ((part partition)) (object* part))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                           get-all-object*                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod get-t-partition ((cl classifier)) (t-partition cl))

(defmethod get-all-object* ((cl classifier))
   (get-all-object* (get-t-partition cl)))

(defun make-set (L)
     (let (r) (loop for x in L unless (member x r :test #'equal)
                do (push x r) finally (return r))))

(defmethod get-all-object* ((pat partition))
   ;;I collect the all objects that the tree whose t-partition is pat contains
   (make-set
     (append (object* pat)
       (loop for c in (child* pat) append
         (get-all-object* c)
         )
       )
     )
   )

;; test
#|
(mapcar #'(lambda (x) (progn (display x)(terpri t)))
        (get-all-object* tree1))
|#
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                show                                ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod show ((cl classifier) method &optional detail indent)
  (let ((t-partition (get-t-partition cl)))
    (and t-partition (show t-partition method detail (or indent 0) ))
    )
  )

(defmethod show ((re partition) method &optional detail indent )
   (if detail
      (format t "~A[~A]~S/~%" (nspace indent) (name re) re)
      (format t "~A[~A]/~%" (nspace indent) (name re))
      )
   (loop for obj in (object* re) do
     (format t "~A" (nspace (1+ indent)))
     (apply method (list obj) )
     (terpri t)
     )
   (loop for c in (child* re) do
     (show c method detail (1+ indent) )
     )
   )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                            show-partition                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod show-partition ((cl classifier) &optional indent)
   (terpri t)
   (show-partition (get-t-partition cl) (or indent 0))
   )

(defmethod show-partition ((re partition) &optional indent)
   (let ((sp (or indent 0)))
      (format t "~A/~A~%" (nspace sp) (name re))
      (loop for sub in (child* re) do
        (show-partition sub (1+ sp))
        )
      )
   )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                          system utility                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun nspace (n) (check-type n integer)
       (make-sequence 'string n :initial-element #\  ))

(defun contain-p (part object)
   (check-type part partition)
   (or (member object (object* part))
      (loop for child in (child* part) thereis
            (contain-p child object))
      )
  )

(defmethod satisfy-p ((part partition) object &optional (samepart nil))
  (and (satisfy-p (category part) object part)
           (equal (name part)
                      (partition-name (category part) object part))
           (loop for parent in (parent* part) always
                    (satisfy-p parent object parent)))
   )

(defun full-satisfy-constraint-p (constraint* obj part)
   (loop for fn in constraint* always
     (progn
       (debug '(constraint*) t "full-satsify-constraint-p ~S ~S => ~S~%"
         fn obj (apply fn  (list obj part)))
       (apply fn  (list obj part))
       )
     )
   )

(defun full-satisfy-p (part obj)
   ;; constraint*́Apartitionfull constraint.
   (check-type part partition)
   (debug '(constraint*) t "full-satisfy//constraint::~S~%" (constraint* part))
   (full-satisfy-constraint-p (constraint* part) obj part)
   )

(defmethod show-partition-info ((cl classifier) &optional indent)
   (show-partition-info (get-t-partition cl) (or indent 0)))

(defmethod show-partition-info ((part partition) &optional indent)
   (format t "~%~A-----------~%" (nspace indent))
   (format t "~Aid         : ~S~%" (nspace indent) part)
   (format t "~Aname       : ~A~%" (nspace indent) (name part))
   (format t "~Aparent*    : ~S~%" (nspace indent) (parent* part))
   (format t "~Acategory   : ~S~%" (nspace indent) (category part))
   (format t "~Aobject*    : ~S~%" (nspace indent) (object* part))
   (format t "~Aconstraint*: ~S~%" (nspace indent) (constraint* part))
   (loop for child in (child* part) do
     (show-partition-info child (1+ indent))
     )
   )


(defmethod show-category-info ((cl classifier) &optional indent)
   (show-category-info (get-t-category cl) (or indent 0)))

(defmethod show-category-info ((cat category) &optional indent)
   (format t "~%~A--------------~%" (nspace indent))
   (format t "~Aid            : ~S~%" (nspace indent) cat)
   (format t "~Aname          : ~A~%" (nspace indent) (name cat))
   (format t "~Aparent*       : ~S~%" (nspace indent) (parent* cat))
   (format t "~Aclassifier    : ~S~%" (nspace indent) (classifier cat))
   (format t "~Arepresentation: ~S~%" (nspace indent) (representation cat))
   (format t "~Aconstraint    : ~S~%" (nspace indent) (constraint cat))
   (loop for child in (child* cat) do
     (show-category-info child (1+ indent))
     )
   )

t
