/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: board_properties.c,v 1.1.1.1 2003/05/10 16:34:25 fuyu Exp $
 */

#include "ochusha_ui.h"

#include "bulletin_board_ui.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "threadlist_filter.h"

#include <gtk/gtk.h>

#include <pthread.h>
#include <stdlib.h>


typedef struct _DialogStates
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;
  BulletinBoardGUIInfo *info;

  ThreadlistFilterSetting *setting;
  GtkToggleButton *use_as_default_button;
  GtkToggleButton *enable_filter_button;

  ThreadlistFilter *last_filter;

  gboolean changed;
  gboolean use_default;
} DialogStates;


static GtkWidget *build_board_info_frame(OchushaApplication *application,
					 OchushaBulletinBoard *board,
					 DialogStates *states);
static GtkWidget *build_filter_setting_frame(OchushaApplication *application,
					     OchushaBulletinBoard *board,
					     DialogStates *states);

static void setting_changed_cb(ThreadlistFilterSetting *setting,
			       ThreadlistFilter *filter, DialogStates *states);
static void restore_default_cb(ThreadlistFilterSetting *setting,
			       DialogStates *states);
static void apply_current_states(GtkWidget *dialog, DialogStates *states);
static void board_properties_response_cb(GtkWidget *dialog, gint response_id,
					 DialogStates *states);
static void button_toggled_cb(GtkToggleButton *button, DialogStates *states);

static pthread_mutex_t board_properties_lock;


#define BOARD_PROPERTIES_LOCK					\
  if (pthread_mutex_lock(&board_properties_lock) != 0)		\
    {								\
      fprintf(stderr, "Couldn't lock a mutex.\n");		\
      abort();							\
    }

#define BOARD_PROPERTIES_UNLOCK					\
  if (pthread_mutex_unlock(&board_properties_lock) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }


void
initialize_board_properties(OchushaApplication *application)
{
  if (pthread_mutex_init(&board_properties_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
}


void
open_board_properties(OchushaApplication *application,
		      OchushaBulletinBoard *board)
{
  GtkWidget *dialog;
  GtkWidget *frame;
  GtkWidget *tmp_widget;
  DialogStates *states;
  BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;

  BOARD_PROPERTIES_LOCK;

  if (info == NULL || info->properties_dialog != NULL)
    {
      BOARD_PROPERTIES_UNLOCK;
      return;
    }

  states = (DialogStates *)calloc(1, sizeof(DialogStates));
  if (states == NULL)
    {
      BOARD_PROPERTIES_UNLOCK;
      return;
    }

  states->application = application;
  states->board = board;
  states->info = info;

  if (info->filter.rule != NULL)
    {
      states->last_filter = &info->filter;
      states->use_default = FALSE;
    }
  else
    {
      states->last_filter = &application->user_default_filter;
      states->use_default = TRUE;
    }

  dialog = gtk_dialog_new_with_buttons(_("Bulletin Board's Properties"),
				       application->top_level,
				       GTK_DIALOG_DESTROY_WITH_PARENT,

				       GTK_STOCK_APPLY,
				       GTK_RESPONSE_APPLY,

				       GTK_STOCK_OK,
				       GTK_RESPONSE_OK,

				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_CANCEL,

				       NULL);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
  g_signal_connect(G_OBJECT(dialog), "response",
		   G_CALLBACK(board_properties_response_cb),
		   states);

  info->properties_dialog = dialog;

  BOARD_PROPERTIES_UNLOCK;

  frame = build_board_info_frame(application, board, states);
  gtk_widget_show(frame);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), frame);

  frame = build_filter_setting_frame(application, board, states);
  gtk_widget_show(frame);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), frame);

  tmp_widget
    = gtk_check_button_new_with_label(_("Apply filter to this board."));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		   tmp_widget, FALSE, FALSE, 0);
  states->enable_filter_button = GTK_TOGGLE_BUTTON(tmp_widget);
  g_signal_connect(G_OBJECT(tmp_widget), "toggled",
		   G_CALLBACK(button_toggled_cb), states);
  gtk_toggle_button_set_active(states->enable_filter_button,
			       info->enable_filter);

  states->changed = FALSE;
  gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
				    GTK_RESPONSE_APPLY, FALSE);
  gtk_widget_show_all(dialog);
}


static void
button_toggled_cb(GtkToggleButton *button, DialogStates *states)
{
  if (!states->changed)
    gtk_dialog_set_response_sensitive(
				GTK_DIALOG(states->info->properties_dialog),
				GTK_RESPONSE_APPLY, TRUE);
  states->changed = TRUE;
}


static void
setting_changed_cb(ThreadlistFilterSetting *setting, ThreadlistFilter *filter,
		   DialogStates *states)
{
  if (!states->changed)
    gtk_dialog_set_response_sensitive(
				GTK_DIALOG(states->info->properties_dialog),
				GTK_RESPONSE_APPLY, TRUE);
  states->changed = TRUE;
  states->use_default = FALSE;
}


static void
restore_default_cb(ThreadlistFilterSetting *setting, DialogStates *states)
{
  threadlist_filter_setting_set_filter(setting,
				&states->application->user_default_filter);
  states->changed = TRUE;
  states->use_default = TRUE;

  states->info->filter.ignore_threshold
    = states->application->user_default_filter.ignore_threshold;
  gtk_toggle_button_set_active(states->use_as_default_button, TRUE);
  gtk_dialog_set_response_sensitive(
				GTK_DIALOG(states->info->properties_dialog),
				GTK_RESPONSE_APPLY, TRUE);
}


static void
apply_current_states(GtkWidget *dialog, DialogStates *states)
{
  const ThreadlistFilter *filter;

  if (!states->changed)
    return;

  filter = threadlist_filter_setting_get_filter(states->setting);
  states->info->enable_filter
    = gtk_toggle_button_get_active(states->enable_filter_button);

  if (gtk_toggle_button_get_active(states->use_as_default_button))
    {
      free(states->application->user_default_filter.rule);
      states->application->user_default_filter.rule = g_strdup(filter->rule);
      states->application->user_default_filter.ignore_threshold
	= filter->ignore_threshold;

      if (states->info->filter.rule != NULL)
	{
	  gchar *tmp = states->info->filter.rule;
	  states->info->filter.rule = NULL;
	  free(tmp);
	}

      states->info->filter.ignore_threshold = filter->ignore_threshold;
      threadlist_filter_setting_set_restore_default_button_sensitive(
						states->setting, FALSE);
      setup_default_threadlist_filter(states->application);
      redraw_current_threadlist_all(states->application);
    }
  else
    {
      if (states->info->filter.rule != NULL)
	{
	  gchar *tmp = states->info->filter.rule;
	  states->info->filter.rule = NULL;
	  free(tmp);
	}

      if (!states->use_default)
	{
	  states->info->filter.rule = g_strdup(filter->rule);
	  states->info->filter.ignore_threshold = filter->ignore_threshold;
	}

      redraw_threadlist_view(states->application, states->board);
      fix_filter_button_state(states->application);
    }

  states->changed = FALSE;
  gtk_dialog_set_response_sensitive(
				GTK_DIALOG(states->info->properties_dialog),
				GTK_RESPONSE_APPLY, FALSE);
}


static void
board_properties_response_cb(GtkWidget *dialog, gint response_id,
			     DialogStates *states)
{
  gboolean close_dialog = TRUE;

  switch (response_id)
    {
    case GTK_RESPONSE_APPLY:
      apply_current_states(dialog, states);
      close_dialog = FALSE;
      break;

    case GTK_RESPONSE_OK:
      apply_current_states(dialog, states);
      break;

    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
      break;
    }

  if (close_dialog)
    {
      BOARD_PROPERTIES_LOCK
	{
	  states->info->properties_dialog = NULL;
	}
      BOARD_PROPERTIES_UNLOCK;
      gtk_widget_destroy(dialog);
    }
}


static GtkWidget *
build_board_info_frame(OchushaApplication *application,
		       OchushaBulletinBoard *board, DialogStates *states)
{
  GtkWidget *table;
  GtkWidget *tmp_widget;
  GtkWidget *board_frame = gtk_frame_new(NULL);

  gtk_frame_set_shadow_type(GTK_FRAME(board_frame), GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(board_frame);

  table = gtk_table_new(2, 5, FALSE);
  gtk_widget_show(table);
  gtk_container_add(GTK_CONTAINER(board_frame), table);

  tmp_widget = gtk_label_new(_("Board's Name"));
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 0, 1);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(tmp_widget), board->name);
  gtk_editable_set_editable(GTK_EDITABLE(tmp_widget), FALSE);
  GTK_WIDGET_UNSET_FLAGS(tmp_widget, GTK_CAN_FOCUS);
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 1, 5, 0, 1);

  tmp_widget = gtk_label_new(_("Board's URL"));
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 1, 2);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(tmp_widget), board->base_url);
  gtk_editable_set_editable(GTK_EDITABLE(tmp_widget), FALSE);
  GTK_WIDGET_UNSET_FLAGS(tmp_widget, GTK_CAN_FOCUS);
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 1, 5, 1, 2);

  return board_frame;
}


static GtkWidget *
build_filter_setting_frame(OchushaApplication *application,
			   OchushaBulletinBoard *board, DialogStates *states)
{
  GtkWidget *filter_frame = gtk_frame_new(_("Filter Setting"));

  GtkWidget *vbox;
  GtkWidget *tmp_widget;

  gtk_frame_set_shadow_type(GTK_FRAME(filter_frame), GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(filter_frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(filter_frame), vbox);

  tmp_widget = gtk_check_button_new_with_label(_("Use this setting as the default setting."));
  states->use_as_default_button = GTK_TOGGLE_BUTTON(tmp_widget);
  g_signal_connect(G_OBJECT(tmp_widget), "toggled",
		   G_CALLBACK(button_toggled_cb), states);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = threadlist_filter_setting_new_with_initial_setting(states->last_filter);
  states->setting = THREADLIST_FILTER_SETTING(tmp_widget);
  g_signal_connect(G_OBJECT(tmp_widget), "changed",
		   G_CALLBACK(setting_changed_cb), states);
  g_signal_connect(G_OBJECT(tmp_widget), "restore_default",
		   G_CALLBACK(restore_default_cb), states);

  if (states->info->filter.rule == NULL)
    {
      threadlist_filter_setting_set_restore_default_button_sensitive(THREADLIST_FILTER_SETTING(tmp_widget), FALSE);
    }

  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);

  return filter_frame;
}
