/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: network.c,v 1.1.1.1 2003/05/10 16:34:28 fuyu Exp $
 */

#include "ochusha.h"
#include "gziputils.h"
#include "worker.h"

#include <ghttp.h>
#include <glib.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/mman.h>

#include <time.h>

#include <unistd.h>

#ifndef MAP_NOCORE
# define MAP_NOCORE	0
#endif

#define DISABLE_PERSISTENT_CONNECTION		0
#define ENABLE_GZIPPED_DIFFERENCIAL_READ	0

#define POLLING_INTERVAL_SEC	0
#define POLLING_INTERVAL_NANO	20000000	/* 20ms */

struct timespec polling_interval =
{
  POLLING_INTERVAL_SEC,
  POLLING_INTERVAL_NANO
};


#define DEBUG_NETWORK		0
#define DEBUG_NETWORK_MOST	0
#define DEBUG_MMAP		0


static int read_cache_to_buffer(OchushaConfig *conf, const char *url,
				OchushaAsyncBuffer *buffer);
static void write_buffer_to_cache(OchushaConfig *conf, const char *url,
				  OchushaAsyncBuffer *buffer);
static void setup_common_request_headers(OchushaConfig *conf,
					 ghttp_request *request);
static gboolean http_read_from_url(OchushaConfig *conf,
				   OchushaAsyncBuffer *buffer,
				   const char *url, const char *last_modified,
				   OchushaNetworkCallbacks *callbacks,
				   gpointer callback_data);
static int get_content_length(ghttp_request *request);
static void try_update(WorkerThread *employee, gpointer args);
static void refresh_after_read(WorkerThread *employee, gpointer args);
static void force_read(WorkerThread *employee, gpointer args);
static OchushaAsyncBuffer *employ_networking_thread(OchushaConfig *conf,
					     const char *url,
					     const char *last_modified,
					     JobFunc *job_function,
					     OchushaNetworkCallbacks *callbacks,
					     gpointer callback_data);
static void munmap_when_finished(OchushaAsyncBuffer *buffer,
				 gpointer user_data);
static OchushaAsyncBuffer *get_mmapped_buffer(int fd);


static const char http_hdr_Cookie[] = "Cookie";
static const char http_hdr_Set_Cookie2[] = "Set-Cookie2";


static int
read_cache_to_buffer(OchushaConfig *conf, const char *url,
		     OchushaAsyncBuffer *buffer)
{
  int len;
  int fd = ochusha_config_cache_open_file(conf, url, O_RDONLY);

  if (fd == -1)
    return 0;	/* åʤξ */

  /* å夢ꡣ*/
  len = lseek(fd, 0, SEEK_END);
  lseek(fd, 0, SEEK_SET);
  if (ochusha_async_buffer_resize(buffer, len,
				  "network.c: read_cache_to_buffer"))
    {
      len = read(fd, (void *)buffer->buffer, len);
      if (!ochusha_async_buffer_update_length(buffer, len,
				"network.c: read_cache_to_buffer"))
	{
#if DEBUG_ASYNC_BUFFER_MOST
	  fprintf(stderr, "read_cache_to_buffer: buffer may have been terminated.\n");
#endif
	  /* ξϤɤ褦ʤ */
	}
    }
  else
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "read_cache_to_buffer: buffer may have been terminated.\n");
#endif
      len = 0;	/* Out of memory? */
    }
  close(fd);

  return len;
}


static void
write_buffer_to_cache(OchushaConfig *conf, const char *url,
		      OchushaAsyncBuffer *buffer)
{
  int fd = ochusha_config_cache_open_file(conf, url,
					  O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      ssize_t len = write(fd, (void *)buffer->buffer, buffer->length);
      close(fd);
      if (len != buffer->length)
	{
	  /* TODO: 񤭹ߤ˼Ԥåä٤*/
#if DEBUG_NETWORK
	  fprintf(stderr, "Couldn't update cache for %s: %s (%d) \n",
		  url, strerror(errno), errno);
#endif
	}
    }
#if DEBUG_NETWORK_MOST
  else
    fprintf(stderr, "Couldn't open cache file to write.\n");
#endif
}


static void
setup_common_request_headers(OchushaConfig *conf, ghttp_request *request)
{
  ghttp_set_header(request, http_hdr_User_Agent, OCHUSHA_USER_AGENT);
}


typedef enum
{
  UNKNOWN_ENCODING = -1,
  STANDARD_ENCODING,
  GZIP_ENCODING,
} ContentEncodingType;


static ContentEncodingType
get_content_encoding_type(ghttp_request *request)
{
  const char *encoding = ghttp_get_header(request, http_hdr_Content_Encoding);

  if (encoding != NULL)
    {
      if (strcmp(encoding, "gzip") == 0)
	return GZIP_ENCODING;
      else
	return UNKNOWN_ENCODING;
    }

  return STANDARD_ENCODING;
}


static gboolean
http_read_from_url(OchushaConfig *conf, OchushaAsyncBuffer *buffer,
		   const char *url, const char *last_modified,
		   OchushaNetworkCallbacks *callbacks, gpointer callback_data)
{
#if DEBUG_NETWORK
  int sock_fd;
#endif
  int status_code = 0;
  ghttp_request *request = NULL;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;
  GzipBuffer *gzip_buffer = NULL;
  ghttp_status state;

  request = ghttp_request_new();
  ghttp_set_uri(request, (char *)url);
  ghttp_set_type(request, ghttp_type_head);
#if DISABLE_PERSISTENT_CONNECTION
  ghttp_set_header(request, http_hdr_Connection, "close");
#endif
  ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
  if (last_modified != NULL)
    {
      int fd = ochusha_config_cache_open_file(conf, url, O_RDONLY);
      if (fd != -1)
	{
	  ghttp_set_header(request, http_hdr_If_Modified_Since, last_modified);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "If-Modified-Since: %s\n", last_modified);
#endif
	  close(fd);
	}
      else
	last_modified = NULL;
    }

  setup_common_request_headers(conf, request);

  if (callbacks != NULL && callbacks->access_started != NULL)
    (*callbacks->access_started)(callback_data);

  ghttp_set_sync(request, ghttp_async);
  ghttp_prepare(request);

  /*
   * MEMO: λޤǥ롼פΤǡsync⡼ɤǥƤ
   *       褤ߤ˽᤯ȿ뤿async⡼ɤ
   *       롼פƤ롣
   */
  while (TRUE)
    {
      state = ghttp_process(request);
#if DEBUG_NETWORK
      sock_fd = ghttp_get_socket(request);
      fprintf(stderr, "1: sock_fd=%d state=%d\n", sock_fd, state);
#endif
      if (!ochusha_async_buffer_check_active(buffer,
					     "network.c: http_read_from_url"))
	state = ghttp_error;
      if (state == ghttp_error || state == ghttp_done)
	break;
      nanosleep(&polling_interval, NULL);
    }

#if DEBUG_NETWORK
  {
    char **hdrs;
    int num_hdrs;
    int i;
    int result = ghttp_get_header_names(request, &hdrs, &num_hdrs);
    if (result == 0)
      {
	for (i = 0; i < num_hdrs; i++)
	  {
	    fprintf(stderr, "%s: %s\n",
		    hdrs[i], ghttp_get_header(request, hdrs[i]));
	    free(hdrs[i]);
	  }
	free(hdrs);
      }
  }
#endif

  status_code = ghttp_status_code(request);

  if (state == ghttp_error || status_code != 200)
    {
      OchushaNetworkStatus *buf_status
	= (OchushaNetworkStatus *)buffer->user_data;
      buf_status->status_code = status_code;

#if DEBUG_NETWORK
      if (status_code != 0)
	fprintf(stderr, "status_code=%d (%s)\n",
		status_code, ghttp_reason_phrase(request));
      else
	fprintf(stderr, "Unknown error.\n");
#endif
      if (status_code != 304
	  && callbacks != NULL && callbacks->access_failed != NULL)
	(*callbacks->access_failed)(callback_data);
      return FALSE;
    }

#if !DISABLE_PERSISTENT_CONNECTION
# if DEBUG_NETWORK_MOST
  fprintf(stderr, "status_code=%d (%s)\n",
	  status_code, ghttp_reason_phrase(request));
# endif
  ghttp_clean(request);
#else
  ghttp_request_destroy(request);
  request = ghttp_request_new();
#endif
  ghttp_set_uri(request, (char *)url);
  ghttp_set_type(request, ghttp_type_get);
  ghttp_set_header(request, http_hdr_Connection, "close");
  ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
  setup_common_request_headers(conf, request);

  ghttp_set_sync(request, ghttp_async);
  ghttp_prepare(request);

  while (TRUE)
    {
      int body_len;
      char *body;
      state = ghttp_process(request);
#if DEBUG_NETWORK
      sock_fd = ghttp_get_socket(request);
      fprintf(stderr, "2: sock_fd=%d state=%d\n", sock_fd, state);
#endif
#if DEBUG_NETWORK_MOST
      {
	/* progress barȤѰդʤ餳Ȥɤ*/
	ghttp_current_status current_status = ghttp_get_status(request);
	fprintf(stderr, "state=%d, %d(read)/%d(total)\n",
		state, current_status.bytes_read, current_status.bytes_total);
      }
#endif

      if (state == ghttp_not_done)
	ghttp_flush_response_buffer(request);

      if (state == ghttp_error)
	{
	  fprintf(stderr, "http_read_from_url: I/O error\n");
	  break;
	}

      body = ghttp_get_body(request);
      body_len = ghttp_get_body_len(request);
      
      if (body_len > 0)
	{
	  if (callbacks != NULL && callbacks->access_progressed != NULL)
	    (*callbacks->access_progressed)(callback_data);

	  if (content_encoding == UNKNOWN_ENCODING)
	    {
	      /* ˤ1٤ʤ */
	      content_encoding = get_content_encoding_type(request);

	      switch (content_encoding)
		{
		case GZIP_ENCODING:
		  gzip_buffer = gzip_buffer_new(buffer);
		  if (gzip_buffer == NULL)
		    {
#if DEBUG_NETWORK_MOST
		      fprintf(stderr, "Out of memory\n");
#endif
		      goto error_exit;
		    }
		  break;

		case UNKNOWN_ENCODING:
		  goto error_exit;

		case STANDARD_ENCODING:
		  break;
		}
	    }

	  if (content_encoding == GZIP_ENCODING)
	    {
	      GzipBufferStatus result
		= gzip_buffer_append_data(gzip_buffer, body, body_len);
	      if (result == GZIP_BUFFER_ERROR)
		{
#if DEBUG_NETWORK_MOST
		  fprintf(stderr, "Inflation failed.\n");
#endif
		  gzip_buffer_free(gzip_buffer);
		  goto error_exit;
		}
	      if (result == GZIP_BUFFER_INFLATION_DONE)
		{
		  gzip_buffer_free(gzip_buffer);
		  gzip_buffer = NULL;
#if DEBUG_NETWORK_MOST
		  if (state != ghttp_done)
		    fprintf(stderr, "Requested file contains garbage?\n");
#endif
		  break;
		}
	    }
	  else if (!ochusha_async_buffer_append_data(buffer, body, body_len,
					"network.c: http_read_from_url"))
	    {	/* content_encoding == STANDARD_ENCODING */
#if DEBUG_ASYNC_BUFFER_MOST
	      fprintf(stderr, "http_read_from_url: buffer may have been terminated.\n");
#endif
	      goto error_exit;
	    }
	}

      if (state == ghttp_done)
	break;

      nanosleep(&polling_interval, NULL);
    }

  if (gzip_buffer != NULL)
    {
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "What's happen?  Connection was closed but inflation hasn't been completed.\n");
#endif
      gzip_buffer_free(gzip_buffer);
    }

  status_code = ghttp_status_code(request);

  if (status_code == 200)
    {
      OchushaNetworkStatus *buf_status
	= (OchushaNetworkStatus *)buffer->user_data;
      buf_status->status_code = status_code;
      buf_status->last_modified
	= g_strdup(ghttp_get_header(request, http_hdr_Last_Modified));

      ghttp_request_destroy(request);

      if (callbacks != NULL && callbacks->access_completed != NULL)
	(*callbacks->access_completed)(callback_data);
      return TRUE;
    }

#if DEBUG_NETWORK
  if (status_code == 0)
    fprintf(stderr, "IP unreachable?\n");
  else
    {
      fprintf(stderr, "HTTP Response: %d: %s\n",
	      status_code, ghttp_reason_phrase(request));
      fprintf(stderr, "Couldn't read url: %s\n", url);
    }
#endif

 error_exit:

  if (callbacks != NULL && callbacks->access_failed != NULL)
    (*callbacks->access_failed)(callback_data);
  ghttp_request_destroy(request);
  return FALSE;
}


typedef struct _NetworkJobArgs
{
  OchushaConfig *conf;
  OchushaAsyncBuffer *buffer;
  char *url;
  char *last_modified;
  OchushaNetworkCallbacks *callbacks;
  gpointer callback_data;
} NetworkJobArgs;


#define CACHE_COMPARE_SIZE	512	/* Ĥѹǽˤ٤ */
#define HEADER_BUFFER_LENGTH	256


static int
get_content_length(ghttp_request *request)
{
  int length;
  const char *content_len = ghttp_get_header(request, http_hdr_Content_Length);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "Content-Type: %s\n",
	  ghttp_get_header(request, http_hdr_Content_Type));
#endif
  if (content_len == NULL)
    return -1;		/* Content-Lengthإåʤ */

  if (sscanf(content_len, "%d", &length) < 1)
    return -1;

  return length;
}


static void
try_update(WorkerThread *employee, gpointer args)
{
  NetworkJobArgs *job_args = (NetworkJobArgs *)args;
  OchushaConfig *conf = job_args->conf;
  OchushaAsyncBuffer *buffer = job_args->buffer;
  char *url = job_args->url;
  OchushaNetworkStatus *buf_status = (OchushaNetworkStatus *)buffer->user_data;
  int len;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;

  if (!ochusha_async_buffer_active_ref(buffer, "network.c: try_update"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      buf_status->state = OCHUSHA_ACCESS_FAILED;

      if (job_args->callbacks != NULL
	  && job_args->callbacks->access_failed != NULL)
	(*job_args->callbacks->access_failed)(job_args->callback_data);

      ochusha_async_buffer_fix(buffer, "network.c: try_update");
      g_object_unref(G_OBJECT(buffer));
      free(url);
      free(args);
      return;
    }

  len = read_cache_to_buffer(conf, url, buffer);

  if (len > 0)
    {
      /* MEMO:
       * ޤHEADꥯȤĹåĹӡ
       * åä麹ʬԡ
       */
      /* ̤ghttpν */
      int status_code;
      ghttp_request *request;
      ghttp_status state = ghttp_not_done;

      buf_status->state = OCHUSHA_CACHE_HIT;
      request = ghttp_request_new();
      if (request == NULL)
	{
	  /* Out of memory? */
	  /* ۤäƥåϤɤΤ */
	  buf_status->state = OCHUSHA_CACHE_IS_OK;

	  if (job_args->callbacks != NULL
	      && job_args->callbacks->access_failed != NULL)
	    (*job_args->callbacks->access_failed)(job_args->callback_data);

	  goto read_done;
	}

      ghttp_set_uri(request, url);

      /* ޤHEADꥯȤͻҤõ */
      ghttp_set_type(request, ghttp_type_head);

#if DISABLE_PERSISTENT_CONNECTION
      ghttp_set_header(request, http_hdr_Connection, "close");
#endif
#if ENABLE_GZIPPED_DIFFERENCIAL_READ
      ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
#endif

      if (job_args->last_modified != NULL)
	{
	  ghttp_set_header(request, http_hdr_If_Modified_Since,
			   job_args->last_modified);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "If-Modified-Since: %s\n", job_args->last_modified);
#endif
	}
      setup_common_request_headers(conf, request);

      if (job_args->callbacks != NULL
	  && job_args->callbacks->access_started != NULL)
	(*job_args->callbacks->access_started)(job_args->callback_data);

      ghttp_set_sync(request, ghttp_async);
      ghttp_prepare(request);

      while (TRUE)
	{
	  state = ghttp_process(request);
	  if (!ochusha_async_buffer_check_active(buffer,
						 "network.c: try_update"))
	    state = ghttp_error;
	  if (state == ghttp_error || state == ghttp_done)
	    break;
	  nanosleep(&polling_interval, NULL);
	}

      status_code = ghttp_status_code(request);

      if (state != ghttp_error
	  && status_code == 304 && job_args->last_modified != NULL)
	{
	  buf_status->status_code = status_code;
	  buf_status->state = OCHUSHA_CACHE_IS_OK;

#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "304\n");
#endif
	  if (job_args->callbacks != NULL
	      && job_args->callbacks->access_completed != NULL)
	    (*job_args->callbacks->access_completed)(job_args->callback_data);

	  ghttp_request_destroy(request);
	  goto read_done;
	}

#if DEBUG_NETWORK_MOST
      /* progress barȤѰդʤ餳Ȥɤ*/
      {
	ghttp_current_status current_status = ghttp_get_status(request);
	fprintf(stderr, "state=%d, proc=%d, %d(read)/%d(total)\n",
		state, current_status.proc, current_status.bytes_read, current_status.bytes_total);
	fprintf(stderr, "status_code=%d (%s)\n",
		status_code, ghttp_reason_phrase(request));
      }
#endif

      if (state != ghttp_error && status_code == 200)
	{
	  char header[HEADER_BUFFER_LENGTH];
	  char cache_compare_buffer[CACHE_COMPARE_SIZE];
	  int offset = len - CACHE_COMPARE_SIZE;
	  GzipBuffer *gzip_buffer = NULL;
	  int new_len = get_content_length(request);

	  if (new_len < len)
	    {
#if DEBUG_NETWORK
	      fprintf(stderr, "cached data is larger than the content that in the server, a-bone?\n");
#endif
	      goto cache_is_dirty;
	    }

	  if (offset < 0)
	    {
	      /* 
	       * MEMO: Υɤˤꡢåʬե
	       *       ˥åconflictˤʤäƤޤ
	       *       If-Modified-SinceդǥåƤΤ
	       *       ѹΤʤˤĤƤϤϲǤ롣
	       */
#if DEBUG_NETWORK
	      fprintf(stderr, "cached data is too small to check its equality.\n");
#endif
	      goto cache_is_dirty;	/* ʬʤˤĤκƼ */
	    }

	  /* ޤǤHEADꥯȤˤͻҸ */

	  /* ʲϺʬμ¹ʬ */
	  if (snprintf(header, HEADER_BUFFER_LENGTH, "bytes=%d-", offset)
	      >= HEADER_BUFFER_LENGTH)
	    {
#if DEBUG_NETWORK
	      fprintf(stderr, "Really?  HEADER_BUFFER_LENGTH is too short.\n");
#endif
	      goto cache_is_dirty;	/* ľʤκƼ*/
	    }

#if !DISABLE_PERSISTENT_CONNECTION
	  ghttp_clean(request);
#else
	  ghttp_request_destroy(request);
	  request = ghttp_request_new();
#endif
	  ghttp_set_uri(request, url);
	  ghttp_set_type(request, ghttp_type_get);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "setting Range: %s\n", header);
#endif
	  ghttp_set_header(request, http_hdr_Connection, "close");
#if ENABLE_GZIPPED_DIFFERENCIAL_READ
	  ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
#endif
	  ghttp_set_header(request, http_hdr_Range, header);
	  setup_common_request_headers(conf, request);
	  ghttp_set_sync(request, ghttp_async);
	  ghttp_prepare(request);

	  /*
	   * Content-Encoding狼äCACHE_COMPARE_SIZEʬ
	   * Хåեɤ߹ߡå夬ɤǧ
	   *
	   * ägzipʾбݤʤΤǡ
	   * ޤOchushaAsyncBufferͭǡCACHE_COMPARE_SIZEʬ
	   * (å夵Ƥǡ)Хåե˥ԡ
	   * OchushaAsyncBufferͭǡĹCACHE_COMPARE_SIZEʬ餷Ƥ
	   * ޤGzipBuffer٤餫GzipBufferAPI
	   * ɲä٤
	   * OchushaAsyncBufferconsumer¦read-onlyǡХåեΰ輫Τ
	   * ͭʤΤɤޤƤʤޤå夬ʾ
	   * Ʊǡ񤭤å夬Ȥ
	   * äˤϡ̵Τǰʥǡ
	   * ޤޤƤƤⵤˤʤǤ
	   */
	  /* λǤϡbuffer->length == len */
	  memcpy(cache_compare_buffer, (char *)buffer->buffer + offset,
		 CACHE_COMPARE_SIZE);

	  if (!ochusha_async_buffer_update_length(buffer, offset,
					"network.c: http_read_from_url"))
	    {
#if DEBUG_ASYNC_BUFFER_MOST
	      fprintf(stderr, "http_read_from_url: buffer may have been terminated.\n");
#endif
	      goto error_exit;
	    }

	  while (TRUE)
	    {
	      /* ̿λ⤷ϥå夬äƤΤȽޤǥ롼*/
	      int body_len;
	      char *body;
	      gboolean before_cache_check = TRUE;

	      /*
	       * åͭǧޤǤν
	       * buffer->length >= lenˤʤ뤫̿ޤpolling
	       */
	      state = ghttp_process(request);
#if DEBUG_NETWORK_MOST
      /* progress barȤѰդʤ餳Ȥɤ*/
	      {
		ghttp_current_status current_status = ghttp_get_status(request);
		fprintf(stderr, "state=%d, proc=%d, %d(read)/%d(total)\n",
			state, current_status.proc, current_status.bytes_read, current_status.bytes_total);
	      }
#endif
	      if (state == ghttp_not_done)
		ghttp_flush_response_buffer(request);
	      else if (state == ghttp_error)
		{
#if DEBUG_NETWORK
		  fprintf(stderr, "I/O error: %s\n", ghttp_get_error(request));
		  status_code = ghttp_status_code(request);
		  fprintf(stderr, "HTTP Response: %d (%s)\n",
			  status_code, ghttp_reason_phrase(request));
#endif
		  goto cache_is_dirty;	/* åäƤ */
		}

	      body = ghttp_get_body(request);
	      body_len = ghttp_get_body_len(request);

#if DEBUG_NETWORK_MOST
	      fprintf(stderr, "body_len=%d\n", body_len);
#endif

	      if (body_len > 0)
		{
		  if (job_args->callbacks != NULL
		      && job_args->callbacks->access_progressed != NULL)
		    (*job_args->callbacks->access_progressed)(job_args->callback_data);

		  if (content_encoding == UNKNOWN_ENCODING)
		    {
		      /* ˤ1٤ʤ */
		      content_encoding = get_content_encoding_type(request);

		      switch (content_encoding)
			{
			case GZIP_ENCODING:
			  gzip_buffer = gzip_buffer_new(buffer);
			  if (gzip_buffer == NULL)
			    {
#if DEBUG_NETWORK
			      fprintf(stderr, "Out of memory\n");
#endif
			      goto error_exit;
			    }
			  break;

			case UNKNOWN_ENCODING:
			  goto cache_is_dirty;	/* ĩ */

			case STANDARD_ENCODING:
			  break;
			}
		    }

		  if (content_encoding == GZIP_ENCODING)
		    {
		      GzipBufferStatus result
			= gzip_buffer_append_data(gzip_buffer, body, body_len);
#if DEBUG_NETWORK
		      fprintf(stderr, "append_gzip_data: body_len=%d\n", body_len);
#endif
		      if (result == GZIP_BUFFER_ERROR)
			{
#if DEBUG_NETWORK
			  fprintf(stderr, "Inflation failed.\n");
#endif
#if DEBUG_ASYNC_BUFFER_MOST
			  fprintf(stderr, "buffer may have been terminated.\n");
#endif
			  gzip_buffer_free(gzip_buffer);
			  goto error_exit;
			}
		      if (result == GZIP_BUFFER_INFLATION_DONE)
			{
			  gzip_buffer_free(gzip_buffer);
			  gzip_buffer = NULL;
#if DEBUG_NETWORK
			  if (state != ghttp_done)
			    fprintf(stderr, "Requested file contains garbage?\n");
#endif
			  break;
			}
		    }
		  else if (!ochusha_async_buffer_append_data(buffer, body,
						body_len,
						"network.c: try_update"))
		    {	/* content_encoding == STANDARD_ENCODING */
#if DEBUG_ASYNC_BUFFER_MOST
		      fprintf(stderr, "buffer may have been terminated.\n");
#endif
		      goto error_exit;
		    }

		  /* buffer->length >= lenˤʤä饭å */
		  if (before_cache_check && buffer->length >= len)
		    {
		      if (memcmp(cache_compare_buffer,
				 (char *)buffer->buffer + offset,
				 CACHE_COMPARE_SIZE) != 0)
			{
#if DEBUG_NETWORK
			  fprintf(stderr, "Cached data isn't up to date.\n");
#endif
			  goto cache_is_dirty;	/* åäƤ */
			}
		      else
			before_cache_check = FALSE;	/* åϿ */
		    }
		}

	      if (state == ghttp_done)
		break;

	      nanosleep(&polling_interval, NULL);
	    }

#if DEBUG_NETWORK
	  status_code = ghttp_status_code(request);
	  if (status_code != 206)
	    {
	      fprintf(stderr, "Content-Length: %d\n", new_len);
	      fprintf(stderr, "Range: %s\n", header);
	      fprintf(stderr, "HTTP Response: %d (%s)\n",
		      status_code, ghttp_reason_phrase(request));
	    }
#endif
	  /* ̿Ǹޤ˽λˤ */
	  if (buffer->length < len)
	    {
#if DEBUG_NETWORK
	      fprintf(stderr, "Cached data is longer than current data.");
#endif
	      goto cache_is_dirty;	/* ͽǡʤäΤ
					 * åäƤ
					 */
	    }

	  buf_status->status_code = status_code;
	  buf_status->last_modified
	    = g_strdup(ghttp_get_header(request, http_hdr_Last_Modified));
	  buf_status->state = OCHUSHA_CACHE_IS_OK;

	  if (job_args->callbacks != NULL
	      && job_args->callbacks->access_completed != NULL)
	    (*job_args->callbacks->access_completed)(job_args->callback_data);

	  ghttp_request_destroy(request);
	  write_buffer_to_cache(conf, url, buffer);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "Update using cache succeeded.\n");
#endif
	  goto read_done;
	}
      else
	{
	  if (job_args->callbacks != NULL
	      && job_args->callbacks->access_failed != NULL)
	    (*job_args->callbacks->access_failed)(job_args->callback_data);

#if DEBUG_NETWORK
	  status_code = ghttp_status_code(request);
	  if (status_code != 0)
	    fprintf(stderr, "HTTP Response: %d (%s)\n",
		    status_code, ghttp_reason_phrase(request));
	  else
	    fprintf(stderr, "Cannot access network?\n");
#endif
	  buf_status->state = OCHUSHA_CACHE_IS_OK;
	  if (!ochusha_async_buffer_update_length(buffer, len,
						  "network.c: try_update"))
	    {
#if DEBUG_ASYNC_BUFFER_MOST
	      fprintf(stderr, "buffer has been terminated.\n");
#endif
	      /* ξϤɤ褦ʤ櫡ȤΥåɤ
	       * ɤ⤦ΥХåե˿ʤΤ֡
	       */
	    }
	  ghttp_request_destroy(request);
	  goto read_done;
	}

    cache_is_dirty:
#if DEBUG_NETWORK
      status_code = ghttp_status_code(request);
      fprintf(stderr, "HTTP Response: %d (%s)\n",
	      status_code, ghttp_reason_phrase(request));
#endif

      buf_status->state = OCHUSHA_CACHE_IS_DIRTY;
      ghttp_request_destroy(request);
      if (!ochusha_async_buffer_update_length(buffer, 0,
					      "network.c: try_update"))
	{
#if DEBUG_ASYNC_BUFFER_MOST
	  fprintf(stderr, "buffer has been terminated.\n");
#endif
	  goto error_exit;
	}
#if DEBUG_NETWORK
      fprintf(stderr, "Cache is dirty.\n");
#endif
    }
  else
    {
      /* å夬ʤä */
#if DEBUG_NETWORK
      fprintf(stderr, "There's no cache.\n");
#endif
      buf_status->state = OCHUSHA_DIRECT_ACCESS;
    }

  /* å夬ʤääƤ */
  if (http_read_from_url(conf, buffer, url, NULL,
			 job_args->callbacks, job_args->callback_data))
    {
      /* ̿ */
      if (buf_status->state == OCHUSHA_DIRECT_ACCESS)
	buf_status->state = OCHUSHA_ACCESS_OK;
      write_buffer_to_cache(conf, url, buffer);
      goto read_done;
    }

 error_exit:
  buf_status->state = OCHUSHA_ACCESS_FAILED;

  if (job_args->callbacks != NULL
      && job_args->callbacks->access_failed != NULL)
    (*job_args->callbacks->access_failed)(job_args->callback_data);

 read_done:
  ochusha_async_buffer_fix(buffer, "network.c: try_update");
  ochusha_async_buffer_active_unref(buffer, "network.c: try_update");
  g_object_unref(G_OBJECT(buffer));
  free(url);
  if (job_args->last_modified != NULL)
    free(job_args->last_modified);
  free(job_args);
}      


static void
refresh_after_read(WorkerThread *employee, gpointer args)
{
  NetworkJobArgs *job_args = (NetworkJobArgs *)args;
  OchushaConfig *conf = job_args->conf;
  OchushaAsyncBuffer *buffer = job_args->buffer;
  char *url = job_args->url;
  OchushaNetworkStatus *status = (OchushaNetworkStatus *)buffer->user_data;

  if (!ochusha_async_buffer_active_ref(buffer,
				       "network.c: refresh_after_read"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      status->state = OCHUSHA_ACCESS_FAILED;
      goto error_exit;
    }

  status->state = OCHUSHA_DIRECT_ACCESS;

  if (http_read_from_url(conf, buffer, url, job_args->last_modified,
			 job_args->callbacks, job_args->callback_data))
    {
      /* ̿ */
      status->state = OCHUSHA_ACCESS_OK;
      /* ХåեƤ򥭥å˽񤭹ࡣ */
      write_buffer_to_cache(conf, url, buffer);
    }
  else
    {
      /* ̿Ԥå夬ʾ */
      /* å夬ФХåե˽񤭹 */
      if (read_cache_to_buffer(conf, url, buffer))
	{
	  status->state = OCHUSHA_CACHE_HIT;
	  if (status->status_code == 304)
	    {
	      if (job_args->callbacks != NULL
		  && job_args->callbacks->access_completed != NULL)
		(*job_args->callbacks->access_completed)(job_args->callback_data);
	    }
#if DEBUG_NETWORK_MOST
	  else
	    fprintf(stderr, "Couldn't read via network, so cached data is used instead.\n");
#endif
	}
      else
	{
	  status->state = OCHUSHA_ACCESS_FAILED;
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "Couldn't open cache file for %s\n", url);
#endif
	  if (status->status_code == 304)
	    {
	      fprintf(stderr, "Cache file is removed during networking.\n");
	      if (job_args->callbacks != NULL
		  && job_args->callbacks->access_failed != NULL)
		(*job_args->callbacks->access_failed)(job_args->callback_data);
	      status->status_code = 0;
	    }
	}
    }

  ochusha_async_buffer_active_unref(buffer, "network.c: refresh_after_read");

 error_exit:
  ochusha_async_buffer_fix(buffer, "network.c: refresh_after_read");
  g_object_unref(G_OBJECT(buffer));
  free(url);
  free(args);
}


static void
force_read(WorkerThread *employee, gpointer args)
{
  NetworkJobArgs *job_args = (NetworkJobArgs *)args;
  OchushaAsyncBuffer *buffer = job_args->buffer;
  OchushaNetworkStatus *status = (OchushaNetworkStatus *)buffer->user_data;

  if (!ochusha_async_buffer_active_ref(buffer, "network.c: force_read"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      status->state = OCHUSHA_ACCESS_FAILED;
      goto error_exit;
    }

#if DEBUG_THREAD_MOST
  fprintf(stderr, "force_read thread is invoked.\n");
#endif

  status->state = OCHUSHA_DIRECT_ACCESS;

  if (http_read_from_url(job_args->conf, buffer, job_args->url, NULL,
			 job_args->callbacks, job_args->callback_data))
    {
      /* ̿ */
      status->state = OCHUSHA_ACCESS_OK;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s succeeded.\n", job_args->url);
#endif
    }
  else
    {
      /* ̿ */
      status->state = OCHUSHA_ACCESS_FAILED;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s failed.\n", job_args->url);
#endif
    }

  ochusha_async_buffer_active_unref(buffer, "network.c: force_read");

 error_exit:
  ochusha_async_buffer_fix(buffer, "network.c: force_read");
  g_object_unref(G_OBJECT(buffer));
  free(job_args->url);
  if (job_args->last_modified != NULL)
    free(job_args->last_modified);
  free(args);
}


static void
network_buffer_free(OchushaAsyncBuffer *buffer, gpointer user_data)
{
  OchushaNetworkStatus *status = (OchushaNetworkStatus *)user_data;
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "network_buffer_free: buffer=0x%x\n", (int)buffer);
#endif
  if (buffer->buffer != NULL)
    free((void *)buffer->buffer);
  if (status != NULL)
    {
      if (status->last_modified != NULL)
	free(status->last_modified);
      free(status);
    }
}


static OchushaAsyncBuffer *
employ_networking_thread(OchushaConfig *conf, const char *url,
			 const char *last_modified,
			 JobFunc *job_function,
			 OchushaNetworkCallbacks *callbacks,
			 gpointer callback_data)
{
  NetworkJobArgs *args;
  WorkerJob *job;
  OchushaNetworkStatus *status
    = (OchushaNetworkStatus *)calloc(1, sizeof(OchushaNetworkStatus));
  OchushaAsyncBuffer *buffer = ochusha_async_buffer_new(NULL, 0, status,
							network_buffer_free);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "employ_networking_thread: url=\"%s\" buffer=0x%x\n",
	  url, (int)buffer);
  fflush(stderr);
#endif

  if (status == NULL || buffer == NULL)
    {
      if (status != NULL)
	free(status);
      else if (buffer != NULL)
	g_object_unref(G_OBJECT(buffer));

      if (callbacks != NULL && callbacks->access_failed != NULL)
	(*callbacks->access_failed)(callback_data);
      return NULL;
    }

  status->state = OCHUSHA_UNKNOWN_STATE;

  args = (NetworkJobArgs *)malloc(sizeof(NetworkJobArgs));
  if (args == NULL)
    {
      g_object_unref(G_OBJECT(buffer));
      if (callbacks != NULL && callbacks->access_failed != NULL)
	(*callbacks->access_failed)(callback_data);
      return NULL;
    }

  args->conf = conf;
  args->buffer = buffer;
  args->url = g_strdup(url);
  args->last_modified
    = (last_modified != NULL) ? g_strdup(last_modified) : NULL;
  args->callbacks = callbacks;
  args->callback_data = callback_data;

  job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
  if (job == NULL)
    {
      g_object_unref(G_OBJECT(buffer));
      free(args->url);
      free(args);
      if (callbacks != NULL && callbacks->access_failed != NULL)
	(*callbacks->access_failed)(callback_data);
      return NULL;
    }

  g_object_ref(G_OBJECT(buffer));	/* åɤΤ */

  job->canceled = FALSE;
  job->job = job_function;
  job->args = args;

#if DEBUG_THREAD_MOST
  fprintf(stderr, "Committing job.\n");
  fflush(stderr);
#endif
  commit_job(job);

  return buffer;
}


static void
munmap_when_finished(OchushaAsyncBuffer *buffer, gpointer user_data)
{
  munmap((void *)buffer->buffer, buffer->length);
  if (user_data != NULL)
    free(user_data);
#if DEBUG_MMAP
  fprintf(stderr, "mmapped buffer at 0x%x is unmapped.\n", (int)buffer);
#endif
}


static OchushaAsyncBuffer *
get_mmapped_buffer(int fd)
{
  char *buf;
  off_t len;
  OchushaAsyncBuffer *buffer;
  OchushaNetworkStatus *status;

  if (fd < 0)
    return NULL;

  status = (OchushaNetworkStatus *)calloc(1, sizeof(OchushaNetworkStatus));
  if (status == NULL)
    return NULL;

  status->state = OCHUSHA_CACHE_HIT;

  len = lseek(fd, 0, SEEK_END);
  buf = mmap(NULL, len, PROT_READ, MAP_NOCORE | MAP_PRIVATE, fd, 0);
  if (buf == MAP_FAILED)
    {
      fprintf(stderr, "mmap failed due to: %s (%d)\n", strerror(errno), errno);
      free(status);
      return NULL;
    }
  close(fd);

#if DEBUG_MMAP
  fprintf(stderr, "allocating sync_buffer using mmaped buffer at 0x%x\n",
	  (int)buf);
#endif

  buffer = ochusha_async_buffer_new(buf, len, status, munmap_when_finished);
  if (!ochusha_async_buffer_fix(buffer, "network.c: get_mmapped_buffer"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      /* ȤϤɤ褦ʤ */
    }
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "get_mmapped_buffer: buffer=0x%x\n", (int)buffer);
#endif
  return buffer;
}


/*
 * ochusha_read_from_url
 *
 * Ϳ줿URLǡɤ߹ࡣ
 * offline⡼ɤǥå夬¸ߤʤNULL֤
 *
 * modeOCHUSHA_CACHE_IGNOREǤ硢ɬͥåȥǡɤ߹ߡ
 * ̤򥭥å夷ʤ
 *
 * ʳξˤɤ߹߻˲餫ηǥå夬ФȤ
 * ˥ͥåȥɹǤˤϥǡ򥭥å¸롣
 *
 * OCHUSHA_CACHE_AS_ISξ硢¸ߤåǡ򤽤Τޤ޻Ȥ
 * OCHUSHA_CACHE_TRY_UPDATEξ硢å夵Ƥǡappend only
 * ǹΤǤȸʤå夵ƤǡȺǿǤκʬ
 * Τߤͥåȥɤ߹ࡣ
 * OCHUSHA_CACHE_TRY_REFRESHξ硢å夬ͥåȥ
 * ˥ǡɤ߹ߡͥåȥɹԲǽǤäˤΤߡ
 * å夵ƤǡȤ
 *
 * ͥåȥ˥硢̿˼ԤǤĹ0ΥХåե
 * ֤Τա
 */
OchushaAsyncBuffer *
ochusha_read_from_url(OchushaConfig *conf, const char *url,
		      const char *last_modified,
		      OchushaCacheMode mode,
		      OchushaNetworkCallbacks *callbacks,
		      gpointer callback_data)
{
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "ochusha_read_from_url: %s\n", url);
  fflush(stderr);
#endif
  if (mode == OCHUSHA_CACHE_IGNORE && conf->offline)
    {
      if (callbacks != NULL && callbacks->access_failed != NULL)
	(*callbacks->access_failed)(callback_data);
      fprintf(stderr, "Couldn't read (offline mode): %s\n", url);
      return NULL;
    }

  if (conf->offline
      || mode == OCHUSHA_CACHE_AS_IS || mode == OCHUSHA_CACHE_ONLY)
    {
      /* å夬ä餽򤽤Τޤ޻Ȥ */
      int fd = ochusha_config_cache_open_file(conf, url, O_RDONLY);
      if (fd >= 0)
	{
	  if (callbacks != NULL && callbacks->access_completed != NULL)
	    (*callbacks->access_completed)(callback_data);
	  return get_mmapped_buffer(fd);
	}
    }

  if (conf->offline || mode == OCHUSHA_CACHE_ONLY)
    {
      /* åʤξ */
      if (callbacks != NULL && callbacks->access_failed != NULL)
	(*callbacks->access_failed)(callback_data);
      return NULL;
    }

  if (mode == OCHUSHA_CACHE_TRY_UPDATE)
    return employ_networking_thread(conf, url, last_modified, try_update,
				    callbacks, callback_data);

  if (mode == OCHUSHA_CACHE_TRY_REFRESH || mode == OCHUSHA_CACHE_AS_IS)
    return employ_networking_thread(conf, url, last_modified,
				    refresh_after_read,
				    callbacks, callback_data);

  return employ_networking_thread(conf, url, NULL, force_read,
				  callbacks, callback_data);
}
