/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: boardlist_view.c,v 1.5 2003/06/06 19:37:23 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "ochusha_ui.h"
#include "boardlist_view.h"
#include "marshal.h"

#include <glib.h>
#include <gtk/gtk.h>

#if DEBUG_WIDGET || DEBUG_WIDGET_MOST
#include <stdio.h>
#endif
#include <stdlib.h>


static void boardlist_view_class_init(BoardlistViewClass *klass);
static void boardlist_view_init(BoardlistView *view);

static void boardlist_view_finalize(GObject *object);
static void boardlist_view_destroy(GtkObject *object);

static void tree_view_row_expanded_cb(GtkTreeView *tree_view,
				      GtkTreeIter *iter, GtkTreePath *path,
				      BoardlistView *view);
static void tree_view_row_collapsed_cb(GtkTreeView *tree_view,
				       GtkTreeIter *iter, GtkTreePath *path,
				       BoardlistView *view);
static gboolean tree_view_button_press_event_cb(GtkWidget *widget,
						GdkEventButton *event,
						BoardlistView *view);
static gboolean tree_view_button_release_event_cb(GtkWidget *widget,
						  GdkEventButton *event,
						  BoardlistView *view);
static void append_board(gpointer data, gpointer user_data);
static void append_category(gpointer data, gpointer user_data);


GType
boardlist_view_get_type(void)
{
  static GType blv_type = 0;

  if (blv_type == 0)
    {
      static const GTypeInfo blv_info =
	{
	  sizeof(BoardlistViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)boardlist_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(BoardlistView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)boardlist_view_init,
	};

      blv_type = g_type_register_static(GTK_TYPE_SCROLLED_WINDOW,
					"BoardlistView", &blv_info, 0);
    }

  return blv_type;
}


typedef struct _TreeNodeInfo
{
  GtkTreeIter iter;
} TreeNodeInfo;


enum {
  OPEN_VIEW_SIGNAL,

  SELECT_CATEGORY_SIGNAL,
  CATEGORY_COLLAPSED_SIGNAL,
  CATEGORY_EXPANDED_SIGNAL,
  LAST_SIGNAL
};


static GtkScrolledWindowClass *parent_class = NULL;
static gint boardlist_view_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };
static GQuark node_info_id;


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static TreeNodeInfo *
ensure_node_info(GObject *object)
{
  TreeNodeInfo *info = g_object_get_qdata(object, node_info_id);
  if (info == NULL)
    {
      info = G_NEW0(TreeNodeInfo, 1);
      g_object_set_qdata_full(object, node_info_id,
			      info, (GDestroyNotify)TRACE_FREE);
    }

  return info;
}


static void
boardlist_view_class_init(BoardlistViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;

  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = boardlist_view_finalize;

  /* GtkObject signals */
  object_class->destroy = boardlist_view_destroy;

  boardlist_view_signals[SELECT_CATEGORY_SIGNAL] =
    g_signal_new("select_category",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, select_category),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  boardlist_view_signals[OPEN_VIEW_SIGNAL] =
    g_signal_new("open_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, open_view),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_BOOLEAN,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 G_TYPE_BOOLEAN);
  boardlist_view_signals[CATEGORY_EXPANDED_SIGNAL] =
    g_signal_new("category_expanded",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, category_expanded),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BOARD_CATEGORY);
  boardlist_view_signals[CATEGORY_COLLAPSED_SIGNAL] =
    g_signal_new("category_collapsed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, category_collapsed),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BOARD_CATEGORY);

  klass->open_view = NULL;
  klass->select_category = NULL;
  klass->category_expanded = NULL;
  klass->category_collapsed = NULL;

  node_info_id = g_quark_from_static_string("BoardlistView::TreeNodeInfo");
}


enum
{
  BOARDLIST_CATEGORY_ROW,
  BOARDLIST_BOARD_ROW
};


enum
{
  BOARDLIST_LABEL = 0,
  BOARDLIST_ROW_TYPE,
  BOARDLIST_DATA,
  BOARDLIST_N_COLUMNS
};


static void
boardlist_view_init(BoardlistView *view)
{
  GtkWidget *tree_view;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  GTK_WIDGET_SET_FLAGS(view, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);

  gtk_scrolled_window_set_policy(&view->container,
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  tree_view = gtk_tree_view_new();
  view->view = GTK_TREE_VIEW(tree_view);

  gtk_scrolled_window_set_hadjustment(&view->container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));
  gtk_scrolled_window_set_vadjustment(&view->container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));

  /* MEMO: Τ褦GtkAdjustmentͿƤʤassertion˰óݤ:
   * Gtk-CRITICAL **: file gtkrange.c: line 438 (gtk_range_get_adjustment):
   * assertion 'GTK_IS_RANGE(range)' failed
   * 2Ф롣
   */
  gtk_container_add(GTK_CONTAINER(&view->container), tree_view);

  gtk_widget_show(tree_view);

  view->selection = gtk_tree_view_get_selection(view->view);
  gtk_tree_selection_set_mode(view->selection, GTK_SELECTION_MULTIPLE);

  g_signal_connect(G_OBJECT(view->view), "row-expanded",
		   G_CALLBACK(tree_view_row_expanded_cb), view);
  g_signal_connect(G_OBJECT(view->view), "row-collapsed",
		   G_CALLBACK(tree_view_row_collapsed_cb), view);
  g_signal_connect(G_OBJECT(view->view), "button_press_event",
		   G_CALLBACK(tree_view_button_press_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "button_release_event",
		   G_CALLBACK(tree_view_button_release_event_cb), view);
  view->last_event_x = 0.0;
  view->last_event_y = 0.0;
  view->last_event_button = 0;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Board List"), renderer,
						    "text", BOARDLIST_LABEL,
						    NULL);
  gtk_tree_view_append_column(view->view, column);
}


static void
boardlist_view_finalize(GObject *object)
{
  BoardlistView *view = (BoardlistView *)object;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "boardlist_view_finalize\n");
#endif
  g_object_unref(G_OBJECT(view->model));

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
boardlist_view_destroy(GtkObject *object)
{
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "boardlist_view_destroy\n");
#endif

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


void
boardlist_view_expand_category(BoardlistView *view,
			       OchushaBoardCategory *category)
{
  gboolean expanded;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(category));
  GtkTreePath *path = gtk_tree_model_get_path(GTK_TREE_MODEL(view->model),
					      &info->iter);
  expanded = gtk_tree_view_expand_row(view->view, path, FALSE);

  gtk_tree_path_free(path);

  if (expanded)
    g_signal_emit(G_OBJECT(view),
		  boardlist_view_signals[CATEGORY_EXPANDED_SIGNAL],
		  0,
		  category);
}


void
boardlist_view_collapse_category(BoardlistView *view,
				 OchushaBoardCategory *category)
{
  gboolean collapsed;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(category));
  GtkTreePath *path = gtk_tree_model_get_path(GTK_TREE_MODEL(view->model),
					      &info->iter);
  collapsed = gtk_tree_view_collapse_row(view->view, path);

  gtk_tree_path_free(path);

  if (collapsed)
    g_signal_emit(G_OBJECT(view),
		  boardlist_view_signals[CATEGORY_COLLAPSED_SIGNAL],
		  0,
		  category);
}


GtkWidget *
boardlist_view_new(void)
{
  return GTK_WIDGET(g_object_new(BOARDLIST_VIEW_TYPE, NULL));
}


static gboolean
tree_view_button_press_event_cb(GtkWidget *widget, GdkEventButton *event,
				BoardlistView *view)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gint row_type;
  gpointer data = NULL;

  view->last_event_x = event->x;
  view->last_event_y = event->y;
  view->last_event_button = event->button;

  if (!gtk_tree_view_get_path_at_pos(view->view, event->x, event->y, &path,
				     NULL, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter,
		     BOARDLIST_ROW_TYPE, &row_type,
		     BOARDLIST_DATA, &data,
		     -1);

  if (data == NULL)
    goto done;

  switch (row_type)
    {
    case BOARDLIST_CATEGORY_ROW:
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[SELECT_CATEGORY_SIGNAL],
		    0,
		    data);
      break;

    case BOARDLIST_BOARD_ROW:
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[OPEN_VIEW_SIGNAL],
		    0,
		    data,
		    event->button == 2);
      break;

    default:
      fprintf(stderr, "GtkTreeModel broken.\n");
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  return FALSE;
}


static gboolean
tree_view_button_release_event_cb(GtkWidget *widget, GdkEventButton *event,
				  BoardlistView *view)
{
  if (view != NULL)
    {
      view->last_event_x = event->x;
      view->last_event_y = event->y;
      view->last_event_button = event->button;
    }

  return FALSE;
}


static void
tree_view_row_expanded_cb(GtkTreeView *tree_view,
			  GtkTreeIter *iter, GtkTreePath *path,
			  BoardlistView *view)
{
  OchushaBoardCategory *category;
#if DEBUG_WIDGET_MOST
  gchar *name;
#endif

  g_return_if_fail(tree_view != NULL);
  g_return_if_fail(view != NULL);

#if DEBUG_WIDGET_MOST
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_LABEL, &name,
		     BOARDLIST_DATA, &category,
		     -1);
  fprintf(stderr, "row_expanded_cb: %s\n", name);
  G_FREE(name);
#else
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_DATA, &category,
		     -1);
#endif

  if (category != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[CATEGORY_EXPANDED_SIGNAL],
		    0,
		    category);
    }
}


static void
tree_view_row_collapsed_cb(GtkTreeView *tree_view,
			   GtkTreeIter *iter, GtkTreePath *path,
			   BoardlistView *view)
{
  OchushaBoardCategory *category;
#if DEBUG_WIDGET_MOST
  gchar *name;
#endif

  g_return_if_fail(tree_view != NULL);
  g_return_if_fail(view != NULL);

#if DEBUG_WIDGET_MOST
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_LABEL, &name,
		     BOARDLIST_DATA, &category,
		     -1);
  fprintf(stderr, "row_collapsed_cb: %s\n", name);
  G_FREE(name);
#else
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_DATA, &category,
		     -1);
#endif

  if (category != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[CATEGORY_COLLAPSED_SIGNAL],
		    0,
		    category);
    }
}


typedef struct _AppendBoardArgs
{
  GtkTreeIter *category_iter;
  GtkTreeStore *store;
} AppendBoardArgs;


static void
append_board(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  AppendBoardArgs *args = (AppendBoardArgs *)user_data;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(board));

  gtk_tree_store_append(args->store, &info->iter, args->category_iter);
  gtk_tree_store_set(args->store, &info->iter,
		     BOARDLIST_LABEL, board->name,
		     BOARDLIST_ROW_TYPE, BOARDLIST_BOARD_ROW,
		     BOARDLIST_DATA, board,
		     -1);
}


static void
append_category(gpointer data, gpointer user_data)
{
  OchushaBoardCategory *category = (OchushaBoardCategory *)data;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(category));
  AppendBoardArgs *args = (AppendBoardArgs *)user_data;

  args->category_iter = &info->iter;

  gtk_tree_store_append(args->store, args->category_iter, NULL);
  gtk_tree_store_set(args->store, args->category_iter,
		     BOARDLIST_LABEL, category->name,
		     BOARDLIST_ROW_TYPE, BOARDLIST_CATEGORY_ROW,
		     BOARDLIST_DATA, category,
		     -1);

  g_slist_foreach(category->board_list, append_board, args);
}


void
boardlist_view_open(BoardlistView *view, GSList *category_list)
{
  GtkTreeStore *store = gtk_tree_store_new(BOARDLIST_N_COLUMNS,
		/* tree view label */	   G_TYPE_STRING,
		/* row type */		   G_TYPE_INT,
		/* data */		   G_TYPE_POINTER);
  		/* (OchushaBoardCategory *)  (OchushaBulletinBoard *) */
  AppendBoardArgs args = { NULL, store };

  g_slist_foreach(category_list, append_category, &args);

  gtk_tree_view_set_model(view->view, GTK_TREE_MODEL(store));

  if (view->model != NULL)
    g_object_unref(G_OBJECT(view->model));
  view->model = GTK_TREE_MODEL(store);
}


