/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: gziputils.c,v 1.2 2003/06/06 19:38:04 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha_async_buffer.h"

#include "gziputils.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <zlib.h>


#define DEBUG_GZIP	0
#define DEBUG_GZIP_MOST	0

static int check_gzip_header(const char *buffer, int buffer_length);
#if 0
static gboolean check_gzip_header_old(z_stream *zstream);
#endif


struct _GzipBuffer
{
  OchushaAsyncBuffer *output;

  int pending_length;
  char *pending_buffer;
  int pending_buffer_length;

  GzipBufferStatus status;
  z_stream zstream;
};


GzipBuffer *
gzip_buffer_new(OchushaAsyncBuffer *async_buffer)
{
  GzipBuffer *buffer = G_NEW0(GzipBuffer, 1);

  buffer->output = async_buffer;
  buffer->status = GZIP_BUFFER_INITIAL;

  buffer->zstream.next_in = Z_NULL;
  buffer->zstream.avail_in = 0;
  buffer->zstream.next_out = Z_NULL;
  buffer->zstream.avail_out = 0;
  buffer->zstream.zalloc = Z_NULL;
  buffer->zstream.zfree = Z_NULL;
  buffer->zstream.opaque = Z_NULL;

  if (inflateInit2(&buffer->zstream, -MAX_WBITS) != Z_OK)
    {
#if DEBUG_GZIP
      fprintf(stderr, "inflateInit2() failed.\n");
#endif
      G_FREE(buffer);
      return NULL;
    }

  return buffer;
}


void
gzip_buffer_free(GzipBuffer *buffer)
{
  inflateEnd(&buffer->zstream);
  if (buffer->pending_buffer != NULL)
    G_FREE(buffer->pending_buffer);
  G_FREE(buffer);
}


GzipBufferStatus
gzip_buffer_append_data(GzipBuffer *buffer, const char *data,
			unsigned int length)
{
  z_stream *zstream = &buffer->zstream;

#if DEBUG_GZIP
  fprintf(stderr, "gzip_buffer_append_data() called.\n");
#endif

  if (buffer->status == GZIP_BUFFER_ERROR
      || buffer->status == GZIP_BUFFER_INFLATION_DONE)
    {
      fprintf(stderr, "Hey, this buffer has already been closed!\n");
      abort();
    }

  if (buffer->pending_length > 0)
    {
      char *new_buf;
      int pending_length = buffer->pending_length + length;
      if (buffer->pending_buffer_length < pending_length)
	new_buf = (char *)G_REALLOC(buffer->pending_buffer, pending_length);
      else
	new_buf = buffer->pending_buffer;

      memcpy(new_buf + buffer->pending_length, data, length);
      length = pending_length;
      data = new_buf;
      buffer->pending_length = length;
      buffer->pending_buffer = new_buf;
    }

  if (buffer->status == GZIP_BUFFER_INITIAL)
    {
      int gzip_header_len = check_gzip_header(data, length);
#if DEBUG_GZIP
      fprintf(stderr, "check_gzip_header() returns %d\n", gzip_header_len);
#endif
      if (gzip_header_len < 0)
	{
	  buffer->status = GZIP_BUFFER_ERROR;
	  return buffer->status;	/* Header error */
	}

      if (gzip_header_len == 0)
	return buffer->status;		/* No enough input so far */

      buffer->status = GZIP_BUFFER_INFLATING;
      length -= gzip_header_len;
      data += gzip_header_len;
    }

  zstream->next_in = (Bytef *)data;
  zstream->avail_in = length;

  while (TRUE)
    {
      int result;
      int total_length = buffer->output->length;
      int initial_space;

      /* ȤꤢϤ2̤ܰ;͵ */
      if (!ochusha_async_buffer_ensure_free_space(buffer->output,
						  zstream->avail_in * 2,
						  "gziputils.c: gzip_buffer_append_data"))
	{
#if DEBUG_ASYNC_BUFFER_MOST
	  fprintf(stderr, "gzip_buffer_append_data: buffer may have been terminated.\n");
#endif
	  buffer->status = GZIP_BUFFER_ERROR;
	  return buffer->status;	/* Out of memory */
	}

      zstream->next_out = (Bytef *)buffer->output->buffer + total_length;
      zstream->avail_out = buffer->output->buffer_length - total_length;

      initial_space = zstream->avail_out;
      result = inflate(zstream, Z_NO_FLUSH);
      total_length += (initial_space - zstream->avail_out);

      if (!ochusha_async_buffer_update_length(buffer->output, total_length,
				"gziputils.c: gzip_buffer_append_data"))
	{
#if DEBUG_ASYNC_BUFFER_MOST
	  fprintf(stderr, "gzip_buffer_append_data: buffer may have been terminated after update length.\n");
#endif
	  buffer->status = GZIP_BUFFER_ERROR;
	  return buffer->status;	/* Out of memory */
	}

      switch (result)
	{
	case Z_STREAM_END:
	  buffer->status = GZIP_BUFFER_INFLATION_DONE;
	  return buffer->status;

	case Z_DATA_ERROR:
#if DEBUG_GZIP
	  fprintf(stderr, "inflation failed: not gzip data?\n");
#endif
	  buffer->status = GZIP_BUFFER_ERROR;
	  return buffer->status;

	case Z_OK:
	  if (zstream->avail_in == 0)
	    {
	      buffer->pending_length = 0;
#if DEBUG_GZIP
	      fprintf(stderr, "gzip_buffer_append_data: need more input?\n");
#endif
	      return buffer->status;	/* ΤȤϤ­ʤ */
	    }
#if DEBUG_GZIP
	  fprintf(stderr, "gzip_buffer_append_data: need more space?: avail_in=%d\n", zstream->avail_in);
#endif
	  break;

	default:
#if DEBUG_GZIP
	  fprintf(stderr, "inflation failed: %s\n",
		  (zstream->msg != NULL) ? zstream->msg : "unknown error");
#endif
	  buffer->status = GZIP_BUFFER_ERROR;
	  return buffer->status;
	}
    }
}


gboolean
inflate_gzip_buffer(const char *input, unsigned int input_length,
		    char **output, unsigned int *output_length)
{
  z_stream zstream;
  char *out_buf = *output;
  unsigned int out_len = *output_length;

  int gzip_header_len = check_gzip_header(input, input_length);
  if (gzip_header_len < 0)
    {
#if DEBUG_GZIP
      fprintf(stderr, "inflation failed: invalid gzip header.\n");
#endif
      return FALSE;
    }

  zstream.next_in = (Bytef *)(input + gzip_header_len);
  zstream.avail_in = input_length - gzip_header_len;
  zstream.next_out = Z_NULL;
  zstream.avail_out = 0;
  zstream.zalloc = Z_NULL;
  zstream.zfree = Z_NULL;
  zstream.opaque = Z_NULL;

  if (inflateInit2(&zstream, -MAX_WBITS) != Z_OK)
    {
#if DEBUG_GZIP
      fprintf(stderr, "inflation failed: inflateInit2 failed.\n");
#endif
      return FALSE;
    }

  if (out_buf == NULL || out_len < input_length * 2)
    {
      out_len = input_length * 2;	/* 褺Ϥܽ */
      out_buf = (char *)G_REALLOC(out_buf, out_len);
    }

  zstream.next_out = out_buf;
  zstream.avail_out = out_len;

  while (TRUE)
    {
      switch (inflate(&zstream, Z_NO_FLUSH))
	{
	case Z_STREAM_END:
	  {
	    inflateEnd(&zstream);
	    *output = (char *)G_REALLOC(out_buf, zstream.total_out);
	    *output_length = zstream.total_out;
	    return TRUE;
	  }

	case Z_DATA_ERROR:
	  fprintf(stderr, "inflation failed: not gzip data.\n");
	  goto inflation_failed;

	case Z_OK:
	  /* output buffer full */
	  {
	    out_len *= 2;	/*  */
	    out_buf = (char *)G_REALLOC(out_buf, out_len);
	    zstream.next_out = out_buf + zstream.total_out;
	    zstream.avail_out += out_len / 2;
	    break;
	  }
	default:
#if DEBUG_GZIP
	  fprintf(stderr, "inflation failed: %s\n",
		  (zstream.msg != NULL) ? zstream.msg : "unknown error");
#endif
	  goto inflation_failed;
	}
    }

 inflation_failed:
  inflateEnd(&zstream);

  if (*output == NULL)
    G_FREE(out_buf);	/* ߤout_bufϤδؿǿ˳ݤ */
  else
    *output = out_buf;	/* reallocΤʤcallerˤޤ */

  *output_length = 0;

  return FALSE;
}


#define HEAD_CRC     0x02 /* bit 1 set: header CRC present */
#define EXTRA_FIELD  0x04 /* bit 2 set: extra field present */
#define ORIG_NAME    0x08 /* bit 3 set: original file name present */
#define COMMENT      0x10 /* bit 4 set: file comment present */
#define RESERVED     0xE0 /* bits 5..7: reserved */


static int
check_gzip_header(const char *buffer, int buffer_length)
{
  unsigned int method;	/* method byte */
  unsigned int flags;	/* flags byte */
  unsigned int len;
  static const char gz_magic[2] = { 0x1f, 0x8b };
  const char *cur_pos = buffer;
  const char *tail_pos = buffer + buffer_length;

  if (cur_pos + 4 >= tail_pos)
    {
#if DEBUG_GZIP
      fprintf(stderr, "Too short for gzip header.\n");
#endif
      return 0;
    }

  /* Check the gzip magic header */
  for (len = 0; len < 2; len++)
    {
      if (*cur_pos++ != gz_magic[len])
	{
#if DEBUG_GZIP
	  fprintf(stderr, "No matching magic number.\n");
#endif
	  return -1;
	}
    }

  method = *cur_pos++;
  flags = *cur_pos++;

#if DEBUG_GZIP_MOST
  fprintf(stderr, "flags: 0x%x\n", flags);
#endif

  if (method != Z_DEFLATED || (flags & RESERVED) != 0)
    return -1;	/* Invalid header */

  /* Discard time, xflags and OS code: */
  cur_pos += 6;
  if (cur_pos >= tail_pos)
    {
#if DEBUG_GZIP
      fprintf(stderr, "Too short for gzip header.\n");
#endif
      return 0;
    }

  if ((flags & EXTRA_FIELD) != 0)
    {
      if (cur_pos + 2 >= tail_pos)
	{
#if DEBUG_GZIP
	  fprintf(stderr, "Too short for gzip header.\n");
#endif
	  return 0;
	}
#if DEBUG_GZIP_MOST
      fprintf(stderr, "skipping the extra field\n", cur_pos);
#endif
      /* skip the extra field */
      len = (unsigned int)*cur_pos++;
      len += ((unsigned int)*cur_pos++) << 8;
      cur_pos += len;

      if (cur_pos >= tail_pos)
	{
	  fprintf(stderr, "Too short for gzip header.\n");
	  return FALSE;
	}
    }

  if ((flags & ORIG_NAME) != 0)
    {
#if DEBUG_GZIP_MOST
      fprintf(stderr, "ORIG_NAME = \"%s\"\n", cur_pos);
#endif
      /* skip the original file name */
      while (*cur_pos++ != '\0')
	if (cur_pos >= tail_pos)
	  {
	    fprintf(stderr, "Too short for gzip header.\n");
	    return FALSE;
	  }
    }

  if ((flags & COMMENT) != 0)
    {
#if DEBUG_GZIP_MOST
      fprintf(stderr, "COMMENT = \"%s\"\n", cur_pos);
#endif
      /* skip the .gz file comment */
      while (*cur_pos++ != '\0')
	if (cur_pos >= tail_pos)
	  {
	    fprintf(stderr, "Too short for gzip header.\n");
	    return FALSE;
	  }
    }

  if ((flags & HEAD_CRC) != 0)
    {
#if DEBUG_GZIP_MOST
      fprintf(stderr, "skipping CRC\n");
#endif
      /* skip the header crc */
      cur_pos += 2;
      if (cur_pos >= tail_pos)
	{
	  fprintf(stderr, "Too short for gzip header.\n");
	  return FALSE;
	}
    }

  return cur_pos - buffer;
}


#if 0
static gboolean
check_gzip_header_old(z_stream *zstream)
{
  unsigned int method;	/* method byte */
  unsigned int flags;	/* flags byte */
  unsigned int len;
  static const char gz_magic[2] = { 0x1f, 0x8b };
  char *cur_pos = zstream->next_in;
  char *tail_pos = zstream->next_in + zstream->avail_in;

  if (cur_pos + 4 >= tail_pos)
    {
      fprintf(stderr, "Too short for gzip header.\n");
      return FALSE;
    }

  /* Check the gzip magic header */
  for (len = 0; len < 2; len++)
    {
      if (*cur_pos++ != gz_magic[len])
	{
	  fprintf(stderr, "No matching magic number.\n");
	  return FALSE;
	}
    }

  method = *cur_pos++;
  flags = *cur_pos++;

#if DEBUG_GZIP_MOST
  fprintf(stderr, "flags: 0x%x\n", flags);
#endif

  if (method != Z_DEFLATED || (flags & RESERVED) != 0)
    return FALSE;	/* Invalid header */

  /* Discard time, xflags and OS code: */
  cur_pos += 6;
  if (cur_pos >= tail_pos)
    {
      fprintf(stderr, "Too short for gzip header.\n");
      return FALSE;
    }

  if ((flags & EXTRA_FIELD) != 0)
    {
      if (cur_pos + 2 >= tail_pos)
	{
	  fprintf(stderr, "Too short for gzip header.\n");
	  return FALSE;
	}
#if DEBUG_GZIP_MOST
      fprintf(stderr, "skipping the extra field\n", cur_pos);
#endif
      /* skip the extra field */
      len = (unsigned int)*cur_pos++;
      len += ((unsigned int)*cur_pos++) << 8;
      cur_pos += len;

      if (cur_pos >= tail_pos)
	{
	  fprintf(stderr, "Too short for gzip header.\n");
	  return FALSE;
	}
    }

  if ((flags & ORIG_NAME) != 0)
    {
#if DEBUG_GZIP_MOST
      fprintf(stderr, "ORIG_NAME = \"%s\"\n", cur_pos);
#endif
      /* skip the original file name */
      while (*cur_pos++ != '\0')
	if (cur_pos >= tail_pos)
	  {
	    fprintf(stderr, "Too short for gzip header.\n");
	    return FALSE;
	  }
    }

  if ((flags & COMMENT) != 0)
    {
#if DEBUG_GZIP_MOST
      fprintf(stderr, "COMMENT = \"%s\"\n", cur_pos);
#endif
      /* skip the .gz file comment */
      while (*cur_pos++ != '\0')
	if (cur_pos >= tail_pos)
	  {
	    fprintf(stderr, "Too short for gzip header.\n");
	    return FALSE;
	  }
    }

  if ((flags & HEAD_CRC) != 0)
    {
#if DEBUG_GZIP_MOST
      fprintf(stderr, "skipping CRC\n");
#endif
      /* skip the header crc */
      cur_pos += 2;
      if (cur_pos >= tail_pos)
	{
	  fprintf(stderr, "Too short for gzip header.\n");
	  return FALSE;
	}
    }

  len = cur_pos - ((char *)zstream->next_in);
  zstream->avail_in -= len;
  zstream->next_in = cur_pos;
  zstream->total_in += len;

  return TRUE;
}
#endif
