/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_network_broker.c,v 1.4 2003/06/07 14:35:32 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_network_broker.h"

#include "gziputils.h"
#include "worker.h"

#include "marshal.h"

#include <ghttp.h>
#include <glib.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/mman.h>

#include <time.h>

#include <unistd.h>

#ifndef MAP_NOCORE
# define MAP_NOCORE	0
#endif


#define DISABLE_PERSISTENT_CONNECTION		0
#define ENABLE_GZIPPED_DIFFERENCIAL_READ	0

#define CACHE_COMPARE_SIZE	512	/* Ĥѹǽˤ٤ */
#define HEADER_BUFFER_LENGTH	256

#define POLLING_INTERVAL_SEC	0
#define POLLING_INTERVAL_NANO	20000000	/* 20ms */


static struct timespec polling_interval =
{
  POLLING_INTERVAL_SEC,
  POLLING_INTERVAL_NANO
};


#define DEBUG_NETWORK		0
#define DEBUG_NETWORK_MOST	0
#define DEBUG_MMAP		0


static void ochusha_network_broker_class_init(OchushaNetworkBrokerClass *klass);
static void ochusha_network_broker_init(OchushaNetworkBroker *broker);
static void ochusha_network_broker_finalize(GObject *object);


GType
ochusha_network_broker_get_type(void)
{
  static GType broker_type = 0;

  if (broker_type == 0)
    {
      static const GTypeInfo broker_info =
	{
	  sizeof(OchushaNetworkBrokerClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)ochusha_network_broker_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(OchushaNetworkBroker),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_network_broker_init,
	};

      broker_type = g_type_register_static(G_TYPE_OBJECT,
					   "OchushaNetworkBroker",
					   &broker_info, 0);
    }

  return broker_type;
}


enum {
  ACCESS_STARTED_SIGNAL,
  ACCESS_PROGRESSED_SIGNAL,
  ACCESS_COMPLETED_SIGNAL,
  ACCESS_TERMINATED_SIGNAL,
  ACCESS_FAILED_SIGNAL,
  LAST_SIGNAL
};


static GObjectClass *parent_class = NULL;
static gint broker_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0 };
static GQuark broker_buffer_status_id;
static GQuark broker_job_args_id;


static void
ochusha_network_broker_class_init(OchushaNetworkBrokerClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_network_broker_finalize;

  broker_signals[ACCESS_STARTED_SIGNAL] =
    g_signal_new("access_started",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_started),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_ASYNC_BUFFER);
  broker_signals[ACCESS_PROGRESSED_SIGNAL] =
    g_signal_new("access_progressed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_progressed),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT_INT_INT,
		 G_TYPE_BOOLEAN, 3,
		 OCHUSHA_TYPE_ASYNC_BUFFER,
		 G_TYPE_INT,
		 G_TYPE_INT);
  broker_signals[ACCESS_COMPLETED_SIGNAL] =
    g_signal_new("access_completed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_completed),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_ASYNC_BUFFER);
  broker_signals[ACCESS_TERMINATED_SIGNAL] =
    g_signal_new("access_terminated",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_terminated),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_ASYNC_BUFFER);
  /*
   * XXX: failˤOchushaAsyncBufferʤȤ⤢롣
   *      OBJECTΥʥΰNULLݥ󥿤ϤȡGLibٹ
   *      Ф⤷ʤΤǡٹ𤬽Ф褦ʤ顢OBJECTǤʤ
   *      POINTERѹ٤
   */
  broker_signals[ACCESS_FAILED_SIGNAL] =
    g_signal_new("access_failed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_failed),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT_INT_STRING,
		 G_TYPE_BOOLEAN, 3,
		 OCHUSHA_TYPE_ASYNC_BUFFER,
		 G_TYPE_INT,
		 G_TYPE_STRING);

  broker_buffer_status_id
    = g_quark_from_static_string("OchushaNetworkBroker::BufferStatus");
  broker_job_args_id
    = g_quark_from_static_string("OchushaNetworkBroker::JobArgs");

  klass->access_started = NULL;
  klass->access_progressed = NULL;
  klass->access_completed = NULL;
  klass->access_terminated = NULL;
  klass->access_failed = NULL;
}


static void
ochusha_network_broker_init(OchushaNetworkBroker *broker)
{
  broker->config = NULL;
}


static void
ochusha_network_broker_finalize(GObject *object)
{
  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


OchushaNetworkBroker *
ochusha_network_broker_new(OchushaConfig *config)
{
  OchushaNetworkBroker *broker
    = OCHUSHA_NETWORK_BROKER(g_object_new(OCHUSHA_TYPE_NETWORK_BROKER, NULL));
  broker->config = config;

  return broker;
}


static void
munmap_when_finished(OchushaAsyncBuffer *buffer)
{
  munmap((void *)buffer->buffer, buffer->length);
#if DEBUG_MMAP
  fprintf(stderr, "mmapped buffer(at %p) is unmapped.\n",
	  buffer->buffer);
#endif
}


static void
ochusha_network_broker_buffer_status_free(OchushaNetworkBrokerBufferStatus *status)
{
  if (status->last_modified != NULL)
    G_FREE(status->last_modified);
  G_FREE(status);
}


static OchushaAsyncBuffer *
get_mmapped_buffer(int fd)
{
  char *buf;
  off_t len;
  OchushaAsyncBuffer *buffer;
  OchushaNetworkBrokerBufferStatus *status;

  if (fd < 0)
    return NULL;

  status = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;

  len = lseek(fd, 0, SEEK_END);
  buf = mmap(NULL, len, PROT_READ, MAP_NOCORE | MAP_PRIVATE, fd, 0);
  if (buf == MAP_FAILED)
    {
      fprintf(stderr, "mmap failed due to: %s (%d)\n", strerror(errno), errno);
      G_FREE(status);
      return NULL;
    }
  close(fd);

#if DEBUG_MMAP
  fprintf(stderr, "allocating mmapped buffer at %p\n", buf);
#endif

  buffer = ochusha_async_buffer_new(buf, len, munmap_when_finished);
  g_object_set_qdata_full(G_OBJECT(buffer), broker_buffer_status_id, status,
		(GDestroyNotify)ochusha_network_broker_buffer_status_free);
  if (!ochusha_async_buffer_fix(buffer,
			"ochusha_network_broker.c: get_mmapped_buffer()"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      /* ȤϤɤ褦ʤ */
    }
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "get_mmapped_buffer(): buffer=%p\n", buffer);
#endif

  return buffer;
}


static int
read_cache_to_buffer(const OchushaConfig *config, const char *url,
		     OchushaAsyncBuffer *buffer)
{
  int len;
  int fd = ochusha_config_cache_open_file(config, url, O_RDONLY);

  if (fd < 0)
    return 0;	/* åʤξ */

  /* å夢ꡣ*/
  len = lseek(fd, 0, SEEK_END);
  lseek(fd, 0, SEEK_SET);
  if (ochusha_async_buffer_resize(buffer, len,
			"ochusha_network_broker.c: read_cache_to_buffer()"))
    {
      len = read(fd, (void *)buffer->buffer, len);
      if (!ochusha_async_buffer_update_length(buffer, len,
			"ochusha_network_broker.c: read_cache_to_buffer()"))
	{
#if DEBUG_ASYNC_BUFFER
	  fprintf(stderr, "read_cache_to_buffer(): buffer may have been terminated.\n");
#endif
	  /* ξϤɤ褦ʤ */
	}
    }
  else
    {
#if DEBUG_ASYNC_BUFFER
      fprintf(stderr, "read_cache_to_buffer(): buffer may have been terminated.\n");
#endif
      len = 0;	/* Out of memory? */
    }
  close(fd);

  return len;
}


static void
write_buffer_to_cache(const OchushaConfig *config, const char *url,
		      OchushaAsyncBuffer *buffer)
{
  int fd = ochusha_config_cache_open_file(config, url,
					  O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      ssize_t len = write(fd, (void *)buffer->buffer, buffer->length);
      close(fd);
      if (len != buffer->length)
	{
	  ochusha_config_cache_unlink_file(config, url);
#if DEBUG_NETWORK
	  fprintf(stderr, "Couldn't update cache file for %s: %s (%d)\n",
		  url, strerror(errno), errno);
#endif
	}
    }
#if DEBUG_NETWORK
  else
    fprintf(stderr, "Couldn't open cache file for %s: %s (%d)\n",
	    url, strerror(errno), errno);
#endif
}


/*
 * ºݤnetwork˥뵡ǽ
 */


static gboolean
setup_common_request_headers(const OchushaConfig *config,
			     ghttp_request *request)
{
  gboolean result = TRUE;

  if (config->enable_proxy && config->proxy_url != NULL)
    {
      int result = ghttp_set_proxy(request, config->proxy_url);
      if (result == 0)
	{
	  if (config->enable_proxy_auth
	      && config->proxy_user != NULL && config->proxy_password != NULL)
	    {
	      result = ghttp_set_proxy_authinfo(request, config->proxy_user,
						config->proxy_password);
	      if (result != 0)
		{
		  fprintf(stderr, "Invalid proxy auth info.\n");
		  result = FALSE;
		}
	    }
	}
      else
	{
	  fprintf(stderr, "Invalid proxy URL: %s\n", config->proxy_url);
	  result = FALSE;
	}
    }

  ghttp_set_header(request, http_hdr_User_Agent, OCHUSHA_USER_AGENT);

  return result;
}


typedef enum
{
  UNKNOWN_ENCODING = -1,
  STANDARD_ENCODING,
  GZIP_ENCODING,
} ContentEncodingType;


static ContentEncodingType
get_content_encoding_type(ghttp_request *request)
{
  const char *encoding = ghttp_get_header(request, http_hdr_Content_Encoding);

  if (encoding != NULL)
    {
      if (strcmp(encoding, "gzip") == 0)
	return GZIP_ENCODING;
      else
	return UNKNOWN_ENCODING;
    }

  return STANDARD_ENCODING;
}


/*
 * OchushaAsyncBufferbroker_job_args_idqdataȤƤäĤ롣
 */
typedef struct _NetworkBrokerJobArgs
{
  OchushaNetworkBroker *broker;

  char *url;
  char *if_modified_since;

  ghttp_request *request;
} NetworkBrokerJobArgs;


static gboolean
http_read_from_url(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  const char *url = args->url;
  const char *if_modified_since = args->if_modified_since;
  const char *error_message = NULL;

  /* ʲϸľɬ */
#if DEBUG_NETWORK
  int sock_fd;
#endif
  int status_code = 0;
  ghttp_request *request = NULL;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;
  GzipBuffer *gzip_buffer = NULL;
  ghttp_status state;

  request = ghttp_request_new();
  args->request = request;	/* 塹Τˡġ */

  ghttp_set_uri(request, (char *)url);

  /*
   * MEMO: HTTPǤ̿ޤstatusɤʤ
   *       404ʤɤˤ̿ԤƤǤäƤ⡢餫Υǡ
   *       ֤äƤΤ̤Ǥꡢ줬ŪΤΤǤϤʤȤϥ쥹
   *       ݥ󥹤μλޤǤ狼ʤΤǡäasync⡼ɤǤ
   *       ʤGETꥯȤϤޤ̤ߤ
   *
   *       Ǥasync⡼ɤȤäƤΤǡǰΤHEADꥯȸ
   *       GETꥯȤȤƧǤ롣̩ˤϡHEADꥯȤ
   *       200ľƱURLФGETꥯȤԤǽϤ
   *       ΤޤǤϤޤǤϥʤ
   */
  ghttp_set_type(request, ghttp_type_head);
#if DISABLE_PERSISTENT_CONNECTION
  ghttp_set_header(request, http_hdr_Connection, "close");
#endif
  ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");

  if (if_modified_since != NULL
      && ochusha_config_cache_file_exist(broker->config, url))
    {
      ghttp_set_header(request, http_hdr_If_Modified_Since, if_modified_since);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "If-Modified-Since: %s\n", if_modified_since);
#endif
    }
  else
    if_modified_since = NULL;

  if (!setup_common_request_headers(broker->config, request))
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
		    _("Proxy setting may be wrong."),
		    &signal_result);

      ghttp_request_destroy(request);
      args->request = NULL;
      return FALSE;
    }    

  g_signal_emit(G_OBJECT(broker),
		broker_signals[ACCESS_STARTED_SIGNAL],
		0,
		buffer,
		&signal_result);

  ghttp_set_sync(request, ghttp_async);
  ghttp_prepare(request);

  /*
   * MEMO: λޤǥ롼פΤǡsync⡼ɤǥƤ
   *       褤ߤ˽᤯ȿ뤿async⡼ɤ
   *       롼פƤ롣
   */
  while (TRUE)
    {
      state = ghttp_process(request);
#if DEBUG_NETWORK_MOST
      sock_fd = ghttp_get_socket(request);
      fprintf(stderr, "1: sock_fd=%d state=%d\n", sock_fd, state);
#endif
      if (!ochusha_async_buffer_check_active(buffer,
			"ochusha_network_broker.c: http_read_from_url()"))
	state = ghttp_error;

      if (state == ghttp_error || state == ghttp_done)
	break;
      nanosleep(&polling_interval, NULL);
    }

#if DEBUG_NETWORK_MOST
  {
    char **hdrs;
    int num_hdrs;
    int i;
    int result = ghttp_get_header_names(request, &hdrs, &num_hdrs);
    if (result == 0)
      {
	for (i = 0; i < num_hdrs; i++)
	  {
	    fprintf(stderr, "%s: %s\n",
		    hdrs[i], ghttp_get_header(request, hdrs[i]));
	    G_FREE(hdrs[i]);
	  }
	G_FREE(hdrs);
      }
  }
#endif

  status_code = ghttp_status_code(request);

  if (state == ghttp_error || status_code != 200)
    {
      status->status_code = status_code;

#if DEBUG_NETWORK
      if (status_code != 0)
	fprintf(stderr, "status_code=%d (%s)\n",
		status_code, ghttp_reason_phrase(request));
      else
	fprintf(stderr, "Unknown error.\n");
#endif
      if (status_code != 304)
	{
	  error_message = _("Unexpected response for HEAD request.");
	  goto error_exit;
	}
      /*
       * MEMO: 304If-Modified-Sinceդѹʤ̣롣
       *       ̤ȤƤǤ̿ԤʤΤFALSE֤
       *       顼ǤϤʤcallerϥåǤΤȤ
       *       ȤȤǤ롣
       */

      ghttp_request_destroy(request);
      args->request = NULL;
      return FALSE;
    }

# if DEBUG_NETWORK_MOST
  fprintf(stderr, "status_code=%d (%s)\n",
	  status_code, ghttp_reason_phrase(request));
# endif

#if DISABLE_PERSISTENT_CONNECTION
  ghttp_request_destroy(request);
  request = ghttp_request_new();
  args->request = request;
#else
  ghttp_clean(request);
#endif

  ghttp_set_uri(request, (char *)url);
  ghttp_set_type(request, ghttp_type_get);
  ghttp_set_header(request, http_hdr_Connection, "close");
  ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
  if (!setup_common_request_headers(broker->config, request))
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
		    _("Proxy setting may be wrong."),
		    &signal_result);

      ghttp_request_destroy(request);
      args->request = NULL;
      return FALSE;
    }    

  ghttp_set_sync(request, ghttp_async);
  ghttp_prepare(request);

  while (TRUE)
    {
      int body_len;
      char *body;
      state = ghttp_process(request);
#if DEBUG_NETWORK_MOST
      sock_fd = ghttp_get_socket(request);
      fprintf(stderr, "2: sock_fd=%d state=%d\n", sock_fd, state);
#endif

      if (state == ghttp_not_done)
	ghttp_flush_response_buffer(request);
      else if (state == ghttp_error)
	{
	  error_message = _("Unknown I/O error.");
	  goto error_exit;
	}

      body = ghttp_get_body(request);
      body_len = ghttp_get_body_len(request);
      
      if (body_len > 0)
	{
	  ghttp_current_status current_status = ghttp_get_status(request);
	  if (current_status.bytes_total > 0)
	    g_signal_emit(G_OBJECT(broker),
			  broker_signals[ACCESS_PROGRESSED_SIGNAL],
			  0,
			  buffer,
			  current_status.bytes_read,
			  current_status.bytes_total,
			  &signal_result);

	  if (content_encoding == UNKNOWN_ENCODING)
	    {
	      /* ˤ1٤ʤ */
	      content_encoding = get_content_encoding_type(request);

	      switch (content_encoding)
		{
		case GZIP_ENCODING:
		  gzip_buffer = gzip_buffer_new(buffer);
		  if (gzip_buffer == NULL)
		    {
		      error_message = _("Out of memory.");
		      goto error_exit;
		    }
		  break;

		case UNKNOWN_ENCODING:
		  error_message = _("Unknown transfer encoding.");
		  goto error_exit;

		case STANDARD_ENCODING:
		  break;
		}
	    }

	  if (content_encoding == GZIP_ENCODING)
	    {
	      GzipBufferStatus result
		= gzip_buffer_append_data(gzip_buffer, body, body_len);
	      if (result == GZIP_BUFFER_ERROR)
		{
		  error_message = _("Inflation failed.");
		  gzip_buffer_free(gzip_buffer);
		  goto error_exit;
		}
	      if (result == GZIP_BUFFER_INFLATION_DONE)
		{
		  gzip_buffer_free(gzip_buffer);
		  gzip_buffer = NULL;
#if DEBUG_NETWORK
		  if (state != ghttp_done)
		    fprintf(stderr, "Requested file contains garbage?\n");
#endif
		  break;
		}
	    }
	  else if (!ochusha_async_buffer_append_data(buffer, body, body_len,
					"network.c: http_read_from_url"))
	    {	/* content_encoding == STANDARD_ENCODING */
	      g_signal_emit(G_OBJECT(broker),
			    broker_signals[ACCESS_TERMINATED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);
	      ghttp_request_destroy(request);
	      args->request = NULL;

	      status->state
		= OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	      return FALSE;
	    }
	}

      if (state == ghttp_done)
	break;

      nanosleep(&polling_interval, NULL);
    }

  if (gzip_buffer != NULL)
    {
#if DEBUG_NETWORK
      fprintf(stderr, "What's happen?  Connection was closed but inflation hasn't been completed.\n");
#endif
      gzip_buffer_free(gzip_buffer);
    }

  status_code = ghttp_status_code(request);

  if (status_code == 200)
    {
      const char *last_modified
	= ghttp_get_header(request, http_hdr_Last_Modified);
      if (last_modified != NULL)
	status->last_modified = G_STRDUP(last_modified);
      else
	status->last_modified = NULL;
      status->status_code = status_code;
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_COMPLETED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);

      ghttp_request_destroy(request);
      args->request = NULL;

      return TRUE;
    }

 error_exit:
  status_code = ghttp_status_code(request);
  if (error_message != NULL)
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    error_message,
		    &signal_result);
    }
  else if (status_code != 0)
    {
      char message[4096];
      snprintf(message, 4096, "Access failed: %d (%s)",
	       status_code, ghttp_reason_phrase(request));
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    message,
		    &signal_result);
    }
  else
    g_signal_emit(G_OBJECT(broker),
		  broker_signals[ACCESS_FAILED_SIGNAL],
		  0,
		  buffer,
		  OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		  _("Access failed: unknown reason."),
		  &signal_result);
  ghttp_request_destroy(request);
  args->request = NULL;

  return FALSE;

}


static void
destruct_job_args(NetworkBrokerJobArgs *job_args)
{
  if (job_args == NULL)
    return;

  if (job_args->broker != NULL)
    g_object_unref(G_OBJECT(job_args->broker));

  if (job_args->url != NULL)
    G_FREE(job_args->url);

  if (job_args->if_modified_since != NULL)
    G_FREE(job_args->if_modified_since);

  if (job_args->request != NULL)
    ghttp_request_destroy(job_args->request);

  G_FREE(job_args);
}


static OchushaAsyncBuffer *
ochusha_network_broker_employ_worker_thread(OchushaNetworkBroker *broker,
					    const char *url,
					    const char *if_modified_since,
					    JobFunc *job_function)
{
  NetworkBrokerJobArgs *args;
  WorkerJob *job;
  OchushaNetworkBrokerBufferStatus *status
    = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
  OchushaAsyncBuffer *buffer = ochusha_async_buffer_new(NULL, 0, NULL);
  g_object_set_qdata_full(G_OBJECT(buffer), broker_buffer_status_id, status,
		(GDestroyNotify)ochusha_network_broker_buffer_status_free);

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_UNKNOWN;

  args = G_NEW0(NetworkBrokerJobArgs, 1);
  args->broker = broker;
  args->url = G_STRDUP(url);
  if (if_modified_since != NULL
      && ochusha_config_cache_file_exist(broker->config, url))
    args->if_modified_since = G_STRDUP(if_modified_since);

  g_object_set_qdata_full(G_OBJECT(buffer), broker_job_args_id,
			  args, (GDestroyNotify)destruct_job_args);

  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = job_function;
  job->args = buffer;

  /* åɤΤ˳ */
  g_object_ref(G_OBJECT(broker));
  g_object_ref(G_OBJECT(buffer));

  commit_job(job);

  return buffer;
}


static int
get_content_length(ghttp_request *request)
{
  int length;
  const char *content_len = ghttp_get_header(request, http_hdr_Content_Length);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "Content-Type: %s\n",
	  ghttp_get_header(request, http_hdr_Content_Type));
#endif
  if (content_len == NULL)
    return -1;		/* Content-Lengthإåʤ */

  if (sscanf(content_len, "%d", &length) < 1)
    return -1;

  return length;
}


static void
try_update_cache(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
						  broker_job_args_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  char *url = args->url;
  int len;
  ghttp_request *request;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;
  int status_code;
  const char *error_message = NULL;

  if (!ochusha_async_buffer_active_ref(buffer,
			"ochusha_network_broker.c: try_update_cache()"))
    {
      g_signal_emit(G_OBJECT(args->broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);

      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(args->broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		    _("Access terminated."),
		    &signal_result);

      ochusha_async_buffer_fix(buffer,
			       "ochusha_network_broker.c: try_update_cache()");
      g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
      g_object_unref(G_OBJECT(buffer));

      return;
    }

  len = read_cache_to_buffer(args->broker->config, url, buffer);

  if (len > 0)
    {	/* å夢ξ */

      /* MEMO:
       * ޤHEADꥯȤĹåĹӡ
       * åä麹ʬԡ
       */
      /* ̤ghttpν */
      ghttp_status state = ghttp_not_done;

      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
      request = ghttp_request_new();
      args->request = request;
      if (request == NULL)
	{
	  /* Out of memory? */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  error_message = _("Out of memory.");
	  goto error_exit;
	}

      ghttp_set_uri(request, url);

      /* ޤHEADꥯȤͻҤõ */
      ghttp_set_type(request, ghttp_type_head);

#if DISABLE_PERSISTENT_CONNECTION
      ghttp_set_header(request, http_hdr_Connection, "close");
#endif
#if ENABLE_GZIPPED_DIFFERENCIAL_READ
      ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
#endif

      if (args->if_modified_since != NULL)
	{
	  ghttp_set_header(request, http_hdr_If_Modified_Since,
			   args->if_modified_since);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "If-Modified-Since: %s\n", args->if_modified_since);
#endif
	}
      if (!setup_common_request_headers(args->broker->config, request))
	{
	  /* ۤäƥåϤɤΤ */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  g_signal_emit(G_OBJECT(args->broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
			_("Proxy setting may be wrong."),
			&signal_result);
	  ghttp_request_destroy(request);
	  args->request = NULL;

	  goto finish_try_update_cache;
	}

      g_signal_emit(G_OBJECT(args->broker),
		    broker_signals[ACCESS_STARTED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);

      ghttp_set_sync(request, ghttp_async);
      ghttp_prepare(request);

      while (TRUE)
	{
	  state = ghttp_process(request);
	  if (!ochusha_async_buffer_check_active(buffer,
			"ochusha_network_broker.c: try_update_cache()"))
	    {
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_TERMINATED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);

	      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	      g_signal_emit(G_OBJECT(args->broker),
		broker_signals[ACCESS_FAILED_SIGNAL],
		0,
		buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."),
		&signal_result);

	      ghttp_request_destroy(request);
	      args->request = NULL;

	      goto finish_try_update_cache;
	    }
	  if (state != ghttp_not_done)
	    break;

	  nanosleep(&polling_interval, NULL);
	}

      status_code = ghttp_status_code(request);

      if (state == ghttp_error)
	goto error_exit;

      if (status_code == 304 && args->if_modified_since != NULL)
	{
	  status->last_modified = G_STRDUP(args->if_modified_since);
	  status->status_code = status_code;
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  g_signal_emit(G_OBJECT(args->broker),
			broker_signals[ACCESS_COMPLETED_SIGNAL],
			0,
			buffer,
			&signal_result);
	  ghttp_request_destroy(request);
	  args->request = NULL;

	  goto finish_try_update_cache;
	}

      if (status_code == 200)
	{
	  char header[HEADER_BUFFER_LENGTH];
	  char cache_compare_buffer[CACHE_COMPARE_SIZE];
	  int offset = len - CACHE_COMPARE_SIZE;
	  GzipBuffer *gzip_buffer = NULL;
	  int new_len = get_content_length(request);

	  if (new_len < len)
	    goto cache_is_dirty;	/* ¿ʬܡȯ */

	  /* 
	   * MEMO: Υɤˤꡢåʬե
	   *       ˥å夬äƤ밷ˤʤäƤޤ
	   *       If-Modified-SinceդǥåƤΤ
	   *       ѹΤʤˤĤƤϤϲǤ롣
	   */
	  if (offset < 0)
	    goto cache_is_dirty;	/* ʬʤˤĤκƼ */


	  /*  ޤǤHEADꥯȤˤͻҸ */


	  /*  ʲϺʬμ¹ʬ */

	  if (snprintf(header, HEADER_BUFFER_LENGTH, "bytes=%d-", offset)
	      >= HEADER_BUFFER_LENGTH)
	    goto cache_is_dirty;	/* ľʤκƼ*/

#if DISABLE_PERSISTENT_CONNECTION
	  ghttp_request_destroy(request);
	  request = ghttp_request_new();
	  args->request = request;
#else
	  ghttp_clean(request);
#endif

	  ghttp_set_uri(request, url);
	  ghttp_set_type(request, ghttp_type_get);
	  ghttp_set_header(request, http_hdr_Connection, "close");
#if ENABLE_GZIPPED_DIFFERENCIAL_READ
	  ghttp_set_header(request, http_hdr_Accept_Encoding, "gzip");
#endif

#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "setting Range: %s\n", header);
#endif
	  ghttp_set_header(request, http_hdr_Range, header);

	  if (!setup_common_request_headers(args->broker->config, request))
	    {
	      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	      g_signal_emit(G_OBJECT(args->broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
			_("Proxy setting may be wrong."),
			&signal_result);
	      ghttp_request_destroy(request);
	      args->request = NULL;

	      goto finish_try_update_cache;
	    }

	  ghttp_set_sync(request, ghttp_async);
	  ghttp_prepare(request);

	  /*
	   * Content-Encoding狼äCACHE_COMPARE_SIZEʬ
	   * Хåեɤ߹ߡå夬ɤǧ
	   *
	   * ägzipʾбݤʤΤǡ
	   * ޤOchushaAsyncBufferͭǡCACHE_COMPARE_SIZEʬ
	   * (å夵Ƥǡ)Хåե˥ԡ
	   * OchushaAsyncBufferͭǡĹCACHE_COMPARE_SIZEʬ餷Ƥ
	   * ޤGzipBuffer٤餫GzipBufferAPI
	   * ɲä٤
	   * OchushaAsyncBufferconsumer¦read-onlyǡХåեΰ輫Τ
	   * ͭʤΤɤޤƤʤޤå夬ʾ
	   * Ʊǡ񤭤å夬Ȥ
	   * äˤϡ̵Τǰʥǡ
	   * ޤޤƤƤⵤˤʤǤ
	   */
	  /* λǤϡbuffer->length == len */
	  memcpy(cache_compare_buffer, (char *)buffer->buffer + offset,
		 CACHE_COMPARE_SIZE);

	  if (!ochusha_async_buffer_update_length(buffer, offset,
			"ochusha_network_broker.c: try_update_cache()"))
	    {
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_TERMINATED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);

	      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	      g_signal_emit(G_OBJECT(args->broker),
		broker_signals[ACCESS_FAILED_SIGNAL],
		0,
		buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."),
		&signal_result);

	      ghttp_request_destroy(request);
	      args->request = NULL;

	      goto finish_try_update_cache;
	    }

	  while (TRUE)
	    {
	      /* ̿λ⤷ϥå夬äƤΤȽޤǥ롼*/
	      int body_len;
	      char *body;
	      gboolean before_cache_check = TRUE;

	      /*
	       * åͭǧޤǤν
	       * buffer->length >= lenˤʤ뤫̿ޤpolling
	       */
	      state = ghttp_process(request);

	      if (state == ghttp_not_done)
		ghttp_flush_response_buffer(request);
	      else if (state == ghttp_error)
		{
#if DEBUG_NETWORK
		  fprintf(stderr, "I/O error: %s\n", ghttp_get_error(request));
		  status_code = ghttp_status_code(request);
		  fprintf(stderr, "HTTP Response: %d (%s)\n",
			  status_code, ghttp_reason_phrase(request));
#endif
		  goto cache_is_dirty;	/* åäƤ */
		}

	      body = ghttp_get_body(request);
	      body_len = ghttp_get_body_len(request);

	      if (body_len > 0)
		{
		  ghttp_current_status current_status
		    = ghttp_get_status(request);

		  if (current_status.bytes_total > 0)
		    g_signal_emit(G_OBJECT(args->broker),
				  broker_signals[ACCESS_PROGRESSED_SIGNAL],
				  0,
				  buffer,
				  current_status.bytes_read,
				  current_status.bytes_total,
				  &signal_result);

		  if (content_encoding == UNKNOWN_ENCODING)
		    {
		      /* ˤ1٤ʤ */
		      content_encoding = get_content_encoding_type(request);

		      switch (content_encoding)
			{
			case GZIP_ENCODING:
			  gzip_buffer = gzip_buffer_new(buffer);
			  if (gzip_buffer == NULL)
			    {
			      error_message = _("Out of memory.");
			      goto error_exit;
			    }
			  break;

			case UNKNOWN_ENCODING:
			  goto cache_is_dirty;	/* ĩ */

			case STANDARD_ENCODING:
			  break;
			}
		    }

		  if (content_encoding == GZIP_ENCODING)
		    {
		      GzipBufferStatus result
			= gzip_buffer_append_data(gzip_buffer, body, body_len);
		      if (result == GZIP_BUFFER_ERROR)
			{
			  error_message = _("Inflation failed.");
			  gzip_buffer_free(gzip_buffer);
			  goto error_exit;
			}
		      if (result == GZIP_BUFFER_INFLATION_DONE)
			{
			  gzip_buffer_free(gzip_buffer);
			  gzip_buffer = NULL;
#if DEBUG_NETWORK
			  if (state != ghttp_done)
			    fprintf(stderr, "Requested file contains garbage?\n");
#endif
			  break;
			}
		    }
		  else if (!ochusha_async_buffer_append_data(buffer, body,
			      body_len,
			      "ochusha_network_broker.c: try_update_cache()"))
		    {	/* content_encoding == STANDARD_ENCODING */
		      g_signal_emit(G_OBJECT(args->broker),
				broker_signals[ACCESS_TERMINATED_SIGNAL],
				0,
				buffer,
				&signal_result);

		      status->state
			= OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
		      g_signal_emit(G_OBJECT(args->broker),
				    broker_signals[ACCESS_FAILED_SIGNAL],
				    0,
				    buffer,
				    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
				    _("Access terminated."),
				    &signal_result);

		      ghttp_request_destroy(request);
		      args->request = NULL;

		      goto finish_try_update_cache;
		    }

		  /* buffer->length >= lenˤʤä饭å */
		  if (before_cache_check && buffer->length >= len)
		    {
		      /*
		       * MEMO: ޤ褿顢consumeråɤȶĴ
		       *       ȻפäƤΤμȤߤconsumer
		       *       åɤ鸫ǽΤ ġĤȡ
		       *       դΤǡα
		       */
		      if (memcmp(cache_compare_buffer,
				 (char *)buffer->buffer + offset,
				 CACHE_COMPARE_SIZE) != 0)
			goto cache_is_dirty;	/* åäƤ */
		      else
			before_cache_check = FALSE;	/* åϿ */
		    }
		}

	      if (state == ghttp_done)
		break;

	      nanosleep(&polling_interval, NULL);
	    }

	  /* ̿Ǹޤ˽λˤ */
	  if (buffer->length < len)
	    goto cache_is_dirty;	/* ͽǡʤäΤ
					 * åäƤ
					 */

	  status_code = ghttp_status_code(request);
	  if (status_code == 206)
	    {	/* ʬžλ */
	      const char *last_modified
		= ghttp_get_header(request, http_hdr_Last_Modified);
	      if (last_modified != NULL)
		status->last_modified = G_STRDUP(last_modified);
	      else
		status->last_modified = NULL;
	      status->status_code = status_code;
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_COMPLETED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);
	      ghttp_request_destroy(request);
	      args->request = NULL;
	      write_buffer_to_cache(args->broker->config, url, buffer);
	      goto finish_try_update_cache;
	    }
	  /*
	   * HEADꥯľRangeդGETꥯȤ̿äƤߤ
	   * 206ʤä(ɤľƤ̵̽)
	   */
	  goto cache_is_dirty;
	}
      else
	{
	  /* HEADꥯȤη̤200ʤä */
	  if (status_code == 304)
	    {
	      /*
	       * MEMO: ѹʤ(If-Modified-Sinceդ)ʤΤ
	       *       åϿ
	       */
	      status->status_code = status_code;
	      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_COMPLETED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);
	      ghttp_request_destroy(request);
	      args->request = NULL;

	      goto finish_try_update_cache;
	    }

	  /*
	   * ƥå򤽤Τޤ޻Ȥ
	   */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  if (!ochusha_async_buffer_update_length(buffer, len,
			"ochusha_network_broker.c: try_update_cache()"))
	    {
	      /* ξϤɤ褦ʤ櫡ȤΥåɤ
	       * ɤ⤦ΥХåե˿ʤΤ֡
	       */
	    }

	  goto error_exit;
	}

    cache_is_dirty:
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY;
      ghttp_request_destroy(request);
      args->request = NULL;

      if (!ochusha_async_buffer_update_length(buffer, 0,
			"ochusha_network_broker.c: try_update_cache()"))
	{
	  g_signal_emit(G_OBJECT(args->broker),
			broker_signals[ACCESS_TERMINATED_SIGNAL],
			0,
			buffer,
			&signal_result);
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	  g_signal_emit(G_OBJECT(args->broker),
		broker_signals[ACCESS_FAILED_SIGNAL],
		0,
		buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."),
		&signal_result);

	  goto finish_try_update_cache;
	}
    }
  else
    {
      /* å夬ʤä */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;
    }

  /* å夬ʤääƤ */
  if (http_read_from_url(args->broker, buffer))
    {
      /* ̿ */
      if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS)
	status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;

      write_buffer_to_cache(args->broker->config, url, buffer);
    }
  /* 褿硢顼äȤƤ⥷ʥȯ */
  goto finish_try_update_cache;

 error_exit:
  if (error_message != NULL)
    {
      g_signal_emit(G_OBJECT(args->broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    error_message,
		    &signal_result);
    }
  else
    {
      status_code = ghttp_status_code(request);
      if (status_code != 0)
	{
	  char message[4096];
	  snprintf(message, 4096, "Access failed: %d (%s)",
		   status_code, ghttp_reason_phrase(request));
	  g_signal_emit(G_OBJECT(args->broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			message,
			&signal_result);
	}
      else
	g_signal_emit(G_OBJECT(args->broker),
		      broker_signals[ACCESS_FAILED_SIGNAL],
		      0,
		      buffer,
		      OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		      _("Access failed: unknown reason."),
		      &signal_result);
    }

  if (request != NULL)
    ghttp_request_destroy(request);
  args->request = NULL;

 finish_try_update_cache:
  ochusha_async_buffer_fix(buffer,
			   "ochusha_network_broker.c: try_update_cache");
  ochusha_async_buffer_active_unref(buffer,
				"ochusha_network_broker.c: try_update_cache");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));

}


static void
refresh_cache_after_read(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);

#if DEBUG_THREAD_MOST
  fprintf(stderr, "refresh_cache_after_read() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer,
		"ochusha_network_broker.c: refresh_cache_after_read()"))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(args->broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);
      goto finish_refresh_cache_after_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (http_read_from_url(args->broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
      /* ХåեƤ򥭥å˽񤭹ࡣ*/
      write_buffer_to_cache(args->broker->config, args->url, buffer);
    }
  else
    {
      /* ̿Ԥå夬(if_modified_sinceNULL)ʾ */
      if (read_cache_to_buffer(args->broker->config, args->url, buffer))
	{
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
	  if (status->status_code == 304)
	    {	/* å夬ä */
	      if (args->if_modified_since != NULL)
		status->last_modified = G_STRDUP(args->if_modified_since);
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_COMPLETED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);
	    }
	  else
	    {	/* ̿˼ԤΤǻʤåȤ */
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_FAILED_SIGNAL],
			    0,
			    buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			    _("Couldn't read from network, cache used."),
			    &signal_result);
	    }
	}
      else
	{	/* åɤʤ */
	  if (status->status_code == 304)
	    {
	      /* ٤å夬Ĥʤ */
	      g_signal_emit(G_OBJECT(args->broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_CACHE_NOT_FOUND,
			_("Couldn't find cache that should exist."),
			&signal_result);
	    }
	  else
	    {
	      /* ̤˼ */
	      g_signal_emit(G_OBJECT(args->broker),
			    broker_signals[ACCESS_FAILED_SIGNAL],
			    0,
			    buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			    _("Couldn't read data via the network."),
			    &signal_result);
	    }
	}
    }

  ochusha_async_buffer_active_unref(buffer,
		"ochusha_network_broker.c: refresh_cache_after_read()");

 finish_refresh_cache_after_read:
  ochusha_async_buffer_fix(buffer,
		"ochusha_network_broker.c: refresh_cache_after_read()");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
}


static void
force_read(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);

#if DEBUG_THREAD_MOST
  fprintf(stderr, "force_read() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer,
				"ochusha_network_broker.c: force_read()"))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(args->broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);
      goto finish_force_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (args->if_modified_since != NULL)
    {
      G_FREE(args->if_modified_since);
      args->if_modified_since = NULL;
    }

  if (http_read_from_url(args->broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s succeeded.\n", args->url);
#endif
    }
  else
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
#if DEBUG_NETWORK
      fprintf(stderr, "reading from %s failed.\n", args->url);
#endif
    }

  ochusha_async_buffer_active_unref(buffer,
				    "ochusha_network_broker.c: force_read()");

 finish_force_read:
  ochusha_async_buffer_fix(buffer, "ochusha_network_broker.c: force_read()");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
}


/*
 * ochusha_network_broker_read_from_url
 *
 * Ϳ줿URLǡɤ߹ࡣ
 * offline⡼ɤǥå夬¸ߤʤNULL֤
 *
 * modeOCHUSHA_NETWORK_BROKER_CACHE_IGNOREǤ硢ɬͥåȥ
 * ǡɤ߹ߡ̤򥭥å夷ʤ
 *
 * ʳξˤɤ߹߻˲餫ηǥå夬ФȤ
 * ˥ͥåȥɹǤˤϥǡ򥭥å¸롣
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_AS_ISξ硢å夬¸ߤˤϤ
 * Τޤ޻Ȥ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATEξ硢å夵Ƥǡ
 * append onlyǹΤǤȸʤå夵Ƥǡ
 * ǿǤκʬΤߤͥåȥɤ߹褦ĩ魯롣⤷append only
 * ǹǤʤäˤɤ߹ľ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESHξ硢å夬ͥå
 * 鴰˥ǡɤ߹ߡͥåȥɹԲǽǤä
 * ˤΤߡå夵ƤǡȤ
 *
 * ͥåȥ˥硢̿˼ԤǤĹ0ΥХåե
 * ֤Τա
 */
OchushaAsyncBuffer *
ochusha_network_broker_read_from_url(OchushaNetworkBroker *broker,
				     const char *url,
				     const char *if_modified_since,
				     OchushaNetworkBrokerCacheMode mode)
{
  gboolean signal_result;

  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && broker->config != NULL && url != NULL, NULL);
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "ochusha_network_broker_read_from_url: %s\n", url);
#endif

  if (mode == OCHUSHA_NETWORK_BROKER_CACHE_IGNORE && broker->config->offline)
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    NULL,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_NOT_CONNECTED,
		    _("Network not connected on offline mode."),
		    &signal_result);
      return NULL;
    }

  if (broker->config->offline
      || mode == OCHUSHA_NETWORK_BROKER_CACHE_AS_IS
      || mode == OCHUSHA_NETWORK_BROKER_CACHE_ONLY)
    {
      /* å夬ä餽򤽤Τޤ޻Ȥ */
      int fd = ochusha_config_cache_open_file(broker->config,
					      url, O_RDONLY);
      if (fd >= 0)
	{
	  OchushaAsyncBuffer *buffer = get_mmapped_buffer(fd);

	  if (buffer != NULL)
	    g_signal_emit(G_OBJECT(broker),
			  broker_signals[ACCESS_COMPLETED_SIGNAL],
			  0,
			  buffer,
			  &signal_result);
	  else
	    g_signal_emit(G_OBJECT(broker),
			  broker_signals[ACCESS_FAILED_SIGNAL],
			  0,
			  NULL,
			  OCHUSHA_NETWORK_BROKER_FAILURE_REASON_MMAP_FAILED,
			  _("Couldn't get mmapped buffer."),
			  &signal_result);
	  return buffer;
	}

      if (mode == OCHUSHA_NETWORK_BROKER_CACHE_ONLY)
	{
	  /* åʤĥå夷Ȥʤ */
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			NULL,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_CACHE_NOT_FOUND,
			_("Couldn't find cache that should exist."),
			&signal_result);
	  return NULL;
	}

      if (broker->config->offline)
	{
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			NULL,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_NOT_CONNECTED,
			_("Network not connected on offline mode."),
			&signal_result);
	  return NULL;
	}
    }

  switch (mode)
    {
    case OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATE:
      return ochusha_network_broker_employ_worker_thread(broker,
						url, if_modified_since,
						(JobFunc *)try_update_cache);

    case OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH:
    case OCHUSHA_NETWORK_BROKER_CACHE_AS_IS:
      return ochusha_network_broker_employ_worker_thread(broker,
					url, if_modified_since,
					(JobFunc *)refresh_cache_after_read);

    case OCHUSHA_NETWORK_BROKER_CACHE_IGNORE:
      return ochusha_network_broker_employ_worker_thread(broker,
						    url, NULL,
						    (JobFunc *)force_read);

    default:
      break;
    }

  /* ߥͭʤΤabort */
  abort();
}


/*
 * ochusha_network_broker_get_response_header:
 *
 * bufferHTTPˤ륢η̤ݻƤˡ
 * HTTP쥹ݥ󥹤λꤷإåƤ֤бإåʤ
 * HTTP쥹ݥ󥹥إåΤΤ̵ˤNULL֤
 *
 * ̾buffer̿åɤνλޤǥ쥹ݥ󥹥إåݻ
 * Ƥ롣
 */
const char *
ochusha_network_broker_get_response_header(OchushaNetworkBroker *broker,
					   OchushaAsyncBuffer *buffer,
					   const char *header)
{
  NetworkBrokerJobArgs *args;
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), NULL);

  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL || args->request == NULL)
    return NULL;

  return ghttp_get_header(args->request, header);
}


/*
 * ochusha_network_broker_get_header_names:
 *
 * bufferHTTPˤ륢η̤ݻƤˡ
 * HTTP쥹ݥ󥹤˴ޤޤƤΥإå̾ȡθĿͿ
 * 줿ݥ󥿤˳Ǽ롣
 * ˤ0֤顼ˤ-1֤
 *
 * ̾buffer̿åɤνλޤǥ쥹ݥ󥹥إåݻ
 * Ƥ롣
 *
 * ƤӽФheadersפˤʤäfreeǤ餦
 */
int
ochusha_network_broker_get_header_names(OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					char ***headers, int *num_headers)
{
  NetworkBrokerJobArgs *args;
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer)
		       && headers != NULL && num_headers != NULL, -1);
  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL || args->request == NULL)
    return -1;

  return ghttp_get_header_names(args->request, headers, num_headers);
}
