/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_board_2ch.c,v 1.15 2003/11/22 08:46:55 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_board_2ch.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include "htmlutils.h"
#include "utils.h"

#include <glib-object.h>
#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#if ENABLE_REGEXP
# if HAVE_ONIGURUMA
#  if HAVE_ONIG_ONIGPOSIX_H
#    include <onig/onigposix.h>
#  elif HAVE_ONIGPOSIX_H
#    include <onigposix.h>
#  else
#    error "Use of oniguruma without onig/onigposix.h header isn't considered."
#  endif
# else
#  if HAVE_POSIX_REGEX
#    if HAVE_SYS_TYPES_H
#      include <sys/types.h>
#    endif
#    if HAVE_REGEX_H
#      include <regex.h>
#    else
#      warning "I don't know suitable header file of your posix regex library."
#    endif
#  endif
# endif
#endif


static OchushaBulletinBoardClass *parent_class = NULL;


static void ochusha_board_2ch_class_init(OchushaBoard2chClass *klass);
static void ochusha_board_2ch_init(OchushaBoard2ch *board);
static void ochusha_board_2ch_finalize(GObject *object);

static void ochusha_board_2ch_read_boardlist_element(
						OchushaBulletinBoard *board,
						GHashTable *board_attributes);
static void ochusha_board_2ch_write_boardlist_element(
						OchushaBulletinBoard *board,
						FILE *boardlist_xml);

static const char *ochusha_board_2ch_get_subject_txt_encoding(
						OchushaBulletinBoard *board);
static const char *ochusha_board_2ch_get_subback_html_encoding(
						OchushaBulletinBoard *board);

static char *ochusha_board_2ch_generate_base_path(OchushaBulletinBoard *board,
						  const char *url);
static char *ochusha_board_2ch_generate_board_id(OchushaBulletinBoard *board,
						 const char *url);
static OchushaBBSThread *ochusha_board_2ch_thread_new(
						OchushaBulletinBoard *board,
						const char *id,
						const gchar *title);
static OchushaBBSThread *ochusha_board_2ch_lookup_thread_by_url(
						OchushaBulletinBoard *board,
						const char *url);
static OchushaAsyncBuffer *ochusha_board_2ch_get_threadlist_source(
					OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaNetworkBrokerCacheMode mode);
static gboolean ochusha_board_2ch_refresh_threadlist(
						OchushaBulletinBoard *board,
						OchushaAsyncBuffer *buffer,
						EachThreadCallback *cb,
						gpointer callback_data);

static const char *ochusha_board_2ch_get_response_character_encoding(
						OchushaBulletinBoard *board);
static iconv_helper *ochusha_board_2ch_get_response_iconv_helper(
						OchushaBulletinBoard *board);

GType
ochusha_board_2ch_get_type(void)
{
  static GType board_2ch_type = 0;

  if (board_2ch_type == 0)
    {
      static const GTypeInfo board_2ch_info =
	{
	  sizeof(OchushaBoard2chClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_board_2ch_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBoard2ch),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_board_2ch_init,
	};

      board_2ch_type = g_type_register_static(OCHUSHA_TYPE_BULLETIN_BOARD,
					      "OchushaBoard2ch",
					      &board_2ch_info, 0);
    }

  return board_2ch_type;
}


static void
ochusha_board_2ch_class_init(OchushaBoard2chClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  OchushaBulletinBoardClass *b_class = OCHUSHA_BULLETIN_BOARD_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_board_2ch_finalize;

  b_class->read_boardlist_element = ochusha_board_2ch_read_boardlist_element;
  b_class->write_boardlist_element = ochusha_board_2ch_write_boardlist_element;

  b_class->generate_base_path = ochusha_board_2ch_generate_base_path;
  b_class->generate_board_id = ochusha_board_2ch_generate_board_id;
  b_class->thread_new = ochusha_board_2ch_thread_new;
  b_class->lookup_thread_by_url = ochusha_board_2ch_lookup_thread_by_url;
  b_class->get_threadlist_source = ochusha_board_2ch_get_threadlist_source;
  b_class->refresh_threadlist = ochusha_board_2ch_refresh_threadlist;

  b_class->get_response_character_encoding
    = ochusha_board_2ch_get_response_character_encoding;
  b_class->get_response_iconv_helper
    = ochusha_board_2ch_get_response_iconv_helper;

  klass->get_read_cgi_url = NULL;
}


static void
ochusha_board_2ch_init(OchushaBoard2ch *board)
{
  OCHUSHA_BULLETIN_BOARD(board)->bbs_type = OCHUSHA_BBS_TYPE_2CH;

  board->read_cgi_url = NULL;
  board->last_modified = NULL;
  board->cookie = NULL;
}


static void
ochusha_board_2ch_finalize(GObject *object)
{
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(object);

  if (board_2ch->read_cgi_url != NULL)
    {
      G_FREE(board_2ch->read_cgi_url);
      board_2ch->read_cgi_url = NULL;
    }

  if (board_2ch->last_modified != NULL)
    {
      G_FREE(board_2ch->last_modified);
      board_2ch->last_modified = NULL;
    }

  if (board_2ch->cookie != NULL)
    {
      G_FREE(board_2ch->cookie);
      board_2ch->cookie = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
ochusha_board_2ch_read_boardlist_element(OchushaBulletinBoard *board,
					 GHashTable *board_attributes)
{
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);
  board_2ch->last_modified
    = ochusha_utils_get_attribute_string(board_attributes, "last_modified");
  board_2ch->cookie
    = ochusha_utils_get_attribute_string(board_attributes, "cookie");

  if (parent_class->read_boardlist_element != NULL)
    (*parent_class->read_boardlist_element)(board, board_attributes);
}


#define OUTPUT_BOARD_ATTRIBUTE_STRING(file, board, attribute)		\
  do {									\
    if ((board)->attribute != NULL)					\
      {									\
        gchar *text = g_markup_escape_text((board)->attribute, -1);	\
        fprintf(file,							\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <string>%s</string>\n"				\
	  "        </attribute>\n", text);				\
        g_free(text);							\
      }									\
  } while (0)


static void
ochusha_board_2ch_write_boardlist_element(OchushaBulletinBoard *board,
					  FILE *boardlist_xml)
{
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

  if (parent_class->write_boardlist_element != NULL)
    (*parent_class->write_boardlist_element)(board, boardlist_xml);

  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, board_2ch, last_modified);
  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, board_2ch, cookie);
}


static const char *
ochusha_board_2ch_get_subject_txt_encoding(OchushaBulletinBoard *board)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return "CP932";

    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static const char *
ochusha_board_2ch_get_subback_html_encoding(OchushaBulletinBoard *board)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      return "CP932";

    case OCHUSHA_BBS_TYPE_MITINOKU:
    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static char *
ochusha_board_2ch_generate_base_path(OchushaBulletinBoard *board,
				     const char *url)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      {
	char *path;
	char *abs_path = ochusha_utils_url_extract_http_absolute_path(url);
	char *tmp_pos;
	g_return_val_if_fail(abs_path != NULL, NULL);

	if (abs_path[0] != '/')
	  {
	    G_FREE(abs_path);
	    return NULL;
	  }
	tmp_pos = strchr(abs_path + 1, '/');

	if (tmp_pos != NULL)
	  {
	    tmp_pos[1] = '\0';
	    path = G_STRDUP(abs_path);
	  }
	else
	  {
	    path = G_STRDUP("/");
	  }
	G_FREE(abs_path);
	return path;
      }

    default:
      return G_STRDUP("/");
    }
}


static char *
ochusha_board_2ch_generate_board_id(OchushaBulletinBoard *board,
				    const char *url)
{
  char *id;
  char *abs_path = ochusha_utils_url_extract_http_absolute_path(url);
  if (abs_path != NULL && abs_path[0] != '/')
    {
      G_FREE(abs_path);
      abs_path = NULL;
    }
  g_return_val_if_fail(abs_path != NULL, NULL);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      {
	char *tmp_pos = strchr(abs_path + 1, '/');
	if (tmp_pos != NULL)
	  {
	    *tmp_pos = '\0';
	  }
	id = G_STRDUP(abs_path + 1);
	break;
      }

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      {
	char *tmp_pos = strchr(abs_path + 1, '/');
	if (tmp_pos == NULL)
	  id = G_STRDUP(abs_path + 1);
	else
	  {
	    char *id_tail = strchr(tmp_pos + 1, '/');
	    if (id_tail != NULL)
	      *id_tail = '\0';
	    id = G_STRDUP(tmp_pos + 1);
	  }
	break;
      }

    default:
      id = NULL;	/* ̤ݡ */
      break;
    }

  G_FREE(abs_path);

  return id;
}


static OchushaBBSThread *
ochusha_board_2ch_thread_new(OchushaBulletinBoard *board, const char *id,
			     const gchar *title)
{
  OchushaBBSThread *thread;
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && id != NULL, NULL);

  thread = ochusha_thread_2ch_new(OCHUSHA_BOARD_2CH(board), id, title);
  g_return_val_if_fail(thread != NULL, NULL);

  return thread;
}


static OchushaBBSThread *
ochusha_board_2ch_lookup_thread_by_url(OchushaBulletinBoard *board,
				       const char *url)
{
  OchushaBBSThread *thread = NULL;
  char *board_url = NULL;
  char *thread_id = NULL;

  if (ochusha_utils_2ch_check_url(url, &board_url, NULL, NULL, &thread_id,
				  NULL, NULL))
    {
      if (thread_id != NULL
	  && g_ascii_strcasecmp(board->base_url, board_url) == 0)
	thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								thread_id);
    }

  if (board_url != NULL)
    G_FREE(board_url);

  if (thread_id != NULL)
    G_FREE(thread_id);

  return thread;
}


static const char *
ochusha_board_2ch_get_response_character_encoding(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), NULL);

  switch (((OchushaBulletinBoard *)board)->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      return "CP932";

    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static iconv_helper *
ochusha_board_2ch_get_response_iconv_helper(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), NULL);

  switch (((OchushaBulletinBoard *)board)->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      return cp932_to_utf8_helper;

    case OCHUSHA_BBS_TYPE_MITINOKU:
    default:
      return NULL;
    }
}


OchushaBulletinBoard *
ochusha_board_2ch_new(const gchar *name, const char *base_url)
{
  g_assert(name != NULL && base_url != NULL);

  return OCHUSHA_BULLETIN_BOARD(g_object_new(OCHUSHA_TYPE_BOARD_2CH,
					     "name", name,
					     "base_url", base_url,
					     NULL));
}


const char *
ochusha_board_2ch_get_read_cgi_url(OchushaBoard2ch *board_2ch)
{
  OchushaBoard2chClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board_2ch), NULL);

  if (board_2ch->read_cgi_url != NULL)
    return board_2ch->read_cgi_url;

  klass = OCHUSHA_BOARD_2CH_GET_CLASS(board_2ch);
  if (klass->get_read_cgi_url != NULL)
    board_2ch->read_cgi_url = (klass->get_read_cgi_url)(board_2ch);
  else
    {
      OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(board_2ch);
      char url[PATH_MAX];

      switch (board->bbs_type)
	{
	case OCHUSHA_BBS_TYPE_2CH:
	case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
	  if (snprintf(url, PATH_MAX, "http://%s/test/read.cgi",
		       ochusha_bulletin_board_get_server(board)) < PATH_MAX)
	    board_2ch->read_cgi_url = G_STRDUP(url);
	  break;

	case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
	  if (snprintf(url, PATH_MAX, "http://%s%sbbs/read.cgi",
		       ochusha_bulletin_board_get_server(board),
		       ochusha_bulletin_board_get_base_path(board)) < PATH_MAX)
	    board_2ch->read_cgi_url = G_STRDUP(url);
	  break;

	case OCHUSHA_BBS_TYPE_JBBS:
	  if (snprintf(url, PATH_MAX, "http://%s/bbs/read.cgi",
		       ochusha_bulletin_board_get_server(board)) < PATH_MAX)
	    board_2ch->read_cgi_url = G_STRDUP(url);
	  break;

	case OCHUSHA_BBS_TYPE_MACHIBBS:
	  if (snprintf(url, PATH_MAX, "http://%s/bbs/read.pl",
		       ochusha_bulletin_board_get_server(board)) < PATH_MAX)
	    board_2ch->read_cgi_url = G_STRDUP(url);
	  break;

	case OCHUSHA_BBS_TYPE_MITINOKU:
	  if (snprintf(url, PATH_MAX, "http://%s/read.cgi",
		       ochusha_bulletin_board_get_server(board)) < PATH_MAX)
	    board_2ch->read_cgi_url = G_STRDUP(url);
	  break;

	default:
	  return NULL;
	}

    }

  return board_2ch->read_cgi_url;
}


const char *
ochusha_board_2ch_get_cookie(OchushaBoard2ch *board)
{
  return board->cookie;
}


void
ochusha_board_2ch_set_cookie(OchushaBoard2ch *board, const char *cookie)
{
  if (board->cookie != NULL)
    G_FREE(board->cookie);
  if (cookie != NULL)
    board->cookie = G_STRDUP(cookie);
  else
    board->cookie = NULL;
}


static OchushaAsyncBuffer *
ochusha_board_2ch_get_threadlist_source(OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaNetworkBrokerCacheMode mode)
{
  OchushaBoard2ch *board_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), NULL);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (snprintf(url, PATH_MAX, "%s" OCHUSHA_SUBBACK_HTML, board->base_url)
	  >= PATH_MAX)
	return NULL;
      return ochusha_network_broker_read_from_cgi(broker, url);

    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    default:
      if (snprintf(url, PATH_MAX, "%s" OCHUSHA_SUBJECT_TXT, board->base_url)
	  >= PATH_MAX)
	return NULL;
    }

  board_2ch = OCHUSHA_BOARD_2CH(board);

  return ochusha_network_broker_read_from_url(broker, url,
					      board_2ch->last_modified, mode);
}


static void
unmark_alive(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread = OCHUSHA_THREAD_2CH(data);
  thread->alive = FALSE;
}


static void
collect_dropped_thread(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(data);
  if (!thread_2ch->alive)
    {
      OchushaBBSThread *thread = (OchushaBBSThread *)thread_2ch;
      /* threadDATäݤ*/
      thread->board->dropped_list
	= g_slist_append(thread->board->dropped_list, thread);
    }
  else
    g_object_unref(G_OBJECT(data));	/* Ƥ륹thread_list
					 * Ѥ;פrefƤ롣
					 */
}


static void
undo_thread_ref(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(data);
  if (thread_2ch->alive)
    g_object_unref(G_OBJECT(data));	/* Ƥ륹thread_list
					 * Ѥ;פrefƤ롣
					 */
}


/*
 * refresh_threadlist_by_subject_txt:
 *
 * Ϳ줿ХåեƤ2chĤsubject.txtȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
refresh_threadlist_by_subject_txt(OchushaBulletinBoard *board,
				     OchushaAsyncBuffer *buffer,
				     EachThreadCallback *each_thread_cb,
				     gpointer callback_data)
{
  char default_buffer[PATH_MAX];
  char scan_buffer[5];
  gboolean result;
  GSList *thread_list = NULL;
  GSList *old_thread_list = board->thread_list;
  iconv_t converter;
  iconv_helper *converter_helper;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_data(G_OBJECT(buffer),
			"OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && status != NULL, FALSE);

  converter = iconv_open("UTF-8//IGNORE",
			 ochusha_board_2ch_get_subject_txt_encoding(board));
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
      converter_helper = cp932_to_utf8_helper;
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
    default:
      converter_helper = NULL;
    }

  if (!ochusha_async_buffer_active_ref(buffer, "ochusha_board_2ch.c: ochusha_board_2ch_refresh_threadlist"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      return FALSE;
    }

  g_slist_foreach(old_thread_list, unmark_alive, NULL);

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int rest_of_data = buffer->length - offset;
	char *eol_pos;

	while (rest_of_data > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    char *thread_id;
	    gchar *thread_title;
	    OchushaBBSThread *thread;
	    OchushaThread2ch *thread_2ch;
	    char *title_pos;
	    int n_responses_on_server = 0;
	    char *tmp_pos;
	    int title_len;

	    /*
	     * XXX: 롼פǤʤȤΤϤ礤»
	     */
	    switch (board->bbs_type)
	      {
	      case OCHUSHA_BBS_TYPE_2CH:
	      case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
	      case OCHUSHA_BBS_TYPE_MITINOKU:
		tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
		break;

	      case OCHUSHA_BBS_TYPE_JBBS:
	      case OCHUSHA_BBS_TYPE_MACHIBBS:
	      case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
		tmp_pos = memchr(cur_pos, ',', eol_pos - cur_pos);
		break;

	      default:
		tmp_pos = NULL;
		break;
	      }

	    if (tmp_pos == NULL)
	      {
#if DEBUG_LIBOCHUSHA
		fprintf(stderr, "Unknown format found in subject.txt.\n");
#endif
		goto prepare_next_line;
	      }
	    if ((tmp_pos - cur_pos) < PATH_MAX)
	      {
		thread_id = memcpy(default_buffer, cur_pos, tmp_pos - cur_pos);
		thread_id[tmp_pos - cur_pos] = '\0';
	      }
	    else
	      thread_id = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

	    /*
	     * XXX: 롼פǤʤȤΤϤ礤»
	     */
	    switch (board->bbs_type)
	      {
	      case OCHUSHA_BBS_TYPE_2CH:
	      case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
	      case OCHUSHA_BBS_TYPE_MITINOKU:
		{
		  char *ext;
		  if ((ext = strstr(thread_id, ".dat")) != NULL)
		    *ext = '\0';
		  title_pos = tmp_pos + 2;	/* skip "<>" */
		  break;
		}

	      case OCHUSHA_BBS_TYPE_JBBS:
	      case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
	      case OCHUSHA_BBS_TYPE_MACHIBBS:
		{
		  char *ext;
		  if ((ext = strstr(thread_id, ".cgi")) != NULL)
		    *ext = '\0';
		  title_pos = tmp_pos + 1;	/* skip "," */
		  break;
		}

	      default:
		title_pos = NULL;
		abort();
	      }

	    title_len = eol_pos - title_pos;

	    for (tmp_pos = eol_pos - 1; tmp_pos > title_pos; tmp_pos--)
	      if (*tmp_pos == '(')
		break;

	    if (*tmp_pos == '(')
	      {
		char *close_paren = memchr(tmp_pos + 1, ')',
					   eol_pos - tmp_pos - 1);
		if (close_paren != NULL)
		  {
		    int tmp_len = close_paren - tmp_pos - 1;
		    if (tmp_len > 0 && tmp_len <= 4)
		      {
			memcpy(scan_buffer, tmp_pos + 1, tmp_len);
			scan_buffer[tmp_len] = '\0';
			sscanf(scan_buffer, "%d", &n_responses_on_server);
			title_len -= (tmp_len + 2);
		      }
		  }
	      }

	    thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								    thread_id);
	    if (thread == NULL)
	      {
		thread_title = simple_string_canon(title_pos, title_len,
						   converter,
						   converter_helper);
		thread = ochusha_bulletin_board_bbs_thread_new(board,
							       thread_id,
							       thread_title);
		G_FREE(thread_title);
	      }

	    if (thread_id != default_buffer)
	      G_FREE(thread_id);

	    thread_2ch = OCHUSHA_THREAD_2CH(thread);

	    if (!thread_2ch->alive)
	      {
		thread_2ch->alive = TRUE;

		thread->number_of_responses_on_server = n_responses_on_server;

		g_object_ref(G_OBJECT(thread));
		thread_list = g_slist_append(thread_list, thread);

		if (each_thread_cb != NULL
		    && !(*each_thread_cb)(thread, callback_data))
		  {
		    result = FALSE;
		    break;
		  }
	      }

	  prepare_next_line:
	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;
	  }

	if (buffer->fixed)
	  {
	    result = TRUE;
	    break;
	  }

	if (!ochusha_async_buffer_wait(buffer, "ochusha_board_2ch.c: ochusha_board_2ch_refresh_threadlist"))
	  {
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "buffer has been terminated.\n");
#endif
	    result = FALSE;
	    break;
	  }

	if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    result = FALSE;
	    break;
	  }
      }
  }
  ochusha_async_buffer_unlock(buffer);

  if (result && status->last_modified != NULL)
    {
      OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

      if (board_2ch->last_modified != NULL)
	G_FREE(board_2ch->last_modified);
      board_2ch->last_modified = G_STRDUP(status->last_modified);
    }

  ochusha_async_buffer_active_unref(buffer, "ochusha_board_2ch.c: ochusha_board_2ch_refresh_threadlist");

  if (result)
    {
      g_slist_foreach(old_thread_list, collect_dropped_thread, NULL);
      g_slist_free(old_thread_list);
      board->thread_list = thread_list;
    }
  else
    {
      /* 顼ϸŤΤͥ */
      g_slist_foreach(old_thread_list, undo_thread_ref, NULL);
      g_slist_free(thread_list);
    }

  iconv_close(converter);

  return result;
}


#if defined(ENABLE_REGEXP) && defined(HAVE_ONIGURUMA)
/*
 * refresh_threadlist_by_subback_html:
 *
 * Ĥsubback.htmlȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
refresh_threadlist_by_subback_html(OchushaBulletinBoard *board,
				   OchushaAsyncBuffer *buffer,
				   EachThreadCallback *each_thread_cb,
				   gpointer callback_data)
{
  gboolean result;
  GSList *thread_list = NULL;
  GSList *old_thread_list = board->thread_list;
  iconv_t converter;
  iconv_helper *converter_helper;
  regex_t threadlist_entry_pattern;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_data(G_OBJECT(buffer),
			"OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && status != NULL, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      converter_helper = NULL;
      if (regcomp(&threadlist_entry_pattern,
		  ".*</a>:<a href=.*key=([1-9][0-9]*)[^>]*>(.*)&lt;([1-9][0-9]*)&gt;</a><br>$", REG_EXTENDED))
	{
	  fprintf(stderr, "invalid regular expression\n");
	  return FALSE;
	}
      break;

    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
    default:
      converter_helper = cp932_to_utf8_helper;
      return FALSE;	/* ̤ݡ */
    }

  converter = iconv_open("UTF-8//IGNORE",
			 ochusha_board_2ch_get_subback_html_encoding(board));
  if (converter == (iconv_t)-1)
    regfree(&threadlist_entry_pattern);
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer, "ochusha_board_2ch.c: ochusha_board_2ch_refresh_threadlist"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      regfree(&threadlist_entry_pattern);
      return FALSE;
    }

  g_slist_foreach(old_thread_list, unmark_alive, NULL);

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int rest_of_data = buffer->length - offset;
	char *eol_pos;

	while (rest_of_data > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    char *thread_id;
	    gchar *thread_title;
	    OchushaBBSThread *thread;
	    OchushaThread2ch *thread_2ch;
	    gchar *html_line;
	    int n_responses_on_server = 0;
	    regmatch_t match[4];

	    *eol_pos = '\0';
	    html_line = convert_string(converter, converter_helper,
				       cur_pos, -1);
	    *eol_pos = '\n';	/* ̣ʤ̣ġ */

	    if (html_line == NULL
		|| regexec(&threadlist_entry_pattern,
			   html_line, 3, match, 0) != 0)
	      {
#if 0
		fprintf(stderr, "html_line: \"%s\"\n", html_line);
#endif
		goto prepare_next_line;
	      }

	    thread_id = html_line + match[1].rm_so;
	    thread_id[match[1].rm_eo - match[1].rm_so] = '\0';

	    thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								    thread_id);
	    if (thread == NULL)
	      {
		thread_title
		  = simple_string_canon(html_line + match[2].rm_so,
					match[2].rm_eo - match[2].rm_so,
					NULL, NULL);
		thread = ochusha_bulletin_board_bbs_thread_new(board,
							       thread_id,
							       thread_title);
		G_FREE(thread_title);
	      }

	    html_line[match[3].rm_eo] = '\0';
	    sscanf(html_line + match[3].rm_so, "%d", &n_responses_on_server);

	    thread_2ch = OCHUSHA_THREAD_2CH(thread);

	    if (!thread_2ch->alive)
	      {
		thread_2ch->alive = TRUE;

		thread->number_of_responses_on_server = n_responses_on_server;

		g_object_ref(G_OBJECT(thread));
		thread_list = g_slist_append(thread_list, thread);

		if (each_thread_cb != NULL
		    && !(*each_thread_cb)(thread, callback_data))
		  {
		    result = FALSE;
		    break;
		  }
	      }

	  prepare_next_line:
	    if (html_line != NULL)
	      G_FREE(html_line);
	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;
	  }

	if (buffer->fixed)
	  {
	    result = TRUE;
	    break;
	  }

	if (!ochusha_async_buffer_wait(buffer, "ochusha_board_2ch.c: ochusha_board_2ch_refresh_threadlist"))
	  {
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "buffer has been terminated.\n");
#endif
	    result = FALSE;
	    break;
	  }

	if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    result = FALSE;
	    break;
	  }
      }
  }
  ochusha_async_buffer_unlock(buffer);

  if (result && status->last_modified != NULL)
    {
      OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

      if (board_2ch->last_modified != NULL)
	G_FREE(board_2ch->last_modified);
      board_2ch->last_modified = G_STRDUP(status->last_modified);
    }

  ochusha_async_buffer_active_unref(buffer, "ochusha_board_2ch.c: ochusha_board_2ch_refresh_threadlist");

  if (result)
    {
      g_slist_foreach(old_thread_list, collect_dropped_thread, NULL);
      g_slist_free(old_thread_list);
      board->thread_list = thread_list;
    }
  else
    {
      /* 顼ϸŤΤͥ */
      g_slist_foreach(old_thread_list, undo_thread_ref, NULL);
      g_slist_free(thread_list);
    }

  regfree(&threadlist_entry_pattern);
  iconv_close(converter);

  return result;
}
#endif


/*
 * ochusha_board_2ch_refresh_threadlist:
 *
 * Ϳ줿ХåեƤ2chĤsubject.txtȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
ochusha_board_2ch_refresh_threadlist(OchushaBulletinBoard *board,
				     OchushaAsyncBuffer *buffer,
				     EachThreadCallback *each_thread_cb,
				     gpointer callback_data)
{
#if defined(ENABLE_REGEXP) && defined(HAVE_ONIGURUMA)
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return refresh_threadlist_by_subback_html(board, buffer,
						each_thread_cb, callback_data);

    default:
      return refresh_threadlist_by_subject_txt(board, buffer,
					       each_thread_cb, callback_data);
    }
#else
  return refresh_threadlist_by_subject_txt(board, buffer,
					   each_thread_cb, callback_data);
#endif
}
