/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: bulletin_board_ui.c,v 1.65 2003/12/05 12:42:51 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "ochusha_ui.h"

#include "bbs_thread_ui.h"
#include "bbs_thread_view.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "bulletin_board_ui.h"
#include "icon_label.h"
#include "text_search_window.h"
#include "threadlist_filter.h"
#include "threadlist_view.h"
#include "paned_notebook.h"
#include "response_editor.h"
#include "ugly_gtk2utils.h"

#include "worker.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <iconv.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static void open_thread_view(WorkerThread *employee, gpointer job_args);
static void show_threadlist(WorkerThread *employee, gpointer args);
static void advance_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
			    PanedNotebook *paned_notebook);
static void back_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
			 PanedNotebook *paned_notebook);
static void write_response_cb(ThreadlistView *view, OchushaBBSThread *thread,
			      OchushaApplication *application);
static void close_threadlist_cb(ThreadlistView *view,
			      OchushaApplication *application);
static void toggle_mark_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   OchushaApplication *application);
static void toggle_hide_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   OchushaApplication *application);
static void reset_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
			    OchushaApplication *application);
static void copy_thread_url_cb(ThreadlistView *view, OchushaBBSThread *thread,
			       OchushaApplication *application);
static void mark_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   gboolean do_mark, OchushaApplication *application);
static void thread_mouse_over_cb(ThreadlistView *view,
				 OchushaBBSThread *thread,
				 OchushaApplication *application);
static void thread_mouse_out_cb(ThreadlistView *view, OchushaBBSThread *thread,
				OchushaApplication *application);
static gboolean thread_title_popup_timeout(gpointer data);
static void thread_title_popup_show();
static gboolean thread_title_popup_paint_window(GtkWidget *widget);

static void item_view_required_cb(PanedNotebook *paned_notebook,
				  OchushaBBSThread *thread,
				  GtkWidget **item_view,
				  const gchar **title, GtkWidget **tab_label,
				  OchushaApplication *application);
static void item_view_being_closed_cb(PanedNotebook *paned_notebook,
				      OchushaBBSThread *thread,
				      GtkWidget *item_view,
				      OchushaApplication *application);
static void item_view_closed_cb(PanedNotebook *paned_notebook,
				OchushaBBSThread *thread, GtkWidget *item_view,
				OchushaApplication *application);
static void page_switched_cb(PanedNotebook *paned_notebook,
			     GtkWidget *previous_page, GtkWidget *new_page,
			     OchushaApplication *application);

static void get_current_thread(OchushaApplication *application,
			       GtkWidget **view, OchushaBBSThread **thread,
			       IconLabel **tab_label);
static void write_response_button_cb(GtkWidget *widget,
				     OchushaApplication *application);
static void select_font_button_cb(GtkWidget *widget,
				  OchushaApplication *application);
static void refresh_thread_button_cb(GtkWidget *widget,
				     OchushaApplication *application);
static void start_thread_search_button_cb(GtkWidget *widget,
					  OchushaApplication *application);
static void jump_to_new_comer_button_cb(GtkWidget *widget,
					OchushaApplication *application);
static void go_to_bottom_button_cb(GtkWidget *widget,
				   OchushaApplication *application);
static void go_to_top_button_cb(GtkWidget *widget,
				OchushaApplication *application);
static GtkWidget *create_write_dialog(OchushaApplication *application,
				      OchushaBBSThread *thread);


static pthread_mutex_t board_info_lock;
static pthread_mutex_t thread_list_lock;
static pthread_cond_t thread_list_condition;

#define BOARD_INFO_LOCK					\
  if (pthread_mutex_lock(&board_info_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define BOARD_INFO_UNLOCK				\
  if (pthread_mutex_unlock(&board_info_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_LOCK				\
  if (pthread_mutex_lock(&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_UNLOCK				\
  if (pthread_mutex_unlock(&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_COND_WAIT				\
  if (pthread_cond_wait(&thread_list_condition,		\
			&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't wait a condition.\n");	\
      abort();						\
    }

#define THREAD_LIST_COND_BROADCAST				\
  if (pthread_cond_broadcast(&thread_list_condition) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }


static OchushaBBSThread *popup_thread = NULL;
static GtkWidget *title_popup_window = NULL;
static GtkWidget *title_popup_label = NULL;
static gint title_popup_x_pos = 0;
static gint title_popup_y_pos = 0;
static guint title_popup_delay_id = 0;


static GQuark board_info_id;
static GQuark application_id;


void
prepare_board_ui_initialization(OchushaApplication *application)
{
 if (pthread_mutex_init(&board_info_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

 if (pthread_mutex_init(&thread_list_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&thread_list_condition, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }

  board_info_id = g_quark_from_static_string("BulletinBoardUI::BoardInfo");
  application_id = g_quark_from_static_string("BulletinBoardUI::Application");
}


void
initialize_board_ui(OchushaApplication *application)
{
  setup_default_threadlist_filter(application);
  initialize_board_properties(application);

  /* ݥåץåפν */
  title_popup_window = gtk_window_new(GTK_WINDOW_POPUP);
  gtk_widget_set_app_paintable(title_popup_window, TRUE);
  gtk_window_set_resizable(GTK_WINDOW(title_popup_window), FALSE);
  gtk_widget_set_name(title_popup_window, "gtk-tooltips");
  gtk_container_set_border_width(GTK_CONTAINER(title_popup_window), 4);

  g_signal_connect(title_popup_window, "expose_event",
		   G_CALLBACK(thread_title_popup_paint_window), NULL);

  title_popup_label = gtk_label_new(NULL);
  gtk_label_set_line_wrap(GTK_LABEL(title_popup_label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(title_popup_label), 0.5, 0.5);
  gtk_widget_show(title_popup_label);

  gtk_container_add(GTK_CONTAINER(title_popup_window), title_popup_label);

  application->thread_search_window = NULL;
}


static void
threadlist_read_thread_element_cb(OchushaBulletinBoard *board,
				  OchushaBBSThread *thread,
				  GHashTable *hash_table,
				  gpointer user_data)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  info->view_flags = ochusha_utils_get_attribute_int(hash_table, "view_flags");
  info->view_rank = ochusha_utils_get_attribute_int(hash_table, "view_rank");
  info->show_mailto_mode
    = (ThreadViewMailtoMode)ochusha_utils_get_attribute_int(hash_table,
							   "show_mailto_mode");
  if (info->show_mailto_mode < THREAD_VIEW_MAILTO_MODE_DEFAULT
      && info->show_mailto_mode > THREAD_VIEW_MAILTO_MODE_HIDE)
    info->show_mailto_mode = THREAD_VIEW_MAILTO_MODE_DEFAULT;

  info->view_ignored = ochusha_utils_get_attribute_int(hash_table,
						       "view_ignored");
  info->last_name = ochusha_utils_get_attribute_string(hash_table,
						       "last_name");
  info->last_mail = ochusha_utils_get_attribute_string(hash_table,
						       "last_mail");
  if ((info->view_flags & BBS_THREAD_HIDDEN) != 0)
    thread->number_of_responses_read = 0;

  return;
}


#define OUTPUT_THREAD_ATTRIBUTE_INT(file, thread, attribute)		\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <int val=\"%d\"/>\n"					\
	  "      </attribute>\n", (thread)->attribute)

#define OUTPUT_THREAD_ATTRIBUTE_STRING(file, thread, attribute)		\
  do {									\
    if ((thread)->attribute != NULL)					\
      {									\
	gchar *text = g_markup_escape_text((thread)->attribute, -1);	\
	fprintf(file,							\
		"      <attribute name=\"" #attribute	"\">\n"		\
		"        <string>%s</string>\n"				\
		"      </attribute>\n", text);				\
	g_free(text);							\
      }									\
  } while (0)



static void
threadlist_write_thread_element_cb(OchushaBulletinBoard *board,
				   OchushaBBSThread *thread,
				   FILE *threadlist_xml,
				   OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (info->view_rank != 0)
    {
      info->view_flags &= ~BBS_THREAD_NEW;
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info,
				  view_flags);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info, view_rank);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info, show_mailto_mode);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info, view_ignored);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info, last_name);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info, last_mail);
    }

  if ((info->view_flags & BBS_THREAD_HIDDEN) != 0)
    {
      thread->number_of_responses_read = 0;
      ochusha_bbs_thread_remove_cache(thread, &application->config);
    }
}


static void
bulletin_board_gui_info_free(BulletinBoardGUIInfo *info)
{
  if (info->filter.rule != NULL)
    {
      G_FREE(info->filter.rule);
      info->filter.rule = NULL;
    }

  if (info->properties_dialog != NULL)
    gtk_widget_destroy(info->properties_dialog);

  if (info->last_name != NULL)
    {
      G_FREE(info->last_name);
      info->last_name = NULL;
    }

  if (info->last_mail != NULL)
    {
      G_FREE(info->last_mail);
      info->last_mail = NULL;
    }

  G_FREE(info);
}


BulletinBoardGUIInfo *
ensure_bulletin_board_info(OchushaBulletinBoard *board,
			   OchushaApplication *application)
{
  BulletinBoardGUIInfo *info;
  BOARD_INFO_LOCK
  {
    info = g_object_get_qdata(G_OBJECT(board), board_info_id);
    if (info == NULL)
      {
	info = G_NEW0(BulletinBoardGUIInfo, 1);
	g_object_set_qdata_full(G_OBJECT(board), board_info_id, info,
				(GDestroyNotify)bulletin_board_gui_info_free);
	g_signal_connect(G_OBJECT(board), "threadlist_read_thread_element",
			 G_CALLBACK(threadlist_read_thread_element_cb), NULL);
	g_signal_connect(G_OBJECT(board), "threadlist_write_thread_element",
			 G_CALLBACK(threadlist_write_thread_element_cb),
			 application);
      }
  }
  BOARD_INFO_UNLOCK;

  return info;
}


typedef struct _ThreadlistJobArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;
  ThreadlistView *view;
  IconLabel *tab_label;
  gboolean increment_ignored;
} ThreadlistJobArgs;


static gboolean
interactive_search_cb(ThreadlistView *view,
		      ThreadlistSearchAction search_action,
		      OchushaApplication *application)
{
  if (search_action != THREADLIST_SEARCH_ACTION_TERMINATE)
    advance_threadlist_search(application, view, search_action);
  else
    terminate_threadlist_search(application, view);

  return TRUE;
}


static gboolean
title_match_func(const gchar *title, ThreadlistSearchQueryData *search_data)
{
  gboolean result;
  gchar *normalized_title;

#if DEBUG_SEARCH
  char *native_string;
  if (title != NULL)
    {
      native_string = convert_string(utf8_to_native, title, -1);
      fprintf(stderr, "title_match_func: title=\"%s\"\n", native_string);
      G_FREE(native_string);
    }
  if (search_data->key != NULL)
    {
      native_string = convert_string(utf8_to_native, search_data->key, -1);
      fprintf(stderr, "title_match_func: key=\"%s\"\n", native_string);
      G_FREE(native_string);
    }
#endif
  if (search_data->key == NULL)
    return FALSE;

  /* match_case˹碌normalizeƤ롣*/
  normalized_title = g_utf8_normalize(title, -1, G_NORMALIZE_ALL);
#if TRACE_MEMORY_USAGE
  {
    gchar *tmp_title = normalized_title;
    normalized_title = G_STRDUP(tmp_title);
    g_free(tmp_title);
  }
#endif
  if (!search_data->match_case)
    {
      gchar *case_normalized_title = g_utf8_casefold(normalized_title, -1);
      G_FREE(normalized_title);
#if TRACE_MEMORY_USAGE
      normalized_title = G_STRDUP(case_normalized_title);
      g_free(case_normalized_title);
#else
      normalized_title = case_normalized_title;
#endif
    }

#if ENABLE_REGEXP
  if (search_data->use_regexp)
    {
      if (search_data->regexp_available)
	result = (regexec(&search_data->regexp, normalized_title,
			  0, NULL, 0) == 0);
      else
	result = TRUE;	/* ̵̤ʸ򤵤ʤ */
    }
  else
    result = (strstr(normalized_title, search_data->key) != NULL);
#else
  result = (strstr(normalized_title, search_data->key) != NULL);
#endif

  G_FREE(normalized_title);

  return result;
}


static void
customize_threadlist_view_contents(OchushaApplication *application,
				   ThreadlistView *view)
{
  const char *column = application->threadlist_view_contents;
  if (column == NULL)
    return;

  g_return_if_fail(strlen(column) > 0);

  while (*column != '\0')
    {
      switch (*column)
	{
	case 'R':
	  threadlist_view_append_rank(view);
	  break;

	case 'M':
	  threadlist_view_append_mark(view);
	  break;

	case 'T':
	  threadlist_view_append_title(view);
	  break;

	case 'N':
	  threadlist_view_append_number_of_responses(view);
	  break;

	case 'n':
	  threadlist_view_append_number_of_responses_read(view);
	  break;

	case 'U':
	  threadlist_view_append_number_of_responses_unread(view);
	  break;

	case 'D':
	  threadlist_view_append_rank_difference(view);
	  break;

	case 'L':
	  threadlist_view_append_last_modified(view);
	  break;

	default:
	  fprintf(stderr, "'%c' doesn't match any column\n", *column);
	  break;
	}
      column++;
    }
}


static void
open_thread_view_cb(GtkWidget *selector, OchushaBBSThread *thread,
		    gboolean in_tab, gboolean with_browser,
		    PanedNotebook *paned_notebook)
{
  if (!with_browser)
    paned_notebook_open_item_view(paned_notebook, thread, in_tab);
  else
    {
      OchushaApplication *application
	= g_object_get_qdata(G_OBJECT(selector), application_id);
      if (application != NULL)
	ochusha_open_url(application,
			 ochusha_bbs_thread_get_url_to_post_response(thread),
			 in_tab, with_browser);
#if 0	/* Թͽ */
      else
	fprintf(stderr, "open_thread_view_cb: %s\n",
		ochusha_bbs_thread_get_url_to_post_response(thread));
#endif
    }
}


GtkWidget *
open_bulletin_board(OchushaApplication *application,
		    OchushaBulletinBoard *board, IconLabel *tab_label)
{
  GtkWidget *paned_notebook;

  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  if (info == NULL)
    {
      fprintf(stderr, "Out of memory.\n");
      return NULL;
    }

  THREAD_LIST_LOCK
  {
    WorkerJob *job;
    ThreadlistJobArgs *job_args;
    GtkWidget *scrolled_window;
    GtkWidget *view;
    GtkToolbar *toolbar;

    if (info->is_busy)
      {
#if DEBUG_THREAD
	fprintf(stderr, "board is busy.\n");
#endif
	THREAD_LIST_UNLOCK;
	return NULL;
      }

    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				   GTK_POLICY_AUTOMATIC,
				   GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window),
					GTK_SHADOW_ETCHED_IN);
    view = threadlist_view_new();
    g_object_set_qdata(G_OBJECT(view), application_id, application);
    customize_threadlist_view_contents(application, THREADLIST_VIEW(view));
    threadlist_view_set_default_open_in_tab(THREADLIST_VIEW(view),
					    application->default_open_in_tab);

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(ThreadlistJobArgs, 1);

    gtk_widget_show(view);
    gtk_container_add(GTK_CONTAINER(scrolled_window), view);
    gtk_widget_show(scrolled_window);

    paned_notebook
      = paned_notebook_new_with_selector(application->threadlist_pane_style,
					 scrolled_window);
    g_signal_connect(G_OBJECT(view), "open_thread_view",
		     G_CALLBACK(open_thread_view_cb), paned_notebook);
    gtk_paned_set_position(GTK_PANED(paned_notebook),
			   application->threadlist_height);
    paned_notebook_set_tab_shrinkable(PANED_NOTEBOOK(paned_notebook),
				      application->thread_tab_shrinkable);
    paned_notebook_set_minimum_tab_label_size(PANED_NOTEBOOK(paned_notebook),
					application->thread_tab_minimum_size);
    paned_notebook_set_tooltips(PANED_NOTEBOOK(paned_notebook),
				application->thread_tab_enable_tooltips);
    paned_notebook_set_tab_policy(PANED_NOTEBOOK(paned_notebook),
				  application->thread_tab_always_show
				  ? GTK_POLICY_ALWAYS
				  : GTK_POLICY_AUTOMATIC);

    toolbar = paned_notebook_get_toolbar(PANED_NOTEBOOK(paned_notebook));

    gtk_toolbar_prepend_space(toolbar);
    gtk_toolbar_insert_stock(toolbar, OCHUSHA_STOCK_WRITE_RESPONSE,
			     _("Write a response to the current thread"),
			     "write_response_to_current_thread",
			     GTK_SIGNAL_FUNC(write_response_button_cb),
			     application,
			     0);

    gtk_toolbar_prepend_space(toolbar);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_SELECT_FONT,
			     _("Select font to render threads"),
			     "select_font_to_render_threads",
			     GTK_SIGNAL_FUNC(select_font_button_cb),
			     application,
			     0);

    gtk_toolbar_prepend_space(toolbar);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_REFRESH,
			     _("Refresh current thread"),
			     "refresh_current_thread",
			     GTK_SIGNAL_FUNC(refresh_thread_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_FIND,
			     _("Start text search on current thread"),
			     "start_text_search_on_thread",
			     GTK_SIGNAL_FUNC(start_thread_search_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_JUMP_TO,
			     _("Jump to new comer response"),
			     "jump_to_new_comer_response",
			     GTK_SIGNAL_FUNC(jump_to_new_comer_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GOTO_BOTTOM,
			     _("Go to bottom of the response view"),
			     "go_to_bottom_of_response",
			     GTK_SIGNAL_FUNC(go_to_bottom_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GOTO_TOP,
			     _("Go to top of the response view"),
			     "go_to_top_of_response",
			     GTK_SIGNAL_FUNC(go_to_top_button_cb),
			     application,
			     0);

    gtk_widget_show(paned_notebook);

    g_signal_connect(G_OBJECT(view), "advance_view",
		     G_CALLBACK(advance_view_cb), paned_notebook);
    g_signal_connect(G_OBJECT(view), "back_view",
		     G_CALLBACK(back_view_cb), paned_notebook);
    g_signal_connect(G_OBJECT(view), "toggle_mark",
		     G_CALLBACK(toggle_mark_cb), application);
    g_signal_connect(G_OBJECT(view), "write_response",
		     G_CALLBACK(write_response_cb), application);
    g_signal_connect(G_OBJECT(view), "close_threadlist",
		     G_CALLBACK(close_threadlist_cb), application);
    g_signal_connect(G_OBJECT(view), "toggle_hide",
		     G_CALLBACK(toggle_hide_cb), application);
    g_signal_connect(G_OBJECT(view), "reset_thread",
		     G_CALLBACK(reset_thread_cb), application);
    g_signal_connect(G_OBJECT(view), "copy_thread_url",
		     G_CALLBACK(copy_thread_url_cb), application);
    g_signal_connect(G_OBJECT(view), "mark_thread",
		     G_CALLBACK(mark_thread_cb), application);
    g_signal_connect(G_OBJECT(view), "thread_mouse_over",
		     G_CALLBACK(thread_mouse_over_cb), application);
    g_signal_connect(G_OBJECT(view), "thread_mouse_out",
		     G_CALLBACK(thread_mouse_out_cb), application);

    g_signal_connect(G_OBJECT(paned_notebook), "item_view_required",
		     G_CALLBACK(item_view_required_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "item_view_being_closed",
		     G_CALLBACK(item_view_being_closed_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "item_view_closed",
		     G_CALLBACK(item_view_closed_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "page_switched",
		     G_CALLBACK(page_switched_cb), application);

    threadlist_view_set_enable_native_search(THREADLIST_VIEW(view), FALSE);
    threadlist_view_set_title_match_func(THREADLIST_VIEW(view),
			(ThreadlistViewTitleMatchFunc *)title_match_func);

    g_signal_connect(G_OBJECT(view), "interactive_search",
		     G_CALLBACK(interactive_search_cb), application);

    job_args->application = application;
    job_args->board = board;
    job_args->view = THREADLIST_VIEW(view);
    job_args->tab_label = tab_label;
    job_args->increment_ignored = TRUE;

    job->canceled = FALSE;
    job->job = show_threadlist;
    job->args = job_args;

    info->is_busy = TRUE;
    info->threadlist_source_buffer
      = ochusha_bulletin_board_get_threadlist_source(board,
				application->broker,
				OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH);
    if (info->threadlist_source_buffer != NULL)
      setup_for_tab_label_animation(info->threadlist_source_buffer,
				    tab_label);

    /* Ȥޤǳ */
    g_object_ref(G_OBJECT(tab_label));
    g_object_ref(G_OBJECT(view));
    if (info->threadlist_source_buffer != NULL)
      g_object_ref(G_OBJECT(info->threadlist_source_buffer));
#if DEBUG_WIDGET
    fprintf(stderr, "threadlist_view->ref_count=%d\n",
	    G_OBJECT(view)->ref_count);
#endif
    commit_job(job);
  }
  THREAD_LIST_UNLOCK;

  return paned_notebook;
}


/*
 * ޤĤ򳫤δλԤäƥ򳫤
 */
typedef struct _OpenThreadJobArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;

  char *thread_id;
  int res_num;

  gboolean in_tab;
} OpenThreadJobArgs;


void
ochusha_open_thread(OchushaApplication *application,
		    OchushaBulletinBoard *board, const char *thread_id,
		    int res_num, gboolean in_tab)
{
  WorkerJob *job;
  OpenThreadJobArgs *args;

  if (board == NULL)
    return;

  paned_notebook_open_item_view(PANED_NOTEBOOK(application->contents_window),
				board, in_tab);
  /*
   * Ĥ򳫤Τ̥åɤǹԤǽꡢ
   * Ǥϡ򳫤ΤϤθˤʤΤǡĤΤλ
   * Ԥθ她򳫤ΥåɤȤ
   */
  job = G_NEW0(WorkerJob, 1);
  args = G_NEW0(OpenThreadJobArgs, 1);

  job->canceled = FALSE;
  job->job = open_thread_view;
  job->args = args;

  args->application = application;
  args->board = board;

  args->thread_id = G_STRDUP(thread_id);
  args->res_num = res_num;

  args->in_tab = in_tab;

  commit_job(job);
}


static void
open_thread_view(WorkerThread *employee, gpointer job_args)
{
  OpenThreadJobArgs *args = (OpenThreadJobArgs *)job_args;
  OchushaApplication *application = args->application;
  OchushaBulletinBoard *board = args->board;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  THREAD_LIST_LOCK
  {
    PanedNotebook *board_view;
    OchushaBBSThread *thread
      = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
						       args->thread_id);
    while (info->is_busy && thread == NULL)
      {
	THREAD_LIST_COND_WAIT;
	thread
	  = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
							   args->thread_id);
      }

#if DEBUG_GUI_MOST
    fprintf(stderr, "Thread ID (e.g. DAT file name): \"%s\"\n",
	    args->thread_id);
#endif
    /* λǡ٤Ϻ줿Ȥݾڤ */
#if 0
    thread = g_hash_table_lookup(board->thread_table, args->thread_id);
#endif

    if (thread != NULL)
      {
	gdk_threads_enter();
	board_view = (PanedNotebook *)paned_notebook_get_item_view(PANED_NOTEBOOK(application->contents_window), board);

	/* TODO:
	 * (board_view != NULL && thread == NULL)ʤ顢¿ʬΥDAT
	 * 긵˥å夷ƤʬȤäƼɽǤ褦ˤ
	 * ޤargs->res_numλȤƻͤ褦
	 */

	if (board_view != NULL)
	  paned_notebook_open_item_view(board_view, thread, args->in_tab);
	gdk_threads_leave();
      }
  }
  THREAD_LIST_UNLOCK;

  G_FREE(args->thread_id);
  G_FREE(args);
}


void
refresh_threadlist_view(OchushaApplication *application,
			OchushaBulletinBoard *board)
{
  PanedNotebook *contents_window;
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *board_view;
  WorkerJob *job;
  ThreadlistJobArgs *job_args;
  IconLabel *tab_label;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  g_return_if_fail(board != NULL && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board_view = (PanedNotebook *)paned_notebook_get_item_view(contents_window,
							     board);
  g_return_if_fail(board_view != NULL);
  tab_label = (IconLabel *)paned_notebook_get_tab_label(contents_window,
							board);

  scrolled_window = paned_notebook_get_selector(board_view);
  g_return_if_fail(scrolled_window != NULL);

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  THREAD_LIST_LOCK
  {
    OchushaAsyncBuffer *old_buffer;
    if (info->is_busy)
      {
	THREAD_LIST_UNLOCK;
#if DEBUG_GUI_MOST
	fprintf(stderr, "board is busy.\n");
#endif
	return;
      }

    old_buffer = info->threadlist_source_buffer;

    info->threadlist_source_buffer
      = ochusha_bulletin_board_get_threadlist_source(board,
				application->broker,
				OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH);
    if (info->threadlist_source_buffer == NULL)
      {
	info->threadlist_source_buffer = old_buffer;
	THREAD_LIST_UNLOCK;
	return;
      }

    setup_for_tab_label_animation(info->threadlist_source_buffer, tab_label);

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(ThreadlistJobArgs, 1);

    job_args->application = application;
    job_args->board = board;
    job_args->view = threadlist_view;
    job_args->tab_label = tab_label;
    job_args->increment_ignored = TRUE;
      
    job->canceled = FALSE;
    job->job = show_threadlist;
    job->args = job_args;
      
    info->is_busy = TRUE;
    /* λޤǳ */
    g_object_ref(G_OBJECT(info->threadlist_source_buffer));
    g_object_ref(G_OBJECT(tab_label));
    g_object_ref(G_OBJECT(threadlist_view));
    commit_job(job);

    g_object_unref(G_OBJECT(old_buffer));
  }
  THREAD_LIST_UNLOCK;
}


void
redraw_threadlist_view(OchushaApplication *application,
		       OchushaBulletinBoard *board)
{
  PanedNotebook *contents_window;
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *board_view;
  WorkerJob *job;
  ThreadlistJobArgs *job_args;
  IconLabel *tab_label;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  g_return_if_fail(board != NULL && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board_view = (PanedNotebook *)paned_notebook_get_item_view(contents_window,
							     board);
  if (board_view == NULL)
    return;	/* ɽƤʤĤξϲ⤻ */

  tab_label = (IconLabel *)paned_notebook_get_tab_label(contents_window,
							board);

  scrolled_window = paned_notebook_get_selector(board_view);
  g_return_if_fail(scrolled_window != NULL);

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  THREAD_LIST_LOCK
  {
    if (info->is_busy)
      {
	info->redraw_required = TRUE;
	THREAD_LIST_UNLOCK;
	return;
      }

    if (info->threadlist_source_buffer == NULL)
      {
	THREAD_LIST_UNLOCK;
	return;
      }

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(ThreadlistJobArgs, 1);

    job_args->application = application;
    job_args->board = board;
    job_args->view = threadlist_view;
    job_args->tab_label = tab_label;
    job_args->increment_ignored = FALSE;
      
    job->canceled = FALSE;
    job->job = show_threadlist;
    job->args = job_args;

    info->is_busy = TRUE;
    /* λޤǳ */
    g_object_ref(G_OBJECT(info->threadlist_source_buffer));
    g_object_ref(G_OBJECT(tab_label));
    g_object_ref(G_OBJECT(threadlist_view));
    commit_job(job);
  }
  THREAD_LIST_UNLOCK;
}


typedef struct _EachThreadArgs
{
  OchushaApplication *application;
  BulletinBoardGUIInfo *info;
  ThreadlistView *view;
  int number_of_threads;

  OchushaBBSThread *last_thread;

  ThreadlistFilter *filter;
  gboolean enable_filter;
  gboolean increment_ignored;
} EachThreadArgs;


static gboolean
each_thread_cb(OchushaBBSThread *thread, gpointer user_data)
{
  EachThreadArgs *args = (EachThreadArgs *)user_data;
  ThreadlistView *view = args->view;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  gboolean filter_result;
  gboolean update_condition
    = (!args->application->config.offline && args->increment_ignored);

#if DEBUG_GUI_MOST
  gchar *title = convert_string(utf8_to_native, NULL, thread->title, -1);
  fprintf(stderr, "each_thread_cb: %s <%d>\n",
	  title, thread->number_of_responses_on_server);
  G_FREE(title);
#endif

  args->number_of_threads++;

  if (update_condition)
    info->view_flags &= ~BBS_THREAD_NEW;

  if (info->view_rank == 0)
    info->view_flags |= BBS_THREAD_NEW;

  if ((info->view_flags & BBS_THREAD_NEW) != 0)
    info->view_rank = 0;	/* ĥ뤿 */

  filter_result = threadlist_filter_check_thread(args->filter,
						 args->application, thread);
  if (filter_result
      || (!args->enable_filter
	  && ((info->view_flags & BBS_THREAD_HIDDEN) == 0
	      || !args->application->hide_hidden_threads)))
    {
      gdk_threads_enter();

#if ENABLE_RENDERING_ON_IDLE
      while (!try_get_rendering_right(args->application,
				      args->info->threadlist_source_buffer,
				      thread->board))
	{
	  gdk_threads_leave();
	  ochusha_async_buffer_wait(args->info->threadlist_source_buffer,
				    "bulletin_board_ui.c: each_thread_cb()");
	  gdk_threads_enter();
	}
#endif

      threadlist_view_append_thread(view, thread,
				    info->view_rank, args->number_of_threads);
      update_threadlist_entry_style(args->application, view, thread);

      gdk_threads_leave();
      args->last_thread = thread;
      if (update_condition)
	info->view_ignored++;
      info->view_rank = args->number_of_threads;
    }

  return TRUE;
}


static void
show_threadlist(WorkerThread *employee, gpointer args)
{
  ThreadlistJobArgs *job_args = (ThreadlistJobArgs *)args;
  OchushaApplication *application = job_args->application;
  OchushaBulletinBoard *board = job_args->board;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);
  ThreadlistView *view = job_args->view;
  OchushaAsyncBuffer *buffer;
  EachThreadArgs cb_args =
    {
      application,
      info,
      view,
      0,			/* number_of_threads */
      NULL,			/* OchushaBBSThread *last_thread */
      &info->filter,
      info->enable_filter,
      job_args->increment_ignored
    };
#if DEBUG_GUI_MOST
  {
    gchar *board_name = convert_string(utf8_to_native, board->name, -1);
    fprintf(stderr, "refresh_current_threadlist: update threadlist for board: %s.\n", board_name);
    G_FREE(board_name);
  }
#endif

  if (board->thread_list == NULL)
    ochusha_bulletin_board_read_threadlist_xml(board, &application->config);

  THREAD_LIST_LOCK
  {
  redraw:
    buffer = info->threadlist_source_buffer;
    info->redraw_required = FALSE;
  }
  THREAD_LIST_UNLOCK;

  if (buffer == NULL)
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "Couldn't get thread list (subject.txt) for %s\n", name);
      G_FREE(name);
#endif
      goto finish_job;
    }

  gdk_threads_enter();
  if (!threadlist_view_open(view))
    {
      gdk_threads_leave();
#if DEBUG_GUI_MOST
      fprintf(stderr, "thread list is now being updated by another thread.\n");
#endif
      goto finish_job;
    }
  gdk_threads_leave();

  if (!ochusha_bulletin_board_refresh_threadlist(board, buffer,
						 each_thread_cb, &cb_args))
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "Broken thread list (subject.txt) for %s\n", name);
      G_FREE(name);
#endif
      gdk_threads_enter();
      threadlist_view_close(view, NULL);
      gdk_threads_leave();
      goto finish_job;
    }
  
#if DEBUG_GUI_MOST
  fprintf(stderr, "Updating threadlist.\n");
#endif

  gdk_threads_enter();
  threadlist_view_close(view, cb_args.last_thread);
  gdk_threads_leave();

 finish_job:
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "show_threadlist: unref OchushaAsyncBuffer(%p)\n",
	  buffer);
  fprintf(stderr, "* before unref: buffer->ref_count=%d\n",
	  G_OBJECT(buffer)->ref_count);
#endif
  THREAD_LIST_LOCK
  {
    if (!info->redraw_required)
      {
	info->is_busy = FALSE;
	THREAD_LIST_COND_BROADCAST;
      }
    else
      goto redraw;
  }
  THREAD_LIST_UNLOCK;

  gdk_threads_enter();
  g_object_unref(G_OBJECT(view));	/* ⤦ʤʤäƤɤ */
  if (job_args->tab_label != NULL)
    g_object_unref(G_OBJECT(job_args->tab_label));
  gdk_threads_leave();

  if (buffer != NULL)
    g_object_unref(G_OBJECT(buffer));

  G_FREE(args);
}


OchushaAsyncBuffer *
snatch_threadlist_source_buffer_for_board(OchushaBulletinBoard *board,
					  OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);
  OchushaAsyncBuffer *buffer;

  THREAD_LIST_LOCK
  {
    buffer = info->threadlist_source_buffer;
    info->is_busy = FALSE;
    info->threadlist_source_buffer = NULL;
    THREAD_LIST_COND_BROADCAST;
  }
  THREAD_LIST_UNLOCK;

  return buffer;
}


void
update_threadlist_entry_style(OchushaApplication *application,
			      ThreadlistView *view, OchushaBBSThread *thread)
{
  BulletinBoardGUIInfo *board_info = ensure_bulletin_board_info(thread->board,
								application);
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  ThreadlistEntryVisualParameter visual_parameter =
    {
      0,	/* flags */
      0,	/* reserved */
      LIST_ENTRY_FOREGROUND_NORMAL,	/* foreground color */
      LIST_ENTRY_BACKGROUND_NORMAL	/* background color */
    };

  gboolean exist_new_responses
    = (thread->number_of_responses_on_server
       > thread->number_of_responses_read);

  /* flags */
  if (info->view_flags & BBS_THREAD_HIDDEN)
    visual_parameter.flags |= THREADLIST_ENTRY_FLAG_STRIKE;
  if (info->view_flags & BBS_THREAD_FAVORITE)
    visual_parameter.flags |= THREADLIST_ENTRY_FLAG_MARK;

  if (info->view_flags & BBS_THREAD_HIDDEN)
    {
      visual_parameter.foreground_color = LIST_ENTRY_FOREGROUND_HIDDEN;
      visual_parameter.background_color = LIST_ENTRY_BACKGROUND_HIDDEN;
    }
  else if (info->view_flags & BBS_THREAD_NEW)
    {
      visual_parameter.foreground_color = LIST_ENTRY_FOREGROUND_STRONG;
      visual_parameter.background_color = LIST_ENTRY_BACKGROUND_STRONG;
    }
  else if (threadlist_filter_check_thread(&board_info->filter,
					  application, thread))
    {
      if (exist_new_responses)
	visual_parameter.foreground_color = LIST_ENTRY_FOREGROUND_EMPH;
      if (!board_info->enable_filter)
	visual_parameter.background_color = LIST_ENTRY_BACKGROUND_EMPH;
    }

  threadlist_view_update_thread(view, thread);
  threadlist_view_update_thread_visual(view, thread, &visual_parameter);
}


static void
advance_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
		PanedNotebook *paned_notebook)
{
  if (paned_notebook_get_current_item(paned_notebook) != thread)
    paned_notebook_open_item_view(paned_notebook, thread, FALSE);
  else
    {
      /* ڡ򥹥뤵 */
      GtkWidget *scrolled_window
	= paned_notebook_get_current_item_view(paned_notebook);
      BBSThreadView *thread_view
	= BBS_THREAD_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));
      g_signal_emit_by_name(G_OBJECT(thread_view), "move_cursor",
			    GTK_MOVEMENT_PAGES, 1, FALSE);
    }
}


static void
back_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
	     PanedNotebook *paned_notebook)
{
  if (paned_notebook_get_current_item(paned_notebook) != thread)
    paned_notebook_open_item_view(paned_notebook, thread, FALSE);
  else
    {
      /* ڡ򥹥뤵 */
      GtkWidget *scrolled_window
	= paned_notebook_get_current_item_view(paned_notebook);
      BBSThreadView *thread_view
	= BBS_THREAD_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));
      g_signal_emit_by_name(G_OBJECT(thread_view), "move_cursor",
			    GTK_MOVEMENT_PAGES, -1, FALSE);
    }
}


static void
write_response_cb(ThreadlistView *view, OchushaBBSThread *thread,
		  OchushaApplication *application)
{
  /* ˤ륫֤Υ˥쥹񤯤Ȥ񤤤
   * 
   */
  write_response(application);
}


static void
close_threadlist_cb(ThreadlistView *view, OchushaApplication *application)
{
  paned_notebook_close_current_item_view(PANED_NOTEBOOK(application->contents_window));
}


static void
toggle_mark_cb(ThreadlistView *view, OchushaBBSThread *thread,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  info->view_flags ^= BBS_THREAD_FAVORITE;
  update_threadlist_entry_style(application, view, thread);
}


static void
toggle_hide_cb(ThreadlistView *view, OchushaBBSThread *thread,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  info->view_flags ^= BBS_THREAD_HIDDEN;
  if ((info->view_flags & BBS_THREAD_HIDDEN) != 0)
    ochusha_bbs_thread_remove_cache(thread, &application->config);
  update_threadlist_entry_style(application, view, thread);
}


static void
reset_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
		OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  ochusha_bbs_thread_remove_cache(thread, &application->config);
  info->view_flags = 0;
  update_threadlist_entry_style(application, view, thread);
}


static void
copy_thread_url_cb(ThreadlistView *view, OchushaBBSThread *thread,
		   OchushaApplication *application)
{
  ochusha_clipboard_set_text(application, ochusha_bbs_thread_get_url(thread));
}


static void
mark_thread_cb(ThreadlistView *view, OchushaBBSThread *thread, gboolean do_mark,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  if (do_mark)
    info->view_flags |= BBS_THREAD_FAVORITE;
  else
    info->view_flags &= ~BBS_THREAD_FAVORITE;
  update_threadlist_entry_style(application, view, thread);
}


static void
thread_mouse_over_cb(ThreadlistView *view, OchushaBBSThread *thread,
		     OchushaApplication *application)
{
  if (!application->enable_popup_title)
    return;

  gdk_window_get_pointer(gdk_screen_get_root_window(gtk_widget_get_screen(GTK_WIDGET(view))),
			 &title_popup_x_pos, &title_popup_y_pos, NULL);
#if DEBUG_GUI
  fprintf(stderr, "thread_mouse_over_cb: x=%d, y=%d, thread=%p\n",
	  title_popup_x_pos, title_popup_y_pos, thread);
#endif

  THREAD_LIST_LOCK
  {
    if (title_popup_delay_id != 0)
      g_source_remove(title_popup_delay_id);
    popup_thread = thread;
    title_popup_delay_id = g_timeout_add(application->popup_title_delay,
					 thread_title_popup_timeout,
					 thread);
  }
  THREAD_LIST_UNLOCK;
}


static void
thread_mouse_out_cb(ThreadlistView *view, OchushaBBSThread *thread,
		    OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "thread_mouse_out_cb: thread=%p\n", thread);
#endif
  THREAD_LIST_LOCK
  {
    if (title_popup_delay_id != 0)
      g_source_remove(title_popup_delay_id);
    popup_thread = NULL;
  }
  THREAD_LIST_UNLOCK;

  gtk_widget_hide_all(title_popup_window);
}


static gboolean
thread_title_popup_timeout(gpointer data)
{
  gboolean result = TRUE;

#if DEBUG_GUI
  fprintf(stderr, "thread_title_popup_timeout: x=%d, y=%d, popup_thread=%p, data=%p\n",
	  title_popup_x_pos, title_popup_y_pos, popup_thread, data);
#endif

  gdk_threads_enter();
  if (data == popup_thread)
    {
      THREAD_LIST_LOCK
      {
	thread_title_popup_show();

	if (title_popup_delay_id != 0)
	  title_popup_delay_id = 0;

	popup_thread = NULL;
      }
      THREAD_LIST_UNLOCK;

      result = FALSE;
    }
  gdk_threads_leave();

  return result;
}


static void
thread_title_popup_show()
{
  gchar buffer[1024];

  if (popup_thread == NULL)
    return;

  snprintf(buffer, 1024, "%s [%d/%d]",
	   popup_thread->title,
	   popup_thread->number_of_responses_read,
	   popup_thread->number_of_responses_on_server);
  
  gtk_label_set_text(GTK_LABEL(title_popup_label), buffer);

  gtk_window_move(GTK_WINDOW(title_popup_window),
		  title_popup_x_pos + 16, title_popup_y_pos);
  gtk_widget_show_all(title_popup_window);
}


static gboolean
thread_title_popup_paint_window(GtkWidget *widget)
{
  gtk_paint_flat_box(widget->style, widget->window,
		     GTK_STATE_NORMAL, GTK_SHADOW_OUT,
		     NULL, widget, "tooltip",
		     0, 0, -1, -1);
  return FALSE;
}


static void
item_view_required_cb(PanedNotebook *paned_notebook,
		      OchushaBBSThread *thread, GtkWidget **item_view,
		      const gchar **title, GtkWidget **tab_label,
		      OchushaApplication *application)
{
  GtkWidget *thread_view;
  *tab_label = get_tab_label(thread->title);

  thread_view = open_bbs_thread(application, thread, ICON_LABEL(*tab_label));
  if (thread_view != NULL)
    {
      g_signal_connect_swapped(G_OBJECT(thread_view), "close_thread",
			       G_CALLBACK(paned_notebook_close_current_item_view),
			       paned_notebook);
      gtk_widget_show(thread_view);
      *item_view = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(*item_view),
				     GTK_POLICY_AUTOMATIC,
				     GTK_POLICY_AUTOMATIC);
      gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(*item_view),
					  GTK_SHADOW_ETCHED_IN);
      gtk_container_add(GTK_CONTAINER(*item_view), thread_view);
      gtk_widget_show(*item_view);
    }
  else
    *item_view = NULL;

  *title = thread->title;
}


static void
free_response_source_buffer(OchushaBBSThread *thread, gpointer unused)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "free_response_source_buffer: unref OchushaAsyncBuffer(%p)\n",
	  info->response_source_buffer);
  fprintf(stderr, "* before unref: response_source_buffer->ref_count=%d\n",
	  G_OBJECT(info->response_source_buffer)->ref_count);
#endif
  if (info->response_source_buffer != NULL)
    {
      g_object_unref(G_OBJECT(info->response_source_buffer));
      info->response_source_buffer = NULL;
    }
}


static void
item_view_being_closed_cb(PanedNotebook *paned_notebook,
			  OchushaBBSThread *thread, GtkWidget *item_view,
			  OchushaApplication *application)
{
  BBSThreadGUIInfo *info;
  OchushaAsyncBuffer *buffer;

  g_return_if_fail(GTK_IS_WIDGET(item_view));

  info = ensure_bbs_thread_info(thread);
  buffer = info->response_source_buffer;

  /*
   * XXX: Ǥinfo->bufferοǤthread safetyݤǤƤʤ
   *      ġġ
   */

#if DEBUG_GUI
  fprintf(stderr, "bulletin_board_ui.c: being_closed thread=%p, item_view=%p\n",
	  thread, item_view);
#endif

  /*
   * MEMO: ɽƤ륦åȤºݤ˼ΤƤ륿ߥ󥰤򤺤餹
   * 
   * GtkTextViewŪidleؿϿƤꡢ줬Ƥ֤˥
   * åȤȲ줿åȤidleؿưƤޤ
   * ͡ˡʤΤǡǤϥåȤref_count1äƤ
   * Υץ饤ƥidleؿˤäơˡref_count򸺤餹
   * Υץ饤ƥidleؿưȤȤϡGTK+Ū
   * ȤäƤidleؿλŻλƤȤȤˤʤΤǡ
   * פ
   */
  g_object_ref(G_OBJECT(item_view));


  /*
   * MEMO: Ĥ褦ȤƤ륹ΡDATɤ߹ߡפȡ֥󥰡
   *       ԤäƤthreadŪ˽λ롣
   */
  if (buffer != NULL)
    {
      gdk_threads_leave();

      ochusha_async_buffer_suspend(buffer);
      ochusha_async_buffer_terminate(buffer);
      ochusha_async_buffer_resume(buffer);

      gdk_threads_enter();
    }

  if (item_view != NULL)
    {
      GtkWidget *thread_view = gtk_bin_get_child(GTK_BIN(item_view));
      if (application->last_search_target_widget == thread_view)
	application->last_search_target_widget = NULL;
    }
}


static void
item_view_closed_cb(PanedNotebook *paned_notebook, OchushaBBSThread *thread,
		    GtkWidget *item_view, OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  /* ΥåɽĤ줿 */
#if DEBUG_GUI_MOST
  fprintf(stderr, "closed: thread=%p, item_view=%p\n", thread, item_view);
#endif

  delayed_g_object_unref(G_OBJECT(item_view));

  info->last_read = NULL;
  info->next_mark = NULL;
  free_response_source_buffer(thread, NULL);
}


static void
page_switched_cb(PanedNotebook *paned_notebook,
		 GtkWidget *previous_page, GtkWidget *new_page,
		 OchushaApplication *application)
{
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  OchushaBBSThread *thread;
  BBSThreadGUIInfo *info;
#if DEBUG_GUI_MOST
  fprintf(stderr, "bulletin_board_ui.c: page_switched_cb\n");
#endif
  /*
   * MEMO: ɤGtkTextViewϡºݤɽ˥뤬ɽ
   *       ޤޤ褦ɽǤޤͤʤΤǡڡڤ
   *       ˡɽ˥ưƤ
   */
  if (previous_page != NULL)
    {
      GtkTextView *thread_view
	= GTK_TEXT_VIEW(gtk_bin_get_child(GTK_BIN(previous_page)));
      thread = (OchushaBBSThread *)paned_notebook_get_item(paned_notebook,
						    previous_page);
      gtk_text_view_place_cursor_onscreen(thread_view);
      if (thread != NULL)
	{
	  info = ensure_bbs_thread_info(thread);
	  info->next_mark = NULL;
	}
    }

#if ENABLE_MAIN_TOOLBAR
  {
    gchar pseudo_url[PATH_MAX];
    OchushaBBSThread *thread = paned_notebook_get_item(paned_notebook, new_page);
    if (thread == NULL)
      set_current_url(application, "empty");
    else
      {
	snprintf(pseudo_url, PATH_MAX, "http://%s/%s",
		 thread->board->server, ochusha_bbs_thread_get_url(thread));
	set_current_url(application, pseudo_url);
      }
  }
#endif

  scrolled_window = paned_notebook_get_selector(paned_notebook);
  thread = paned_notebook_get_item(paned_notebook, new_page);
  if (scrolled_window == NULL || thread == NULL)
    return;

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  info = ensure_bbs_thread_info(thread);
  if (info->next_mark != NULL)
    {
      BBSThreadView *thread_view
	= BBS_THREAD_VIEW(gtk_bin_get_child(GTK_BIN(new_page)));
      bbs_thread_view_scroll_to_mark(thread_view, info->next_mark);
    }

  if (application->show_threadlist_entry_when_thread_selected)
    threadlist_view_scroll_to_thread(threadlist_view, thread);
}


static void
get_current_thread(OchushaApplication *application,
		   GtkWidget **view, OchushaBBSThread **thread, IconLabel **tab_label)
{
  PanedNotebook *paned_notebook;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  paned_notebook = PANED_NOTEBOOK(application->contents_window);
  paned_notebook = PANED_NOTEBOOK(paned_notebook_get_current_item_view(paned_notebook));
  if (paned_notebook == NULL)
    {
      if (view != NULL)
	*view = NULL;
      if (thread != NULL)
	*thread = NULL;
      return;
    }

  if (view != NULL)
    {
      GtkWidget *bin = paned_notebook_get_current_item_view(paned_notebook);
      *view = gtk_bin_get_child(GTK_BIN(bin));
    }
  if (thread != NULL)
    *thread = (OchushaBBSThread *)paned_notebook_get_current_item(paned_notebook);
  if (tab_label != NULL)
    *tab_label = (IconLabel *)paned_notebook_get_tab_label(paned_notebook,
							   *thread);
}


void
finalize_bulletin_board_view(OchushaBulletinBoard *board, GtkWidget *widget)
{
  PanedNotebook *board_view;

  if (widget == NULL)
    return;

  board_view = PANED_NOTEBOOK(widget);
  paned_notebook_foreach_item(board_view, (GFunc)free_response_source_buffer,
			      NULL);
#if DEBUG_WIDGET
  {
    GtkWidget *scrolled_window = paned_notebook_get_selector(board_view);
    if (scrolled_window != NULL)
      {
	ThreadlistView *threadlist_view
	  = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));
	fprintf(stderr, "finalize: threadlist_view->ref_count=%d\n",
		G_OBJECT(threadlist_view)->ref_count);
      }
  }
#endif
}


static const gchar *
reverse_strpbrk(const gchar *string, const gchar *charset)
{
  const gchar *s = string;
  while (*s != '\0')
    {
      if (strchr(charset, *s) == NULL)
	return s;
      s++;
    }

  return NULL;
}


typedef struct _WriteResponseJobArgs
{
  OchushaBBSThread *thread;
  gchar *name;
  gchar *mailto;
  gchar *message;
  GtkWidget *dialog;
  OchushaApplication *application;
} WriteResponseJobArgs;


static void
background_write_response(WorkerThread *employee, gpointer job_args)
{
  WriteResponseJobArgs *args = (WriteResponseJobArgs *)job_args;
  OchushaBBSResponse response;
  gboolean result;

  response.name = args->name;
  response.mailto = args->mailto;
  response.content = args->message;

  result = ochusha_bbs_thread_post_response(args->thread,
					    args->application->broker,
					    &response);

  if (result)
    {
      BBSThreadGUIInfo *thread_info = ensure_bbs_thread_info(args->thread);
      BulletinBoardGUIInfo *board_info
	= ensure_bulletin_board_info(args->thread->board, args->application);
      gboolean number_only
	= (response.name[0] != '\0'
	   && reverse_strpbrk(response.name, "0123456789") == NULL);
      GtkWidget *notebook;

      gdk_threads_enter();
      if (thread_info->last_name == NULL
	  || strcmp(thread_info->last_name, response.name) != 0)
	{
	  if (thread_info->last_name != NULL)
	    G_FREE(thread_info->last_name);
	  thread_info->last_name = G_STRDUP(response.name);
	}

      if (thread_info->last_mail == NULL
	  || strcmp(thread_info->last_mail, response.mailto) != 0)
	{
	  if (thread_info->last_mail != NULL)
	    G_FREE(thread_info->last_mail);
	  thread_info->last_mail = G_STRDUP(response.mailto);
	}

      if (!number_only)
	{
	  if (board_info->last_name == NULL
	      || strcmp(board_info->last_name, response.name) != 0)
	    {
	      if (board_info->last_name != NULL)
		G_FREE(board_info->last_name);
	      board_info->last_name = G_STRDUP(response.name);
	    }

	  if (board_info->last_mail == NULL
	      || strcmp(board_info->last_mail, response.mailto) != 0)
	    {
	      if (board_info->last_mail != NULL)
		G_FREE(board_info->last_mail);
	      board_info->last_mail = G_STRDUP(response.mailto);
	    }

	  if (args->application->last_name == NULL
	      || strcmp(args->application->last_name, response.name) != 0)
	    {
	      if (args->application->last_name != NULL)
		G_FREE(args->application->last_name);
	      args->application->last_name = G_STRDUP(response.name);
	    }

	  if (args->application->last_mail == NULL
	      || strcmp(args->application->last_mail, response.mailto) != 0)
	    {
	      if (args->application->last_mail != NULL)
		G_FREE(args->application->last_mail);
	      args->application->last_mail = G_STRDUP(response.mailto);
	    }
	}

      notebook = paned_notebook_get_item_view(PANED_NOTEBOOK(args->application->contents_window), args->thread->board);
      if (notebook != NULL)
	{
	  GtkWidget *view = paned_notebook_get_item_view(PANED_NOTEBOOK(notebook), args->thread);
	  GtkWidget *label = paned_notebook_get_tab_label(PANED_NOTEBOOK(notebook), args->thread);
	  if (view != NULL && label != NULL)
	    {
	      GtkWidget *thread_view = gtk_bin_get_child(GTK_BIN(view));
	      refresh_thread(args->application, thread_view, args->thread,
			     ICON_LABEL(label));
	    }
	}

      gtk_widget_destroy(args->dialog);
      gdk_threads_leave();
    }
  else
    {
      gdk_threads_enter();
      ochusha_open_url(args->application,
		       ochusha_bbs_thread_get_url_to_post_response(args->thread),
		       FALSE, TRUE);
      gtk_widget_show_all(args->dialog);
      gdk_threads_leave();
    }

  G_FREE(args->name);
  G_FREE(args->mailto);
  G_FREE(args->message);
  G_FREE(args);
}


static void
write_dialog_response_cb(GtkWidget *write_dialog, gint response_id,
			 OchushaApplication *application)
{
  OchushaBBSThread *thread = g_object_get_data(G_OBJECT(write_dialog),
					       "thread");
  ResponseEditor *editor;
  const gchar *name;
  const gchar *mailto;
  gchar *message;

  if (response_id != GTK_RESPONSE_OK)
    {
      gtk_widget_destroy(write_dialog);
      return;
    }

  editor = g_object_get_data(G_OBJECT(write_dialog), "editor");

  name = response_editor_get_name(editor);
  mailto = response_editor_get_mail(editor);
  message = response_editor_get_response(editor);

  if (message != NULL && *message != '\0')
    {
      WorkerJob *job = G_NEW0(WorkerJob, 1);
      WriteResponseJobArgs *job_args = G_NEW0(WriteResponseJobArgs, 1);
      
      job_args->thread = thread;
      job_args->name = G_STRDUP(name);
      job_args->mailto = G_STRDUP(mailto);
      job_args->message = message;
      job_args->dialog = write_dialog;
      job_args->application = application;

      job->canceled = FALSE;
      job->job = background_write_response;
      job->args = job_args;

      gtk_widget_hide(write_dialog);

      commit_job(job);

      return;
    }

  G_FREE(message);
}


void
write_response(OchushaApplication *application)
{
  BBSThreadGUIInfo *info;
  GtkWidget *view;
  OchushaBBSThread *thread;
  IconLabel *tab_label;
  GtkWidget *write_dialog;

  if (application->config.offline)
    return;

  get_current_thread(application, &view, &thread, &tab_label);
  if (view == NULL || thread == NULL)
    return;

  if (thread->flags & OCHUSHA_BBS_THREAD_DAT_DROPPED)
    return;

  if (!ochusha_bbs_thread_is_post_supported(thread))
    {
      ochusha_open_url(application,
		       ochusha_bbs_thread_get_url_to_post_response(thread),
		       FALSE, TRUE);
      return;
    }

  info = ensure_bbs_thread_info(thread);

  if (info->write_dialog == NULL)
    {
      write_dialog = create_write_dialog(application, thread);
      info->write_dialog = write_dialog;
    }
  else
    return;

  g_signal_connect(G_OBJECT(write_dialog), "response",
		   G_CALLBACK(write_dialog_response_cb), application);
  g_signal_connect(G_OBJECT(write_dialog), "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &info->write_dialog);

  gtk_widget_show_all(write_dialog);
}


void
refresh_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  IconLabel *tab_label;
  get_current_thread(application, &view, &thread, &tab_label);
  if (view == NULL || thread == NULL)
    return;
  refresh_thread(application, view, thread, tab_label);
}


static void
write_response_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  write_response(application);
}


static void
select_font_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  select_thread_view_font(application);
}


static void
refresh_thread_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  refresh_current_thread(application);
}


static void
text_search_window_response_cb(TextSearchWindow *window, gint response_id,
			       OchushaApplication *application)
{
  switch (response_id)
    {
    case GTK_RESPONSE_DELETE_EVENT:
      application->thread_search_window = NULL;
      /* fall through */

    case GTK_RESPONSE_CANCEL:
      /* XXX: ɥξȥϿ٤ */
      gtk_widget_hide(GTK_WIDGET(window));
      if (application->last_search_target_widget != NULL)
	{
	  bbs_thread_view_invalidate_search_result(BBS_THREAD_VIEW(application->last_search_target_widget));
	  application->last_search_target_widget = NULL;
	}
      return;
    }

  fprintf(stderr, "text_search_window_response_cb: unknown response(%d)\n",
	  response_id);
}


static gboolean
text_search_window_query_changed_cb(TextSearchWindow *window, const gchar *key,
				    TextSearchDirection direction,
				    gboolean enable_wrap, gboolean match_case,
				    gboolean use_regexp,
				    OchushaApplication *application)
{
  GtkWidget *widget;
  get_current_thread(application, &widget, NULL, NULL);

  if (application->last_search_target_widget != widget)
    {
      if (application->last_search_target_widget != NULL
	  && IS_BBS_THREAD_VIEW(application->last_search_target_widget))
	bbs_thread_view_invalidate_search_result(BBS_THREAD_VIEW(application->last_search_target_widget));
      application->last_search_target_widget = widget;
    }

  if (widget == NULL)
    return FALSE;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(widget), FALSE);
  return bbs_thread_view_find(
		BBS_THREAD_VIEW(application->last_search_target_widget),
		key, direction, enable_wrap, match_case, use_regexp);
}


static gboolean
text_search_window_find_next_cb(TextSearchWindow *window, const gchar *key,
				TextSearchDirection direction,
				gboolean enable_wrap, gboolean match_case,
				gboolean use_regexp,
				OchushaApplication *application)
{
  GtkWidget *widget;

  get_current_thread(application, &widget, NULL, NULL);

  if (application->last_search_target_widget == widget)
    {
      if (widget == NULL)
	return FALSE;
      return bbs_thread_view_find_next(BBS_THREAD_VIEW(widget));
    }

  if (application->last_search_target_widget != NULL)
    bbs_thread_view_invalidate_search_result(BBS_THREAD_VIEW(application->last_search_target_widget));
  application->last_search_target_widget = widget;

  if (widget == NULL)
    return FALSE;
  return bbs_thread_view_find(
		BBS_THREAD_VIEW(application->last_search_target_widget),
		key, direction, enable_wrap, match_case, use_regexp);
}


static void
setup_thread_search_window(OchushaApplication *application)
{
  application->thread_search_window = text_search_window_new();
  text_search_window_set_enable_incremental_search(
			TEXT_SEARCH_WINDOW(application->thread_search_window),
			TRUE);
  g_signal_connect(application->thread_search_window, "response",
		   G_CALLBACK(text_search_window_response_cb),
		   application);
  g_signal_connect(application->thread_search_window, "query_changed",
		   G_CALLBACK(text_search_window_query_changed_cb),
		   application);
  g_signal_connect(application->thread_search_window, "find_next",
		   G_CALLBACK(text_search_window_find_next_cb),
		   application);
  gtk_window_set_title(GTK_WINDOW(application->thread_search_window),
		       _("Find in the current thread"));
  gtk_window_set_transient_for(GTK_WINDOW(application->thread_search_window),
			       application->top_level);
}


void
start_thread_search(OchushaApplication *application)
{
  if (application->thread_search_window == NULL)
    setup_thread_search_window(application);

  gtk_widget_show(application->thread_search_window);
  gtk_widget_grab_focus(application->thread_search_window);
}


static void
start_thread_search_button_cb(GtkWidget *widget,
			      OchushaApplication *application)
{
  start_thread_search(application);
}


void
jump_to_new_comer_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  jump_to_new_comer_response(view, thread);
}


static void
jump_to_new_comer_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  jump_to_new_comer_response_of_current_thread(application);
}


void
go_to_the_last_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  go_to_the_last_response(view, thread);
}


static void
go_to_bottom_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  go_to_the_last_response_of_current_thread(application);
}


void
go_to_the_first_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  go_to_the_first_response(view, thread);
}


static void
go_to_top_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  go_to_the_first_response_of_current_thread(application);
}


static GtkWidget *
create_write_dialog(OchushaApplication *application, OchushaBBSThread *thread)
{
  GtkWidget *write_dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *url_hbox;
  GtkWidget *url_entry;
  GtkWidget *title_hbox;
  GtkWidget *title_label;
  GtkWidget *board_label;
  GtkWidget *editor;
  ResponseEditor *response_editor;
  BBSThreadGUIInfo *thread_info = ensure_bbs_thread_info(thread);
  BulletinBoardGUIInfo *board_info = ensure_bulletin_board_info(thread->board,
								application);
  gboolean has_name = FALSE;
  gboolean has_mail = FALSE;

  write_dialog = gtk_dialog_new_with_buttons(_("Write a response"),
					 application->top_level,
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					 GTK_STOCK_OK, GTK_RESPONSE_OK,
					 NULL);
  gtk_window_set_position(GTK_WINDOW(write_dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_default_size(GTK_WINDOW(write_dialog), -1, 250);

  dialog_vbox = GTK_DIALOG(write_dialog)->vbox;

  title_hbox = gtk_hbox_new(FALSE, 5);
  title_label = gtk_label_new(ochusha_bbs_thread_get_title(thread));
  gtk_label_set_justify(GTK_LABEL(title_label), GTK_JUSTIFY_RIGHT);
  gtk_box_pack_start(GTK_BOX(title_hbox), title_label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(title_hbox),
		     gtk_label_new(_("@")), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(title_hbox),
		gtk_label_new(ochusha_bulletin_board_get_name(thread->board)),
		FALSE, FALSE, 0);
  board_label = gtk_label_new(_("board"));
  gtk_label_set_justify(GTK_LABEL(board_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(title_hbox), board_label, FALSE, FALSE, 5);

  gtk_widget_show_all(title_hbox);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), title_hbox, FALSE, FALSE, 0);


  url_hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(url_hbox), gtk_label_new("URL: "),
		     FALSE, FALSE, 0);
  url_entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(url_entry), ochusha_bbs_thread_get_url(thread));
  gtk_editable_set_editable(GTK_EDITABLE(url_entry), FALSE);
  GTK_WIDGET_UNSET_FLAGS(url_entry, GTK_CAN_FOCUS);
  gtk_box_pack_start(GTK_BOX(url_hbox), url_entry, TRUE, TRUE, 0);
  gtk_widget_show_all(url_hbox);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), url_hbox, FALSE, FALSE, 0);


  editor = response_editor_new();
  response_editor = RESPONSE_EDITOR(editor);

  if (thread_info->last_name != NULL)
    {
      response_editor_add_name(response_editor, thread_info->last_name);
      response_editor_set_name(response_editor, thread_info->last_name);
      has_name = TRUE;
    }

  if (thread_info->last_mail != NULL)
    {
      response_editor_add_mail(response_editor, thread_info->last_mail);
      response_editor_set_mail(response_editor, thread_info->last_mail);
      has_mail = TRUE;
    }

  if (board_info->last_name != NULL)
    {
      response_editor_add_name(response_editor, board_info->last_name);
      if (!has_name)
	{
	  response_editor_set_name(response_editor, board_info->last_name);
	  has_name = TRUE;
	}
    }

  if (board_info->last_mail != NULL)
    {
      response_editor_add_mail(response_editor, board_info->last_mail);
      if (!has_mail)
	{
	  response_editor_set_mail(response_editor, board_info->last_mail);
	  has_mail = TRUE;
	}
    }

  if (application->last_name != NULL)
    {
      response_editor_add_name(response_editor, application->last_name);
      if (!has_name)
	response_editor_set_name(response_editor, application->last_name);
    }

  if (application->last_mail != NULL)
    {
      response_editor_add_mail(response_editor, application->last_mail);
      if (!has_mail)
	response_editor_set_mail(response_editor, application->last_mail);
    }

  gtk_widget_show(editor);

  gtk_box_pack_start(GTK_BOX(dialog_vbox), editor, TRUE, TRUE, 0);

  g_object_set_data(G_OBJECT(write_dialog), "editor", editor);
  g_object_set_data(G_OBJECT(write_dialog), "thread", thread);

  return write_dialog;
}
