/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_ui.c,v 1.44 2003/12/12 21:43:52 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_table.h"
#include "ochusha_network_broker.h"
#include "ochusha_utils_2ch.h"
#include "worker.h"

#include "ochusha_ui.h"
#include "bbs_thread_ui.h"
#include "boardlist_ui.h"
#include "bulletin_board_ui.h"

#include "icon_label.h"
#include "paned_notebook.h"
#include "regex_utils.h"

#include "ugly_gtk2utils.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>

#include "write_response_24.h"

#include "blue0.xpm"
#include "blue1.xpm"
#include "blue2.xpm"
#include "blue3.xpm"
#include "orange.xpm"
#include "black.xpm"
#include "red.xpm"

#include "ochusha48.h"

#include "ochusha-banner.h"
#include "ochusha-banner1.h"


static void setup_ochusha_stock_items(void);
static void setup_tab_label_icons(void);
static void delete_event_cb(GtkWidget *widget, GdkEventAny *event,
			    gpointer user_data);
#if ENABLE_MAIN_TOOLBAR
static void go_forward_button_cb(gpointer unused,
				 OchushaApplication *application);
static void go_back_button_cb(gpointer unused,
			      OchushaApplication *application);
static void refresh_button_cb(gpointer unused,
			      OchushaApplication *application);
static void cancel_button_cb(gpointer unused, OchushaApplication *application);
#endif

static void on_quit(gpointer data, guint action, GtkWidget *widget);
static void main_menu_callback(gpointer data, guint action, GtkWidget *widget);
static void write_optional_prefs(FILE *file, gpointer user_data);
static void configure_application(OchushaApplication *application);
static void read_optional_prefs(GHashTable *pref_attrs, gpointer user_data);


static gboolean access_started_cb(OchushaNetworkBroker *broker,
				  OchushaAsyncBuffer *buffer,
				  OchushaApplication *application);
static gboolean access_progressed_cb(OchushaNetworkBroker *broker,
				     OchushaAsyncBuffer *buffer,
				     int bytes_read, int bytes_total,
				     OchushaApplication *application);
static gboolean access_completed_cb(OchushaNetworkBroker *broker,
				    OchushaAsyncBuffer *buffer,
				    OchushaApplication *application);
static gboolean access_terminated_cb(OchushaNetworkBroker *broker,
				     OchushaAsyncBuffer *buffer,
				     OchushaApplication *application);
static gboolean access_failed_cb(OchushaNetworkBroker *broker,
				 OchushaAsyncBuffer *buffer,
				 OchushaNetworkBrokerFailureReason reason_code,
				 const gchar *reason,
				 OchushaApplication *application);
static void output_log_cb(OchushaNetworkBroker *broker, const gchar *message,
			  OchushaApplication *application);

#if ENABLE_DEBUG_CONSOLE
static void open_debug_console(OchushaApplication *application);
#endif
static void open_preference_dialog(OchushaApplication *application);
static void open_about_dialog(OchushaApplication *application);
static void about_dialog_response_cb(GtkWidget *dialog, gint response_id,
				     gpointer unused);

static void initialize_open_url_dialog(OchushaApplication *application);
static void open_url_combo_entry_activate_cb(GtkWidget *entry,
					     GtkDialog *dialog);
static void open_url_dialog_response_cb(GtkWidget *dialog, gint response_id,
					OchushaApplication *application);
static void popup_open_url_dialog(OchushaApplication *application);


static GtkItemFactory *main_menu_item_factory;
static GdkPixbuf *tab_label_icons[7];
static GQuark tab_label_id;
static GQuark tab_label_icon_counter_id;

static pthread_mutex_t main_ui_lock;

#define MAIN_UI_LOCK					\
  if (pthread_mutex_lock(&main_ui_lock) != 0)		\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define MAIN_UI_UNLOCK					\
  if (pthread_mutex_unlock(&main_ui_lock) != 0)		\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

static GtkWidget *preference_dialog = NULL;
static GtkWidget *about_dialog = NULL;
static GdkPixbuf *banner_pixbuf = NULL;
static GdkPixbuf *banner1_pixbuf = NULL;
static GtkWidget *open_url_dialog = NULL;
static GtkWidget *open_url_combo = NULL;
static GtkWidget *open_url_in_tab_button = NULL;
static GList *open_url_history = NULL;

#if ENABLE_DEBUG_CONSOLE
static GtkWidget *debug_console = NULL;
static GtkTextBuffer *log_buffer = NULL;
static GtkTextIter log_tail_iter;
#endif


void
ochusha_init_application(OchushaApplication *application)
{
  GtkItemFactoryEntry menu_items[] = {
    {
      _("/_File"),		/* menu path */
      NULL,			/* accelerator */
      NULL,			/* callback_func */
      0,			/* act */
      "<Branch>"		/* type */
    },
    {
      _("/File/_Open"),
      NULL,
      main_menu_callback,
      MAIN_MENU_OPEN_URL,
      NULL
    },
    {
      _("/File/-------"),
      NULL,
      NULL,
      0,
      "<Separator>"
    },
    {
      _("/File/_Quit"),
      "<control>Q",
      on_quit,
      MAIN_MENU_QUIT,
      NULL
    },
    {
      _("/_Board"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
    {
      _("/Board/_Refresh"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_BOARDLIST,
      NULL
    },
    {
      _("/Board/Hide Hidden Boardlist Items"),
      NULL,
      main_menu_callback,
      MAIN_MENU_HIDE_HIDDEN_BOARDLIST_ITEMS,
      "<CheckItem>",
    },
    {
      _("/_Thread"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
    {
      _("/Thread/_Refresh"),
      "F5",
      main_menu_callback,
      MAIN_MENU_REFRESH_THREAD,
      NULL
    },
    {
      _("/Thread/Refresh _All"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREAD_ALL,
      NULL
    },
    {
      _("/Thread/-------"),
      NULL,
      NULL,
      0,
      "<Separator>"
    },
    {
      _("/Thread/Refresh _Threadlist"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREADLIST,
      NULL
    },
    {
      _("/Thread/Refresh Threadlist All"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREADLIST_ALL,
      NULL
    },
    {
      _("/Thread/Hide Hidden Threads"),
      NULL,
      main_menu_callback,
      MAIN_MENU_HIDE_HIDDEN_THREADS,
      "<CheckItem>",
    },
    {
      _("/Tools"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
#if ENABLE_DEBUG_CONSOLE
    {
      _("/Tools/Debug Console"),
      NULL,
      main_menu_callback,
      MAIN_MENU_DEBUG_CONSOLE,
      NULL,
    },
    {
      _("/Tools/-------"),
      NULL,
      NULL,
      0,
      "<Separator>"
    },
#endif
    {
      _("/Tools/Options"),
      NULL,
      main_menu_callback,
      MAIN_MENU_OPTIONS,
      NULL,
    },
    {
      _("/_Help"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
    {
      _("/Help/_About"),
      NULL,
      main_menu_callback,
      MAIN_MENU_ABOUT,
      NULL
    }
  };
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  GdkGeometry geometry = { 0, 0, 0, 0, 0, 0, 0, 0, 0.0, 0.0, 0 };
  GtkCheckMenuItem *item;

  if (pthread_mutex_init(&main_ui_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  setup_ochusha_stock_items();
  setup_tab_label_icons();

  banner_pixbuf
    = gdk_pixbuf_new_from_inline(-1, ochusha_banner_inline, FALSE, NULL);
  banner1_pixbuf
    = gdk_pixbuf_new_from_inline(-1, ochusha_banner1_inline, FALSE, NULL);

  configure_application(application);

  application->top_level = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));
  g_signal_connect(G_OBJECT(application->top_level), "delete_event",
		   G_CALLBACK(delete_event_cb), application);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->top_level), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &application->top_level);
#endif

  gtk_window_set_title(application->top_level, _("ochusha"));

  gtk_window_set_icon(application->top_level,
		      gdk_pixbuf_new_from_inline(-1, ochusha48_inline, FALSE, NULL));

  application->main_window = GTK_BOX(gtk_vbox_new(FALSE, 0));
  gtk_container_add(GTK_CONTAINER(application->top_level),
		    GTK_WIDGET(application->main_window));
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->main_window), "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &application->main_window);
#endif

  gtk_window_set_geometry_hints(application->top_level,
				GTK_WIDGET(application->main_window),
				&geometry, GDK_HINT_MIN_SIZE);
  gtk_widget_set_size_request(GTK_WIDGET(application->top_level),
			      application->width, application->height);

  /* menu barŹ */
  accel_group = gtk_accel_group_new();
  main_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
						accel_group);
  gtk_item_factory_create_items(main_menu_item_factory,
				nmenu_items, menu_items, application);
  gtk_window_add_accel_group(application->top_level, accel_group);

  application->menu_bar = gtk_item_factory_get_widget(main_menu_item_factory,
						      "<main>");
  /* ̤ʵǽΥ˥塼򻦤 */
  ochusha_set_main_menu_sensitive(MAIN_MENU_REFRESH_THREAD_ALL, FALSE);
#if 0
  ochusha_set_main_menu_sensitive(MAIN_MENU_DEBUG_CONSOLE, FALSE);
  ochusha_set_main_menu_sensitive(MAIN_MENU_OPTIONS, FALSE);
#endif

  item = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget_by_action(
				main_menu_item_factory,
				MAIN_MENU_HIDE_HIDDEN_BOARDLIST_ITEMS));
  gtk_check_menu_item_set_active(item,
				 application->hide_hidden_boardlist_items);
  item = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget_by_action(
				main_menu_item_factory,
				MAIN_MENU_HIDE_HIDDEN_THREADS));
  gtk_check_menu_item_set_active(item,
				 application->hide_hidden_threads);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->menu_bar), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &application->menu_bar);
#endif

  gtk_box_pack_start(application->main_window,
		     application->menu_bar, FALSE, TRUE, 0);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->menu_bar), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &application->menu_bar);
#endif

#if ENABLE_MAIN_TOOLBAR
  /* toolbarŹ */
  application->toolbar = GTK_TOOLBAR(gtk_toolbar_new());
  gtk_toolbar_set_style(application->toolbar, GTK_TOOLBAR_ICONS);
  gtk_box_pack_start(application->main_window,
		     GTK_WIDGET(application->toolbar),
		     FALSE, TRUE, 0);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->toolbar), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &application->toolbar);
#endif

  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_GO_FORWARD,
			   _("Go next page"),
			   "go_next_page",
			   GTK_SIGNAL_FUNC(go_forward_button_cb),
			   application,
			   0);
  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_GO_BACK,
			   _("Go next page"),
			   "go_next_page",
			   GTK_SIGNAL_FUNC(go_back_button_cb),
			   application,
			   0);
  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_REFRESH,
			   _("Refresh current page"),
			   "refresh_current_page",
			   GTK_SIGNAL_FUNC(refresh_button_cb),
			   application,
			   -1);
  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_CANCEL,
			   _("Cancel current request"),
			   "cancel_current_request",
			   GTK_SIGNAL_FUNC(cancel_button_cb),
			   application,
			   -1);
  application->url_box = GTK_COMBO(gtk_combo_new());
  gtk_toolbar_append_widget(application->toolbar,
			    GTK_WIDGET(application->url_box),
			    _("Input URL to show"),
			    "input_url_to_show");
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->url_box), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &application->url_box);
#endif
#endif	/* ENABLE_MAIN_TOOLBAR */

  /* ν */
  initialize_open_url_dialog(application);


  /* GUIν */
  prepare_boardlist_ui_initialization(application);
  prepare_board_ui_initialization(application);
  prepare_thread_ui_initialization(application);


  /* İν */
  initialize_boardlist(application);

  gtk_box_pack_start(application->main_window,
		     application->contents_window, TRUE, TRUE, 0);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->contents_window), "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &application->contents_window);
#endif

  application->statusbar = GTK_STATUSBAR(gtk_statusbar_new());

  initialize_board_ui(application);
  initialize_thread_ui(application);

  gtk_box_pack_end(application->main_window,
		   GTK_WIDGET(application->statusbar), FALSE, TRUE, 0);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->statusbar), "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &application->statusbar);
#endif

  application->clipboard = NULL;

#if ENABLE_DEBUG_CONSOLE
  log_buffer = gtk_text_buffer_new(NULL);
  gtk_text_buffer_get_start_iter(log_buffer, &log_tail_iter);
#endif

  gtk_widget_show_all(GTK_WIDGET(application->top_level));
}


void
ochusha_set_main_menu_sensitive(OchushaMainMenuAction action,
				gboolean sensitive)
{
  GtkWidget *item;
  item = gtk_item_factory_get_widget_by_action(main_menu_item_factory,
					       action);
  if (item != NULL)
    gtk_widget_set_sensitive(item, sensitive);
}


void
ochusha_clipboard_set_text(OchushaApplication *application, const gchar *text)
{
  MAIN_UI_LOCK
  {
    if (application->clipboard == NULL)
      application->clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
#if DEBUG_GUI_MOST
    fprintf(stderr, "setting text on clipboard: \"%s\"\n", text);
#endif
    gtk_clipboard_set_text(application->clipboard, text, -1);
  }
  MAIN_UI_UNLOCK
}


#define COMMAND_LINE_BUFFER_SIZE	4096
void
ochusha_open_url(OchushaApplication *application, const gchar *url,
		 gboolean in_tab, gboolean use_web_browser)
{
  OchushaBulletinBoard *board;
  char *board_url = NULL;
  OchushaBBSType bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
  char *thread_id = NULL;
  unsigned int from = 0;
  unsigned int to = 0;
  char *scheme;
  char command_line[COMMAND_LINE_BUFFER_SIZE];

  if (!use_web_browser
      && ochusha_bbs_table_check_url(application->table, url,
				     &board, &board_url, &bbs_type,
				     &thread_id, &from, &to))
    {
#if 0
      if (board == NULL)
	{
	  /* TODO: ɲåɤߤʤΤư*/
	  fprintf(stderr, "ochusha_open_url()\n");
	  fprintf(stderr, "  url: %s\n", url);
	  fprintf(stderr, "  board_url: %s\n", board_url);
	  fprintf(stderr, "  bbs_type: %d\n", bbs_type);
	  if (thread_id != NULL)
	    fprintf(stderr, "  thread_id: %s\n", thread_id);
	  fprintf(stderr, "  from=%d, to=%d\n", from, to);
	}
#endif

      if (board != NULL)
	{
	  /* ɽ */
	  if (thread_id != NULL)
	    {
	      ochusha_open_thread(application, board, thread_id, from, in_tab);
	    }
	  else
	    {
	      paned_notebook_open_item_view(PANED_NOTEBOOK(application->contents_window),
					    board, in_tab);
	    }

	  if (board_url != NULL)
	    G_FREE(board_url);
	  if (thread_id != NULL)
	    G_FREE(thread_id);
	  return;
	}
    }

  if (board_url != NULL)
    G_FREE(board_url);
  if (thread_id != NULL)
    G_FREE(thread_id);

  scheme = ochusha_utils_url_extract_scheme(url);

  if (scheme == NULL)
    return;

  /* TODO: scheme̤˵ư볰ץѹǽˤ٤ */

  G_FREE(scheme);

  if (snprintf(command_line, COMMAND_LINE_BUFFER_SIZE,
	       application->web_browser_template, url)
      < COMMAND_LINE_BUFFER_SIZE)
    {
      GError *error = NULL;
      char *native_command_line
	= g_locale_from_utf8(command_line, -1, NULL, NULL, NULL);

      /*
       * MEMO: UTF-8Ǥ'~'E280BEˤʤäƤꡢ줬ޤޤ줿URL
       *       mozillaϤƤŪΥڡɽʤȤ
       *       餷(̤ǧδĶǤɽƤ褦ʵ)
       *       Τᤳnative localeλꤹencodingѴ롣
       *
       *       ȥѥå by Motonobu Ichimura <famao@momonga-linux.org>
       *
       * XXX:  UTF-8ʸnativeencodingѴǽȤϸ¤ʤ
       *       Τϥ顼åȤкפ롣
       */
      if (!g_spawn_command_line_async(native_command_line, &error))
	{
	  /* TODO: 顼ɽ */
	  fprintf(stderr, "spawning error: %s (%d)\n",
		  error->message, error->code);
	}
      g_free(native_command_line);
    }

  return;
}


#if ENABLE_MAIN_TOOLBAR
void
set_current_url(OchushaApplication *application, const gchar *url)
{
  g_return_if_fail(application->url_box != NULL);

  gtk_entry_set_text(GTK_ENTRY(application->url_box->entry), url);
}


const gchar *
get_current_url(OchushaApplication *application)
{
  g_return_val_if_fail(application->url_box != NULL, NULL);

  return gtk_entry_get_text(GTK_ENTRY(application->url_box->entry));
}
#endif


static void
invalidate_tab_label_animation(OchushaAsyncBuffer *buffer)
{
  g_object_set_qdata(G_OBJECT(buffer), tab_label_id, NULL);
}


void
setup_for_tab_label_animation(OchushaAsyncBuffer *buffer, IconLabel *tab_label)
{
  g_return_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer)
		   && IS_ICON_LABEL(tab_label));
  g_signal_connect(G_OBJECT(tab_label), "destroy",
		   G_CALLBACK(invalidate_tab_label_animation),
		   buffer);
  g_object_set_qdata(G_OBJECT(buffer), tab_label_id, tab_label);

  if (buffer->fixed)
    icon_label_set_icon(tab_label, tab_label_icons[TAB_LABEL_ICON_DONE]);
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


GtkWidget *
get_tab_label(const gchar *label_text)
{
  GtkWidget *label = icon_label_new(label_text,
				tab_label_icons[TAB_LABEL_ICON_PREPARATION]);
  g_object_set_qdata_full(G_OBJECT(label), tab_label_icon_counter_id,
			  G_NEW(int, 1), (GDestroyNotify)TRACE_FREE);
  return label;
}


static void
add_stock_icon(GtkIconFactory *factory, const char *stock_id,
	       GdkPixbuf *pixbuf, GtkIconSize size)
{
  GtkIconSource *source = gtk_icon_source_new();
  GtkIconSet *set = gtk_icon_set_new();

  gtk_icon_source_set_size(source, size);
  gtk_icon_source_set_pixbuf(source, pixbuf);

#if 0	/* ׽ */
  gtk_icon_source_set_size_wildcarded(source, FALSE);
  gtk_icon_set_add_source(set, source);
#endif

  gtk_icon_source_set_size_wildcarded(source, TRUE);
  gtk_icon_set_add_source(set, source);

  gtk_icon_source_free(source);

  gtk_icon_factory_add(factory, stock_id, set);

  gtk_icon_set_unref(set);
  g_object_unref(pixbuf);	/* ݤʤΤǤunrefƤޤ */
}


static const GtkStockItem ochusha_stock_items[] =
{
  { OCHUSHA_STOCK_WRITE_RESPONSE, N_("_Write"), 0, 0, PACKAGE_NAME }
};


static void
setup_ochusha_stock_items(void)
{
  GtkIconFactory *factory = gtk_icon_factory_new();

  add_stock_icon(factory, OCHUSHA_STOCK_WRITE_RESPONSE,
		 gdk_pixbuf_new_from_inline(-1, write_response_24,
					    FALSE, NULL),
		 GTK_ICON_SIZE_LARGE_TOOLBAR);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_PROGRESS0,
		 gdk_pixbuf_new_from_xpm_data((const char **)blue0_xpm),
		 GTK_ICON_SIZE_MENU);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_PROGRESS1,
		 gdk_pixbuf_new_from_xpm_data((const char **)blue1_xpm),
		 GTK_ICON_SIZE_MENU);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_PROGRESS2,
		 gdk_pixbuf_new_from_xpm_data((const char **)blue2_xpm),
		 GTK_ICON_SIZE_MENU);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_PROGRESS3,
		 gdk_pixbuf_new_from_xpm_data((const char **)blue3_xpm),
		 GTK_ICON_SIZE_MENU);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_PREPARATION,
		 gdk_pixbuf_new_from_xpm_data((const char **)orange_xpm),
		 GTK_ICON_SIZE_MENU);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_DONE,
		 gdk_pixbuf_new_from_xpm_data((const char **)black_xpm),
		 GTK_ICON_SIZE_MENU);

  add_stock_icon(factory, OCHUSHA_STOCK_NET_ERROR,
		 gdk_pixbuf_new_from_xpm_data((const char **)red_xpm),
		 GTK_ICON_SIZE_MENU);

  gtk_icon_factory_add_default(factory);

  g_object_unref(factory);

  gtk_stock_add_static(ochusha_stock_items, G_N_ELEMENTS(ochusha_stock_items));
}


static void
setup_tab_label_icons(void)
{
  GtkWidget *dummy = icon_label_new("", NULL);

  tab_label_icons[TAB_LABEL_ICON_PROGRESS0]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_PROGRESS0,
			     GTK_ICON_SIZE_MENU, "ochusha::net-progress-0");

  tab_label_icons[TAB_LABEL_ICON_PROGRESS1]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_PROGRESS1,
			     GTK_ICON_SIZE_MENU, "ochusha::net-progress-1");

  tab_label_icons[TAB_LABEL_ICON_PROGRESS2]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_PROGRESS2,
			     GTK_ICON_SIZE_MENU, "ochusha::net-progress-2");

  tab_label_icons[TAB_LABEL_ICON_PROGRESS3]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_PROGRESS3,
			     GTK_ICON_SIZE_MENU, "ochusha::net-progress-3");

  tab_label_icons[TAB_LABEL_ICON_PREPARATION]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_PREPARATION,
			     GTK_ICON_SIZE_MENU, "ochusha::net-preparation");

  tab_label_icons[TAB_LABEL_ICON_DONE]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_DONE,
			     GTK_ICON_SIZE_MENU, "ochusha::net-done");

  tab_label_icons[TAB_LABEL_ICON_ERROR]
    = gtk_widget_render_icon(dummy, OCHUSHA_STOCK_NET_ERROR,
			     GTK_ICON_SIZE_MENU, "ochusha::net-error");

  gtk_object_sink(GTK_OBJECT(dummy));

  tab_label_id = g_quark_from_static_string("OchushaUI::icon-label");
  tab_label_icon_counter_id
    = g_quark_from_static_string("OchushaUI::icon-counter");
}


static void
delete_event_cb(GtkWidget *widget, GdkEventAny *event, gpointer user_data)
{
  on_quit(user_data, 0, widget);
}


#if ENABLE_MAIN_TOOLBAR
static void
go_forward_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "go_forward_button_cb: not implemented.\n");
#endif
}


static void
go_back_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "go_forward_button_cb: not implemented.\n");
#endif
}


static void
refresh_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "refresh_button_cb: not implemented.\n");
#endif
}


static void
cancel_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "cancel_button_cb: not implemented.\n");
#endif
}
#endif


static void
on_quit(gpointer data, guint action, GtkWidget *widget)
{
  OchushaApplication *application = (OchushaApplication *)data;
  GtkAllocation *allocation = &GTK_WIDGET(application->top_level)->allocation;

  application->width = allocation->width;
  application->height = allocation->height;

  allocation = &application->boardlist_view->allocation;
  application->boardlist_width = allocation->width;

  finalize_boardlist(application);

  if (!ochusha_write_config_xml(&application->config,
				write_optional_prefs, application))
    fprintf(stderr, "Couldn't write %s.\n", OCHUSHA_CONFIG_XML);

  gtk_main_quit();
}


static void
main_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  OchushaApplication *application = (OchushaApplication *)data;
  switch (action)
    {
    case MAIN_MENU_OPEN_URL:
      popup_open_url_dialog(application);
      return;

    case MAIN_MENU_REFRESH_BOARDLIST:
      refresh_boardlist(application);
      return;

    case MAIN_MENU_HIDE_HIDDEN_BOARDLIST_ITEMS:
    case MAIN_MENU_HIDE_HIDDEN_THREADS:
      {
	GtkCheckMenuItem *item
	  = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget_by_action(
							main_menu_item_factory,
							action));
	if (action == MAIN_MENU_HIDE_HIDDEN_BOARDLIST_ITEMS)
	  {
	    application->hide_hidden_boardlist_items
	      = gtk_check_menu_item_get_active(item);
	    if (application->boardlist_view != NULL)
	      redraw_boardlist(application);
	  }
	else
	  application->hide_hidden_threads
	    = gtk_check_menu_item_get_active(item);

	return;
      }

    case MAIN_MENU_REFRESH_THREAD:
      refresh_current_thread(application);
      return;

#if 0
    case MAIN_MENU_REFRESH_THREAD_ALL:
      refresh_current_thread_all(application);
      return;
#endif

    case MAIN_MENU_REFRESH_THREADLIST:
      refresh_current_threadlist(application);
      return;

    case MAIN_MENU_REFRESH_THREADLIST_ALL:
      refresh_current_threadlist_all(application);
      return;

#if ENABLE_DEBUG_CONSOLE
    case MAIN_MENU_DEBUG_CONSOLE:
      open_debug_console(application);
      return;
#endif

    case MAIN_MENU_OPTIONS:
      open_preference_dialog(application);
      return;

    case MAIN_MENU_ABOUT:
      open_about_dialog(application);
      return;
    }

  fprintf(stderr, "Not implemented yet.\n");
}


#define OUTPUT_CONFIG_ATTRIBUTE_INT(file, config, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <int val=\"%d\"/>\n"				\
	  "    </attribute>\n", (config)->attribute)

#define OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, config, attribute)	\
  fprintf(file,								\
	  "    <attribute name=\"" #attribute	"\">\n"			\
	  "      <boolean val=\"%s\"/>\n"				\
	  "    </attribute>\n", (config)->attribute ? "true" : "false" )

#define OUTPUT_CONFIG_ATTRIBUTE_STRING(file, config, attribute)		\
  do {									\
    if ((config)->attribute != NULL)					\
      {									\
	gchar *text = g_markup_escape_text((config)->attribute, -1);	\
	fprintf(file,							\
		"    <attribute name=\"" #attribute	"\">\n"		\
		"      <string>%s</string>\n"				\
		"    </attribute>\n", text);				\
	g_free(text);							\
      }									\
  } while (0)


static void
write_optional_prefs(FILE *file, gpointer user_data)
{
  OchushaApplication *application = (OchushaApplication *)user_data;

  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, width);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, height);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, boardlist_width);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, threadlist_height);

#if ENABLE_RENDERING_ON_IDLE
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, rendering_unit);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application,
			      weight_of_threadlist_rendering);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, weight_of_response_rendering);
#endif

  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, boardlist_pane_style);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, board_tab_shrinkable);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, board_tab_minimum_size);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  board_tab_enable_tooltips);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  board_tab_always_show);
  
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, threadlist_pane_style);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, thread_tab_shrinkable);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, thread_tab_minimum_size);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  thread_tab_enable_tooltips);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  thread_tab_always_show);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  show_threadlist_entry_when_thread_selected);
  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, threadlist_view_contents);

  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, thread_view_font_name);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  enable_transparent_a_bone);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, disable_popup_a_bone);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, default_open_in_tab);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, show_mailto_mode);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, a_bone_by_name);
  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, a_bone_by_name_pattern);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, a_bone_by_id);
  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, a_bone_by_id_pattern);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, a_bone_by_content);
  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application,
				 a_bone_by_content_pattern);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, enable_popup_title);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, popup_title_delay);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, enable_popup_response);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, popup_response_delay);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, popup_close_delay);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  enable_smart_popup_placement);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application,
			      maximum_number_of_popup_responses);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  analyze_informal_links);

  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, web_browser_template);

  if (application->user_default_filter.rule != NULL)
    OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application,
				   user_default_filter.rule);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application,
			      user_default_filter.ignore_threshold);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  hide_hidden_boardlist_items);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, hide_hidden_threads);

  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, last_name);
  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, last_mail);
}


static void
configure_application(OchushaApplication *application)
{
  ochusha_read_config_xml(&application->config, read_optional_prefs,
			  application);
  application->broker = ochusha_network_broker_new(&application->config);
  g_signal_connect(G_OBJECT(application->broker), "access_started",
		   G_CALLBACK(access_started_cb), application);
  g_signal_connect(G_OBJECT(application->broker), "access_progressed",
		   G_CALLBACK(access_progressed_cb), application);
  g_signal_connect(G_OBJECT(application->broker), "access_completed",
		   G_CALLBACK(access_completed_cb), application);
  g_signal_connect(G_OBJECT(application->broker), "access_terminated",
		   G_CALLBACK(access_terminated_cb), application);
  g_signal_connect(G_OBJECT(application->broker), "access_failed",
		   G_CALLBACK(access_failed_cb), application);
  g_signal_connect(G_OBJECT(application->broker), "output_log",
		   G_CALLBACK(output_log_cb), application);
  /* Sets unspecified attributes */
  if (application->width == 0)
    {	/* ƵưȻפ */
      /* ɥν */
      application->width = OCHUSHA_DEFAULT_WIDTH;
      application->height = OCHUSHA_DEFAULT_HEIGHT;
      application->boardlist_width = OCHUSHA_DEFAULT_BOARDLIST_WIDTH;
      application->threadlist_height = OCHUSHA_DEFAULT_THREADLIST_HEIGHT;

#if ENABLE_RENDERING_ON_IDLE
      application->rendering_unit
	= OCHUSHA_DEFAULT_RENDERING_UNIT;
      application->weight_of_threadlist_rendering
	= OCHUSHA_DEFAULT_WEIGHT_OF_THREADLIST_RENDERING;
      application->weight_of_response_rendering
	= OCHUSHA_DEFAULT_WEIGHT_OF_RESPONSE_RENDERING;
#endif

      /* İϢ */
      application->boardlist_pane_style = HPANED_STYLE;
      application->board_tab_shrinkable = OCHUSHA_DEFAULT_BOARD_TAB_SHRINKABLE;
      application->board_tab_minimum_size
	= OCHUSHA_DEFAULT_BOARD_TAB_MINIMUM_SIZE;
      application->board_tab_enable_tooltips = FALSE;


      /* Ϣ */
      application->threadlist_pane_style = VPANED_STYLE;
      application->thread_tab_shrinkable
	= OCHUSHA_DEFAULT_THREAD_TAB_SHRINKABLE;
      application->thread_tab_minimum_size
	= OCHUSHA_DEFAULT_THREAD_TAB_MINIMUM_SIZE;
      application->thread_tab_enable_tooltips = TRUE;
      application->show_threadlist_entry_when_thread_selected = FALSE;


      /* ɽϢ */
      application->thread_view_font_name = NULL;
      application->enable_transparent_a_bone = FALSE;
      application->disable_popup_a_bone = FALSE;

      application->default_open_in_tab = FALSE;
      application->show_mailto_mode = OCHUSHA_DEFAULT_SHOW_MAILTO_MODE;
      application->a_bone_by_name = FALSE;
      application->a_bone_by_name_pattern = G_STRDUP("");
      application->a_bone_by_id = FALSE;
      application->a_bone_by_id_pattern = G_STRDUP("");
      application->a_bone_by_content = FALSE;
      application->a_bone_by_content_pattern = G_STRDUP("");


      /* 쥿Υݥåץå״Ϣ */
      application->enable_popup_title = OCHUSHA_DEFAULT_ENABLE_POPUP_TITLE;
      application->popup_title_delay = OCHUSHA_DEFAULT_POPUP_DELAY;


      /* 쥹Υݥåץå״Ϣ */
      application->enable_popup_response
	= OCHUSHA_DEFAULT_ENABLE_POPUP_RESPONSE;
      application->popup_response_delay = OCHUSHA_DEFAULT_POPUP_DELAY;
      application->popup_close_delay = OCHUSHA_DEFAULT_POPUP_CLOSE_DELAY;
      application->enable_smart_popup_placement
	= OCHUSHA_DEFAULT_ENABLE_SMART_POPUP_PLACEMENT;
      application->maximum_number_of_popup_responses
	= OCHUSHA_DEFAULT_MAXIMUM_NUMBER_OF_POPUP_RESPONSES;
      application->analyze_informal_links
	= OCHUSHA_DEFAULT_ANALYZE_INFORMAL_LINKS;

      application->show_threadlist_entry_when_thread_selected
	= OCHUSHA_DEFAULT_SHOW_THREADLIST_ENTRY_WHEN_THREAD_SELECTED;
      application->hide_hidden_boardlist_items
	= OCHUSHA_DEFAULT_HIDE_HIDDEN_BOARDLIST_ITEMS;
      application->hide_hidden_threads
	= OCHUSHA_DEFAULT_HIDE_HIDDEN_THREADS;
    }

  if (application->threadlist_view_contents == NULL)
    application->threadlist_view_contents
      = G_STRDUP(OCHUSHA_DEFAULT_THREADLIST_VIEW_CONTENTS);

#if ENABLE_RENDERING_ON_IDLE
  if (application->rendering_unit < 0)
    application->rendering_unit = 0;

  if (application->weight_of_threadlist_rendering < 0)
    application->weight_of_threadlist_rendering = 0;

  if (application->weight_of_response_rendering < 0)
    application->weight_of_response_rendering = 0;
#endif


  if (application->board_tab_minimum_size == 0)
    application->board_tab_minimum_size
      = OCHUSHA_DEFAULT_BOARD_TAB_MINIMUM_SIZE;

  if (application->thread_tab_minimum_size == 0)
    application->thread_tab_minimum_size
      = OCHUSHA_DEFAULT_THREAD_TAB_MINIMUM_SIZE;

  if (application->popup_title_delay == 0)
    application->popup_title_delay = OCHUSHA_DEFAULT_POPUP_DELAY;

  if (application->web_browser_template == NULL)
    application->web_browser_template
      = G_STRDUP(OCHUSHA_DEFAULT_WEB_BROWSER_TEMPLATE);
}


static void
read_optional_prefs(GHashTable *pref_attrs, gpointer user_data)
{
  OchushaApplication *application = (OchushaApplication *)user_data;

  application->width = ochusha_utils_get_attribute_int(pref_attrs, "width");

  application->height = ochusha_utils_get_attribute_int(pref_attrs, "height");

  application->boardlist_width
    = ochusha_utils_get_attribute_int(pref_attrs, "boardlist_width");

  application->threadlist_height
    = ochusha_utils_get_attribute_int(pref_attrs, "threadlist_height");

#if ENABLE_RENDERING_ON_IDLE
  if (g_hash_table_lookup(pref_attrs, "rendering_unit") != NULL)
    application->rendering_unit
      = ochusha_utils_get_attribute_int(pref_attrs, "rendering_unit");
  else
    application->rendering_unit = OCHUSHA_DEFAULT_RENDERING_UNIT;

  if (g_hash_table_lookup(pref_attrs,
			  "weight_of_threadlist_rendering") != NULL)
    application->weight_of_threadlist_rendering
      = ochusha_utils_get_attribute_int(pref_attrs,
					"weight_of_threadlist_rendering");
  else
    application->weight_of_threadlist_rendering
      = OCHUSHA_DEFAULT_WEIGHT_OF_THREADLIST_RENDERING;
  
  if (g_hash_table_lookup(pref_attrs, "weight_of_response_rendering") != NULL)
    application->weight_of_response_rendering
      = ochusha_utils_get_attribute_int(pref_attrs,
					"weight_of_response_rendering");
  else
    application->weight_of_response_rendering
      = OCHUSHA_DEFAULT_WEIGHT_OF_RESPONSE_RENDERING;
#endif

  if (g_hash_table_lookup(pref_attrs, "boardlist_pane_style") != NULL)
    application->boardlist_pane_style
      = ochusha_utils_get_attribute_int(pref_attrs, "boardlist_pane_style");
  else
    application->boardlist_pane_style = HPANED_STYLE;

  if (g_hash_table_lookup(pref_attrs, "board_tab_shrinkable") != NULL)
    application->board_tab_shrinkable
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "board_tab_shrinkable");
  else
    application->board_tab_shrinkable = OCHUSHA_DEFAULT_BOARD_TAB_SHRINKABLE;

  if (g_hash_table_lookup(pref_attrs, "board_tab_minimum_size") != NULL)
    application->board_tab_minimum_size
      = ochusha_utils_get_attribute_int(pref_attrs, "board_tab_minimum_size");
  else
    application->board_tab_minimum_size
      = OCHUSHA_DEFAULT_BOARD_TAB_MINIMUM_SIZE;

  if (g_hash_table_lookup(pref_attrs, "board_tab_enable_tooltips") != NULL)
    application->board_tab_enable_tooltips
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "board_tab_enable_tooltips");
  else
    application->board_tab_enable_tooltips
      = OCHUSHA_DEFAULT_BOARD_TAB_ENABLE_TOOLTIPS;

  application->board_tab_always_show
    = ochusha_utils_get_attribute_boolean(pref_attrs, "board_tab_always_show");

  if (g_hash_table_lookup(pref_attrs, "threadlist_pane_style") != NULL)
    application->threadlist_pane_style
      = ochusha_utils_get_attribute_int(pref_attrs, "threadlist_pane_style");
  else
    application->threadlist_pane_style = VPANED_STYLE;

  if (g_hash_table_lookup(pref_attrs, "thread_tab_shrinkable") != NULL)
    application->thread_tab_shrinkable
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "thread_tab_shrinkable");
  else
    application->thread_tab_shrinkable = OCHUSHA_DEFAULT_THREAD_TAB_SHRINKABLE;

  if (g_hash_table_lookup(pref_attrs, "thread_tab_minimum_size") != NULL)
    application->thread_tab_minimum_size
      = ochusha_utils_get_attribute_int(pref_attrs, "thread_tab_minimum_size");
  else
    application->thread_tab_minimum_size
      = OCHUSHA_DEFAULT_THREAD_TAB_MINIMUM_SIZE;

  if (g_hash_table_lookup(pref_attrs, "thread_tab_enable_tooltips") != NULL)
    application->thread_tab_enable_tooltips
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "thread_tab_enable_tooltips");
  else
    application->thread_tab_enable_tooltips
      = OCHUSHA_DEFAULT_THREAD_TAB_ENABLE_TOOLTIPS;

  application->thread_tab_always_show
    = ochusha_utils_get_attribute_boolean(pref_attrs,
					  "thread_tab_always_show");

  if (g_hash_table_lookup(pref_attrs,
			  "show_threadlist_entry_when_thread_selected")
      != NULL)
    application->show_threadlist_entry_when_thread_selected
      = ochusha_utils_get_attribute_boolean(pref_attrs,
				"show_threadlist_entry_when_thread_selected");
  else
    application->show_threadlist_entry_when_thread_selected
      = OCHUSHA_DEFAULT_SHOW_THREADLIST_ENTRY_WHEN_THREAD_SELECTED;

  if (g_hash_table_lookup(pref_attrs, "threadlist_view_contents") != NULL)
    application->threadlist_view_contents
      = ochusha_utils_get_attribute_string(pref_attrs,
					   "threadlist_view_contents");
  else
    application->threadlist_view_contents
      = G_STRDUP(OCHUSHA_DEFAULT_THREADLIST_VIEW_CONTENTS);

  application->default_open_in_tab
    = ochusha_utils_get_attribute_boolean(pref_attrs, "default_open_in_tab");

  application->thread_view_font_name
    = ochusha_utils_get_attribute_string(pref_attrs, "thread_view_font_name");

  application->enable_transparent_a_bone
    = ochusha_utils_get_attribute_boolean(pref_attrs,
					  "enable_transparent_a_bone");
  application->disable_popup_a_bone
    = ochusha_utils_get_attribute_boolean(pref_attrs,
					  "disable_popup_a_bone");

  application->show_mailto_mode
    = (ThreadViewMailtoMode)ochusha_utils_get_attribute_int(pref_attrs,
							   "show_mailto_mode");
  if (application->show_mailto_mode < THREAD_VIEW_MAILTO_MODE_DEFAULT
      || application->show_mailto_mode > THREAD_VIEW_MAILTO_MODE_HIDE)
    application->show_mailto_mode = THREAD_VIEW_MAILTO_MODE_DEFAULT;

  application->a_bone_by_name
    = ochusha_utils_get_attribute_boolean(pref_attrs, "a_bone_by_name");
  application->a_bone_by_name_pattern
    = ochusha_utils_get_attribute_string(pref_attrs, "a_bone_by_name_pattern");
  if (application->a_bone_by_name_pattern == NULL)
    application->a_bone_by_name_pattern = G_STRDUP("");

  application->a_bone_by_id
    = ochusha_utils_get_attribute_boolean(pref_attrs, "a_bone_by_id");
  application->a_bone_by_id_pattern
    = ochusha_utils_get_attribute_string(pref_attrs, "a_bone_by_id_pattern");
  if (application->a_bone_by_id_pattern == NULL)
    application->a_bone_by_id_pattern = G_STRDUP("");

  application->a_bone_by_content
    = ochusha_utils_get_attribute_boolean(pref_attrs, "a_bone_by_content");
  application->a_bone_by_content_pattern
    = ochusha_utils_get_attribute_string(pref_attrs,
					 "a_bone_by_content_pattern");
  if (application->a_bone_by_content_pattern == NULL)
    application->a_bone_by_content_pattern = G_STRDUP("");

  if (g_hash_table_lookup(pref_attrs, "enable_popup_title") != NULL)
    application->enable_popup_title
      = ochusha_utils_get_attribute_boolean(pref_attrs, "enable_popup_title");
  else
    application->enable_popup_title = FALSE;

  application->popup_title_delay
    = ochusha_utils_get_attribute_int(pref_attrs, "popup_title_delay");

  if (g_hash_table_lookup(pref_attrs, "enable_popup_response") != NULL)
    application->enable_popup_response
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "enable_popup_response");
  else
    application->enable_popup_response = TRUE;

  if (g_hash_table_lookup(pref_attrs, "popup_response_delay") != NULL)
    application->popup_response_delay
      = ochusha_utils_get_attribute_int(pref_attrs, "popup_response_delay");
  else
    application->popup_response_delay = OCHUSHA_DEFAULT_POPUP_DELAY;

  if (g_hash_table_lookup(pref_attrs, "popup_close_delay") != NULL)
    application->popup_close_delay
      = ochusha_utils_get_attribute_int(pref_attrs, "popup_close_delay");
  else
    application->popup_close_delay = OCHUSHA_DEFAULT_POPUP_CLOSE_DELAY;

  if (g_hash_table_lookup(pref_attrs, "enable_smart_popup_placement") != NULL)
    application->enable_smart_popup_placement
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "enable_smart_popup_placement");
  else
    application->enable_smart_popup_placement
      = OCHUSHA_DEFAULT_ENABLE_SMART_POPUP_PLACEMENT;

  if (g_hash_table_lookup(pref_attrs, "maximum_number_of_popup_responses")
      != NULL)
    application->maximum_number_of_popup_responses
      = ochusha_utils_get_attribute_int(pref_attrs,
					"maximum_number_of_popup_responses");
  else
    application->maximum_number_of_popup_responses
      = OCHUSHA_DEFAULT_MAXIMUM_NUMBER_OF_POPUP_RESPONSES;

  if (g_hash_table_lookup(pref_attrs, "analyze_informal_links") != NULL)
    application->analyze_informal_links
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "analyze_informal_links");
  else
    application->analyze_informal_links
      = OCHUSHA_DEFAULT_ANALYZE_INFORMAL_LINKS;

  application->web_browser_template
    = ochusha_utils_get_attribute_string(pref_attrs, "web_browser_template");

  application->user_default_filter.rule
    = ochusha_utils_get_attribute_string(pref_attrs,
					 "user_default_filter.rule");

  application->user_default_filter.ignore_threshold
    = ochusha_utils_get_attribute_int(pref_attrs,
				      "user_default_filter.ignore_threshold");

  application->hide_hidden_boardlist_items
    = ochusha_utils_get_attribute_boolean(pref_attrs,
					  "hide_hidden_boardlist_items");
  application->hide_hidden_threads
    = ochusha_utils_get_attribute_boolean(pref_attrs, "hide_hidden_threads");

  application->last_name = ochusha_utils_get_attribute_string(pref_attrs,
							      "last_name");
  application->last_mail = ochusha_utils_get_attribute_string(pref_attrs,
							      "last_mail");
}


static gboolean
access_started_cb(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer,
		  OchushaApplication *application)
{
  int *counter;
  IconLabel *tab_label = g_object_get_qdata(G_OBJECT(buffer), tab_label_id);

#if DEBUG_GUI
  fprintf(stderr, "access_started_cb\n");
#endif

  if (tab_label == NULL)
    return TRUE;

  counter = g_object_get_qdata(G_OBJECT(tab_label), tab_label_icon_counter_id);
  *counter = 0;

  gdk_threads_enter();
  icon_label_set_icon(tab_label, tab_label_icons[TAB_LABEL_ICON_PREPARATION]);
  gdk_threads_leave();

  return TRUE;
}


static int progress_icon_number[7] =
{
  TAB_LABEL_ICON_PROGRESS0,
  TAB_LABEL_ICON_PROGRESS1,
  TAB_LABEL_ICON_PROGRESS2,
  TAB_LABEL_ICON_PROGRESS3,
  TAB_LABEL_ICON_PROGRESS2,
  TAB_LABEL_ICON_PROGRESS1,
  TAB_LABEL_ICON_PROGRESS0,
};


static gboolean
access_progressed_cb(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer,
		     int bytes_read, int bytes_total,
		     OchushaApplication *application)
{
  IconLabel *tab_label = g_object_get_qdata(G_OBJECT(buffer), tab_label_id);
  int *counter;
#if DEBUG_GUI_MOST
  fprintf(stderr, "access_progressed_cb\n");
#endif

  if (tab_label == NULL)
    return TRUE;

  counter = g_object_get_qdata(G_OBJECT(tab_label), tab_label_icon_counter_id);
  *counter = (*counter + 1) % 7;

  gdk_threads_enter();
  icon_label_set_icon(tab_label,
		      tab_label_icons[progress_icon_number[*counter]]);
  gdk_threads_leave();

  return TRUE;
}


static gboolean
access_completed_cb(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer,
		    OchushaApplication *application)
{
  IconLabel *tab_label = g_object_get_qdata(G_OBJECT(buffer), tab_label_id);

#if DEBUG_GUI
  fprintf(stderr, "access_completed_cb\n");
#endif

  if (tab_label == NULL)
    return TRUE;

  gdk_threads_enter();
  icon_label_set_icon(tab_label, tab_label_icons[TAB_LABEL_ICON_DONE]);
  gdk_threads_leave();

  return TRUE;
}


static gboolean
access_terminated_cb(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer,
		     OchushaApplication *application)
{
  IconLabel *tab_label = g_object_get_qdata(G_OBJECT(buffer), tab_label_id);

#if DEBUG_GUI
  fprintf(stderr, "access_terminated_cb\n");
#endif

  if (tab_label == NULL)
    return TRUE;

  gdk_threads_enter();
  icon_label_set_icon(tab_label, tab_label_icons[TAB_LABEL_ICON_DONE]);
  gdk_threads_leave();

  return TRUE;
}


static gboolean
access_failed_cb(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer,
		 OchushaNetworkBrokerFailureReason reason_code,
		 const gchar *reason, OchushaApplication *application)
{
  IconLabel *tab_label = g_object_get_qdata(G_OBJECT(buffer), tab_label_id);

#if DEBUG_GUI
  gchar *native_reason = g_locale_from_utf8(reason, -1, NULL, NULL, NULL);
  fprintf(stderr,
	  "access_failed_cb(): buffer=%p, reason_code=%d, reason=\"%s\"\n",
	  buffer, reason_code, native_reason);
  g_free(native_reason);
#endif

  if (tab_label == NULL)
    return TRUE;

  gdk_threads_enter();
  icon_label_set_icon(tab_label, tab_label_icons[TAB_LABEL_ICON_ERROR]);
  gdk_threads_leave();

  return TRUE;
}


static void
output_log_cb(OchushaNetworkBroker *broker, const gchar *message,
	      OchushaApplication *application)
{
#if 0
  fprintf(stderr, "LOG: %s", message);
#endif
#if ENABLE_DEBUG_CONSOLE
  g_return_if_fail(GTK_IS_TEXT_BUFFER(log_buffer));

  gtk_text_buffer_insert(log_buffer, &log_tail_iter, message, -1);
#endif
}


static void
preference_dialog_response_cb(GtkWidget *widget, gint response_id,
			      OchushaApplication *application)
{
  switch (response_id)
    {
    case GTK_RESPONSE_APPLY:
    case GTK_RESPONSE_OK:
      {
	PanedNotebook *boardlist_pane
	  = PANED_NOTEBOOK(application->contents_window);
	GtkWidget *current_board_pane
	  = paned_notebook_get_current_item_view(boardlist_pane);
	gpointer tmp_data;

	tmp_data = g_object_get_data(G_OBJECT(widget), "web_browser_template");
	if (tmp_data != NULL)
	  {
	    const gchar *template = gtk_entry_get_text(GTK_ENTRY(tmp_data));
	    if (template != NULL && *template != '\0')
	      {
		G_FREE(application->web_browser_template);
		application->web_browser_template = G_STRDUP(template);
	      }
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "board_tab_enable_tooltips");
	if (tmp_data != NULL)
	  {
	    application->board_tab_enable_tooltips
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	    paned_notebook_set_tooltips(boardlist_pane,
				application->board_tab_enable_tooltips);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "board_tab_always_show");
	if (tmp_data != NULL)
	  {
	    gboolean always_show
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	    application->board_tab_always_show = always_show;
	    paned_notebook_set_tab_policy(boardlist_pane,
					  always_show
					  ? GTK_POLICY_ALWAYS
					  : GTK_POLICY_AUTOMATIC);
	  }

#if 0	/* ɤ */
	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "boardlist_pane_style");
	if (tmp_data != NULL)
	  {
	    application->boardlist_pane_style
	      = (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data))
		 ? VPANED_STYLE : HPANED_STYLE);
	    paned_notebook_set_paned_style(boardlist_pane,
					   application->boardlist_pane_style);
	  }
#endif

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "threadlist_pane_style");
	if (tmp_data != NULL)
	  {
	    application->threadlist_pane_style
	      = (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data))
		 ? VPANED_STYLE : HPANED_STYLE);
	    if (current_board_pane != NULL)
	      paned_notebook_set_paned_style(PANED_NOTEBOOK(current_board_pane), application->threadlist_pane_style);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "enable_proxy");
	if (tmp_data != NULL)
	  {
	    application->config.enable_proxy
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "enable_proxy_only_for_posting");
	if (tmp_data != NULL)
	  {
	    application->config.enable_proxy_only_for_posting
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "proxy_url");
	if (tmp_data != NULL)
	  {
	    const gchar *url;
	    if (application->config.proxy_url != NULL)
	      G_FREE(application->config.proxy_url);
	    url = gtk_entry_get_text(GTK_ENTRY(tmp_data));
	    if (*url == '\0')
	      application->config.proxy_url = NULL;
	    else
	      application->config.proxy_url = G_STRDUP(url);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "proxy_user");
	if (tmp_data != NULL)
	  {
	    const gchar *name;
	    if (application->config.proxy_user != NULL)
	      G_FREE(application->config.proxy_user);
	    name = gtk_entry_get_text(GTK_ENTRY(tmp_data));
	    if (*name == '\0')
	      application->config.proxy_user = NULL;
	    else
	      application->config.proxy_user = G_STRDUP(name);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "proxy_password");
	if (tmp_data != NULL)
	  {
	    const gchar *password;
	    if (application->config.proxy_password != NULL)
	      G_FREE(application->config.proxy_password);
	    password = gtk_entry_get_text(GTK_ENTRY(tmp_data));
	    if (*password == '\0')
	      application->config.proxy_password = NULL;
	    else
	      application->config.proxy_password = G_STRDUP(password);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "enable_popup_title");
	if (tmp_data != NULL)
	  {
	    application->enable_popup_title
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "thread_tab_enable_tooltips");
	if (tmp_data != NULL)
	  {
	    application->thread_tab_enable_tooltips
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	    if (current_board_pane != NULL)
	      paned_notebook_set_tooltips(PANED_NOTEBOOK(current_board_pane),
				application->thread_tab_enable_tooltips);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "thread_tab_always_show");
	if (tmp_data != NULL)
	  {
	    gboolean always_show
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	    application->thread_tab_always_show = always_show;
	    if (current_board_pane != NULL)
	      paned_notebook_set_tab_policy(PANED_NOTEBOOK(current_board_pane),
					    always_show
					    ? GTK_POLICY_ALWAYS
					    : GTK_POLICY_AUTOMATIC);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "show_threadlist_entry");
	if (tmp_data != NULL)
	  {
	    application->show_threadlist_entry_when_thread_selected
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "filter_setting");

	if (tmp_data != NULL)
	  {
	    const ThreadlistFilter *filter;
	    filter = threadlist_filter_setting_get_filter(THREADLIST_FILTER_SETTING(tmp_data));
	    G_FREE(application->user_default_filter.rule);
	    application->user_default_filter.rule = G_STRDUP(filter->rule);
	    application->user_default_filter.ignore_threshold
	      = filter->ignore_threshold;
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "enable_popup_response");
	if (tmp_data != NULL)
	  {
	    application->enable_popup_response
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "popup_response_delay");
	if (tmp_data != NULL)
	  {
	    application->popup_response_delay
	      = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "popup_close_delay");
	if (tmp_data != NULL)
	  {
	    application->popup_close_delay
	      = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "enable_transparent_a_bone");
	if (tmp_data != NULL)
	  {
	    application->enable_transparent_a_bone
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "disable_popup_a_bone");
	if (tmp_data != NULL)
	  {
	    application->disable_popup_a_bone
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "show_mailto_literally");
	if (tmp_data != NULL)
	  {
	    application->show_mailto_mode
	      = (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data))
		 ? THREAD_VIEW_MAILTO_MODE_SHOW
		 : THREAD_VIEW_MAILTO_MODE_HIDE);
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "a_bone_by_name");
	if (tmp_data != NULL)
	  {
	    application->a_bone_by_name
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "a_bone_by_name_pattern");
	if (tmp_data != NULL)
	  {
	    if (application->a_bone_by_name_pattern != NULL)
	      G_FREE(application->a_bone_by_name_pattern);
	    application->a_bone_by_name_pattern
	      = regex_editor_get_regex_text(REGEX_EDITOR(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "a_bone_by_id");
	if (tmp_data != NULL)
	  {
	    application->a_bone_by_id
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "a_bone_by_id_pattern");
	if (tmp_data != NULL)
	  {
	    if (application->a_bone_by_id_pattern != NULL)
	      G_FREE(application->a_bone_by_id_pattern);
	    application->a_bone_by_id_pattern
	      = regex_editor_get_regex_text(REGEX_EDITOR(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "a_bone_by_content");
	if (tmp_data != NULL)
	  {
	    application->a_bone_by_content
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget),
				     "a_bone_by_content_pattern");
	if (tmp_data != NULL)
	  {
	    if (application->a_bone_by_content_pattern != NULL)
	      G_FREE(application->a_bone_by_content_pattern);
	    application->a_bone_by_content_pattern
	      = regex_editor_get_regex_text(REGEX_EDITOR(tmp_data));
	  }

	tmp_data = g_object_get_data(G_OBJECT(widget), "default_open_in_tab");
	if (tmp_data != NULL)
	  {
	    application->default_open_in_tab
	      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tmp_data));
	    boardlist_view_set_default_open_in_tab(
				BOARDLIST_VIEW(application->boardlist_view),
				application->default_open_in_tab);
	    if (current_board_pane != NULL)
	      {
		GtkWidget *tmp_widget
		  = paned_notebook_get_selector(PANED_NOTEBOOK(current_board_pane));
		if (tmp_widget != NULL)
		  {
		    tmp_widget = gtk_bin_get_child(GTK_BIN(tmp_widget));
		    threadlist_view_set_default_open_in_tab(
					THREADLIST_VIEW(tmp_widget),
					application->default_open_in_tab);
		  }
	      }
	  }

	break;
      }

    case GTK_RESPONSE_CANCEL:
      break;
    }

  if (response_id == GTK_RESPONSE_APPLY)
    return;

  gtk_widget_destroy(widget);
}


static GtkWidget *
build_ochusha_setting_frame(OchushaApplication *application,
			    GtkWidget *dialog)
{
  GtkWidget *ochusha_frame = gtk_frame_new(_("Setting of Ochusha"));
  GtkWidget *outer_vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *proxy_frame;
  GtkWidget *proxy_vbox;
  GtkWidget *hbox;
  GtkWidget *tmp_widget;

  hbox = gtk_hbox_new(FALSE, 0);

  tmp_widget = gtk_label_new(_("External Web Browser:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(tmp_widget), application->web_browser_template);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, TRUE, TRUE, 5);
  g_object_set_data(G_OBJECT(dialog), "web_browser_template",
		    tmp_widget);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(outer_vbox), hbox, FALSE, FALSE, 5);

  hbox = gtk_hbox_new(FALSE, 0);

  tmp_widget = gtk_label_new(_("(%s will be replaced with URI)"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 100);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(outer_vbox), hbox, FALSE, FALSE, 0);

  gtk_frame_set_shadow_type(GTK_FRAME(ochusha_frame), GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(outer_vbox);
  gtk_container_add(GTK_CONTAINER(ochusha_frame), outer_vbox);

  tmp_widget = gtk_check_button_new_with_label(_("Open in Tab by Default."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->default_open_in_tab);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 5);
  g_object_set_data(G_OBJECT(dialog), "default_open_in_tab", tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("Enable Tooltips of Board Tab Label."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->board_tab_enable_tooltips);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 5);
  g_object_set_data(G_OBJECT(dialog), "board_tab_enable_tooltips",
		    tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("Always Show Board Tab."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->board_tab_always_show);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 5);
  g_object_set_data(G_OBJECT(dialog), "board_tab_always_show", tmp_widget);

#if 0
  tmp_widget = gtk_check_button_new_with_label(_("Place Boardlist and Threadlist Vertically."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->boardlist_pane_style
			       == VPANED_STYLE);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 5);
  g_object_set_data(G_OBJECT(dialog), "boardlist_pane_style",
		    tmp_widget);
#endif

  tmp_widget = gtk_check_button_new_with_label(_("Place Threadlist and Thread Vertically."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->threadlist_pane_style
			       == VPANED_STYLE);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 5);
  g_object_set_data(G_OBJECT(dialog), "threadlist_pane_style",
		    tmp_widget);

  proxy_frame = gtk_frame_new(_("Proxy Setting"));
  proxy_vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(proxy_vbox);
  gtk_container_add(GTK_CONTAINER(proxy_frame), proxy_vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  tmp_widget = gtk_label_new(_("WARNING: Use of PROXY never tested.  If you know whether this works well or not, please report that."));
  gtk_label_set_justify(GTK_LABEL(tmp_widget), GTK_JUSTIFY_LEFT);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(proxy_vbox), hbox, FALSE, FALSE, 5);

  hbox = gtk_hbox_new(FALSE, 0);
  tmp_widget = gtk_label_new(_("WARNING: If specified PROXY server isn't alive, Ochusha may hang up."));
  gtk_label_set_justify(GTK_LABEL(tmp_widget), GTK_JUSTIFY_LEFT);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(proxy_vbox), hbox, FALSE, FALSE, 5);

  hbox = gtk_hbox_new(FALSE, 0);
  tmp_widget = gtk_label_new(_("WARNING: Sorry, I cannot recommend to use PROXY at this time."));
  gtk_label_set_justify(GTK_LABEL(tmp_widget), GTK_JUSTIFY_LEFT);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(proxy_vbox), hbox, FALSE, FALSE, 5);

  hbox = gtk_hbox_new(FALSE, 0);

  tmp_widget = gtk_radio_button_new_with_label(NULL, _("Don't Use Proxy"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);

  tmp_widget
    = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tmp_widget),
						  _("Use Proxy"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->config.enable_proxy);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "enable_proxy", tmp_widget);

  tmp_widget
    = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tmp_widget),
						  _("Use Proxy Only for Posting"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->config.enable_proxy_only_for_posting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "enable_proxy_only_for_posting",
		    tmp_widget);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(proxy_vbox), hbox, FALSE, FALSE, 0);



  hbox = gtk_hbox_new(FALSE, 0);

  tmp_widget = gtk_label_new(_("Proxy URL:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  tmp_widget = gtk_entry_new();
  if (application->config.proxy_url != NULL)
    gtk_entry_set_text(GTK_ENTRY(tmp_widget), application->config.proxy_url);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, TRUE, TRUE, 5);
  g_object_set_data(G_OBJECT(dialog), "proxy_url", tmp_widget);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(proxy_vbox), hbox, FALSE, FALSE, 0);


  hbox = gtk_hbox_new(FALSE, 0);

  tmp_widget = gtk_label_new(_("Proxy User:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  tmp_widget = gtk_entry_new();
  if (application->config.proxy_user != NULL)
    gtk_entry_set_text(GTK_ENTRY(tmp_widget), application->config.proxy_user);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, TRUE, TRUE, 5);
  g_object_set_data(G_OBJECT(dialog), "proxy_user", tmp_widget);

  tmp_widget = gtk_label_new(_("Proxy Password:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  tmp_widget = gtk_entry_new();
  if (application->config.proxy_password != NULL)
    gtk_entry_set_text(GTK_ENTRY(tmp_widget),
		       application->config.proxy_password);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, TRUE, TRUE, 5);
  g_object_set_data(G_OBJECT(dialog), "proxy_password", tmp_widget);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(proxy_vbox), hbox, FALSE, FALSE, 5);


  gtk_widget_show(proxy_frame);
  gtk_box_pack_end(GTK_BOX(outer_vbox), proxy_frame, FALSE, FALSE, 5);

  return ochusha_frame;
}


static void
restore_default_cb(ThreadlistFilterSetting *setting,
		   OchushaApplication *application)
{
  threadlist_filter_setting_set_filter(setting,
				       &application->user_default_filter);

  setting->filter.ignore_threshold
    = application->user_default_filter.ignore_threshold;

}


static GtkWidget *
build_threadlist_view_setting_frame(OchushaApplication *application,
				    GtkWidget *dialog)
{
  GtkWidget *threadlist_frame = gtk_frame_new(_("Setting of Threadlist View"));
  GtkWidget *filter_frame
    = gtk_frame_new(_("Default Setting of Threadlist's Filter"));
  GtkWidget *outer_vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *vbox;
  GtkWidget *tmp_widget;

  gtk_frame_set_shadow_type(GTK_FRAME(threadlist_frame),
			    GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(outer_vbox);
  gtk_container_add(GTK_CONTAINER(threadlist_frame), outer_vbox);

  tmp_widget = gtk_check_button_new_with_label(_("Enable Popup of Thread Title."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->enable_popup_title);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "enable_popup_title",
		    tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("Enable Tooltips of Thread Tab Label."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->thread_tab_enable_tooltips);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "thread_tab_enable_tooltips",
		    tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("Always Show Thread Tab."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->thread_tab_always_show);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "thread_tab_always_show", tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("Show Threadlist Entry for Selected Thread."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->show_threadlist_entry_when_thread_selected);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "show_threadlist_entry",
		    tmp_widget);

  gtk_frame_set_shadow_type(GTK_FRAME(filter_frame), GTK_SHADOW_ETCHED_OUT);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(filter_frame), vbox);

  tmp_widget = threadlist_filter_setting_new_with_initial_setting(&application->user_default_filter);
  g_signal_connect(G_OBJECT(tmp_widget), "restore_default",
		   G_CALLBACK(restore_default_cb), application);

  g_object_set_data(G_OBJECT(dialog), "filter_setting", tmp_widget);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  gtk_widget_show(filter_frame);

  gtk_box_pack_start(GTK_BOX(outer_vbox), filter_frame, FALSE, FALSE, 5);

  return threadlist_frame;
}


static GtkWidget *
build_thread_view_setting_frame(OchushaApplication *application,
				GtkWidget *dialog)
{
  GtkWidget *outer_frame = gtk_frame_new(_("Setting of Thread View"));
  GtkWidget *outer_vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *thread_frame;
  GtkWidget *tmp_widget;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkObject *adjustment;

  gtk_frame_set_shadow_type(GTK_FRAME(outer_frame), GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(outer_vbox);
  gtk_container_add(GTK_CONTAINER(outer_frame), outer_vbox);

  tmp_widget = gtk_check_button_new_with_label(_("Enable Popup of Response."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->enable_popup_response);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "enable_popup_response",
		    tmp_widget);

  hbox = gtk_hbox_new(FALSE, 0);
  tmp_widget = gtk_label_new(_("Delay to Show Response Popup:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  adjustment = gtk_adjustment_new(application->popup_response_delay,
				  0.0, 10000.0, 100.0, 1000.0, 1000.0);
  tmp_widget = gtk_spin_button_new((GtkAdjustment *)adjustment, 100.0, 0);
  gtk_widget_show(tmp_widget);
  g_object_set_data(G_OBJECT(dialog), "popup_response_delay",
		    tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_label_new(_("(ms)"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(outer_vbox), hbox, FALSE, FALSE, 0);
  
  hbox = gtk_hbox_new(FALSE, 0);
  tmp_widget = gtk_label_new(_("Delay to Close Response Popup:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  adjustment = gtk_adjustment_new(application->popup_close_delay,
				  0.0, 10000.0, 100.0, 1000.0, 1000.0);
  tmp_widget = gtk_spin_button_new((GtkAdjustment *)adjustment, 100.0, 0);
  gtk_widget_show(tmp_widget);
  g_object_set_data(G_OBJECT(dialog), "popup_close_delay",
		    tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_label_new(_("(ms)"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(outer_vbox), hbox, FALSE, FALSE, 0);


  tmp_widget = gtk_check_button_new_with_label(_("Enable transparent a bone."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->enable_transparent_a_bone);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "enable_transparent_a_bone",
		    tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("Disable popup of response made a bone."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->disable_popup_a_bone);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(outer_vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "disable_popup_a_bone", tmp_widget);



  thread_frame = gtk_frame_new(_("Default Setting of Thread Views"));
  gtk_frame_set_shadow_type(GTK_FRAME(thread_frame), GTK_SHADOW_ETCHED_OUT);

  vbox = gtk_vbox_new(FALSE, 0);

  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(thread_frame), vbox);

  tmp_widget = gtk_check_button_new_with_label(_("Show mailto literally."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->show_mailto_mode
			       == THREAD_VIEW_MAILTO_MODE_SHOW);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "show_mailto_literally",
		    tmp_widget);


  tmp_widget = gtk_check_button_new_with_label(_("A bone by name field."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->a_bone_by_name);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "a_bone_by_name", tmp_widget);

  tmp_widget = regex_editor_new(application->a_bone_by_name_pattern);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, TRUE, TRUE, 0);
  g_object_set_data(G_OBJECT(dialog), "a_bone_by_name_pattern", tmp_widget);


  tmp_widget = gtk_check_button_new_with_label(_("A bone by ID."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->a_bone_by_id);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "a_bone_by_id", tmp_widget);

  tmp_widget = regex_editor_new(application->a_bone_by_id_pattern);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, TRUE, TRUE, 0);
  g_object_set_data(G_OBJECT(dialog), "a_bone_by_id_pattern", tmp_widget);


  tmp_widget = gtk_check_button_new_with_label(_("A bone by content."));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp_widget),
			       application->a_bone_by_content);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), "a_bone_by_content", tmp_widget);

  tmp_widget = regex_editor_new(application->a_bone_by_content_pattern);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, TRUE, TRUE, 0);
  g_object_set_data(G_OBJECT(dialog), "a_bone_by_content_pattern",
		    tmp_widget);


  gtk_widget_show(thread_frame);
  gtk_box_pack_start(GTK_BOX(outer_vbox), thread_frame, TRUE, TRUE, 5);

  return outer_frame;
}


#if ENABLE_DEBUG_CONSOLE
static void
debug_console_response_cb(GtkWidget *widget, gint response_id,
			  gpointer unused)
{
  if (widget == NULL)
    return;

  gtk_widget_hide(widget);
}


static void
open_debug_console(OchushaApplication *application)
{
  GtkWidget *dialog;
  GtkWidget *scroll;
  GtkWidget *text_view;
  if (debug_console != NULL)
    {
      gtk_widget_show_all(debug_console);
      return;
    }

  dialog = gtk_dialog_new_with_buttons(_("Debug Outputs"),
				       application->top_level,
				       GTK_DIALOG_DESTROY_WITH_PARENT,
				       GTK_STOCK_OK, GTK_RESPONSE_OK,
				       NULL);
  g_signal_connect(G_OBJECT(dialog), "response",
		   G_CALLBACK(debug_console_response_cb), application);
  g_signal_connect(G_OBJECT(dialog), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &debug_console);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);

  scroll = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
				      GTK_SHADOW_ETCHED_IN);
  text_view = gtk_text_view_new_with_buffer(log_buffer);
  gtk_widget_show(text_view);
  gtk_container_add(GTK_CONTAINER(scroll), text_view);
  gtk_widget_show(scroll);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), scroll);

  gtk_widget_show(dialog);
  debug_console = dialog;
}
#endif


static void
open_preference_dialog(OchushaApplication *application)
{
  MAIN_UI_LOCK
  {
    if (preference_dialog == NULL)
      {
	GtkWidget *notebook;
	GtkWidget *tmp_widget;
	preference_dialog
	  = gtk_dialog_new_with_buttons(_("Preferences of Ochusha"),
					application->top_level,
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
					GTK_STOCK_OK, GTK_RESPONSE_OK,
					NULL);
	gtk_dialog_set_default_response(GTK_DIALOG(preference_dialog),
					GTK_RESPONSE_OK);
	g_signal_connect(G_OBJECT(preference_dialog), "response",
			 G_CALLBACK(preference_dialog_response_cb),
			 application);
	g_signal_connect(G_OBJECT(preference_dialog), "destroy",
			 G_CALLBACK(gtk_widget_destroyed),
			 &preference_dialog);

	notebook = gtk_notebook_new();
	gtk_widget_show(notebook);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(preference_dialog)->vbox),
			  notebook);

	tmp_widget = build_ochusha_setting_frame(application,
						 preference_dialog);
	gtk_widget_show(tmp_widget);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), tmp_widget,
				 gtk_label_new(_("Setting of Ochusha")));

	tmp_widget = build_threadlist_view_setting_frame(application,
							 preference_dialog);
	gtk_widget_show(tmp_widget);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), tmp_widget,
				 gtk_label_new(_("Setting of Threadlist View")));

	tmp_widget = build_thread_view_setting_frame(application,
						     preference_dialog);
	gtk_widget_show(tmp_widget);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), tmp_widget,
				 gtk_label_new(_("Setting of Thread View")));

	gtk_widget_show(preference_dialog);
      }
  }
  MAIN_UI_UNLOCK;
}


static void
open_about_dialog(OchushaApplication *application)
{
  MAIN_UI_LOCK
  {
    if (about_dialog == NULL)
      {
	gchar text_buffer[1024];
	GtkWidget *label;
	GdkPixbuf *banner = NULL;

	about_dialog = gtk_dialog_new_with_buttons(_("About Ochusha"),
					application->top_level,
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_STOCK_OK, GTK_RESPONSE_OK,
					NULL);
	gtk_dialog_set_default_response(GTK_DIALOG(about_dialog),
					GTK_RESPONSE_OK);
	g_signal_connect(G_OBJECT(about_dialog), "response",
			 G_CALLBACK(about_dialog_response_cb),
			 NULL);
	g_signal_connect(G_OBJECT(about_dialog), "destroy",
			 G_CALLBACK(gtk_widget_destroyed),
			 &about_dialog);

	snprintf(text_buffer, 1024, "%s %s %s",
		 _("ochusha"), _("Version"), PACKAGE_VERSION);
	label = gtk_label_new(text_buffer);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
			  label);

#ifdef HAVE_LOCALTIME_R
	{
	  struct tm tm;
	  time_t utc = time(NULL);
	  localtime_r(&utc, &tm);

	  if (tm.tm_year == 104 && tm.tm_yday < 7)
	    banner = banner1_pixbuf;
	  else
	    banner = banner_pixbuf;
	}
#else
	banner = banner_pixbuf;
#endif
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
			  gtk_image_new_from_pixbuf(banner));

	label = gtk_label_new("Copyright(c) 2003 The Ochusha Project");
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
			  label);

	gtk_widget_show_all(about_dialog);
      }
  }
  MAIN_UI_UNLOCK;
}


static void
about_dialog_response_cb(GtkWidget *widget, gint response_id, gpointer unused)
{
  if (widget == NULL)
    return;

  gtk_widget_destroy(widget);
}


static void
initialize_open_url_dialog(OchushaApplication *application)
{
  GtkWidget *tmp_hbox;
  open_url_dialog = gtk_message_dialog_new(application->top_level,
					   GTK_DIALOG_DESTROY_WITH_PARENT,
					   GTK_MESSAGE_QUESTION,
					   GTK_BUTTONS_OK_CANCEL,
					   _("Input the URL of the document you want to open."));
  gtk_dialog_set_default_response(GTK_DIALOG(open_url_dialog),
				  GTK_RESPONSE_OK);
  g_signal_connect(G_OBJECT(open_url_dialog), "response",
		   G_CALLBACK(open_url_dialog_response_cb),
		   application);
  g_signal_connect(G_OBJECT(open_url_dialog), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &open_url_dialog);


  open_url_combo = gtk_combo_new();
  if (open_url_history != NULL)
    gtk_combo_set_popdown_strings(GTK_COMBO(open_url_combo),
				  open_url_history);
  gtk_combo_disable_activate(GTK_COMBO(open_url_combo));
  g_signal_connect(G_OBJECT(GTK_COMBO(open_url_combo)->entry), "activate",
		   G_CALLBACK(open_url_combo_entry_activate_cb),
		   open_url_dialog);
  tmp_hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tmp_hbox), gtk_label_new(_("URL: ")),
		     FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tmp_hbox), open_url_combo, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(open_url_dialog)->vbox),
		     tmp_hbox, FALSE, FALSE, 0);
  open_url_in_tab_button = gtk_check_button_new_with_label(_("Open in tab"));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(open_url_dialog)->vbox),
		     open_url_in_tab_button, FALSE, FALSE, 0);
}


static void
open_url_combo_entry_activate_cb(GtkWidget *entry, GtkDialog *dialog)
{
  gtk_dialog_response(dialog, GTK_RESPONSE_OK);
}


static void
open_url_dialog_response_cb(GtkWidget *dialog, gint response_id,
			    OchushaApplication *application)
{
  if (response_id == GTK_RESPONSE_OK)
    {
      const gchar *url
	= gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(open_url_combo)->entry));
      gboolean in_tab = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(open_url_in_tab_button));
      ochusha_open_url(application, url, in_tab, FALSE);
    }

  gtk_widget_hide_all(open_url_dialog);

#if 0	/* ZZZ */
  if (response_id == GTK_RESPONSE_DELETE_EVENT)
    {
      open_url_dialog = NULL;
    }
#endif
}


static void
popup_open_url_dialog(OchushaApplication *application)
{
  if (open_url_dialog == NULL)
    initialize_open_url_dialog(application);
  gtk_widget_show_all(open_url_dialog);
}


#if ENABLE_RENDERING_ON_IDLE
/*
 * ĤGDKΥƥ륻Ǽ롣
 */
static gint current_rendering_load = 0;
static GSList *rendered_buffers = NULL;
static guint advance_rendering_on_idle_func_id = 0;


static void
wake_up_helper(OchushaAsyncBuffer *buffer, OchushaApplication *application)
{
  ochusha_async_buffer_broadcast(buffer, "ochusha_ui.c: wake_up_helper()");
  g_object_unref(G_OBJECT(buffer));
}


/*
 * advance_rendering_on_idle:
 *
 * GLibΥᥤ롼פidleؿȤϿᥤ롼פˤˤʤä
 * ԤäƤ󥰥åɤ򵯤
 */
static gboolean
advance_rendering_on_idle(gpointer application)
{
  GSList *old_rendered_buffers;

  gdk_threads_enter();

  old_rendered_buffers = rendered_buffers;
  rendered_buffers = NULL;
  advance_rendering_on_idle_func_id = 0;
  current_rendering_load = 0;

  gdk_threads_leave();

#if DEBUG_RENDERING_ON_IDLE
  fprintf(stderr, "advance_rendering_on_idle()\n");
#endif
  /*
   * 򵯤Τ̵̤ͥåȥξ֤
   * ԤΥǡ̤ʤɺ٤򵤤ˤƤޤǸ̩˥åɿĴ
   * ɬפϤʤȻפΤǡޤåɥå
   * ƤåɤοȤΤ̵̤餯ʤΤ
   * GUIΥ쥹ݥ󥹤˱ƶͿʤ
   * ΤϥǤȤȤǡ
   * application->maximum_number_of_renderingοåɤ򵯤
   * Żʤåɤ̤Υåɤ򵯤Ȥץȥ
   * Ȥˤ롣
   */
  g_slist_foreach(old_rendered_buffers, (GFunc)wake_up_helper, application);

  return FALSE;
}


/*
 * try_get_rendering_right:
 *
 * item󥰤륹åɤߥ󥰤Ԥäɤɤ
 * Ƚꤹ롣ߥ󥰤ԤäƤ륹åɿ
 * application->maximum_number_of_rendering_threads̤ǤС
 * 󥰤ĤȤ̣TRUE֤ʳξˤFALSE֤
 * 󥰤ԤåɤϤδؿFALSE֤Ȥˤϥ
 * Ԥ鷺bufferwait롣TRUEåɤ1ñʬ󥰤
 * Ԥä顢return_rendering_semaphore()ˤꥻޥե֤
 *
 * gdk_threads_enter()ĶƤӽФȡ
 */
gboolean
try_get_rendering_right(OchushaApplication *application,
			OchushaAsyncBuffer *buffer, gpointer item)
{
  gboolean result = FALSE;
  PanedNotebook *contents_window
    = PANED_NOTEBOOK(application->contents_window);
  OchushaBulletinBoard *current_board
    = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);

#if DEBUG_RENDERING_ON_IDLE
  fprintf(stderr, "try_get_rendering_semaphore(): buffer=%p, item=%p\n",
	  buffer, item);
#endif

  if (current_board == NULL || application->rendering_unit == 0)
    result = TRUE;
  else
    {
      if (OCHUSHA_IS_BULLETIN_BOARD(item)
	  && (current_board == item
	      || current_rendering_load < application->rendering_unit))
	result = TRUE;
      else if (OCHUSHA_IS_BBS_THREAD(item))
	{
	  PanedNotebook *board_pane = (PanedNotebook *)paned_notebook_get_current_item_view(contents_window);
	  if (board_pane == NULL)
	    result = TRUE;
	  else
	    {
	      OchushaBBSThread *current_thread
		= (OchushaBBSThread *)paned_notebook_get_current_item(board_pane);
	      if (current_thread == NULL || current_thread == item
		  || current_rendering_load < application->rendering_unit)
		result = TRUE;
	    }
	}
    }

  if (result)
    {
      if (OCHUSHA_IS_BULLETIN_BOARD(item))
	current_rendering_load += application->weight_of_threadlist_rendering;
      else if (OCHUSHA_IS_BBS_THREAD(item))
	current_rendering_load += application->weight_of_response_rendering;
#if DEBUG_RENDERING_ON_IDLE
      else
	fprintf(stderr, "Unknown item(%p) found\n", item);
#endif
    }
  else
    {
      g_object_ref(G_OBJECT(buffer));
      rendered_buffers = g_slist_append(rendered_buffers, buffer);

      if (advance_rendering_on_idle_func_id == 0)
	advance_rendering_on_idle_func_id
	  = g_idle_add_full(G_PRIORITY_HIGH,
			    advance_rendering_on_idle, application, NULL);
    }

#if DEBUG_RENDERING_ON_IDLE
  fprintf(stderr, "  result=%d, current_rendering_load=%d\n",
	  result, current_rendering_load);
#endif

  return result;
}
#endif
